/*
 *  ShipClass.cpp
 */

#include "ShipClass.h"

#include "XWingDefs.h"
#include "RaptorGame.h"
#include "Shot.h"
#include "Str.h"
#include <fstream>
#include <algorithm>
#include "Num.h"


ShipClass::ShipClass( uint32_t id ) : GameObject( id, XWing::Object::SHIP_CLASS )
{
	Category = CATEGORY_FIGHTER;
	Team = XWing::Team::NONE;
	Radius = 4.5;
	CollisionDamage = 1000.;
	MaxSpeed = 170.;
	Acceleration = 85.;
	RollSlow = RollFast = 180.;
	PitchSlow = PitchFast = 100.;
	YawSlow = YawFast = 80.;
	RollExponent = PitchExponent = YawExponent = 1.;
	RollChangeSlow = RollChangeFast = 8. * RollSlow;
	PitchChangeSlow = PitchChangeFast = 8. * PitchSlow;
	YawChangeSlow = YawChangeFast = 8. * YawSlow;
	RollChangeExponent = PitchChangeExponent = YawChangeExponent = 2.;
	MaxHealth = 100.;
	MaxShield = 0.;
	ShieldRechargeDelay = 5.;
	ShieldRechargeRate = 0.;
	ExplosionRate = 1.;
	ModelScale = 0.022;
}


ShipClass::ShipClass( const ShipClass &other ) : GameObject( 0, XWing::Object::SHIP_CLASS )
{
	ShortName = other.ShortName;
	LongName = other.LongName;
	Squadron = other.Squadron;
	Category = other.Category;
	Team = other.Team;
	Radius = other.Radius;
	CollisionDamage = other.CollisionDamage;
	MaxSpeed = other.MaxSpeed;
	Acceleration = other.Acceleration;
	RollSlow = other.RollSlow;
	PitchSlow = other.PitchSlow;
	YawSlow = other.YawSlow;
	RollFast = other.RollFast;
	PitchFast = other.PitchFast;
	YawFast = other.YawFast;
	RollExponent = other.RollExponent;
	PitchExponent = other.PitchExponent;
	YawExponent = other.YawExponent;
	RollChangeSlow = other.RollChangeSlow;
	PitchChangeSlow = other.PitchChangeSlow;
	YawChangeSlow = other.YawChangeSlow;
	RollChangeFast = other.RollChangeFast;
	PitchChangeFast = other.PitchChangeFast;
	YawChangeFast = other.YawChangeFast;
	RollChangeExponent = other.RollChangeExponent;
	PitchChangeExponent = other.PitchChangeExponent;
	YawChangeExponent = other.YawChangeExponent;
	MaxHealth = other.MaxHealth;
	MaxShield = other.MaxShield;
	ShieldRechargeDelay = other.ShieldRechargeDelay;
	ShieldRechargeRate = other.ShieldRechargeRate;
	ExplosionRate = other.ExplosionRate;
	Subsystems = other.Subsystems;
	Weapons = other.Weapons;
	FireTime = other.FireTime;
	Ammo = other.Ammo;
	Turrets = other.Turrets;
	CollisionModel = other.CollisionModel;
	ExternalModel = other.ExternalModel;
	CockpitModel = other.CockpitModel;
	CockpitPos.Copy( &(other.CockpitPos) );
	ModelScale = other.ModelScale;
	FlybySounds = other.FlybySounds;
}


ShipClass::~ShipClass()
{
}


static uint8_t ShotTypeFromString( std::string type, uint32_t team )
{
	std::transform( type.begin(), type.end(), type.begin(), tolower );
	
	if( (type == "red_laser") || (type == "laser_red") )
		return Shot::TYPE_LASER_RED;
	else if( (type == "green_laser") || (type == "laser_green") )
		return Shot::TYPE_LASER_GREEN;
	else if( (type == "red_turbolaser") || (type == "turbolaser_red") )
		return Shot::TYPE_TURBO_LASER_RED;
	else if( (type == "green_turbolaser") || (type == "turbolaser_green") )
		return Shot::TYPE_TURBO_LASER_GREEN;
	else if( type == "ion_cannon" )
		return Shot::TYPE_ION_CANNON;
	else if( type == "torpedo" )
		return Shot::TYPE_TORPEDO;
	else if( type == "missile" )
		return Shot::TYPE_MISSILE;
	else if( type == "turbolaser" )
		return (team == XWing::Team::EMPIRE) ? Shot::TYPE_TURBO_LASER_GREEN : Shot::TYPE_TURBO_LASER_RED;
	
	return (team == XWing::Team::EMPIRE) ? Shot::TYPE_LASER_GREEN : Shot::TYPE_LASER_RED;
}


bool ShipClass::Load( const std::string &filename )
{
	std::ifstream input( filename.c_str() );
	if( ! input.is_open() )
		return false;
	
	char buffer[ 1024 ] = "";
	while( ! input.eof() )
	{
		buffer[ 0 ] = '\0';
		input.getline( buffer, sizeof(buffer) );
		
		std::list<std::string> parsed = CStr::ParseCommand( buffer );
		if( ! parsed.size() )
			continue;
		
		std::string var = *(parsed.begin());
		std::transform( var.begin(), var.end(), var.begin(), tolower );
		parsed.erase( parsed.begin() );
		std::vector<std::string> args( parsed.begin(), parsed.end() );
		
		if( (var == "id") && args.size() )
		{
			ShortName = args.at(0);
			std::transform( ShortName.begin(), ShortName.end(), ShortName.begin(), toupper );
		}
		else if( (var == "name") && args.size() )
		{
			LongName = Str::Join( args, " " );
		}
		else if( (var == "squadron") && args.size() )
		{
			Squadron = Str::Join( args, " " );
		}
		else if( (var == "category") && args.size() )
		{
			std::string category = args.at(0);
			std::transform( category.begin(), category.end(), category.begin(), tolower );
			if( category == "fighter" )
				Category = CATEGORY_FIGHTER;
			else if( category == "bomber" )
				Category = CATEGORY_BOMBER;
			else if( category == "capital" )
				Category = CATEGORY_CAPITAL;
			else if( category == "target" )
				Category = CATEGORY_TARGET;
		}
		else if( (var == "team") && args.size() )
		{
			std::string team = args.at(0);
			std::transform( team.begin(), team.end(), team.begin(), tolower );
			if( team == "rebel" )
				Team = XWing::Team::REBEL;
			else if( team == "empire" )
				Team = XWing::Team::EMPIRE;
			else
				Team = XWing::Team::NONE;
		}
		else if( (var == "radius") && args.size() )
		{
			Radius = atof( args.at(0).c_str() );
		}
		else if( (var == "ramming") && args.size() )
		{
			CollisionDamage = atof( args.at(0).c_str() );
		}
		else if( (var == "speed") && args.size() )
		{
			MaxSpeed = atof( args.at(0).c_str() );
		}
		else if( (var == "accel") && args.size() )
		{
			Acceleration = atof( args.at(0).c_str() );
		}
		else if( (var == "roll") && args.size() )
		{
			RollSlow = atof( args.at(0).c_str() );
			if( args.size() >= 2 )
				RollFast = atof( args.at(1).c_str() );
			else
				RollFast = RollSlow;
			if( args.size() >= 3 )
				RollExponent = atof( args.at(2).c_str() );
			else
				RollExponent = 2.;
			RollChangeSlow = 8. * RollSlow;
			RollChangeFast = 8. * RollFast;
		}
		else if( (var == "pitch") && args.size() )
		{
			PitchSlow = atof( args.at(0).c_str() );
			if( args.size() >= 2 )
				PitchFast = atof( args.at(1).c_str() );
			else
				PitchFast = PitchSlow;
			if( args.size() >= 3 )
				PitchExponent = atof( args.at(2).c_str() );
			else
				PitchExponent = 2.;
			PitchChangeSlow = 8. * PitchSlow;
			PitchChangeFast = 8. * PitchFast;
		}
		else if( (var == "yaw") && args.size() )
		{
			YawSlow = atof( args.at(0).c_str() );
			if( args.size() >= 2 )
				YawFast = atof( args.at(1).c_str() );
			else
				YawFast = YawSlow;
			if( args.size() >= 3 )
				YawExponent = atof( args.at(2).c_str() );
			else
				YawExponent = 2.;
			YawChangeSlow = 8. * YawSlow;
			YawChangeFast = 8. * YawFast;
		}
		else if( (var == "roll_change") && args.size() )
		{
			RollChangeSlow = atof( args.at(0).c_str() );
			if( args.size() >= 2 )
				RollChangeFast = atof( args.at(1).c_str() );
			else
				RollChangeFast = RollChangeSlow;
			if( args.size() >= 3 )
				RollChangeExponent = atof( args.at(2).c_str() );
			else
				RollChangeExponent = 2.;
		}
		else if( (var == "pitch_change") && args.size() )
		{
			PitchChangeSlow = atof( args.at(0).c_str() );
			if( args.size() >= 2 )
				PitchChangeFast = atof( args.at(1).c_str() );
			else
				PitchChangeFast = PitchChangeSlow;
			if( args.size() >= 3 )
				PitchChangeExponent = atof( args.at(2).c_str() );
			else
				PitchChangeExponent = 2.;
		}
		else if( (var == "yaw_change") && args.size() )
		{
			YawChangeSlow = atof( args.at(0).c_str() );
			if( args.size() >= 2 )
				YawChangeFast = atof( args.at(1).c_str() );
			else
				YawChangeFast = YawChangeSlow;
			if( args.size() >= 3 )
				YawChangeExponent = atof( args.at(2).c_str() );
			else
				YawChangeExponent = 2.;
		}
		else if( (var == "health") && args.size() )
		{
			MaxHealth = atof( args.at(0).c_str() );
		}
		else if( (var == "shield") && args.size() )
		{
			MaxShield = atof( args.at(0).c_str() );
		}
		else if( (var == "recover") && args.size() )
		{
			ShieldRechargeDelay = atof( args.at(0).c_str() );
		}
		else if( (var == "recharge") && args.size() )
		{
			ShieldRechargeRate = atof( args.at(0).c_str() );
		}
		else if( (var == "explode") && args.size() )
		{
			ExplosionRate = atof( args.at(0).c_str() );
		}
		else if( (var == "subsystem") && (args.size() >= 2) )
		{
			Subsystems[ args.at(0) ] = atof( args.at(1).c_str() );
		}
		else if( (var == "weapon") && (args.size() >= 4) )
		{
			uint8_t type = ShotTypeFromString( args.at(0), Team );
			double right = atof( args.at(1).c_str() );
			double up    = atof( args.at(2).c_str() );
			double fwd   = atof( args.at(3).c_str() );
			Weapons[ type ].push_back( Pos3D( fwd, up, right ) );
		}
		else if( (var == "firetime") && (args.size() >= 2) )
		{
			uint8_t type = ShotTypeFromString( args.at(0), Team );
			FireTime[ type ] = atof( args.at(1).c_str() );
		}
		else if( (var == "ammo") && (args.size() >= 2) )
		{
			uint8_t type = ShotTypeFromString( args.at(0), Team );
			Ammo[ type ] = atoi( args.at(1).c_str() );
		}
		else if( (var == "turret") && (args.size() >= 4) )
		{
			uint8_t type = ShotTypeFromString( args.at(0), Team );
			double right = atof( args.at(1).c_str() );
			double up    = atof( args.at(2).c_str() );
			double fwd   = atof( args.at(3).c_str() );
			args.erase( args.begin() );
			args.erase( args.begin() );
			args.erase( args.begin() );
			args.erase( args.begin() );
			Turrets.push_back( ShipClassTurret( fwd, up, right, type ) );
			
			while( args.size() )
			{
				std::string subvar = *(args.begin());
				std::transform( subvar.begin(), subvar.end(), subvar.begin(), tolower );
				args.erase( args.begin() );
				
				if( (subvar == "up") && (args.size() >= 3) )
				{
					double right = atof( args.at(0).c_str() );
					double up    = atof( args.at(1).c_str() );
					double fwd   = atof( args.at(2).c_str() );
					args.erase( args.begin() );
					args.erase( args.begin() );
					args.erase( args.begin() );
					Turrets.back().Up.Set( fwd, up, right );
				}
				else if( (subvar == "dir") && (args.size() >= 3) )
				{
					double right = atof( args.at(0).c_str() );
					double up    = atof( args.at(1).c_str() );
					double fwd   = atof( args.at(2).c_str() );
					args.erase( args.begin() );
					args.erase( args.begin() );
					args.erase( args.begin() );
					Turrets.back().Fwd.Set( fwd, up, right );
				}
				else if( (subvar == "arc") && args.size() )
				{
					Turrets.back().TargetArc = atof( args.at(0).c_str() );
					args.erase( args.begin() );
				}
				else if( (subvar == "pitch") && (args.size() >= 2) )
				{
					Turrets.back().MinGunPitch = atof( args.at(0).c_str() );
					Turrets.back().MaxGunPitch = atof( args.at(1).c_str() );
					args.erase( args.begin() );
					args.erase( args.begin() );
				}
				else if( subvar == "visible" )
				{
					Turrets.back().Visible = true;
				}
				else if( subvar == "hidden" )
				{
					Turrets.back().Visible = false;
				}
				else if( subvar == "independent" )
				{
					Turrets.back().ParentControl = false;
				}
				else if( subvar == "linked" )
				{
					Turrets.back().ParentControl = true;
				}
				else if( subvar == "dual" )
				{
					Turrets.back().FiringMode = 2;
				}
				else if( (subvar == "firetime") && args.size() )
				{
					Turrets.back().SingleShotDelay = atof( args.at(0).c_str() );
					args.erase( args.begin() );
				}
			}
		}
		else if( (var == "cockpit") && (args.size() >= 4) )
		{
			CockpitModel = args.at(0);
			CockpitPos.Z = atof( args.at(1).c_str() ); // Right
			CockpitPos.Y = atof( args.at(2).c_str() ); // Up
			CockpitPos.X = atof( args.at(3).c_str() ); // Fwd
		}
		else if( (var == "model") && args.size() )
		{
			ExternalModel = args.at(0);
		}
		else if( (var == "model_collision") && args.size() )
		{
			CollisionModel = args.at(0);
		}
		else if( (var == "model_scale") && args.size() )
		{
			ModelScale = atof( args.at(0).c_str() );
		}
		else if( (var == "flyby") && (args.size() >= 2) )
		{
			FlybySounds[ atof( args.at(0).c_str() ) ] = args.at(1);
		}
	}
	input.close();
	
	for( std::map< uint8_t, std::vector<Pos3D> >::const_iterator weapon_iter = Weapons.begin(); weapon_iter != Weapons.end(); weapon_iter ++ )
	{
		if( Ammo.find( weapon_iter->first ) == Ammo.end() )
			Ammo[ weapon_iter->first ] = -1;
		if( FireTime.find( weapon_iter->first ) == FireTime.end() )
			FireTime[ weapon_iter->first ] = 0.25;
	}
	
	// Return true if we loaded at least some basic ship data from the file.
	return (ShortName.length() && LongName.length());
}


void ShipClass::AddToInitPacket( Packet *packet, int8_t precision )
{
	packet->AddString( ShortName );
	packet->AddString( LongName );
	packet->AddChar( Category );
	packet->AddUInt( Team );
	packet->AddFloat( Radius );
	packet->AddFloat( MaxSpeed );
	packet->AddFloat( Acceleration );
	packet->AddFloat( RollSlow );
	packet->AddFloat( PitchSlow );
	packet->AddFloat( YawSlow );
	packet->AddFloat( RollFast );
	packet->AddFloat( PitchFast );
	packet->AddFloat( YawFast );
	packet->AddFloat( RollExponent );
	packet->AddFloat( PitchExponent );
	packet->AddFloat( YawExponent );
	packet->AddFloat( RollChangeSlow );
	packet->AddFloat( PitchChangeSlow );
	packet->AddFloat( YawChangeSlow );
	packet->AddFloat( RollChangeFast );
	packet->AddFloat( PitchChangeFast );
	packet->AddFloat( YawChangeFast );
	packet->AddFloat( RollChangeExponent );
	packet->AddFloat( PitchChangeExponent );
	packet->AddFloat( YawChangeExponent );
	packet->AddFloat( MaxHealth );
	
	packet->AddFloat( MaxShield );
	if( MaxShield )
	{
		packet->AddFloat( ShieldRechargeDelay );
		packet->AddFloat( ShieldRechargeRate );
	}
	
	packet->AddFloat( ExplosionRate );
	
	packet->AddUChar( Weapons.size() );
	for( std::map< uint8_t, std::vector<Pos3D> >::const_iterator weapon_iter = Weapons.begin(); weapon_iter != Weapons.end(); weapon_iter ++ )
	{
		packet->AddUInt( weapon_iter->first );
		
		std::map<uint8_t,int8_t>::const_iterator ammo_iter = Ammo.find( weapon_iter->first );
		packet->AddChar( (ammo_iter != Ammo.end()) ? ammo_iter->second : -1 );
		
		packet->AddUChar( weapon_iter->second.size() );
		for( size_t i = 0; i < weapon_iter->second.size(); i ++ )
		{
			packet->AddFloat( weapon_iter->second[ i ].X );
			packet->AddFloat( weapon_iter->second[ i ].Y );
			packet->AddFloat( weapon_iter->second[ i ].Z );
		}
	}
	
	packet->AddString( CollisionModel );
	packet->AddString( ExternalModel );
	
	packet->AddString( CockpitModel );
	packet->AddFloat( CockpitPos.X );
	packet->AddFloat( CockpitPos.Y );
	packet->AddFloat( CockpitPos.Z );
	
	packet->AddFloat( ModelScale );
	
	packet->AddUChar( FlybySounds.size() );
	for( std::map< double, std::string >::const_iterator flyby_iter = FlybySounds.begin(); flyby_iter != FlybySounds.end(); flyby_iter ++ )
	{
		packet->AddFloat( flyby_iter->first );
		packet->AddString( flyby_iter->second );
	}
}


void ShipClass::ReadFromInitPacket( Packet *packet, int8_t precision )
{
	ShortName           = packet->NextString();
	LongName            = packet->NextString();
	Category            = packet->NextChar();
	Team                = packet->NextUInt();
	Radius              = packet->NextFloat();
	MaxSpeed            = packet->NextFloat();
	Acceleration        = packet->NextFloat();
	RollSlow            = packet->NextFloat();
	PitchSlow           = packet->NextFloat();
	YawSlow             = packet->NextFloat();
	RollFast            = packet->NextFloat();
	PitchFast           = packet->NextFloat();
	YawFast             = packet->NextFloat();
	RollExponent        = packet->NextFloat();
	PitchExponent       = packet->NextFloat();
	YawExponent         = packet->NextFloat();
	RollChangeSlow      = packet->NextFloat();
	PitchChangeSlow     = packet->NextFloat();
	YawChangeSlow       = packet->NextFloat();
	RollChangeFast      = packet->NextFloat();
	PitchChangeFast     = packet->NextFloat();
	YawChangeFast       = packet->NextFloat();
	RollChangeExponent  = packet->NextFloat();
	PitchChangeExponent = packet->NextFloat();
	YawChangeExponent   = packet->NextFloat();
	MaxHealth           = packet->NextFloat();
	
	MaxShield           = packet->NextFloat();
	if( MaxShield )
	{
		ShieldRechargeDelay = packet->NextFloat();
		ShieldRechargeRate  = packet->NextFloat();
	}
	
	ExplosionRate = packet->NextFloat();
	
	size_t num_weapons = packet->NextUChar();
	for( size_t i = 0; i < num_weapons; i ++ )
	{
		uint8_t weapon_id = packet->NextUInt();
		
		Ammo[ weapon_id ] = packet->NextChar();
		
		size_t weapon_count = packet->NextUChar();
		Weapons[ weapon_id ] = std::vector<Pos3D>();
		for( size_t j = 0; j < weapon_count; j ++ )
		{
			double x = packet->NextFloat();
			double y = packet->NextFloat();
			double z = packet->NextFloat();
			Weapons[ weapon_id ].push_back( Pos3D(x,y,z) );
		}
	}
	
	CollisionModel = packet->NextString();
	ExternalModel = packet->NextString();
	
	CockpitModel = packet->NextString();
	CockpitPos.X = packet->NextFloat();
	CockpitPos.Y = packet->NextFloat();
	CockpitPos.Z = packet->NextFloat();
	
	ModelScale = packet->NextFloat();
	
	size_t num_flyby = packet->NextUChar();
	for( size_t i = 0; i < num_flyby; i ++ )
	{
		double flyby_speed = packet->NextFloat();
		FlybySounds[ flyby_speed ] = packet->NextString();
	}
}


bool ShipClass::ServerShouldUpdateOthers( void ) const
{
	return false;
}

bool ShipClass::IsMoving( void ) const
{
	return false;
}


void ShipClass::Update( double dt )
{
}


bool ShipClass::PlayersCanFly( void ) const
{
	return (Category == CATEGORY_FIGHTER) || (Category == CATEGORY_BOMBER);
}


bool ShipClass::operator < ( const ShipClass &other ) const
{
	if( (Category == CATEGORY_TARGET) && (other.Category != CATEGORY_TARGET) )
		return false;
	if( (Category != CATEGORY_TARGET) && (other.Category == CATEGORY_TARGET) )
		return true;
	if( (Category == CATEGORY_CAPITAL) && (other.Category != CATEGORY_CAPITAL) )
		return false;
	if( (Category != CATEGORY_CAPITAL) && (other.Category == CATEGORY_CAPITAL) )
		return true;
	
	if( (Team == XWing::Team::NONE) && (other.Team != XWing::Team::NONE) )
		return true;
	if( (Team != XWing::Team::NONE) && (other.Team == XWing::Team::NONE) )
		return false;
	if( (Team == XWing::Team::REBEL) && (other.Team != XWing::Team::REBEL) )
		return true;
	if( (Team != XWing::Team::REBEL) && (other.Team == XWing::Team::REBEL) )
		return false;
	
	/*
	// This section is disabled because YT-1300 is currently defined as a fighter.
	if( (Category == CATEGORY_FIGHTER) && (other.Category != CATEGORY_FIGHTER) )
		return true;
	if( (Category != CATEGORY_FIGHTER) && (other.Category == CATEGORY_FIGHTER) )
		return false;
	if( (Category == CATEGORY_BOMBER) && (other.Category != CATEGORY_BOMBER) )
		return true;
	if( (Category != CATEGORY_BOMBER) && (other.Category == CATEGORY_BOMBER) )
		return false;
	*/
	
	if( Radius < other.Radius )
		return true;
	if( Radius > other.Radius )
		return false;
	if( CollisionDamage < other.CollisionDamage )
		return true;
	if( CollisionDamage > other.CollisionDamage )
		return false;
	
	return (strcasecmp( ShortName.c_str(), other.ShortName.c_str() ) < 0);
}


// ---------------------------------------------------------------------------


ShipClassTurret::ShipClassTurret( double fwd, double up, double right, uint8_t weapon ) : Pos3D( fwd, up, right )
{
	Weapon = weapon;
	
	Visible = false;
	ParentControl = true;
	FiringMode = 1;
	SingleShotDelay = 0.5;
	TargetArc = 360.;
	MinGunPitch = -10.;
	MaxGunPitch = 90.;
}
