#pragma once

/**
 * \file
 * Contains declarations of all functions and types which represent a public TDLib interface.
 */
#include "td/tl/TlObject.h"

#include <cstdint>
#include <utility>
#include <vector>

#include <jni.h>

#include <string>

namespace td {
class TlStorerToString;

namespace td_api {

/**
 * This type is used to store 32-bit signed integers, which can be represented as Number in JSON.
 */
using int32 = std::int32_t;
/**
 * This type is used to store 53-bit signed integers, which can be represented as Number in JSON.
 */
using int53 = std::int64_t;
/**
 * This type is used to store 64-bit signed integers, which can't be represented as Number in JSON and are represented as String instead.
 */
using int64 = std::int64_t;

/**
 * This type is used to store UTF-8 strings.
 */
using string = std::string;

/**
 * This type is used to store arbitrary sequences of bytes. In JSON interface the bytes are base64-encoded.
 */
using bytes = std::string;

/**
 * This type is used to store a list of objects of any type and is represented as Array in JSON.
 */
template <class Type>
using array = std::vector<Type>;

class Object;
/**
 * This class is a base class for all TDLib API classes and functions.
 */
using BaseObject = Object;

/**
 * A smart wrapper to store a pointer to a TDLib API object. Can be treated as an analogue of std::unique_ptr.
 */
template <class Type>
using object_ptr = ::td::tl_object_ptr<Type>;

/**
 * A function to create a dynamically allocated TDLib API object. Can be treated as an analogue of std::make_unique.
 * Usage example:
 * \code
 * auto get_me_request = td::td_api::make_object<td::td_api::getMe>();
 * auto message_text = td::td_api::make_object<td::td_api::formattedText>("Hello, world!!!",
 *                     td::td_api::array<td::td_api::object_ptr<td::td_api::textEntity>>());
 * auto send_message_request = td::td_api::make_object<td::td_api::sendMessage>(chat_id, 0, nullptr, nullptr, nullptr,
 *      td::td_api::make_object<td::td_api::inputMessageText>(std::move(message_text), nullptr, true));
 * \endcode
 *
 * \tparam Type Type of object to construct.
 * \param[in] args Arguments to pass to the object constructor.
 * \return Wrapped pointer to the created object.
 */
template <class Type, class... Args>
object_ptr<Type> make_object(Args &&... args) {
  return object_ptr<Type>(new Type(std::forward<Args>(args)...));
}

/**
 * A function to cast a wrapped in td::td_api::object_ptr TDLib API object to its subclass or superclass.
 * Casting an object to an incorrect type will lead to undefined behaviour.
 * Usage example:
 * \code
 * td::td_api::object_ptr<td::td_api::callState> call_state = ...;
 * switch (call_state->get_id()) {
 *   case td::td_api::callStatePending::ID: {
 *     auto state = td::td_api::move_object_as<td::td_api::callStatePending>(call_state);
 *     // use state
 *     break;
 *   }
 *   case td::td_api::callStateExchangingKeys::ID: {
 *     // no additional fields, no casting is needed
 *     break;
 *   }
 *   case td::td_api::callStateReady::ID: {
 *     auto state = td::td_api::move_object_as<td::td_api::callStateReady>(call_state);
 *     // use state
 *     break;
 *   }
 *   case td::td_api::callStateHangingUp::ID: {
 *     // no additional fields, no casting is needed
 *     break;
 *   }
 *   case td::td_api::callStateDiscarded::ID: {
 *     auto state = td::td_api::move_object_as<td::td_api::callStateDiscarded>(call_state);
 *     // use state
 *     break;
 *   }
 *   case td::td_api::callStateError::ID: {
 *     auto state = td::td_api::move_object_as<td::td_api::callStateError>(call_state);
 *     // use state
 *     break;
 *   }
 *   default:
 *     assert(false);
 * }
 * \endcode
 *
 * \tparam ToType Type of TDLib API object to move to.
 * \tparam FromType Type of TDLib API object to move from, this is auto-deduced.
 * \param[in] from Wrapped in td::td_api::object_ptr pointer to a TDLib API object.
 */
template <class ToType, class FromType>
object_ptr<ToType> move_object_as(FromType &&from) {
  return object_ptr<ToType>(static_cast<ToType *>(from.release()));
}

/**
 * Returns a string representation of a TDLib API object.
 * \param[in] value The object.
 * \return Object string representation.
 */
std::string to_string(const BaseObject &value);

/**
 * Returns a string representation of a TDLib API object.
 * \tparam T Object type, auto-deduced.
 * \param[in] value The object.
 * \return Object string representation.
 */
template <class T>
std::string to_string(const object_ptr<T> &value) {
  if (value == nullptr) {
    return "null";
  }

  return to_string(*value);
}

/**
 * Returns a string representation of a list of TDLib API objects.
 * \tparam T Object type, auto-deduced.
 * \param[in] values The objects.
 * \return Objects string representation.
 */
template <class T>
std::string to_string(const std::vector<object_ptr<T>> &values) {
  std::string result = "{\n";
  for (const auto &value : values) {
    if (value == nullptr) {
      result += "null\n";
    } else {
      result += to_string(*value);
    }
  }
  result += "}\n";
  return result;
}

const char *&get_package_name_ref();
const char *get_git_commit_hash();

/**
 * This class is a base class for all TDLib API classes.
 */
class Object {
 public:
  virtual ~Object() {
  }

  virtual void store(JNIEnv *env, jobject &s) const {
  }

  virtual void store(TlStorerToString &s, const char *field_name) const = 0;

  static jclass Class;

  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  virtual std::int32_t get_id() const = 0;

  static object_ptr<Object> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is a base class for all TDLib API functions.
 */
class Function: public Object {
 public:
  static jclass Class;

  static object_ptr<Function> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains information about supported accent color for user/chat name, background of empty chat photo, replies to messages and link previews.
 */
class accentColor final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Accent color identifier.
  int32 id_;
  static jfieldID id_fieldID;
  /// Identifier of a built-in color to use in places, where only one color is needed; 0-6.
  int32 built_in_accent_color_id_;
  static jfieldID built_in_accent_color_id_fieldID;
  /// The list of 1-3 colors in RGB format, describing the accent color, as expected to be shown in light themes.
  array<int32> light_theme_colors_;
  static jfieldID light_theme_colors_fieldID;
  /// The list of 1-3 colors in RGB format, describing the accent color, as expected to be shown in dark themes.
  array<int32> dark_theme_colors_;
  static jfieldID dark_theme_colors_fieldID;
  /// The minimum chat boost level required to use the color in a channel chat.
  int32 min_channel_chat_boost_level_;
  static jfieldID min_channel_chat_boost_level_fieldID;

  /**
   * Contains information about supported accent color for user/chat name, background of empty chat photo, replies to messages and link previews.
   */
  accentColor();

  /**
   * Contains information about supported accent color for user/chat name, background of empty chat photo, replies to messages and link previews.
   *
   * \param[in] id_ Accent color identifier.
   * \param[in] built_in_accent_color_id_ Identifier of a built-in color to use in places, where only one color is needed; 0-6.
   * \param[in] light_theme_colors_ The list of 1-3 colors in RGB format, describing the accent color, as expected to be shown in light themes.
   * \param[in] dark_theme_colors_ The list of 1-3 colors in RGB format, describing the accent color, as expected to be shown in dark themes.
   * \param[in] min_channel_chat_boost_level_ The minimum chat boost level required to use the color in a channel chat.
   */
  accentColor(int32 id_, int32 built_in_accent_color_id_, array<int32> &&light_theme_colors_, array<int32> &&dark_theme_colors_, int32 min_channel_chat_boost_level_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -496870680;

  static object_ptr<accentColor> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes gift types that are accepted by a user.
 */
class acceptedGiftTypes final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// True, if unlimited regular gifts are accepted.
  bool unlimited_gifts_;
  static jfieldID unlimited_gifts_fieldID;
  /// True, if limited regular gifts are accepted.
  bool limited_gifts_;
  static jfieldID limited_gifts_fieldID;
  /// True, if upgraded gifts and regular gifts that can be upgraded for free are accepted.
  bool upgraded_gifts_;
  static jfieldID upgraded_gifts_fieldID;
  /// True, if Telegram Premium subscription is accepted.
  bool premium_subscription_;
  static jfieldID premium_subscription_fieldID;

  /**
   * Describes gift types that are accepted by a user.
   */
  acceptedGiftTypes();

  /**
   * Describes gift types that are accepted by a user.
   *
   * \param[in] unlimited_gifts_ True, if unlimited regular gifts are accepted.
   * \param[in] limited_gifts_ True, if limited regular gifts are accepted.
   * \param[in] upgraded_gifts_ True, if upgraded gifts and regular gifts that can be upgraded for free are accepted.
   * \param[in] premium_subscription_ True, if Telegram Premium subscription is accepted.
   */
  acceptedGiftTypes(bool unlimited_gifts_, bool limited_gifts_, bool upgraded_gifts_, bool premium_subscription_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1783521300;

  static object_ptr<acceptedGiftTypes> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains basic information about another user that started a chat with the current user.
 */
class accountInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Month when the user was registered in Telegram; 0-12; may be 0 if unknown.
  int32 registration_month_;
  static jfieldID registration_month_fieldID;
  /// Year when the user was registered in Telegram; 0-9999; may be 0 if unknown.
  int32 registration_year_;
  static jfieldID registration_year_fieldID;
  /// A two-letter ISO 3166-1 alpha-2 country code based on the phone number of the user; may be empty if unknown.
  string phone_number_country_code_;
  static jfieldID phone_number_country_code_fieldID;
  /// Point in time (Unix timestamp) when the user changed name last time; 0 if unknown.
  int32 last_name_change_date_;
  static jfieldID last_name_change_date_fieldID;
  /// Point in time (Unix timestamp) when the user changed photo last time; 0 if unknown.
  int32 last_photo_change_date_;
  static jfieldID last_photo_change_date_fieldID;

  /**
   * Contains basic information about another user that started a chat with the current user.
   */
  accountInfo();

  /**
   * Contains basic information about another user that started a chat with the current user.
   *
   * \param[in] registration_month_ Month when the user was registered in Telegram; 0-12; may be 0 if unknown.
   * \param[in] registration_year_ Year when the user was registered in Telegram; 0-9999; may be 0 if unknown.
   * \param[in] phone_number_country_code_ A two-letter ISO 3166-1 alpha-2 country code based on the phone number of the user; may be empty if unknown.
   * \param[in] last_name_change_date_ Point in time (Unix timestamp) when the user changed name last time; 0 if unknown.
   * \param[in] last_photo_change_date_ Point in time (Unix timestamp) when the user changed photo last time; 0 if unknown.
   */
  accountInfo(int32 registration_month_, int32 registration_year_, string const &phone_number_country_code_, int32 last_name_change_date_, int32 last_photo_change_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1803492711;

  static object_ptr<accountInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains information about the period of inactivity after which the current user's account will automatically be deleted.
 */
class accountTtl final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Number of days of inactivity before the account will be flagged for deletion; 30-730 days.
  int32 days_;
  static jfieldID days_fieldID;

  /**
   * Contains information about the period of inactivity after which the current user's account will automatically be deleted.
   */
  accountTtl();

  /**
   * Contains information about the period of inactivity after which the current user's account will automatically be deleted.
   *
   * \param[in] days_ Number of days of inactivity before the account will be flagged for deletion; 30-730 days.
   */
  explicit accountTtl(int32 days_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1324495492;

  static object_ptr<accountTtl> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class MessageSender;

class ReactionType;

/**
 * Represents a reaction applied to a message.
 */
class addedReaction final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Type of the reaction.
  object_ptr<ReactionType> type_;
  static jfieldID type_fieldID;
  /// Identifier of the chat member, applied the reaction.
  object_ptr<MessageSender> sender_id_;
  static jfieldID sender_id_fieldID;
  /// True, if the reaction was added by the current user.
  bool is_outgoing_;
  static jfieldID is_outgoing_fieldID;
  /// Point in time (Unix timestamp) when the reaction was added.
  int32 date_;
  static jfieldID date_fieldID;

  /**
   * Represents a reaction applied to a message.
   */
  addedReaction();

  /**
   * Represents a reaction applied to a message.
   *
   * \param[in] type_ Type of the reaction.
   * \param[in] sender_id_ Identifier of the chat member, applied the reaction.
   * \param[in] is_outgoing_ True, if the reaction was added by the current user.
   * \param[in] date_ Point in time (Unix timestamp) when the reaction was added.
   */
  addedReaction(object_ptr<ReactionType> &&type_, object_ptr<MessageSender> &&sender_id_, bool is_outgoing_, int32 date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1258586525;

  static object_ptr<addedReaction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class addedReaction;

/**
 * Represents a list of reactions added to a message.
 */
class addedReactions final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The total number of found reactions.
  int32 total_count_;
  static jfieldID total_count_fieldID;
  /// The list of added reactions.
  array<object_ptr<addedReaction>> reactions_;
  static jfieldID reactions_fieldID;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;
  static jfieldID next_offset_fieldID;

  /**
   * Represents a list of reactions added to a message.
   */
  addedReactions();

  /**
   * Represents a list of reactions added to a message.
   *
   * \param[in] total_count_ The total number of found reactions.
   * \param[in] reactions_ The list of added reactions.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  addedReactions(int32 total_count_, array<object_ptr<addedReaction>> &&reactions_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 226352304;

  static object_ptr<addedReactions> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes an address.
 */
class address final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A two-letter ISO 3166-1 alpha-2 country code.
  string country_code_;
  static jfieldID country_code_fieldID;
  /// State, if applicable.
  string state_;
  static jfieldID state_fieldID;
  /// City.
  string city_;
  static jfieldID city_fieldID;
  /// First line of the address.
  string street_line1_;
  static jfieldID street_line1_fieldID;
  /// Second line of the address.
  string street_line2_;
  static jfieldID street_line2_fieldID;
  /// Address postal code.
  string postal_code_;
  static jfieldID postal_code_fieldID;

  /**
   * Describes an address.
   */
  address();

  /**
   * Describes an address.
   *
   * \param[in] country_code_ A two-letter ISO 3166-1 alpha-2 country code.
   * \param[in] state_ State, if applicable.
   * \param[in] city_ City.
   * \param[in] street_line1_ First line of the address.
   * \param[in] street_line2_ Second line of the address.
   * \param[in] postal_code_ Address postal code.
   */
  address(string const &country_code_, string const &state_, string const &city_, string const &street_line1_, string const &street_line2_, string const &postal_code_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2043654342;

  static object_ptr<address> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class photo;

/**
 * Information about the sponsor of an advertisement.
 */
class advertisementSponsor final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// URL of the sponsor to be opened when the advertisement is clicked.
  string url_;
  static jfieldID url_fieldID;
  /// Photo of the sponsor; may be null if must not be shown.
  object_ptr<photo> photo_;
  static jfieldID photo_fieldID;
  /// Additional optional information about the sponsor to be shown along with the advertisement.
  string info_;
  static jfieldID info_fieldID;

  /**
   * Information about the sponsor of an advertisement.
   */
  advertisementSponsor();

  /**
   * Information about the sponsor of an advertisement.
   *
   * \param[in] url_ URL of the sponsor to be opened when the advertisement is clicked.
   * \param[in] photo_ Photo of the sponsor; may be null if must not be shown.
   * \param[in] info_ Additional optional information about the sponsor to be shown along with the advertisement.
   */
  advertisementSponsor(string const &url_, object_ptr<photo> &&photo_, string const &info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1388914159;

  static object_ptr<advertisementSponsor> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class starAmount;

/**
 * Contains information about an affiliate that received commission from a Telegram Star transaction.
 */
class affiliateInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The number of Telegram Stars received by the affiliate for each 1000 Telegram Stars received by the program owner.
  int32 commission_per_mille_;
  static jfieldID commission_per_mille_fieldID;
  /// Identifier of the chat which received the commission.
  int53 affiliate_chat_id_;
  static jfieldID affiliate_chat_id_fieldID;
  /// The amount of Telegram Stars that were received by the affiliate; can be negative for refunds.
  object_ptr<starAmount> star_amount_;
  static jfieldID star_amount_fieldID;

  /**
   * Contains information about an affiliate that received commission from a Telegram Star transaction.
   */
  affiliateInfo();

  /**
   * Contains information about an affiliate that received commission from a Telegram Star transaction.
   *
   * \param[in] commission_per_mille_ The number of Telegram Stars received by the affiliate for each 1000 Telegram Stars received by the program owner.
   * \param[in] affiliate_chat_id_ Identifier of the chat which received the commission.
   * \param[in] star_amount_ The amount of Telegram Stars that were received by the affiliate; can be negative for refunds.
   */
  affiliateInfo(int32 commission_per_mille_, int53 affiliate_chat_id_, object_ptr<starAmount> &&star_amount_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1312695046;

  static object_ptr<affiliateInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class affiliateProgramParameters;

class starAmount;

/**
 * Contains information about an active affiliate program.
 */
class affiliateProgramInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Parameters of the affiliate program.
  object_ptr<affiliateProgramParameters> parameters_;
  static jfieldID parameters_fieldID;
  /// Point in time (Unix timestamp) when the affiliate program will be closed; 0 if the affiliate program isn't scheduled to be closed. If positive, then the program can't be connected using connectAffiliateProgram, but active connections will work until the date.
  int32 end_date_;
  static jfieldID end_date_fieldID;
  /// The amount of daily revenue per user in Telegram Stars of the bot that created the affiliate program.
  object_ptr<starAmount> daily_revenue_per_user_amount_;
  static jfieldID daily_revenue_per_user_amount_fieldID;

  /**
   * Contains information about an active affiliate program.
   */
  affiliateProgramInfo();

  /**
   * Contains information about an active affiliate program.
   *
   * \param[in] parameters_ Parameters of the affiliate program.
   * \param[in] end_date_ Point in time (Unix timestamp) when the affiliate program will be closed; 0 if the affiliate program isn't scheduled to be closed. If positive, then the program can't be connected using connectAffiliateProgram, but active connections will work until the date.
   * \param[in] daily_revenue_per_user_amount_ The amount of daily revenue per user in Telegram Stars of the bot that created the affiliate program.
   */
  affiliateProgramInfo(object_ptr<affiliateProgramParameters> &&parameters_, int32 end_date_, object_ptr<starAmount> &&daily_revenue_per_user_amount_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1761810251;

  static object_ptr<affiliateProgramInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes parameters of an affiliate program.
 */
class affiliateProgramParameters final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The number of Telegram Stars received by the affiliate for each 1000 Telegram Stars received by the program owner; getOption(&quot;affiliate_program_commission_per_mille_min&quot;)-getOption(&quot;affiliate_program_commission_per_mille_max&quot;).
  int32 commission_per_mille_;
  static jfieldID commission_per_mille_fieldID;
  /// Number of months the program will be active; 0-36. If 0, then the program is eternal.
  int32 month_count_;
  static jfieldID month_count_fieldID;

  /**
   * Describes parameters of an affiliate program.
   */
  affiliateProgramParameters();

  /**
   * Describes parameters of an affiliate program.
   *
   * \param[in] commission_per_mille_ The number of Telegram Stars received by the affiliate for each 1000 Telegram Stars received by the program owner; getOption(&quot;affiliate_program_commission_per_mille_min&quot;)-getOption(&quot;affiliate_program_commission_per_mille_max&quot;).
   * \param[in] month_count_ Number of months the program will be active; 0-36. If 0, then the program is eternal.
   */
  affiliateProgramParameters(int32 commission_per_mille_, int32 month_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1642662996;

  static object_ptr<affiliateProgramParameters> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes the order of the found affiliate programs.
 */
class AffiliateProgramSortOrder: public Object {
 public:
  static jclass Class;

  static object_ptr<AffiliateProgramSortOrder> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The affiliate programs must be sorted by the profitability.
 */
class affiliateProgramSortOrderProfitability final : public AffiliateProgramSortOrder {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The affiliate programs must be sorted by the profitability.
   */
  affiliateProgramSortOrderProfitability();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1963282585;

  static object_ptr<AffiliateProgramSortOrder> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The affiliate programs must be sorted by creation date.
 */
class affiliateProgramSortOrderCreationDate final : public AffiliateProgramSortOrder {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The affiliate programs must be sorted by creation date.
   */
  affiliateProgramSortOrderCreationDate();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1558628083;

  static object_ptr<AffiliateProgramSortOrder> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The affiliate programs must be sorted by the expected revenue.
 */
class affiliateProgramSortOrderRevenue final : public AffiliateProgramSortOrder {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The affiliate programs must be sorted by the expected revenue.
   */
  affiliateProgramSortOrderRevenue();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1923269304;

  static object_ptr<AffiliateProgramSortOrder> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes type of affiliate for an affiliate program.
 */
class AffiliateType: public Object {
 public:
  static jclass Class;

  static object_ptr<AffiliateType> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The affiliate is the current user.
 */
class affiliateTypeCurrentUser final : public AffiliateType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The affiliate is the current user.
   */
  affiliateTypeCurrentUser();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1453785589;

  static object_ptr<AffiliateType> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The affiliate is a bot owned by the current user.
 */
class affiliateTypeBot final : public AffiliateType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// User identifier of the bot.
  int53 user_id_;
  static jfieldID user_id_fieldID;

  /**
   * The affiliate is a bot owned by the current user.
   */
  affiliateTypeBot();

  /**
   * The affiliate is a bot owned by the current user.
   *
   * \param[in] user_id_ User identifier of the bot.
   */
  explicit affiliateTypeBot(int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1032587200;

  static object_ptr<AffiliateType> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The affiliate is a channel chat where the current user has can_post_messages administrator right.
 */
class affiliateTypeChannel final : public AffiliateType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the channel chat.
  int53 chat_id_;
  static jfieldID chat_id_fieldID;

  /**
   * The affiliate is a channel chat where the current user has can_post_messages administrator right.
   */
  affiliateTypeChannel();

  /**
   * The affiliate is a channel chat where the current user has can_post_messages administrator right.
   *
   * \param[in] chat_id_ Identifier of the channel chat.
   */
  explicit affiliateTypeChannel(int53 chat_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -683939735;

  static object_ptr<AffiliateType> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class file;

/**
 * Describes an alternative re-encoded quality of a video file.
 */
class alternativeVideo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the alternative video, which is used in the HLS file.
  int64 id_;
  static jfieldID id_fieldID;
  /// Video width.
  int32 width_;
  static jfieldID width_fieldID;
  /// Video height.
  int32 height_;
  static jfieldID height_fieldID;
  /// Codec used for video file encoding, for example, &quot;h264&quot;, &quot;h265&quot;, or &quot;av1&quot;.
  string codec_;
  static jfieldID codec_fieldID;
  /// HLS file describing the video.
  object_ptr<file> hls_file_;
  static jfieldID hls_file_fieldID;
  /// File containing the video.
  object_ptr<file> video_;
  static jfieldID video_fieldID;

  /**
   * Describes an alternative re-encoded quality of a video file.
   */
  alternativeVideo();

  /**
   * Describes an alternative re-encoded quality of a video file.
   *
   * \param[in] id_ Unique identifier of the alternative video, which is used in the HLS file.
   * \param[in] width_ Video width.
   * \param[in] height_ Video height.
   * \param[in] codec_ Codec used for video file encoding, for example, &quot;h264&quot;, &quot;h265&quot;, or &quot;av1&quot;.
   * \param[in] hls_file_ HLS file describing the video.
   * \param[in] video_ File containing the video.
   */
  alternativeVideo(int64 id_, int32 width_, int32 height_, string const &codec_, object_ptr<file> &&hls_file_, object_ptr<file> &&video_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 483379470;

  static object_ptr<alternativeVideo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class file;

/**
 * Animated variant of a chat photo in MPEG4 format.
 */
class animatedChatPhoto final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Animation width and height.
  int32 length_;
  static jfieldID length_fieldID;
  /// Information about the animation file.
  object_ptr<file> file_;
  static jfieldID file_fieldID;
  /// Timestamp of the frame, used as a static chat photo.
  double main_frame_timestamp_;
  static jfieldID main_frame_timestamp_fieldID;

  /**
   * Animated variant of a chat photo in MPEG4 format.
   */
  animatedChatPhoto();

  /**
   * Animated variant of a chat photo in MPEG4 format.
   *
   * \param[in] length_ Animation width and height.
   * \param[in] file_ Information about the animation file.
   * \param[in] main_frame_timestamp_ Timestamp of the frame, used as a static chat photo.
   */
  animatedChatPhoto(int32 length_, object_ptr<file> &&file_, double main_frame_timestamp_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 191994926;

  static object_ptr<animatedChatPhoto> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class file;

class sticker;

/**
 * Describes an animated or custom representation of an emoji.
 */
class animatedEmoji final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Sticker for the emoji; may be null if yet unknown for a custom emoji. If the sticker is a custom emoji, then it can have arbitrary format.
  object_ptr<sticker> sticker_;
  static jfieldID sticker_fieldID;
  /// Expected width of the sticker, which can be used if the sticker is null.
  int32 sticker_width_;
  static jfieldID sticker_width_fieldID;
  /// Expected height of the sticker, which can be used if the sticker is null.
  int32 sticker_height_;
  static jfieldID sticker_height_fieldID;
  /// Emoji modifier fitzpatrick type; 0-6; 0 if none.
  int32 fitzpatrick_type_;
  static jfieldID fitzpatrick_type_fieldID;
  /// File containing the sound to be played when the sticker is clicked; may be null. The sound is encoded with the Opus codec, and stored inside an OGG container.
  object_ptr<file> sound_;
  static jfieldID sound_fieldID;

  /**
   * Describes an animated or custom representation of an emoji.
   */
  animatedEmoji();

  /**
   * Describes an animated or custom representation of an emoji.
   *
   * \param[in] sticker_ Sticker for the emoji; may be null if yet unknown for a custom emoji. If the sticker is a custom emoji, then it can have arbitrary format.
   * \param[in] sticker_width_ Expected width of the sticker, which can be used if the sticker is null.
   * \param[in] sticker_height_ Expected height of the sticker, which can be used if the sticker is null.
   * \param[in] fitzpatrick_type_ Emoji modifier fitzpatrick type; 0-6; 0 if none.
   * \param[in] sound_ File containing the sound to be played when the sticker is clicked; may be null. The sound is encoded with the Opus codec, and stored inside an OGG container.
   */
  animatedEmoji(object_ptr<sticker> &&sticker_, int32 sticker_width_, int32 sticker_height_, int32 fitzpatrick_type_, object_ptr<file> &&sound_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1378918079;

  static object_ptr<animatedEmoji> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class file;

class minithumbnail;

class thumbnail;

/**
 * Describes an animation file. The animation must be encoded in GIF or MPEG4 format.
 */
class animation final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Duration of the animation, in seconds; as defined by the sender.
  int32 duration_;
  static jfieldID duration_fieldID;
  /// Width of the animation.
  int32 width_;
  static jfieldID width_fieldID;
  /// Height of the animation.
  int32 height_;
  static jfieldID height_fieldID;
  /// Original name of the file; as defined by the sender.
  string file_name_;
  static jfieldID file_name_fieldID;
  /// MIME type of the file, usually &quot;image/gif&quot; or &quot;video/mp4&quot;.
  string mime_type_;
  static jfieldID mime_type_fieldID;
  /// True, if stickers were added to the animation. The list of corresponding sticker set can be received using getAttachedStickerSets.
  bool has_stickers_;
  static jfieldID has_stickers_fieldID;
  /// Animation minithumbnail; may be null.
  object_ptr<minithumbnail> minithumbnail_;
  static jfieldID minithumbnail_fieldID;
  /// Animation thumbnail in JPEG or MPEG4 format; may be null.
  object_ptr<thumbnail> thumbnail_;
  static jfieldID thumbnail_fieldID;
  /// File containing the animation.
  object_ptr<file> animation_;
  static jfieldID animation_fieldID;

  /**
   * Describes an animation file. The animation must be encoded in GIF or MPEG4 format.
   */
  animation();

  /**
   * Describes an animation file. The animation must be encoded in GIF or MPEG4 format.
   *
   * \param[in] duration_ Duration of the animation, in seconds; as defined by the sender.
   * \param[in] width_ Width of the animation.
   * \param[in] height_ Height of the animation.
   * \param[in] file_name_ Original name of the file; as defined by the sender.
   * \param[in] mime_type_ MIME type of the file, usually &quot;image/gif&quot; or &quot;video/mp4&quot;.
   * \param[in] has_stickers_ True, if stickers were added to the animation. The list of corresponding sticker set can be received using getAttachedStickerSets.
   * \param[in] minithumbnail_ Animation minithumbnail; may be null.
   * \param[in] thumbnail_ Animation thumbnail in JPEG or MPEG4 format; may be null.
   * \param[in] animation_ File containing the animation.
   */
  animation(int32 duration_, int32 width_, int32 height_, string const &file_name_, string const &mime_type_, bool has_stickers_, object_ptr<minithumbnail> &&minithumbnail_, object_ptr<thumbnail> &&thumbnail_, object_ptr<file> &&animation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -872359106;

  static object_ptr<animation> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class animation;

/**
 * Represents a list of animations.
 */
class animations final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// List of animations.
  array<object_ptr<animation>> animations_;
  static jfieldID animations_fieldID;

  /**
   * Represents a list of animations.
   */
  animations();

  /**
   * Represents a list of animations.
   *
   * \param[in] animations_ List of animations.
   */
  explicit animations(array<object_ptr<animation>> &&animations_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 344216945;

  static object_ptr<animations> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains settings for automatic moving of chats to and from the Archive chat lists.
 */
class archiveChatListSettings final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// True, if new chats from non-contacts will be automatically archived and muted. Can be set to true only if the option &quot;can_archive_and_mute_new_chats_from_unknown_users&quot; is true.
  bool archive_and_mute_new_chats_from_unknown_users_;
  static jfieldID archive_and_mute_new_chats_from_unknown_users_fieldID;
  /// True, if unmuted chats will be kept in the Archive chat list when they get a new message.
  bool keep_unmuted_chats_archived_;
  static jfieldID keep_unmuted_chats_archived_fieldID;
  /// True, if unmuted chats, that are always included or pinned in a folder, will be kept in the Archive chat list when they get a new message. Ignored if keep_unmuted_chats_archived == true.
  bool keep_chats_from_folders_archived_;
  static jfieldID keep_chats_from_folders_archived_fieldID;

  /**
   * Contains settings for automatic moving of chats to and from the Archive chat lists.
   */
  archiveChatListSettings();

  /**
   * Contains settings for automatic moving of chats to and from the Archive chat lists.
   *
   * \param[in] archive_and_mute_new_chats_from_unknown_users_ True, if new chats from non-contacts will be automatically archived and muted. Can be set to true only if the option &quot;can_archive_and_mute_new_chats_from_unknown_users&quot; is true.
   * \param[in] keep_unmuted_chats_archived_ True, if unmuted chats will be kept in the Archive chat list when they get a new message.
   * \param[in] keep_chats_from_folders_archived_ True, if unmuted chats, that are always included or pinned in a folder, will be kept in the Archive chat list when they get a new message. Ignored if keep_unmuted_chats_archived == true.
   */
  archiveChatListSettings(bool archive_and_mute_new_chats_from_unknown_users_, bool keep_unmuted_chats_archived_, bool keep_chats_from_folders_archived_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1058499236;

  static object_ptr<archiveChatListSettings> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class attachmentMenuBotColor;

class file;

/**
 * Represents a bot, which can be added to attachment or side menu.
 */
class attachmentMenuBot final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// User identifier of the bot.
  int53 bot_user_id_;
  static jfieldID bot_user_id_fieldID;
  /// True, if the bot supports opening from attachment menu in the chat with the bot.
  bool supports_self_chat_;
  static jfieldID supports_self_chat_fieldID;
  /// True, if the bot supports opening from attachment menu in private chats with ordinary users.
  bool supports_user_chats_;
  static jfieldID supports_user_chats_fieldID;
  /// True, if the bot supports opening from attachment menu in private chats with other bots.
  bool supports_bot_chats_;
  static jfieldID supports_bot_chats_fieldID;
  /// True, if the bot supports opening from attachment menu in basic group and supergroup chats.
  bool supports_group_chats_;
  static jfieldID supports_group_chats_fieldID;
  /// True, if the bot supports opening from attachment menu in channel chats.
  bool supports_channel_chats_;
  static jfieldID supports_channel_chats_fieldID;
  /// True, if the user must be asked for the permission to send messages to the bot.
  bool request_write_access_;
  static jfieldID request_write_access_fieldID;
  /// True, if the bot was explicitly added by the user. If the bot isn't added, then on the first bot launch toggleBotIsAddedToAttachmentMenu must be called and the bot must be added or removed.
  bool is_added_;
  static jfieldID is_added_fieldID;
  /// True, if the bot must be shown in the attachment menu.
  bool show_in_attachment_menu_;
  static jfieldID show_in_attachment_menu_fieldID;
  /// True, if the bot must be shown in the side menu.
  bool show_in_side_menu_;
  static jfieldID show_in_side_menu_fieldID;
  /// True, if a disclaimer, why the bot is shown in the side menu, is needed.
  bool show_disclaimer_in_side_menu_;
  static jfieldID show_disclaimer_in_side_menu_fieldID;
  /// Name for the bot in attachment menu.
  string name_;
  static jfieldID name_fieldID;
  /// Color to highlight selected name of the bot if appropriate; may be null.
  object_ptr<attachmentMenuBotColor> name_color_;
  static jfieldID name_color_fieldID;
  /// Default icon for the bot in SVG format; may be null.
  object_ptr<file> default_icon_;
  static jfieldID default_icon_fieldID;
  /// Icon for the bot in SVG format for the official iOS app; may be null.
  object_ptr<file> ios_static_icon_;
  static jfieldID ios_static_icon_fieldID;
  /// Icon for the bot in TGS format for the official iOS app; may be null.
  object_ptr<file> ios_animated_icon_;
  static jfieldID ios_animated_icon_fieldID;
  /// Icon for the bot in PNG format for the official iOS app side menu; may be null.
  object_ptr<file> ios_side_menu_icon_;
  static jfieldID ios_side_menu_icon_fieldID;
  /// Icon for the bot in TGS format for the official Android app; may be null.
  object_ptr<file> android_icon_;
  static jfieldID android_icon_fieldID;
  /// Icon for the bot in SVG format for the official Android app side menu; may be null.
  object_ptr<file> android_side_menu_icon_;
  static jfieldID android_side_menu_icon_fieldID;
  /// Icon for the bot in TGS format for the official native macOS app; may be null.
  object_ptr<file> macos_icon_;
  static jfieldID macos_icon_fieldID;
  /// Icon for the bot in PNG format for the official macOS app side menu; may be null.
  object_ptr<file> macos_side_menu_icon_;
  static jfieldID macos_side_menu_icon_fieldID;
  /// Color to highlight selected icon of the bot if appropriate; may be null.
  object_ptr<attachmentMenuBotColor> icon_color_;
  static jfieldID icon_color_fieldID;
  /// Default placeholder for opened Web Apps in SVG format; may be null.
  object_ptr<file> web_app_placeholder_;
  static jfieldID web_app_placeholder_fieldID;

  /**
   * Represents a bot, which can be added to attachment or side menu.
   */
  attachmentMenuBot();

  /**
   * Represents a bot, which can be added to attachment or side menu.
   *
   * \param[in] bot_user_id_ User identifier of the bot.
   * \param[in] supports_self_chat_ True, if the bot supports opening from attachment menu in the chat with the bot.
   * \param[in] supports_user_chats_ True, if the bot supports opening from attachment menu in private chats with ordinary users.
   * \param[in] supports_bot_chats_ True, if the bot supports opening from attachment menu in private chats with other bots.
   * \param[in] supports_group_chats_ True, if the bot supports opening from attachment menu in basic group and supergroup chats.
   * \param[in] supports_channel_chats_ True, if the bot supports opening from attachment menu in channel chats.
   * \param[in] request_write_access_ True, if the user must be asked for the permission to send messages to the bot.
   * \param[in] is_added_ True, if the bot was explicitly added by the user. If the bot isn't added, then on the first bot launch toggleBotIsAddedToAttachmentMenu must be called and the bot must be added or removed.
   * \param[in] show_in_attachment_menu_ True, if the bot must be shown in the attachment menu.
   * \param[in] show_in_side_menu_ True, if the bot must be shown in the side menu.
   * \param[in] show_disclaimer_in_side_menu_ True, if a disclaimer, why the bot is shown in the side menu, is needed.
   * \param[in] name_ Name for the bot in attachment menu.
   * \param[in] name_color_ Color to highlight selected name of the bot if appropriate; may be null.
   * \param[in] default_icon_ Default icon for the bot in SVG format; may be null.
   * \param[in] ios_static_icon_ Icon for the bot in SVG format for the official iOS app; may be null.
   * \param[in] ios_animated_icon_ Icon for the bot in TGS format for the official iOS app; may be null.
   * \param[in] ios_side_menu_icon_ Icon for the bot in PNG format for the official iOS app side menu; may be null.
   * \param[in] android_icon_ Icon for the bot in TGS format for the official Android app; may be null.
   * \param[in] android_side_menu_icon_ Icon for the bot in SVG format for the official Android app side menu; may be null.
   * \param[in] macos_icon_ Icon for the bot in TGS format for the official native macOS app; may be null.
   * \param[in] macos_side_menu_icon_ Icon for the bot in PNG format for the official macOS app side menu; may be null.
   * \param[in] icon_color_ Color to highlight selected icon of the bot if appropriate; may be null.
   * \param[in] web_app_placeholder_ Default placeholder for opened Web Apps in SVG format; may be null.
   */
  attachmentMenuBot(int53 bot_user_id_, bool supports_self_chat_, bool supports_user_chats_, bool supports_bot_chats_, bool supports_group_chats_, bool supports_channel_chats_, bool request_write_access_, bool is_added_, bool show_in_attachment_menu_, bool show_in_side_menu_, bool show_disclaimer_in_side_menu_, string const &name_, object_ptr<attachmentMenuBotColor> &&name_color_, object_ptr<file> &&default_icon_, object_ptr<file> &&ios_static_icon_, object_ptr<file> &&ios_animated_icon_, object_ptr<file> &&ios_side_menu_icon_, object_ptr<file> &&android_icon_, object_ptr<file> &&android_side_menu_icon_, object_ptr<file> &&macos_icon_, object_ptr<file> &&macos_side_menu_icon_, object_ptr<attachmentMenuBotColor> &&icon_color_, object_ptr<file> &&web_app_placeholder_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1183966273;

  static object_ptr<attachmentMenuBot> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes a color to highlight a bot added to attachment menu.
 */
class attachmentMenuBotColor final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Color in the RGB format for light themes.
  int32 light_color_;
  static jfieldID light_color_fieldID;
  /// Color in the RGB format for dark themes.
  int32 dark_color_;
  static jfieldID dark_color_fieldID;

  /**
   * Describes a color to highlight a bot added to attachment menu.
   */
  attachmentMenuBotColor();

  /**
   * Describes a color to highlight a bot added to attachment menu.
   *
   * \param[in] light_color_ Color in the RGB format for light themes.
   * \param[in] dark_color_ Color in the RGB format for dark themes.
   */
  attachmentMenuBotColor(int32 light_color_, int32 dark_color_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1680039612;

  static object_ptr<attachmentMenuBotColor> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class file;

class minithumbnail;

class thumbnail;

/**
 * Describes an audio file. Audio is usually in MP3 or M4A format.
 */
class audio final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Duration of the audio, in seconds; as defined by the sender.
  int32 duration_;
  static jfieldID duration_fieldID;
  /// Title of the audio; as defined by the sender.
  string title_;
  static jfieldID title_fieldID;
  /// Performer of the audio; as defined by the sender.
  string performer_;
  static jfieldID performer_fieldID;
  /// Original name of the file; as defined by the sender.
  string file_name_;
  static jfieldID file_name_fieldID;
  /// The MIME type of the file; as defined by the sender.
  string mime_type_;
  static jfieldID mime_type_fieldID;
  /// The minithumbnail of the album cover; may be null.
  object_ptr<minithumbnail> album_cover_minithumbnail_;
  static jfieldID album_cover_minithumbnail_fieldID;
  /// The thumbnail of the album cover in JPEG format; as defined by the sender. The full size thumbnail is expected to be extracted from the downloaded audio file; may be null.
  object_ptr<thumbnail> album_cover_thumbnail_;
  static jfieldID album_cover_thumbnail_fieldID;
  /// Album cover variants to use if the downloaded audio file contains no album cover. Provided thumbnail dimensions are approximate.
  array<object_ptr<thumbnail>> external_album_covers_;
  static jfieldID external_album_covers_fieldID;
  /// File containing the audio.
  object_ptr<file> audio_;
  static jfieldID audio_fieldID;

  /**
   * Describes an audio file. Audio is usually in MP3 or M4A format.
   */
  audio();

  /**
   * Describes an audio file. Audio is usually in MP3 or M4A format.
   *
   * \param[in] duration_ Duration of the audio, in seconds; as defined by the sender.
   * \param[in] title_ Title of the audio; as defined by the sender.
   * \param[in] performer_ Performer of the audio; as defined by the sender.
   * \param[in] file_name_ Original name of the file; as defined by the sender.
   * \param[in] mime_type_ The MIME type of the file; as defined by the sender.
   * \param[in] album_cover_minithumbnail_ The minithumbnail of the album cover; may be null.
   * \param[in] album_cover_thumbnail_ The thumbnail of the album cover in JPEG format; as defined by the sender. The full size thumbnail is expected to be extracted from the downloaded audio file; may be null.
   * \param[in] external_album_covers_ Album cover variants to use if the downloaded audio file contains no album cover. Provided thumbnail dimensions are approximate.
   * \param[in] audio_ File containing the audio.
   */
  audio(int32 duration_, string const &title_, string const &performer_, string const &file_name_, string const &mime_type_, object_ptr<minithumbnail> &&album_cover_minithumbnail_, object_ptr<thumbnail> &&album_cover_thumbnail_, array<object_ptr<thumbnail>> &&external_album_covers_, object_ptr<file> &&audio_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -166398841;

  static object_ptr<audio> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class AuthenticationCodeType;

/**
 * Information about the authentication code that was sent.
 */
class authenticationCodeInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A phone number that is being authenticated.
  string phone_number_;
  static jfieldID phone_number_fieldID;
  /// The way the code was sent to the user.
  object_ptr<AuthenticationCodeType> type_;
  static jfieldID type_fieldID;
  /// The way the next code will be sent to the user; may be null.
  object_ptr<AuthenticationCodeType> next_type_;
  static jfieldID next_type_fieldID;
  /// Timeout before the code can be re-sent, in seconds.
  int32 timeout_;
  static jfieldID timeout_fieldID;

  /**
   * Information about the authentication code that was sent.
   */
  authenticationCodeInfo();

  /**
   * Information about the authentication code that was sent.
   *
   * \param[in] phone_number_ A phone number that is being authenticated.
   * \param[in] type_ The way the code was sent to the user.
   * \param[in] next_type_ The way the next code will be sent to the user; may be null.
   * \param[in] timeout_ Timeout before the code can be re-sent, in seconds.
   */
  authenticationCodeInfo(string const &phone_number_, object_ptr<AuthenticationCodeType> &&type_, object_ptr<AuthenticationCodeType> &&next_type_, int32 timeout_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -860345416;

  static object_ptr<authenticationCodeInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class FirebaseDeviceVerificationParameters;

/**
 * This class is an abstract base class.
 * Provides information about the method by which an authentication code is delivered to the user.
 */
class AuthenticationCodeType: public Object {
 public:
  static jclass Class;

  static object_ptr<AuthenticationCodeType> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A digit-only authentication code is delivered via a private Telegram message, which can be viewed from another active session.
 */
class authenticationCodeTypeTelegramMessage final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Length of the code.
  int32 length_;
  static jfieldID length_fieldID;

  /**
   * A digit-only authentication code is delivered via a private Telegram message, which can be viewed from another active session.
   */
  authenticationCodeTypeTelegramMessage();

  /**
   * A digit-only authentication code is delivered via a private Telegram message, which can be viewed from another active session.
   *
   * \param[in] length_ Length of the code.
   */
  explicit authenticationCodeTypeTelegramMessage(int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2079628074;

  static object_ptr<AuthenticationCodeType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A digit-only authentication code is delivered via an SMS message to the specified phone number; non-official applications may not receive this type of code.
 */
class authenticationCodeTypeSms final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Length of the code.
  int32 length_;
  static jfieldID length_fieldID;

  /**
   * A digit-only authentication code is delivered via an SMS message to the specified phone number; non-official applications may not receive this type of code.
   */
  authenticationCodeTypeSms();

  /**
   * A digit-only authentication code is delivered via an SMS message to the specified phone number; non-official applications may not receive this type of code.
   *
   * \param[in] length_ Length of the code.
   */
  explicit authenticationCodeTypeSms(int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 962650760;

  static object_ptr<AuthenticationCodeType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * An authentication code is a word delivered via an SMS message to the specified phone number; non-official applications may not receive this type of code.
 */
class authenticationCodeTypeSmsWord final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The first letters of the word if known.
  string first_letter_;
  static jfieldID first_letter_fieldID;

  /**
   * An authentication code is a word delivered via an SMS message to the specified phone number; non-official applications may not receive this type of code.
   */
  authenticationCodeTypeSmsWord();

  /**
   * An authentication code is a word delivered via an SMS message to the specified phone number; non-official applications may not receive this type of code.
   *
   * \param[in] first_letter_ The first letters of the word if known.
   */
  explicit authenticationCodeTypeSmsWord(string const &first_letter_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1509540765;

  static object_ptr<AuthenticationCodeType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * An authentication code is a phrase from multiple words delivered via an SMS message to the specified phone number; non-official applications may not receive this type of code.
 */
class authenticationCodeTypeSmsPhrase final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The first word of the phrase if known.
  string first_word_;
  static jfieldID first_word_fieldID;

  /**
   * An authentication code is a phrase from multiple words delivered via an SMS message to the specified phone number; non-official applications may not receive this type of code.
   */
  authenticationCodeTypeSmsPhrase();

  /**
   * An authentication code is a phrase from multiple words delivered via an SMS message to the specified phone number; non-official applications may not receive this type of code.
   *
   * \param[in] first_word_ The first word of the phrase if known.
   */
  explicit authenticationCodeTypeSmsPhrase(string const &first_word_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 784108753;

  static object_ptr<AuthenticationCodeType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A digit-only authentication code is delivered via a phone call to the specified phone number.
 */
class authenticationCodeTypeCall final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Length of the code.
  int32 length_;
  static jfieldID length_fieldID;

  /**
   * A digit-only authentication code is delivered via a phone call to the specified phone number.
   */
  authenticationCodeTypeCall();

  /**
   * A digit-only authentication code is delivered via a phone call to the specified phone number.
   *
   * \param[in] length_ Length of the code.
   */
  explicit authenticationCodeTypeCall(int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1636265063;

  static object_ptr<AuthenticationCodeType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * An authentication code is delivered by an immediately canceled call to the specified phone number. The phone number that calls is the code that must be entered automatically.
 */
class authenticationCodeTypeFlashCall final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Pattern of the phone number from which the call will be made.
  string pattern_;
  static jfieldID pattern_fieldID;

  /**
   * An authentication code is delivered by an immediately canceled call to the specified phone number. The phone number that calls is the code that must be entered automatically.
   */
  authenticationCodeTypeFlashCall();

  /**
   * An authentication code is delivered by an immediately canceled call to the specified phone number. The phone number that calls is the code that must be entered automatically.
   *
   * \param[in] pattern_ Pattern of the phone number from which the call will be made.
   */
  explicit authenticationCodeTypeFlashCall(string const &pattern_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1395882402;

  static object_ptr<AuthenticationCodeType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * An authentication code is delivered by an immediately canceled call to the specified phone number. The last digits of the phone number that calls are the code that must be entered manually by the user.
 */
class authenticationCodeTypeMissedCall final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Prefix of the phone number from which the call will be made.
  string phone_number_prefix_;
  static jfieldID phone_number_prefix_fieldID;
  /// Number of digits in the code, excluding the prefix.
  int32 length_;
  static jfieldID length_fieldID;

  /**
   * An authentication code is delivered by an immediately canceled call to the specified phone number. The last digits of the phone number that calls are the code that must be entered manually by the user.
   */
  authenticationCodeTypeMissedCall();

  /**
   * An authentication code is delivered by an immediately canceled call to the specified phone number. The last digits of the phone number that calls are the code that must be entered manually by the user.
   *
   * \param[in] phone_number_prefix_ Prefix of the phone number from which the call will be made.
   * \param[in] length_ Number of digits in the code, excluding the prefix.
   */
  authenticationCodeTypeMissedCall(string const &phone_number_prefix_, int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 700123783;

  static object_ptr<AuthenticationCodeType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A digit-only authentication code is delivered to https://fragment.com. The user must be logged in there via a wallet owning the phone number's NFT.
 */
class authenticationCodeTypeFragment final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// URL to open to receive the code.
  string url_;
  static jfieldID url_fieldID;
  /// Length of the code.
  int32 length_;
  static jfieldID length_fieldID;

  /**
   * A digit-only authentication code is delivered to https://fragment.com. The user must be logged in there via a wallet owning the phone number's NFT.
   */
  authenticationCodeTypeFragment();

  /**
   * A digit-only authentication code is delivered to https://fragment.com. The user must be logged in there via a wallet owning the phone number's NFT.
   *
   * \param[in] url_ URL to open to receive the code.
   * \param[in] length_ Length of the code.
   */
  authenticationCodeTypeFragment(string const &url_, int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2129693491;

  static object_ptr<AuthenticationCodeType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A digit-only authentication code is delivered via Firebase Authentication to the official Android application.
 */
class authenticationCodeTypeFirebaseAndroid final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Parameters to be used for device verification.
  object_ptr<FirebaseDeviceVerificationParameters> device_verification_parameters_;
  static jfieldID device_verification_parameters_fieldID;
  /// Length of the code.
  int32 length_;
  static jfieldID length_fieldID;

  /**
   * A digit-only authentication code is delivered via Firebase Authentication to the official Android application.
   */
  authenticationCodeTypeFirebaseAndroid();

  /**
   * A digit-only authentication code is delivered via Firebase Authentication to the official Android application.
   *
   * \param[in] device_verification_parameters_ Parameters to be used for device verification.
   * \param[in] length_ Length of the code.
   */
  authenticationCodeTypeFirebaseAndroid(object_ptr<FirebaseDeviceVerificationParameters> &&device_verification_parameters_, int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1872475422;

  static object_ptr<AuthenticationCodeType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A digit-only authentication code is delivered via Firebase Authentication to the official iOS application.
 */
class authenticationCodeTypeFirebaseIos final : public AuthenticationCodeType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Receipt of successful application token validation to compare with receipt from push notification.
  string receipt_;
  static jfieldID receipt_fieldID;
  /// Time after the next authentication method is expected to be used if verification push notification isn't received, in seconds.
  int32 push_timeout_;
  static jfieldID push_timeout_fieldID;
  /// Length of the code.
  int32 length_;
  static jfieldID length_fieldID;

  /**
   * A digit-only authentication code is delivered via Firebase Authentication to the official iOS application.
   */
  authenticationCodeTypeFirebaseIos();

  /**
   * A digit-only authentication code is delivered via Firebase Authentication to the official iOS application.
   *
   * \param[in] receipt_ Receipt of successful application token validation to compare with receipt from push notification.
   * \param[in] push_timeout_ Time after the next authentication method is expected to be used if verification push notification isn't received, in seconds.
   * \param[in] length_ Length of the code.
   */
  authenticationCodeTypeFirebaseIos(string const &receipt_, int32 push_timeout_, int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -11162989;

  static object_ptr<AuthenticationCodeType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class EmailAddressResetState;

class authenticationCodeInfo;

class emailAddressAuthenticationCodeInfo;

class termsOfService;

/**
 * This class is an abstract base class.
 * Represents the current authorization state of the TDLib client.
 */
class AuthorizationState: public Object {
 public:
  static jclass Class;

  static object_ptr<AuthorizationState> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Initialization parameters are needed. Call setTdlibParameters to provide them.
 */
class authorizationStateWaitTdlibParameters final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * Initialization parameters are needed. Call setTdlibParameters to provide them.
   */
  authorizationStateWaitTdlibParameters();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 904720988;

  static object_ptr<AuthorizationState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * TDLib needs the user's phone number to authorize. Call setAuthenticationPhoneNumber to provide the phone number, or use requestQrCodeAuthentication or checkAuthenticationBotToken for other authentication options.
 */
class authorizationStateWaitPhoneNumber final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * TDLib needs the user's phone number to authorize. Call setAuthenticationPhoneNumber to provide the phone number, or use requestQrCodeAuthentication or checkAuthenticationBotToken for other authentication options.
   */
  authorizationStateWaitPhoneNumber();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 306402531;

  static object_ptr<AuthorizationState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user must buy Telegram Premium as an in-store purchase to log in. Call checkAuthenticationPremiumPurchase and then setAuthenticationPremiumPurchaseTransaction.
 */
class authorizationStateWaitPremiumPurchase final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the store product that must be bought.
  string store_product_id_;
  static jfieldID store_product_id_fieldID;

  /**
   * The user must buy Telegram Premium as an in-store purchase to log in. Call checkAuthenticationPremiumPurchase and then setAuthenticationPremiumPurchaseTransaction.
   */
  authorizationStateWaitPremiumPurchase();

  /**
   * The user must buy Telegram Premium as an in-store purchase to log in. Call checkAuthenticationPremiumPurchase and then setAuthenticationPremiumPurchaseTransaction.
   *
   * \param[in] store_product_id_ Identifier of the store product that must be bought.
   */
  explicit authorizationStateWaitPremiumPurchase(string const &store_product_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -862487743;

  static object_ptr<AuthorizationState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * TDLib needs the user's email address to authorize. Call setAuthenticationEmailAddress to provide the email address, or directly call checkAuthenticationEmailCode with Apple ID/Google ID token if allowed.
 */
class authorizationStateWaitEmailAddress final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// True, if authorization through Apple ID is allowed.
  bool allow_apple_id_;
  static jfieldID allow_apple_id_fieldID;
  /// True, if authorization through Google ID is allowed.
  bool allow_google_id_;
  static jfieldID allow_google_id_fieldID;

  /**
   * TDLib needs the user's email address to authorize. Call setAuthenticationEmailAddress to provide the email address, or directly call checkAuthenticationEmailCode with Apple ID/Google ID token if allowed.
   */
  authorizationStateWaitEmailAddress();

  /**
   * TDLib needs the user's email address to authorize. Call setAuthenticationEmailAddress to provide the email address, or directly call checkAuthenticationEmailCode with Apple ID/Google ID token if allowed.
   *
   * \param[in] allow_apple_id_ True, if authorization through Apple ID is allowed.
   * \param[in] allow_google_id_ True, if authorization through Google ID is allowed.
   */
  authorizationStateWaitEmailAddress(bool allow_apple_id_, bool allow_google_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1040478663;

  static object_ptr<AuthorizationState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * TDLib needs the user's authentication code sent to an email address to authorize. Call checkAuthenticationEmailCode to provide the code.
 */
class authorizationStateWaitEmailCode final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// True, if authorization through Apple ID is allowed.
  bool allow_apple_id_;
  static jfieldID allow_apple_id_fieldID;
  /// True, if authorization through Google ID is allowed.
  bool allow_google_id_;
  static jfieldID allow_google_id_fieldID;
  /// Information about the sent authentication code.
  object_ptr<emailAddressAuthenticationCodeInfo> code_info_;
  static jfieldID code_info_fieldID;
  /// Reset state of the email address; may be null if the email address can't be reset.
  object_ptr<EmailAddressResetState> email_address_reset_state_;
  static jfieldID email_address_reset_state_fieldID;

  /**
   * TDLib needs the user's authentication code sent to an email address to authorize. Call checkAuthenticationEmailCode to provide the code.
   */
  authorizationStateWaitEmailCode();

  /**
   * TDLib needs the user's authentication code sent to an email address to authorize. Call checkAuthenticationEmailCode to provide the code.
   *
   * \param[in] allow_apple_id_ True, if authorization through Apple ID is allowed.
   * \param[in] allow_google_id_ True, if authorization through Google ID is allowed.
   * \param[in] code_info_ Information about the sent authentication code.
   * \param[in] email_address_reset_state_ Reset state of the email address; may be null if the email address can't be reset.
   */
  authorizationStateWaitEmailCode(bool allow_apple_id_, bool allow_google_id_, object_ptr<emailAddressAuthenticationCodeInfo> &&code_info_, object_ptr<EmailAddressResetState> &&email_address_reset_state_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1868627365;

  static object_ptr<AuthorizationState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * TDLib needs the user's authentication code to authorize. Call checkAuthenticationCode to check the code.
 */
class authorizationStateWaitCode final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Information about the authorization code that was sent.
  object_ptr<authenticationCodeInfo> code_info_;
  static jfieldID code_info_fieldID;

  /**
   * TDLib needs the user's authentication code to authorize. Call checkAuthenticationCode to check the code.
   */
  authorizationStateWaitCode();

  /**
   * TDLib needs the user's authentication code to authorize. Call checkAuthenticationCode to check the code.
   *
   * \param[in] code_info_ Information about the authorization code that was sent.
   */
  explicit authorizationStateWaitCode(object_ptr<authenticationCodeInfo> &&code_info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 52643073;

  static object_ptr<AuthorizationState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user needs to confirm authorization on another logged in device by scanning a QR code with the provided link.
 */
class authorizationStateWaitOtherDeviceConfirmation final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A tg:// URL for the QR code. The link will be updated frequently.
  string link_;
  static jfieldID link_fieldID;

  /**
   * The user needs to confirm authorization on another logged in device by scanning a QR code with the provided link.
   */
  authorizationStateWaitOtherDeviceConfirmation();

  /**
   * The user needs to confirm authorization on another logged in device by scanning a QR code with the provided link.
   *
   * \param[in] link_ A tg:// URL for the QR code. The link will be updated frequently.
   */
  explicit authorizationStateWaitOtherDeviceConfirmation(string const &link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 860166378;

  static object_ptr<AuthorizationState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is unregistered and need to accept terms of service and enter their first name and last name to finish registration. Call registerUser to accept the terms of service and provide the data.
 */
class authorizationStateWaitRegistration final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Telegram terms of service.
  object_ptr<termsOfService> terms_of_service_;
  static jfieldID terms_of_service_fieldID;

  /**
   * The user is unregistered and need to accept terms of service and enter their first name and last name to finish registration. Call registerUser to accept the terms of service and provide the data.
   */
  authorizationStateWaitRegistration();

  /**
   * The user is unregistered and need to accept terms of service and enter their first name and last name to finish registration. Call registerUser to accept the terms of service and provide the data.
   *
   * \param[in] terms_of_service_ Telegram terms of service.
   */
  explicit authorizationStateWaitRegistration(object_ptr<termsOfService> &&terms_of_service_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 550350511;

  static object_ptr<AuthorizationState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user has been authorized, but needs to enter a 2-step verification password to start using the application. Call checkAuthenticationPassword to provide the password, or requestAuthenticationPasswordRecovery to recover the password, or deleteAccount to delete the account after a week.
 */
class authorizationStateWaitPassword final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Hint for the password; may be empty.
  string password_hint_;
  static jfieldID password_hint_fieldID;
  /// True, if a recovery email address has been set up.
  bool has_recovery_email_address_;
  static jfieldID has_recovery_email_address_fieldID;
  /// True, if some Telegram Passport elements were saved.
  bool has_passport_data_;
  static jfieldID has_passport_data_fieldID;
  /// Pattern of the email address to which the recovery email was sent; empty until a recovery email has been sent.
  string recovery_email_address_pattern_;
  static jfieldID recovery_email_address_pattern_fieldID;

  /**
   * The user has been authorized, but needs to enter a 2-step verification password to start using the application. Call checkAuthenticationPassword to provide the password, or requestAuthenticationPasswordRecovery to recover the password, or deleteAccount to delete the account after a week.
   */
  authorizationStateWaitPassword();

  /**
   * The user has been authorized, but needs to enter a 2-step verification password to start using the application. Call checkAuthenticationPassword to provide the password, or requestAuthenticationPasswordRecovery to recover the password, or deleteAccount to delete the account after a week.
   *
   * \param[in] password_hint_ Hint for the password; may be empty.
   * \param[in] has_recovery_email_address_ True, if a recovery email address has been set up.
   * \param[in] has_passport_data_ True, if some Telegram Passport elements were saved.
   * \param[in] recovery_email_address_pattern_ Pattern of the email address to which the recovery email was sent; empty until a recovery email has been sent.
   */
  authorizationStateWaitPassword(string const &password_hint_, bool has_recovery_email_address_, bool has_passport_data_, string const &recovery_email_address_pattern_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 112238030;

  static object_ptr<AuthorizationState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user has been successfully authorized. TDLib is now ready to answer general requests.
 */
class authorizationStateReady final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user has been successfully authorized. TDLib is now ready to answer general requests.
   */
  authorizationStateReady();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1834871737;

  static object_ptr<AuthorizationState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is currently logging out.
 */
class authorizationStateLoggingOut final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user is currently logging out.
   */
  authorizationStateLoggingOut();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 154449270;

  static object_ptr<AuthorizationState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * TDLib is closing, all subsequent queries will be answered with the error 500. Note that closing TDLib can take a while. All resources will be freed only after authorizationStateClosed has been received.
 */
class authorizationStateClosing final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * TDLib is closing, all subsequent queries will be answered with the error 500. Note that closing TDLib can take a while. All resources will be freed only after authorizationStateClosed has been received.
   */
  authorizationStateClosing();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 445855311;

  static object_ptr<AuthorizationState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * TDLib client is in its final state. All databases are closed and all resources are released. No other updates will be received after this. All queries will be responded to with error code 500. To continue working, one must create a new instance of the TDLib client.
 */
class authorizationStateClosed final : public AuthorizationState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * TDLib client is in its final state. All databases are closed and all resources are released. No other updates will be received after this. All queries will be responded to with error code 500. To continue working, one must create a new instance of the TDLib client.
   */
  authorizationStateClosed();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1526047584;

  static object_ptr<AuthorizationState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains auto-download settings.
 */
class autoDownloadSettings final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// True, if the auto-download is enabled.
  bool is_auto_download_enabled_;
  static jfieldID is_auto_download_enabled_fieldID;
  /// The maximum size of a photo file to be auto-downloaded, in bytes.
  int32 max_photo_file_size_;
  static jfieldID max_photo_file_size_fieldID;
  /// The maximum size of a video file to be auto-downloaded, in bytes.
  int53 max_video_file_size_;
  static jfieldID max_video_file_size_fieldID;
  /// The maximum size of other file types to be auto-downloaded, in bytes.
  int53 max_other_file_size_;
  static jfieldID max_other_file_size_fieldID;
  /// The maximum suggested bitrate for uploaded videos, in kbit/s.
  int32 video_upload_bitrate_;
  static jfieldID video_upload_bitrate_fieldID;
  /// True, if the beginning of video files needs to be preloaded for instant playback.
  bool preload_large_videos_;
  static jfieldID preload_large_videos_fieldID;
  /// True, if the next audio track needs to be preloaded while the user is listening to an audio file.
  bool preload_next_audio_;
  static jfieldID preload_next_audio_fieldID;
  /// True, if stories needs to be preloaded.
  bool preload_stories_;
  static jfieldID preload_stories_fieldID;
  /// True, if &quot;use less data for calls&quot; option needs to be enabled.
  bool use_less_data_for_calls_;
  static jfieldID use_less_data_for_calls_fieldID;

  /**
   * Contains auto-download settings.
   */
  autoDownloadSettings();

  /**
   * Contains auto-download settings.
   *
   * \param[in] is_auto_download_enabled_ True, if the auto-download is enabled.
   * \param[in] max_photo_file_size_ The maximum size of a photo file to be auto-downloaded, in bytes.
   * \param[in] max_video_file_size_ The maximum size of a video file to be auto-downloaded, in bytes.
   * \param[in] max_other_file_size_ The maximum size of other file types to be auto-downloaded, in bytes.
   * \param[in] video_upload_bitrate_ The maximum suggested bitrate for uploaded videos, in kbit/s.
   * \param[in] preload_large_videos_ True, if the beginning of video files needs to be preloaded for instant playback.
   * \param[in] preload_next_audio_ True, if the next audio track needs to be preloaded while the user is listening to an audio file.
   * \param[in] preload_stories_ True, if stories needs to be preloaded.
   * \param[in] use_less_data_for_calls_ True, if &quot;use less data for calls&quot; option needs to be enabled.
   */
  autoDownloadSettings(bool is_auto_download_enabled_, int32 max_photo_file_size_, int53 max_video_file_size_, int53 max_other_file_size_, int32 video_upload_bitrate_, bool preload_large_videos_, bool preload_next_audio_, bool preload_stories_, bool use_less_data_for_calls_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 991433696;

  static object_ptr<autoDownloadSettings> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class autoDownloadSettings;

/**
 * Contains auto-download settings presets for the current user.
 */
class autoDownloadSettingsPresets final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Preset with lowest settings; expected to be used by default when roaming.
  object_ptr<autoDownloadSettings> low_;
  static jfieldID low_fieldID;
  /// Preset with medium settings; expected to be used by default when using mobile data.
  object_ptr<autoDownloadSettings> medium_;
  static jfieldID medium_fieldID;
  /// Preset with highest settings; expected to be used by default when connected on Wi-Fi.
  object_ptr<autoDownloadSettings> high_;
  static jfieldID high_fieldID;

  /**
   * Contains auto-download settings presets for the current user.
   */
  autoDownloadSettingsPresets();

  /**
   * Contains auto-download settings presets for the current user.
   *
   * \param[in] low_ Preset with lowest settings; expected to be used by default when roaming.
   * \param[in] medium_ Preset with medium settings; expected to be used by default when using mobile data.
   * \param[in] high_ Preset with highest settings; expected to be used by default when connected on Wi-Fi.
   */
  autoDownloadSettingsPresets(object_ptr<autoDownloadSettings> &&low_, object_ptr<autoDownloadSettings> &&medium_, object_ptr<autoDownloadSettings> &&high_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -782099166;

  static object_ptr<autoDownloadSettingsPresets> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class autosaveSettingsException;

class scopeAutosaveSettings;

/**
 * Describes autosave settings.
 */
class autosaveSettings final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Default autosave settings for private chats.
  object_ptr<scopeAutosaveSettings> private_chat_settings_;
  static jfieldID private_chat_settings_fieldID;
  /// Default autosave settings for basic group and supergroup chats.
  object_ptr<scopeAutosaveSettings> group_settings_;
  static jfieldID group_settings_fieldID;
  /// Default autosave settings for channel chats.
  object_ptr<scopeAutosaveSettings> channel_settings_;
  static jfieldID channel_settings_fieldID;
  /// Autosave settings for specific chats.
  array<object_ptr<autosaveSettingsException>> exceptions_;
  static jfieldID exceptions_fieldID;

  /**
   * Describes autosave settings.
   */
  autosaveSettings();

  /**
   * Describes autosave settings.
   *
   * \param[in] private_chat_settings_ Default autosave settings for private chats.
   * \param[in] group_settings_ Default autosave settings for basic group and supergroup chats.
   * \param[in] channel_settings_ Default autosave settings for channel chats.
   * \param[in] exceptions_ Autosave settings for specific chats.
   */
  autosaveSettings(object_ptr<scopeAutosaveSettings> &&private_chat_settings_, object_ptr<scopeAutosaveSettings> &&group_settings_, object_ptr<scopeAutosaveSettings> &&channel_settings_, array<object_ptr<autosaveSettingsException>> &&exceptions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1629412502;

  static object_ptr<autosaveSettings> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class scopeAutosaveSettings;

/**
 * Contains autosave settings for a chat, which overrides default settings for the corresponding scope.
 */
class autosaveSettingsException final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Chat identifier.
  int53 chat_id_;
  static jfieldID chat_id_fieldID;
  /// Autosave settings for the chat.
  object_ptr<scopeAutosaveSettings> settings_;
  static jfieldID settings_fieldID;

  /**
   * Contains autosave settings for a chat, which overrides default settings for the corresponding scope.
   */
  autosaveSettingsException();

  /**
   * Contains autosave settings for a chat, which overrides default settings for the corresponding scope.
   *
   * \param[in] chat_id_ Chat identifier.
   * \param[in] settings_ Autosave settings for the chat.
   */
  autosaveSettingsException(int53 chat_id_, object_ptr<scopeAutosaveSettings> &&settings_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1483470280;

  static object_ptr<autosaveSettingsException> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes scope of autosave settings.
 */
class AutosaveSettingsScope: public Object {
 public:
  static jclass Class;

  static object_ptr<AutosaveSettingsScope> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Autosave settings applied to all private chats without chat-specific settings.
 */
class autosaveSettingsScopePrivateChats final : public AutosaveSettingsScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * Autosave settings applied to all private chats without chat-specific settings.
   */
  autosaveSettingsScopePrivateChats();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1395227007;

  static object_ptr<AutosaveSettingsScope> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Autosave settings applied to all basic group and supergroup chats without chat-specific settings.
 */
class autosaveSettingsScopeGroupChats final : public AutosaveSettingsScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * Autosave settings applied to all basic group and supergroup chats without chat-specific settings.
   */
  autosaveSettingsScopeGroupChats();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 853544526;

  static object_ptr<AutosaveSettingsScope> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Autosave settings applied to all channel chats without chat-specific settings.
 */
class autosaveSettingsScopeChannelChats final : public AutosaveSettingsScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * Autosave settings applied to all channel chats without chat-specific settings.
   */
  autosaveSettingsScopeChannelChats();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -499572783;

  static object_ptr<AutosaveSettingsScope> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Autosave settings applied to a chat.
 */
class autosaveSettingsScopeChat final : public AutosaveSettingsScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Chat identifier.
  int53 chat_id_;
  static jfieldID chat_id_fieldID;

  /**
   * Autosave settings applied to a chat.
   */
  autosaveSettingsScopeChat();

  /**
   * Autosave settings applied to a chat.
   *
   * \param[in] chat_id_ Chat identifier.
   */
  explicit autosaveSettingsScopeChat(int53 chat_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1632255255;

  static object_ptr<AutosaveSettingsScope> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class gift;

/**
 * Describes a gift that is available for purchase.
 */
class availableGift final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The gift.
  object_ptr<gift> gift_;
  static jfieldID gift_fieldID;
  /// Number of gifts that are available for resale.
  int32 resale_count_;
  static jfieldID resale_count_fieldID;
  /// The minimum price for the gifts available for resale; 0 if there are no such gifts.
  int53 min_resale_star_count_;
  static jfieldID min_resale_star_count_fieldID;
  /// The title of the upgraded gift; empty if the gift isn't available for resale.
  string title_;
  static jfieldID title_fieldID;

  /**
   * Describes a gift that is available for purchase.
   */
  availableGift();

  /**
   * Describes a gift that is available for purchase.
   *
   * \param[in] gift_ The gift.
   * \param[in] resale_count_ Number of gifts that are available for resale.
   * \param[in] min_resale_star_count_ The minimum price for the gifts available for resale; 0 if there are no such gifts.
   * \param[in] title_ The title of the upgraded gift; empty if the gift isn't available for resale.
   */
  availableGift(object_ptr<gift> &&gift_, int32 resale_count_, int53 min_resale_star_count_, string const &title_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 850530502;

  static object_ptr<availableGift> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class availableGift;

/**
 * Contains a list of gifts that can be sent to another user or channel chat.
 */
class availableGifts final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The list of gifts.
  array<object_ptr<availableGift>> gifts_;
  static jfieldID gifts_fieldID;

  /**
   * Contains a list of gifts that can be sent to another user or channel chat.
   */
  availableGifts();

  /**
   * Contains a list of gifts that can be sent to another user or channel chat.
   *
   * \param[in] gifts_ The list of gifts.
   */
  explicit availableGifts(array<object_ptr<availableGift>> &&gifts_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1450341886;

  static object_ptr<availableGifts> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class ReactionType;

/**
 * Represents an available reaction.
 */
class availableReaction final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Type of the reaction.
  object_ptr<ReactionType> type_;
  static jfieldID type_fieldID;
  /// True, if Telegram Premium is needed to send the reaction.
  bool needs_premium_;
  static jfieldID needs_premium_fieldID;

  /**
   * Represents an available reaction.
   */
  availableReaction();

  /**
   * Represents an available reaction.
   *
   * \param[in] type_ Type of the reaction.
   * \param[in] needs_premium_ True, if Telegram Premium is needed to send the reaction.
   */
  availableReaction(object_ptr<ReactionType> &&type_, bool needs_premium_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -117292153;

  static object_ptr<availableReaction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class ReactionUnavailabilityReason;

class availableReaction;

/**
 * Represents a list of reactions that can be added to a message.
 */
class availableReactions final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// List of reactions to be shown at the top.
  array<object_ptr<availableReaction>> top_reactions_;
  static jfieldID top_reactions_fieldID;
  /// List of recently used reactions.
  array<object_ptr<availableReaction>> recent_reactions_;
  static jfieldID recent_reactions_fieldID;
  /// List of popular reactions.
  array<object_ptr<availableReaction>> popular_reactions_;
  static jfieldID popular_reactions_fieldID;
  /// True, if any custom emoji reaction can be added by Telegram Premium subscribers.
  bool allow_custom_emoji_;
  static jfieldID allow_custom_emoji_fieldID;
  /// True, if the reactions will be tags and the message can be found by them.
  bool are_tags_;
  static jfieldID are_tags_fieldID;
  /// The reason why the current user can't add reactions to the message, despite some other users can; may be null if none.
  object_ptr<ReactionUnavailabilityReason> unavailability_reason_;
  static jfieldID unavailability_reason_fieldID;

  /**
   * Represents a list of reactions that can be added to a message.
   */
  availableReactions();

  /**
   * Represents a list of reactions that can be added to a message.
   *
   * \param[in] top_reactions_ List of reactions to be shown at the top.
   * \param[in] recent_reactions_ List of recently used reactions.
   * \param[in] popular_reactions_ List of popular reactions.
   * \param[in] allow_custom_emoji_ True, if any custom emoji reaction can be added by Telegram Premium subscribers.
   * \param[in] are_tags_ True, if the reactions will be tags and the message can be found by them.
   * \param[in] unavailability_reason_ The reason why the current user can't add reactions to the message, despite some other users can; may be null if none.
   */
  availableReactions(array<object_ptr<availableReaction>> &&top_reactions_, array<object_ptr<availableReaction>> &&recent_reactions_, array<object_ptr<availableReaction>> &&popular_reactions_, bool allow_custom_emoji_, bool are_tags_, object_ptr<ReactionUnavailabilityReason> &&unavailability_reason_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 912529522;

  static object_ptr<availableReactions> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class BackgroundType;

class document;

/**
 * Describes a chat background.
 */
class background final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique background identifier.
  int64 id_;
  static jfieldID id_fieldID;
  /// True, if this is one of default backgrounds.
  bool is_default_;
  static jfieldID is_default_fieldID;
  /// True, if the background is dark and is recommended to be used with dark theme.
  bool is_dark_;
  static jfieldID is_dark_fieldID;
  /// Unique background name.
  string name_;
  static jfieldID name_fieldID;
  /// Document with the background; may be null. Null only for filled and chat theme backgrounds.
  object_ptr<document> document_;
  static jfieldID document_fieldID;
  /// Type of the background.
  object_ptr<BackgroundType> type_;
  static jfieldID type_fieldID;

  /**
   * Describes a chat background.
   */
  background();

  /**
   * Describes a chat background.
   *
   * \param[in] id_ Unique background identifier.
   * \param[in] is_default_ True, if this is one of default backgrounds.
   * \param[in] is_dark_ True, if the background is dark and is recommended to be used with dark theme.
   * \param[in] name_ Unique background name.
   * \param[in] document_ Document with the background; may be null. Null only for filled and chat theme backgrounds.
   * \param[in] type_ Type of the background.
   */
  background(int64 id_, bool is_default_, bool is_dark_, string const &name_, object_ptr<document> &&document_, object_ptr<BackgroundType> &&type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -429971172;

  static object_ptr<background> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes a fill of a background.
 */
class BackgroundFill: public Object {
 public:
  static jclass Class;

  static object_ptr<BackgroundFill> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes a solid fill of a background.
 */
class backgroundFillSolid final : public BackgroundFill {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A color of the background in the RGB format.
  int32 color_;
  static jfieldID color_fieldID;

  /**
   * Describes a solid fill of a background.
   */
  backgroundFillSolid();

  /**
   * Describes a solid fill of a background.
   *
   * \param[in] color_ A color of the background in the RGB format.
   */
  explicit backgroundFillSolid(int32 color_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1010678813;

  static object_ptr<BackgroundFill> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes a gradient fill of a background.
 */
class backgroundFillGradient final : public BackgroundFill {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A top color of the background in the RGB format.
  int32 top_color_;
  static jfieldID top_color_fieldID;
  /// A bottom color of the background in the RGB format.
  int32 bottom_color_;
  static jfieldID bottom_color_fieldID;
  /// Clockwise rotation angle of the gradient, in degrees; 0-359. Must always be divisible by 45.
  int32 rotation_angle_;
  static jfieldID rotation_angle_fieldID;

  /**
   * Describes a gradient fill of a background.
   */
  backgroundFillGradient();

  /**
   * Describes a gradient fill of a background.
   *
   * \param[in] top_color_ A top color of the background in the RGB format.
   * \param[in] bottom_color_ A bottom color of the background in the RGB format.
   * \param[in] rotation_angle_ Clockwise rotation angle of the gradient, in degrees; 0-359. Must always be divisible by 45.
   */
  backgroundFillGradient(int32 top_color_, int32 bottom_color_, int32 rotation_angle_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1839206017;

  static object_ptr<BackgroundFill> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes a freeform gradient fill of a background.
 */
class backgroundFillFreeformGradient final : public BackgroundFill {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A list of 3 or 4 colors of the freeform gradient in the RGB format.
  array<int32> colors_;
  static jfieldID colors_fieldID;

  /**
   * Describes a freeform gradient fill of a background.
   */
  backgroundFillFreeformGradient();

  /**
   * Describes a freeform gradient fill of a background.
   *
   * \param[in] colors_ A list of 3 or 4 colors of the freeform gradient in the RGB format.
   */
  explicit backgroundFillFreeformGradient(array<int32> &&colors_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1145469255;

  static object_ptr<BackgroundFill> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class BackgroundFill;

/**
 * This class is an abstract base class.
 * Describes the type of background.
 */
class BackgroundType: public Object {
 public:
  static jclass Class;

  static object_ptr<BackgroundType> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A wallpaper in JPEG format.
 */
class backgroundTypeWallpaper final : public BackgroundType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// True, if the wallpaper must be downscaled to fit in 450x450 square and then box-blurred with radius 12.
  bool is_blurred_;
  static jfieldID is_blurred_fieldID;
  /// True, if the background needs to be slightly moved when device is tilted.
  bool is_moving_;
  static jfieldID is_moving_fieldID;

  /**
   * A wallpaper in JPEG format.
   */
  backgroundTypeWallpaper();

  /**
   * A wallpaper in JPEG format.
   *
   * \param[in] is_blurred_ True, if the wallpaper must be downscaled to fit in 450x450 square and then box-blurred with radius 12.
   * \param[in] is_moving_ True, if the background needs to be slightly moved when device is tilted.
   */
  backgroundTypeWallpaper(bool is_blurred_, bool is_moving_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1972128891;

  static object_ptr<BackgroundType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A PNG or TGV (gzipped subset of SVG with MIME type &quot;application/x-tgwallpattern&quot;) pattern to be combined with the background fill chosen by the user.
 */
class backgroundTypePattern final : public BackgroundType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Fill of the background.
  object_ptr<BackgroundFill> fill_;
  static jfieldID fill_fieldID;
  /// Intensity of the pattern when it is shown above the filled background; 0-100.
  int32 intensity_;
  static jfieldID intensity_fieldID;
  /// True, if the background fill must be applied only to the pattern itself. All other pixels are black in this case. For dark themes only.
  bool is_inverted_;
  static jfieldID is_inverted_fieldID;
  /// True, if the background needs to be slightly moved when device is tilted.
  bool is_moving_;
  static jfieldID is_moving_fieldID;

  /**
   * A PNG or TGV (gzipped subset of SVG with MIME type &quot;application/x-tgwallpattern&quot;) pattern to be combined with the background fill chosen by the user.
   */
  backgroundTypePattern();

  /**
   * A PNG or TGV (gzipped subset of SVG with MIME type &quot;application/x-tgwallpattern&quot;) pattern to be combined with the background fill chosen by the user.
   *
   * \param[in] fill_ Fill of the background.
   * \param[in] intensity_ Intensity of the pattern when it is shown above the filled background; 0-100.
   * \param[in] is_inverted_ True, if the background fill must be applied only to the pattern itself. All other pixels are black in this case. For dark themes only.
   * \param[in] is_moving_ True, if the background needs to be slightly moved when device is tilted.
   */
  backgroundTypePattern(object_ptr<BackgroundFill> &&fill_, int32 intensity_, bool is_inverted_, bool is_moving_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1290213117;

  static object_ptr<BackgroundType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A filled background.
 */
class backgroundTypeFill final : public BackgroundType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The background fill.
  object_ptr<BackgroundFill> fill_;
  static jfieldID fill_fieldID;

  /**
   * A filled background.
   */
  backgroundTypeFill();

  /**
   * A filled background.
   *
   * \param[in] fill_ The background fill.
   */
  explicit backgroundTypeFill(object_ptr<BackgroundFill> &&fill_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 993008684;

  static object_ptr<BackgroundType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A background from a chat theme; can be used only as a chat background in channels.
 */
class backgroundTypeChatTheme final : public BackgroundType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Name of the chat theme.
  string theme_name_;
  static jfieldID theme_name_fieldID;

  /**
   * A background from a chat theme; can be used only as a chat background in channels.
   */
  backgroundTypeChatTheme();

  /**
   * A background from a chat theme; can be used only as a chat background in channels.
   *
   * \param[in] theme_name_ Name of the chat theme.
   */
  explicit backgroundTypeChatTheme(string const &theme_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1299879762;

  static object_ptr<BackgroundType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class background;

/**
 * Contains a list of backgrounds.
 */
class backgrounds final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A list of backgrounds.
  array<object_ptr<background>> backgrounds_;
  static jfieldID backgrounds_fieldID;

  /**
   * Contains a list of backgrounds.
   */
  backgrounds();

  /**
   * Contains a list of backgrounds.
   *
   * \param[in] backgrounds_ A list of backgrounds.
   */
  explicit backgrounds(array<object_ptr<background>> &&backgrounds_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 724728704;

  static object_ptr<backgrounds> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes an action associated with a bank card number.
 */
class bankCardActionOpenUrl final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Action text.
  string text_;
  static jfieldID text_fieldID;
  /// The URL to be opened.
  string url_;
  static jfieldID url_fieldID;

  /**
   * Describes an action associated with a bank card number.
   */
  bankCardActionOpenUrl();

  /**
   * Describes an action associated with a bank card number.
   *
   * \param[in] text_ Action text.
   * \param[in] url_ The URL to be opened.
   */
  bankCardActionOpenUrl(string const &text_, string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -196454267;

  static object_ptr<bankCardActionOpenUrl> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class bankCardActionOpenUrl;

/**
 * Information about a bank card.
 */
class bankCardInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Title of the bank card description.
  string title_;
  static jfieldID title_fieldID;
  /// Actions that can be done with the bank card number.
  array<object_ptr<bankCardActionOpenUrl>> actions_;
  static jfieldID actions_fieldID;

  /**
   * Information about a bank card.
   */
  bankCardInfo();

  /**
   * Information about a bank card.
   *
   * \param[in] title_ Title of the bank card description.
   * \param[in] actions_ Actions that can be done with the bank card number.
   */
  bankCardInfo(string const &title_, array<object_ptr<bankCardActionOpenUrl>> &&actions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2116647730;

  static object_ptr<bankCardInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class ChatMemberStatus;

/**
 * Represents a basic group of 0-200 users (must be upgraded to a supergroup to accommodate more than 200 users).
 */
class basicGroup final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Group identifier.
  int53 id_;
  static jfieldID id_fieldID;
  /// Number of members in the group.
  int32 member_count_;
  static jfieldID member_count_fieldID;
  /// Status of the current user in the group.
  object_ptr<ChatMemberStatus> status_;
  static jfieldID status_fieldID;
  /// True, if the group is active.
  bool is_active_;
  static jfieldID is_active_fieldID;
  /// Identifier of the supergroup to which this group was upgraded; 0 if none.
  int53 upgraded_to_supergroup_id_;
  static jfieldID upgraded_to_supergroup_id_fieldID;

  /**
   * Represents a basic group of 0-200 users (must be upgraded to a supergroup to accommodate more than 200 users).
   */
  basicGroup();

  /**
   * Represents a basic group of 0-200 users (must be upgraded to a supergroup to accommodate more than 200 users).
   *
   * \param[in] id_ Group identifier.
   * \param[in] member_count_ Number of members in the group.
   * \param[in] status_ Status of the current user in the group.
   * \param[in] is_active_ True, if the group is active.
   * \param[in] upgraded_to_supergroup_id_ Identifier of the supergroup to which this group was upgraded; 0 if none.
   */
  basicGroup(int53 id_, int32 member_count_, object_ptr<ChatMemberStatus> &&status_, bool is_active_, int53 upgraded_to_supergroup_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -194767217;

  static object_ptr<basicGroup> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class botCommands;

class chatInviteLink;

class chatMember;

class chatPhoto;

/**
 * Contains full information about a basic group.
 */
class basicGroupFullInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Chat photo; may be null if empty or unknown. If non-null, then it is the same photo as in chat.photo.
  object_ptr<chatPhoto> photo_;
  static jfieldID photo_fieldID;
  /// Group description. Updated only after the basic group is opened.
  string description_;
  static jfieldID description_fieldID;
  /// User identifier of the creator of the group; 0 if unknown.
  int53 creator_user_id_;
  static jfieldID creator_user_id_fieldID;
  /// Group members.
  array<object_ptr<chatMember>> members_;
  static jfieldID members_fieldID;
  /// True, if non-administrators and non-bots can be hidden in responses to getSupergroupMembers and searchChatMembers for non-administrators after upgrading the basic group to a supergroup.
  bool can_hide_members_;
  static jfieldID can_hide_members_fieldID;
  /// True, if aggressive anti-spam checks can be enabled or disabled in the supergroup after upgrading the basic group to a supergroup.
  bool can_toggle_aggressive_anti_spam_;
  static jfieldID can_toggle_aggressive_anti_spam_fieldID;
  /// Primary invite link for this group; may be null. For chat administrators with can_invite_users right only. Updated only after the basic group is opened.
  object_ptr<chatInviteLink> invite_link_;
  static jfieldID invite_link_fieldID;
  /// List of commands of bots in the group.
  array<object_ptr<botCommands>> bot_commands_;
  static jfieldID bot_commands_fieldID;

  /**
   * Contains full information about a basic group.
   */
  basicGroupFullInfo();

  /**
   * Contains full information about a basic group.
   *
   * \param[in] photo_ Chat photo; may be null if empty or unknown. If non-null, then it is the same photo as in chat.photo.
   * \param[in] description_ Group description. Updated only after the basic group is opened.
   * \param[in] creator_user_id_ User identifier of the creator of the group; 0 if unknown.
   * \param[in] members_ Group members.
   * \param[in] can_hide_members_ True, if non-administrators and non-bots can be hidden in responses to getSupergroupMembers and searchChatMembers for non-administrators after upgrading the basic group to a supergroup.
   * \param[in] can_toggle_aggressive_anti_spam_ True, if aggressive anti-spam checks can be enabled or disabled in the supergroup after upgrading the basic group to a supergroup.
   * \param[in] invite_link_ Primary invite link for this group; may be null. For chat administrators with can_invite_users right only. Updated only after the basic group is opened.
   * \param[in] bot_commands_ List of commands of bots in the group.
   */
  basicGroupFullInfo(object_ptr<chatPhoto> &&photo_, string const &description_, int53 creator_user_id_, array<object_ptr<chatMember>> &&members_, bool can_hide_members_, bool can_toggle_aggressive_anti_spam_, object_ptr<chatInviteLink> &&invite_link_, array<object_ptr<botCommands>> &&bot_commands_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1879035520;

  static object_ptr<basicGroupFullInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents a birthdate of a user.
 */
class birthdate final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Day of the month; 1-31.
  int32 day_;
  static jfieldID day_fieldID;
  /// Month of the year; 1-12.
  int32 month_;
  static jfieldID month_fieldID;
  /// Birth year; 0 if unknown.
  int32 year_;
  static jfieldID year_fieldID;

  /**
   * Represents a birthdate of a user.
   */
  birthdate();

  /**
   * Represents a birthdate of a user.
   *
   * \param[in] day_ Day of the month; 1-31.
   * \param[in] month_ Month of the year; 1-12.
   * \param[in] year_ Birth year; 0 if unknown.
   */
  birthdate(int32 day_, int32 month_, int32 year_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1644064030;

  static object_ptr<birthdate> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes type of block list.
 */
class BlockList: public Object {
 public:
  static jclass Class;

  static object_ptr<BlockList> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The main block list that disallows writing messages to the current user, receiving their status and photo, viewing of stories, and some other actions.
 */
class blockListMain final : public BlockList {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The main block list that disallows writing messages to the current user, receiving their status and photo, viewing of stories, and some other actions.
   */
  blockListMain();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1352930172;

  static object_ptr<BlockList> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The block list that disallows viewing of stories of the current user.
 */
class blockListStories final : public BlockList {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The block list that disallows viewing of stories of the current user.
   */
  blockListStories();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 103323228;

  static object_ptr<BlockList> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents a command supported by a bot.
 */
class botCommand final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Text of the bot command.
  string command_;
  static jfieldID command_fieldID;
  /// Description of the bot command.
  string description_;
  static jfieldID description_fieldID;

  /**
   * Represents a command supported by a bot.
   */
  botCommand();

  /**
   * Represents a command supported by a bot.
   *
   * \param[in] command_ Text of the bot command.
   * \param[in] description_ Description of the bot command.
   */
  botCommand(string const &command_, string const &description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1032140601;

  static object_ptr<botCommand> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Represents the scope to which bot commands are relevant.
 */
class BotCommandScope: public Object {
 public:
  static jclass Class;

  static object_ptr<BotCommandScope> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A scope covering all users.
 */
class botCommandScopeDefault final : public BotCommandScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * A scope covering all users.
   */
  botCommandScopeDefault();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 795652779;

  static object_ptr<BotCommandScope> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A scope covering all private chats.
 */
class botCommandScopeAllPrivateChats final : public BotCommandScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * A scope covering all private chats.
   */
  botCommandScopeAllPrivateChats();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -344889543;

  static object_ptr<BotCommandScope> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A scope covering all group and supergroup chats.
 */
class botCommandScopeAllGroupChats final : public BotCommandScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * A scope covering all group and supergroup chats.
   */
  botCommandScopeAllGroupChats();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -981088162;

  static object_ptr<BotCommandScope> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A scope covering all group and supergroup chat administrators.
 */
class botCommandScopeAllChatAdministrators final : public BotCommandScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * A scope covering all group and supergroup chat administrators.
   */
  botCommandScopeAllChatAdministrators();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1998329169;

  static object_ptr<BotCommandScope> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A scope covering all members of a chat.
 */
class botCommandScopeChat final : public BotCommandScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Chat identifier.
  int53 chat_id_;
  static jfieldID chat_id_fieldID;

  /**
   * A scope covering all members of a chat.
   */
  botCommandScopeChat();

  /**
   * A scope covering all members of a chat.
   *
   * \param[in] chat_id_ Chat identifier.
   */
  explicit botCommandScopeChat(int53 chat_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -430234971;

  static object_ptr<BotCommandScope> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A scope covering all administrators of a chat.
 */
class botCommandScopeChatAdministrators final : public BotCommandScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Chat identifier.
  int53 chat_id_;
  static jfieldID chat_id_fieldID;

  /**
   * A scope covering all administrators of a chat.
   */
  botCommandScopeChatAdministrators();

  /**
   * A scope covering all administrators of a chat.
   *
   * \param[in] chat_id_ Chat identifier.
   */
  explicit botCommandScopeChatAdministrators(int53 chat_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1119682126;

  static object_ptr<BotCommandScope> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A scope covering a member of a chat.
 */
class botCommandScopeChatMember final : public BotCommandScope {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Chat identifier.
  int53 chat_id_;
  static jfieldID chat_id_fieldID;
  /// User identifier.
  int53 user_id_;
  static jfieldID user_id_fieldID;

  /**
   * A scope covering a member of a chat.
   */
  botCommandScopeChatMember();

  /**
   * A scope covering a member of a chat.
   *
   * \param[in] chat_id_ Chat identifier.
   * \param[in] user_id_ User identifier.
   */
  botCommandScopeChatMember(int53 chat_id_, int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -211380494;

  static object_ptr<BotCommandScope> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class botCommand;

/**
 * Contains a list of bot commands.
 */
class botCommands final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Bot's user identifier.
  int53 bot_user_id_;
  static jfieldID bot_user_id_fieldID;
  /// List of bot commands.
  array<object_ptr<botCommand>> commands_;
  static jfieldID commands_fieldID;

  /**
   * Contains a list of bot commands.
   */
  botCommands();

  /**
   * Contains a list of bot commands.
   *
   * \param[in] bot_user_id_ Bot's user identifier.
   * \param[in] commands_ List of bot commands.
   */
  botCommands(int53 bot_user_id_, array<object_ptr<botCommand>> &&commands_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1741364468;

  static object_ptr<botCommands> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class InternalLinkType;

class affiliateProgramInfo;

class animation;

class botCommand;

class botMenuButton;

class botVerificationParameters;

class chatAdministratorRights;

class photo;

/**
 * Contains information about a bot.
 */
class botInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The text that is shown on the bot's profile page and is sent together with the link when users share the bot.
  string short_description_;
  static jfieldID short_description_fieldID;
  /// The text shown in the chat with the bot if the chat is empty.
  string description_;
  static jfieldID description_fieldID;
  /// Photo shown in the chat with the bot if the chat is empty; may be null.
  object_ptr<photo> photo_;
  static jfieldID photo_fieldID;
  /// Animation shown in the chat with the bot if the chat is empty; may be null.
  object_ptr<animation> animation_;
  static jfieldID animation_fieldID;
  /// Information about a button to show instead of the bot commands menu button; may be null if ordinary bot commands menu must be shown.
  object_ptr<botMenuButton> menu_button_;
  static jfieldID menu_button_fieldID;
  /// List of the bot commands.
  array<object_ptr<botCommand>> commands_;
  static jfieldID commands_fieldID;
  /// The HTTP link to the privacy policy of the bot. If empty, then /privacy command must be used if supported by the bot. If the command isn't supported, then https://telegram.org/privacy-tpa must be opened.
  string privacy_policy_url_;
  static jfieldID privacy_policy_url_fieldID;
  /// Default administrator rights for adding the bot to basic group and supergroup chats; may be null.
  object_ptr<chatAdministratorRights> default_group_administrator_rights_;
  static jfieldID default_group_administrator_rights_fieldID;
  /// Default administrator rights for adding the bot to channels; may be null.
  object_ptr<chatAdministratorRights> default_channel_administrator_rights_;
  static jfieldID default_channel_administrator_rights_fieldID;
  /// Information about the affiliate program of the bot; may be null if none.
  object_ptr<affiliateProgramInfo> affiliate_program_;
  static jfieldID affiliate_program_fieldID;
  /// Default light background color for bot Web Apps; -1 if not specified.
  int32 web_app_background_light_color_;
  static jfieldID web_app_background_light_color_fieldID;
  /// Default dark background color for bot Web Apps; -1 if not specified.
  int32 web_app_background_dark_color_;
  static jfieldID web_app_background_dark_color_fieldID;
  /// Default light header color for bot Web Apps; -1 if not specified.
  int32 web_app_header_light_color_;
  static jfieldID web_app_header_light_color_fieldID;
  /// Default dark header color for bot Web Apps; -1 if not specified.
  int32 web_app_header_dark_color_;
  static jfieldID web_app_header_dark_color_fieldID;
  /// Parameters of the verification that can be provided by the bot; may be null if none or the current user isn't the owner of the bot.
  object_ptr<botVerificationParameters> verification_parameters_;
  static jfieldID verification_parameters_fieldID;
  /// True, if the bot's revenue statistics are available to the current user.
  bool can_get_revenue_statistics_;
  static jfieldID can_get_revenue_statistics_fieldID;
  /// True, if the bot can manage emoji status of the current user.
  bool can_manage_emoji_status_;
  static jfieldID can_manage_emoji_status_fieldID;
  /// True, if the bot has media previews.
  bool has_media_previews_;
  static jfieldID has_media_previews_fieldID;
  /// The internal link, which can be used to edit bot commands; may be null.
  object_ptr<InternalLinkType> edit_commands_link_;
  static jfieldID edit_commands_link_fieldID;
  /// The internal link, which can be used to edit bot description; may be null.
  object_ptr<InternalLinkType> edit_description_link_;
  static jfieldID edit_description_link_fieldID;
  /// The internal link, which can be used to edit the photo or animation shown in the chat with the bot if the chat is empty; may be null.
  object_ptr<InternalLinkType> edit_description_media_link_;
  static jfieldID edit_description_media_link_fieldID;
  /// The internal link, which can be used to edit bot settings; may be null.
  object_ptr<InternalLinkType> edit_settings_link_;
  static jfieldID edit_settings_link_fieldID;

  /**
   * Contains information about a bot.
   */
  botInfo();

  /**
   * Contains information about a bot.
   *
   * \param[in] short_description_ The text that is shown on the bot's profile page and is sent together with the link when users share the bot.
   * \param[in] description_ The text shown in the chat with the bot if the chat is empty.
   * \param[in] photo_ Photo shown in the chat with the bot if the chat is empty; may be null.
   * \param[in] animation_ Animation shown in the chat with the bot if the chat is empty; may be null.
   * \param[in] menu_button_ Information about a button to show instead of the bot commands menu button; may be null if ordinary bot commands menu must be shown.
   * \param[in] commands_ List of the bot commands.
   * \param[in] privacy_policy_url_ The HTTP link to the privacy policy of the bot. If empty, then /privacy command must be used if supported by the bot. If the command isn't supported, then https://telegram.org/privacy-tpa must be opened.
   * \param[in] default_group_administrator_rights_ Default administrator rights for adding the bot to basic group and supergroup chats; may be null.
   * \param[in] default_channel_administrator_rights_ Default administrator rights for adding the bot to channels; may be null.
   * \param[in] affiliate_program_ Information about the affiliate program of the bot; may be null if none.
   * \param[in] web_app_background_light_color_ Default light background color for bot Web Apps; -1 if not specified.
   * \param[in] web_app_background_dark_color_ Default dark background color for bot Web Apps; -1 if not specified.
   * \param[in] web_app_header_light_color_ Default light header color for bot Web Apps; -1 if not specified.
   * \param[in] web_app_header_dark_color_ Default dark header color for bot Web Apps; -1 if not specified.
   * \param[in] verification_parameters_ Parameters of the verification that can be provided by the bot; may be null if none or the current user isn't the owner of the bot.
   * \param[in] can_get_revenue_statistics_ True, if the bot's revenue statistics are available to the current user.
   * \param[in] can_manage_emoji_status_ True, if the bot can manage emoji status of the current user.
   * \param[in] has_media_previews_ True, if the bot has media previews.
   * \param[in] edit_commands_link_ The internal link, which can be used to edit bot commands; may be null.
   * \param[in] edit_description_link_ The internal link, which can be used to edit bot description; may be null.
   * \param[in] edit_description_media_link_ The internal link, which can be used to edit the photo or animation shown in the chat with the bot if the chat is empty; may be null.
   * \param[in] edit_settings_link_ The internal link, which can be used to edit bot settings; may be null.
   */
  botInfo(string const &short_description_, string const &description_, object_ptr<photo> &&photo_, object_ptr<animation> &&animation_, object_ptr<botMenuButton> &&menu_button_, array<object_ptr<botCommand>> &&commands_, string const &privacy_policy_url_, object_ptr<chatAdministratorRights> &&default_group_administrator_rights_, object_ptr<chatAdministratorRights> &&default_channel_administrator_rights_, object_ptr<affiliateProgramInfo> &&affiliate_program_, int32 web_app_background_light_color_, int32 web_app_background_dark_color_, int32 web_app_header_light_color_, int32 web_app_header_dark_color_, object_ptr<botVerificationParameters> &&verification_parameters_, bool can_get_revenue_statistics_, bool can_manage_emoji_status_, bool has_media_previews_, object_ptr<InternalLinkType> &&edit_commands_link_, object_ptr<InternalLinkType> &&edit_description_link_, object_ptr<InternalLinkType> &&edit_description_media_link_, object_ptr<InternalLinkType> &&edit_settings_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1771886272;

  static object_ptr<botInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class StoryContent;

/**
 * Describes media previews of a bot.
 */
class botMediaPreview final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Point in time (Unix timestamp) when the preview was added or changed last time.
  int32 date_;
  static jfieldID date_fieldID;
  /// Content of the preview.
  object_ptr<StoryContent> content_;
  static jfieldID content_fieldID;

  /**
   * Describes media previews of a bot.
   */
  botMediaPreview();

  /**
   * Describes media previews of a bot.
   *
   * \param[in] date_ Point in time (Unix timestamp) when the preview was added or changed last time.
   * \param[in] content_ Content of the preview.
   */
  botMediaPreview(int32 date_, object_ptr<StoryContent> &&content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1632264984;

  static object_ptr<botMediaPreview> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class botMediaPreview;

/**
 * Contains a list of media previews of a bot for the given language and the list of languages for which the bot has dedicated previews.
 */
class botMediaPreviewInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// List of media previews.
  array<object_ptr<botMediaPreview>> previews_;
  static jfieldID previews_fieldID;
  /// List of language codes for which the bot has dedicated previews.
  array<string> language_codes_;
  static jfieldID language_codes_fieldID;

  /**
   * Contains a list of media previews of a bot for the given language and the list of languages for which the bot has dedicated previews.
   */
  botMediaPreviewInfo();

  /**
   * Contains a list of media previews of a bot for the given language and the list of languages for which the bot has dedicated previews.
   *
   * \param[in] previews_ List of media previews.
   * \param[in] language_codes_ List of language codes for which the bot has dedicated previews.
   */
  botMediaPreviewInfo(array<object_ptr<botMediaPreview>> &&previews_, array<string> &&language_codes_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -284783012;

  static object_ptr<botMediaPreviewInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class botMediaPreview;

/**
 * Contains a list of media previews of a bot.
 */
class botMediaPreviews final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// List of media previews.
  array<object_ptr<botMediaPreview>> previews_;
  static jfieldID previews_fieldID;

  /**
   * Contains a list of media previews of a bot.
   */
  botMediaPreviews();

  /**
   * Contains a list of media previews of a bot.
   *
   * \param[in] previews_ List of media previews.
   */
  explicit botMediaPreviews(array<object_ptr<botMediaPreview>> &&previews_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1787720586;

  static object_ptr<botMediaPreviews> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes a button to be shown instead of bot commands menu button.
 */
class botMenuButton final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Text of the button.
  string text_;
  static jfieldID text_fieldID;
  /// URL of a Web App to open when the button is pressed. If the link is of the type internalLinkTypeWebApp, then it must be processed accordingly. Otherwise, the link must be passed to openWebApp.
  string url_;
  static jfieldID url_fieldID;

  /**
   * Describes a button to be shown instead of bot commands menu button.
   */
  botMenuButton();

  /**
   * Describes a button to be shown instead of bot commands menu button.
   *
   * \param[in] text_ Text of the button.
   * \param[in] url_ URL of a Web App to open when the button is pressed. If the link is of the type internalLinkTypeWebApp, then it must be processed accordingly. Otherwise, the link must be passed to openWebApp.
   */
  botMenuButton(string const &text_, string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -944407322;

  static object_ptr<botMenuButton> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class formattedText;

/**
 * Describes verification status provided by a bot.
 */
class botVerification final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the bot that provided the verification.
  int53 bot_user_id_;
  static jfieldID bot_user_id_fieldID;
  /// Identifier of the custom emoji that is used as the verification sign.
  int64 icon_custom_emoji_id_;
  static jfieldID icon_custom_emoji_id_fieldID;
  /// Custom description of verification reason set by the bot. Can contain only Mention, Hashtag, Cashtag, PhoneNumber, BankCardNumber, Url, and EmailAddress entities.
  object_ptr<formattedText> custom_description_;
  static jfieldID custom_description_fieldID;

  /**
   * Describes verification status provided by a bot.
   */
  botVerification();

  /**
   * Describes verification status provided by a bot.
   *
   * \param[in] bot_user_id_ Identifier of the bot that provided the verification.
   * \param[in] icon_custom_emoji_id_ Identifier of the custom emoji that is used as the verification sign.
   * \param[in] custom_description_ Custom description of verification reason set by the bot. Can contain only Mention, Hashtag, Cashtag, PhoneNumber, BankCardNumber, Url, and EmailAddress entities.
   */
  botVerification(int53 bot_user_id_, int64 icon_custom_emoji_id_, object_ptr<formattedText> &&custom_description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1319061774;

  static object_ptr<botVerification> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class formattedText;

/**
 * Describes parameters of verification that is provided by a bot.
 */
class botVerificationParameters final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the custom emoji that is used as the verification sign.
  int64 icon_custom_emoji_id_;
  static jfieldID icon_custom_emoji_id_fieldID;
  /// Name of the organization that provides verification.
  string organization_name_;
  static jfieldID organization_name_fieldID;
  /// Default custom description of verification reason to be used as placeholder in setMessageSenderBotVerification; may be null if none.
  object_ptr<formattedText> default_custom_description_;
  static jfieldID default_custom_description_fieldID;
  /// True, if the bot is allowed to provide custom description for verified entities.
  bool can_set_custom_description_;
  static jfieldID can_set_custom_description_fieldID;

  /**
   * Describes parameters of verification that is provided by a bot.
   */
  botVerificationParameters();

  /**
   * Describes parameters of verification that is provided by a bot.
   *
   * \param[in] icon_custom_emoji_id_ Identifier of the custom emoji that is used as the verification sign.
   * \param[in] organization_name_ Name of the organization that provides verification.
   * \param[in] default_custom_description_ Default custom description of verification reason to be used as placeholder in setMessageSenderBotVerification; may be null if none.
   * \param[in] can_set_custom_description_ True, if the bot is allowed to provide custom description for verified entities.
   */
  botVerificationParameters(int64 icon_custom_emoji_id_, string const &organization_name_, object_ptr<formattedText> &&default_custom_description_, bool can_set_custom_description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -723737249;

  static object_ptr<botVerificationParameters> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class webApp;

/**
 * This class is an abstract base class.
 * Describes a reason why a bot was allowed to write messages to the current user.
 */
class BotWriteAccessAllowReason: public Object {
 public:
  static jclass Class;

  static object_ptr<BotWriteAccessAllowReason> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user connected a website by logging in using Telegram Login Widget on it.
 */
class botWriteAccessAllowReasonConnectedWebsite final : public BotWriteAccessAllowReason {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Domain name of the connected website.
  string domain_name_;
  static jfieldID domain_name_fieldID;

  /**
   * The user connected a website by logging in using Telegram Login Widget on it.
   */
  botWriteAccessAllowReasonConnectedWebsite();

  /**
   * The user connected a website by logging in using Telegram Login Widget on it.
   *
   * \param[in] domain_name_ Domain name of the connected website.
   */
  explicit botWriteAccessAllowReasonConnectedWebsite(string const &domain_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2016325603;

  static object_ptr<BotWriteAccessAllowReason> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user added the bot to attachment or side menu using toggleBotIsAddedToAttachmentMenu.
 */
class botWriteAccessAllowReasonAddedToAttachmentMenu final : public BotWriteAccessAllowReason {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user added the bot to attachment or side menu using toggleBotIsAddedToAttachmentMenu.
   */
  botWriteAccessAllowReasonAddedToAttachmentMenu();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2104795235;

  static object_ptr<BotWriteAccessAllowReason> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user launched a Web App using getWebAppLinkUrl.
 */
class botWriteAccessAllowReasonLaunchedWebApp final : public BotWriteAccessAllowReason {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Information about the Web App.
  object_ptr<webApp> web_app_;
  static jfieldID web_app_fieldID;

  /**
   * The user launched a Web App using getWebAppLinkUrl.
   */
  botWriteAccessAllowReasonLaunchedWebApp();

  /**
   * The user launched a Web App using getWebAppLinkUrl.
   *
   * \param[in] web_app_ Information about the Web App.
   */
  explicit botWriteAccessAllowReasonLaunchedWebApp(object_ptr<webApp> &&web_app_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -240843561;

  static object_ptr<BotWriteAccessAllowReason> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user accepted bot's request to send messages with allowBotToSendMessages.
 */
class botWriteAccessAllowReasonAcceptedRequest final : public BotWriteAccessAllowReason {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user accepted bot's request to send messages with allowBotToSendMessages.
   */
  botWriteAccessAllowReasonAcceptedRequest();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1983497220;

  static object_ptr<BotWriteAccessAllowReason> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes conditions for sending of away messages by a Telegram Business account.
 */
class BusinessAwayMessageSchedule: public Object {
 public:
  static jclass Class;

  static object_ptr<BusinessAwayMessageSchedule> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Send away messages always.
 */
class businessAwayMessageScheduleAlways final : public BusinessAwayMessageSchedule {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * Send away messages always.
   */
  businessAwayMessageScheduleAlways();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -910564679;

  static object_ptr<BusinessAwayMessageSchedule> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Send away messages outside of the business opening hours.
 */
class businessAwayMessageScheduleOutsideOfOpeningHours final : public BusinessAwayMessageSchedule {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * Send away messages outside of the business opening hours.
   */
  businessAwayMessageScheduleOutsideOfOpeningHours();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -968630506;

  static object_ptr<BusinessAwayMessageSchedule> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Send away messages only in the specified time span.
 */
class businessAwayMessageScheduleCustom final : public BusinessAwayMessageSchedule {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Point in time (Unix timestamp) when the away messages will start to be sent.
  int32 start_date_;
  static jfieldID start_date_fieldID;
  /// Point in time (Unix timestamp) when the away messages will stop to be sent.
  int32 end_date_;
  static jfieldID end_date_fieldID;

  /**
   * Send away messages only in the specified time span.
   */
  businessAwayMessageScheduleCustom();

  /**
   * Send away messages only in the specified time span.
   *
   * \param[in] start_date_ Point in time (Unix timestamp) when the away messages will start to be sent.
   * \param[in] end_date_ Point in time (Unix timestamp) when the away messages will stop to be sent.
   */
  businessAwayMessageScheduleCustom(int32 start_date_, int32 end_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1967108654;

  static object_ptr<BusinessAwayMessageSchedule> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class BusinessAwayMessageSchedule;

class businessRecipients;

/**
 * Describes settings for messages that are automatically sent by a Telegram Business account when it is away.
 */
class businessAwayMessageSettings final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique quick reply shortcut identifier for the away messages.
  int32 shortcut_id_;
  static jfieldID shortcut_id_fieldID;
  /// Chosen recipients of the away messages.
  object_ptr<businessRecipients> recipients_;
  static jfieldID recipients_fieldID;
  /// Settings used to check whether the current user is away.
  object_ptr<BusinessAwayMessageSchedule> schedule_;
  static jfieldID schedule_fieldID;
  /// True, if the messages must not be sent if the account was online in the last 10 minutes.
  bool offline_only_;
  static jfieldID offline_only_fieldID;

  /**
   * Describes settings for messages that are automatically sent by a Telegram Business account when it is away.
   */
  businessAwayMessageSettings();

  /**
   * Describes settings for messages that are automatically sent by a Telegram Business account when it is away.
   *
   * \param[in] shortcut_id_ Unique quick reply shortcut identifier for the away messages.
   * \param[in] recipients_ Chosen recipients of the away messages.
   * \param[in] schedule_ Settings used to check whether the current user is away.
   * \param[in] offline_only_ True, if the messages must not be sent if the account was online in the last 10 minutes.
   */
  businessAwayMessageSettings(int32 shortcut_id_, object_ptr<businessRecipients> &&recipients_, object_ptr<BusinessAwayMessageSchedule> &&schedule_, bool offline_only_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 353084137;

  static object_ptr<businessAwayMessageSettings> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains information about a business bot that manages the chat.
 */
class businessBotManageBar final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// User identifier of the bot.
  int53 bot_user_id_;
  static jfieldID bot_user_id_fieldID;
  /// URL to be opened to manage the bot.
  string manage_url_;
  static jfieldID manage_url_fieldID;
  /// True, if the bot is paused. Use toggleBusinessConnectedBotChatIsPaused to change the value of the field.
  bool is_bot_paused_;
  static jfieldID is_bot_paused_fieldID;
  /// True, if the bot can reply.
  bool can_bot_reply_;
  static jfieldID can_bot_reply_fieldID;

  /**
   * Contains information about a business bot that manages the chat.
   */
  businessBotManageBar();

  /**
   * Contains information about a business bot that manages the chat.
   *
   * \param[in] bot_user_id_ User identifier of the bot.
   * \param[in] manage_url_ URL to be opened to manage the bot.
   * \param[in] is_bot_paused_ True, if the bot is paused. Use toggleBusinessConnectedBotChatIsPaused to change the value of the field.
   * \param[in] can_bot_reply_ True, if the bot can reply.
   */
  businessBotManageBar(int53 bot_user_id_, string const &manage_url_, bool is_bot_paused_, bool can_bot_reply_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -311399806;

  static object_ptr<businessBotManageBar> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes rights of a business bot.
 */
class businessBotRights final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// True, if the bot can send and edit messages in the private chats that had incoming messages in the last 24 hours.
  bool can_reply_;
  static jfieldID can_reply_fieldID;
  /// True, if the bot can mark incoming private messages as read.
  bool can_read_messages_;
  static jfieldID can_read_messages_fieldID;
  /// True, if the bot can delete sent messages.
  bool can_delete_sent_messages_;
  static jfieldID can_delete_sent_messages_fieldID;
  /// True, if the bot can delete any message.
  bool can_delete_all_messages_;
  static jfieldID can_delete_all_messages_fieldID;
  /// True, if the bot can edit name of the business account.
  bool can_edit_name_;
  static jfieldID can_edit_name_fieldID;
  /// True, if the bot can edit bio of the business account.
  bool can_edit_bio_;
  static jfieldID can_edit_bio_fieldID;
  /// True, if the bot can edit profile photo of the business account.
  bool can_edit_profile_photo_;
  static jfieldID can_edit_profile_photo_fieldID;
  /// True, if the bot can edit username of the business account.
  bool can_edit_username_;
  static jfieldID can_edit_username_fieldID;
  /// True, if the bot can view gifts and amount of Telegram Stars owned by the business account.
  bool can_view_gifts_and_stars_;
  static jfieldID can_view_gifts_and_stars_fieldID;
  /// True, if the bot can sell regular gifts received by the business account.
  bool can_sell_gifts_;
  static jfieldID can_sell_gifts_fieldID;
  /// True, if the bot can change gift receiving settings of the business account.
  bool can_change_gift_settings_;
  static jfieldID can_change_gift_settings_fieldID;
  /// True, if the bot can transfer and upgrade gifts received by the business account.
  bool can_transfer_and_upgrade_gifts_;
  static jfieldID can_transfer_and_upgrade_gifts_fieldID;
  /// True, if the bot can transfer Telegram Stars received by the business account to account of the bot, or use them to upgrade and transfer gifts.
  bool can_transfer_stars_;
  static jfieldID can_transfer_stars_fieldID;
  /// True, if the bot can post, edit and delete stories.
  bool can_manage_stories_;
  static jfieldID can_manage_stories_fieldID;

  /**
   * Describes rights of a business bot.
   */
  businessBotRights();

  /**
   * Describes rights of a business bot.
   *
   * \param[in] can_reply_ True, if the bot can send and edit messages in the private chats that had incoming messages in the last 24 hours.
   * \param[in] can_read_messages_ True, if the bot can mark incoming private messages as read.
   * \param[in] can_delete_sent_messages_ True, if the bot can delete sent messages.
   * \param[in] can_delete_all_messages_ True, if the bot can delete any message.
   * \param[in] can_edit_name_ True, if the bot can edit name of the business account.
   * \param[in] can_edit_bio_ True, if the bot can edit bio of the business account.
   * \param[in] can_edit_profile_photo_ True, if the bot can edit profile photo of the business account.
   * \param[in] can_edit_username_ True, if the bot can edit username of the business account.
   * \param[in] can_view_gifts_and_stars_ True, if the bot can view gifts and amount of Telegram Stars owned by the business account.
   * \param[in] can_sell_gifts_ True, if the bot can sell regular gifts received by the business account.
   * \param[in] can_change_gift_settings_ True, if the bot can change gift receiving settings of the business account.
   * \param[in] can_transfer_and_upgrade_gifts_ True, if the bot can transfer and upgrade gifts received by the business account.
   * \param[in] can_transfer_stars_ True, if the bot can transfer Telegram Stars received by the business account to account of the bot, or use them to upgrade and transfer gifts.
   * \param[in] can_manage_stories_ True, if the bot can post, edit and delete stories.
   */
  businessBotRights(bool can_reply_, bool can_read_messages_, bool can_delete_sent_messages_, bool can_delete_all_messages_, bool can_edit_name_, bool can_edit_bio_, bool can_edit_profile_photo_, bool can_edit_username_, bool can_view_gifts_and_stars_, bool can_sell_gifts_, bool can_change_gift_settings_, bool can_transfer_and_upgrade_gifts_, bool can_transfer_stars_, bool can_manage_stories_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1224839038;

  static object_ptr<businessBotRights> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class formattedText;

/**
 * Contains information about a business chat link.
 */
class businessChatLink final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The HTTPS link.
  string link_;
  static jfieldID link_fieldID;
  /// Message draft text that will be added to the input field.
  object_ptr<formattedText> text_;
  static jfieldID text_fieldID;
  /// Link title.
  string title_;
  static jfieldID title_fieldID;
  /// Number of times the link was used.
  int32 view_count_;
  static jfieldID view_count_fieldID;

  /**
   * Contains information about a business chat link.
   */
  businessChatLink();

  /**
   * Contains information about a business chat link.
   *
   * \param[in] link_ The HTTPS link.
   * \param[in] text_ Message draft text that will be added to the input field.
   * \param[in] title_ Link title.
   * \param[in] view_count_ Number of times the link was used.
   */
  businessChatLink(string const &link_, object_ptr<formattedText> &&text_, string const &title_, int32 view_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1902539901;

  static object_ptr<businessChatLink> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class formattedText;

/**
 * Contains information about a business chat link.
 */
class businessChatLinkInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the private chat that created the link.
  int53 chat_id_;
  static jfieldID chat_id_fieldID;
  /// Message draft text that must be added to the input field.
  object_ptr<formattedText> text_;
  static jfieldID text_fieldID;

  /**
   * Contains information about a business chat link.
   */
  businessChatLinkInfo();

  /**
   * Contains information about a business chat link.
   *
   * \param[in] chat_id_ Identifier of the private chat that created the link.
   * \param[in] text_ Message draft text that must be added to the input field.
   */
  businessChatLinkInfo(int53 chat_id_, object_ptr<formattedText> &&text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -864865105;

  static object_ptr<businessChatLinkInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class businessChatLink;

/**
 * Contains a list of business chat links created by the user.
 */
class businessChatLinks final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// List of links.
  array<object_ptr<businessChatLink>> links_;
  static jfieldID links_fieldID;

  /**
   * Contains a list of business chat links created by the user.
   */
  businessChatLinks();

  /**
   * Contains a list of business chat links created by the user.
   *
   * \param[in] links_ List of links.
   */
  explicit businessChatLinks(array<object_ptr<businessChatLink>> &&links_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 79067036;

  static object_ptr<businessChatLinks> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class businessBotRights;

class businessRecipients;

/**
 * Describes a bot connected to a business account.
 */
class businessConnectedBot final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// User identifier of the bot.
  int53 bot_user_id_;
  static jfieldID bot_user_id_fieldID;
  /// Private chats that will be accessible to the bot.
  object_ptr<businessRecipients> recipients_;
  static jfieldID recipients_fieldID;
  /// Rights of the bot.
  object_ptr<businessBotRights> rights_;
  static jfieldID rights_fieldID;

  /**
   * Describes a bot connected to a business account.
   */
  businessConnectedBot();

  /**
   * Describes a bot connected to a business account.
   *
   * \param[in] bot_user_id_ User identifier of the bot.
   * \param[in] recipients_ Private chats that will be accessible to the bot.
   * \param[in] rights_ Rights of the bot.
   */
  businessConnectedBot(int53 bot_user_id_, object_ptr<businessRecipients> &&recipients_, object_ptr<businessBotRights> &&rights_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1815439021;

  static object_ptr<businessConnectedBot> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class businessBotRights;

/**
 * Describes a connection of the bot with a business account.
 */
class businessConnection final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the connection.
  string id_;
  static jfieldID id_fieldID;
  /// Identifier of the business user that created the connection.
  int53 user_id_;
  static jfieldID user_id_fieldID;
  /// Chat identifier of the private chat with the user.
  int53 user_chat_id_;
  static jfieldID user_chat_id_fieldID;
  /// Point in time (Unix timestamp) when the connection was established.
  int32 date_;
  static jfieldID date_fieldID;
  /// Rights of the bot; may be null if the connection was disabled.
  object_ptr<businessBotRights> rights_;
  static jfieldID rights_fieldID;
  /// True, if the connection is enabled; false otherwise.
  bool is_enabled_;
  static jfieldID is_enabled_fieldID;

  /**
   * Describes a connection of the bot with a business account.
   */
  businessConnection();

  /**
   * Describes a connection of the bot with a business account.
   *
   * \param[in] id_ Unique identifier of the connection.
   * \param[in] user_id_ Identifier of the business user that created the connection.
   * \param[in] user_chat_id_ Chat identifier of the private chat with the user.
   * \param[in] date_ Point in time (Unix timestamp) when the connection was established.
   * \param[in] rights_ Rights of the bot; may be null if the connection was disabled.
   * \param[in] is_enabled_ True, if the connection is enabled; false otherwise.
   */
  businessConnection(string const &id_, int53 user_id_, int53 user_chat_id_, int32 date_, object_ptr<businessBotRights> &&rights_, bool is_enabled_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -995703933;

  static object_ptr<businessConnection> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes a feature available to Business user accounts.
 */
class BusinessFeature: public Object {
 public:
  static jclass Class;

  static object_ptr<BusinessFeature> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The ability to set location.
 */
class businessFeatureLocation final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The ability to set location.
   */
  businessFeatureLocation();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1064304004;

  static object_ptr<BusinessFeature> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The ability to set opening hours.
 */
class businessFeatureOpeningHours final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The ability to set opening hours.
   */
  businessFeatureOpeningHours();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 461054701;

  static object_ptr<BusinessFeature> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The ability to use quick replies.
 */
class businessFeatureQuickReplies final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The ability to use quick replies.
   */
  businessFeatureQuickReplies();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1674048894;

  static object_ptr<BusinessFeature> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The ability to set up a greeting message.
 */
class businessFeatureGreetingMessage final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The ability to set up a greeting message.
   */
  businessFeatureGreetingMessage();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1789424756;

  static object_ptr<BusinessFeature> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The ability to set up an away message.
 */
class businessFeatureAwayMessage final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The ability to set up an away message.
   */
  businessFeatureAwayMessage();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1090119901;

  static object_ptr<BusinessFeature> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The ability to create links to the business account with predefined message text.
 */
class businessFeatureAccountLinks final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The ability to create links to the business account with predefined message text.
   */
  businessFeatureAccountLinks();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1878693646;

  static object_ptr<BusinessFeature> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The ability to customize start page.
 */
class businessFeatureStartPage final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The ability to customize start page.
   */
  businessFeatureStartPage();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 401471457;

  static object_ptr<BusinessFeature> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The ability to connect a bot to the account.
 */
class businessFeatureBots final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The ability to connect a bot to the account.
   */
  businessFeatureBots();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 275084773;

  static object_ptr<BusinessFeature> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The ability to show an emoji status along with the business name.
 */
class businessFeatureEmojiStatus final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The ability to show an emoji status along with the business name.
   */
  businessFeatureEmojiStatus();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -846282523;

  static object_ptr<BusinessFeature> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The ability to display folder names for each chat in the chat list.
 */
class businessFeatureChatFolderTags final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The ability to display folder names for each chat in the chat list.
   */
  businessFeatureChatFolderTags();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -543880918;

  static object_ptr<BusinessFeature> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Allowed to use many additional features for stories.
 */
class businessFeatureUpgradedStories final : public BusinessFeature {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * Allowed to use many additional features for stories.
   */
  businessFeatureUpgradedStories();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1812245550;

  static object_ptr<BusinessFeature> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class BusinessFeature;

class animation;

/**
 * Describes a promotion animation for a Business feature.
 */
class businessFeaturePromotionAnimation final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Business feature.
  object_ptr<BusinessFeature> feature_;
  static jfieldID feature_fieldID;
  /// Promotion animation for the feature.
  object_ptr<animation> animation_;
  static jfieldID animation_fieldID;

  /**
   * Describes a promotion animation for a Business feature.
   */
  businessFeaturePromotionAnimation();

  /**
   * Describes a promotion animation for a Business feature.
   *
   * \param[in] feature_ Business feature.
   * \param[in] animation_ Promotion animation for the feature.
   */
  businessFeaturePromotionAnimation(object_ptr<BusinessFeature> &&feature_, object_ptr<animation> &&animation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2047174666;

  static object_ptr<businessFeaturePromotionAnimation> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class BusinessFeature;

/**
 * Contains information about features, available to Business user accounts.
 */
class businessFeatures final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The list of available business features.
  array<object_ptr<BusinessFeature>> features_;
  static jfieldID features_fieldID;

  /**
   * Contains information about features, available to Business user accounts.
   */
  businessFeatures();

  /**
   * Contains information about features, available to Business user accounts.
   *
   * \param[in] features_ The list of available business features.
   */
  explicit businessFeatures(array<object_ptr<BusinessFeature>> &&features_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1532468184;

  static object_ptr<businessFeatures> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class businessRecipients;

/**
 * Describes settings for greeting messages that are automatically sent by a Telegram Business account as response to incoming messages in an inactive private chat.
 */
class businessGreetingMessageSettings final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique quick reply shortcut identifier for the greeting messages.
  int32 shortcut_id_;
  static jfieldID shortcut_id_fieldID;
  /// Chosen recipients of the greeting messages.
  object_ptr<businessRecipients> recipients_;
  static jfieldID recipients_fieldID;
  /// The number of days after which a chat will be considered as inactive; currently, must be on of 7, 14, 21, or 28.
  int32 inactivity_days_;
  static jfieldID inactivity_days_fieldID;

  /**
   * Describes settings for greeting messages that are automatically sent by a Telegram Business account as response to incoming messages in an inactive private chat.
   */
  businessGreetingMessageSettings();

  /**
   * Describes settings for greeting messages that are automatically sent by a Telegram Business account as response to incoming messages in an inactive private chat.
   *
   * \param[in] shortcut_id_ Unique quick reply shortcut identifier for the greeting messages.
   * \param[in] recipients_ Chosen recipients of the greeting messages.
   * \param[in] inactivity_days_ The number of days after which a chat will be considered as inactive; currently, must be on of 7, 14, 21, or 28.
   */
  businessGreetingMessageSettings(int32 shortcut_id_, object_ptr<businessRecipients> &&recipients_, int32 inactivity_days_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1689140754;

  static object_ptr<businessGreetingMessageSettings> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class businessAwayMessageSettings;

class businessGreetingMessageSettings;

class businessLocation;

class businessOpeningHours;

class businessStartPage;

/**
 * Contains information about a Telegram Business account.
 */
class businessInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Location of the business; may be null if none.
  object_ptr<businessLocation> location_;
  static jfieldID location_fieldID;
  /// Opening hours of the business; may be null if none. The hours are guaranteed to be valid and has already been split by week days.
  object_ptr<businessOpeningHours> opening_hours_;
  static jfieldID opening_hours_fieldID;
  /// Opening hours of the business in the local time; may be null if none. The hours are guaranteed to be valid and has already been split by week days. Local time zone identifier will be empty. An updateUserFullInfo update is not triggered when value of this field changes.
  object_ptr<businessOpeningHours> local_opening_hours_;
  static jfieldID local_opening_hours_fieldID;
  /// Time left before the business will open the next time, in seconds; 0 if unknown. An updateUserFullInfo update is not triggered when value of this field changes.
  int32 next_open_in_;
  static jfieldID next_open_in_fieldID;
  /// Time left before the business will close the next time, in seconds; 0 if unknown. An updateUserFullInfo update is not triggered when value of this field changes.
  int32 next_close_in_;
  static jfieldID next_close_in_fieldID;
  /// The greeting message; may be null if none or the Business account is not of the current user.
  object_ptr<businessGreetingMessageSettings> greeting_message_settings_;
  static jfieldID greeting_message_settings_fieldID;
  /// The away message; may be null if none or the Business account is not of the current user.
  object_ptr<businessAwayMessageSettings> away_message_settings_;
  static jfieldID away_message_settings_fieldID;
  /// Information about start page of the account; may be null if none.
  object_ptr<businessStartPage> start_page_;
  static jfieldID start_page_fieldID;

  /**
   * Contains information about a Telegram Business account.
   */
  businessInfo();

  /**
   * Contains information about a Telegram Business account.
   *
   * \param[in] location_ Location of the business; may be null if none.
   * \param[in] opening_hours_ Opening hours of the business; may be null if none. The hours are guaranteed to be valid and has already been split by week days.
   * \param[in] local_opening_hours_ Opening hours of the business in the local time; may be null if none. The hours are guaranteed to be valid and has already been split by week days. Local time zone identifier will be empty. An updateUserFullInfo update is not triggered when value of this field changes.
   * \param[in] next_open_in_ Time left before the business will open the next time, in seconds; 0 if unknown. An updateUserFullInfo update is not triggered when value of this field changes.
   * \param[in] next_close_in_ Time left before the business will close the next time, in seconds; 0 if unknown. An updateUserFullInfo update is not triggered when value of this field changes.
   * \param[in] greeting_message_settings_ The greeting message; may be null if none or the Business account is not of the current user.
   * \param[in] away_message_settings_ The away message; may be null if none or the Business account is not of the current user.
   * \param[in] start_page_ Information about start page of the account; may be null if none.
   */
  businessInfo(object_ptr<businessLocation> &&location_, object_ptr<businessOpeningHours> &&opening_hours_, object_ptr<businessOpeningHours> &&local_opening_hours_, int32 next_open_in_, int32 next_close_in_, object_ptr<businessGreetingMessageSettings> &&greeting_message_settings_, object_ptr<businessAwayMessageSettings> &&away_message_settings_, object_ptr<businessStartPage> &&start_page_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1428179342;

  static object_ptr<businessInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class location;

/**
 * Represents a location of a business.
 */
class businessLocation final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The location; may be null if not specified.
  object_ptr<location> location_;
  static jfieldID location_fieldID;
  /// Location address; 1-96 characters.
  string address_;
  static jfieldID address_fieldID;

  /**
   * Represents a location of a business.
   */
  businessLocation();

  /**
   * Represents a location of a business.
   *
   * \param[in] location_ The location; may be null if not specified.
   * \param[in] address_ Location address; 1-96 characters.
   */
  businessLocation(object_ptr<location> &&location_, string const &address_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1084969126;

  static object_ptr<businessLocation> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class message;

/**
 * Describes a message from a business account as received by a bot.
 */
class businessMessage final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The message.
  object_ptr<message> message_;
  static jfieldID message_fieldID;
  /// Message that is replied by the message in the same chat; may be null if none.
  object_ptr<message> reply_to_message_;
  static jfieldID reply_to_message_fieldID;

  /**
   * Describes a message from a business account as received by a bot.
   */
  businessMessage();

  /**
   * Describes a message from a business account as received by a bot.
   *
   * \param[in] message_ The message.
   * \param[in] reply_to_message_ Message that is replied by the message in the same chat; may be null if none.
   */
  businessMessage(object_ptr<message> &&message_, object_ptr<message> &&reply_to_message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -94353850;

  static object_ptr<businessMessage> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class businessMessage;

/**
 * Contains a list of messages from a business account as received by a bot.
 */
class businessMessages final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// List of business messages.
  array<object_ptr<businessMessage>> messages_;
  static jfieldID messages_fieldID;

  /**
   * Contains a list of messages from a business account as received by a bot.
   */
  businessMessages();

  /**
   * Contains a list of messages from a business account as received by a bot.
   *
   * \param[in] messages_ List of business messages.
   */
  explicit businessMessages(array<object_ptr<businessMessage>> &&messages_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -764562473;

  static object_ptr<businessMessages> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class businessOpeningHoursInterval;

/**
 * Describes opening hours of a business.
 */
class businessOpeningHours final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique time zone identifier.
  string time_zone_id_;
  static jfieldID time_zone_id_fieldID;
  /// Intervals of the time when the business is open.
  array<object_ptr<businessOpeningHoursInterval>> opening_hours_;
  static jfieldID opening_hours_fieldID;

  /**
   * Describes opening hours of a business.
   */
  businessOpeningHours();

  /**
   * Describes opening hours of a business.
   *
   * \param[in] time_zone_id_ Unique time zone identifier.
   * \param[in] opening_hours_ Intervals of the time when the business is open.
   */
  businessOpeningHours(string const &time_zone_id_, array<object_ptr<businessOpeningHoursInterval>> &&opening_hours_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 816603700;

  static object_ptr<businessOpeningHours> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes an interval of time when the business is open.
 */
class businessOpeningHoursInterval final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The minute's sequence number in a week, starting on Monday, marking the start of the time interval during which the business is open; 0-7*24*60.
  int32 start_minute_;
  static jfieldID start_minute_fieldID;
  /// The minute's sequence number in a week, starting on Monday, marking the end of the time interval during which the business is open; 1-8*24*60.
  int32 end_minute_;
  static jfieldID end_minute_fieldID;

  /**
   * Describes an interval of time when the business is open.
   */
  businessOpeningHoursInterval();

  /**
   * Describes an interval of time when the business is open.
   *
   * \param[in] start_minute_ The minute's sequence number in a week, starting on Monday, marking the start of the time interval during which the business is open; 0-7*24*60.
   * \param[in] end_minute_ The minute's sequence number in a week, starting on Monday, marking the end of the time interval during which the business is open; 1-8*24*60.
   */
  businessOpeningHoursInterval(int32 start_minute_, int32 end_minute_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1108322732;

  static object_ptr<businessOpeningHoursInterval> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes private chats chosen for automatic interaction with a business.
 */
class businessRecipients final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifiers of selected private chats.
  array<int53> chat_ids_;
  static jfieldID chat_ids_fieldID;
  /// Identifiers of private chats that are always excluded; for businessConnectedBot only.
  array<int53> excluded_chat_ids_;
  static jfieldID excluded_chat_ids_fieldID;
  /// True, if all existing private chats are selected.
  bool select_existing_chats_;
  static jfieldID select_existing_chats_fieldID;
  /// True, if all new private chats are selected.
  bool select_new_chats_;
  static jfieldID select_new_chats_fieldID;
  /// True, if all private chats with contacts are selected.
  bool select_contacts_;
  static jfieldID select_contacts_fieldID;
  /// True, if all private chats with non-contacts are selected.
  bool select_non_contacts_;
  static jfieldID select_non_contacts_fieldID;
  /// If true, then all private chats except the selected are chosen. Otherwise, only the selected chats are chosen.
  bool exclude_selected_;
  static jfieldID exclude_selected_fieldID;

  /**
   * Describes private chats chosen for automatic interaction with a business.
   */
  businessRecipients();

  /**
   * Describes private chats chosen for automatic interaction with a business.
   *
   * \param[in] chat_ids_ Identifiers of selected private chats.
   * \param[in] excluded_chat_ids_ Identifiers of private chats that are always excluded; for businessConnectedBot only.
   * \param[in] select_existing_chats_ True, if all existing private chats are selected.
   * \param[in] select_new_chats_ True, if all new private chats are selected.
   * \param[in] select_contacts_ True, if all private chats with contacts are selected.
   * \param[in] select_non_contacts_ True, if all private chats with non-contacts are selected.
   * \param[in] exclude_selected_ If true, then all private chats except the selected are chosen. Otherwise, only the selected chats are chosen.
   */
  businessRecipients(array<int53> &&chat_ids_, array<int53> &&excluded_chat_ids_, bool select_existing_chats_, bool select_new_chats_, bool select_contacts_, bool select_non_contacts_, bool exclude_selected_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 868656909;

  static object_ptr<businessRecipients> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class sticker;

/**
 * Describes settings for a business account start page.
 */
class businessStartPage final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Title text of the start page.
  string title_;
  static jfieldID title_fieldID;
  /// Message text of the start page.
  string message_;
  static jfieldID message_fieldID;
  /// Greeting sticker of the start page; may be null if none.
  object_ptr<sticker> sticker_;
  static jfieldID sticker_fieldID;

  /**
   * Describes settings for a business account start page.
   */
  businessStartPage();

  /**
   * Describes settings for a business account start page.
   *
   * \param[in] title_ Title text of the start page.
   * \param[in] message_ Message text of the start page.
   * \param[in] sticker_ Greeting sticker of the start page; may be null if none.
   */
  businessStartPage(string const &title_, string const &message_, object_ptr<sticker> &&sticker_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1616709681;

  static object_ptr<businessStartPage> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class CallState;

/**
 * Describes a call.
 */
class call final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Call identifier, not persistent.
  int32 id_;
  static jfieldID id_fieldID;
  /// User identifier of the other call participant.
  int53 user_id_;
  static jfieldID user_id_fieldID;
  /// True, if the call is outgoing.
  bool is_outgoing_;
  static jfieldID is_outgoing_fieldID;
  /// True, if the call is a video call.
  bool is_video_;
  static jfieldID is_video_fieldID;
  /// Call state.
  object_ptr<CallState> state_;
  static jfieldID state_fieldID;

  /**
   * Describes a call.
   */
  call();

  /**
   * Describes a call.
   *
   * \param[in] id_ Call identifier, not persistent.
   * \param[in] user_id_ User identifier of the other call participant.
   * \param[in] is_outgoing_ True, if the call is outgoing.
   * \param[in] is_video_ True, if the call is a video call.
   * \param[in] state_ Call state.
   */
  call(int32 id_, int53 user_id_, bool is_outgoing_, bool is_video_, object_ptr<CallState> &&state_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 920360804;

  static object_ptr<call> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes the reason why a call was discarded.
 */
class CallDiscardReason: public Object {
 public:
  static jclass Class;

  static object_ptr<CallDiscardReason> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The call wasn't discarded, or the reason is unknown.
 */
class callDiscardReasonEmpty final : public CallDiscardReason {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The call wasn't discarded, or the reason is unknown.
   */
  callDiscardReasonEmpty();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1258917949;

  static object_ptr<CallDiscardReason> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The call was ended before the conversation started. It was canceled by the caller or missed by the other party.
 */
class callDiscardReasonMissed final : public CallDiscardReason {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The call was ended before the conversation started. It was canceled by the caller or missed by the other party.
   */
  callDiscardReasonMissed();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1680358012;

  static object_ptr<CallDiscardReason> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The call was ended before the conversation started. It was declined by the other party.
 */
class callDiscardReasonDeclined final : public CallDiscardReason {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The call was ended before the conversation started. It was declined by the other party.
   */
  callDiscardReasonDeclined();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1729926094;

  static object_ptr<CallDiscardReason> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The call was ended during the conversation because the users were disconnected.
 */
class callDiscardReasonDisconnected final : public CallDiscardReason {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The call was ended during the conversation because the users were disconnected.
   */
  callDiscardReasonDisconnected();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1342872670;

  static object_ptr<CallDiscardReason> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The call was ended because one of the parties hung up.
 */
class callDiscardReasonHungUp final : public CallDiscardReason {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The call was ended because one of the parties hung up.
   */
  callDiscardReasonHungUp();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 438216166;

  static object_ptr<CallDiscardReason> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The call was ended because it has been upgraded to a group call.
 */
class callDiscardReasonUpgradeToGroupCall final : public CallDiscardReason {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Invite link for the group call.
  string invite_link_;
  static jfieldID invite_link_fieldID;

  /**
   * The call was ended because it has been upgraded to a group call.
   */
  callDiscardReasonUpgradeToGroupCall();

  /**
   * The call was ended because it has been upgraded to a group call.
   *
   * \param[in] invite_link_ Invite link for the group call.
   */
  explicit callDiscardReasonUpgradeToGroupCall(string const &invite_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1254509319;

  static object_ptr<CallDiscardReason> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains the call identifier.
 */
class callId final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Call identifier.
  int32 id_;
  static jfieldID id_fieldID;

  /**
   * Contains the call identifier.
   */
  callId();

  /**
   * Contains the call identifier.
   *
   * \param[in] id_ Call identifier.
   */
  explicit callId(int32 id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 65717769;

  static object_ptr<callId> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes the exact type of problem with a call.
 */
class CallProblem: public Object {
 public:
  static jclass Class;

  static object_ptr<CallProblem> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user heard their own voice.
 */
class callProblemEcho final : public CallProblem {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user heard their own voice.
   */
  callProblemEcho();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 801116548;

  static object_ptr<CallProblem> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user heard background noise.
 */
class callProblemNoise final : public CallProblem {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user heard background noise.
   */
  callProblemNoise();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1053065359;

  static object_ptr<CallProblem> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The other side kept disappearing.
 */
class callProblemInterruptions final : public CallProblem {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The other side kept disappearing.
   */
  callProblemInterruptions();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1119493218;

  static object_ptr<CallProblem> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The speech was distorted.
 */
class callProblemDistortedSpeech final : public CallProblem {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The speech was distorted.
   */
  callProblemDistortedSpeech();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 379960581;

  static object_ptr<CallProblem> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user couldn't hear the other side.
 */
class callProblemSilentLocal final : public CallProblem {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user couldn't hear the other side.
   */
  callProblemSilentLocal();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 253652790;

  static object_ptr<CallProblem> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The other side couldn't hear the user.
 */
class callProblemSilentRemote final : public CallProblem {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The other side couldn't hear the user.
   */
  callProblemSilentRemote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 573634714;

  static object_ptr<CallProblem> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The call ended unexpectedly.
 */
class callProblemDropped final : public CallProblem {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The call ended unexpectedly.
   */
  callProblemDropped();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1207311487;

  static object_ptr<CallProblem> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The video was distorted.
 */
class callProblemDistortedVideo final : public CallProblem {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The video was distorted.
   */
  callProblemDistortedVideo();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 385245706;

  static object_ptr<CallProblem> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The video was pixelated.
 */
class callProblemPixelatedVideo final : public CallProblem {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The video was pixelated.
   */
  callProblemPixelatedVideo();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2115315411;

  static object_ptr<CallProblem> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Specifies the supported call protocols.
 */
class callProtocol final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// True, if UDP peer-to-peer connections are supported.
  bool udp_p2p_;
  static jfieldID udp_p2p_fieldID;
  /// True, if connection through UDP reflectors is supported.
  bool udp_reflector_;
  static jfieldID udp_reflector_fieldID;
  /// The minimum supported API layer; use 65.
  int32 min_layer_;
  static jfieldID min_layer_fieldID;
  /// The maximum supported API layer; use 92.
  int32 max_layer_;
  static jfieldID max_layer_fieldID;
  /// List of supported tgcalls versions.
  array<string> library_versions_;
  static jfieldID library_versions_fieldID;

  /**
   * Specifies the supported call protocols.
   */
  callProtocol();

  /**
   * Specifies the supported call protocols.
   *
   * \param[in] udp_p2p_ True, if UDP peer-to-peer connections are supported.
   * \param[in] udp_reflector_ True, if connection through UDP reflectors is supported.
   * \param[in] min_layer_ The minimum supported API layer; use 65.
   * \param[in] max_layer_ The maximum supported API layer; use 92.
   * \param[in] library_versions_ List of supported tgcalls versions.
   */
  callProtocol(bool udp_p2p_, bool udp_reflector_, int32 min_layer_, int32 max_layer_, array<string> &&library_versions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1075562897;

  static object_ptr<callProtocol> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class CallServerType;

/**
 * Describes a server for relaying call data.
 */
class callServer final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Server identifier.
  int64 id_;
  static jfieldID id_fieldID;
  /// Server IPv4 address.
  string ip_address_;
  static jfieldID ip_address_fieldID;
  /// Server IPv6 address.
  string ipv6_address_;
  static jfieldID ipv6_address_fieldID;
  /// Server port number.
  int32 port_;
  static jfieldID port_fieldID;
  /// Server type.
  object_ptr<CallServerType> type_;
  static jfieldID type_fieldID;

  /**
   * Describes a server for relaying call data.
   */
  callServer();

  /**
   * Describes a server for relaying call data.
   *
   * \param[in] id_ Server identifier.
   * \param[in] ip_address_ Server IPv4 address.
   * \param[in] ipv6_address_ Server IPv6 address.
   * \param[in] port_ Server port number.
   * \param[in] type_ Server type.
   */
  callServer(int64 id_, string const &ip_address_, string const &ipv6_address_, int32 port_, object_ptr<CallServerType> &&type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1865932695;

  static object_ptr<callServer> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes the type of call server.
 */
class CallServerType: public Object {
 public:
  static jclass Class;

  static object_ptr<CallServerType> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A Telegram call reflector.
 */
class callServerTypeTelegramReflector final : public CallServerType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A peer tag to be used with the reflector.
  bytes peer_tag_;
  static jfieldID peer_tag_fieldID;
  /// True, if the server uses TCP instead of UDP.
  bool is_tcp_;
  static jfieldID is_tcp_fieldID;

  /**
   * A Telegram call reflector.
   */
  callServerTypeTelegramReflector();

  /**
   * A Telegram call reflector.
   *
   * \param[in] peer_tag_ A peer tag to be used with the reflector.
   * \param[in] is_tcp_ True, if the server uses TCP instead of UDP.
   */
  callServerTypeTelegramReflector(bytes const &peer_tag_, bool is_tcp_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 850343189;

  static object_ptr<CallServerType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A WebRTC server.
 */
class callServerTypeWebrtc final : public CallServerType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Username to be used for authentication.
  string username_;
  static jfieldID username_fieldID;
  /// Authentication password.
  string password_;
  static jfieldID password_fieldID;
  /// True, if the server supports TURN.
  bool supports_turn_;
  static jfieldID supports_turn_fieldID;
  /// True, if the server supports STUN.
  bool supports_stun_;
  static jfieldID supports_stun_fieldID;

  /**
   * A WebRTC server.
   */
  callServerTypeWebrtc();

  /**
   * A WebRTC server.
   *
   * \param[in] username_ Username to be used for authentication.
   * \param[in] password_ Authentication password.
   * \param[in] supports_turn_ True, if the server supports TURN.
   * \param[in] supports_stun_ True, if the server supports STUN.
   */
  callServerTypeWebrtc(string const &username_, string const &password_, bool supports_turn_, bool supports_stun_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1250622821;

  static object_ptr<CallServerType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class CallDiscardReason;

class callProtocol;

class callServer;

class error;

/**
 * This class is an abstract base class.
 * Describes the current call state.
 */
class CallState: public Object {
 public:
  static jclass Class;

  static object_ptr<CallState> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The call is pending, waiting to be accepted by a user.
 */
class callStatePending final : public CallState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// True, if the call has already been created by the server.
  bool is_created_;
  static jfieldID is_created_fieldID;
  /// True, if the call has already been received by the other party.
  bool is_received_;
  static jfieldID is_received_fieldID;

  /**
   * The call is pending, waiting to be accepted by a user.
   */
  callStatePending();

  /**
   * The call is pending, waiting to be accepted by a user.
   *
   * \param[in] is_created_ True, if the call has already been created by the server.
   * \param[in] is_received_ True, if the call has already been received by the other party.
   */
  callStatePending(bool is_created_, bool is_received_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1073048620;

  static object_ptr<CallState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The call has been answered and encryption keys are being exchanged.
 */
class callStateExchangingKeys final : public CallState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The call has been answered and encryption keys are being exchanged.
   */
  callStateExchangingKeys();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1848149403;

  static object_ptr<CallState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The call is ready to use.
 */
class callStateReady final : public CallState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Call protocols supported by the other call participant.
  object_ptr<callProtocol> protocol_;
  static jfieldID protocol_fieldID;
  /// List of available call servers.
  array<object_ptr<callServer>> servers_;
  static jfieldID servers_fieldID;
  /// A JSON-encoded call config.
  string config_;
  static jfieldID config_fieldID;
  /// Call encryption key.
  bytes encryption_key_;
  static jfieldID encryption_key_fieldID;
  /// Encryption key fingerprint represented as 4 emoji.
  array<string> emojis_;
  static jfieldID emojis_fieldID;
  /// True, if peer-to-peer connection is allowed by users privacy settings.
  bool allow_p2p_;
  static jfieldID allow_p2p_fieldID;
  /// True, if the other party supports upgrading of the call to a group call.
  bool is_group_call_supported_;
  static jfieldID is_group_call_supported_fieldID;
  /// Custom JSON-encoded call parameters to be passed to tgcalls.
  string custom_parameters_;
  static jfieldID custom_parameters_fieldID;

  /**
   * The call is ready to use.
   */
  callStateReady();

  /**
   * The call is ready to use.
   *
   * \param[in] protocol_ Call protocols supported by the other call participant.
   * \param[in] servers_ List of available call servers.
   * \param[in] config_ A JSON-encoded call config.
   * \param[in] encryption_key_ Call encryption key.
   * \param[in] emojis_ Encryption key fingerprint represented as 4 emoji.
   * \param[in] allow_p2p_ True, if peer-to-peer connection is allowed by users privacy settings.
   * \param[in] is_group_call_supported_ True, if the other party supports upgrading of the call to a group call.
   * \param[in] custom_parameters_ Custom JSON-encoded call parameters to be passed to tgcalls.
   */
  callStateReady(object_ptr<callProtocol> &&protocol_, array<object_ptr<callServer>> &&servers_, string const &config_, bytes const &encryption_key_, array<string> &&emojis_, bool allow_p2p_, bool is_group_call_supported_, string const &custom_parameters_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -281776921;

  static object_ptr<CallState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The call is hanging up after discardCall has been called.
 */
class callStateHangingUp final : public CallState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The call is hanging up after discardCall has been called.
   */
  callStateHangingUp();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2133790038;

  static object_ptr<CallState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The call has ended successfully.
 */
class callStateDiscarded final : public CallState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The reason why the call has ended.
  object_ptr<CallDiscardReason> reason_;
  static jfieldID reason_fieldID;
  /// True, if the call rating must be sent to the server.
  bool need_rating_;
  static jfieldID need_rating_fieldID;
  /// True, if the call debug information must be sent to the server.
  bool need_debug_information_;
  static jfieldID need_debug_information_fieldID;
  /// True, if the call log must be sent to the server.
  bool need_log_;
  static jfieldID need_log_fieldID;

  /**
   * The call has ended successfully.
   */
  callStateDiscarded();

  /**
   * The call has ended successfully.
   *
   * \param[in] reason_ The reason why the call has ended.
   * \param[in] need_rating_ True, if the call rating must be sent to the server.
   * \param[in] need_debug_information_ True, if the call debug information must be sent to the server.
   * \param[in] need_log_ True, if the call log must be sent to the server.
   */
  callStateDiscarded(object_ptr<CallDiscardReason> &&reason_, bool need_rating_, bool need_debug_information_, bool need_log_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1394310213;

  static object_ptr<CallState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The call has ended with an error.
 */
class callStateError final : public CallState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Error. An error with the code 4005000 will be returned if an outgoing call is missed because of an expired timeout.
  object_ptr<error> error_;
  static jfieldID error_fieldID;

  /**
   * The call has ended with an error.
   */
  callStateError();

  /**
   * The call has ended with an error.
   *
   * \param[in] error_ Error. An error with the code 4005000 will be returned if an outgoing call is missed because of an expired timeout.
   */
  explicit callStateError(object_ptr<error> &&error_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -975215467;

  static object_ptr<CallState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains a bot's answer to a callback query.
 */
class callbackQueryAnswer final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Text of the answer.
  string text_;
  static jfieldID text_fieldID;
  /// True, if an alert must be shown to the user instead of a toast notification.
  bool show_alert_;
  static jfieldID show_alert_fieldID;
  /// URL to be opened.
  string url_;
  static jfieldID url_fieldID;

  /**
   * Contains a bot's answer to a callback query.
   */
  callbackQueryAnswer();

  /**
   * Contains a bot's answer to a callback query.
   *
   * \param[in] text_ Text of the answer.
   * \param[in] show_alert_ True, if an alert must be shown to the user instead of a toast notification.
   * \param[in] url_ URL to be opened.
   */
  callbackQueryAnswer(string const &text_, bool show_alert_, string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 360867933;

  static object_ptr<callbackQueryAnswer> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Represents a payload of a callback query.
 */
class CallbackQueryPayload: public Object {
 public:
  static jclass Class;

  static object_ptr<CallbackQueryPayload> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The payload for a general callback button.
 */
class callbackQueryPayloadData final : public CallbackQueryPayload {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Data that was attached to the callback button.
  bytes data_;
  static jfieldID data_fieldID;

  /**
   * The payload for a general callback button.
   */
  callbackQueryPayloadData();

  /**
   * The payload for a general callback button.
   *
   * \param[in] data_ Data that was attached to the callback button.
   */
  explicit callbackQueryPayloadData(bytes const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1977729946;

  static object_ptr<CallbackQueryPayload> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The payload for a callback button requiring password.
 */
class callbackQueryPayloadDataWithPassword final : public CallbackQueryPayload {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The 2-step verification password for the current user.
  string password_;
  static jfieldID password_fieldID;
  /// Data that was attached to the callback button.
  bytes data_;
  static jfieldID data_fieldID;

  /**
   * The payload for a callback button requiring password.
   */
  callbackQueryPayloadDataWithPassword();

  /**
   * The payload for a callback button requiring password.
   *
   * \param[in] password_ The 2-step verification password for the current user.
   * \param[in] data_ Data that was attached to the callback button.
   */
  callbackQueryPayloadDataWithPassword(string const &password_, bytes const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1340266738;

  static object_ptr<CallbackQueryPayload> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The payload for a game callback button.
 */
class callbackQueryPayloadGame final : public CallbackQueryPayload {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A short name of the game that was attached to the callback button.
  string game_short_name_;
  static jfieldID game_short_name_fieldID;

  /**
   * The payload for a game callback button.
   */
  callbackQueryPayloadGame();

  /**
   * The payload for a game callback button.
   *
   * \param[in] game_short_name_ A short name of the game that was attached to the callback button.
   */
  explicit callbackQueryPayloadGame(string const &game_short_name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1303571512;

  static object_ptr<CallbackQueryPayload> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Represents result of checking whether the current user can post a story on behalf of the specific chat.
 */
class CanPostStoryResult: public Object {
 public:
  static jclass Class;

  static object_ptr<CanPostStoryResult> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A story can be sent.
 */
class canPostStoryResultOk final : public CanPostStoryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Number of stories that can be posted by the user.
  int32 story_count_;
  static jfieldID story_count_fieldID;

  /**
   * A story can be sent.
   */
  canPostStoryResultOk();

  /**
   * A story can be sent.
   *
   * \param[in] story_count_ Number of stories that can be posted by the user.
   */
  explicit canPostStoryResultOk(int32 story_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2083205610;

  static object_ptr<CanPostStoryResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user must subscribe to Telegram Premium to be able to post stories.
 */
class canPostStoryResultPremiumNeeded final : public CanPostStoryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user must subscribe to Telegram Premium to be able to post stories.
   */
  canPostStoryResultPremiumNeeded();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 935130501;

  static object_ptr<CanPostStoryResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat must be boosted first by Telegram Premium subscribers to post more stories. Call getChatBoostStatus to get current boost status of the chat.
 */
class canPostStoryResultBoostNeeded final : public CanPostStoryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The chat must be boosted first by Telegram Premium subscribers to post more stories. Call getChatBoostStatus to get current boost status of the chat.
   */
  canPostStoryResultBoostNeeded();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 80246195;

  static object_ptr<CanPostStoryResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The limit for the number of active stories exceeded. The user can buy Telegram Premium, delete an active story, or wait for the oldest story to expire.
 */
class canPostStoryResultActiveStoryLimitExceeded final : public CanPostStoryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The limit for the number of active stories exceeded. The user can buy Telegram Premium, delete an active story, or wait for the oldest story to expire.
   */
  canPostStoryResultActiveStoryLimitExceeded();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1640759002;

  static object_ptr<CanPostStoryResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The weekly limit for the number of posted stories exceeded. The user needs to buy Telegram Premium or wait specified time.
 */
class canPostStoryResultWeeklyLimitExceeded final : public CanPostStoryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Time left before the user can post the next story.
  int32 retry_after_;
  static jfieldID retry_after_fieldID;

  /**
   * The weekly limit for the number of posted stories exceeded. The user needs to buy Telegram Premium or wait specified time.
   */
  canPostStoryResultWeeklyLimitExceeded();

  /**
   * The weekly limit for the number of posted stories exceeded. The user needs to buy Telegram Premium or wait specified time.
   *
   * \param[in] retry_after_ Time left before the user can post the next story.
   */
  explicit canPostStoryResultWeeklyLimitExceeded(int32 retry_after_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 552858605;

  static object_ptr<CanPostStoryResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The monthly limit for the number of posted stories exceeded. The user needs to buy Telegram Premium or wait specified time.
 */
class canPostStoryResultMonthlyLimitExceeded final : public CanPostStoryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Time left before the user can post the next story.
  int32 retry_after_;
  static jfieldID retry_after_fieldID;

  /**
   * The monthly limit for the number of posted stories exceeded. The user needs to buy Telegram Premium or wait specified time.
   */
  canPostStoryResultMonthlyLimitExceeded();

  /**
   * The monthly limit for the number of posted stories exceeded. The user needs to buy Telegram Premium or wait specified time.
   *
   * \param[in] retry_after_ Time left before the user can post the next story.
   */
  explicit canPostStoryResultMonthlyLimitExceeded(int32 retry_after_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -784208562;

  static object_ptr<CanPostStoryResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes result of canSendMessageToUser.
 */
class CanSendMessageToUserResult: public Object {
 public:
  static jclass Class;

  static object_ptr<CanSendMessageToUserResult> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user can be messaged.
 */
class canSendMessageToUserResultOk final : public CanSendMessageToUserResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user can be messaged.
   */
  canSendMessageToUserResultOk();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1530583042;

  static object_ptr<CanSendMessageToUserResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user can be messaged, but the messages are paid.
 */
class canSendMessageToUserResultUserHasPaidMessages final : public CanSendMessageToUserResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Number of Telegram Stars that must be paid by the current user for each sent message to the user.
  int53 outgoing_paid_message_star_count_;
  static jfieldID outgoing_paid_message_star_count_fieldID;

  /**
   * The user can be messaged, but the messages are paid.
   */
  canSendMessageToUserResultUserHasPaidMessages();

  /**
   * The user can be messaged, but the messages are paid.
   *
   * \param[in] outgoing_paid_message_star_count_ Number of Telegram Stars that must be paid by the current user for each sent message to the user.
   */
  explicit canSendMessageToUserResultUserHasPaidMessages(int53 outgoing_paid_message_star_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1346487602;

  static object_ptr<CanSendMessageToUserResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user can't be messaged, because they are deleted or unknown.
 */
class canSendMessageToUserResultUserIsDeleted final : public CanSendMessageToUserResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user can't be messaged, because they are deleted or unknown.
   */
  canSendMessageToUserResultUserIsDeleted();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1944639903;

  static object_ptr<CanSendMessageToUserResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user can't be messaged, because they restrict new chats with non-contacts.
 */
class canSendMessageToUserResultUserRestrictsNewChats final : public CanSendMessageToUserResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user can't be messaged, because they restrict new chats with non-contacts.
   */
  canSendMessageToUserResultUserRestrictsNewChats();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1929699797;

  static object_ptr<CanSendMessageToUserResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Represents result of checking whether the current session can be used to transfer a chat ownership to another user.
 */
class CanTransferOwnershipResult: public Object {
 public:
  static jclass Class;

  static object_ptr<CanTransferOwnershipResult> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The session can be used.
 */
class canTransferOwnershipResultOk final : public CanTransferOwnershipResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The session can be used.
   */
  canTransferOwnershipResultOk();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -89881021;

  static object_ptr<CanTransferOwnershipResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The 2-step verification needs to be enabled first.
 */
class canTransferOwnershipResultPasswordNeeded final : public CanTransferOwnershipResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The 2-step verification needs to be enabled first.
   */
  canTransferOwnershipResultPasswordNeeded();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1548372703;

  static object_ptr<CanTransferOwnershipResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The 2-step verification was enabled recently, user needs to wait.
 */
class canTransferOwnershipResultPasswordTooFresh final : public CanTransferOwnershipResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Time left before the session can be used to transfer ownership of a chat, in seconds.
  int32 retry_after_;
  static jfieldID retry_after_fieldID;

  /**
   * The 2-step verification was enabled recently, user needs to wait.
   */
  canTransferOwnershipResultPasswordTooFresh();

  /**
   * The 2-step verification was enabled recently, user needs to wait.
   *
   * \param[in] retry_after_ Time left before the session can be used to transfer ownership of a chat, in seconds.
   */
  explicit canTransferOwnershipResultPasswordTooFresh(int32 retry_after_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 811440913;

  static object_ptr<CanTransferOwnershipResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The session was created recently, user needs to wait.
 */
class canTransferOwnershipResultSessionTooFresh final : public CanTransferOwnershipResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Time left before the session can be used to transfer ownership of a chat, in seconds.
  int32 retry_after_;
  static jfieldID retry_after_fieldID;

  /**
   * The session was created recently, user needs to wait.
   */
  canTransferOwnershipResultSessionTooFresh();

  /**
   * The session was created recently, user needs to wait.
   *
   * \param[in] retry_after_ Time left before the session can be used to transfer ownership of a chat, in seconds.
   */
  explicit canTransferOwnershipResultSessionTooFresh(int32 retry_after_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 984664289;

  static object_ptr<CanTransferOwnershipResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class BlockList;

class ChatActionBar;

class ChatAvailableReactions;

class ChatList;

class ChatType;

class MessageSender;

class businessBotManageBar;

class chatBackground;

class chatJoinRequestsInfo;

class chatNotificationSettings;

class chatPermissions;

class chatPhotoInfo;

class chatPosition;

class draftMessage;

class emojiStatus;

class message;

class videoChat;

/**
 * A chat. (Can be a private chat, basic group, supergroup, or secret chat.)
 */
class chat final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Chat unique identifier.
  int53 id_;
  static jfieldID id_fieldID;
  /// Type of the chat.
  object_ptr<ChatType> type_;
  static jfieldID type_fieldID;
  /// Chat title.
  string title_;
  static jfieldID title_fieldID;
  /// Chat photo; may be null.
  object_ptr<chatPhotoInfo> photo_;
  static jfieldID photo_fieldID;
  /// Identifier of the accent color for message sender name, and backgrounds of chat photo, reply header, and link preview.
  int32 accent_color_id_;
  static jfieldID accent_color_id_fieldID;
  /// Identifier of a custom emoji to be shown on the reply header and link preview background for messages sent by the chat; 0 if none.
  int64 background_custom_emoji_id_;
  static jfieldID background_custom_emoji_id_fieldID;
  /// Identifier of the profile accent color for the chat's profile; -1 if none.
  int32 profile_accent_color_id_;
  static jfieldID profile_accent_color_id_fieldID;
  /// Identifier of a custom emoji to be shown on the background of the chat's profile; 0 if none.
  int64 profile_background_custom_emoji_id_;
  static jfieldID profile_background_custom_emoji_id_fieldID;
  /// Actions that non-administrator chat members are allowed to take in the chat.
  object_ptr<chatPermissions> permissions_;
  static jfieldID permissions_fieldID;
  /// Last message in the chat; may be null if none or unknown.
  object_ptr<message> last_message_;
  static jfieldID last_message_fieldID;
  /// Positions of the chat in chat lists.
  array<object_ptr<chatPosition>> positions_;
  static jfieldID positions_fieldID;
  /// Chat lists to which the chat belongs. A chat can have a non-zero position in a chat list even if it doesn't belong to the chat list and have no position in a chat list even if it belongs to the chat list.
  array<object_ptr<ChatList>> chat_lists_;
  static jfieldID chat_lists_fieldID;
  /// Identifier of a user or chat that is selected to send messages in the chat; may be null if the user can't change message sender.
  object_ptr<MessageSender> message_sender_id_;
  static jfieldID message_sender_id_fieldID;
  /// Block list to which the chat is added; may be null if none.
  object_ptr<BlockList> block_list_;
  static jfieldID block_list_fieldID;
  /// True, if chat content can't be saved locally, forwarded, or copied.
  bool has_protected_content_;
  static jfieldID has_protected_content_fieldID;
  /// True, if translation of all messages in the chat must be suggested to the user.
  bool is_translatable_;
  static jfieldID is_translatable_fieldID;
  /// True, if the chat is marked as unread.
  bool is_marked_as_unread_;
  static jfieldID is_marked_as_unread_fieldID;
  /// True, if the chat is a forum supergroup that must be shown in the &quot;View as topics&quot; mode, or Saved Messages chat that must be shown in the &quot;View as chats&quot;.
  bool view_as_topics_;
  static jfieldID view_as_topics_fieldID;
  /// True, if the chat has scheduled messages.
  bool has_scheduled_messages_;
  static jfieldID has_scheduled_messages_fieldID;
  /// True, if the chat messages can be deleted only for the current user while other users will continue to see the messages.
  bool can_be_deleted_only_for_self_;
  static jfieldID can_be_deleted_only_for_self_fieldID;
  /// True, if the chat messages can be deleted for all users.
  bool can_be_deleted_for_all_users_;
  static jfieldID can_be_deleted_for_all_users_fieldID;
  /// True, if the chat can be reported to Telegram moderators through reportChat or reportChatPhoto.
  bool can_be_reported_;
  static jfieldID can_be_reported_fieldID;
  /// Default value of the disable_notification parameter, used when a message is sent to the chat.
  bool default_disable_notification_;
  static jfieldID default_disable_notification_fieldID;
  /// Number of unread messages in the chat.
  int32 unread_count_;
  static jfieldID unread_count_fieldID;
  /// Identifier of the last read incoming message.
  int53 last_read_inbox_message_id_;
  static jfieldID last_read_inbox_message_id_fieldID;
  /// Identifier of the last read outgoing message.
  int53 last_read_outbox_message_id_;
  static jfieldID last_read_outbox_message_id_fieldID;
  /// Number of unread messages with a mention/reply in the chat.
  int32 unread_mention_count_;
  static jfieldID unread_mention_count_fieldID;
  /// Number of messages with unread reactions in the chat.
  int32 unread_reaction_count_;
  static jfieldID unread_reaction_count_fieldID;
  /// Notification settings for the chat.
  object_ptr<chatNotificationSettings> notification_settings_;
  static jfieldID notification_settings_fieldID;
  /// Types of reaction, available in the chat.
  object_ptr<ChatAvailableReactions> available_reactions_;
  static jfieldID available_reactions_fieldID;
  /// Current message auto-delete or self-destruct timer setting for the chat, in seconds; 0 if disabled. Self-destruct timer in secret chats starts after the message or its content is viewed. Auto-delete timer in other chats starts from the send date.
  int32 message_auto_delete_time_;
  static jfieldID message_auto_delete_time_fieldID;
  /// Emoji status to be shown along with chat title; may be null.
  object_ptr<emojiStatus> emoji_status_;
  static jfieldID emoji_status_fieldID;
  /// Background set for the chat; may be null if none.
  object_ptr<chatBackground> background_;
  static jfieldID background_fieldID;
  /// If non-empty, name of a theme, set for the chat.
  string theme_name_;
  static jfieldID theme_name_fieldID;
  /// Information about actions which must be possible to do through the chat action bar; may be null if none.
  object_ptr<ChatActionBar> action_bar_;
  static jfieldID action_bar_fieldID;
  /// Information about bar for managing a business bot in the chat; may be null if none.
  object_ptr<businessBotManageBar> business_bot_manage_bar_;
  static jfieldID business_bot_manage_bar_fieldID;
  /// Information about video chat of the chat.
  object_ptr<videoChat> video_chat_;
  static jfieldID video_chat_fieldID;
  /// Information about pending join requests; may be null if none.
  object_ptr<chatJoinRequestsInfo> pending_join_requests_;
  static jfieldID pending_join_requests_fieldID;
  /// Identifier of the message from which reply markup needs to be used; 0 if there is no default custom reply markup in the chat.
  int53 reply_markup_message_id_;
  static jfieldID reply_markup_message_id_fieldID;
  /// A draft of a message in the chat; may be null if none.
  object_ptr<draftMessage> draft_message_;
  static jfieldID draft_message_fieldID;
  /// Application-specific data associated with the chat. (For example, the chat scroll position or local chat notification settings can be stored here.) Persistent if the message database is used.
  string client_data_;
  static jfieldID client_data_fieldID;

  /**
   * A chat. (Can be a private chat, basic group, supergroup, or secret chat.)
   */
  chat();

  /**
   * A chat. (Can be a private chat, basic group, supergroup, or secret chat.)
   *
   * \param[in] id_ Chat unique identifier.
   * \param[in] type_ Type of the chat.
   * \param[in] title_ Chat title.
   * \param[in] photo_ Chat photo; may be null.
   * \param[in] accent_color_id_ Identifier of the accent color for message sender name, and backgrounds of chat photo, reply header, and link preview.
   * \param[in] background_custom_emoji_id_ Identifier of a custom emoji to be shown on the reply header and link preview background for messages sent by the chat; 0 if none.
   * \param[in] profile_accent_color_id_ Identifier of the profile accent color for the chat's profile; -1 if none.
   * \param[in] profile_background_custom_emoji_id_ Identifier of a custom emoji to be shown on the background of the chat's profile; 0 if none.
   * \param[in] permissions_ Actions that non-administrator chat members are allowed to take in the chat.
   * \param[in] last_message_ Last message in the chat; may be null if none or unknown.
   * \param[in] positions_ Positions of the chat in chat lists.
   * \param[in] chat_lists_ Chat lists to which the chat belongs. A chat can have a non-zero position in a chat list even if it doesn't belong to the chat list and have no position in a chat list even if it belongs to the chat list.
   * \param[in] message_sender_id_ Identifier of a user or chat that is selected to send messages in the chat; may be null if the user can't change message sender.
   * \param[in] block_list_ Block list to which the chat is added; may be null if none.
   * \param[in] has_protected_content_ True, if chat content can't be saved locally, forwarded, or copied.
   * \param[in] is_translatable_ True, if translation of all messages in the chat must be suggested to the user.
   * \param[in] is_marked_as_unread_ True, if the chat is marked as unread.
   * \param[in] view_as_topics_ True, if the chat is a forum supergroup that must be shown in the &quot;View as topics&quot; mode, or Saved Messages chat that must be shown in the &quot;View as chats&quot;.
   * \param[in] has_scheduled_messages_ True, if the chat has scheduled messages.
   * \param[in] can_be_deleted_only_for_self_ True, if the chat messages can be deleted only for the current user while other users will continue to see the messages.
   * \param[in] can_be_deleted_for_all_users_ True, if the chat messages can be deleted for all users.
   * \param[in] can_be_reported_ True, if the chat can be reported to Telegram moderators through reportChat or reportChatPhoto.
   * \param[in] default_disable_notification_ Default value of the disable_notification parameter, used when a message is sent to the chat.
   * \param[in] unread_count_ Number of unread messages in the chat.
   * \param[in] last_read_inbox_message_id_ Identifier of the last read incoming message.
   * \param[in] last_read_outbox_message_id_ Identifier of the last read outgoing message.
   * \param[in] unread_mention_count_ Number of unread messages with a mention/reply in the chat.
   * \param[in] unread_reaction_count_ Number of messages with unread reactions in the chat.
   * \param[in] notification_settings_ Notification settings for the chat.
   * \param[in] available_reactions_ Types of reaction, available in the chat.
   * \param[in] message_auto_delete_time_ Current message auto-delete or self-destruct timer setting for the chat, in seconds; 0 if disabled. Self-destruct timer in secret chats starts after the message or its content is viewed. Auto-delete timer in other chats starts from the send date.
   * \param[in] emoji_status_ Emoji status to be shown along with chat title; may be null.
   * \param[in] background_ Background set for the chat; may be null if none.
   * \param[in] theme_name_ If non-empty, name of a theme, set for the chat.
   * \param[in] action_bar_ Information about actions which must be possible to do through the chat action bar; may be null if none.
   * \param[in] business_bot_manage_bar_ Information about bar for managing a business bot in the chat; may be null if none.
   * \param[in] video_chat_ Information about video chat of the chat.
   * \param[in] pending_join_requests_ Information about pending join requests; may be null if none.
   * \param[in] reply_markup_message_id_ Identifier of the message from which reply markup needs to be used; 0 if there is no default custom reply markup in the chat.
   * \param[in] draft_message_ A draft of a message in the chat; may be null if none.
   * \param[in] client_data_ Application-specific data associated with the chat. (For example, the chat scroll position or local chat notification settings can be stored here.) Persistent if the message database is used.
   */
  chat(int53 id_, object_ptr<ChatType> &&type_, string const &title_, object_ptr<chatPhotoInfo> &&photo_, int32 accent_color_id_, int64 background_custom_emoji_id_, int32 profile_accent_color_id_, int64 profile_background_custom_emoji_id_, object_ptr<chatPermissions> &&permissions_, object_ptr<message> &&last_message_, array<object_ptr<chatPosition>> &&positions_, array<object_ptr<ChatList>> &&chat_lists_, object_ptr<MessageSender> &&message_sender_id_, object_ptr<BlockList> &&block_list_, bool has_protected_content_, bool is_translatable_, bool is_marked_as_unread_, bool view_as_topics_, bool has_scheduled_messages_, bool can_be_deleted_only_for_self_, bool can_be_deleted_for_all_users_, bool can_be_reported_, bool default_disable_notification_, int32 unread_count_, int53 last_read_inbox_message_id_, int53 last_read_outbox_message_id_, int32 unread_mention_count_, int32 unread_reaction_count_, object_ptr<chatNotificationSettings> &&notification_settings_, object_ptr<ChatAvailableReactions> &&available_reactions_, int32 message_auto_delete_time_, object_ptr<emojiStatus> &&emoji_status_, object_ptr<chatBackground> &&background_, string const &theme_name_, object_ptr<ChatActionBar> &&action_bar_, object_ptr<businessBotManageBar> &&business_bot_manage_bar_, object_ptr<videoChat> &&video_chat_, object_ptr<chatJoinRequestsInfo> &&pending_join_requests_, int53 reply_markup_message_id_, object_ptr<draftMessage> &&draft_message_, string const &client_data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 830601369;

  static object_ptr<chat> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes the different types of activity in a chat.
 */
class ChatAction: public Object {
 public:
  static jclass Class;

  static object_ptr<ChatAction> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is typing a message.
 */
class chatActionTyping final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user is typing a message.
   */
  chatActionTyping();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 380122167;

  static object_ptr<ChatAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is recording a video.
 */
class chatActionRecordingVideo final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user is recording a video.
   */
  chatActionRecordingVideo();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 216553362;

  static object_ptr<ChatAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is uploading a video.
 */
class chatActionUploadingVideo final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Upload progress, as a percentage.
  int32 progress_;
  static jfieldID progress_fieldID;

  /**
   * The user is uploading a video.
   */
  chatActionUploadingVideo();

  /**
   * The user is uploading a video.
   *
   * \param[in] progress_ Upload progress, as a percentage.
   */
  explicit chatActionUploadingVideo(int32 progress_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1234185270;

  static object_ptr<ChatAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is recording a voice note.
 */
class chatActionRecordingVoiceNote final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user is recording a voice note.
   */
  chatActionRecordingVoiceNote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -808850058;

  static object_ptr<ChatAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is uploading a voice note.
 */
class chatActionUploadingVoiceNote final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Upload progress, as a percentage.
  int32 progress_;
  static jfieldID progress_fieldID;

  /**
   * The user is uploading a voice note.
   */
  chatActionUploadingVoiceNote();

  /**
   * The user is uploading a voice note.
   *
   * \param[in] progress_ Upload progress, as a percentage.
   */
  explicit chatActionUploadingVoiceNote(int32 progress_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -613643666;

  static object_ptr<ChatAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is uploading a photo.
 */
class chatActionUploadingPhoto final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Upload progress, as a percentage.
  int32 progress_;
  static jfieldID progress_fieldID;

  /**
   * The user is uploading a photo.
   */
  chatActionUploadingPhoto();

  /**
   * The user is uploading a photo.
   *
   * \param[in] progress_ Upload progress, as a percentage.
   */
  explicit chatActionUploadingPhoto(int32 progress_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 654240583;

  static object_ptr<ChatAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is uploading a document.
 */
class chatActionUploadingDocument final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Upload progress, as a percentage.
  int32 progress_;
  static jfieldID progress_fieldID;

  /**
   * The user is uploading a document.
   */
  chatActionUploadingDocument();

  /**
   * The user is uploading a document.
   *
   * \param[in] progress_ Upload progress, as a percentage.
   */
  explicit chatActionUploadingDocument(int32 progress_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 167884362;

  static object_ptr<ChatAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is picking a sticker to send.
 */
class chatActionChoosingSticker final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user is picking a sticker to send.
   */
  chatActionChoosingSticker();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 372753697;

  static object_ptr<ChatAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is picking a location or venue to send.
 */
class chatActionChoosingLocation final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user is picking a location or venue to send.
   */
  chatActionChoosingLocation();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2017893596;

  static object_ptr<ChatAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is picking a contact to send.
 */
class chatActionChoosingContact final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user is picking a contact to send.
   */
  chatActionChoosingContact();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1222507496;

  static object_ptr<ChatAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user has started to play a game.
 */
class chatActionStartPlayingGame final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user has started to play a game.
   */
  chatActionStartPlayingGame();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -865884164;

  static object_ptr<ChatAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is recording a video note.
 */
class chatActionRecordingVideoNote final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user is recording a video note.
   */
  chatActionRecordingVideoNote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 16523393;

  static object_ptr<ChatAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is uploading a video note.
 */
class chatActionUploadingVideoNote final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Upload progress, as a percentage.
  int32 progress_;
  static jfieldID progress_fieldID;

  /**
   * The user is uploading a video note.
   */
  chatActionUploadingVideoNote();

  /**
   * The user is uploading a video note.
   *
   * \param[in] progress_ Upload progress, as a percentage.
   */
  explicit chatActionUploadingVideoNote(int32 progress_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1172364918;

  static object_ptr<ChatAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is watching animations sent by the other party by clicking on an animated emoji.
 */
class chatActionWatchingAnimations final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The animated emoji.
  string emoji_;
  static jfieldID emoji_fieldID;

  /**
   * The user is watching animations sent by the other party by clicking on an animated emoji.
   */
  chatActionWatchingAnimations();

  /**
   * The user is watching animations sent by the other party by clicking on an animated emoji.
   *
   * \param[in] emoji_ The animated emoji.
   */
  explicit chatActionWatchingAnimations(string const &emoji_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2052990641;

  static object_ptr<ChatAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user has canceled the previous action.
 */
class chatActionCancel final : public ChatAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user has canceled the previous action.
   */
  chatActionCancel();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1160523958;

  static object_ptr<ChatAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class accountInfo;

/**
 * This class is an abstract base class.
 * Describes actions which must be possible to do through a chat action bar.
 */
class ChatActionBar: public Object {
 public:
  static jclass Class;

  static object_ptr<ChatActionBar> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat can be reported as spam using the method reportChat with an empty option_id and message_ids. If the chat is a private chat with a user with an emoji status, then a notice about emoji status usage must be shown.
 */
class chatActionBarReportSpam final : public ChatActionBar {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// If true, the chat was automatically archived and can be moved back to the main chat list using addChatToList simultaneously with setting chat notification settings to default using setChatNotificationSettings.
  bool can_unarchive_;
  static jfieldID can_unarchive_fieldID;

  /**
   * The chat can be reported as spam using the method reportChat with an empty option_id and message_ids. If the chat is a private chat with a user with an emoji status, then a notice about emoji status usage must be shown.
   */
  chatActionBarReportSpam();

  /**
   * The chat can be reported as spam using the method reportChat with an empty option_id and message_ids. If the chat is a private chat with a user with an emoji status, then a notice about emoji status usage must be shown.
   *
   * \param[in] can_unarchive_ If true, the chat was automatically archived and can be moved back to the main chat list using addChatToList simultaneously with setting chat notification settings to default using setChatNotificationSettings.
   */
  explicit chatActionBarReportSpam(bool can_unarchive_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1312758246;

  static object_ptr<ChatActionBar> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat is a recently created group chat to which new members can be invited.
 */
class chatActionBarInviteMembers final : public ChatActionBar {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The chat is a recently created group chat to which new members can be invited.
   */
  chatActionBarInviteMembers();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1985313904;

  static object_ptr<ChatActionBar> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat is a private or secret chat, which can be reported using the method reportChat, or the other user can be blocked using the method setMessageSenderBlockList, or the other user can be added to the contact list using the method addContact. If the chat is a private chat with a user with an emoji status, then a notice about emoji status usage must be shown.
 */
class chatActionBarReportAddBlock final : public ChatActionBar {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// If true, the chat was automatically archived and can be moved back to the main chat list using addChatToList simultaneously with setting chat notification settings to default using setChatNotificationSettings.
  bool can_unarchive_;
  static jfieldID can_unarchive_fieldID;
  /// Basic information about the other user in the chat; may be null if unknown.
  object_ptr<accountInfo> account_info_;
  static jfieldID account_info_fieldID;

  /**
   * The chat is a private or secret chat, which can be reported using the method reportChat, or the other user can be blocked using the method setMessageSenderBlockList, or the other user can be added to the contact list using the method addContact. If the chat is a private chat with a user with an emoji status, then a notice about emoji status usage must be shown.
   */
  chatActionBarReportAddBlock();

  /**
   * The chat is a private or secret chat, which can be reported using the method reportChat, or the other user can be blocked using the method setMessageSenderBlockList, or the other user can be added to the contact list using the method addContact. If the chat is a private chat with a user with an emoji status, then a notice about emoji status usage must be shown.
   *
   * \param[in] can_unarchive_ If true, the chat was automatically archived and can be moved back to the main chat list using addChatToList simultaneously with setting chat notification settings to default using setChatNotificationSettings.
   * \param[in] account_info_ Basic information about the other user in the chat; may be null if unknown.
   */
  chatActionBarReportAddBlock(bool can_unarchive_, object_ptr<accountInfo> &&account_info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1476817269;

  static object_ptr<ChatActionBar> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat is a private or secret chat and the other user can be added to the contact list using the method addContact.
 */
class chatActionBarAddContact final : public ChatActionBar {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The chat is a private or secret chat and the other user can be added to the contact list using the method addContact.
   */
  chatActionBarAddContact();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -733325295;

  static object_ptr<ChatActionBar> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat is a private or secret chat with a mutual contact and the user's phone number can be shared with the other user using the method sharePhoneNumber.
 */
class chatActionBarSharePhoneNumber final : public ChatActionBar {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The chat is a private or secret chat with a mutual contact and the user's phone number can be shared with the other user using the method sharePhoneNumber.
   */
  chatActionBarSharePhoneNumber();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 35188697;

  static object_ptr<ChatActionBar> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat is a private chat with an administrator of a chat to which the user sent join request.
 */
class chatActionBarJoinRequest final : public ChatActionBar {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Title of the chat to which the join request was sent.
  string title_;
  static jfieldID title_fieldID;
  /// True, if the join request was sent to a channel chat.
  bool is_channel_;
  static jfieldID is_channel_fieldID;
  /// Point in time (Unix timestamp) when the join request was sent.
  int32 request_date_;
  static jfieldID request_date_fieldID;

  /**
   * The chat is a private chat with an administrator of a chat to which the user sent join request.
   */
  chatActionBarJoinRequest();

  /**
   * The chat is a private chat with an administrator of a chat to which the user sent join request.
   *
   * \param[in] title_ Title of the chat to which the join request was sent.
   * \param[in] is_channel_ True, if the join request was sent to a channel chat.
   * \param[in] request_date_ Point in time (Unix timestamp) when the join request was sent.
   */
  chatActionBarJoinRequest(string const &title_, bool is_channel_, int32 request_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1037140744;

  static object_ptr<ChatActionBar> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class StoryList;

class storyInfo;

/**
 * Describes active stories posted by a chat.
 */
class chatActiveStories final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the chat that posted the stories.
  int53 chat_id_;
  static jfieldID chat_id_fieldID;
  /// Identifier of the story list in which the stories are shown; may be null if the stories aren't shown in a story list.
  object_ptr<StoryList> list_;
  static jfieldID list_fieldID;
  /// A parameter used to determine order of the stories in the story list; 0 if the stories doesn't need to be shown in the story list. Stories must be sorted by the pair (order, story_poster_chat_id) in descending order.
  int53 order_;
  static jfieldID order_fieldID;
  /// Identifier of the last read active story.
  int32 max_read_story_id_;
  static jfieldID max_read_story_id_fieldID;
  /// Basic information about the stories; use getStory to get full information about the stories. The stories are in chronological order (i.e., in order of increasing story identifiers).
  array<object_ptr<storyInfo>> stories_;
  static jfieldID stories_fieldID;

  /**
   * Describes active stories posted by a chat.
   */
  chatActiveStories();

  /**
   * Describes active stories posted by a chat.
   *
   * \param[in] chat_id_ Identifier of the chat that posted the stories.
   * \param[in] list_ Identifier of the story list in which the stories are shown; may be null if the stories aren't shown in a story list.
   * \param[in] order_ A parameter used to determine order of the stories in the story list; 0 if the stories doesn't need to be shown in the story list. Stories must be sorted by the pair (order, story_poster_chat_id) in descending order.
   * \param[in] max_read_story_id_ Identifier of the last read active story.
   * \param[in] stories_ Basic information about the stories; use getStory to get full information about the stories. The stories are in chronological order (i.e., in order of increasing story identifiers).
   */
  chatActiveStories(int53 chat_id_, object_ptr<StoryList> &&list_, int53 order_, int32 max_read_story_id_, array<object_ptr<storyInfo>> &&stories_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1398869529;

  static object_ptr<chatActiveStories> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains information about a chat administrator.
 */
class chatAdministrator final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// User identifier of the administrator.
  int53 user_id_;
  static jfieldID user_id_fieldID;
  /// Custom title of the administrator.
  string custom_title_;
  static jfieldID custom_title_fieldID;
  /// True, if the user is the owner of the chat.
  bool is_owner_;
  static jfieldID is_owner_fieldID;

  /**
   * Contains information about a chat administrator.
   */
  chatAdministrator();

  /**
   * Contains information about a chat administrator.
   *
   * \param[in] user_id_ User identifier of the administrator.
   * \param[in] custom_title_ Custom title of the administrator.
   * \param[in] is_owner_ True, if the user is the owner of the chat.
   */
  chatAdministrator(int53 user_id_, string const &custom_title_, bool is_owner_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1920449836;

  static object_ptr<chatAdministrator> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes rights of the administrator.
 */
class chatAdministratorRights final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// True, if the administrator can access the chat event log, get boost list, see hidden supergroup and channel members, report supergroup spam messages, ignore slow mode, and send messages to the chat without paying Telegram Stars. Implied by any other privilege; applicable to supergroups and channels only.
  bool can_manage_chat_;
  static jfieldID can_manage_chat_fieldID;
  /// True, if the administrator can change the chat title, photo, and other settings.
  bool can_change_info_;
  static jfieldID can_change_info_fieldID;
  /// True, if the administrator can create channel posts, approve suggested channel posts, or view channel statistics; applicable to channels only.
  bool can_post_messages_;
  static jfieldID can_post_messages_fieldID;
  /// True, if the administrator can edit messages of other users and pin messages; applicable to channels only.
  bool can_edit_messages_;
  static jfieldID can_edit_messages_fieldID;
  /// True, if the administrator can delete messages of other users.
  bool can_delete_messages_;
  static jfieldID can_delete_messages_fieldID;
  /// True, if the administrator can invite new users to the chat.
  bool can_invite_users_;
  static jfieldID can_invite_users_fieldID;
  /// True, if the administrator can restrict, ban, or unban chat members or view supergroup statistics; always true for channels.
  bool can_restrict_members_;
  static jfieldID can_restrict_members_fieldID;
  /// True, if the administrator can pin messages; applicable to basic groups and supergroups only.
  bool can_pin_messages_;
  static jfieldID can_pin_messages_fieldID;
  /// True, if the administrator can create, rename, close, reopen, hide, and unhide forum topics; applicable to forum supergroups only.
  bool can_manage_topics_;
  static jfieldID can_manage_topics_fieldID;
  /// True, if the administrator can add new administrators with a subset of their own privileges or demote administrators that were directly or indirectly promoted by them.
  bool can_promote_members_;
  static jfieldID can_promote_members_fieldID;
  /// True, if the administrator can manage video chats.
  bool can_manage_video_chats_;
  static jfieldID can_manage_video_chats_fieldID;
  /// True, if the administrator can create new chat stories, or edit and delete posted stories; applicable to supergroups and channels only.
  bool can_post_stories_;
  static jfieldID can_post_stories_fieldID;
  /// True, if the administrator can edit stories posted by other users, post stories to the chat page, pin chat stories, and access story archive; applicable to supergroups and channels only.
  bool can_edit_stories_;
  static jfieldID can_edit_stories_fieldID;
  /// True, if the administrator can delete stories posted by other users; applicable to supergroups and channels only.
  bool can_delete_stories_;
  static jfieldID can_delete_stories_fieldID;
  /// True, if the administrator can answer to channel direct messages; applicable to channels only.
  bool can_manage_direct_messages_;
  static jfieldID can_manage_direct_messages_fieldID;
  /// True, if the administrator isn't shown in the chat member list and sends messages anonymously; applicable to supergroups only.
  bool is_anonymous_;
  static jfieldID is_anonymous_fieldID;

  /**
   * Describes rights of the administrator.
   */
  chatAdministratorRights();

  /**
   * Describes rights of the administrator.
   *
   * \param[in] can_manage_chat_ True, if the administrator can access the chat event log, get boost list, see hidden supergroup and channel members, report supergroup spam messages, ignore slow mode, and send messages to the chat without paying Telegram Stars. Implied by any other privilege; applicable to supergroups and channels only.
   * \param[in] can_change_info_ True, if the administrator can change the chat title, photo, and other settings.
   * \param[in] can_post_messages_ True, if the administrator can create channel posts, approve suggested channel posts, or view channel statistics; applicable to channels only.
   * \param[in] can_edit_messages_ True, if the administrator can edit messages of other users and pin messages; applicable to channels only.
   * \param[in] can_delete_messages_ True, if the administrator can delete messages of other users.
   * \param[in] can_invite_users_ True, if the administrator can invite new users to the chat.
   * \param[in] can_restrict_members_ True, if the administrator can restrict, ban, or unban chat members or view supergroup statistics; always true for channels.
   * \param[in] can_pin_messages_ True, if the administrator can pin messages; applicable to basic groups and supergroups only.
   * \param[in] can_manage_topics_ True, if the administrator can create, rename, close, reopen, hide, and unhide forum topics; applicable to forum supergroups only.
   * \param[in] can_promote_members_ True, if the administrator can add new administrators with a subset of their own privileges or demote administrators that were directly or indirectly promoted by them.
   * \param[in] can_manage_video_chats_ True, if the administrator can manage video chats.
   * \param[in] can_post_stories_ True, if the administrator can create new chat stories, or edit and delete posted stories; applicable to supergroups and channels only.
   * \param[in] can_edit_stories_ True, if the administrator can edit stories posted by other users, post stories to the chat page, pin chat stories, and access story archive; applicable to supergroups and channels only.
   * \param[in] can_delete_stories_ True, if the administrator can delete stories posted by other users; applicable to supergroups and channels only.
   * \param[in] can_manage_direct_messages_ True, if the administrator can answer to channel direct messages; applicable to channels only.
   * \param[in] is_anonymous_ True, if the administrator isn't shown in the chat member list and sends messages anonymously; applicable to supergroups only.
   */
  chatAdministratorRights(bool can_manage_chat_, bool can_change_info_, bool can_post_messages_, bool can_edit_messages_, bool can_delete_messages_, bool can_invite_users_, bool can_restrict_members_, bool can_pin_messages_, bool can_manage_topics_, bool can_promote_members_, bool can_manage_video_chats_, bool can_post_stories_, bool can_edit_stories_, bool can_delete_stories_, bool can_manage_direct_messages_, bool is_anonymous_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1142880374;

  static object_ptr<chatAdministratorRights> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class chatAdministrator;

/**
 * Represents a list of chat administrators.
 */
class chatAdministrators final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A list of chat administrators.
  array<object_ptr<chatAdministrator>> administrators_;
  static jfieldID administrators_fieldID;

  /**
   * Represents a list of chat administrators.
   */
  chatAdministrators();

  /**
   * Represents a list of chat administrators.
   *
   * \param[in] administrators_ A list of chat administrators.
   */
  explicit chatAdministrators(array<object_ptr<chatAdministrator>> &&administrators_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2126186435;

  static object_ptr<chatAdministrators> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class ReactionType;

/**
 * This class is an abstract base class.
 * Describes reactions available in the chat.
 */
class ChatAvailableReactions: public Object {
 public:
  static jclass Class;

  static object_ptr<ChatAvailableReactions> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * All reactions are available in the chat, excluding the paid reaction and custom reactions in channel chats.
 */
class chatAvailableReactionsAll final : public ChatAvailableReactions {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The maximum allowed number of reactions per message; 1-11.
  int32 max_reaction_count_;
  static jfieldID max_reaction_count_fieldID;

  /**
   * All reactions are available in the chat, excluding the paid reaction and custom reactions in channel chats.
   */
  chatAvailableReactionsAll();

  /**
   * All reactions are available in the chat, excluding the paid reaction and custom reactions in channel chats.
   *
   * \param[in] max_reaction_count_ The maximum allowed number of reactions per message; 1-11.
   */
  explicit chatAvailableReactionsAll(int32 max_reaction_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 694160279;

  static object_ptr<ChatAvailableReactions> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Only specific reactions are available in the chat.
 */
class chatAvailableReactionsSome final : public ChatAvailableReactions {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The list of reactions.
  array<object_ptr<ReactionType>> reactions_;
  static jfieldID reactions_fieldID;
  /// The maximum allowed number of reactions per message; 1-11.
  int32 max_reaction_count_;
  static jfieldID max_reaction_count_fieldID;

  /**
   * Only specific reactions are available in the chat.
   */
  chatAvailableReactionsSome();

  /**
   * Only specific reactions are available in the chat.
   *
   * \param[in] reactions_ The list of reactions.
   * \param[in] max_reaction_count_ The maximum allowed number of reactions per message; 1-11.
   */
  chatAvailableReactionsSome(array<object_ptr<ReactionType>> &&reactions_, int32 max_reaction_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 152513153;

  static object_ptr<ChatAvailableReactions> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class background;

/**
 * Describes a background set for a specific chat.
 */
class chatBackground final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The background.
  object_ptr<background> background_;
  static jfieldID background_fieldID;
  /// Dimming of the background in dark themes, as a percentage; 0-100. Applied only to Wallpaper and Fill types of background.
  int32 dark_theme_dimming_;
  static jfieldID dark_theme_dimming_fieldID;

  /**
   * Describes a background set for a specific chat.
   */
  chatBackground();

  /**
   * Describes a background set for a specific chat.
   *
   * \param[in] background_ The background.
   * \param[in] dark_theme_dimming_ Dimming of the background in dark themes, as a percentage; 0-100. Applied only to Wallpaper and Fill types of background.
   */
  chatBackground(object_ptr<background> &&background_, int32 dark_theme_dimming_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1653152104;

  static object_ptr<chatBackground> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class ChatBoostSource;

/**
 * Describes a boost applied to a chat.
 */
class chatBoost final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the boost.
  string id_;
  static jfieldID id_fieldID;
  /// The number of identical boosts applied.
  int32 count_;
  static jfieldID count_fieldID;
  /// Source of the boost.
  object_ptr<ChatBoostSource> source_;
  static jfieldID source_fieldID;
  /// Point in time (Unix timestamp) when the chat was boosted.
  int32 start_date_;
  static jfieldID start_date_fieldID;
  /// Point in time (Unix timestamp) when the boost will expire.
  int32 expiration_date_;
  static jfieldID expiration_date_fieldID;

  /**
   * Describes a boost applied to a chat.
   */
  chatBoost();

  /**
   * Describes a boost applied to a chat.
   *
   * \param[in] id_ Unique identifier of the boost.
   * \param[in] count_ The number of identical boosts applied.
   * \param[in] source_ Source of the boost.
   * \param[in] start_date_ Point in time (Unix timestamp) when the chat was boosted.
   * \param[in] expiration_date_ Point in time (Unix timestamp) when the boost will expire.
   */
  chatBoost(string const &id_, int32 count_, object_ptr<ChatBoostSource> &&source_, int32 start_date_, int32 expiration_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1765815118;

  static object_ptr<chatBoost> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class chatBoostLevelFeatures;

/**
 * Contains a list of features available on the first chat boost levels.
 */
class chatBoostFeatures final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The list of features.
  array<object_ptr<chatBoostLevelFeatures>> features_;
  static jfieldID features_fieldID;
  /// The minimum boost level required to set custom emoji for profile background.
  int32 min_profile_background_custom_emoji_boost_level_;
  static jfieldID min_profile_background_custom_emoji_boost_level_fieldID;
  /// The minimum boost level required to set custom emoji for reply header and link preview background; for channel chats only.
  int32 min_background_custom_emoji_boost_level_;
  static jfieldID min_background_custom_emoji_boost_level_fieldID;
  /// The minimum boost level required to set emoji status.
  int32 min_emoji_status_boost_level_;
  static jfieldID min_emoji_status_boost_level_fieldID;
  /// The minimum boost level required to set a chat theme background as chat background.
  int32 min_chat_theme_background_boost_level_;
  static jfieldID min_chat_theme_background_boost_level_fieldID;
  /// The minimum boost level required to set custom chat background.
  int32 min_custom_background_boost_level_;
  static jfieldID min_custom_background_boost_level_fieldID;
  /// The minimum boost level required to set custom emoji sticker set for the chat; for supergroup chats only.
  int32 min_custom_emoji_sticker_set_boost_level_;
  static jfieldID min_custom_emoji_sticker_set_boost_level_fieldID;
  /// The minimum boost level allowing to enable automatic translation of messages for non-Premium users; for channel chats only.
  int32 min_automatic_translation_boost_level_;
  static jfieldID min_automatic_translation_boost_level_fieldID;
  /// The minimum boost level allowing to recognize speech in video note and voice note messages for non-Premium users; for supergroup chats only.
  int32 min_speech_recognition_boost_level_;
  static jfieldID min_speech_recognition_boost_level_fieldID;
  /// The minimum boost level allowing to disable sponsored messages in the chat; for channel chats only.
  int32 min_sponsored_message_disable_boost_level_;
  static jfieldID min_sponsored_message_disable_boost_level_fieldID;

  /**
   * Contains a list of features available on the first chat boost levels.
   */
  chatBoostFeatures();

  /**
   * Contains a list of features available on the first chat boost levels.
   *
   * \param[in] features_ The list of features.
   * \param[in] min_profile_background_custom_emoji_boost_level_ The minimum boost level required to set custom emoji for profile background.
   * \param[in] min_background_custom_emoji_boost_level_ The minimum boost level required to set custom emoji for reply header and link preview background; for channel chats only.
   * \param[in] min_emoji_status_boost_level_ The minimum boost level required to set emoji status.
   * \param[in] min_chat_theme_background_boost_level_ The minimum boost level required to set a chat theme background as chat background.
   * \param[in] min_custom_background_boost_level_ The minimum boost level required to set custom chat background.
   * \param[in] min_custom_emoji_sticker_set_boost_level_ The minimum boost level required to set custom emoji sticker set for the chat; for supergroup chats only.
   * \param[in] min_automatic_translation_boost_level_ The minimum boost level allowing to enable automatic translation of messages for non-Premium users; for channel chats only.
   * \param[in] min_speech_recognition_boost_level_ The minimum boost level allowing to recognize speech in video note and voice note messages for non-Premium users; for supergroup chats only.
   * \param[in] min_sponsored_message_disable_boost_level_ The minimum boost level allowing to disable sponsored messages in the chat; for channel chats only.
   */
  chatBoostFeatures(array<object_ptr<chatBoostLevelFeatures>> &&features_, int32 min_profile_background_custom_emoji_boost_level_, int32 min_background_custom_emoji_boost_level_, int32 min_emoji_status_boost_level_, int32 min_chat_theme_background_boost_level_, int32 min_custom_background_boost_level_, int32 min_custom_emoji_sticker_set_boost_level_, int32 min_automatic_translation_boost_level_, int32 min_speech_recognition_boost_level_, int32 min_sponsored_message_disable_boost_level_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -940531367;

  static object_ptr<chatBoostFeatures> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains a list of features available on a specific chat boost level.
 */
class chatBoostLevelFeatures final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Target chat boost level.
  int32 level_;
  static jfieldID level_fieldID;
  /// Number of stories that the chat can publish daily.
  int32 story_per_day_count_;
  static jfieldID story_per_day_count_fieldID;
  /// Number of custom emoji reactions that can be added to the list of available reactions.
  int32 custom_emoji_reaction_count_;
  static jfieldID custom_emoji_reaction_count_fieldID;
  /// Number of custom colors for chat title.
  int32 title_color_count_;
  static jfieldID title_color_count_fieldID;
  /// Number of custom colors for profile photo background.
  int32 profile_accent_color_count_;
  static jfieldID profile_accent_color_count_fieldID;
  /// True, if custom emoji for profile background can be set.
  bool can_set_profile_background_custom_emoji_;
  static jfieldID can_set_profile_background_custom_emoji_fieldID;
  /// Number of custom colors for background of empty chat photo, replies to messages and link previews.
  int32 accent_color_count_;
  static jfieldID accent_color_count_fieldID;
  /// True, if custom emoji for reply header and link preview background can be set.
  bool can_set_background_custom_emoji_;
  static jfieldID can_set_background_custom_emoji_fieldID;
  /// True, if emoji status can be set.
  bool can_set_emoji_status_;
  static jfieldID can_set_emoji_status_fieldID;
  /// Number of chat theme backgrounds that can be set as chat background.
  int32 chat_theme_background_count_;
  static jfieldID chat_theme_background_count_fieldID;
  /// True, if custom background can be set in the chat for all users.
  bool can_set_custom_background_;
  static jfieldID can_set_custom_background_fieldID;
  /// True, if custom emoji sticker set can be set for the chat.
  bool can_set_custom_emoji_sticker_set_;
  static jfieldID can_set_custom_emoji_sticker_set_fieldID;
  /// True, if automatic translation of messages can be enabled in the chat.
  bool can_enable_automatic_translation_;
  static jfieldID can_enable_automatic_translation_fieldID;
  /// True, if speech recognition can be used for video note and voice note messages by all users.
  bool can_recognize_speech_;
  static jfieldID can_recognize_speech_fieldID;
  /// True, if sponsored messages can be disabled in the chat.
  bool can_disable_sponsored_messages_;
  static jfieldID can_disable_sponsored_messages_fieldID;

  /**
   * Contains a list of features available on a specific chat boost level.
   */
  chatBoostLevelFeatures();

  /**
   * Contains a list of features available on a specific chat boost level.
   *
   * \param[in] level_ Target chat boost level.
   * \param[in] story_per_day_count_ Number of stories that the chat can publish daily.
   * \param[in] custom_emoji_reaction_count_ Number of custom emoji reactions that can be added to the list of available reactions.
   * \param[in] title_color_count_ Number of custom colors for chat title.
   * \param[in] profile_accent_color_count_ Number of custom colors for profile photo background.
   * \param[in] can_set_profile_background_custom_emoji_ True, if custom emoji for profile background can be set.
   * \param[in] accent_color_count_ Number of custom colors for background of empty chat photo, replies to messages and link previews.
   * \param[in] can_set_background_custom_emoji_ True, if custom emoji for reply header and link preview background can be set.
   * \param[in] can_set_emoji_status_ True, if emoji status can be set.
   * \param[in] chat_theme_background_count_ Number of chat theme backgrounds that can be set as chat background.
   * \param[in] can_set_custom_background_ True, if custom background can be set in the chat for all users.
   * \param[in] can_set_custom_emoji_sticker_set_ True, if custom emoji sticker set can be set for the chat.
   * \param[in] can_enable_automatic_translation_ True, if automatic translation of messages can be enabled in the chat.
   * \param[in] can_recognize_speech_ True, if speech recognition can be used for video note and voice note messages by all users.
   * \param[in] can_disable_sponsored_messages_ True, if sponsored messages can be disabled in the chat.
   */
  chatBoostLevelFeatures(int32 level_, int32 story_per_day_count_, int32 custom_emoji_reaction_count_, int32 title_color_count_, int32 profile_accent_color_count_, bool can_set_profile_background_custom_emoji_, int32 accent_color_count_, bool can_set_background_custom_emoji_, bool can_set_emoji_status_, int32 chat_theme_background_count_, bool can_set_custom_background_, bool can_set_custom_emoji_sticker_set_, bool can_enable_automatic_translation_, bool can_recognize_speech_, bool can_disable_sponsored_messages_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 975439470;

  static object_ptr<chatBoostLevelFeatures> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains an HTTPS link to boost a chat.
 */
class chatBoostLink final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The link.
  string link_;
  static jfieldID link_fieldID;
  /// True, if the link will work for non-members of the chat.
  bool is_public_;
  static jfieldID is_public_fieldID;

  /**
   * Contains an HTTPS link to boost a chat.
   */
  chatBoostLink();

  /**
   * Contains an HTTPS link to boost a chat.
   *
   * \param[in] link_ The link.
   * \param[in] is_public_ True, if the link will work for non-members of the chat.
   */
  chatBoostLink(string const &link_, bool is_public_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1253999503;

  static object_ptr<chatBoostLink> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains information about a link to boost a chat.
 */
class chatBoostLinkInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// True, if the link will work for non-members of the chat.
  bool is_public_;
  static jfieldID is_public_fieldID;
  /// Identifier of the chat to which the link points; 0 if the chat isn't found.
  int53 chat_id_;
  static jfieldID chat_id_fieldID;

  /**
   * Contains information about a link to boost a chat.
   */
  chatBoostLinkInfo();

  /**
   * Contains information about a link to boost a chat.
   *
   * \param[in] is_public_ True, if the link will work for non-members of the chat.
   * \param[in] chat_id_ Identifier of the chat to which the link points; 0 if the chat isn't found.
   */
  chatBoostLinkInfo(bool is_public_, int53 chat_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -602785660;

  static object_ptr<chatBoostLinkInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes a slot for chat boost.
 */
class chatBoostSlot final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the slot.
  int32 slot_id_;
  static jfieldID slot_id_fieldID;
  /// Identifier of the currently boosted chat; 0 if none.
  int53 currently_boosted_chat_id_;
  static jfieldID currently_boosted_chat_id_fieldID;
  /// Point in time (Unix timestamp) when the chat was boosted; 0 if none.
  int32 start_date_;
  static jfieldID start_date_fieldID;
  /// Point in time (Unix timestamp) when the boost will expire.
  int32 expiration_date_;
  static jfieldID expiration_date_fieldID;
  /// Point in time (Unix timestamp) after which the boost can be used for another chat.
  int32 cooldown_until_date_;
  static jfieldID cooldown_until_date_fieldID;

  /**
   * Describes a slot for chat boost.
   */
  chatBoostSlot();

  /**
   * Describes a slot for chat boost.
   *
   * \param[in] slot_id_ Unique identifier of the slot.
   * \param[in] currently_boosted_chat_id_ Identifier of the currently boosted chat; 0 if none.
   * \param[in] start_date_ Point in time (Unix timestamp) when the chat was boosted; 0 if none.
   * \param[in] expiration_date_ Point in time (Unix timestamp) when the boost will expire.
   * \param[in] cooldown_until_date_ Point in time (Unix timestamp) after which the boost can be used for another chat.
   */
  chatBoostSlot(int32 slot_id_, int53 currently_boosted_chat_id_, int32 start_date_, int32 expiration_date_, int32 cooldown_until_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 123206343;

  static object_ptr<chatBoostSlot> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class chatBoostSlot;

/**
 * Contains a list of chat boost slots.
 */
class chatBoostSlots final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// List of boost slots.
  array<object_ptr<chatBoostSlot>> slots_;
  static jfieldID slots_fieldID;

  /**
   * Contains a list of chat boost slots.
   */
  chatBoostSlots();

  /**
   * Contains a list of chat boost slots.
   *
   * \param[in] slots_ List of boost slots.
   */
  explicit chatBoostSlots(array<object_ptr<chatBoostSlot>> &&slots_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1014966293;

  static object_ptr<chatBoostSlots> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes source of a chat boost.
 */
class ChatBoostSource: public Object {
 public:
  static jclass Class;

  static object_ptr<ChatBoostSource> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat created a Telegram Premium gift code for a user.
 */
class chatBoostSourceGiftCode final : public ChatBoostSource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of a user, for which the gift code was created.
  int53 user_id_;
  static jfieldID user_id_fieldID;
  /// The created Telegram Premium gift code, which is known only if this is a gift code for the current user, or it has already been claimed.
  string gift_code_;
  static jfieldID gift_code_fieldID;

  /**
   * The chat created a Telegram Premium gift code for a user.
   */
  chatBoostSourceGiftCode();

  /**
   * The chat created a Telegram Premium gift code for a user.
   *
   * \param[in] user_id_ Identifier of a user, for which the gift code was created.
   * \param[in] gift_code_ The created Telegram Premium gift code, which is known only if this is a gift code for the current user, or it has already been claimed.
   */
  chatBoostSourceGiftCode(int53 user_id_, string const &gift_code_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -98299206;

  static object_ptr<ChatBoostSource> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat created a giveaway.
 */
class chatBoostSourceGiveaway final : public ChatBoostSource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of a user that won in the giveaway; 0 if none.
  int53 user_id_;
  static jfieldID user_id_fieldID;
  /// The created Telegram Premium gift code if it was used by the user or can be claimed by the current user; an empty string otherwise; for Telegram Premium giveways only.
  string gift_code_;
  static jfieldID gift_code_fieldID;
  /// Number of Telegram Stars distributed among winners of the giveaway.
  int53 star_count_;
  static jfieldID star_count_fieldID;
  /// Identifier of the corresponding giveaway message; can be an identifier of a deleted message.
  int53 giveaway_message_id_;
  static jfieldID giveaway_message_id_fieldID;
  /// True, if the winner for the corresponding giveaway prize wasn't chosen, because there were not enough participants.
  bool is_unclaimed_;
  static jfieldID is_unclaimed_fieldID;

  /**
   * The chat created a giveaway.
   */
  chatBoostSourceGiveaway();

  /**
   * The chat created a giveaway.
   *
   * \param[in] user_id_ Identifier of a user that won in the giveaway; 0 if none.
   * \param[in] gift_code_ The created Telegram Premium gift code if it was used by the user or can be claimed by the current user; an empty string otherwise; for Telegram Premium giveways only.
   * \param[in] star_count_ Number of Telegram Stars distributed among winners of the giveaway.
   * \param[in] giveaway_message_id_ Identifier of the corresponding giveaway message; can be an identifier of a deleted message.
   * \param[in] is_unclaimed_ True, if the winner for the corresponding giveaway prize wasn't chosen, because there were not enough participants.
   */
  chatBoostSourceGiveaway(int53 user_id_, string const &gift_code_, int53 star_count_, int53 giveaway_message_id_, bool is_unclaimed_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1918145690;

  static object_ptr<ChatBoostSource> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A user with Telegram Premium subscription or gifted Telegram Premium boosted the chat.
 */
class chatBoostSourcePremium final : public ChatBoostSource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the user.
  int53 user_id_;
  static jfieldID user_id_fieldID;

  /**
   * A user with Telegram Premium subscription or gifted Telegram Premium boosted the chat.
   */
  chatBoostSourcePremium();

  /**
   * A user with Telegram Premium subscription or gifted Telegram Premium boosted the chat.
   *
   * \param[in] user_id_ Identifier of the user.
   */
  explicit chatBoostSourcePremium(int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 972011;

  static object_ptr<ChatBoostSource> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class prepaidGiveaway;

/**
 * Describes current boost status of a chat.
 */
class chatBoostStatus final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// An HTTP URL, which can be used to boost the chat.
  string boost_url_;
  static jfieldID boost_url_fieldID;
  /// Identifiers of boost slots of the current user applied to the chat.
  array<int32> applied_slot_ids_;
  static jfieldID applied_slot_ids_fieldID;
  /// Current boost level of the chat.
  int32 level_;
  static jfieldID level_fieldID;
  /// The number of boosts received by the chat from created Telegram Premium gift codes and giveaways; always 0 if the current user isn't an administrator in the chat.
  int32 gift_code_boost_count_;
  static jfieldID gift_code_boost_count_fieldID;
  /// The number of boosts received by the chat.
  int32 boost_count_;
  static jfieldID boost_count_fieldID;
  /// The number of boosts added to reach the current level.
  int32 current_level_boost_count_;
  static jfieldID current_level_boost_count_fieldID;
  /// The number of boosts needed to reach the next level; 0 if the next level isn't available.
  int32 next_level_boost_count_;
  static jfieldID next_level_boost_count_fieldID;
  /// Approximate number of Telegram Premium subscribers joined the chat; always 0 if the current user isn't an administrator in the chat.
  int32 premium_member_count_;
  static jfieldID premium_member_count_fieldID;
  /// A percentage of Telegram Premium subscribers joined the chat; always 0 if the current user isn't an administrator in the chat.
  double premium_member_percentage_;
  static jfieldID premium_member_percentage_fieldID;
  /// The list of prepaid giveaways available for the chat; only for chat administrators.
  array<object_ptr<prepaidGiveaway>> prepaid_giveaways_;
  static jfieldID prepaid_giveaways_fieldID;

  /**
   * Describes current boost status of a chat.
   */
  chatBoostStatus();

  /**
   * Describes current boost status of a chat.
   *
   * \param[in] boost_url_ An HTTP URL, which can be used to boost the chat.
   * \param[in] applied_slot_ids_ Identifiers of boost slots of the current user applied to the chat.
   * \param[in] level_ Current boost level of the chat.
   * \param[in] gift_code_boost_count_ The number of boosts received by the chat from created Telegram Premium gift codes and giveaways; always 0 if the current user isn't an administrator in the chat.
   * \param[in] boost_count_ The number of boosts received by the chat.
   * \param[in] current_level_boost_count_ The number of boosts added to reach the current level.
   * \param[in] next_level_boost_count_ The number of boosts needed to reach the next level; 0 if the next level isn't available.
   * \param[in] premium_member_count_ Approximate number of Telegram Premium subscribers joined the chat; always 0 if the current user isn't an administrator in the chat.
   * \param[in] premium_member_percentage_ A percentage of Telegram Premium subscribers joined the chat; always 0 if the current user isn't an administrator in the chat.
   * \param[in] prepaid_giveaways_ The list of prepaid giveaways available for the chat; only for chat administrators.
   */
  chatBoostStatus(string const &boost_url_, array<int32> &&applied_slot_ids_, int32 level_, int32 gift_code_boost_count_, int32 boost_count_, int32 current_level_boost_count_, int32 next_level_boost_count_, int32 premium_member_count_, double premium_member_percentage_, array<object_ptr<prepaidGiveaway>> &&prepaid_giveaways_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1050332618;

  static object_ptr<chatBoostStatus> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class ChatEventAction;

class MessageSender;

/**
 * Represents a chat event.
 */
class chatEvent final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Chat event identifier.
  int64 id_;
  static jfieldID id_fieldID;
  /// Point in time (Unix timestamp) when the event happened.
  int32 date_;
  static jfieldID date_fieldID;
  /// Identifier of the user or chat who performed the action.
  object_ptr<MessageSender> member_id_;
  static jfieldID member_id_fieldID;
  /// The action.
  object_ptr<ChatEventAction> action_;
  static jfieldID action_fieldID;

  /**
   * Represents a chat event.
   */
  chatEvent();

  /**
   * Represents a chat event.
   *
   * \param[in] id_ Chat event identifier.
   * \param[in] date_ Point in time (Unix timestamp) when the event happened.
   * \param[in] member_id_ Identifier of the user or chat who performed the action.
   * \param[in] action_ The action.
   */
  chatEvent(int64 id_, int32 date_, object_ptr<MessageSender> &&member_id_, object_ptr<ChatEventAction> &&action_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -652102704;

  static object_ptr<chatEvent> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class ChatAvailableReactions;

class ChatMemberStatus;

class MessageSender;

class chatBackground;

class chatInviteLink;

class chatLocation;

class chatPermissions;

class chatPhoto;

class emojiStatus;

class forumTopicInfo;

class message;

/**
 * This class is an abstract base class.
 * Represents a chat event.
 */
class ChatEventAction: public Object {
 public:
  static jclass Class;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A message was edited.
 */
class chatEventMessageEdited final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The original message before the edit.
  object_ptr<message> old_message_;
  static jfieldID old_message_fieldID;
  /// The message after it was edited.
  object_ptr<message> new_message_;
  static jfieldID new_message_fieldID;

  /**
   * A message was edited.
   */
  chatEventMessageEdited();

  /**
   * A message was edited.
   *
   * \param[in] old_message_ The original message before the edit.
   * \param[in] new_message_ The message after it was edited.
   */
  chatEventMessageEdited(object_ptr<message> &&old_message_, object_ptr<message> &&new_message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -430967304;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A message was deleted.
 */
class chatEventMessageDeleted final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Deleted message.
  object_ptr<message> message_;
  static jfieldID message_fieldID;
  /// True, if the message deletion can be reported via reportSupergroupAntiSpamFalsePositive.
  bool can_report_anti_spam_false_positive_;
  static jfieldID can_report_anti_spam_false_positive_fieldID;

  /**
   * A message was deleted.
   */
  chatEventMessageDeleted();

  /**
   * A message was deleted.
   *
   * \param[in] message_ Deleted message.
   * \param[in] can_report_anti_spam_false_positive_ True, if the message deletion can be reported via reportSupergroupAntiSpamFalsePositive.
   */
  chatEventMessageDeleted(object_ptr<message> &&message_, bool can_report_anti_spam_false_positive_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 935316851;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A message was pinned.
 */
class chatEventMessagePinned final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Pinned message.
  object_ptr<message> message_;
  static jfieldID message_fieldID;

  /**
   * A message was pinned.
   */
  chatEventMessagePinned();

  /**
   * A message was pinned.
   *
   * \param[in] message_ Pinned message.
   */
  explicit chatEventMessagePinned(object_ptr<message> &&message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 438742298;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A message was unpinned.
 */
class chatEventMessageUnpinned final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unpinned message.
  object_ptr<message> message_;
  static jfieldID message_fieldID;

  /**
   * A message was unpinned.
   */
  chatEventMessageUnpinned();

  /**
   * A message was unpinned.
   *
   * \param[in] message_ Unpinned message.
   */
  explicit chatEventMessageUnpinned(object_ptr<message> &&message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -376161513;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A poll in a message was stopped.
 */
class chatEventPollStopped final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The message with the poll.
  object_ptr<message> message_;
  static jfieldID message_fieldID;

  /**
   * A poll in a message was stopped.
   */
  chatEventPollStopped();

  /**
   * A poll in a message was stopped.
   *
   * \param[in] message_ The message with the poll.
   */
  explicit chatEventPollStopped(object_ptr<message> &&message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2009893861;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A new member joined the chat.
 */
class chatEventMemberJoined final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * A new member joined the chat.
   */
  chatEventMemberJoined();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -235468508;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A new member joined the chat via an invite link.
 */
class chatEventMemberJoinedByInviteLink final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Invite link used to join the chat.
  object_ptr<chatInviteLink> invite_link_;
  static jfieldID invite_link_fieldID;
  /// True, if the user has joined the chat using an invite link for a chat folder.
  bool via_chat_folder_invite_link_;
  static jfieldID via_chat_folder_invite_link_fieldID;

  /**
   * A new member joined the chat via an invite link.
   */
  chatEventMemberJoinedByInviteLink();

  /**
   * A new member joined the chat via an invite link.
   *
   * \param[in] invite_link_ Invite link used to join the chat.
   * \param[in] via_chat_folder_invite_link_ True, if the user has joined the chat using an invite link for a chat folder.
   */
  chatEventMemberJoinedByInviteLink(object_ptr<chatInviteLink> &&invite_link_, bool via_chat_folder_invite_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1445536390;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A new member was accepted to the chat by an administrator.
 */
class chatEventMemberJoinedByRequest final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// User identifier of the chat administrator, approved user join request.
  int53 approver_user_id_;
  static jfieldID approver_user_id_fieldID;
  /// Invite link used to join the chat; may be null.
  object_ptr<chatInviteLink> invite_link_;
  static jfieldID invite_link_fieldID;

  /**
   * A new member was accepted to the chat by an administrator.
   */
  chatEventMemberJoinedByRequest();

  /**
   * A new member was accepted to the chat by an administrator.
   *
   * \param[in] approver_user_id_ User identifier of the chat administrator, approved user join request.
   * \param[in] invite_link_ Invite link used to join the chat; may be null.
   */
  chatEventMemberJoinedByRequest(int53 approver_user_id_, object_ptr<chatInviteLink> &&invite_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1647804865;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A new chat member was invited.
 */
class chatEventMemberInvited final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// New member user identifier.
  int53 user_id_;
  static jfieldID user_id_fieldID;
  /// New member status.
  object_ptr<ChatMemberStatus> status_;
  static jfieldID status_fieldID;

  /**
   * A new chat member was invited.
   */
  chatEventMemberInvited();

  /**
   * A new chat member was invited.
   *
   * \param[in] user_id_ New member user identifier.
   * \param[in] status_ New member status.
   */
  chatEventMemberInvited(int53 user_id_, object_ptr<ChatMemberStatus> &&status_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 953663433;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A member left the chat.
 */
class chatEventMemberLeft final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * A member left the chat.
   */
  chatEventMemberLeft();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -948420593;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A chat member has gained/lost administrator status, or the list of their administrator privileges has changed.
 */
class chatEventMemberPromoted final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Affected chat member user identifier.
  int53 user_id_;
  static jfieldID user_id_fieldID;
  /// Previous status of the chat member.
  object_ptr<ChatMemberStatus> old_status_;
  static jfieldID old_status_fieldID;
  /// New status of the chat member.
  object_ptr<ChatMemberStatus> new_status_;
  static jfieldID new_status_fieldID;

  /**
   * A chat member has gained/lost administrator status, or the list of their administrator privileges has changed.
   */
  chatEventMemberPromoted();

  /**
   * A chat member has gained/lost administrator status, or the list of their administrator privileges has changed.
   *
   * \param[in] user_id_ Affected chat member user identifier.
   * \param[in] old_status_ Previous status of the chat member.
   * \param[in] new_status_ New status of the chat member.
   */
  chatEventMemberPromoted(int53 user_id_, object_ptr<ChatMemberStatus> &&old_status_, object_ptr<ChatMemberStatus> &&new_status_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 525297761;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A chat member was restricted/unrestricted or banned/unbanned, or the list of their restrictions has changed.
 */
class chatEventMemberRestricted final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Affected chat member identifier.
  object_ptr<MessageSender> member_id_;
  static jfieldID member_id_fieldID;
  /// Previous status of the chat member.
  object_ptr<ChatMemberStatus> old_status_;
  static jfieldID old_status_fieldID;
  /// New status of the chat member.
  object_ptr<ChatMemberStatus> new_status_;
  static jfieldID new_status_fieldID;

  /**
   * A chat member was restricted/unrestricted or banned/unbanned, or the list of their restrictions has changed.
   */
  chatEventMemberRestricted();

  /**
   * A chat member was restricted/unrestricted or banned/unbanned, or the list of their restrictions has changed.
   *
   * \param[in] member_id_ Affected chat member identifier.
   * \param[in] old_status_ Previous status of the chat member.
   * \param[in] new_status_ New status of the chat member.
   */
  chatEventMemberRestricted(object_ptr<MessageSender> &&member_id_, object_ptr<ChatMemberStatus> &&old_status_, object_ptr<ChatMemberStatus> &&new_status_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1603608069;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A chat member extended their subscription to the chat.
 */
class chatEventMemberSubscriptionExtended final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Affected chat member user identifier.
  int53 user_id_;
  static jfieldID user_id_fieldID;
  /// Previous status of the chat member.
  object_ptr<ChatMemberStatus> old_status_;
  static jfieldID old_status_fieldID;
  /// New status of the chat member.
  object_ptr<ChatMemberStatus> new_status_;
  static jfieldID new_status_fieldID;

  /**
   * A chat member extended their subscription to the chat.
   */
  chatEventMemberSubscriptionExtended();

  /**
   * A chat member extended their subscription to the chat.
   *
   * \param[in] user_id_ Affected chat member user identifier.
   * \param[in] old_status_ Previous status of the chat member.
   * \param[in] new_status_ New status of the chat member.
   */
  chatEventMemberSubscriptionExtended(int53 user_id_, object_ptr<ChatMemberStatus> &&old_status_, object_ptr<ChatMemberStatus> &&new_status_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1141198846;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat available reactions were changed.
 */
class chatEventAvailableReactionsChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Previous chat available reactions.
  object_ptr<ChatAvailableReactions> old_available_reactions_;
  static jfieldID old_available_reactions_fieldID;
  /// New chat available reactions.
  object_ptr<ChatAvailableReactions> new_available_reactions_;
  static jfieldID new_available_reactions_fieldID;

  /**
   * The chat available reactions were changed.
   */
  chatEventAvailableReactionsChanged();

  /**
   * The chat available reactions were changed.
   *
   * \param[in] old_available_reactions_ Previous chat available reactions.
   * \param[in] new_available_reactions_ New chat available reactions.
   */
  chatEventAvailableReactionsChanged(object_ptr<ChatAvailableReactions> &&old_available_reactions_, object_ptr<ChatAvailableReactions> &&new_available_reactions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1749491521;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat background was changed.
 */
class chatEventBackgroundChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Previous background; may be null if none.
  object_ptr<chatBackground> old_background_;
  static jfieldID old_background_fieldID;
  /// New background; may be null if none.
  object_ptr<chatBackground> new_background_;
  static jfieldID new_background_fieldID;

  /**
   * The chat background was changed.
   */
  chatEventBackgroundChanged();

  /**
   * The chat background was changed.
   *
   * \param[in] old_background_ Previous background; may be null if none.
   * \param[in] new_background_ New background; may be null if none.
   */
  chatEventBackgroundChanged(object_ptr<chatBackground> &&old_background_, object_ptr<chatBackground> &&new_background_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1225953992;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat description was changed.
 */
class chatEventDescriptionChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Previous chat description.
  string old_description_;
  static jfieldID old_description_fieldID;
  /// New chat description.
  string new_description_;
  static jfieldID new_description_fieldID;

  /**
   * The chat description was changed.
   */
  chatEventDescriptionChanged();

  /**
   * The chat description was changed.
   *
   * \param[in] old_description_ Previous chat description.
   * \param[in] new_description_ New chat description.
   */
  chatEventDescriptionChanged(string const &old_description_, string const &new_description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 39112478;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat emoji status was changed.
 */
class chatEventEmojiStatusChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Previous emoji status; may be null if none.
  object_ptr<emojiStatus> old_emoji_status_;
  static jfieldID old_emoji_status_fieldID;
  /// New emoji status; may be null if none.
  object_ptr<emojiStatus> new_emoji_status_;
  static jfieldID new_emoji_status_fieldID;

  /**
   * The chat emoji status was changed.
   */
  chatEventEmojiStatusChanged();

  /**
   * The chat emoji status was changed.
   *
   * \param[in] old_emoji_status_ Previous emoji status; may be null if none.
   * \param[in] new_emoji_status_ New emoji status; may be null if none.
   */
  chatEventEmojiStatusChanged(object_ptr<emojiStatus> &&old_emoji_status_, object_ptr<emojiStatus> &&new_emoji_status_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2081850594;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The linked chat of a supergroup was changed.
 */
class chatEventLinkedChatChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Previous supergroup linked chat identifier.
  int53 old_linked_chat_id_;
  static jfieldID old_linked_chat_id_fieldID;
  /// New supergroup linked chat identifier.
  int53 new_linked_chat_id_;
  static jfieldID new_linked_chat_id_fieldID;

  /**
   * The linked chat of a supergroup was changed.
   */
  chatEventLinkedChatChanged();

  /**
   * The linked chat of a supergroup was changed.
   *
   * \param[in] old_linked_chat_id_ Previous supergroup linked chat identifier.
   * \param[in] new_linked_chat_id_ New supergroup linked chat identifier.
   */
  chatEventLinkedChatChanged(int53 old_linked_chat_id_, int53 new_linked_chat_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1797419439;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The supergroup location was changed.
 */
class chatEventLocationChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Previous location; may be null.
  object_ptr<chatLocation> old_location_;
  static jfieldID old_location_fieldID;
  /// New location; may be null.
  object_ptr<chatLocation> new_location_;
  static jfieldID new_location_fieldID;

  /**
   * The supergroup location was changed.
   */
  chatEventLocationChanged();

  /**
   * The supergroup location was changed.
   *
   * \param[in] old_location_ Previous location; may be null.
   * \param[in] new_location_ New location; may be null.
   */
  chatEventLocationChanged(object_ptr<chatLocation> &&old_location_, object_ptr<chatLocation> &&new_location_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -405930674;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The message auto-delete timer was changed.
 */
class chatEventMessageAutoDeleteTimeChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Previous value of message_auto_delete_time.
  int32 old_message_auto_delete_time_;
  static jfieldID old_message_auto_delete_time_fieldID;
  /// New value of message_auto_delete_time.
  int32 new_message_auto_delete_time_;
  static jfieldID new_message_auto_delete_time_fieldID;

  /**
   * The message auto-delete timer was changed.
   */
  chatEventMessageAutoDeleteTimeChanged();

  /**
   * The message auto-delete timer was changed.
   *
   * \param[in] old_message_auto_delete_time_ Previous value of message_auto_delete_time.
   * \param[in] new_message_auto_delete_time_ New value of message_auto_delete_time.
   */
  chatEventMessageAutoDeleteTimeChanged(int32 old_message_auto_delete_time_, int32 new_message_auto_delete_time_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 17317668;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat permissions were changed.
 */
class chatEventPermissionsChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Previous chat permissions.
  object_ptr<chatPermissions> old_permissions_;
  static jfieldID old_permissions_fieldID;
  /// New chat permissions.
  object_ptr<chatPermissions> new_permissions_;
  static jfieldID new_permissions_fieldID;

  /**
   * The chat permissions were changed.
   */
  chatEventPermissionsChanged();

  /**
   * The chat permissions were changed.
   *
   * \param[in] old_permissions_ Previous chat permissions.
   * \param[in] new_permissions_ New chat permissions.
   */
  chatEventPermissionsChanged(object_ptr<chatPermissions> &&old_permissions_, object_ptr<chatPermissions> &&new_permissions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1311557720;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat photo was changed.
 */
class chatEventPhotoChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Previous chat photo value; may be null.
  object_ptr<chatPhoto> old_photo_;
  static jfieldID old_photo_fieldID;
  /// New chat photo value; may be null.
  object_ptr<chatPhoto> new_photo_;
  static jfieldID new_photo_fieldID;

  /**
   * The chat photo was changed.
   */
  chatEventPhotoChanged();

  /**
   * The chat photo was changed.
   *
   * \param[in] old_photo_ Previous chat photo value; may be null.
   * \param[in] new_photo_ New chat photo value; may be null.
   */
  chatEventPhotoChanged(object_ptr<chatPhoto> &&old_photo_, object_ptr<chatPhoto> &&new_photo_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -811572541;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The slow_mode_delay setting of a supergroup was changed.
 */
class chatEventSlowModeDelayChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Previous value of slow_mode_delay, in seconds.
  int32 old_slow_mode_delay_;
  static jfieldID old_slow_mode_delay_fieldID;
  /// New value of slow_mode_delay, in seconds.
  int32 new_slow_mode_delay_;
  static jfieldID new_slow_mode_delay_fieldID;

  /**
   * The slow_mode_delay setting of a supergroup was changed.
   */
  chatEventSlowModeDelayChanged();

  /**
   * The slow_mode_delay setting of a supergroup was changed.
   *
   * \param[in] old_slow_mode_delay_ Previous value of slow_mode_delay, in seconds.
   * \param[in] new_slow_mode_delay_ New value of slow_mode_delay, in seconds.
   */
  chatEventSlowModeDelayChanged(int32 old_slow_mode_delay_, int32 new_slow_mode_delay_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1653195765;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The supergroup sticker set was changed.
 */
class chatEventStickerSetChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Previous identifier of the chat sticker set; 0 if none.
  int64 old_sticker_set_id_;
  static jfieldID old_sticker_set_id_fieldID;
  /// New identifier of the chat sticker set; 0 if none.
  int64 new_sticker_set_id_;
  static jfieldID new_sticker_set_id_fieldID;

  /**
   * The supergroup sticker set was changed.
   */
  chatEventStickerSetChanged();

  /**
   * The supergroup sticker set was changed.
   *
   * \param[in] old_sticker_set_id_ Previous identifier of the chat sticker set; 0 if none.
   * \param[in] new_sticker_set_id_ New identifier of the chat sticker set; 0 if none.
   */
  chatEventStickerSetChanged(int64 old_sticker_set_id_, int64 new_sticker_set_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1243130481;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The supergroup sticker set with allowed custom emoji was changed.
 */
class chatEventCustomEmojiStickerSetChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Previous identifier of the chat sticker set; 0 if none.
  int64 old_sticker_set_id_;
  static jfieldID old_sticker_set_id_fieldID;
  /// New identifier of the chat sticker set; 0 if none.
  int64 new_sticker_set_id_;
  static jfieldID new_sticker_set_id_fieldID;

  /**
   * The supergroup sticker set with allowed custom emoji was changed.
   */
  chatEventCustomEmojiStickerSetChanged();

  /**
   * The supergroup sticker set with allowed custom emoji was changed.
   *
   * \param[in] old_sticker_set_id_ Previous identifier of the chat sticker set; 0 if none.
   * \param[in] new_sticker_set_id_ New identifier of the chat sticker set; 0 if none.
   */
  chatEventCustomEmojiStickerSetChanged(int64 old_sticker_set_id_, int64 new_sticker_set_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 118244123;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat title was changed.
 */
class chatEventTitleChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Previous chat title.
  string old_title_;
  static jfieldID old_title_fieldID;
  /// New chat title.
  string new_title_;
  static jfieldID new_title_fieldID;

  /**
   * The chat title was changed.
   */
  chatEventTitleChanged();

  /**
   * The chat title was changed.
   *
   * \param[in] old_title_ Previous chat title.
   * \param[in] new_title_ New chat title.
   */
  chatEventTitleChanged(string const &old_title_, string const &new_title_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1134103250;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat editable username was changed.
 */
class chatEventUsernameChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Previous chat username.
  string old_username_;
  static jfieldID old_username_fieldID;
  /// New chat username.
  string new_username_;
  static jfieldID new_username_fieldID;

  /**
   * The chat editable username was changed.
   */
  chatEventUsernameChanged();

  /**
   * The chat editable username was changed.
   *
   * \param[in] old_username_ Previous chat username.
   * \param[in] new_username_ New chat username.
   */
  chatEventUsernameChanged(string const &old_username_, string const &new_username_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1728558443;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat active usernames were changed.
 */
class chatEventActiveUsernamesChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Previous list of active usernames.
  array<string> old_usernames_;
  static jfieldID old_usernames_fieldID;
  /// New list of active usernames.
  array<string> new_usernames_;
  static jfieldID new_usernames_fieldID;

  /**
   * The chat active usernames were changed.
   */
  chatEventActiveUsernamesChanged();

  /**
   * The chat active usernames were changed.
   *
   * \param[in] old_usernames_ Previous list of active usernames.
   * \param[in] new_usernames_ New list of active usernames.
   */
  chatEventActiveUsernamesChanged(array<string> &&old_usernames_, array<string> &&new_usernames_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1508790810;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat accent color or background custom emoji were changed.
 */
class chatEventAccentColorChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Previous identifier of chat accent color.
  int32 old_accent_color_id_;
  static jfieldID old_accent_color_id_fieldID;
  /// Previous identifier of the custom emoji; 0 if none.
  int64 old_background_custom_emoji_id_;
  static jfieldID old_background_custom_emoji_id_fieldID;
  /// New identifier of chat accent color.
  int32 new_accent_color_id_;
  static jfieldID new_accent_color_id_fieldID;
  /// New identifier of the custom emoji; 0 if none.
  int64 new_background_custom_emoji_id_;
  static jfieldID new_background_custom_emoji_id_fieldID;

  /**
   * The chat accent color or background custom emoji were changed.
   */
  chatEventAccentColorChanged();

  /**
   * The chat accent color or background custom emoji were changed.
   *
   * \param[in] old_accent_color_id_ Previous identifier of chat accent color.
   * \param[in] old_background_custom_emoji_id_ Previous identifier of the custom emoji; 0 if none.
   * \param[in] new_accent_color_id_ New identifier of chat accent color.
   * \param[in] new_background_custom_emoji_id_ New identifier of the custom emoji; 0 if none.
   */
  chatEventAccentColorChanged(int32 old_accent_color_id_, int64 old_background_custom_emoji_id_, int32 new_accent_color_id_, int64 new_background_custom_emoji_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -427591885;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat's profile accent color or profile background custom emoji were changed.
 */
class chatEventProfileAccentColorChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Previous identifier of chat's profile accent color; -1 if none.
  int32 old_profile_accent_color_id_;
  static jfieldID old_profile_accent_color_id_fieldID;
  /// Previous identifier of the custom emoji; 0 if none.
  int64 old_profile_background_custom_emoji_id_;
  static jfieldID old_profile_background_custom_emoji_id_fieldID;
  /// New identifier of chat's profile accent color; -1 if none.
  int32 new_profile_accent_color_id_;
  static jfieldID new_profile_accent_color_id_fieldID;
  /// New identifier of the custom emoji; 0 if none.
  int64 new_profile_background_custom_emoji_id_;
  static jfieldID new_profile_background_custom_emoji_id_fieldID;

  /**
   * The chat's profile accent color or profile background custom emoji were changed.
   */
  chatEventProfileAccentColorChanged();

  /**
   * The chat's profile accent color or profile background custom emoji were changed.
   *
   * \param[in] old_profile_accent_color_id_ Previous identifier of chat's profile accent color; -1 if none.
   * \param[in] old_profile_background_custom_emoji_id_ Previous identifier of the custom emoji; 0 if none.
   * \param[in] new_profile_accent_color_id_ New identifier of chat's profile accent color; -1 if none.
   * \param[in] new_profile_background_custom_emoji_id_ New identifier of the custom emoji; 0 if none.
   */
  chatEventProfileAccentColorChanged(int32 old_profile_accent_color_id_, int64 old_profile_background_custom_emoji_id_, int32 new_profile_accent_color_id_, int64 new_profile_background_custom_emoji_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1514612124;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The has_protected_content setting of a channel was toggled.
 */
class chatEventHasProtectedContentToggled final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// New value of has_protected_content.
  bool has_protected_content_;
  static jfieldID has_protected_content_fieldID;

  /**
   * The has_protected_content setting of a channel was toggled.
   */
  chatEventHasProtectedContentToggled();

  /**
   * The has_protected_content setting of a channel was toggled.
   *
   * \param[in] has_protected_content_ New value of has_protected_content.
   */
  explicit chatEventHasProtectedContentToggled(bool has_protected_content_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -184270335;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The can_invite_users permission of a supergroup chat was toggled.
 */
class chatEventInvitesToggled final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// New value of can_invite_users permission.
  bool can_invite_users_;
  static jfieldID can_invite_users_fieldID;

  /**
   * The can_invite_users permission of a supergroup chat was toggled.
   */
  chatEventInvitesToggled();

  /**
   * The can_invite_users permission of a supergroup chat was toggled.
   *
   * \param[in] can_invite_users_ New value of can_invite_users permission.
   */
  explicit chatEventInvitesToggled(bool can_invite_users_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -62548373;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The is_all_history_available setting of a supergroup was toggled.
 */
class chatEventIsAllHistoryAvailableToggled final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// New value of is_all_history_available.
  bool is_all_history_available_;
  static jfieldID is_all_history_available_fieldID;

  /**
   * The is_all_history_available setting of a supergroup was toggled.
   */
  chatEventIsAllHistoryAvailableToggled();

  /**
   * The is_all_history_available setting of a supergroup was toggled.
   *
   * \param[in] is_all_history_available_ New value of is_all_history_available.
   */
  explicit chatEventIsAllHistoryAvailableToggled(bool is_all_history_available_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1599063019;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The has_aggressive_anti_spam_enabled setting of a supergroup was toggled.
 */
class chatEventHasAggressiveAntiSpamEnabledToggled final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// New value of has_aggressive_anti_spam_enabled.
  bool has_aggressive_anti_spam_enabled_;
  static jfieldID has_aggressive_anti_spam_enabled_fieldID;

  /**
   * The has_aggressive_anti_spam_enabled setting of a supergroup was toggled.
   */
  chatEventHasAggressiveAntiSpamEnabledToggled();

  /**
   * The has_aggressive_anti_spam_enabled setting of a supergroup was toggled.
   *
   * \param[in] has_aggressive_anti_spam_enabled_ New value of has_aggressive_anti_spam_enabled.
   */
  explicit chatEventHasAggressiveAntiSpamEnabledToggled(bool has_aggressive_anti_spam_enabled_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -125348094;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The sign_messages setting of a channel was toggled.
 */
class chatEventSignMessagesToggled final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// New value of sign_messages.
  bool sign_messages_;
  static jfieldID sign_messages_fieldID;

  /**
   * The sign_messages setting of a channel was toggled.
   */
  chatEventSignMessagesToggled();

  /**
   * The sign_messages setting of a channel was toggled.
   *
   * \param[in] sign_messages_ New value of sign_messages.
   */
  explicit chatEventSignMessagesToggled(bool sign_messages_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1313265634;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The show_message_sender setting of a channel was toggled.
 */
class chatEventShowMessageSenderToggled final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// New value of show_message_sender.
  bool show_message_sender_;
  static jfieldID show_message_sender_fieldID;

  /**
   * The show_message_sender setting of a channel was toggled.
   */
  chatEventShowMessageSenderToggled();

  /**
   * The show_message_sender setting of a channel was toggled.
   *
   * \param[in] show_message_sender_ New value of show_message_sender.
   */
  explicit chatEventShowMessageSenderToggled(bool show_message_sender_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -794343453;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The has_automatic_translation setting of a channel was toggled.
 */
class chatEventAutomaticTranslationToggled final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// New value of has_automatic_translation.
  bool has_automatic_translation_;
  static jfieldID has_automatic_translation_fieldID;

  /**
   * The has_automatic_translation setting of a channel was toggled.
   */
  chatEventAutomaticTranslationToggled();

  /**
   * The has_automatic_translation setting of a channel was toggled.
   *
   * \param[in] has_automatic_translation_ New value of has_automatic_translation.
   */
  explicit chatEventAutomaticTranslationToggled(bool has_automatic_translation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 194147926;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A chat invite link was edited.
 */
class chatEventInviteLinkEdited final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Previous information about the invite link.
  object_ptr<chatInviteLink> old_invite_link_;
  static jfieldID old_invite_link_fieldID;
  /// New information about the invite link.
  object_ptr<chatInviteLink> new_invite_link_;
  static jfieldID new_invite_link_fieldID;

  /**
   * A chat invite link was edited.
   */
  chatEventInviteLinkEdited();

  /**
   * A chat invite link was edited.
   *
   * \param[in] old_invite_link_ Previous information about the invite link.
   * \param[in] new_invite_link_ New information about the invite link.
   */
  chatEventInviteLinkEdited(object_ptr<chatInviteLink> &&old_invite_link_, object_ptr<chatInviteLink> &&new_invite_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -460190366;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A chat invite link was revoked.
 */
class chatEventInviteLinkRevoked final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The invite link.
  object_ptr<chatInviteLink> invite_link_;
  static jfieldID invite_link_fieldID;

  /**
   * A chat invite link was revoked.
   */
  chatEventInviteLinkRevoked();

  /**
   * A chat invite link was revoked.
   *
   * \param[in] invite_link_ The invite link.
   */
  explicit chatEventInviteLinkRevoked(object_ptr<chatInviteLink> &&invite_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1579417629;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A revoked chat invite link was deleted.
 */
class chatEventInviteLinkDeleted final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The invite link.
  object_ptr<chatInviteLink> invite_link_;
  static jfieldID invite_link_fieldID;

  /**
   * A revoked chat invite link was deleted.
   */
  chatEventInviteLinkDeleted();

  /**
   * A revoked chat invite link was deleted.
   *
   * \param[in] invite_link_ The invite link.
   */
  explicit chatEventInviteLinkDeleted(object_ptr<chatInviteLink> &&invite_link_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1394974361;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A video chat was created.
 */
class chatEventVideoChatCreated final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the video chat. The video chat can be received through the method getGroupCall.
  int32 group_call_id_;
  static jfieldID group_call_id_fieldID;

  /**
   * A video chat was created.
   */
  chatEventVideoChatCreated();

  /**
   * A video chat was created.
   *
   * \param[in] group_call_id_ Identifier of the video chat. The video chat can be received through the method getGroupCall.
   */
  explicit chatEventVideoChatCreated(int32 group_call_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1822853755;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A video chat was ended.
 */
class chatEventVideoChatEnded final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the video chat. The video chat can be received through the method getGroupCall.
  int32 group_call_id_;
  static jfieldID group_call_id_fieldID;

  /**
   * A video chat was ended.
   */
  chatEventVideoChatEnded();

  /**
   * A video chat was ended.
   *
   * \param[in] group_call_id_ Identifier of the video chat. The video chat can be received through the method getGroupCall.
   */
  explicit chatEventVideoChatEnded(int32 group_call_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1630039112;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The mute_new_participants setting of a video chat was toggled.
 */
class chatEventVideoChatMuteNewParticipantsToggled final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// New value of the mute_new_participants setting.
  bool mute_new_participants_;
  static jfieldID mute_new_participants_fieldID;

  /**
   * The mute_new_participants setting of a video chat was toggled.
   */
  chatEventVideoChatMuteNewParticipantsToggled();

  /**
   * The mute_new_participants setting of a video chat was toggled.
   *
   * \param[in] mute_new_participants_ New value of the mute_new_participants setting.
   */
  explicit chatEventVideoChatMuteNewParticipantsToggled(bool mute_new_participants_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -126547970;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A video chat participant was muted or unmuted.
 */
class chatEventVideoChatParticipantIsMutedToggled final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the affected group call participant.
  object_ptr<MessageSender> participant_id_;
  static jfieldID participant_id_fieldID;
  /// New value of is_muted.
  bool is_muted_;
  static jfieldID is_muted_fieldID;

  /**
   * A video chat participant was muted or unmuted.
   */
  chatEventVideoChatParticipantIsMutedToggled();

  /**
   * A video chat participant was muted or unmuted.
   *
   * \param[in] participant_id_ Identifier of the affected group call participant.
   * \param[in] is_muted_ New value of is_muted.
   */
  chatEventVideoChatParticipantIsMutedToggled(object_ptr<MessageSender> &&participant_id_, bool is_muted_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 521165047;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A video chat participant volume level was changed.
 */
class chatEventVideoChatParticipantVolumeLevelChanged final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the affected group call participant.
  object_ptr<MessageSender> participant_id_;
  static jfieldID participant_id_fieldID;
  /// New value of volume_level; 1-20000 in hundreds of percents.
  int32 volume_level_;
  static jfieldID volume_level_fieldID;

  /**
   * A video chat participant volume level was changed.
   */
  chatEventVideoChatParticipantVolumeLevelChanged();

  /**
   * A video chat participant volume level was changed.
   *
   * \param[in] participant_id_ Identifier of the affected group call participant.
   * \param[in] volume_level_ New value of volume_level; 1-20000 in hundreds of percents.
   */
  chatEventVideoChatParticipantVolumeLevelChanged(object_ptr<MessageSender> &&participant_id_, int32 volume_level_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1131385534;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The is_forum setting of a channel was toggled.
 */
class chatEventIsForumToggled final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// New value of is_forum.
  bool is_forum_;
  static jfieldID is_forum_fieldID;

  /**
   * The is_forum setting of a channel was toggled.
   */
  chatEventIsForumToggled();

  /**
   * The is_forum setting of a channel was toggled.
   *
   * \param[in] is_forum_ New value of is_forum.
   */
  explicit chatEventIsForumToggled(bool is_forum_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1516491033;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A new forum topic was created.
 */
class chatEventForumTopicCreated final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Information about the topic.
  object_ptr<forumTopicInfo> topic_info_;
  static jfieldID topic_info_fieldID;

  /**
   * A new forum topic was created.
   */
  chatEventForumTopicCreated();

  /**
   * A new forum topic was created.
   *
   * \param[in] topic_info_ Information about the topic.
   */
  explicit chatEventForumTopicCreated(object_ptr<forumTopicInfo> &&topic_info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2005269314;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A forum topic was edited.
 */
class chatEventForumTopicEdited final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Old information about the topic.
  object_ptr<forumTopicInfo> old_topic_info_;
  static jfieldID old_topic_info_fieldID;
  /// New information about the topic.
  object_ptr<forumTopicInfo> new_topic_info_;
  static jfieldID new_topic_info_fieldID;

  /**
   * A forum topic was edited.
   */
  chatEventForumTopicEdited();

  /**
   * A forum topic was edited.
   *
   * \param[in] old_topic_info_ Old information about the topic.
   * \param[in] new_topic_info_ New information about the topic.
   */
  chatEventForumTopicEdited(object_ptr<forumTopicInfo> &&old_topic_info_, object_ptr<forumTopicInfo> &&new_topic_info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1624910860;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A forum topic was closed or reopened.
 */
class chatEventForumTopicToggleIsClosed final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// New information about the topic.
  object_ptr<forumTopicInfo> topic_info_;
  static jfieldID topic_info_fieldID;

  /**
   * A forum topic was closed or reopened.
   */
  chatEventForumTopicToggleIsClosed();

  /**
   * A forum topic was closed or reopened.
   *
   * \param[in] topic_info_ New information about the topic.
   */
  explicit chatEventForumTopicToggleIsClosed(object_ptr<forumTopicInfo> &&topic_info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -962704070;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The General forum topic was hidden or unhidden.
 */
class chatEventForumTopicToggleIsHidden final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// New information about the topic.
  object_ptr<forumTopicInfo> topic_info_;
  static jfieldID topic_info_fieldID;

  /**
   * The General forum topic was hidden or unhidden.
   */
  chatEventForumTopicToggleIsHidden();

  /**
   * The General forum topic was hidden or unhidden.
   *
   * \param[in] topic_info_ New information about the topic.
   */
  explicit chatEventForumTopicToggleIsHidden(object_ptr<forumTopicInfo> &&topic_info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1609175250;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A forum topic was deleted.
 */
class chatEventForumTopicDeleted final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Information about the topic.
  object_ptr<forumTopicInfo> topic_info_;
  static jfieldID topic_info_fieldID;

  /**
   * A forum topic was deleted.
   */
  chatEventForumTopicDeleted();

  /**
   * A forum topic was deleted.
   *
   * \param[in] topic_info_ Information about the topic.
   */
  explicit chatEventForumTopicDeleted(object_ptr<forumTopicInfo> &&topic_info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1332795123;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A pinned forum topic was changed.
 */
class chatEventForumTopicPinned final : public ChatEventAction {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Information about the old pinned topic; may be null.
  object_ptr<forumTopicInfo> old_topic_info_;
  static jfieldID old_topic_info_fieldID;
  /// Information about the new pinned topic; may be null.
  object_ptr<forumTopicInfo> new_topic_info_;
  static jfieldID new_topic_info_fieldID;

  /**
   * A pinned forum topic was changed.
   */
  chatEventForumTopicPinned();

  /**
   * A pinned forum topic was changed.
   *
   * \param[in] old_topic_info_ Information about the old pinned topic; may be null.
   * \param[in] new_topic_info_ Information about the new pinned topic; may be null.
   */
  chatEventForumTopicPinned(object_ptr<forumTopicInfo> &&old_topic_info_, object_ptr<forumTopicInfo> &&new_topic_info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2143626222;

  static object_ptr<ChatEventAction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents a set of filters used to obtain a chat event log.
 */
class chatEventLogFilters final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// True, if message edits need to be returned.
  bool message_edits_;
  static jfieldID message_edits_fieldID;
  /// True, if message deletions need to be returned.
  bool message_deletions_;
  static jfieldID message_deletions_fieldID;
  /// True, if pin/unpin events need to be returned.
  bool message_pins_;
  static jfieldID message_pins_fieldID;
  /// True, if members joining events need to be returned.
  bool member_joins_;
  static jfieldID member_joins_fieldID;
  /// True, if members leaving events need to be returned.
  bool member_leaves_;
  static jfieldID member_leaves_fieldID;
  /// True, if invited member events need to be returned.
  bool member_invites_;
  static jfieldID member_invites_fieldID;
  /// True, if member promotion/demotion events need to be returned.
  bool member_promotions_;
  static jfieldID member_promotions_fieldID;
  /// True, if member restricted/unrestricted/banned/unbanned events need to be returned.
  bool member_restrictions_;
  static jfieldID member_restrictions_fieldID;
  /// True, if changes in chat information need to be returned.
  bool info_changes_;
  static jfieldID info_changes_fieldID;
  /// True, if changes in chat settings need to be returned.
  bool setting_changes_;
  static jfieldID setting_changes_fieldID;
  /// True, if changes to invite links need to be returned.
  bool invite_link_changes_;
  static jfieldID invite_link_changes_fieldID;
  /// True, if video chat actions need to be returned.
  bool video_chat_changes_;
  static jfieldID video_chat_changes_fieldID;
  /// True, if forum-related actions need to be returned.
  bool forum_changes_;
  static jfieldID forum_changes_fieldID;
  /// True, if subscription extensions need to be returned.
  bool subscription_extensions_;
  static jfieldID subscription_extensions_fieldID;

  /**
   * Represents a set of filters used to obtain a chat event log.
   */
  chatEventLogFilters();

  /**
   * Represents a set of filters used to obtain a chat event log.
   *
   * \param[in] message_edits_ True, if message edits need to be returned.
   * \param[in] message_deletions_ True, if message deletions need to be returned.
   * \param[in] message_pins_ True, if pin/unpin events need to be returned.
   * \param[in] member_joins_ True, if members joining events need to be returned.
   * \param[in] member_leaves_ True, if members leaving events need to be returned.
   * \param[in] member_invites_ True, if invited member events need to be returned.
   * \param[in] member_promotions_ True, if member promotion/demotion events need to be returned.
   * \param[in] member_restrictions_ True, if member restricted/unrestricted/banned/unbanned events need to be returned.
   * \param[in] info_changes_ True, if changes in chat information need to be returned.
   * \param[in] setting_changes_ True, if changes in chat settings need to be returned.
   * \param[in] invite_link_changes_ True, if changes to invite links need to be returned.
   * \param[in] video_chat_changes_ True, if video chat actions need to be returned.
   * \param[in] forum_changes_ True, if forum-related actions need to be returned.
   * \param[in] subscription_extensions_ True, if subscription extensions need to be returned.
   */
  chatEventLogFilters(bool message_edits_, bool message_deletions_, bool message_pins_, bool member_joins_, bool member_leaves_, bool member_invites_, bool member_promotions_, bool member_restrictions_, bool info_changes_, bool setting_changes_, bool invite_link_changes_, bool video_chat_changes_, bool forum_changes_, bool subscription_extensions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1032965711;

  static object_ptr<chatEventLogFilters> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class chatEvent;

/**
 * Contains a list of chat events.
 */
class chatEvents final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// List of events.
  array<object_ptr<chatEvent>> events_;
  static jfieldID events_fieldID;

  /**
   * Contains a list of chat events.
   */
  chatEvents();

  /**
   * Contains a list of chat events.
   *
   * \param[in] events_ List of events.
   */
  explicit chatEvents(array<object_ptr<chatEvent>> &&events_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -585329664;

  static object_ptr<chatEvents> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class chatFolderIcon;

class chatFolderName;

/**
 * Represents a folder for user chats.
 */
class chatFolder final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The name of the folder.
  object_ptr<chatFolderName> name_;
  static jfieldID name_fieldID;
  /// The chosen icon for the chat folder; may be null. If null, use getChatFolderDefaultIconName to get default icon name for the folder.
  object_ptr<chatFolderIcon> icon_;
  static jfieldID icon_fieldID;
  /// The identifier of the chosen color for the chat folder icon; from -1 to 6. If -1, then color is disabled. Can't be changed if folder tags are disabled or the current user doesn't have Telegram Premium subscription.
  int32 color_id_;
  static jfieldID color_id_fieldID;
  /// True, if at least one link has been created for the folder.
  bool is_shareable_;
  static jfieldID is_shareable_fieldID;
  /// The chat identifiers of pinned chats in the folder. There can be up to getOption(&quot;chat_folder_chosen_chat_count_max&quot;) pinned and always included non-secret chats and the same number of secret chats, but the limit can be increased with Telegram Premium.
  array<int53> pinned_chat_ids_;
  static jfieldID pinned_chat_ids_fieldID;
  /// The chat identifiers of always included chats in the folder. There can be up to getOption(&quot;chat_folder_chosen_chat_count_max&quot;) pinned and always included non-secret chats and the same number of secret chats, but the limit can be increased with Telegram Premium.
  array<int53> included_chat_ids_;
  static jfieldID included_chat_ids_fieldID;
  /// The chat identifiers of always excluded chats in the folder. There can be up to getOption(&quot;chat_folder_chosen_chat_count_max&quot;) always excluded non-secret chats and the same number of secret chats, but the limit can be increased with Telegram Premium.
  array<int53> excluded_chat_ids_;
  static jfieldID excluded_chat_ids_fieldID;
  /// True, if muted chats need to be excluded.
  bool exclude_muted_;
  static jfieldID exclude_muted_fieldID;
  /// True, if read chats need to be excluded.
  bool exclude_read_;
  static jfieldID exclude_read_fieldID;
  /// True, if archived chats need to be excluded.
  bool exclude_archived_;
  static jfieldID exclude_archived_fieldID;
  /// True, if contacts need to be included.
  bool include_contacts_;
  static jfieldID include_contacts_fieldID;
  /// True, if non-contact users need to be included.
  bool include_non_contacts_;
  static jfieldID include_non_contacts_fieldID;
  /// True, if bots need to be included.
  bool include_bots_;
  static jfieldID include_bots_fieldID;
  /// True, if basic groups and supergroups need to be included.
  bool include_groups_;
  static jfieldID include_groups_fieldID;
  /// True, if channels need to be included.
  bool include_channels_;
  static jfieldID include_channels_fieldID;

  /**
   * Represents a folder for user chats.
   */
  chatFolder();

  /**
   * Represents a folder for user chats.
   *
   * \param[in] name_ The name of the folder.
   * \param[in] icon_ The chosen icon for the chat folder; may be null. If null, use getChatFolderDefaultIconName to get default icon name for the folder.
   * \param[in] color_id_ The identifier of the chosen color for the chat folder icon; from -1 to 6. If -1, then color is disabled. Can't be changed if folder tags are disabled or the current user doesn't have Telegram Premium subscription.
   * \param[in] is_shareable_ True, if at least one link has been created for the folder.
   * \param[in] pinned_chat_ids_ The chat identifiers of pinned chats in the folder. There can be up to getOption(&quot;chat_folder_chosen_chat_count_max&quot;) pinned and always included non-secret chats and the same number of secret chats, but the limit can be increased with Telegram Premium.
   * \param[in] included_chat_ids_ The chat identifiers of always included chats in the folder. There can be up to getOption(&quot;chat_folder_chosen_chat_count_max&quot;) pinned and always included non-secret chats and the same number of secret chats, but the limit can be increased with Telegram Premium.
   * \param[in] excluded_chat_ids_ The chat identifiers of always excluded chats in the folder. There can be up to getOption(&quot;chat_folder_chosen_chat_count_max&quot;) always excluded non-secret chats and the same number of secret chats, but the limit can be increased with Telegram Premium.
   * \param[in] exclude_muted_ True, if muted chats need to be excluded.
   * \param[in] exclude_read_ True, if read chats need to be excluded.
   * \param[in] exclude_archived_ True, if archived chats need to be excluded.
   * \param[in] include_contacts_ True, if contacts need to be included.
   * \param[in] include_non_contacts_ True, if non-contact users need to be included.
   * \param[in] include_bots_ True, if bots need to be included.
   * \param[in] include_groups_ True, if basic groups and supergroups need to be included.
   * \param[in] include_channels_ True, if channels need to be included.
   */
  chatFolder(object_ptr<chatFolderName> &&name_, object_ptr<chatFolderIcon> &&icon_, int32 color_id_, bool is_shareable_, array<int53> &&pinned_chat_ids_, array<int53> &&included_chat_ids_, array<int53> &&excluded_chat_ids_, bool exclude_muted_, bool exclude_read_, bool exclude_archived_, bool include_contacts_, bool include_non_contacts_, bool include_bots_, bool include_groups_, bool include_channels_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1596164696;

  static object_ptr<chatFolder> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents an icon for a chat folder.
 */
class chatFolderIcon final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The chosen icon name for short folder representation; one of &quot;All&quot;, &quot;Unread&quot;, &quot;Unmuted&quot;, &quot;Bots&quot;, &quot;Channels&quot;, &quot;Groups&quot;, &quot;Private&quot;, &quot;Custom&quot;, &quot;Setup&quot;, &quot;Cat&quot;, &quot;Crown&quot;, &quot;Favorite&quot;, &quot;Flower&quot;, &quot;Game&quot;, &quot;Home&quot;, &quot;Love&quot;, &quot;Mask&quot;, &quot;Party&quot;, &quot;Sport&quot;, &quot;Study&quot;, &quot;Trade&quot;, &quot;Travel&quot;, &quot;Work&quot;, &quot;Airplane&quot;, &quot;Book&quot;, &quot;Light&quot;, &quot;Like&quot;, &quot;Money&quot;, &quot;Note&quot;, &quot;Palette&quot;.
  string name_;
  static jfieldID name_fieldID;

  /**
   * Represents an icon for a chat folder.
   */
  chatFolderIcon();

  /**
   * Represents an icon for a chat folder.
   *
   * \param[in] name_ The chosen icon name for short folder representation; one of &quot;All&quot;, &quot;Unread&quot;, &quot;Unmuted&quot;, &quot;Bots&quot;, &quot;Channels&quot;, &quot;Groups&quot;, &quot;Private&quot;, &quot;Custom&quot;, &quot;Setup&quot;, &quot;Cat&quot;, &quot;Crown&quot;, &quot;Favorite&quot;, &quot;Flower&quot;, &quot;Game&quot;, &quot;Home&quot;, &quot;Love&quot;, &quot;Mask&quot;, &quot;Party&quot;, &quot;Sport&quot;, &quot;Study&quot;, &quot;Trade&quot;, &quot;Travel&quot;, &quot;Work&quot;, &quot;Airplane&quot;, &quot;Book&quot;, &quot;Light&quot;, &quot;Like&quot;, &quot;Money&quot;, &quot;Note&quot;, &quot;Palette&quot;.
   */
  explicit chatFolderIcon(string const &name_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -146104090;

  static object_ptr<chatFolderIcon> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class chatFolderIcon;

class chatFolderName;

/**
 * Contains basic information about a chat folder.
 */
class chatFolderInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique chat folder identifier.
  int32 id_;
  static jfieldID id_fieldID;
  /// The name of the folder.
  object_ptr<chatFolderName> name_;
  static jfieldID name_fieldID;
  /// The chosen or default icon for the chat folder.
  object_ptr<chatFolderIcon> icon_;
  static jfieldID icon_fieldID;
  /// The identifier of the chosen color for the chat folder icon; from -1 to 6. If -1, then color is disabled.
  int32 color_id_;
  static jfieldID color_id_fieldID;
  /// True, if at least one link has been created for the folder.
  bool is_shareable_;
  static jfieldID is_shareable_fieldID;
  /// True, if the chat folder has invite links created by the current user.
  bool has_my_invite_links_;
  static jfieldID has_my_invite_links_fieldID;

  /**
   * Contains basic information about a chat folder.
   */
  chatFolderInfo();

  /**
   * Contains basic information about a chat folder.
   *
   * \param[in] id_ Unique chat folder identifier.
   * \param[in] name_ The name of the folder.
   * \param[in] icon_ The chosen or default icon for the chat folder.
   * \param[in] color_id_ The identifier of the chosen color for the chat folder icon; from -1 to 6. If -1, then color is disabled.
   * \param[in] is_shareable_ True, if at least one link has been created for the folder.
   * \param[in] has_my_invite_links_ True, if the chat folder has invite links created by the current user.
   */
  chatFolderInfo(int32 id_, object_ptr<chatFolderName> &&name_, object_ptr<chatFolderIcon> &&icon_, int32 color_id_, bool is_shareable_, bool has_my_invite_links_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 815535117;

  static object_ptr<chatFolderInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains a chat folder invite link.
 */
class chatFolderInviteLink final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The chat folder invite link.
  string invite_link_;
  static jfieldID invite_link_fieldID;
  /// Name of the link.
  string name_;
  static jfieldID name_fieldID;
  /// Identifiers of chats, included in the link.
  array<int53> chat_ids_;
  static jfieldID chat_ids_fieldID;

  /**
   * Contains a chat folder invite link.
   */
  chatFolderInviteLink();

  /**
   * Contains a chat folder invite link.
   *
   * \param[in] invite_link_ The chat folder invite link.
   * \param[in] name_ Name of the link.
   * \param[in] chat_ids_ Identifiers of chats, included in the link.
   */
  chatFolderInviteLink(string const &invite_link_, string const &name_, array<int53> &&chat_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 493969661;

  static object_ptr<chatFolderInviteLink> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class chatFolderInfo;

/**
 * Contains information about an invite link to a chat folder.
 */
class chatFolderInviteLinkInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Basic information about the chat folder; chat folder identifier will be 0 if the user didn't have the chat folder yet.
  object_ptr<chatFolderInfo> chat_folder_info_;
  static jfieldID chat_folder_info_fieldID;
  /// Identifiers of the chats from the link, which aren't added to the folder yet.
  array<int53> missing_chat_ids_;
  static jfieldID missing_chat_ids_fieldID;
  /// Identifiers of the chats from the link, which are added to the folder already.
  array<int53> added_chat_ids_;
  static jfieldID added_chat_ids_fieldID;

  /**
   * Contains information about an invite link to a chat folder.
   */
  chatFolderInviteLinkInfo();

  /**
   * Contains information about an invite link to a chat folder.
   *
   * \param[in] chat_folder_info_ Basic information about the chat folder; chat folder identifier will be 0 if the user didn't have the chat folder yet.
   * \param[in] missing_chat_ids_ Identifiers of the chats from the link, which aren't added to the folder yet.
   * \param[in] added_chat_ids_ Identifiers of the chats from the link, which are added to the folder already.
   */
  chatFolderInviteLinkInfo(object_ptr<chatFolderInfo> &&chat_folder_info_, array<int53> &&missing_chat_ids_, array<int53> &&added_chat_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1119450395;

  static object_ptr<chatFolderInviteLinkInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class chatFolderInviteLink;

/**
 * Represents a list of chat folder invite links.
 */
class chatFolderInviteLinks final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// List of the invite links.
  array<object_ptr<chatFolderInviteLink>> invite_links_;
  static jfieldID invite_links_fieldID;

  /**
   * Represents a list of chat folder invite links.
   */
  chatFolderInviteLinks();

  /**
   * Represents a list of chat folder invite links.
   *
   * \param[in] invite_links_ List of the invite links.
   */
  explicit chatFolderInviteLinks(array<object_ptr<chatFolderInviteLink>> &&invite_links_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1853351525;

  static object_ptr<chatFolderInviteLinks> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class formattedText;

/**
 * Describes name of a chat folder.
 */
class chatFolderName final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The text of the chat folder name; 1-12 characters without line feeds. May contain only CustomEmoji entities.
  object_ptr<formattedText> text_;
  static jfieldID text_fieldID;
  /// True, if custom emoji in the name must be animated.
  bool animate_custom_emoji_;
  static jfieldID animate_custom_emoji_fieldID;

  /**
   * Describes name of a chat folder.
   */
  chatFolderName();

  /**
   * Describes name of a chat folder.
   *
   * \param[in] text_ The text of the chat folder name; 1-12 characters without line feeds. May contain only CustomEmoji entities.
   * \param[in] animate_custom_emoji_ True, if custom emoji in the name must be animated.
   */
  chatFolderName(object_ptr<formattedText> &&text_, bool animate_custom_emoji_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -330482274;

  static object_ptr<chatFolderName> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class starSubscriptionPricing;

/**
 * Contains a chat invite link.
 */
class chatInviteLink final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Chat invite link.
  string invite_link_;
  static jfieldID invite_link_fieldID;
  /// Name of the link.
  string name_;
  static jfieldID name_fieldID;
  /// User identifier of an administrator created the link.
  int53 creator_user_id_;
  static jfieldID creator_user_id_fieldID;
  /// Point in time (Unix timestamp) when the link was created.
  int32 date_;
  static jfieldID date_fieldID;
  /// Point in time (Unix timestamp) when the link was last edited; 0 if never or unknown.
  int32 edit_date_;
  static jfieldID edit_date_fieldID;
  /// Point in time (Unix timestamp) when the link will expire; 0 if never.
  int32 expiration_date_;
  static jfieldID expiration_date_fieldID;
  /// Information about subscription plan that is applied to the users joining the chat by the link; may be null if the link doesn't require subscription.
  object_ptr<starSubscriptionPricing> subscription_pricing_;
  static jfieldID subscription_pricing_fieldID;
  /// The maximum number of members, which can join the chat using the link simultaneously; 0 if not limited. Always 0 if the link requires approval.
  int32 member_limit_;
  static jfieldID member_limit_fieldID;
  /// Number of chat members, which joined the chat using the link.
  int32 member_count_;
  static jfieldID member_count_fieldID;
  /// Number of chat members, which joined the chat using the link, but have already left because of expired subscription; for subscription links only.
  int32 expired_member_count_;
  static jfieldID expired_member_count_fieldID;
  /// Number of pending join requests created using this link.
  int32 pending_join_request_count_;
  static jfieldID pending_join_request_count_fieldID;
  /// True, if the link only creates join request. If true, total number of joining members will be unlimited.
  bool creates_join_request_;
  static jfieldID creates_join_request_fieldID;
  /// True, if the link is primary. Primary invite link can't have name, expiration date, or usage limit. There is exactly one primary invite link for each administrator with can_invite_users right at a given time.
  bool is_primary_;
  static jfieldID is_primary_fieldID;
  /// True, if the link was revoked.
  bool is_revoked_;
  static jfieldID is_revoked_fieldID;

  /**
   * Contains a chat invite link.
   */
  chatInviteLink();

  /**
   * Contains a chat invite link.
   *
   * \param[in] invite_link_ Chat invite link.
   * \param[in] name_ Name of the link.
   * \param[in] creator_user_id_ User identifier of an administrator created the link.
   * \param[in] date_ Point in time (Unix timestamp) when the link was created.
   * \param[in] edit_date_ Point in time (Unix timestamp) when the link was last edited; 0 if never or unknown.
   * \param[in] expiration_date_ Point in time (Unix timestamp) when the link will expire; 0 if never.
   * \param[in] subscription_pricing_ Information about subscription plan that is applied to the users joining the chat by the link; may be null if the link doesn't require subscription.
   * \param[in] member_limit_ The maximum number of members, which can join the chat using the link simultaneously; 0 if not limited. Always 0 if the link requires approval.
   * \param[in] member_count_ Number of chat members, which joined the chat using the link.
   * \param[in] expired_member_count_ Number of chat members, which joined the chat using the link, but have already left because of expired subscription; for subscription links only.
   * \param[in] pending_join_request_count_ Number of pending join requests created using this link.
   * \param[in] creates_join_request_ True, if the link only creates join request. If true, total number of joining members will be unlimited.
   * \param[in] is_primary_ True, if the link is primary. Primary invite link can't have name, expiration date, or usage limit. There is exactly one primary invite link for each administrator with can_invite_users right at a given time.
   * \param[in] is_revoked_ True, if the link was revoked.
   */
  chatInviteLink(string const &invite_link_, string const &name_, int53 creator_user_id_, int32 date_, int32 edit_date_, int32 expiration_date_, object_ptr<starSubscriptionPricing> &&subscription_pricing_, int32 member_limit_, int32 member_count_, int32 expired_member_count_, int32 pending_join_request_count_, bool creates_join_request_, bool is_primary_, bool is_revoked_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -957651664;

  static object_ptr<chatInviteLink> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes a chat administrator with a number of active and revoked chat invite links.
 */
class chatInviteLinkCount final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Administrator's user identifier.
  int53 user_id_;
  static jfieldID user_id_fieldID;
  /// Number of active invite links.
  int32 invite_link_count_;
  static jfieldID invite_link_count_fieldID;
  /// Number of revoked invite links.
  int32 revoked_invite_link_count_;
  static jfieldID revoked_invite_link_count_fieldID;

  /**
   * Describes a chat administrator with a number of active and revoked chat invite links.
   */
  chatInviteLinkCount();

  /**
   * Describes a chat administrator with a number of active and revoked chat invite links.
   *
   * \param[in] user_id_ Administrator's user identifier.
   * \param[in] invite_link_count_ Number of active invite links.
   * \param[in] revoked_invite_link_count_ Number of revoked invite links.
   */
  chatInviteLinkCount(int53 user_id_, int32 invite_link_count_, int32 revoked_invite_link_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1021999210;

  static object_ptr<chatInviteLinkCount> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class chatInviteLinkCount;

/**
 * Contains a list of chat invite link counts.
 */
class chatInviteLinkCounts final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// List of invite link counts.
  array<object_ptr<chatInviteLinkCount>> invite_link_counts_;
  static jfieldID invite_link_counts_fieldID;

  /**
   * Contains a list of chat invite link counts.
   */
  chatInviteLinkCounts();

  /**
   * Contains a list of chat invite link counts.
   *
   * \param[in] invite_link_counts_ List of invite link counts.
   */
  explicit chatInviteLinkCounts(array<object_ptr<chatInviteLinkCount>> &&invite_link_counts_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 920326637;

  static object_ptr<chatInviteLinkCounts> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class InviteLinkChatType;

class chatInviteLinkSubscriptionInfo;

class chatPhotoInfo;

class verificationStatus;

/**
 * Contains information about a chat invite link.
 */
class chatInviteLinkInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Chat identifier of the invite link; 0 if the user has no access to the chat before joining.
  int53 chat_id_;
  static jfieldID chat_id_fieldID;
  /// If non-zero, the amount of time for which read access to the chat will remain available, in seconds.
  int32 accessible_for_;
  static jfieldID accessible_for_fieldID;
  /// Type of the chat.
  object_ptr<InviteLinkChatType> type_;
  static jfieldID type_fieldID;
  /// Title of the chat.
  string title_;
  static jfieldID title_fieldID;
  /// Chat photo; may be null.
  object_ptr<chatPhotoInfo> photo_;
  static jfieldID photo_fieldID;
  /// Identifier of the accent color for chat title and background of chat photo.
  int32 accent_color_id_;
  static jfieldID accent_color_id_fieldID;
  /// Chat description.
  string description_;
  static jfieldID description_fieldID;
  /// Number of members in the chat.
  int32 member_count_;
  static jfieldID member_count_fieldID;
  /// User identifiers of some chat members that may be known to the current user.
  array<int53> member_user_ids_;
  static jfieldID member_user_ids_fieldID;
  /// Information about subscription plan that must be paid by the user to use the link; may be null if the link doesn't require subscription.
  object_ptr<chatInviteLinkSubscriptionInfo> subscription_info_;
  static jfieldID subscription_info_fieldID;
  /// True, if the link only creates join request.
  bool creates_join_request_;
  static jfieldID creates_join_request_fieldID;
  /// True, if the chat is a public supergroup or channel, i.e. it has a username or it is a location-based supergroup.
  bool is_public_;
  static jfieldID is_public_fieldID;
  /// Information about verification status of the chat; may be null if none.
  object_ptr<verificationStatus> verification_status_;
  static jfieldID verification_status_fieldID;

  /**
   * Contains information about a chat invite link.
   */
  chatInviteLinkInfo();

  /**
   * Contains information about a chat invite link.
   *
   * \param[in] chat_id_ Chat identifier of the invite link; 0 if the user has no access to the chat before joining.
   * \param[in] accessible_for_ If non-zero, the amount of time for which read access to the chat will remain available, in seconds.
   * \param[in] type_ Type of the chat.
   * \param[in] title_ Title of the chat.
   * \param[in] photo_ Chat photo; may be null.
   * \param[in] accent_color_id_ Identifier of the accent color for chat title and background of chat photo.
   * \param[in] description_ Chat description.
   * \param[in] member_count_ Number of members in the chat.
   * \param[in] member_user_ids_ User identifiers of some chat members that may be known to the current user.
   * \param[in] subscription_info_ Information about subscription plan that must be paid by the user to use the link; may be null if the link doesn't require subscription.
   * \param[in] creates_join_request_ True, if the link only creates join request.
   * \param[in] is_public_ True, if the chat is a public supergroup or channel, i.e. it has a username or it is a location-based supergroup.
   * \param[in] verification_status_ Information about verification status of the chat; may be null if none.
   */
  chatInviteLinkInfo(int53 chat_id_, int32 accessible_for_, object_ptr<InviteLinkChatType> &&type_, string const &title_, object_ptr<chatPhotoInfo> &&photo_, int32 accent_color_id_, string const &description_, int32 member_count_, array<int53> &&member_user_ids_, object_ptr<chatInviteLinkSubscriptionInfo> &&subscription_info_, bool creates_join_request_, bool is_public_, object_ptr<verificationStatus> &&verification_status_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1145310535;

  static object_ptr<chatInviteLinkInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes a chat member joined a chat via an invite link.
 */
class chatInviteLinkMember final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// User identifier.
  int53 user_id_;
  static jfieldID user_id_fieldID;
  /// Point in time (Unix timestamp) when the user joined the chat.
  int32 joined_chat_date_;
  static jfieldID joined_chat_date_fieldID;
  /// True, if the user has joined the chat using an invite link for a chat folder.
  bool via_chat_folder_invite_link_;
  static jfieldID via_chat_folder_invite_link_fieldID;
  /// User identifier of the chat administrator, approved user join request.
  int53 approver_user_id_;
  static jfieldID approver_user_id_fieldID;

  /**
   * Describes a chat member joined a chat via an invite link.
   */
  chatInviteLinkMember();

  /**
   * Describes a chat member joined a chat via an invite link.
   *
   * \param[in] user_id_ User identifier.
   * \param[in] joined_chat_date_ Point in time (Unix timestamp) when the user joined the chat.
   * \param[in] via_chat_folder_invite_link_ True, if the user has joined the chat using an invite link for a chat folder.
   * \param[in] approver_user_id_ User identifier of the chat administrator, approved user join request.
   */
  chatInviteLinkMember(int53 user_id_, int32 joined_chat_date_, bool via_chat_folder_invite_link_, int53 approver_user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 29156795;

  static object_ptr<chatInviteLinkMember> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class chatInviteLinkMember;

/**
 * Contains a list of chat members joined a chat via an invite link.
 */
class chatInviteLinkMembers final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Approximate total number of chat members found.
  int32 total_count_;
  static jfieldID total_count_fieldID;
  /// List of chat members, joined a chat via an invite link.
  array<object_ptr<chatInviteLinkMember>> members_;
  static jfieldID members_fieldID;

  /**
   * Contains a list of chat members joined a chat via an invite link.
   */
  chatInviteLinkMembers();

  /**
   * Contains a list of chat members joined a chat via an invite link.
   *
   * \param[in] total_count_ Approximate total number of chat members found.
   * \param[in] members_ List of chat members, joined a chat via an invite link.
   */
  chatInviteLinkMembers(int32 total_count_, array<object_ptr<chatInviteLinkMember>> &&members_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 315635051;

  static object_ptr<chatInviteLinkMembers> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class starSubscriptionPricing;

/**
 * Contains information about subscription plan that must be paid by the user to use a chat invite link.
 */
class chatInviteLinkSubscriptionInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Information about subscription plan that must be paid by the user to use the link.
  object_ptr<starSubscriptionPricing> pricing_;
  static jfieldID pricing_fieldID;
  /// True, if the user has already paid for the subscription and can use joinChatByInviteLink to join the subscribed chat again.
  bool can_reuse_;
  static jfieldID can_reuse_fieldID;
  /// Identifier of the payment form to use for subscription payment; 0 if the subscription can't be paid.
  int64 form_id_;
  static jfieldID form_id_fieldID;

  /**
   * Contains information about subscription plan that must be paid by the user to use a chat invite link.
   */
  chatInviteLinkSubscriptionInfo();

  /**
   * Contains information about subscription plan that must be paid by the user to use a chat invite link.
   *
   * \param[in] pricing_ Information about subscription plan that must be paid by the user to use the link.
   * \param[in] can_reuse_ True, if the user has already paid for the subscription and can use joinChatByInviteLink to join the subscribed chat again.
   * \param[in] form_id_ Identifier of the payment form to use for subscription payment; 0 if the subscription can't be paid.
   */
  chatInviteLinkSubscriptionInfo(object_ptr<starSubscriptionPricing> &&pricing_, bool can_reuse_, int64 form_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 953119592;

  static object_ptr<chatInviteLinkSubscriptionInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class chatInviteLink;

/**
 * Contains a list of chat invite links.
 */
class chatInviteLinks final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Approximate total number of chat invite links found.
  int32 total_count_;
  static jfieldID total_count_fieldID;
  /// List of invite links.
  array<object_ptr<chatInviteLink>> invite_links_;
  static jfieldID invite_links_fieldID;

  /**
   * Contains a list of chat invite links.
   */
  chatInviteLinks();

  /**
   * Contains a list of chat invite links.
   *
   * \param[in] total_count_ Approximate total number of chat invite links found.
   * \param[in] invite_links_ List of invite links.
   */
  chatInviteLinks(int32 total_count_, array<object_ptr<chatInviteLink>> &&invite_links_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 112891427;

  static object_ptr<chatInviteLinks> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes a user that sent a join request and waits for administrator approval.
 */
class chatJoinRequest final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// User identifier.
  int53 user_id_;
  static jfieldID user_id_fieldID;
  /// Point in time (Unix timestamp) when the user sent the join request.
  int32 date_;
  static jfieldID date_fieldID;
  /// A short bio of the user.
  string bio_;
  static jfieldID bio_fieldID;

  /**
   * Describes a user that sent a join request and waits for administrator approval.
   */
  chatJoinRequest();

  /**
   * Describes a user that sent a join request and waits for administrator approval.
   *
   * \param[in] user_id_ User identifier.
   * \param[in] date_ Point in time (Unix timestamp) when the user sent the join request.
   * \param[in] bio_ A short bio of the user.
   */
  chatJoinRequest(int53 user_id_, int32 date_, string const &bio_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 59341416;

  static object_ptr<chatJoinRequest> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class chatJoinRequest;

/**
 * Contains a list of requests to join a chat.
 */
class chatJoinRequests final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Approximate total number of requests found.
  int32 total_count_;
  static jfieldID total_count_fieldID;
  /// List of the requests.
  array<object_ptr<chatJoinRequest>> requests_;
  static jfieldID requests_fieldID;

  /**
   * Contains a list of requests to join a chat.
   */
  chatJoinRequests();

  /**
   * Contains a list of requests to join a chat.
   *
   * \param[in] total_count_ Approximate total number of requests found.
   * \param[in] requests_ List of the requests.
   */
  chatJoinRequests(int32 total_count_, array<object_ptr<chatJoinRequest>> &&requests_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1291680519;

  static object_ptr<chatJoinRequests> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains information about pending join requests for a chat.
 */
class chatJoinRequestsInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Total number of pending join requests.
  int32 total_count_;
  static jfieldID total_count_fieldID;
  /// Identifiers of at most 3 users sent the newest pending join requests.
  array<int53> user_ids_;
  static jfieldID user_ids_fieldID;

  /**
   * Contains information about pending join requests for a chat.
   */
  chatJoinRequestsInfo();

  /**
   * Contains information about pending join requests for a chat.
   *
   * \param[in] total_count_ Total number of pending join requests.
   * \param[in] user_ids_ Identifiers of at most 3 users sent the newest pending join requests.
   */
  chatJoinRequestsInfo(int32 total_count_, array<int53> &&user_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 888534463;

  static object_ptr<chatJoinRequestsInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes a list of chats.
 */
class ChatList: public Object {
 public:
  static jclass Class;

  static object_ptr<ChatList> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A main list of chats.
 */
class chatListMain final : public ChatList {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * A main list of chats.
   */
  chatListMain();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -400991316;

  static object_ptr<ChatList> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A list of chats usually located at the top of the main chat list. Unmuted chats are automatically moved from the Archive to the Main chat list when a new message arrives.
 */
class chatListArchive final : public ChatList {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * A list of chats usually located at the top of the main chat list. Unmuted chats are automatically moved from the Archive to the Main chat list when a new message arrives.
   */
  chatListArchive();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 362770115;

  static object_ptr<ChatList> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A list of chats added to a chat folder.
 */
class chatListFolder final : public ChatList {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Chat folder identifier.
  int32 chat_folder_id_;
  static jfieldID chat_folder_id_fieldID;

  /**
   * A list of chats added to a chat folder.
   */
  chatListFolder();

  /**
   * A list of chats added to a chat folder.
   *
   * \param[in] chat_folder_id_ Chat folder identifier.
   */
  explicit chatListFolder(int32 chat_folder_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 385760856;

  static object_ptr<ChatList> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class ChatList;

/**
 * Contains a list of chat lists.
 */
class chatLists final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// List of chat lists.
  array<object_ptr<ChatList>> chat_lists_;
  static jfieldID chat_lists_fieldID;

  /**
   * Contains a list of chat lists.
   */
  chatLists();

  /**
   * Contains a list of chat lists.
   *
   * \param[in] chat_lists_ List of chat lists.
   */
  explicit chatLists(array<object_ptr<ChatList>> &&chat_lists_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -258292771;

  static object_ptr<chatLists> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class location;

/**
 * Represents a location to which a chat is connected.
 */
class chatLocation final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The location.
  object_ptr<location> location_;
  static jfieldID location_fieldID;
  /// Location address; 1-64 characters, as defined by the chat owner.
  string address_;
  static jfieldID address_fieldID;

  /**
   * Represents a location to which a chat is connected.
   */
  chatLocation();

  /**
   * Represents a location to which a chat is connected.
   *
   * \param[in] location_ The location.
   * \param[in] address_ Location address; 1-64 characters, as defined by the chat owner.
   */
  chatLocation(object_ptr<location> &&location_, string const &address_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1566863583;

  static object_ptr<chatLocation> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class ChatMemberStatus;

class MessageSender;

/**
 * Describes a user or a chat as a member of another chat.
 */
class chatMember final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the chat member. Currently, other chats can be only Left or Banned. Only supergroups and channels can have other chats as Left or Banned members and these chats must be supergroups or channels.
  object_ptr<MessageSender> member_id_;
  static jfieldID member_id_fieldID;
  /// Identifier of a user that invited/promoted/banned this member in the chat; 0 if unknown.
  int53 inviter_user_id_;
  static jfieldID inviter_user_id_fieldID;
  /// Point in time (Unix timestamp) when the user joined/was promoted/was banned in the chat.
  int32 joined_chat_date_;
  static jfieldID joined_chat_date_fieldID;
  /// Status of the member in the chat.
  object_ptr<ChatMemberStatus> status_;
  static jfieldID status_fieldID;

  /**
   * Describes a user or a chat as a member of another chat.
   */
  chatMember();

  /**
   * Describes a user or a chat as a member of another chat.
   *
   * \param[in] member_id_ Identifier of the chat member. Currently, other chats can be only Left or Banned. Only supergroups and channels can have other chats as Left or Banned members and these chats must be supergroups or channels.
   * \param[in] inviter_user_id_ Identifier of a user that invited/promoted/banned this member in the chat; 0 if unknown.
   * \param[in] joined_chat_date_ Point in time (Unix timestamp) when the user joined/was promoted/was banned in the chat.
   * \param[in] status_ Status of the member in the chat.
   */
  chatMember(object_ptr<MessageSender> &&member_id_, int53 inviter_user_id_, int32 joined_chat_date_, object_ptr<ChatMemberStatus> &&status_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1829953909;

  static object_ptr<chatMember> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class chatAdministratorRights;

class chatPermissions;

/**
 * This class is an abstract base class.
 * Provides information about the status of a member in a chat.
 */
class ChatMemberStatus: public Object {
 public:
  static jclass Class;

  static object_ptr<ChatMemberStatus> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is the owner of the chat and has all the administrator privileges.
 */
class chatMemberStatusCreator final : public ChatMemberStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A custom title of the owner; 0-16 characters without emoji; applicable to supergroups only.
  string custom_title_;
  static jfieldID custom_title_fieldID;
  /// True, if the creator isn't shown in the chat member list and sends messages anonymously; applicable to supergroups only.
  bool is_anonymous_;
  static jfieldID is_anonymous_fieldID;
  /// True, if the user is a member of the chat.
  bool is_member_;
  static jfieldID is_member_fieldID;

  /**
   * The user is the owner of the chat and has all the administrator privileges.
   */
  chatMemberStatusCreator();

  /**
   * The user is the owner of the chat and has all the administrator privileges.
   *
   * \param[in] custom_title_ A custom title of the owner; 0-16 characters without emoji; applicable to supergroups only.
   * \param[in] is_anonymous_ True, if the creator isn't shown in the chat member list and sends messages anonymously; applicable to supergroups only.
   * \param[in] is_member_ True, if the user is a member of the chat.
   */
  chatMemberStatusCreator(string const &custom_title_, bool is_anonymous_, bool is_member_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -160019714;

  static object_ptr<ChatMemberStatus> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is a member of the chat and has some additional privileges. In basic groups, administrators can edit and delete messages sent by others, add new members, ban unprivileged members, and manage video chats. In supergroups and channels, there are more detailed options for administrator privileges.
 */
class chatMemberStatusAdministrator final : public ChatMemberStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A custom title of the administrator; 0-16 characters without emoji; applicable to supergroups only.
  string custom_title_;
  static jfieldID custom_title_fieldID;
  /// True, if the current user can edit the administrator privileges for the called user.
  bool can_be_edited_;
  static jfieldID can_be_edited_fieldID;
  /// Rights of the administrator.
  object_ptr<chatAdministratorRights> rights_;
  static jfieldID rights_fieldID;

  /**
   * The user is a member of the chat and has some additional privileges. In basic groups, administrators can edit and delete messages sent by others, add new members, ban unprivileged members, and manage video chats. In supergroups and channels, there are more detailed options for administrator privileges.
   */
  chatMemberStatusAdministrator();

  /**
   * The user is a member of the chat and has some additional privileges. In basic groups, administrators can edit and delete messages sent by others, add new members, ban unprivileged members, and manage video chats. In supergroups and channels, there are more detailed options for administrator privileges.
   *
   * \param[in] custom_title_ A custom title of the administrator; 0-16 characters without emoji; applicable to supergroups only.
   * \param[in] can_be_edited_ True, if the current user can edit the administrator privileges for the called user.
   * \param[in] rights_ Rights of the administrator.
   */
  chatMemberStatusAdministrator(string const &custom_title_, bool can_be_edited_, object_ptr<chatAdministratorRights> &&rights_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -70024163;

  static object_ptr<ChatMemberStatus> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is a member of the chat, without any additional privileges or restrictions.
 */
class chatMemberStatusMember final : public ChatMemberStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Point in time (Unix timestamp) when the user will be removed from the chat because of the expired subscription; 0 if never. Ignored in setChatMemberStatus.
  int32 member_until_date_;
  static jfieldID member_until_date_fieldID;

  /**
   * The user is a member of the chat, without any additional privileges or restrictions.
   */
  chatMemberStatusMember();

  /**
   * The user is a member of the chat, without any additional privileges or restrictions.
   *
   * \param[in] member_until_date_ Point in time (Unix timestamp) when the user will be removed from the chat because of the expired subscription; 0 if never. Ignored in setChatMemberStatus.
   */
  explicit chatMemberStatusMember(int32 member_until_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -32707562;

  static object_ptr<ChatMemberStatus> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is under certain restrictions in the chat. Not supported in basic groups and channels.
 */
class chatMemberStatusRestricted final : public ChatMemberStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// True, if the user is a member of the chat.
  bool is_member_;
  static jfieldID is_member_fieldID;
  /// Point in time (Unix timestamp) when restrictions will be lifted from the user; 0 if never. If the user is restricted for more than 366 days or for less than 30 seconds from the current time, the user is considered to be restricted forever.
  int32 restricted_until_date_;
  static jfieldID restricted_until_date_fieldID;
  /// User permissions in the chat.
  object_ptr<chatPermissions> permissions_;
  static jfieldID permissions_fieldID;

  /**
   * The user is under certain restrictions in the chat. Not supported in basic groups and channels.
   */
  chatMemberStatusRestricted();

  /**
   * The user is under certain restrictions in the chat. Not supported in basic groups and channels.
   *
   * \param[in] is_member_ True, if the user is a member of the chat.
   * \param[in] restricted_until_date_ Point in time (Unix timestamp) when restrictions will be lifted from the user; 0 if never. If the user is restricted for more than 366 days or for less than 30 seconds from the current time, the user is considered to be restricted forever.
   * \param[in] permissions_ User permissions in the chat.
   */
  chatMemberStatusRestricted(bool is_member_, int32 restricted_until_date_, object_ptr<chatPermissions> &&permissions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1661432998;

  static object_ptr<ChatMemberStatus> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user or the chat is not a chat member.
 */
class chatMemberStatusLeft final : public ChatMemberStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user or the chat is not a chat member.
   */
  chatMemberStatusLeft();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -5815259;

  static object_ptr<ChatMemberStatus> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user or the chat was banned (and hence is not a member of the chat). Implies the user can't return to the chat, view messages, or be used as a participant identifier to join a video chat of the chat.
 */
class chatMemberStatusBanned final : public ChatMemberStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Point in time (Unix timestamp) when the user will be unbanned; 0 if never. If the user is banned for more than 366 days or for less than 30 seconds from the current time, the user is considered to be banned forever. Always 0 in basic groups.
  int32 banned_until_date_;
  static jfieldID banned_until_date_fieldID;

  /**
   * The user or the chat was banned (and hence is not a member of the chat). Implies the user can't return to the chat, view messages, or be used as a participant identifier to join a video chat of the chat.
   */
  chatMemberStatusBanned();

  /**
   * The user or the chat was banned (and hence is not a member of the chat). Implies the user can't return to the chat, view messages, or be used as a participant identifier to join a video chat of the chat.
   *
   * \param[in] banned_until_date_ Point in time (Unix timestamp) when the user will be unbanned; 0 if never. If the user is banned for more than 366 days or for less than 30 seconds from the current time, the user is considered to be banned forever. Always 0 in basic groups.
   */
  explicit chatMemberStatusBanned(int32 banned_until_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1653518666;

  static object_ptr<ChatMemberStatus> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class chatMember;

/**
 * Contains a list of chat members.
 */
class chatMembers final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Approximate total number of chat members found.
  int32 total_count_;
  static jfieldID total_count_fieldID;
  /// A list of chat members.
  array<object_ptr<chatMember>> members_;
  static jfieldID members_fieldID;

  /**
   * Contains a list of chat members.
   */
  chatMembers();

  /**
   * Contains a list of chat members.
   *
   * \param[in] total_count_ Approximate total number of chat members found.
   * \param[in] members_ A list of chat members.
   */
  chatMembers(int32 total_count_, array<object_ptr<chatMember>> &&members_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -497558622;

  static object_ptr<chatMembers> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Specifies the kind of chat members to return in searchChatMembers.
 */
class ChatMembersFilter: public Object {
 public:
  static jclass Class;

  static object_ptr<ChatMembersFilter> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Returns contacts of the user.
 */
class chatMembersFilterContacts final : public ChatMembersFilter {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * Returns contacts of the user.
   */
  chatMembersFilterContacts();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1774485671;

  static object_ptr<ChatMembersFilter> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Returns the owner and administrators.
 */
class chatMembersFilterAdministrators final : public ChatMembersFilter {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * Returns the owner and administrators.
   */
  chatMembersFilterAdministrators();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1266893796;

  static object_ptr<ChatMembersFilter> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Returns all chat members, including restricted chat members.
 */
class chatMembersFilterMembers final : public ChatMembersFilter {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * Returns all chat members, including restricted chat members.
   */
  chatMembersFilterMembers();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 670504342;

  static object_ptr<ChatMembersFilter> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Returns users which can be mentioned in the chat.
 */
class chatMembersFilterMention final : public ChatMembersFilter {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// If non-zero, the identifier of the current message thread.
  int53 message_thread_id_;
  static jfieldID message_thread_id_fieldID;

  /**
   * Returns users which can be mentioned in the chat.
   */
  chatMembersFilterMention();

  /**
   * Returns users which can be mentioned in the chat.
   *
   * \param[in] message_thread_id_ If non-zero, the identifier of the current message thread.
   */
  explicit chatMembersFilterMention(int53 message_thread_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 856419831;

  static object_ptr<ChatMembersFilter> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Returns users under certain restrictions in the chat; can be used only by administrators in a supergroup.
 */
class chatMembersFilterRestricted final : public ChatMembersFilter {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * Returns users under certain restrictions in the chat; can be used only by administrators in a supergroup.
   */
  chatMembersFilterRestricted();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1256282813;

  static object_ptr<ChatMembersFilter> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Returns users banned from the chat; can be used only by administrators in a supergroup or in a channel.
 */
class chatMembersFilterBanned final : public ChatMembersFilter {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * Returns users banned from the chat; can be used only by administrators in a supergroup or in a channel.
   */
  chatMembersFilterBanned();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1863102648;

  static object_ptr<ChatMembersFilter> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Returns bot members of the chat.
 */
class chatMembersFilterBots final : public ChatMembersFilter {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * Returns bot members of the chat.
   */
  chatMembersFilterBots();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1422567288;

  static object_ptr<ChatMembersFilter> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class MessageSender;

/**
 * Represents a message sender, which can be used to send messages in a chat.
 */
class chatMessageSender final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The message sender.
  object_ptr<MessageSender> sender_;
  static jfieldID sender_fieldID;
  /// True, if Telegram Premium is needed to use the message sender.
  bool needs_premium_;
  static jfieldID needs_premium_fieldID;

  /**
   * Represents a message sender, which can be used to send messages in a chat.
   */
  chatMessageSender();

  /**
   * Represents a message sender, which can be used to send messages in a chat.
   *
   * \param[in] sender_ The message sender.
   * \param[in] needs_premium_ True, if Telegram Premium is needed to use the message sender.
   */
  chatMessageSender(object_ptr<MessageSender> &&sender_, bool needs_premium_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 760590010;

  static object_ptr<chatMessageSender> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class chatMessageSender;

/**
 * Represents a list of message senders, which can be used to send messages in a chat.
 */
class chatMessageSenders final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// List of available message senders.
  array<object_ptr<chatMessageSender>> senders_;
  static jfieldID senders_fieldID;

  /**
   * Represents a list of message senders, which can be used to send messages in a chat.
   */
  chatMessageSenders();

  /**
   * Represents a list of message senders, which can be used to send messages in a chat.
   *
   * \param[in] senders_ List of available message senders.
   */
  explicit chatMessageSenders(array<object_ptr<chatMessageSender>> &&senders_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1866230970;

  static object_ptr<chatMessageSenders> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains information about notification settings for a chat or a forum topic.
 */
class chatNotificationSettings final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// If true, the value for the relevant type of chat or the forum chat is used instead of mute_for.
  bool use_default_mute_for_;
  static jfieldID use_default_mute_for_fieldID;
  /// Time left before notifications will be unmuted, in seconds.
  int32 mute_for_;
  static jfieldID mute_for_fieldID;
  /// If true, the value for the relevant type of chat or the forum chat is used instead of sound_id.
  bool use_default_sound_;
  static jfieldID use_default_sound_fieldID;
  /// Identifier of the notification sound to be played for messages; 0 if sound is disabled.
  int64 sound_id_;
  static jfieldID sound_id_fieldID;
  /// If true, the value for the relevant type of chat or the forum chat is used instead of show_preview.
  bool use_default_show_preview_;
  static jfieldID use_default_show_preview_fieldID;
  /// True, if message content must be displayed in notifications.
  bool show_preview_;
  static jfieldID show_preview_fieldID;
  /// If true, the value for the relevant type of chat is used instead of mute_stories.
  bool use_default_mute_stories_;
  static jfieldID use_default_mute_stories_fieldID;
  /// True, if story notifications are disabled for the chat.
  bool mute_stories_;
  static jfieldID mute_stories_fieldID;
  /// If true, the value for the relevant type of chat is used instead of story_sound_id.
  bool use_default_story_sound_;
  static jfieldID use_default_story_sound_fieldID;
  /// Identifier of the notification sound to be played for stories; 0 if sound is disabled.
  int64 story_sound_id_;
  static jfieldID story_sound_id_fieldID;
  /// If true, the value for the relevant type of chat is used instead of show_story_poster.
  bool use_default_show_story_poster_;
  static jfieldID use_default_show_story_poster_fieldID;
  /// True, if the chat that posted a story must be displayed in notifications.
  bool show_story_poster_;
  static jfieldID show_story_poster_fieldID;
  /// If true, the value for the relevant type of chat or the forum chat is used instead of disable_pinned_message_notifications.
  bool use_default_disable_pinned_message_notifications_;
  static jfieldID use_default_disable_pinned_message_notifications_fieldID;
  /// If true, notifications for incoming pinned messages will be created as for an ordinary unread message.
  bool disable_pinned_message_notifications_;
  static jfieldID disable_pinned_message_notifications_fieldID;
  /// If true, the value for the relevant type of chat or the forum chat is used instead of disable_mention_notifications.
  bool use_default_disable_mention_notifications_;
  static jfieldID use_default_disable_mention_notifications_fieldID;
  /// If true, notifications for messages with mentions will be created as for an ordinary unread message.
  bool disable_mention_notifications_;
  static jfieldID disable_mention_notifications_fieldID;

  /**
   * Contains information about notification settings for a chat or a forum topic.
   */
  chatNotificationSettings();

  /**
   * Contains information about notification settings for a chat or a forum topic.
   *
   * \param[in] use_default_mute_for_ If true, the value for the relevant type of chat or the forum chat is used instead of mute_for.
   * \param[in] mute_for_ Time left before notifications will be unmuted, in seconds.
   * \param[in] use_default_sound_ If true, the value for the relevant type of chat or the forum chat is used instead of sound_id.
   * \param[in] sound_id_ Identifier of the notification sound to be played for messages; 0 if sound is disabled.
   * \param[in] use_default_show_preview_ If true, the value for the relevant type of chat or the forum chat is used instead of show_preview.
   * \param[in] show_preview_ True, if message content must be displayed in notifications.
   * \param[in] use_default_mute_stories_ If true, the value for the relevant type of chat is used instead of mute_stories.
   * \param[in] mute_stories_ True, if story notifications are disabled for the chat.
   * \param[in] use_default_story_sound_ If true, the value for the relevant type of chat is used instead of story_sound_id.
   * \param[in] story_sound_id_ Identifier of the notification sound to be played for stories; 0 if sound is disabled.
   * \param[in] use_default_show_story_poster_ If true, the value for the relevant type of chat is used instead of show_story_poster.
   * \param[in] show_story_poster_ True, if the chat that posted a story must be displayed in notifications.
   * \param[in] use_default_disable_pinned_message_notifications_ If true, the value for the relevant type of chat or the forum chat is used instead of disable_pinned_message_notifications.
   * \param[in] disable_pinned_message_notifications_ If true, notifications for incoming pinned messages will be created as for an ordinary unread message.
   * \param[in] use_default_disable_mention_notifications_ If true, the value for the relevant type of chat or the forum chat is used instead of disable_mention_notifications.
   * \param[in] disable_mention_notifications_ If true, notifications for messages with mentions will be created as for an ordinary unread message.
   */
  chatNotificationSettings(bool use_default_mute_for_, int32 mute_for_, bool use_default_sound_, int64 sound_id_, bool use_default_show_preview_, bool show_preview_, bool use_default_mute_stories_, bool mute_stories_, bool use_default_story_sound_, int64 story_sound_id_, bool use_default_show_story_poster_, bool show_story_poster_, bool use_default_disable_pinned_message_notifications_, bool disable_pinned_message_notifications_, bool use_default_disable_mention_notifications_, bool disable_mention_notifications_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1459533846;

  static object_ptr<chatNotificationSettings> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes actions that a user is allowed to take in a chat.
 */
class chatPermissions final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// True, if the user can send text messages, contacts, giveaways, giveaway winners, invoices, locations, and venues.
  bool can_send_basic_messages_;
  static jfieldID can_send_basic_messages_fieldID;
  /// True, if the user can send music files.
  bool can_send_audios_;
  static jfieldID can_send_audios_fieldID;
  /// True, if the user can send documents.
  bool can_send_documents_;
  static jfieldID can_send_documents_fieldID;
  /// True, if the user can send photos.
  bool can_send_photos_;
  static jfieldID can_send_photos_fieldID;
  /// True, if the user can send videos.
  bool can_send_videos_;
  static jfieldID can_send_videos_fieldID;
  /// True, if the user can send video notes.
  bool can_send_video_notes_;
  static jfieldID can_send_video_notes_fieldID;
  /// True, if the user can send voice notes.
  bool can_send_voice_notes_;
  static jfieldID can_send_voice_notes_fieldID;
  /// True, if the user can send polls and checklists.
  bool can_send_polls_;
  static jfieldID can_send_polls_fieldID;
  /// True, if the user can send animations, games, stickers, and dice and use inline bots.
  bool can_send_other_messages_;
  static jfieldID can_send_other_messages_fieldID;
  /// True, if the user may add a link preview to their messages.
  bool can_add_link_previews_;
  static jfieldID can_add_link_previews_fieldID;
  /// True, if the user can change the chat title, photo, and other settings.
  bool can_change_info_;
  static jfieldID can_change_info_fieldID;
  /// True, if the user can invite new users to the chat.
  bool can_invite_users_;
  static jfieldID can_invite_users_fieldID;
  /// True, if the user can pin messages.
  bool can_pin_messages_;
  static jfieldID can_pin_messages_fieldID;
  /// True, if the user can create topics.
  bool can_create_topics_;
  static jfieldID can_create_topics_fieldID;

  /**
   * Describes actions that a user is allowed to take in a chat.
   */
  chatPermissions();

  /**
   * Describes actions that a user is allowed to take in a chat.
   *
   * \param[in] can_send_basic_messages_ True, if the user can send text messages, contacts, giveaways, giveaway winners, invoices, locations, and venues.
   * \param[in] can_send_audios_ True, if the user can send music files.
   * \param[in] can_send_documents_ True, if the user can send documents.
   * \param[in] can_send_photos_ True, if the user can send photos.
   * \param[in] can_send_videos_ True, if the user can send videos.
   * \param[in] can_send_video_notes_ True, if the user can send video notes.
   * \param[in] can_send_voice_notes_ True, if the user can send voice notes.
   * \param[in] can_send_polls_ True, if the user can send polls and checklists.
   * \param[in] can_send_other_messages_ True, if the user can send animations, games, stickers, and dice and use inline bots.
   * \param[in] can_add_link_previews_ True, if the user may add a link preview to their messages.
   * \param[in] can_change_info_ True, if the user can change the chat title, photo, and other settings.
   * \param[in] can_invite_users_ True, if the user can invite new users to the chat.
   * \param[in] can_pin_messages_ True, if the user can pin messages.
   * \param[in] can_create_topics_ True, if the user can create topics.
   */
  chatPermissions(bool can_send_basic_messages_, bool can_send_audios_, bool can_send_documents_, bool can_send_photos_, bool can_send_videos_, bool can_send_video_notes_, bool can_send_voice_notes_, bool can_send_polls_, bool can_send_other_messages_, bool can_add_link_previews_, bool can_change_info_, bool can_invite_users_, bool can_pin_messages_, bool can_create_topics_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -118334855;

  static object_ptr<chatPermissions> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class animatedChatPhoto;

class chatPhotoSticker;

class minithumbnail;

class photoSize;

/**
 * Describes a chat or user profile photo.
 */
class chatPhoto final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique photo identifier.
  int64 id_;
  static jfieldID id_fieldID;
  /// Point in time (Unix timestamp) when the photo has been added.
  int32 added_date_;
  static jfieldID added_date_fieldID;
  /// Photo minithumbnail; may be null.
  object_ptr<minithumbnail> minithumbnail_;
  static jfieldID minithumbnail_fieldID;
  /// Available variants of the photo in JPEG format, in different size.
  array<object_ptr<photoSize>> sizes_;
  static jfieldID sizes_fieldID;
  /// A big (up to 1280x1280) animated variant of the photo in MPEG4 format; may be null.
  object_ptr<animatedChatPhoto> animation_;
  static jfieldID animation_fieldID;
  /// A small (160x160) animated variant of the photo in MPEG4 format; may be null even if the big animation is available.
  object_ptr<animatedChatPhoto> small_animation_;
  static jfieldID small_animation_fieldID;
  /// Sticker-based version of the chat photo; may be null.
  object_ptr<chatPhotoSticker> sticker_;
  static jfieldID sticker_fieldID;

  /**
   * Describes a chat or user profile photo.
   */
  chatPhoto();

  /**
   * Describes a chat or user profile photo.
   *
   * \param[in] id_ Unique photo identifier.
   * \param[in] added_date_ Point in time (Unix timestamp) when the photo has been added.
   * \param[in] minithumbnail_ Photo minithumbnail; may be null.
   * \param[in] sizes_ Available variants of the photo in JPEG format, in different size.
   * \param[in] animation_ A big (up to 1280x1280) animated variant of the photo in MPEG4 format; may be null.
   * \param[in] small_animation_ A small (160x160) animated variant of the photo in MPEG4 format; may be null even if the big animation is available.
   * \param[in] sticker_ Sticker-based version of the chat photo; may be null.
   */
  chatPhoto(int64 id_, int32 added_date_, object_ptr<minithumbnail> &&minithumbnail_, array<object_ptr<photoSize>> &&sizes_, object_ptr<animatedChatPhoto> &&animation_, object_ptr<animatedChatPhoto> &&small_animation_, object_ptr<chatPhotoSticker> &&sticker_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1430870201;

  static object_ptr<chatPhoto> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class file;

class minithumbnail;

/**
 * Contains basic information about the photo of a chat.
 */
class chatPhotoInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A small (160x160) chat photo variant in JPEG format. The file can be downloaded only before the photo is changed.
  object_ptr<file> small_;
  static jfieldID small_fieldID;
  /// A big (640x640) chat photo variant in JPEG format. The file can be downloaded only before the photo is changed.
  object_ptr<file> big_;
  static jfieldID big_fieldID;
  /// Chat photo minithumbnail; may be null.
  object_ptr<minithumbnail> minithumbnail_;
  static jfieldID minithumbnail_fieldID;
  /// True, if the photo has animated variant.
  bool has_animation_;
  static jfieldID has_animation_fieldID;
  /// True, if the photo is visible only for the current user.
  bool is_personal_;
  static jfieldID is_personal_fieldID;

  /**
   * Contains basic information about the photo of a chat.
   */
  chatPhotoInfo();

  /**
   * Contains basic information about the photo of a chat.
   *
   * \param[in] small_ A small (160x160) chat photo variant in JPEG format. The file can be downloaded only before the photo is changed.
   * \param[in] big_ A big (640x640) chat photo variant in JPEG format. The file can be downloaded only before the photo is changed.
   * \param[in] minithumbnail_ Chat photo minithumbnail; may be null.
   * \param[in] has_animation_ True, if the photo has animated variant.
   * \param[in] is_personal_ True, if the photo is visible only for the current user.
   */
  chatPhotoInfo(object_ptr<file> &&small_, object_ptr<file> &&big_, object_ptr<minithumbnail> &&minithumbnail_, bool has_animation_, bool is_personal_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 281195686;

  static object_ptr<chatPhotoInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class BackgroundFill;

class ChatPhotoStickerType;

/**
 * Information about the sticker, which was used to create the chat photo. The sticker is shown at the center of the photo and occupies at most 67% of it.
 */
class chatPhotoSticker final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Type of the sticker.
  object_ptr<ChatPhotoStickerType> type_;
  static jfieldID type_fieldID;
  /// The fill to be used as background for the sticker; rotation angle in backgroundFillGradient isn't supported.
  object_ptr<BackgroundFill> background_fill_;
  static jfieldID background_fill_fieldID;

  /**
   * Information about the sticker, which was used to create the chat photo. The sticker is shown at the center of the photo and occupies at most 67% of it.
   */
  chatPhotoSticker();

  /**
   * Information about the sticker, which was used to create the chat photo. The sticker is shown at the center of the photo and occupies at most 67% of it.
   *
   * \param[in] type_ Type of the sticker.
   * \param[in] background_fill_ The fill to be used as background for the sticker; rotation angle in backgroundFillGradient isn't supported.
   */
  chatPhotoSticker(object_ptr<ChatPhotoStickerType> &&type_, object_ptr<BackgroundFill> &&background_fill_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1459387485;

  static object_ptr<chatPhotoSticker> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes type of sticker, which was used to create a chat photo.
 */
class ChatPhotoStickerType: public Object {
 public:
  static jclass Class;

  static object_ptr<ChatPhotoStickerType> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Information about the sticker, which was used to create the chat photo.
 */
class chatPhotoStickerTypeRegularOrMask final : public ChatPhotoStickerType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Sticker set identifier.
  int64 sticker_set_id_;
  static jfieldID sticker_set_id_fieldID;
  /// Identifier of the sticker in the set.
  int64 sticker_id_;
  static jfieldID sticker_id_fieldID;

  /**
   * Information about the sticker, which was used to create the chat photo.
   */
  chatPhotoStickerTypeRegularOrMask();

  /**
   * Information about the sticker, which was used to create the chat photo.
   *
   * \param[in] sticker_set_id_ Sticker set identifier.
   * \param[in] sticker_id_ Identifier of the sticker in the set.
   */
  chatPhotoStickerTypeRegularOrMask(int64 sticker_set_id_, int64 sticker_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -415147620;

  static object_ptr<ChatPhotoStickerType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Information about the custom emoji, which was used to create the chat photo.
 */
class chatPhotoStickerTypeCustomEmoji final : public ChatPhotoStickerType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the custom emoji.
  int64 custom_emoji_id_;
  static jfieldID custom_emoji_id_fieldID;

  /**
   * Information about the custom emoji, which was used to create the chat photo.
   */
  chatPhotoStickerTypeCustomEmoji();

  /**
   * Information about the custom emoji, which was used to create the chat photo.
   *
   * \param[in] custom_emoji_id_ Identifier of the custom emoji.
   */
  explicit chatPhotoStickerTypeCustomEmoji(int64 custom_emoji_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -266224943;

  static object_ptr<ChatPhotoStickerType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class chatPhoto;

/**
 * Contains a list of chat or user profile photos.
 */
class chatPhotos final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Total number of photos.
  int32 total_count_;
  static jfieldID total_count_fieldID;
  /// List of photos.
  array<object_ptr<chatPhoto>> photos_;
  static jfieldID photos_fieldID;

  /**
   * Contains a list of chat or user profile photos.
   */
  chatPhotos();

  /**
   * Contains a list of chat or user profile photos.
   *
   * \param[in] total_count_ Total number of photos.
   * \param[in] photos_ List of photos.
   */
  chatPhotos(int32 total_count_, array<object_ptr<chatPhoto>> &&photos_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1510699180;

  static object_ptr<chatPhotos> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class ChatList;

class ChatSource;

/**
 * Describes a position of a chat in a chat list.
 */
class chatPosition final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The chat list.
  object_ptr<ChatList> list_;
  static jfieldID list_fieldID;
  /// A parameter used to determine order of the chat in the chat list. Chats must be sorted by the pair (order, chat.id) in descending order.
  int64 order_;
  static jfieldID order_fieldID;
  /// True, if the chat is pinned in the chat list.
  bool is_pinned_;
  static jfieldID is_pinned_fieldID;
  /// Source of the chat in the chat list; may be null.
  object_ptr<ChatSource> source_;
  static jfieldID source_fieldID;

  /**
   * Describes a position of a chat in a chat list.
   */
  chatPosition();

  /**
   * Describes a position of a chat in a chat list.
   *
   * \param[in] list_ The chat list.
   * \param[in] order_ A parameter used to determine order of the chat in the chat list. Chats must be sorted by the pair (order, chat.id) in descending order.
   * \param[in] is_pinned_ True, if the chat is pinned in the chat list.
   * \param[in] source_ Source of the chat in the chat list; may be null.
   */
  chatPosition(object_ptr<ChatList> &&list_, int64 order_, bool is_pinned_, object_ptr<ChatSource> &&source_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -622557355;

  static object_ptr<chatPosition> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains information about revenue earned from sponsored messages in a chat.
 */
class chatRevenueAmount final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Cryptocurrency in which revenue is calculated.
  string cryptocurrency_;
  static jfieldID cryptocurrency_fieldID;
  /// Total amount of the cryptocurrency earned, in the smallest units of the cryptocurrency.
  int64 total_amount_;
  static jfieldID total_amount_fieldID;
  /// Amount of the cryptocurrency that isn't withdrawn yet, in the smallest units of the cryptocurrency.
  int64 balance_amount_;
  static jfieldID balance_amount_fieldID;
  /// Amount of the cryptocurrency available for withdrawal, in the smallest units of the cryptocurrency.
  int64 available_amount_;
  static jfieldID available_amount_fieldID;
  /// True, if Telegram Stars can be withdrawn now or later.
  bool withdrawal_enabled_;
  static jfieldID withdrawal_enabled_fieldID;

  /**
   * Contains information about revenue earned from sponsored messages in a chat.
   */
  chatRevenueAmount();

  /**
   * Contains information about revenue earned from sponsored messages in a chat.
   *
   * \param[in] cryptocurrency_ Cryptocurrency in which revenue is calculated.
   * \param[in] total_amount_ Total amount of the cryptocurrency earned, in the smallest units of the cryptocurrency.
   * \param[in] balance_amount_ Amount of the cryptocurrency that isn't withdrawn yet, in the smallest units of the cryptocurrency.
   * \param[in] available_amount_ Amount of the cryptocurrency available for withdrawal, in the smallest units of the cryptocurrency.
   * \param[in] withdrawal_enabled_ True, if Telegram Stars can be withdrawn now or later.
   */
  chatRevenueAmount(string const &cryptocurrency_, int64 total_amount_, int64 balance_amount_, int64 available_amount_, bool withdrawal_enabled_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1505178024;

  static object_ptr<chatRevenueAmount> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class StatisticalGraph;

class chatRevenueAmount;

/**
 * A detailed statistics about revenue earned from sponsored messages in a chat.
 */
class chatRevenueStatistics final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A graph containing amount of revenue in a given hour.
  object_ptr<StatisticalGraph> revenue_by_hour_graph_;
  static jfieldID revenue_by_hour_graph_fieldID;
  /// A graph containing amount of revenue.
  object_ptr<StatisticalGraph> revenue_graph_;
  static jfieldID revenue_graph_fieldID;
  /// Amount of earned revenue.
  object_ptr<chatRevenueAmount> revenue_amount_;
  static jfieldID revenue_amount_fieldID;
  /// Current conversion rate of the cryptocurrency in which revenue is calculated to USD.
  double usd_rate_;
  static jfieldID usd_rate_fieldID;

  /**
   * A detailed statistics about revenue earned from sponsored messages in a chat.
   */
  chatRevenueStatistics();

  /**
   * A detailed statistics about revenue earned from sponsored messages in a chat.
   *
   * \param[in] revenue_by_hour_graph_ A graph containing amount of revenue in a given hour.
   * \param[in] revenue_graph_ A graph containing amount of revenue.
   * \param[in] revenue_amount_ Amount of earned revenue.
   * \param[in] usd_rate_ Current conversion rate of the cryptocurrency in which revenue is calculated to USD.
   */
  chatRevenueStatistics(object_ptr<StatisticalGraph> &&revenue_by_hour_graph_, object_ptr<StatisticalGraph> &&revenue_graph_, object_ptr<chatRevenueAmount> &&revenue_amount_, double usd_rate_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1667438779;

  static object_ptr<chatRevenueStatistics> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class ChatRevenueTransactionType;

/**
 * Contains a chat revenue transactions.
 */
class chatRevenueTransaction final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Cryptocurrency in which revenue is calculated.
  string cryptocurrency_;
  static jfieldID cryptocurrency_fieldID;
  /// The withdrawn amount, in the smallest units of the cryptocurrency.
  int64 cryptocurrency_amount_;
  static jfieldID cryptocurrency_amount_fieldID;
  /// Type of the transaction.
  object_ptr<ChatRevenueTransactionType> type_;
  static jfieldID type_fieldID;

  /**
   * Contains a chat revenue transactions.
   */
  chatRevenueTransaction();

  /**
   * Contains a chat revenue transactions.
   *
   * \param[in] cryptocurrency_ Cryptocurrency in which revenue is calculated.
   * \param[in] cryptocurrency_amount_ The withdrawn amount, in the smallest units of the cryptocurrency.
   * \param[in] type_ Type of the transaction.
   */
  chatRevenueTransaction(string const &cryptocurrency_, int64 cryptocurrency_amount_, object_ptr<ChatRevenueTransactionType> &&type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 80192767;

  static object_ptr<chatRevenueTransaction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class RevenueWithdrawalState;

/**
 * This class is an abstract base class.
 * Describes type of transaction for revenue earned from sponsored messages in a chat.
 */
class ChatRevenueTransactionType: public Object {
 public:
  static jclass Class;

  static object_ptr<ChatRevenueTransactionType> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes an unsupported transaction.
 */
class chatRevenueTransactionTypeUnsupported final : public ChatRevenueTransactionType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * Describes an unsupported transaction.
   */
  chatRevenueTransactionTypeUnsupported();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -27518756;

  static object_ptr<ChatRevenueTransactionType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes earnings from sponsored messages in a chat in some time frame.
 */
class chatRevenueTransactionTypeSponsoredMessageEarnings final : public ChatRevenueTransactionType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Point in time (Unix timestamp) when the earnings started.
  int32 start_date_;
  static jfieldID start_date_fieldID;
  /// Point in time (Unix timestamp) when the earnings ended.
  int32 end_date_;
  static jfieldID end_date_fieldID;

  /**
   * Describes earnings from sponsored messages in a chat in some time frame.
   */
  chatRevenueTransactionTypeSponsoredMessageEarnings();

  /**
   * Describes earnings from sponsored messages in a chat in some time frame.
   *
   * \param[in] start_date_ Point in time (Unix timestamp) when the earnings started.
   * \param[in] end_date_ Point in time (Unix timestamp) when the earnings ended.
   */
  chatRevenueTransactionTypeSponsoredMessageEarnings(int32 start_date_, int32 end_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1907391317;

  static object_ptr<ChatRevenueTransactionType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes earnings from a published suggested post.
 */
class chatRevenueTransactionTypeSuggestedPostEarnings final : public ChatRevenueTransactionType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the user that paid for the suggested post.
  int53 user_id_;
  static jfieldID user_id_fieldID;

  /**
   * Describes earnings from a published suggested post.
   */
  chatRevenueTransactionTypeSuggestedPostEarnings();

  /**
   * Describes earnings from a published suggested post.
   *
   * \param[in] user_id_ Identifier of the user that paid for the suggested post.
   */
  explicit chatRevenueTransactionTypeSuggestedPostEarnings(int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1252049103;

  static object_ptr<ChatRevenueTransactionType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes a withdrawal of earnings through Fragment.
 */
class chatRevenueTransactionTypeFragmentWithdrawal final : public ChatRevenueTransactionType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Point in time (Unix timestamp) when the earnings withdrawal started.
  int32 withdrawal_date_;
  static jfieldID withdrawal_date_fieldID;
  /// State of the withdrawal.
  object_ptr<RevenueWithdrawalState> state_;
  static jfieldID state_fieldID;

  /**
   * Describes a withdrawal of earnings through Fragment.
   */
  chatRevenueTransactionTypeFragmentWithdrawal();

  /**
   * Describes a withdrawal of earnings through Fragment.
   *
   * \param[in] withdrawal_date_ Point in time (Unix timestamp) when the earnings withdrawal started.
   * \param[in] state_ State of the withdrawal.
   */
  chatRevenueTransactionTypeFragmentWithdrawal(int32 withdrawal_date_, object_ptr<RevenueWithdrawalState> &&state_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 327153867;

  static object_ptr<ChatRevenueTransactionType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes a refund for failed withdrawal of earnings through Fragment.
 */
class chatRevenueTransactionTypeFragmentRefund final : public ChatRevenueTransactionType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Point in time (Unix timestamp) when the transaction was refunded.
  int32 refund_date_;
  static jfieldID refund_date_fieldID;

  /**
   * Describes a refund for failed withdrawal of earnings through Fragment.
   */
  chatRevenueTransactionTypeFragmentRefund();

  /**
   * Describes a refund for failed withdrawal of earnings through Fragment.
   *
   * \param[in] refund_date_ Point in time (Unix timestamp) when the transaction was refunded.
   */
  explicit chatRevenueTransactionTypeFragmentRefund(int32 refund_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1488694273;

  static object_ptr<ChatRevenueTransactionType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class chatRevenueTransaction;

/**
 * Contains a list of chat revenue transactions.
 */
class chatRevenueTransactions final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The amount of owned Toncoins; in the smallest units of the cryptocurrency.
  int53 ton_amount_;
  static jfieldID ton_amount_fieldID;
  /// List of transactions.
  array<object_ptr<chatRevenueTransaction>> transactions_;
  static jfieldID transactions_fieldID;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;
  static jfieldID next_offset_fieldID;

  /**
   * Contains a list of chat revenue transactions.
   */
  chatRevenueTransactions();

  /**
   * Contains a list of chat revenue transactions.
   *
   * \param[in] ton_amount_ The amount of owned Toncoins; in the smallest units of the cryptocurrency.
   * \param[in] transactions_ List of transactions.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  chatRevenueTransactions(int53 ton_amount_, array<object_ptr<chatRevenueTransaction>> &&transactions_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2017122771;

  static object_ptr<chatRevenueTransactions> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes a reason why an external chat is shown in a chat list.
 */
class ChatSource: public Object {
 public:
  static jclass Class;

  static object_ptr<ChatSource> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat is sponsored by the user's MTProxy server.
 */
class chatSourceMtprotoProxy final : public ChatSource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The chat is sponsored by the user's MTProxy server.
   */
  chatSourceMtprotoProxy();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 394074115;

  static object_ptr<ChatSource> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The chat contains a public service announcement.
 */
class chatSourcePublicServiceAnnouncement final : public ChatSource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The type of the announcement.
  string type_;
  static jfieldID type_fieldID;
  /// The text of the announcement.
  string text_;
  static jfieldID text_fieldID;

  /**
   * The chat contains a public service announcement.
   */
  chatSourcePublicServiceAnnouncement();

  /**
   * The chat contains a public service announcement.
   *
   * \param[in] type_ The type of the announcement.
   * \param[in] text_ The text of the announcement.
   */
  chatSourcePublicServiceAnnouncement(string const &type_, string const &text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -328571244;

  static object_ptr<ChatSource> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class StatisticalGraph;

class chatStatisticsAdministratorActionsInfo;

class chatStatisticsInteractionInfo;

class chatStatisticsInviterInfo;

class chatStatisticsMessageSenderInfo;

class dateRange;

class statisticalValue;

/**
 * This class is an abstract base class.
 * Contains a detailed statistics about a chat.
 */
class ChatStatistics: public Object {
 public:
  static jclass Class;

  static object_ptr<ChatStatistics> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A detailed statistics about a supergroup chat.
 */
class chatStatisticsSupergroup final : public ChatStatistics {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A period to which the statistics applies.
  object_ptr<dateRange> period_;
  static jfieldID period_fieldID;
  /// Number of members in the chat.
  object_ptr<statisticalValue> member_count_;
  static jfieldID member_count_fieldID;
  /// Number of messages sent to the chat.
  object_ptr<statisticalValue> message_count_;
  static jfieldID message_count_fieldID;
  /// Number of users who viewed messages in the chat.
  object_ptr<statisticalValue> viewer_count_;
  static jfieldID viewer_count_fieldID;
  /// Number of users who sent messages to the chat.
  object_ptr<statisticalValue> sender_count_;
  static jfieldID sender_count_fieldID;
  /// A graph containing number of members in the chat.
  object_ptr<StatisticalGraph> member_count_graph_;
  static jfieldID member_count_graph_fieldID;
  /// A graph containing number of members joined and left the chat.
  object_ptr<StatisticalGraph> join_graph_;
  static jfieldID join_graph_fieldID;
  /// A graph containing number of new member joins per source.
  object_ptr<StatisticalGraph> join_by_source_graph_;
  static jfieldID join_by_source_graph_fieldID;
  /// A graph containing distribution of active users per language.
  object_ptr<StatisticalGraph> language_graph_;
  static jfieldID language_graph_fieldID;
  /// A graph containing distribution of sent messages by content type.
  object_ptr<StatisticalGraph> message_content_graph_;
  static jfieldID message_content_graph_fieldID;
  /// A graph containing number of different actions in the chat.
  object_ptr<StatisticalGraph> action_graph_;
  static jfieldID action_graph_fieldID;
  /// A graph containing distribution of message views per hour.
  object_ptr<StatisticalGraph> day_graph_;
  static jfieldID day_graph_fieldID;
  /// A graph containing distribution of message views per day of week.
  object_ptr<StatisticalGraph> week_graph_;
  static jfieldID week_graph_fieldID;
  /// List of users sent most messages in the last week.
  array<object_ptr<chatStatisticsMessageSenderInfo>> top_senders_;
  static jfieldID top_senders_fieldID;
  /// List of most active administrators in the last week.
  array<object_ptr<chatStatisticsAdministratorActionsInfo>> top_administrators_;
  static jfieldID top_administrators_fieldID;
  /// List of most active inviters of new members in the last week.
  array<object_ptr<chatStatisticsInviterInfo>> top_inviters_;
  static jfieldID top_inviters_fieldID;

  /**
   * A detailed statistics about a supergroup chat.
   */
  chatStatisticsSupergroup();

  /**
   * A detailed statistics about a supergroup chat.
   *
   * \param[in] period_ A period to which the statistics applies.
   * \param[in] member_count_ Number of members in the chat.
   * \param[in] message_count_ Number of messages sent to the chat.
   * \param[in] viewer_count_ Number of users who viewed messages in the chat.
   * \param[in] sender_count_ Number of users who sent messages to the chat.
   * \param[in] member_count_graph_ A graph containing number of members in the chat.
   * \param[in] join_graph_ A graph containing number of members joined and left the chat.
   * \param[in] join_by_source_graph_ A graph containing number of new member joins per source.
   * \param[in] language_graph_ A graph containing distribution of active users per language.
   * \param[in] message_content_graph_ A graph containing distribution of sent messages by content type.
   * \param[in] action_graph_ A graph containing number of different actions in the chat.
   * \param[in] day_graph_ A graph containing distribution of message views per hour.
   * \param[in] week_graph_ A graph containing distribution of message views per day of week.
   * \param[in] top_senders_ List of users sent most messages in the last week.
   * \param[in] top_administrators_ List of most active administrators in the last week.
   * \param[in] top_inviters_ List of most active inviters of new members in the last week.
   */
  chatStatisticsSupergroup(object_ptr<dateRange> &&period_, object_ptr<statisticalValue> &&member_count_, object_ptr<statisticalValue> &&message_count_, object_ptr<statisticalValue> &&viewer_count_, object_ptr<statisticalValue> &&sender_count_, object_ptr<StatisticalGraph> &&member_count_graph_, object_ptr<StatisticalGraph> &&join_graph_, object_ptr<StatisticalGraph> &&join_by_source_graph_, object_ptr<StatisticalGraph> &&language_graph_, object_ptr<StatisticalGraph> &&message_content_graph_, object_ptr<StatisticalGraph> &&action_graph_, object_ptr<StatisticalGraph> &&day_graph_, object_ptr<StatisticalGraph> &&week_graph_, array<object_ptr<chatStatisticsMessageSenderInfo>> &&top_senders_, array<object_ptr<chatStatisticsAdministratorActionsInfo>> &&top_administrators_, array<object_ptr<chatStatisticsInviterInfo>> &&top_inviters_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -17244633;

  static object_ptr<ChatStatistics> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A detailed statistics about a channel chat.
 */
class chatStatisticsChannel final : public ChatStatistics {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A period to which the statistics applies.
  object_ptr<dateRange> period_;
  static jfieldID period_fieldID;
  /// Number of members in the chat.
  object_ptr<statisticalValue> member_count_;
  static jfieldID member_count_fieldID;
  /// Mean number of times the recently sent messages were viewed.
  object_ptr<statisticalValue> mean_message_view_count_;
  static jfieldID mean_message_view_count_fieldID;
  /// Mean number of times the recently sent messages were shared.
  object_ptr<statisticalValue> mean_message_share_count_;
  static jfieldID mean_message_share_count_fieldID;
  /// Mean number of times reactions were added to the recently sent messages.
  object_ptr<statisticalValue> mean_message_reaction_count_;
  static jfieldID mean_message_reaction_count_fieldID;
  /// Mean number of times the recently posted stories were viewed.
  object_ptr<statisticalValue> mean_story_view_count_;
  static jfieldID mean_story_view_count_fieldID;
  /// Mean number of times the recently posted stories were shared.
  object_ptr<statisticalValue> mean_story_share_count_;
  static jfieldID mean_story_share_count_fieldID;
  /// Mean number of times reactions were added to the recently posted stories.
  object_ptr<statisticalValue> mean_story_reaction_count_;
  static jfieldID mean_story_reaction_count_fieldID;
  /// A percentage of users with enabled notifications for the chat; 0-100.
  double enabled_notifications_percentage_;
  static jfieldID enabled_notifications_percentage_fieldID;
  /// A graph containing number of members in the chat.
  object_ptr<StatisticalGraph> member_count_graph_;
  static jfieldID member_count_graph_fieldID;
  /// A graph containing number of members joined and left the chat.
  object_ptr<StatisticalGraph> join_graph_;
  static jfieldID join_graph_fieldID;
  /// A graph containing number of members muted and unmuted the chat.
  object_ptr<StatisticalGraph> mute_graph_;
  static jfieldID mute_graph_fieldID;
  /// A graph containing number of message views in a given hour in the last two weeks.
  object_ptr<StatisticalGraph> view_count_by_hour_graph_;
  static jfieldID view_count_by_hour_graph_fieldID;
  /// A graph containing number of message views per source.
  object_ptr<StatisticalGraph> view_count_by_source_graph_;
  static jfieldID view_count_by_source_graph_fieldID;
  /// A graph containing number of new member joins per source.
  object_ptr<StatisticalGraph> join_by_source_graph_;
  static jfieldID join_by_source_graph_fieldID;
  /// A graph containing number of users viewed chat messages per language.
  object_ptr<StatisticalGraph> language_graph_;
  static jfieldID language_graph_fieldID;
  /// A graph containing number of chat message views and shares.
  object_ptr<StatisticalGraph> message_interaction_graph_;
  static jfieldID message_interaction_graph_fieldID;
  /// A graph containing number of reactions on messages.
  object_ptr<StatisticalGraph> message_reaction_graph_;
  static jfieldID message_reaction_graph_fieldID;
  /// A graph containing number of story views and shares.
  object_ptr<StatisticalGraph> story_interaction_graph_;
  static jfieldID story_interaction_graph_fieldID;
  /// A graph containing number of reactions on stories.
  object_ptr<StatisticalGraph> story_reaction_graph_;
  static jfieldID story_reaction_graph_fieldID;
  /// A graph containing number of views of associated with the chat instant views.
  object_ptr<StatisticalGraph> instant_view_interaction_graph_;
  static jfieldID instant_view_interaction_graph_fieldID;
  /// Detailed statistics about number of views and shares of recently sent messages and posted stories.
  array<object_ptr<chatStatisticsInteractionInfo>> recent_interactions_;
  static jfieldID recent_interactions_fieldID;

  /**
   * A detailed statistics about a channel chat.
   */
  chatStatisticsChannel();

  /**
   * A detailed statistics about a channel chat.
   *
   * \param[in] period_ A period to which the statistics applies.
   * \param[in] member_count_ Number of members in the chat.
   * \param[in] mean_message_view_count_ Mean number of times the recently sent messages were viewed.
   * \param[in] mean_message_share_count_ Mean number of times the recently sent messages were shared.
   * \param[in] mean_message_reaction_count_ Mean number of times reactions were added to the recently sent messages.
   * \param[in] mean_story_view_count_ Mean number of times the recently posted stories were viewed.
   * \param[in] mean_story_share_count_ Mean number of times the recently posted stories were shared.
   * \param[in] mean_story_reaction_count_ Mean number of times reactions were added to the recently posted stories.
   * \param[in] enabled_notifications_percentage_ A percentage of users with enabled notifications for the chat; 0-100.
   * \param[in] member_count_graph_ A graph containing number of members in the chat.
   * \param[in] join_graph_ A graph containing number of members joined and left the chat.
   * \param[in] mute_graph_ A graph containing number of members muted and unmuted the chat.
   * \param[in] view_count_by_hour_graph_ A graph containing number of message views in a given hour in the last two weeks.
   * \param[in] view_count_by_source_graph_ A graph containing number of message views per source.
   * \param[in] join_by_source_graph_ A graph containing number of new member joins per source.
   * \param[in] language_graph_ A graph containing number of users viewed chat messages per language.
   * \param[in] message_interaction_graph_ A graph containing number of chat message views and shares.
   * \param[in] message_reaction_graph_ A graph containing number of reactions on messages.
   * \param[in] story_interaction_graph_ A graph containing number of story views and shares.
   * \param[in] story_reaction_graph_ A graph containing number of reactions on stories.
   * \param[in] instant_view_interaction_graph_ A graph containing number of views of associated with the chat instant views.
   * \param[in] recent_interactions_ Detailed statistics about number of views and shares of recently sent messages and posted stories.
   */
  chatStatisticsChannel(object_ptr<dateRange> &&period_, object_ptr<statisticalValue> &&member_count_, object_ptr<statisticalValue> &&mean_message_view_count_, object_ptr<statisticalValue> &&mean_message_share_count_, object_ptr<statisticalValue> &&mean_message_reaction_count_, object_ptr<statisticalValue> &&mean_story_view_count_, object_ptr<statisticalValue> &&mean_story_share_count_, object_ptr<statisticalValue> &&mean_story_reaction_count_, double enabled_notifications_percentage_, object_ptr<StatisticalGraph> &&member_count_graph_, object_ptr<StatisticalGraph> &&join_graph_, object_ptr<StatisticalGraph> &&mute_graph_, object_ptr<StatisticalGraph> &&view_count_by_hour_graph_, object_ptr<StatisticalGraph> &&view_count_by_source_graph_, object_ptr<StatisticalGraph> &&join_by_source_graph_, object_ptr<StatisticalGraph> &&language_graph_, object_ptr<StatisticalGraph> &&message_interaction_graph_, object_ptr<StatisticalGraph> &&message_reaction_graph_, object_ptr<StatisticalGraph> &&story_interaction_graph_, object_ptr<StatisticalGraph> &&story_reaction_graph_, object_ptr<StatisticalGraph> &&instant_view_interaction_graph_, array<object_ptr<chatStatisticsInteractionInfo>> &&recent_interactions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1375151660;

  static object_ptr<ChatStatistics> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains statistics about administrator actions done by a user.
 */
class chatStatisticsAdministratorActionsInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Administrator user identifier.
  int53 user_id_;
  static jfieldID user_id_fieldID;
  /// Number of messages deleted by the administrator.
  int32 deleted_message_count_;
  static jfieldID deleted_message_count_fieldID;
  /// Number of users banned by the administrator.
  int32 banned_user_count_;
  static jfieldID banned_user_count_fieldID;
  /// Number of users restricted by the administrator.
  int32 restricted_user_count_;
  static jfieldID restricted_user_count_fieldID;

  /**
   * Contains statistics about administrator actions done by a user.
   */
  chatStatisticsAdministratorActionsInfo();

  /**
   * Contains statistics about administrator actions done by a user.
   *
   * \param[in] user_id_ Administrator user identifier.
   * \param[in] deleted_message_count_ Number of messages deleted by the administrator.
   * \param[in] banned_user_count_ Number of users banned by the administrator.
   * \param[in] restricted_user_count_ Number of users restricted by the administrator.
   */
  chatStatisticsAdministratorActionsInfo(int53 user_id_, int32 deleted_message_count_, int32 banned_user_count_, int32 restricted_user_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -406467202;

  static object_ptr<chatStatisticsAdministratorActionsInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class ChatStatisticsObjectType;

/**
 * Contains statistics about interactions with a message sent in the chat or a story posted on behalf of the chat.
 */
class chatStatisticsInteractionInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Type of the object.
  object_ptr<ChatStatisticsObjectType> object_type_;
  static jfieldID object_type_fieldID;
  /// Number of times the object was viewed.
  int32 view_count_;
  static jfieldID view_count_fieldID;
  /// Number of times the object was forwarded.
  int32 forward_count_;
  static jfieldID forward_count_fieldID;
  /// Number of times reactions were added to the object.
  int32 reaction_count_;
  static jfieldID reaction_count_fieldID;

  /**
   * Contains statistics about interactions with a message sent in the chat or a story posted on behalf of the chat.
   */
  chatStatisticsInteractionInfo();

  /**
   * Contains statistics about interactions with a message sent in the chat or a story posted on behalf of the chat.
   *
   * \param[in] object_type_ Type of the object.
   * \param[in] view_count_ Number of times the object was viewed.
   * \param[in] forward_count_ Number of times the object was forwarded.
   * \param[in] reaction_count_ Number of times reactions were added to the object.
   */
  chatStatisticsInteractionInfo(object_ptr<ChatStatisticsObjectType> &&object_type_, int32 view_count_, int32 forward_count_, int32 reaction_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1766496909;

  static object_ptr<chatStatisticsInteractionInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains statistics about number of new members invited by a user.
 */
class chatStatisticsInviterInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// User identifier.
  int53 user_id_;
  static jfieldID user_id_fieldID;
  /// Number of new members invited by the user.
  int32 added_member_count_;
  static jfieldID added_member_count_fieldID;

  /**
   * Contains statistics about number of new members invited by a user.
   */
  chatStatisticsInviterInfo();

  /**
   * Contains statistics about number of new members invited by a user.
   *
   * \param[in] user_id_ User identifier.
   * \param[in] added_member_count_ Number of new members invited by the user.
   */
  chatStatisticsInviterInfo(int53 user_id_, int32 added_member_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 629396619;

  static object_ptr<chatStatisticsInviterInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains statistics about messages sent by a user.
 */
class chatStatisticsMessageSenderInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// User identifier.
  int53 user_id_;
  static jfieldID user_id_fieldID;
  /// Number of sent messages.
  int32 sent_message_count_;
  static jfieldID sent_message_count_fieldID;
  /// Average number of characters in sent messages; 0 if unknown.
  int32 average_character_count_;
  static jfieldID average_character_count_fieldID;

  /**
   * Contains statistics about messages sent by a user.
   */
  chatStatisticsMessageSenderInfo();

  /**
   * Contains statistics about messages sent by a user.
   *
   * \param[in] user_id_ User identifier.
   * \param[in] sent_message_count_ Number of sent messages.
   * \param[in] average_character_count_ Average number of characters in sent messages; 0 if unknown.
   */
  chatStatisticsMessageSenderInfo(int53 user_id_, int32 sent_message_count_, int32 average_character_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1762295371;

  static object_ptr<chatStatisticsMessageSenderInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes type of object, for which statistics are provided.
 */
class ChatStatisticsObjectType: public Object {
 public:
  static jclass Class;

  static object_ptr<ChatStatisticsObjectType> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes a message sent in the chat.
 */
class chatStatisticsObjectTypeMessage final : public ChatStatisticsObjectType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Message identifier.
  int53 message_id_;
  static jfieldID message_id_fieldID;

  /**
   * Describes a message sent in the chat.
   */
  chatStatisticsObjectTypeMessage();

  /**
   * Describes a message sent in the chat.
   *
   * \param[in] message_id_ Message identifier.
   */
  explicit chatStatisticsObjectTypeMessage(int53 message_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1872700662;

  static object_ptr<ChatStatisticsObjectType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes a story posted on behalf of the chat.
 */
class chatStatisticsObjectTypeStory final : public ChatStatisticsObjectType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Story identifier.
  int32 story_id_;
  static jfieldID story_id_fieldID;

  /**
   * Describes a story posted on behalf of the chat.
   */
  chatStatisticsObjectTypeStory();

  /**
   * Describes a story posted on behalf of the chat.
   *
   * \param[in] story_id_ Story identifier.
   */
  explicit chatStatisticsObjectTypeStory(int32 story_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 364575152;

  static object_ptr<ChatStatisticsObjectType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class themeSettings;

/**
 * Describes a chat theme.
 */
class chatTheme final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Theme name.
  string name_;
  static jfieldID name_fieldID;
  /// Theme settings for a light chat theme.
  object_ptr<themeSettings> light_settings_;
  static jfieldID light_settings_fieldID;
  /// Theme settings for a dark chat theme.
  object_ptr<themeSettings> dark_settings_;
  static jfieldID dark_settings_fieldID;

  /**
   * Describes a chat theme.
   */
  chatTheme();

  /**
   * Describes a chat theme.
   *
   * \param[in] name_ Theme name.
   * \param[in] light_settings_ Theme settings for a light chat theme.
   * \param[in] dark_settings_ Theme settings for a dark chat theme.
   */
  chatTheme(string const &name_, object_ptr<themeSettings> &&light_settings_, object_ptr<themeSettings> &&dark_settings_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -113218503;

  static object_ptr<chatTheme> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes the type of chat.
 */
class ChatType: public Object {
 public:
  static jclass Class;

  static object_ptr<ChatType> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * An ordinary chat with a user.
 */
class chatTypePrivate final : public ChatType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// User identifier.
  int53 user_id_;
  static jfieldID user_id_fieldID;

  /**
   * An ordinary chat with a user.
   */
  chatTypePrivate();

  /**
   * An ordinary chat with a user.
   *
   * \param[in] user_id_ User identifier.
   */
  explicit chatTypePrivate(int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1579049844;

  static object_ptr<ChatType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A basic group (a chat with 0-200 other users).
 */
class chatTypeBasicGroup final : public ChatType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Basic group identifier.
  int53 basic_group_id_;
  static jfieldID basic_group_id_fieldID;

  /**
   * A basic group (a chat with 0-200 other users).
   */
  chatTypeBasicGroup();

  /**
   * A basic group (a chat with 0-200 other users).
   *
   * \param[in] basic_group_id_ Basic group identifier.
   */
  explicit chatTypeBasicGroup(int53 basic_group_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 973884508;

  static object_ptr<ChatType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A supergroup or channel (with unlimited members).
 */
class chatTypeSupergroup final : public ChatType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Supergroup or channel identifier.
  int53 supergroup_id_;
  static jfieldID supergroup_id_fieldID;
  /// True, if the supergroup is a channel.
  bool is_channel_;
  static jfieldID is_channel_fieldID;

  /**
   * A supergroup or channel (with unlimited members).
   */
  chatTypeSupergroup();

  /**
   * A supergroup or channel (with unlimited members).
   *
   * \param[in] supergroup_id_ Supergroup or channel identifier.
   * \param[in] is_channel_ True, if the supergroup is a channel.
   */
  chatTypeSupergroup(int53 supergroup_id_, bool is_channel_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1472570774;

  static object_ptr<ChatType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A secret chat with a user.
 */
class chatTypeSecret final : public ChatType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Secret chat identifier.
  int32 secret_chat_id_;
  static jfieldID secret_chat_id_fieldID;
  /// User identifier of the other user in the secret chat.
  int53 user_id_;
  static jfieldID user_id_fieldID;

  /**
   * A secret chat with a user.
   */
  chatTypeSecret();

  /**
   * A secret chat with a user.
   *
   * \param[in] secret_chat_id_ Secret chat identifier.
   * \param[in] user_id_ User identifier of the other user in the secret chat.
   */
  chatTypeSecret(int32 secret_chat_id_, int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 862366513;

  static object_ptr<ChatType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents a list of chats.
 */
class chats final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Approximate total number of chats found.
  int32 total_count_;
  static jfieldID total_count_fieldID;
  /// List of chat identifiers.
  array<int53> chat_ids_;
  static jfieldID chat_ids_fieldID;

  /**
   * Represents a list of chats.
   */
  chats();

  /**
   * Represents a list of chats.
   *
   * \param[in] total_count_ Approximate total number of chats found.
   * \param[in] chat_ids_ List of chat identifiers.
   */
  chats(int32 total_count_, array<int53> &&chat_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1809654812;

  static object_ptr<chats> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Represents result of checking whether a username can be set for a chat.
 */
class CheckChatUsernameResult: public Object {
 public:
  static jclass Class;

  static object_ptr<CheckChatUsernameResult> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The username can be set.
 */
class checkChatUsernameResultOk final : public CheckChatUsernameResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The username can be set.
   */
  checkChatUsernameResultOk();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1498956964;

  static object_ptr<CheckChatUsernameResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The username is invalid.
 */
class checkChatUsernameResultUsernameInvalid final : public CheckChatUsernameResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The username is invalid.
   */
  checkChatUsernameResultUsernameInvalid();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -636979370;

  static object_ptr<CheckChatUsernameResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The username is occupied.
 */
class checkChatUsernameResultUsernameOccupied final : public CheckChatUsernameResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The username is occupied.
   */
  checkChatUsernameResultUsernameOccupied();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1320892201;

  static object_ptr<CheckChatUsernameResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The username can be purchased at https://fragment.com. Information about the username can be received using getCollectibleItemInfo.
 */
class checkChatUsernameResultUsernamePurchasable final : public CheckChatUsernameResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The username can be purchased at https://fragment.com. Information about the username can be received using getCollectibleItemInfo.
   */
  checkChatUsernameResultUsernamePurchasable();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 5885529;

  static object_ptr<CheckChatUsernameResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user has too many chats with username, one of them must be made private first.
 */
class checkChatUsernameResultPublicChatsTooMany final : public CheckChatUsernameResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user has too many chats with username, one of them must be made private first.
   */
  checkChatUsernameResultPublicChatsTooMany();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -659264388;

  static object_ptr<CheckChatUsernameResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user can't be a member of a public supergroup.
 */
class checkChatUsernameResultPublicGroupsUnavailable final : public CheckChatUsernameResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user can't be a member of a public supergroup.
   */
  checkChatUsernameResultPublicGroupsUnavailable();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -51833641;

  static object_ptr<CheckChatUsernameResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Represents result of checking whether a name can be used for a new sticker set.
 */
class CheckStickerSetNameResult: public Object {
 public:
  static jclass Class;

  static object_ptr<CheckStickerSetNameResult> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The name can be set.
 */
class checkStickerSetNameResultOk final : public CheckStickerSetNameResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The name can be set.
   */
  checkStickerSetNameResultOk();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1404308904;

  static object_ptr<CheckStickerSetNameResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The name is invalid.
 */
class checkStickerSetNameResultNameInvalid final : public CheckStickerSetNameResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The name is invalid.
   */
  checkStickerSetNameResultNameInvalid();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 177992244;

  static object_ptr<CheckStickerSetNameResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The name is occupied.
 */
class checkStickerSetNameResultNameOccupied final : public CheckStickerSetNameResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The name is occupied.
   */
  checkStickerSetNameResultNameOccupied();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1012980872;

  static object_ptr<CheckStickerSetNameResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class checklistTask;

class formattedText;

/**
 * Describes a checklist.
 */
class checklist final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Title of the checklist; may contain only Bold, Italic, Underline, Strikethrough, Spoiler, and CustomEmoji entities.
  object_ptr<formattedText> title_;
  static jfieldID title_fieldID;
  /// List of tasks in the checklist.
  array<object_ptr<checklistTask>> tasks_;
  static jfieldID tasks_fieldID;
  /// True, if users other than creator of the list can add tasks to the list.
  bool others_can_add_tasks_;
  static jfieldID others_can_add_tasks_fieldID;
  /// True, if the current user can add tasks to the list if they have Telegram Premium subscription.
  bool can_add_tasks_;
  static jfieldID can_add_tasks_fieldID;
  /// True, if users other than creator of the list can mark tasks as done or not done. If true, then the checklist is called &quot;group checklist&quot;.
  bool others_can_mark_tasks_as_done_;
  static jfieldID others_can_mark_tasks_as_done_fieldID;
  /// True, if the current user can mark tasks as done or not done if they have Telegram Premium subscription.
  bool can_mark_tasks_as_done_;
  static jfieldID can_mark_tasks_as_done_fieldID;

  /**
   * Describes a checklist.
   */
  checklist();

  /**
   * Describes a checklist.
   *
   * \param[in] title_ Title of the checklist; may contain only Bold, Italic, Underline, Strikethrough, Spoiler, and CustomEmoji entities.
   * \param[in] tasks_ List of tasks in the checklist.
   * \param[in] others_can_add_tasks_ True, if users other than creator of the list can add tasks to the list.
   * \param[in] can_add_tasks_ True, if the current user can add tasks to the list if they have Telegram Premium subscription.
   * \param[in] others_can_mark_tasks_as_done_ True, if users other than creator of the list can mark tasks as done or not done. If true, then the checklist is called &quot;group checklist&quot;.
   * \param[in] can_mark_tasks_as_done_ True, if the current user can mark tasks as done or not done if they have Telegram Premium subscription.
   */
  checklist(object_ptr<formattedText> &&title_, array<object_ptr<checklistTask>> &&tasks_, bool others_can_add_tasks_, bool can_add_tasks_, bool others_can_mark_tasks_as_done_, bool can_mark_tasks_as_done_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -987598247;

  static object_ptr<checklist> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class formattedText;

/**
 * Describes a task in a checklist.
 */
class checklistTask final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the task.
  int32 id_;
  static jfieldID id_fieldID;
  /// Text of the task; may contain only Bold, Italic, Underline, Strikethrough, Spoiler, CustomEmoji, Url, EmailAddress, Mention, Hashtag, Cashtag and PhoneNumber entities.
  object_ptr<formattedText> text_;
  static jfieldID text_fieldID;
  /// Identifier of the user that completed the task; 0 if the task isn't completed.
  int53 completed_by_user_id_;
  static jfieldID completed_by_user_id_fieldID;
  /// Point in time (Unix timestamp) when the task was completed; 0 if the task isn't completed.
  int32 completion_date_;
  static jfieldID completion_date_fieldID;

  /**
   * Describes a task in a checklist.
   */
  checklistTask();

  /**
   * Describes a task in a checklist.
   *
   * \param[in] id_ Unique identifier of the task.
   * \param[in] text_ Text of the task; may contain only Bold, Italic, Underline, Strikethrough, Spoiler, CustomEmoji, Url, EmailAddress, Mention, Hashtag, Cashtag and PhoneNumber entities.
   * \param[in] completed_by_user_id_ Identifier of the user that completed the task; 0 if the task isn't completed.
   * \param[in] completion_date_ Point in time (Unix timestamp) when the task was completed; 0 if the task isn't completed.
   */
  checklistTask(int32 id_, object_ptr<formattedText> &&text_, int53 completed_by_user_id_, int32 completion_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 772992512;

  static object_ptr<checklistTask> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class birthdate;

/**
 * Describes a user that had or will have a birthday soon.
 */
class closeBirthdayUser final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// User identifier.
  int53 user_id_;
  static jfieldID user_id_fieldID;
  /// Birthdate of the user.
  object_ptr<birthdate> birthdate_;
  static jfieldID birthdate_fieldID;

  /**
   * Describes a user that had or will have a birthday soon.
   */
  closeBirthdayUser();

  /**
   * Describes a user that had or will have a birthday soon.
   *
   * \param[in] user_id_ User identifier.
   * \param[in] birthdate_ Birthdate of the user.
   */
  closeBirthdayUser(int53 user_id_, object_ptr<birthdate> &&birthdate_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2147067410;

  static object_ptr<closeBirthdayUser> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class VectorPathCommand;

/**
 * Represents a closed vector path. The path begins at the end point of the last command. The coordinate system origin is in the upper-left corner.
 */
class closedVectorPath final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// List of vector path commands.
  array<object_ptr<VectorPathCommand>> commands_;
  static jfieldID commands_fieldID;

  /**
   * Represents a closed vector path. The path begins at the end point of the last command. The coordinate system origin is in the upper-left corner.
   */
  closedVectorPath();

  /**
   * Represents a closed vector path. The path begins at the end point of the last command. The coordinate system origin is in the upper-left corner.
   *
   * \param[in] commands_ List of vector path commands.
   */
  explicit closedVectorPath(array<object_ptr<VectorPathCommand>> &&commands_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 589951657;

  static object_ptr<closedVectorPath> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains information about a collectible item and its last purchase.
 */
class collectibleItemInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Point in time (Unix timestamp) when the item was purchased.
  int32 purchase_date_;
  static jfieldID purchase_date_fieldID;
  /// Currency for the paid amount.
  string currency_;
  static jfieldID currency_fieldID;
  /// The paid amount, in the smallest units of the currency.
  int53 amount_;
  static jfieldID amount_fieldID;
  /// Cryptocurrency used to pay for the item.
  string cryptocurrency_;
  static jfieldID cryptocurrency_fieldID;
  /// The paid amount, in the smallest units of the cryptocurrency.
  int64 cryptocurrency_amount_;
  static jfieldID cryptocurrency_amount_fieldID;
  /// Individual URL for the item on https://fragment.com.
  string url_;
  static jfieldID url_fieldID;

  /**
   * Contains information about a collectible item and its last purchase.
   */
  collectibleItemInfo();

  /**
   * Contains information about a collectible item and its last purchase.
   *
   * \param[in] purchase_date_ Point in time (Unix timestamp) when the item was purchased.
   * \param[in] currency_ Currency for the paid amount.
   * \param[in] amount_ The paid amount, in the smallest units of the currency.
   * \param[in] cryptocurrency_ Cryptocurrency used to pay for the item.
   * \param[in] cryptocurrency_amount_ The paid amount, in the smallest units of the cryptocurrency.
   * \param[in] url_ Individual URL for the item on https://fragment.com.
   */
  collectibleItemInfo(int32 purchase_date_, string const &currency_, int53 amount_, string const &cryptocurrency_, int64 cryptocurrency_amount_, string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1460640717;

  static object_ptr<collectibleItemInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes a collectible item that can be purchased at https://fragment.com.
 */
class CollectibleItemType: public Object {
 public:
  static jclass Class;

  static object_ptr<CollectibleItemType> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A username.
 */
class collectibleItemTypeUsername final : public CollectibleItemType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The username.
  string username_;
  static jfieldID username_fieldID;

  /**
   * A username.
   */
  collectibleItemTypeUsername();

  /**
   * A username.
   *
   * \param[in] username_ The username.
   */
  explicit collectibleItemTypeUsername(string const &username_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 458680273;

  static object_ptr<CollectibleItemType> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A phone number.
 */
class collectibleItemTypePhoneNumber final : public CollectibleItemType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The phone number.
  string phone_number_;
  static jfieldID phone_number_fieldID;

  /**
   * A phone number.
   */
  collectibleItemTypePhoneNumber();

  /**
   * A phone number.
   *
   * \param[in] phone_number_ The phone number.
   */
  explicit collectibleItemTypePhoneNumber(string const &phone_number_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1256251714;

  static object_ptr<CollectibleItemType> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class affiliateProgramParameters;

/**
 * Describes an affiliate program that was connected to an affiliate.
 */
class connectedAffiliateProgram final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The link that can be used to refer users if the program is still active.
  string url_;
  static jfieldID url_fieldID;
  /// User identifier of the bot created the program.
  int53 bot_user_id_;
  static jfieldID bot_user_id_fieldID;
  /// The parameters of the affiliate program.
  object_ptr<affiliateProgramParameters> parameters_;
  static jfieldID parameters_fieldID;
  /// Point in time (Unix timestamp) when the affiliate program was connected.
  int32 connection_date_;
  static jfieldID connection_date_fieldID;
  /// True, if the program was canceled by the bot, or disconnected by the chat owner and isn't available anymore.
  bool is_disconnected_;
  static jfieldID is_disconnected_fieldID;
  /// The number of users that used the affiliate program.
  int64 user_count_;
  static jfieldID user_count_fieldID;
  /// The number of Telegram Stars that were earned by the affiliate program.
  int64 revenue_star_count_;
  static jfieldID revenue_star_count_fieldID;

  /**
   * Describes an affiliate program that was connected to an affiliate.
   */
  connectedAffiliateProgram();

  /**
   * Describes an affiliate program that was connected to an affiliate.
   *
   * \param[in] url_ The link that can be used to refer users if the program is still active.
   * \param[in] bot_user_id_ User identifier of the bot created the program.
   * \param[in] parameters_ The parameters of the affiliate program.
   * \param[in] connection_date_ Point in time (Unix timestamp) when the affiliate program was connected.
   * \param[in] is_disconnected_ True, if the program was canceled by the bot, or disconnected by the chat owner and isn't available anymore.
   * \param[in] user_count_ The number of users that used the affiliate program.
   * \param[in] revenue_star_count_ The number of Telegram Stars that were earned by the affiliate program.
   */
  connectedAffiliateProgram(string const &url_, int53 bot_user_id_, object_ptr<affiliateProgramParameters> &&parameters_, int32 connection_date_, bool is_disconnected_, int64 user_count_, int64 revenue_star_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1488942101;

  static object_ptr<connectedAffiliateProgram> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class connectedAffiliateProgram;

/**
 * Represents a list of affiliate programs that were connected to an affiliate.
 */
class connectedAffiliatePrograms final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The total number of affiliate programs that were connected to the affiliate.
  int32 total_count_;
  static jfieldID total_count_fieldID;
  /// The list of connected affiliate programs.
  array<object_ptr<connectedAffiliateProgram>> programs_;
  static jfieldID programs_fieldID;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;
  static jfieldID next_offset_fieldID;

  /**
   * Represents a list of affiliate programs that were connected to an affiliate.
   */
  connectedAffiliatePrograms();

  /**
   * Represents a list of affiliate programs that were connected to an affiliate.
   *
   * \param[in] total_count_ The total number of affiliate programs that were connected to the affiliate.
   * \param[in] programs_ The list of connected affiliate programs.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  connectedAffiliatePrograms(int32 total_count_, array<object_ptr<connectedAffiliateProgram>> &&programs_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1505880847;

  static object_ptr<connectedAffiliatePrograms> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains information about one website the current user is logged in with Telegram.
 */
class connectedWebsite final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Website identifier.
  int64 id_;
  static jfieldID id_fieldID;
  /// The domain name of the website.
  string domain_name_;
  static jfieldID domain_name_fieldID;
  /// User identifier of a bot linked with the website.
  int53 bot_user_id_;
  static jfieldID bot_user_id_fieldID;
  /// The version of a browser used to log in.
  string browser_;
  static jfieldID browser_fieldID;
  /// Operating system the browser is running on.
  string platform_;
  static jfieldID platform_fieldID;
  /// Point in time (Unix timestamp) when the user was logged in.
  int32 log_in_date_;
  static jfieldID log_in_date_fieldID;
  /// Point in time (Unix timestamp) when obtained authorization was last used.
  int32 last_active_date_;
  static jfieldID last_active_date_fieldID;
  /// IP address from which the user was logged in, in human-readable format.
  string ip_address_;
  static jfieldID ip_address_fieldID;
  /// Human-readable description of a country and a region from which the user was logged in, based on the IP address.
  string location_;
  static jfieldID location_fieldID;

  /**
   * Contains information about one website the current user is logged in with Telegram.
   */
  connectedWebsite();

  /**
   * Contains information about one website the current user is logged in with Telegram.
   *
   * \param[in] id_ Website identifier.
   * \param[in] domain_name_ The domain name of the website.
   * \param[in] bot_user_id_ User identifier of a bot linked with the website.
   * \param[in] browser_ The version of a browser used to log in.
   * \param[in] platform_ Operating system the browser is running on.
   * \param[in] log_in_date_ Point in time (Unix timestamp) when the user was logged in.
   * \param[in] last_active_date_ Point in time (Unix timestamp) when obtained authorization was last used.
   * \param[in] ip_address_ IP address from which the user was logged in, in human-readable format.
   * \param[in] location_ Human-readable description of a country and a region from which the user was logged in, based on the IP address.
   */
  connectedWebsite(int64 id_, string const &domain_name_, int53 bot_user_id_, string const &browser_, string const &platform_, int32 log_in_date_, int32 last_active_date_, string const &ip_address_, string const &location_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1978115978;

  static object_ptr<connectedWebsite> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class connectedWebsite;

/**
 * Contains a list of websites the current user is logged in with Telegram.
 */
class connectedWebsites final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// List of connected websites.
  array<object_ptr<connectedWebsite>> websites_;
  static jfieldID websites_fieldID;

  /**
   * Contains a list of websites the current user is logged in with Telegram.
   */
  connectedWebsites();

  /**
   * Contains a list of websites the current user is logged in with Telegram.
   *
   * \param[in] websites_ List of connected websites.
   */
  explicit connectedWebsites(array<object_ptr<connectedWebsite>> &&websites_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1727949694;

  static object_ptr<connectedWebsites> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes the current state of the connection to Telegram servers.
 */
class ConnectionState: public Object {
 public:
  static jclass Class;

  static object_ptr<ConnectionState> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Waiting for the network to become available. Use setNetworkType to change the available network type.
 */
class connectionStateWaitingForNetwork final : public ConnectionState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * Waiting for the network to become available. Use setNetworkType to change the available network type.
   */
  connectionStateWaitingForNetwork();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1695405912;

  static object_ptr<ConnectionState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Establishing a connection with a proxy server.
 */
class connectionStateConnectingToProxy final : public ConnectionState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * Establishing a connection with a proxy server.
   */
  connectionStateConnectingToProxy();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -93187239;

  static object_ptr<ConnectionState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Establishing a connection to the Telegram servers.
 */
class connectionStateConnecting final : public ConnectionState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * Establishing a connection to the Telegram servers.
   */
  connectionStateConnecting();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1298400670;

  static object_ptr<ConnectionState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Downloading data expected to be received while the application was offline.
 */
class connectionStateUpdating final : public ConnectionState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * Downloading data expected to be received while the application was offline.
   */
  connectionStateUpdating();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -188104009;

  static object_ptr<ConnectionState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * There is a working connection to the Telegram servers.
 */
class connectionStateReady final : public ConnectionState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * There is a working connection to the Telegram servers.
   */
  connectionStateReady();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 48608492;

  static object_ptr<ConnectionState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes a user contact.
 */
class contact final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Phone number of the user.
  string phone_number_;
  static jfieldID phone_number_fieldID;
  /// First name of the user; 1-255 characters in length.
  string first_name_;
  static jfieldID first_name_fieldID;
  /// Last name of the user.
  string last_name_;
  static jfieldID last_name_fieldID;
  /// Additional data about the user in a form of vCard; 0-2048 bytes in length.
  string vcard_;
  static jfieldID vcard_fieldID;
  /// Identifier of the user, if known; 0 otherwise.
  int53 user_id_;
  static jfieldID user_id_fieldID;

  /**
   * Describes a user contact.
   */
  contact();

  /**
   * Describes a user contact.
   *
   * \param[in] phone_number_ Phone number of the user.
   * \param[in] first_name_ First name of the user; 1-255 characters in length.
   * \param[in] last_name_ Last name of the user.
   * \param[in] vcard_ Additional data about the user in a form of vCard; 0-2048 bytes in length.
   * \param[in] user_id_ Identifier of the user, if known; 0 otherwise.
   */
  contact(string const &phone_number_, string const &first_name_, string const &last_name_, string const &vcard_, int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1993844876;

  static object_ptr<contact> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains a counter.
 */
class count final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Count.
  int32 count_;
  static jfieldID count_fieldID;

  /**
   * Contains a counter.
   */
  count();

  /**
   * Contains a counter.
   *
   * \param[in] count_ Count.
   */
  explicit count(int32 count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1295577348;

  static object_ptr<count> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class countryInfo;

/**
 * Contains information about countries.
 */
class countries final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The list of countries.
  array<object_ptr<countryInfo>> countries_;
  static jfieldID countries_fieldID;

  /**
   * Contains information about countries.
   */
  countries();

  /**
   * Contains information about countries.
   *
   * \param[in] countries_ The list of countries.
   */
  explicit countries(array<object_ptr<countryInfo>> &&countries_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1854211813;

  static object_ptr<countries> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains information about a country.
 */
class countryInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A two-letter ISO 3166-1 alpha-2 country code.
  string country_code_;
  static jfieldID country_code_fieldID;
  /// Native name of the country.
  string name_;
  static jfieldID name_fieldID;
  /// English name of the country.
  string english_name_;
  static jfieldID english_name_fieldID;
  /// True, if the country must be hidden from the list of all countries.
  bool is_hidden_;
  static jfieldID is_hidden_fieldID;
  /// List of country calling codes.
  array<string> calling_codes_;
  static jfieldID calling_codes_fieldID;

  /**
   * Contains information about a country.
   */
  countryInfo();

  /**
   * Contains information about a country.
   *
   * \param[in] country_code_ A two-letter ISO 3166-1 alpha-2 country code.
   * \param[in] name_ Native name of the country.
   * \param[in] english_name_ English name of the country.
   * \param[in] is_hidden_ True, if the country must be hidden from the list of all countries.
   * \param[in] calling_codes_ List of country calling codes.
   */
  countryInfo(string const &country_code_, string const &name_, string const &english_name_, bool is_hidden_, array<string> &&calling_codes_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1617195722;

  static object_ptr<countryInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class failedToAddMembers;

/**
 * Contains information about a newly created basic group chat.
 */
class createdBasicGroupChat final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Chat identifier.
  int53 chat_id_;
  static jfieldID chat_id_fieldID;
  /// Information about failed to add members.
  object_ptr<failedToAddMembers> failed_to_add_members_;
  static jfieldID failed_to_add_members_fieldID;

  /**
   * Contains information about a newly created basic group chat.
   */
  createdBasicGroupChat();

  /**
   * Contains information about a newly created basic group chat.
   *
   * \param[in] chat_id_ Chat identifier.
   * \param[in] failed_to_add_members_ Information about failed to add members.
   */
  createdBasicGroupChat(int53 chat_id_, object_ptr<failedToAddMembers> &&failed_to_add_members_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -20417068;

  static object_ptr<createdBasicGroupChat> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes the current weather.
 */
class currentWeather final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Temperature, in degree Celsius.
  double temperature_;
  static jfieldID temperature_fieldID;
  /// Emoji representing the weather.
  string emoji_;
  static jfieldID emoji_fieldID;

  /**
   * Describes the current weather.
   */
  currentWeather();

  /**
   * Describes the current weather.
   *
   * \param[in] temperature_ Temperature, in degree Celsius.
   * \param[in] emoji_ Emoji representing the weather.
   */
  currentWeather(double temperature_, string const &emoji_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -355555136;

  static object_ptr<currentWeather> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains the result of a custom request.
 */
class customRequestResult final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A JSON-serialized result.
  string result_;
  static jfieldID result_fieldID;

  /**
   * Contains the result of a custom request.
   */
  customRequestResult();

  /**
   * Contains the result of a custom request.
   *
   * \param[in] result_ A JSON-serialized result.
   */
  explicit customRequestResult(string const &result_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2009960452;

  static object_ptr<customRequestResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains some binary data.
 */
class data final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Data.
  bytes data_;
  static jfieldID data_fieldID;

  /**
   * Contains some binary data.
   */
  data();

  /**
   * Contains some binary data.
   *
   * \param[in] data_ Data.
   */
  explicit data(bytes const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 221197337;

  static object_ptr<data> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains database statistics.
 */
class databaseStatistics final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Database statistics in an unspecified human-readable format.
  string statistics_;
  static jfieldID statistics_fieldID;

  /**
   * Contains database statistics.
   */
  databaseStatistics();

  /**
   * Contains database statistics.
   *
   * \param[in] statistics_ Database statistics in an unspecified human-readable format.
   */
  explicit databaseStatistics(string const &statistics_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1123912880;

  static object_ptr<databaseStatistics> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents a date according to the Gregorian calendar.
 */
class date final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Day of the month; 1-31.
  int32 day_;
  static jfieldID day_fieldID;
  /// Month; 1-12.
  int32 month_;
  static jfieldID month_fieldID;
  /// Year; 1-9999.
  int32 year_;
  static jfieldID year_fieldID;

  /**
   * Represents a date according to the Gregorian calendar.
   */
  date();

  /**
   * Represents a date according to the Gregorian calendar.
   *
   * \param[in] day_ Day of the month; 1-31.
   * \param[in] month_ Month; 1-12.
   * \param[in] year_ Year; 1-9999.
   */
  date(int32 day_, int32 month_, int32 year_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -277956960;

  static object_ptr<date> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents a date range.
 */
class dateRange final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Point in time (Unix timestamp) at which the date range begins.
  int32 start_date_;
  static jfieldID start_date_fieldID;
  /// Point in time (Unix timestamp) at which the date range ends.
  int32 end_date_;
  static jfieldID end_date_fieldID;

  /**
   * Represents a date range.
   */
  dateRange();

  /**
   * Represents a date range.
   *
   * \param[in] start_date_ Point in time (Unix timestamp) at which the date range begins.
   * \param[in] end_date_ Point in time (Unix timestamp) at which the date range ends.
   */
  dateRange(int32 start_date_, int32 end_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1360333926;

  static object_ptr<dateRange> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class file;

/**
 * File with the date it was uploaded.
 */
class datedFile final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The file.
  object_ptr<file> file_;
  static jfieldID file_fieldID;
  /// Point in time (Unix timestamp) when the file was uploaded.
  int32 date_;
  static jfieldID date_fieldID;

  /**
   * File with the date it was uploaded.
   */
  datedFile();

  /**
   * File with the date it was uploaded.
   *
   * \param[in] file_ The file.
   * \param[in] date_ Point in time (Unix timestamp) when the file was uploaded.
   */
  datedFile(object_ptr<file> &&file_, int32 date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1840795491;

  static object_ptr<datedFile> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class formattedText;

/**
 * Contains information about a tg: deep link.
 */
class deepLinkInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Text to be shown to the user.
  object_ptr<formattedText> text_;
  static jfieldID text_fieldID;
  /// True, if the user must be asked to update the application.
  bool need_update_application_;
  static jfieldID need_update_application_fieldID;

  /**
   * Contains information about a tg: deep link.
   */
  deepLinkInfo();

  /**
   * Contains information about a tg: deep link.
   *
   * \param[in] text_ Text to be shown to the user.
   * \param[in] need_update_application_ True, if the user must be asked to update the application.
   */
  deepLinkInfo(object_ptr<formattedText> &&text_, bool need_update_application_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1864081662;

  static object_ptr<deepLinkInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Represents a data needed to subscribe for push notifications through registerDevice method. To use specific push notification service, the correct application platform must be specified and a valid server authentication data must be uploaded at https://my.telegram.org.
 */
class DeviceToken: public Object {
 public:
  static jclass Class;

  static object_ptr<DeviceToken> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A token for Firebase Cloud Messaging.
 */
class deviceTokenFirebaseCloudMessaging final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Device registration token; may be empty to deregister a device.
  string token_;
  static jfieldID token_fieldID;
  /// True, if push notifications must be additionally encrypted.
  bool encrypt_;
  static jfieldID encrypt_fieldID;

  /**
   * A token for Firebase Cloud Messaging.
   */
  deviceTokenFirebaseCloudMessaging();

  /**
   * A token for Firebase Cloud Messaging.
   *
   * \param[in] token_ Device registration token; may be empty to deregister a device.
   * \param[in] encrypt_ True, if push notifications must be additionally encrypted.
   */
  deviceTokenFirebaseCloudMessaging(string const &token_, bool encrypt_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -797881849;

  static object_ptr<DeviceToken> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A token for Apple Push Notification service.
 */
class deviceTokenApplePush final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Device token; may be empty to deregister a device.
  string device_token_;
  static jfieldID device_token_fieldID;
  /// True, if App Sandbox is enabled.
  bool is_app_sandbox_;
  static jfieldID is_app_sandbox_fieldID;

  /**
   * A token for Apple Push Notification service.
   */
  deviceTokenApplePush();

  /**
   * A token for Apple Push Notification service.
   *
   * \param[in] device_token_ Device token; may be empty to deregister a device.
   * \param[in] is_app_sandbox_ True, if App Sandbox is enabled.
   */
  deviceTokenApplePush(string const &device_token_, bool is_app_sandbox_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 387541955;

  static object_ptr<DeviceToken> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A token for Apple Push Notification service VoIP notifications.
 */
class deviceTokenApplePushVoIP final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Device token; may be empty to deregister a device.
  string device_token_;
  static jfieldID device_token_fieldID;
  /// True, if App Sandbox is enabled.
  bool is_app_sandbox_;
  static jfieldID is_app_sandbox_fieldID;
  /// True, if push notifications must be additionally encrypted.
  bool encrypt_;
  static jfieldID encrypt_fieldID;

  /**
   * A token for Apple Push Notification service VoIP notifications.
   */
  deviceTokenApplePushVoIP();

  /**
   * A token for Apple Push Notification service VoIP notifications.
   *
   * \param[in] device_token_ Device token; may be empty to deregister a device.
   * \param[in] is_app_sandbox_ True, if App Sandbox is enabled.
   * \param[in] encrypt_ True, if push notifications must be additionally encrypted.
   */
  deviceTokenApplePushVoIP(string const &device_token_, bool is_app_sandbox_, bool encrypt_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 804275689;

  static object_ptr<DeviceToken> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A token for Windows Push Notification Services.
 */
class deviceTokenWindowsPush final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The access token that will be used to send notifications; may be empty to deregister a device.
  string access_token_;
  static jfieldID access_token_fieldID;

  /**
   * A token for Windows Push Notification Services.
   */
  deviceTokenWindowsPush();

  /**
   * A token for Windows Push Notification Services.
   *
   * \param[in] access_token_ The access token that will be used to send notifications; may be empty to deregister a device.
   */
  explicit deviceTokenWindowsPush(string const &access_token_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1410514289;

  static object_ptr<DeviceToken> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A token for Microsoft Push Notification Service.
 */
class deviceTokenMicrosoftPush final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Push notification channel URI; may be empty to deregister a device.
  string channel_uri_;
  static jfieldID channel_uri_fieldID;

  /**
   * A token for Microsoft Push Notification Service.
   */
  deviceTokenMicrosoftPush();

  /**
   * A token for Microsoft Push Notification Service.
   *
   * \param[in] channel_uri_ Push notification channel URI; may be empty to deregister a device.
   */
  explicit deviceTokenMicrosoftPush(string const &channel_uri_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1224269900;

  static object_ptr<DeviceToken> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A token for Microsoft Push Notification Service VoIP channel.
 */
class deviceTokenMicrosoftPushVoIP final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Push notification channel URI; may be empty to deregister a device.
  string channel_uri_;
  static jfieldID channel_uri_fieldID;

  /**
   * A token for Microsoft Push Notification Service VoIP channel.
   */
  deviceTokenMicrosoftPushVoIP();

  /**
   * A token for Microsoft Push Notification Service VoIP channel.
   *
   * \param[in] channel_uri_ Push notification channel URI; may be empty to deregister a device.
   */
  explicit deviceTokenMicrosoftPushVoIP(string const &channel_uri_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -785603759;

  static object_ptr<DeviceToken> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A token for web Push API.
 */
class deviceTokenWebPush final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Absolute URL exposed by the push service where the application server can send push messages; may be empty to deregister a device.
  string endpoint_;
  static jfieldID endpoint_fieldID;
  /// Base64url-encoded P-256 elliptic curve Diffie-Hellman public key.
  string p256dh_base64url_;
  static jfieldID p256dh_base64url_fieldID;
  /// Base64url-encoded authentication secret.
  string auth_base64url_;
  static jfieldID auth_base64url_fieldID;

  /**
   * A token for web Push API.
   */
  deviceTokenWebPush();

  /**
   * A token for web Push API.
   *
   * \param[in] endpoint_ Absolute URL exposed by the push service where the application server can send push messages; may be empty to deregister a device.
   * \param[in] p256dh_base64url_ Base64url-encoded P-256 elliptic curve Diffie-Hellman public key.
   * \param[in] auth_base64url_ Base64url-encoded authentication secret.
   */
  deviceTokenWebPush(string const &endpoint_, string const &p256dh_base64url_, string const &auth_base64url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1694507273;

  static object_ptr<DeviceToken> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A token for Simple Push API for Firefox OS.
 */
class deviceTokenSimplePush final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Absolute URL exposed by the push service where the application server can send push messages; may be empty to deregister a device.
  string endpoint_;
  static jfieldID endpoint_fieldID;

  /**
   * A token for Simple Push API for Firefox OS.
   */
  deviceTokenSimplePush();

  /**
   * A token for Simple Push API for Firefox OS.
   *
   * \param[in] endpoint_ Absolute URL exposed by the push service where the application server can send push messages; may be empty to deregister a device.
   */
  explicit deviceTokenSimplePush(string const &endpoint_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 49584736;

  static object_ptr<DeviceToken> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A token for Ubuntu Push Client service.
 */
class deviceTokenUbuntuPush final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Token; may be empty to deregister a device.
  string token_;
  static jfieldID token_fieldID;

  /**
   * A token for Ubuntu Push Client service.
   */
  deviceTokenUbuntuPush();

  /**
   * A token for Ubuntu Push Client service.
   *
   * \param[in] token_ Token; may be empty to deregister a device.
   */
  explicit deviceTokenUbuntuPush(string const &token_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1782320422;

  static object_ptr<DeviceToken> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A token for BlackBerry Push Service.
 */
class deviceTokenBlackBerryPush final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Token; may be empty to deregister a device.
  string token_;
  static jfieldID token_fieldID;

  /**
   * A token for BlackBerry Push Service.
   */
  deviceTokenBlackBerryPush();

  /**
   * A token for BlackBerry Push Service.
   *
   * \param[in] token_ Token; may be empty to deregister a device.
   */
  explicit deviceTokenBlackBerryPush(string const &token_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1559167234;

  static object_ptr<DeviceToken> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A token for Tizen Push Service.
 */
class deviceTokenTizenPush final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Push service registration identifier; may be empty to deregister a device.
  string reg_id_;
  static jfieldID reg_id_fieldID;

  /**
   * A token for Tizen Push Service.
   */
  deviceTokenTizenPush();

  /**
   * A token for Tizen Push Service.
   *
   * \param[in] reg_id_ Push service registration identifier; may be empty to deregister a device.
   */
  explicit deviceTokenTizenPush(string const &reg_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1359947213;

  static object_ptr<DeviceToken> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A token for HUAWEI Push Service.
 */
class deviceTokenHuaweiPush final : public DeviceToken {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Device registration token; may be empty to deregister a device.
  string token_;
  static jfieldID token_fieldID;
  /// True, if push notifications must be additionally encrypted.
  bool encrypt_;
  static jfieldID encrypt_fieldID;

  /**
   * A token for HUAWEI Push Service.
   */
  deviceTokenHuaweiPush();

  /**
   * A token for HUAWEI Push Service.
   *
   * \param[in] token_ Device registration token; may be empty to deregister a device.
   * \param[in] encrypt_ True, if push notifications must be additionally encrypted.
   */
  deviceTokenHuaweiPush(string const &token_, bool encrypt_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1989103142;

  static object_ptr<DeviceToken> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class sticker;

/**
 * This class is an abstract base class.
 * Contains animated stickers which must be used for dice animation rendering.
 */
class DiceStickers: public Object {
 public:
  static jclass Class;

  static object_ptr<DiceStickers> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A regular animated sticker.
 */
class diceStickersRegular final : public DiceStickers {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The animated sticker with the dice animation.
  object_ptr<sticker> sticker_;
  static jfieldID sticker_fieldID;

  /**
   * A regular animated sticker.
   */
  diceStickersRegular();

  /**
   * A regular animated sticker.
   *
   * \param[in] sticker_ The animated sticker with the dice animation.
   */
  explicit diceStickersRegular(object_ptr<sticker> &&sticker_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -740299570;

  static object_ptr<DiceStickers> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Animated stickers to be combined into a slot machine.
 */
class diceStickersSlotMachine final : public DiceStickers {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The animated sticker with the slot machine background. The background animation must start playing after all reel animations finish.
  object_ptr<sticker> background_;
  static jfieldID background_fieldID;
  /// The animated sticker with the lever animation. The lever animation must play once in the initial dice state.
  object_ptr<sticker> lever_;
  static jfieldID lever_fieldID;
  /// The animated sticker with the left reel.
  object_ptr<sticker> left_reel_;
  static jfieldID left_reel_fieldID;
  /// The animated sticker with the center reel.
  object_ptr<sticker> center_reel_;
  static jfieldID center_reel_fieldID;
  /// The animated sticker with the right reel.
  object_ptr<sticker> right_reel_;
  static jfieldID right_reel_fieldID;

  /**
   * Animated stickers to be combined into a slot machine.
   */
  diceStickersSlotMachine();

  /**
   * Animated stickers to be combined into a slot machine.
   *
   * \param[in] background_ The animated sticker with the slot machine background. The background animation must start playing after all reel animations finish.
   * \param[in] lever_ The animated sticker with the lever animation. The lever animation must play once in the initial dice state.
   * \param[in] left_reel_ The animated sticker with the left reel.
   * \param[in] center_reel_ The animated sticker with the center reel.
   * \param[in] right_reel_ The animated sticker with the right reel.
   */
  diceStickersSlotMachine(object_ptr<sticker> &&background_, object_ptr<sticker> &&lever_, object_ptr<sticker> &&left_reel_, object_ptr<sticker> &&center_reel_, object_ptr<sticker> &&right_reel_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -375223124;

  static object_ptr<DiceStickers> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class MessageSender;

class draftMessage;

class message;

/**
 * Contains information about a topic in a channel direct messages chat administered by the current user.
 */
class directMessagesChatTopic final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the chat to which the topic belongs.
  int53 chat_id_;
  static jfieldID chat_id_fieldID;
  /// Unique topic identifier.
  int53 id_;
  static jfieldID id_fieldID;
  /// Identifier of the user or chat that sends the messages to the topic.
  object_ptr<MessageSender> sender_id_;
  static jfieldID sender_id_fieldID;
  /// A parameter used to determine order of the topic in the topic list. Topics must be sorted by the order in descending order.
  int64 order_;
  static jfieldID order_fieldID;
  /// True, if the other party can send unpaid messages even if the chat has paid messages enabled.
  bool can_send_unpaid_messages_;
  static jfieldID can_send_unpaid_messages_fieldID;
  /// True, if the forum topic is marked as unread.
  bool is_marked_as_unread_;
  static jfieldID is_marked_as_unread_fieldID;
  /// Number of unread messages in the chat.
  int53 unread_count_;
  static jfieldID unread_count_fieldID;
  /// Identifier of the last read incoming message.
  int53 last_read_inbox_message_id_;
  static jfieldID last_read_inbox_message_id_fieldID;
  /// Identifier of the last read outgoing message.
  int53 last_read_outbox_message_id_;
  static jfieldID last_read_outbox_message_id_fieldID;
  /// Number of messages with unread reactions in the chat.
  int53 unread_reaction_count_;
  static jfieldID unread_reaction_count_fieldID;
  /// Last message in the topic; may be null if none or unknown.
  object_ptr<message> last_message_;
  static jfieldID last_message_fieldID;
  /// A draft of a message in the topic; may be null if none.
  object_ptr<draftMessage> draft_message_;
  static jfieldID draft_message_fieldID;

  /**
   * Contains information about a topic in a channel direct messages chat administered by the current user.
   */
  directMessagesChatTopic();

  /**
   * Contains information about a topic in a channel direct messages chat administered by the current user.
   *
   * \param[in] chat_id_ Identifier of the chat to which the topic belongs.
   * \param[in] id_ Unique topic identifier.
   * \param[in] sender_id_ Identifier of the user or chat that sends the messages to the topic.
   * \param[in] order_ A parameter used to determine order of the topic in the topic list. Topics must be sorted by the order in descending order.
   * \param[in] can_send_unpaid_messages_ True, if the other party can send unpaid messages even if the chat has paid messages enabled.
   * \param[in] is_marked_as_unread_ True, if the forum topic is marked as unread.
   * \param[in] unread_count_ Number of unread messages in the chat.
   * \param[in] last_read_inbox_message_id_ Identifier of the last read incoming message.
   * \param[in] last_read_outbox_message_id_ Identifier of the last read outgoing message.
   * \param[in] unread_reaction_count_ Number of messages with unread reactions in the chat.
   * \param[in] last_message_ Last message in the topic; may be null if none or unknown.
   * \param[in] draft_message_ A draft of a message in the topic; may be null if none.
   */
  directMessagesChatTopic(int53 chat_id_, int53 id_, object_ptr<MessageSender> &&sender_id_, int64 order_, bool can_send_unpaid_messages_, bool is_marked_as_unread_, int53 unread_count_, int53 last_read_inbox_message_id_, int53 last_read_outbox_message_id_, int53 unread_reaction_count_, object_ptr<message> &&last_message_, object_ptr<draftMessage> &&draft_message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1778377757;

  static object_ptr<directMessagesChatTopic> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class file;

class minithumbnail;

class thumbnail;

/**
 * Describes a document of any type.
 */
class document final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Original name of the file; as defined by the sender.
  string file_name_;
  static jfieldID file_name_fieldID;
  /// MIME type of the file; as defined by the sender.
  string mime_type_;
  static jfieldID mime_type_fieldID;
  /// Document minithumbnail; may be null.
  object_ptr<minithumbnail> minithumbnail_;
  static jfieldID minithumbnail_fieldID;
  /// Document thumbnail in JPEG or PNG format (PNG will be used only for background patterns); as defined by the sender; may be null.
  object_ptr<thumbnail> thumbnail_;
  static jfieldID thumbnail_fieldID;
  /// File containing the document.
  object_ptr<file> document_;
  static jfieldID document_fieldID;

  /**
   * Describes a document of any type.
   */
  document();

  /**
   * Describes a document of any type.
   *
   * \param[in] file_name_ Original name of the file; as defined by the sender.
   * \param[in] mime_type_ MIME type of the file; as defined by the sender.
   * \param[in] minithumbnail_ Document minithumbnail; may be null.
   * \param[in] thumbnail_ Document thumbnail in JPEG or PNG format (PNG will be used only for background patterns); as defined by the sender; may be null.
   * \param[in] document_ File containing the document.
   */
  document(string const &file_name_, string const &mime_type_, object_ptr<minithumbnail> &&minithumbnail_, object_ptr<thumbnail> &&thumbnail_, object_ptr<file> &&document_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1357271080;

  static object_ptr<document> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains number of being downloaded and recently downloaded files found.
 */
class downloadedFileCounts final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Number of active file downloads found, including paused.
  int32 active_count_;
  static jfieldID active_count_fieldID;
  /// Number of paused file downloads found.
  int32 paused_count_;
  static jfieldID paused_count_fieldID;
  /// Number of completed file downloads found.
  int32 completed_count_;
  static jfieldID completed_count_fieldID;

  /**
   * Contains number of being downloaded and recently downloaded files found.
   */
  downloadedFileCounts();

  /**
   * Contains number of being downloaded and recently downloaded files found.
   *
   * \param[in] active_count_ Number of active file downloads found, including paused.
   * \param[in] paused_count_ Number of paused file downloads found.
   * \param[in] completed_count_ Number of completed file downloads found.
   */
  downloadedFileCounts(int32 active_count_, int32 paused_count_, int32 completed_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1973999550;

  static object_ptr<downloadedFileCounts> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class InputMessageContent;

class InputMessageReplyTo;

class inputSuggestedPostInfo;

/**
 * Contains information about a message draft.
 */
class draftMessage final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Information about the message to be replied; must be of the type inputMessageReplyToMessage; may be null if none.
  object_ptr<InputMessageReplyTo> reply_to_;
  static jfieldID reply_to_fieldID;
  /// Point in time (Unix timestamp) when the draft was created.
  int32 date_;
  static jfieldID date_fieldID;
  /// Content of the message draft; must be of the type inputMessageText, inputMessageVideoNote, or inputMessageVoiceNote.
  object_ptr<InputMessageContent> input_message_text_;
  static jfieldID input_message_text_fieldID;
  /// Identifier of the effect to apply to the message when it is sent; 0 if none.
  int64 effect_id_;
  static jfieldID effect_id_fieldID;
  /// Information about the suggested post; may be null if none.
  object_ptr<inputSuggestedPostInfo> suggested_post_info_;
  static jfieldID suggested_post_info_fieldID;

  /**
   * Contains information about a message draft.
   */
  draftMessage();

  /**
   * Contains information about a message draft.
   *
   * \param[in] reply_to_ Information about the message to be replied; must be of the type inputMessageReplyToMessage; may be null if none.
   * \param[in] date_ Point in time (Unix timestamp) when the draft was created.
   * \param[in] input_message_text_ Content of the message draft; must be of the type inputMessageText, inputMessageVideoNote, or inputMessageVoiceNote.
   * \param[in] effect_id_ Identifier of the effect to apply to the message when it is sent; 0 if none.
   * \param[in] suggested_post_info_ Information about the suggested post; may be null if none.
   */
  draftMessage(object_ptr<InputMessageReplyTo> &&reply_to_, int32 date_, object_ptr<InputMessageContent> &&input_message_text_, int64 effect_id_, object_ptr<inputSuggestedPostInfo> &&suggested_post_info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1165040650;

  static object_ptr<draftMessage> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Contains authentication data for an email address.
 */
class EmailAddressAuthentication: public Object {
 public:
  static jclass Class;

  static object_ptr<EmailAddressAuthentication> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * An authentication code delivered to a user's email address.
 */
class emailAddressAuthenticationCode final : public EmailAddressAuthentication {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The code.
  string code_;
  static jfieldID code_fieldID;

  /**
   * An authentication code delivered to a user's email address.
   */
  emailAddressAuthenticationCode();

  /**
   * An authentication code delivered to a user's email address.
   *
   * \param[in] code_ The code.
   */
  explicit emailAddressAuthenticationCode(string const &code_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -993257022;

  static object_ptr<EmailAddressAuthentication> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * An authentication token received through Apple ID.
 */
class emailAddressAuthenticationAppleId final : public EmailAddressAuthentication {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The token.
  string token_;
  static jfieldID token_fieldID;

  /**
   * An authentication token received through Apple ID.
   */
  emailAddressAuthenticationAppleId();

  /**
   * An authentication token received through Apple ID.
   *
   * \param[in] token_ The token.
   */
  explicit emailAddressAuthenticationAppleId(string const &token_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 633948265;

  static object_ptr<EmailAddressAuthentication> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * An authentication token received through Google ID.
 */
class emailAddressAuthenticationGoogleId final : public EmailAddressAuthentication {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The token.
  string token_;
  static jfieldID token_fieldID;

  /**
   * An authentication token received through Google ID.
   */
  emailAddressAuthenticationGoogleId();

  /**
   * An authentication token received through Google ID.
   *
   * \param[in] token_ The token.
   */
  explicit emailAddressAuthenticationGoogleId(string const &token_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -19142846;

  static object_ptr<EmailAddressAuthentication> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Information about the email address authentication code that was sent.
 */
class emailAddressAuthenticationCodeInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Pattern of the email address to which an authentication code was sent.
  string email_address_pattern_;
  static jfieldID email_address_pattern_fieldID;
  /// Length of the code; 0 if unknown.
  int32 length_;
  static jfieldID length_fieldID;

  /**
   * Information about the email address authentication code that was sent.
   */
  emailAddressAuthenticationCodeInfo();

  /**
   * Information about the email address authentication code that was sent.
   *
   * \param[in] email_address_pattern_ Pattern of the email address to which an authentication code was sent.
   * \param[in] length_ Length of the code; 0 if unknown.
   */
  emailAddressAuthenticationCodeInfo(string const &email_address_pattern_, int32 length_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1151066659;

  static object_ptr<emailAddressAuthenticationCodeInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes reset state of an email address.
 */
class EmailAddressResetState: public Object {
 public:
  static jclass Class;

  static object_ptr<EmailAddressResetState> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Email address can be reset after the given period. Call resetAuthenticationEmailAddress to reset it and allow the user to authorize with a code sent to the user's phone number.
 */
class emailAddressResetStateAvailable final : public EmailAddressResetState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Time required to wait before the email address can be reset; 0 if the user is subscribed to Telegram Premium.
  int32 wait_period_;
  static jfieldID wait_period_fieldID;

  /**
   * Email address can be reset after the given period. Call resetAuthenticationEmailAddress to reset it and allow the user to authorize with a code sent to the user's phone number.
   */
  emailAddressResetStateAvailable();

  /**
   * Email address can be reset after the given period. Call resetAuthenticationEmailAddress to reset it and allow the user to authorize with a code sent to the user's phone number.
   *
   * \param[in] wait_period_ Time required to wait before the email address can be reset; 0 if the user is subscribed to Telegram Premium.
   */
  explicit emailAddressResetStateAvailable(int32 wait_period_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1917177600;

  static object_ptr<EmailAddressResetState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Email address reset has already been requested. Call resetAuthenticationEmailAddress to check whether immediate reset is possible.
 */
class emailAddressResetStatePending final : public EmailAddressResetState {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Left time before the email address will be reset, in seconds. updateAuthorizationState is not sent when this field changes.
  int32 reset_in_;
  static jfieldID reset_in_fieldID;

  /**
   * Email address reset has already been requested. Call resetAuthenticationEmailAddress to check whether immediate reset is possible.
   */
  emailAddressResetStatePending();

  /**
   * Email address reset has already been requested. Call resetAuthenticationEmailAddress to check whether immediate reset is possible.
   *
   * \param[in] reset_in_ Left time before the email address will be reset, in seconds. updateAuthorizationState is not sent when this field changes.
   */
  explicit emailAddressResetStatePending(int32 reset_in_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1885966805;

  static object_ptr<EmailAddressResetState> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class emojiCategory;

/**
 * Represents a list of emoji categories.
 */
class emojiCategories final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// List of categories.
  array<object_ptr<emojiCategory>> categories_;
  static jfieldID categories_fieldID;

  /**
   * Represents a list of emoji categories.
   */
  emojiCategories();

  /**
   * Represents a list of emoji categories.
   *
   * \param[in] categories_ List of categories.
   */
  explicit emojiCategories(array<object_ptr<emojiCategory>> &&categories_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1455387824;

  static object_ptr<emojiCategories> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class EmojiCategorySource;

class sticker;

/**
 * Describes an emoji category.
 */
class emojiCategory final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Name of the category.
  string name_;
  static jfieldID name_fieldID;
  /// Custom emoji sticker, which represents icon of the category.
  object_ptr<sticker> icon_;
  static jfieldID icon_fieldID;
  /// Source of stickers for the emoji category.
  object_ptr<EmojiCategorySource> source_;
  static jfieldID source_fieldID;
  /// True, if the category must be shown first when choosing a sticker for the start page.
  bool is_greeting_;
  static jfieldID is_greeting_fieldID;

  /**
   * Describes an emoji category.
   */
  emojiCategory();

  /**
   * Describes an emoji category.
   *
   * \param[in] name_ Name of the category.
   * \param[in] icon_ Custom emoji sticker, which represents icon of the category.
   * \param[in] source_ Source of stickers for the emoji category.
   * \param[in] is_greeting_ True, if the category must be shown first when choosing a sticker for the start page.
   */
  emojiCategory(string const &name_, object_ptr<sticker> &&icon_, object_ptr<EmojiCategorySource> &&source_, bool is_greeting_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 571335919;

  static object_ptr<emojiCategory> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes source of stickers for an emoji category.
 */
class EmojiCategorySource: public Object {
 public:
  static jclass Class;

  static object_ptr<EmojiCategorySource> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The category contains a list of similar emoji to search for in getStickers and searchStickers for stickers, or getInlineQueryResults with the bot getOption(&quot;animation_search_bot_username&quot;) for animations.
 */
class emojiCategorySourceSearch final : public EmojiCategorySource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// List of emojis to search for.
  array<string> emojis_;
  static jfieldID emojis_fieldID;

  /**
   * The category contains a list of similar emoji to search for in getStickers and searchStickers for stickers, or getInlineQueryResults with the bot getOption(&quot;animation_search_bot_username&quot;) for animations.
   */
  emojiCategorySourceSearch();

  /**
   * The category contains a list of similar emoji to search for in getStickers and searchStickers for stickers, or getInlineQueryResults with the bot getOption(&quot;animation_search_bot_username&quot;) for animations.
   *
   * \param[in] emojis_ List of emojis to search for.
   */
  explicit emojiCategorySourceSearch(array<string> &&emojis_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -453260262;

  static object_ptr<EmojiCategorySource> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The category contains premium stickers that must be found by getPremiumStickers.
 */
class emojiCategorySourcePremium final : public EmojiCategorySource {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The category contains premium stickers that must be found by getPremiumStickers.
   */
  emojiCategorySourcePremium();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1932358388;

  static object_ptr<EmojiCategorySource> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes type of emoji category.
 */
class EmojiCategoryType: public Object {
 public:
  static jclass Class;

  static object_ptr<EmojiCategoryType> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The category must be used by default (e.g., for custom emoji or animation search).
 */
class emojiCategoryTypeDefault final : public EmojiCategoryType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The category must be used by default (e.g., for custom emoji or animation search).
   */
  emojiCategoryTypeDefault();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1188782699;

  static object_ptr<EmojiCategoryType> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The category must be used by default for regular sticker selection. It may contain greeting emoji category and premium stickers.
 */
class emojiCategoryTypeRegularStickers final : public EmojiCategoryType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The category must be used by default for regular sticker selection. It may contain greeting emoji category and premium stickers.
   */
  emojiCategoryTypeRegularStickers();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1337484846;

  static object_ptr<EmojiCategoryType> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The category must be used for emoji status selection.
 */
class emojiCategoryTypeEmojiStatus final : public EmojiCategoryType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The category must be used for emoji status selection.
   */
  emojiCategoryTypeEmojiStatus();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1381282631;

  static object_ptr<EmojiCategoryType> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The category must be used for chat photo emoji selection.
 */
class emojiCategoryTypeChatPhoto final : public EmojiCategoryType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The category must be used for chat photo emoji selection.
   */
  emojiCategoryTypeChatPhoto();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1059063081;

  static object_ptr<EmojiCategoryType> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents an emoji with its keyword.
 */
class emojiKeyword final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The emoji.
  string emoji_;
  static jfieldID emoji_fieldID;
  /// The keyword.
  string keyword_;
  static jfieldID keyword_fieldID;

  /**
   * Represents an emoji with its keyword.
   */
  emojiKeyword();

  /**
   * Represents an emoji with its keyword.
   *
   * \param[in] emoji_ The emoji.
   * \param[in] keyword_ The keyword.
   */
  emojiKeyword(string const &emoji_, string const &keyword_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2112285985;

  static object_ptr<emojiKeyword> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class emojiKeyword;

/**
 * Represents a list of emojis with their keywords.
 */
class emojiKeywords final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// List of emojis with their keywords.
  array<object_ptr<emojiKeyword>> emoji_keywords_;
  static jfieldID emoji_keywords_fieldID;

  /**
   * Represents a list of emojis with their keywords.
   */
  emojiKeywords();

  /**
   * Represents a list of emojis with their keywords.
   *
   * \param[in] emoji_keywords_ List of emojis with their keywords.
   */
  explicit emojiKeywords(array<object_ptr<emojiKeyword>> &&emoji_keywords_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 689723339;

  static object_ptr<emojiKeywords> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class sticker;

/**
 * Contains information about an emoji reaction.
 */
class emojiReaction final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Text representation of the reaction.
  string emoji_;
  static jfieldID emoji_fieldID;
  /// Reaction title.
  string title_;
  static jfieldID title_fieldID;
  /// True, if the reaction can be added to new messages and enabled in chats.
  bool is_active_;
  static jfieldID is_active_fieldID;
  /// Static icon for the reaction.
  object_ptr<sticker> static_icon_;
  static jfieldID static_icon_fieldID;
  /// Appear animation for the reaction.
  object_ptr<sticker> appear_animation_;
  static jfieldID appear_animation_fieldID;
  /// Select animation for the reaction.
  object_ptr<sticker> select_animation_;
  static jfieldID select_animation_fieldID;
  /// Activate animation for the reaction.
  object_ptr<sticker> activate_animation_;
  static jfieldID activate_animation_fieldID;
  /// Effect animation for the reaction.
  object_ptr<sticker> effect_animation_;
  static jfieldID effect_animation_fieldID;
  /// Around animation for the reaction; may be null.
  object_ptr<sticker> around_animation_;
  static jfieldID around_animation_fieldID;
  /// Center animation for the reaction; may be null.
  object_ptr<sticker> center_animation_;
  static jfieldID center_animation_fieldID;

  /**
   * Contains information about an emoji reaction.
   */
  emojiReaction();

  /**
   * Contains information about an emoji reaction.
   *
   * \param[in] emoji_ Text representation of the reaction.
   * \param[in] title_ Reaction title.
   * \param[in] is_active_ True, if the reaction can be added to new messages and enabled in chats.
   * \param[in] static_icon_ Static icon for the reaction.
   * \param[in] appear_animation_ Appear animation for the reaction.
   * \param[in] select_animation_ Select animation for the reaction.
   * \param[in] activate_animation_ Activate animation for the reaction.
   * \param[in] effect_animation_ Effect animation for the reaction.
   * \param[in] around_animation_ Around animation for the reaction; may be null.
   * \param[in] center_animation_ Center animation for the reaction; may be null.
   */
  emojiReaction(string const &emoji_, string const &title_, bool is_active_, object_ptr<sticker> &&static_icon_, object_ptr<sticker> &&appear_animation_, object_ptr<sticker> &&select_animation_, object_ptr<sticker> &&activate_animation_, object_ptr<sticker> &&effect_animation_, object_ptr<sticker> &&around_animation_, object_ptr<sticker> &&center_animation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1616063583;

  static object_ptr<emojiReaction> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class EmojiStatusType;

/**
 * Describes an emoji to be shown instead of the Telegram Premium badge.
 */
class emojiStatus final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Type of the emoji status.
  object_ptr<EmojiStatusType> type_;
  static jfieldID type_fieldID;
  /// Point in time (Unix timestamp) when the status will expire; 0 if never.
  int32 expiration_date_;
  static jfieldID expiration_date_fieldID;

  /**
   * Describes an emoji to be shown instead of the Telegram Premium badge.
   */
  emojiStatus();

  /**
   * Describes an emoji to be shown instead of the Telegram Premium badge.
   *
   * \param[in] type_ Type of the emoji status.
   * \param[in] expiration_date_ Point in time (Unix timestamp) when the status will expire; 0 if never.
   */
  emojiStatus(object_ptr<EmojiStatusType> &&type_, int32 expiration_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 973424912;

  static object_ptr<emojiStatus> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains a list of custom emoji identifiers for emoji statuses.
 */
class emojiStatusCustomEmojis final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The list of custom emoji identifiers.
  array<int64> custom_emoji_ids_;
  static jfieldID custom_emoji_ids_fieldID;

  /**
   * Contains a list of custom emoji identifiers for emoji statuses.
   */
  emojiStatusCustomEmojis();

  /**
   * Contains a list of custom emoji identifiers for emoji statuses.
   *
   * \param[in] custom_emoji_ids_ The list of custom emoji identifiers.
   */
  explicit emojiStatusCustomEmojis(array<int64> &&custom_emoji_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 917123337;

  static object_ptr<emojiStatusCustomEmojis> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class upgradedGiftBackdropColors;

/**
 * This class is an abstract base class.
 * Describes type of emoji status.
 */
class EmojiStatusType: public Object {
 public:
  static jclass Class;

  static object_ptr<EmojiStatusType> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A custom emoji set as emoji status.
 */
class emojiStatusTypeCustomEmoji final : public EmojiStatusType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the custom emoji in stickerFormatTgs format.
  int64 custom_emoji_id_;
  static jfieldID custom_emoji_id_fieldID;

  /**
   * A custom emoji set as emoji status.
   */
  emojiStatusTypeCustomEmoji();

  /**
   * A custom emoji set as emoji status.
   *
   * \param[in] custom_emoji_id_ Identifier of the custom emoji in stickerFormatTgs format.
   */
  explicit emojiStatusTypeCustomEmoji(int64 custom_emoji_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1666780939;

  static object_ptr<EmojiStatusType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * An upgraded gift set as emoji status.
 */
class emojiStatusTypeUpgradedGift final : public EmojiStatusType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the upgraded gift.
  int64 upgraded_gift_id_;
  static jfieldID upgraded_gift_id_fieldID;
  /// The title of the upgraded gift.
  string gift_title_;
  static jfieldID gift_title_fieldID;
  /// Unique name of the upgraded gift that can be used with internalLinkTypeUpgradedGift.
  string gift_name_;
  static jfieldID gift_name_fieldID;
  /// Custom emoji identifier of the model of the upgraded gift.
  int64 model_custom_emoji_id_;
  static jfieldID model_custom_emoji_id_fieldID;
  /// Custom emoji identifier of the symbol of the upgraded gift.
  int64 symbol_custom_emoji_id_;
  static jfieldID symbol_custom_emoji_id_fieldID;
  /// Colors of the backdrop of the upgraded gift.
  object_ptr<upgradedGiftBackdropColors> backdrop_colors_;
  static jfieldID backdrop_colors_fieldID;

  /**
   * An upgraded gift set as emoji status.
   */
  emojiStatusTypeUpgradedGift();

  /**
   * An upgraded gift set as emoji status.
   *
   * \param[in] upgraded_gift_id_ Identifier of the upgraded gift.
   * \param[in] gift_title_ The title of the upgraded gift.
   * \param[in] gift_name_ Unique name of the upgraded gift that can be used with internalLinkTypeUpgradedGift.
   * \param[in] model_custom_emoji_id_ Custom emoji identifier of the model of the upgraded gift.
   * \param[in] symbol_custom_emoji_id_ Custom emoji identifier of the symbol of the upgraded gift.
   * \param[in] backdrop_colors_ Colors of the backdrop of the upgraded gift.
   */
  emojiStatusTypeUpgradedGift(int64 upgraded_gift_id_, string const &gift_title_, string const &gift_name_, int64 model_custom_emoji_id_, int64 symbol_custom_emoji_id_, object_ptr<upgradedGiftBackdropColors> &&backdrop_colors_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -837921804;

  static object_ptr<EmojiStatusType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class emojiStatus;

/**
 * Contains a list of emoji statuses.
 */
class emojiStatuses final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The list of emoji statuses identifiers.
  array<object_ptr<emojiStatus>> emoji_statuses_;
  static jfieldID emoji_statuses_fieldID;

  /**
   * Contains a list of emoji statuses.
   */
  emojiStatuses();

  /**
   * Contains a list of emoji statuses.
   *
   * \param[in] emoji_statuses_ The list of emoji statuses identifiers.
   */
  explicit emojiStatuses(array<object_ptr<emojiStatus>> &&emoji_statuses_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1186104146;

  static object_ptr<emojiStatuses> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents a list of emojis.
 */
class emojis final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// List of emojis.
  array<string> emojis_;
  static jfieldID emojis_fieldID;

  /**
   * Represents a list of emojis.
   */
  emojis();

  /**
   * Represents a list of emojis.
   *
   * \param[in] emojis_ List of emojis.
   */
  explicit emojis(array<string> &&emojis_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 950339552;

  static object_ptr<emojis> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains encrypted Telegram Passport data credentials.
 */
class encryptedCredentials final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The encrypted credentials.
  bytes data_;
  static jfieldID data_fieldID;
  /// The decrypted data hash.
  bytes hash_;
  static jfieldID hash_fieldID;
  /// Secret for data decryption, encrypted with the service's public key.
  bytes secret_;
  static jfieldID secret_fieldID;

  /**
   * Contains encrypted Telegram Passport data credentials.
   */
  encryptedCredentials();

  /**
   * Contains encrypted Telegram Passport data credentials.
   *
   * \param[in] data_ The encrypted credentials.
   * \param[in] hash_ The decrypted data hash.
   * \param[in] secret_ Secret for data decryption, encrypted with the service's public key.
   */
  encryptedCredentials(bytes const &data_, bytes const &hash_, bytes const &secret_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1331106766;

  static object_ptr<encryptedCredentials> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class PassportElementType;

class datedFile;

/**
 * Contains information about an encrypted Telegram Passport element; for bots only.
 */
class encryptedPassportElement final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Type of Telegram Passport element.
  object_ptr<PassportElementType> type_;
  static jfieldID type_fieldID;
  /// Encrypted JSON-encoded data about the user.
  bytes data_;
  static jfieldID data_fieldID;
  /// The front side of an identity document.
  object_ptr<datedFile> front_side_;
  static jfieldID front_side_fieldID;
  /// The reverse side of an identity document; may be null.
  object_ptr<datedFile> reverse_side_;
  static jfieldID reverse_side_fieldID;
  /// Selfie with the document; may be null.
  object_ptr<datedFile> selfie_;
  static jfieldID selfie_fieldID;
  /// List of files containing a certified English translation of the document.
  array<object_ptr<datedFile>> translation_;
  static jfieldID translation_fieldID;
  /// List of attached files.
  array<object_ptr<datedFile>> files_;
  static jfieldID files_fieldID;
  /// Unencrypted data, phone number or email address.
  string value_;
  static jfieldID value_fieldID;
  /// Hash of the entire element.
  string hash_;
  static jfieldID hash_fieldID;

  /**
   * Contains information about an encrypted Telegram Passport element; for bots only.
   */
  encryptedPassportElement();

  /**
   * Contains information about an encrypted Telegram Passport element; for bots only.
   *
   * \param[in] type_ Type of Telegram Passport element.
   * \param[in] data_ Encrypted JSON-encoded data about the user.
   * \param[in] front_side_ The front side of an identity document.
   * \param[in] reverse_side_ The reverse side of an identity document; may be null.
   * \param[in] selfie_ Selfie with the document; may be null.
   * \param[in] translation_ List of files containing a certified English translation of the document.
   * \param[in] files_ List of attached files.
   * \param[in] value_ Unencrypted data, phone number or email address.
   * \param[in] hash_ Hash of the entire element.
   */
  encryptedPassportElement(object_ptr<PassportElementType> &&type_, bytes const &data_, object_ptr<datedFile> &&front_side_, object_ptr<datedFile> &&reverse_side_, object_ptr<datedFile> &&selfie_, array<object_ptr<datedFile>> &&translation_, array<object_ptr<datedFile>> &&files_, string const &value_, string const &hash_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2002386193;

  static object_ptr<encryptedPassportElement> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * An object of this type can be returned on every function call, in case of an error.
 */
class error final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Error code; subject to future changes. If the error code is 406, the error message must not be processed in any way and must not be displayed to the user.
  int32 code_;
  static jfieldID code_fieldID;
  /// Error message; subject to future changes.
  string message_;
  static jfieldID message_fieldID;

  /**
   * An object of this type can be returned on every function call, in case of an error.
   */
  error();

  /**
   * An object of this type can be returned on every function call, in case of an error.
   *
   * \param[in] code_ Error code; subject to future changes. If the error code is 406, the error message must not be processed in any way and must not be displayed to the user.
   * \param[in] message_ Error message; subject to future changes.
   */
  error(int32 code_, string const &message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1679978726;

  static object_ptr<error> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class formattedText;

/**
 * Describes a fact-check added to the message by an independent checker.
 */
class factCheck final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Text of the fact-check.
  object_ptr<formattedText> text_;
  static jfieldID text_fieldID;
  /// A two-letter ISO 3166-1 alpha-2 country code of the country for which the fact-check is shown.
  string country_code_;
  static jfieldID country_code_fieldID;

  /**
   * Describes a fact-check added to the message by an independent checker.
   */
  factCheck();

  /**
   * Describes a fact-check added to the message by an independent checker.
   *
   * \param[in] text_ Text of the fact-check.
   * \param[in] country_code_ A two-letter ISO 3166-1 alpha-2 country code of the country for which the fact-check is shown.
   */
  factCheck(object_ptr<formattedText> &&text_, string const &country_code_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1048184552;

  static object_ptr<factCheck> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains information about a user that has failed to be added to a chat.
 */
class failedToAddMember final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// User identifier.
  int53 user_id_;
  static jfieldID user_id_fieldID;
  /// True, if subscription to Telegram Premium would have allowed to add the user to the chat.
  bool premium_would_allow_invite_;
  static jfieldID premium_would_allow_invite_fieldID;
  /// True, if subscription to Telegram Premium is required to send the user chat invite link.
  bool premium_required_to_send_messages_;
  static jfieldID premium_required_to_send_messages_fieldID;

  /**
   * Contains information about a user that has failed to be added to a chat.
   */
  failedToAddMember();

  /**
   * Contains information about a user that has failed to be added to a chat.
   *
   * \param[in] user_id_ User identifier.
   * \param[in] premium_would_allow_invite_ True, if subscription to Telegram Premium would have allowed to add the user to the chat.
   * \param[in] premium_required_to_send_messages_ True, if subscription to Telegram Premium is required to send the user chat invite link.
   */
  failedToAddMember(int53 user_id_, bool premium_would_allow_invite_, bool premium_required_to_send_messages_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -282891070;

  static object_ptr<failedToAddMember> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class failedToAddMember;

/**
 * Represents a list of users that has failed to be added to a chat.
 */
class failedToAddMembers final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Information about users that weren't added to the chat.
  array<object_ptr<failedToAddMember>> failed_to_add_members_;
  static jfieldID failed_to_add_members_fieldID;

  /**
   * Represents a list of users that has failed to be added to a chat.
   */
  failedToAddMembers();

  /**
   * Represents a list of users that has failed to be added to a chat.
   *
   * \param[in] failed_to_add_members_ Information about users that weren't added to the chat.
   */
  explicit failedToAddMembers(array<object_ptr<failedToAddMember>> &&failed_to_add_members_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -272587152;

  static object_ptr<failedToAddMembers> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class localFile;

class remoteFile;

/**
 * Represents a file.
 */
class file final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique file identifier.
  int32 id_;
  static jfieldID id_fieldID;
  /// File size, in bytes; 0 if unknown.
  int53 size_;
  static jfieldID size_fieldID;
  /// Approximate file size in bytes in case the exact file size is unknown. Can be used to show download/upload progress.
  int53 expected_size_;
  static jfieldID expected_size_fieldID;
  /// Information about the local copy of the file.
  object_ptr<localFile> local_;
  static jfieldID local_fieldID;
  /// Information about the remote copy of the file.
  object_ptr<remoteFile> remote_;
  static jfieldID remote_fieldID;

  /**
   * Represents a file.
   */
  file();

  /**
   * Represents a file.
   *
   * \param[in] id_ Unique file identifier.
   * \param[in] size_ File size, in bytes; 0 if unknown.
   * \param[in] expected_size_ Approximate file size in bytes in case the exact file size is unknown. Can be used to show download/upload progress.
   * \param[in] local_ Information about the local copy of the file.
   * \param[in] remote_ Information about the remote copy of the file.
   */
  file(int32 id_, int53 size_, int53 expected_size_, object_ptr<localFile> &&local_, object_ptr<remoteFile> &&remote_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1263291956;

  static object_ptr<file> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class message;

/**
 * Describes a file added to file download list.
 */
class fileDownload final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// File identifier.
  int32 file_id_;
  static jfieldID file_id_fieldID;
  /// The message with the file.
  object_ptr<message> message_;
  static jfieldID message_fieldID;
  /// Point in time (Unix timestamp) when the file was added to the download list.
  int32 add_date_;
  static jfieldID add_date_fieldID;
  /// Point in time (Unix timestamp) when the file downloading was completed; 0 if the file downloading isn't completed.
  int32 complete_date_;
  static jfieldID complete_date_fieldID;
  /// True, if downloading of the file is paused.
  bool is_paused_;
  static jfieldID is_paused_fieldID;

  /**
   * Describes a file added to file download list.
   */
  fileDownload();

  /**
   * Describes a file added to file download list.
   *
   * \param[in] file_id_ File identifier.
   * \param[in] message_ The message with the file.
   * \param[in] add_date_ Point in time (Unix timestamp) when the file was added to the download list.
   * \param[in] complete_date_ Point in time (Unix timestamp) when the file downloading was completed; 0 if the file downloading isn't completed.
   * \param[in] is_paused_ True, if downloading of the file is paused.
   */
  fileDownload(int32 file_id_, object_ptr<message> &&message_, int32 add_date_, int32 complete_date_, bool is_paused_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2092100780;

  static object_ptr<fileDownload> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains size of downloaded prefix of a file.
 */
class fileDownloadedPrefixSize final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The prefix size, in bytes.
  int53 size_;
  static jfieldID size_fieldID;

  /**
   * Contains size of downloaded prefix of a file.
   */
  fileDownloadedPrefixSize();

  /**
   * Contains size of downloaded prefix of a file.
   *
   * \param[in] size_ The prefix size, in bytes.
   */
  explicit fileDownloadedPrefixSize(int53 size_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2015205381;

  static object_ptr<fileDownloadedPrefixSize> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Represents the type of file.
 */
class FileType: public Object {
 public:
  static jclass Class;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The data is not a file.
 */
class fileTypeNone final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The data is not a file.
   */
  fileTypeNone();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2003009189;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is an animation.
 */
class fileTypeAnimation final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is an animation.
   */
  fileTypeAnimation();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -290816582;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is an audio file.
 */
class fileTypeAudio final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is an audio file.
   */
  fileTypeAudio();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -709112160;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is a document.
 */
class fileTypeDocument final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is a document.
   */
  fileTypeDocument();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -564722929;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is a notification sound.
 */
class fileTypeNotificationSound final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is a notification sound.
   */
  fileTypeNotificationSound();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1020289271;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is a photo.
 */
class fileTypePhoto final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is a photo.
   */
  fileTypePhoto();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1718914651;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is a photo published as a story.
 */
class fileTypePhotoStory final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is a photo published as a story.
   */
  fileTypePhotoStory();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2018995956;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is a profile photo.
 */
class fileTypeProfilePhoto final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is a profile photo.
   */
  fileTypeProfilePhoto();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1795089315;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file was sent to a secret chat (the file type is not known to the server).
 */
class fileTypeSecret final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file was sent to a secret chat (the file type is not known to the server).
   */
  fileTypeSecret();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1871899401;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is a thumbnail of a file from a secret chat.
 */
class fileTypeSecretThumbnail final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is a thumbnail of a file from a secret chat.
   */
  fileTypeSecretThumbnail();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1401326026;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is a file from Secure storage used for storing Telegram Passport files.
 */
class fileTypeSecure final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is a file from Secure storage used for storing Telegram Passport files.
   */
  fileTypeSecure();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1419133146;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is a self-destructing photo in a private chat.
 */
class fileTypeSelfDestructingPhoto final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is a self-destructing photo in a private chat.
   */
  fileTypeSelfDestructingPhoto();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2077176475;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is a self-destructing video in a private chat.
 */
class fileTypeSelfDestructingVideo final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is a self-destructing video in a private chat.
   */
  fileTypeSelfDestructingVideo();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1223900123;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is a self-destructing video note in a private chat.
 */
class fileTypeSelfDestructingVideoNote final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is a self-destructing video note in a private chat.
   */
  fileTypeSelfDestructingVideoNote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1495274177;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is a self-destructing voice note in a private chat.
 */
class fileTypeSelfDestructingVoiceNote final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is a self-destructing voice note in a private chat.
   */
  fileTypeSelfDestructingVoiceNote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1691409181;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is a sticker.
 */
class fileTypeSticker final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is a sticker.
   */
  fileTypeSticker();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 475233385;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is a thumbnail of another file.
 */
class fileTypeThumbnail final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is a thumbnail of another file.
   */
  fileTypeThumbnail();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -12443298;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file type is not yet known.
 */
class fileTypeUnknown final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file type is not yet known.
   */
  fileTypeUnknown();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2011566768;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is a video.
 */
class fileTypeVideo final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is a video.
   */
  fileTypeVideo();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1430816539;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is a video note.
 */
class fileTypeVideoNote final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is a video note.
   */
  fileTypeVideoNote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -518412385;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is a video published as a story.
 */
class fileTypeVideoStory final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is a video published as a story.
   */
  fileTypeVideoStory();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2146754143;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is a voice note.
 */
class fileTypeVoiceNote final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is a voice note.
   */
  fileTypeVoiceNote();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -588681661;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The file is a wallpaper or a background pattern.
 */
class fileTypeWallpaper final : public FileType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The file is a wallpaper or a background pattern.
   */
  fileTypeWallpaper();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1854930076;

  static object_ptr<FileType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Contains settings for Firebase Authentication in the official applications.
 */
class FirebaseAuthenticationSettings: public Object {
 public:
  static jclass Class;

  static object_ptr<FirebaseAuthenticationSettings> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Settings for Firebase Authentication in the official Android application.
 */
class firebaseAuthenticationSettingsAndroid final : public FirebaseAuthenticationSettings {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * Settings for Firebase Authentication in the official Android application.
   */
  firebaseAuthenticationSettingsAndroid();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1771112932;

  static object_ptr<FirebaseAuthenticationSettings> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Settings for Firebase Authentication in the official iOS application.
 */
class firebaseAuthenticationSettingsIos final : public FirebaseAuthenticationSettings {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Device token from Apple Push Notification service.
  string device_token_;
  static jfieldID device_token_fieldID;
  /// True, if App Sandbox is enabled.
  bool is_app_sandbox_;
  static jfieldID is_app_sandbox_fieldID;

  /**
   * Settings for Firebase Authentication in the official iOS application.
   */
  firebaseAuthenticationSettingsIos();

  /**
   * Settings for Firebase Authentication in the official iOS application.
   *
   * \param[in] device_token_ Device token from Apple Push Notification service.
   * \param[in] is_app_sandbox_ True, if App Sandbox is enabled.
   */
  firebaseAuthenticationSettingsIos(string const &device_token_, bool is_app_sandbox_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 222930116;

  static object_ptr<FirebaseAuthenticationSettings> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes parameters to be used for device verification.
 */
class FirebaseDeviceVerificationParameters: public Object {
 public:
  static jclass Class;

  static object_ptr<FirebaseDeviceVerificationParameters> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Device verification must be performed with the SafetyNet Attestation API.
 */
class firebaseDeviceVerificationParametersSafetyNet final : public FirebaseDeviceVerificationParameters {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Nonce to pass to the SafetyNet Attestation API.
  bytes nonce_;
  static jfieldID nonce_fieldID;

  /**
   * Device verification must be performed with the SafetyNet Attestation API.
   */
  firebaseDeviceVerificationParametersSafetyNet();

  /**
   * Device verification must be performed with the SafetyNet Attestation API.
   *
   * \param[in] nonce_ Nonce to pass to the SafetyNet Attestation API.
   */
  explicit firebaseDeviceVerificationParametersSafetyNet(bytes const &nonce_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 731296497;

  static object_ptr<FirebaseDeviceVerificationParameters> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Device verification must be performed with the classic Play Integrity verification (https://developer.android.com/google/play/integrity/classic).
 */
class firebaseDeviceVerificationParametersPlayIntegrity final : public FirebaseDeviceVerificationParameters {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Base64url-encoded nonce to pass to the Play Integrity API.
  string nonce_;
  static jfieldID nonce_fieldID;
  /// Cloud project number to pass to the Play Integrity API.
  int64 cloud_project_number_;
  static jfieldID cloud_project_number_fieldID;

  /**
   * Device verification must be performed with the classic Play Integrity verification (https://developer.android.com/google/play/integrity/classic).
   */
  firebaseDeviceVerificationParametersPlayIntegrity();

  /**
   * Device verification must be performed with the classic Play Integrity verification (https://developer.android.com/google/play/integrity/classic).
   *
   * \param[in] nonce_ Base64url-encoded nonce to pass to the Play Integrity API.
   * \param[in] cloud_project_number_ Cloud project number to pass to the Play Integrity API.
   */
  firebaseDeviceVerificationParametersPlayIntegrity(string const &nonce_, int64 cloud_project_number_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -889936502;

  static object_ptr<FirebaseDeviceVerificationParameters> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class textEntity;

/**
 * A text with some entities.
 */
class formattedText final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The text.
  string text_;
  static jfieldID text_fieldID;
  /// Entities contained in the text. Entities can be nested, but must not mutually intersect with each other. Pre, Code and PreCode entities can't contain other entities. BlockQuote entities can't contain other BlockQuote entities. Bold, Italic, Underline, Strikethrough, and Spoiler entities can contain and can be part of any other entities. All other entities can't contain each other.
  array<object_ptr<textEntity>> entities_;
  static jfieldID entities_fieldID;

  /**
   * A text with some entities.
   */
  formattedText();

  /**
   * A text with some entities.
   *
   * \param[in] text_ The text.
   * \param[in] entities_ Entities contained in the text. Entities can be nested, but must not mutually intersect with each other. Pre, Code and PreCode entities can't contain other entities. BlockQuote entities can't contain other BlockQuote entities. Bold, Italic, Underline, Strikethrough, and Spoiler entities can contain and can be part of any other entities. All other entities can't contain each other.
   */
  formattedText(string const &text_, array<object_ptr<textEntity>> &&entities_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -252624564;

  static object_ptr<formattedText> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class chatNotificationSettings;

class draftMessage;

class forumTopicInfo;

class message;

/**
 * Describes a forum topic.
 */
class forumTopic final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Basic information about the topic.
  object_ptr<forumTopicInfo> info_;
  static jfieldID info_fieldID;
  /// Last message in the topic; may be null if unknown.
  object_ptr<message> last_message_;
  static jfieldID last_message_fieldID;
  /// A parameter used to determine order of the topic in the topic list. Topics must be sorted by the order in descending order.
  int64 order_;
  static jfieldID order_fieldID;
  /// True, if the topic is pinned in the topic list.
  bool is_pinned_;
  static jfieldID is_pinned_fieldID;
  /// Number of unread messages in the topic.
  int32 unread_count_;
  static jfieldID unread_count_fieldID;
  /// Identifier of the last read incoming message.
  int53 last_read_inbox_message_id_;
  static jfieldID last_read_inbox_message_id_fieldID;
  /// Identifier of the last read outgoing message.
  int53 last_read_outbox_message_id_;
  static jfieldID last_read_outbox_message_id_fieldID;
  /// Number of unread messages with a mention/reply in the topic.
  int32 unread_mention_count_;
  static jfieldID unread_mention_count_fieldID;
  /// Number of messages with unread reactions in the topic.
  int32 unread_reaction_count_;
  static jfieldID unread_reaction_count_fieldID;
  /// Notification settings for the topic.
  object_ptr<chatNotificationSettings> notification_settings_;
  static jfieldID notification_settings_fieldID;
  /// A draft of a message in the topic; may be null if none.
  object_ptr<draftMessage> draft_message_;
  static jfieldID draft_message_fieldID;

  /**
   * Describes a forum topic.
   */
  forumTopic();

  /**
   * Describes a forum topic.
   *
   * \param[in] info_ Basic information about the topic.
   * \param[in] last_message_ Last message in the topic; may be null if unknown.
   * \param[in] order_ A parameter used to determine order of the topic in the topic list. Topics must be sorted by the order in descending order.
   * \param[in] is_pinned_ True, if the topic is pinned in the topic list.
   * \param[in] unread_count_ Number of unread messages in the topic.
   * \param[in] last_read_inbox_message_id_ Identifier of the last read incoming message.
   * \param[in] last_read_outbox_message_id_ Identifier of the last read outgoing message.
   * \param[in] unread_mention_count_ Number of unread messages with a mention/reply in the topic.
   * \param[in] unread_reaction_count_ Number of messages with unread reactions in the topic.
   * \param[in] notification_settings_ Notification settings for the topic.
   * \param[in] draft_message_ A draft of a message in the topic; may be null if none.
   */
  forumTopic(object_ptr<forumTopicInfo> &&info_, object_ptr<message> &&last_message_, int64 order_, bool is_pinned_, int32 unread_count_, int53 last_read_inbox_message_id_, int53 last_read_outbox_message_id_, int32 unread_mention_count_, int32 unread_reaction_count_, object_ptr<chatNotificationSettings> &&notification_settings_, object_ptr<draftMessage> &&draft_message_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2094608976;

  static object_ptr<forumTopic> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes a forum topic icon.
 */
class forumTopicIcon final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Color of the topic icon in RGB format.
  int32 color_;
  static jfieldID color_fieldID;
  /// Unique identifier of the custom emoji shown on the topic icon; 0 if none.
  int64 custom_emoji_id_;
  static jfieldID custom_emoji_id_fieldID;

  /**
   * Describes a forum topic icon.
   */
  forumTopicIcon();

  /**
   * Describes a forum topic icon.
   *
   * \param[in] color_ Color of the topic icon in RGB format.
   * \param[in] custom_emoji_id_ Unique identifier of the custom emoji shown on the topic icon; 0 if none.
   */
  forumTopicIcon(int32 color_, int64 custom_emoji_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -818765421;

  static object_ptr<forumTopicIcon> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class MessageSender;

class forumTopicIcon;

/**
 * Contains basic information about a forum topic.
 */
class forumTopicInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the forum chat to which the topic belongs.
  int53 chat_id_;
  static jfieldID chat_id_fieldID;
  /// Forum topic identifier of the topic.
  int53 forum_topic_id_;
  static jfieldID forum_topic_id_fieldID;
  /// Message thread identifier of the topic.
  int53 message_thread_id_;
  static jfieldID message_thread_id_fieldID;
  /// Name of the topic.
  string name_;
  static jfieldID name_fieldID;
  /// Icon of the topic.
  object_ptr<forumTopicIcon> icon_;
  static jfieldID icon_fieldID;
  /// Point in time (Unix timestamp) when the topic was created.
  int32 creation_date_;
  static jfieldID creation_date_fieldID;
  /// Identifier of the creator of the topic.
  object_ptr<MessageSender> creator_id_;
  static jfieldID creator_id_fieldID;
  /// True, if the topic is the General topic list.
  bool is_general_;
  static jfieldID is_general_fieldID;
  /// True, if the topic was created by the current user.
  bool is_outgoing_;
  static jfieldID is_outgoing_fieldID;
  /// True, if the topic is closed. If the topic is closed, then the user must have can_manage_topics administrator right in the supergroup or must be the creator of the topic to send messages there.
  bool is_closed_;
  static jfieldID is_closed_fieldID;
  /// True, if the topic is hidden above the topic list and closed; for General topic only.
  bool is_hidden_;
  static jfieldID is_hidden_fieldID;

  /**
   * Contains basic information about a forum topic.
   */
  forumTopicInfo();

  /**
   * Contains basic information about a forum topic.
   *
   * \param[in] chat_id_ Identifier of the forum chat to which the topic belongs.
   * \param[in] forum_topic_id_ Forum topic identifier of the topic.
   * \param[in] message_thread_id_ Message thread identifier of the topic.
   * \param[in] name_ Name of the topic.
   * \param[in] icon_ Icon of the topic.
   * \param[in] creation_date_ Point in time (Unix timestamp) when the topic was created.
   * \param[in] creator_id_ Identifier of the creator of the topic.
   * \param[in] is_general_ True, if the topic is the General topic list.
   * \param[in] is_outgoing_ True, if the topic was created by the current user.
   * \param[in] is_closed_ True, if the topic is closed. If the topic is closed, then the user must have can_manage_topics administrator right in the supergroup or must be the creator of the topic to send messages there.
   * \param[in] is_hidden_ True, if the topic is hidden above the topic list and closed; for General topic only.
   */
  forumTopicInfo(int53 chat_id_, int53 forum_topic_id_, int53 message_thread_id_, string const &name_, object_ptr<forumTopicIcon> &&icon_, int32 creation_date_, object_ptr<MessageSender> &&creator_id_, bool is_general_, bool is_outgoing_, bool is_closed_, bool is_hidden_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1573650873;

  static object_ptr<forumTopicInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class forumTopic;

/**
 * Describes a list of forum topics.
 */
class forumTopics final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Approximate total number of forum topics found.
  int32 total_count_;
  static jfieldID total_count_fieldID;
  /// List of forum topics.
  array<object_ptr<forumTopic>> topics_;
  static jfieldID topics_fieldID;
  /// Offset date for the next getForumTopics request.
  int32 next_offset_date_;
  static jfieldID next_offset_date_fieldID;
  /// Offset message identifier for the next getForumTopics request.
  int53 next_offset_message_id_;
  static jfieldID next_offset_message_id_fieldID;
  /// Offset message thread identifier for the next getForumTopics request.
  int53 next_offset_message_thread_id_;
  static jfieldID next_offset_message_thread_id_fieldID;

  /**
   * Describes a list of forum topics.
   */
  forumTopics();

  /**
   * Describes a list of forum topics.
   *
   * \param[in] total_count_ Approximate total number of forum topics found.
   * \param[in] topics_ List of forum topics.
   * \param[in] next_offset_date_ Offset date for the next getForumTopics request.
   * \param[in] next_offset_message_id_ Offset message identifier for the next getForumTopics request.
   * \param[in] next_offset_message_thread_id_ Offset message thread identifier for the next getForumTopics request.
   */
  forumTopics(int32 total_count_, array<object_ptr<forumTopic>> &&topics_, int32 next_offset_date_, int53 next_offset_message_id_, int53 next_offset_message_thread_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 732819537;

  static object_ptr<forumTopics> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class MessageSender;

/**
 * Contains information about the last message from which a new message was forwarded last time.
 */
class forwardSource final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the chat to which the message that was forwarded belonged; may be 0 if unknown.
  int53 chat_id_;
  static jfieldID chat_id_fieldID;
  /// Identifier of the message; may be 0 if unknown.
  int53 message_id_;
  static jfieldID message_id_fieldID;
  /// Identifier of the sender of the message; may be null if unknown or the new message was forwarded not to Saved Messages.
  object_ptr<MessageSender> sender_id_;
  static jfieldID sender_id_fieldID;
  /// Name of the sender of the message if the sender is hidden by their privacy settings.
  string sender_name_;
  static jfieldID sender_name_fieldID;
  /// Point in time (Unix timestamp) when the message is sent; 0 if unknown.
  int32 date_;
  static jfieldID date_fieldID;
  /// True, if the message that was forwarded is outgoing; always false if sender is unknown.
  bool is_outgoing_;
  static jfieldID is_outgoing_fieldID;

  /**
   * Contains information about the last message from which a new message was forwarded last time.
   */
  forwardSource();

  /**
   * Contains information about the last message from which a new message was forwarded last time.
   *
   * \param[in] chat_id_ Identifier of the chat to which the message that was forwarded belonged; may be 0 if unknown.
   * \param[in] message_id_ Identifier of the message; may be 0 if unknown.
   * \param[in] sender_id_ Identifier of the sender of the message; may be null if unknown or the new message was forwarded not to Saved Messages.
   * \param[in] sender_name_ Name of the sender of the message if the sender is hidden by their privacy settings.
   * \param[in] date_ Point in time (Unix timestamp) when the message is sent; 0 if unknown.
   * \param[in] is_outgoing_ True, if the message that was forwarded is outgoing; always false if sender is unknown.
   */
  forwardSource(int53 chat_id_, int53 message_id_, object_ptr<MessageSender> &&sender_id_, string const &sender_name_, int32 date_, bool is_outgoing_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1795337929;

  static object_ptr<forwardSource> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class affiliateProgramInfo;

/**
 * Describes a found affiliate program.
 */
class foundAffiliateProgram final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// User identifier of the bot created the program.
  int53 bot_user_id_;
  static jfieldID bot_user_id_fieldID;
  /// Information about the affiliate program.
  object_ptr<affiliateProgramInfo> info_;
  static jfieldID info_fieldID;

  /**
   * Describes a found affiliate program.
   */
  foundAffiliateProgram();

  /**
   * Describes a found affiliate program.
   *
   * \param[in] bot_user_id_ User identifier of the bot created the program.
   * \param[in] info_ Information about the affiliate program.
   */
  foundAffiliateProgram(int53 bot_user_id_, object_ptr<affiliateProgramInfo> &&info_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -966565242;

  static object_ptr<foundAffiliateProgram> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class foundAffiliateProgram;

/**
 * Represents a list of found affiliate programs.
 */
class foundAffiliatePrograms final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The total number of found affiliate programs.
  int32 total_count_;
  static jfieldID total_count_fieldID;
  /// The list of affiliate programs.
  array<object_ptr<foundAffiliateProgram>> programs_;
  static jfieldID programs_fieldID;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;
  static jfieldID next_offset_fieldID;

  /**
   * Represents a list of found affiliate programs.
   */
  foundAffiliatePrograms();

  /**
   * Represents a list of found affiliate programs.
   *
   * \param[in] total_count_ The total number of found affiliate programs.
   * \param[in] programs_ The list of affiliate programs.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  foundAffiliatePrograms(int32 total_count_, array<object_ptr<foundAffiliateProgram>> &&programs_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 186317057;

  static object_ptr<foundAffiliatePrograms> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class chatBoost;

/**
 * Contains a list of boosts applied to a chat.
 */
class foundChatBoosts final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Total number of boosts applied to the chat.
  int32 total_count_;
  static jfieldID total_count_fieldID;
  /// List of boosts.
  array<object_ptr<chatBoost>> boosts_;
  static jfieldID boosts_fieldID;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;
  static jfieldID next_offset_fieldID;

  /**
   * Contains a list of boosts applied to a chat.
   */
  foundChatBoosts();

  /**
   * Contains a list of boosts applied to a chat.
   *
   * \param[in] total_count_ Total number of boosts applied to the chat.
   * \param[in] boosts_ List of boosts.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  foundChatBoosts(int32 total_count_, array<object_ptr<chatBoost>> &&boosts_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 51457680;

  static object_ptr<foundChatBoosts> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class message;

/**
 * Contains a list of messages found by a search in a given chat.
 */
class foundChatMessages final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Approximate total number of messages found; -1 if unknown.
  int32 total_count_;
  static jfieldID total_count_fieldID;
  /// List of messages.
  array<object_ptr<message>> messages_;
  static jfieldID messages_fieldID;
  /// The offset for the next request. If 0, there are no more results.
  int53 next_from_message_id_;
  static jfieldID next_from_message_id_fieldID;

  /**
   * Contains a list of messages found by a search in a given chat.
   */
  foundChatMessages();

  /**
   * Contains a list of messages found by a search in a given chat.
   *
   * \param[in] total_count_ Approximate total number of messages found; -1 if unknown.
   * \param[in] messages_ List of messages.
   * \param[in] next_from_message_id_ The offset for the next request. If 0, there are no more results.
   */
  foundChatMessages(int32 total_count_, array<object_ptr<message>> &&messages_, int53 next_from_message_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 427484196;

  static object_ptr<foundChatMessages> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class downloadedFileCounts;

class fileDownload;

/**
 * Contains a list of downloaded files, found by a search.
 */
class foundFileDownloads final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Total number of suitable files, ignoring offset.
  object_ptr<downloadedFileCounts> total_counts_;
  static jfieldID total_counts_fieldID;
  /// The list of files.
  array<object_ptr<fileDownload>> files_;
  static jfieldID files_fieldID;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;
  static jfieldID next_offset_fieldID;

  /**
   * Contains a list of downloaded files, found by a search.
   */
  foundFileDownloads();

  /**
   * Contains a list of downloaded files, found by a search.
   *
   * \param[in] total_counts_ Total number of suitable files, ignoring offset.
   * \param[in] files_ The list of files.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  foundFileDownloads(object_ptr<downloadedFileCounts> &&total_counts_, array<object_ptr<fileDownload>> &&files_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1395890392;

  static object_ptr<foundFileDownloads> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class message;

/**
 * Contains a list of messages found by a search.
 */
class foundMessages final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Approximate total number of messages found; -1 if unknown.
  int32 total_count_;
  static jfieldID total_count_fieldID;
  /// List of messages.
  array<object_ptr<message>> messages_;
  static jfieldID messages_fieldID;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;
  static jfieldID next_offset_fieldID;

  /**
   * Contains a list of messages found by a search.
   */
  foundMessages();

  /**
   * Contains a list of messages found by a search.
   *
   * \param[in] total_count_ Approximate total number of messages found; -1 if unknown.
   * \param[in] messages_ List of messages.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  foundMessages(int32 total_count_, array<object_ptr<message>> &&messages_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -529809608;

  static object_ptr<foundMessages> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains 0-based match position.
 */
class foundPosition final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The position of the match.
  int32 position_;
  static jfieldID position_fieldID;

  /**
   * Contains 0-based match position.
   */
  foundPosition();

  /**
   * Contains 0-based match position.
   *
   * \param[in] position_ The position of the match.
   */
  explicit foundPosition(int32 position_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1886724216;

  static object_ptr<foundPosition> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains 0-based positions of matched objects.
 */
class foundPositions final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Total number of matched objects.
  int32 total_count_;
  static jfieldID total_count_fieldID;
  /// The positions of the matched objects.
  array<int32> positions_;
  static jfieldID positions_fieldID;

  /**
   * Contains 0-based positions of matched objects.
   */
  foundPositions();

  /**
   * Contains 0-based positions of matched objects.
   *
   * \param[in] total_count_ Total number of matched objects.
   * \param[in] positions_ The positions of the matched objects.
   */
  foundPositions(int32 total_count_, array<int32> &&positions_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -80518368;

  static object_ptr<foundPositions> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class story;

/**
 * Contains a list of stories found by a search.
 */
class foundStories final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Approximate total number of stories found.
  int32 total_count_;
  static jfieldID total_count_fieldID;
  /// List of stories.
  array<object_ptr<story>> stories_;
  static jfieldID stories_fieldID;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;
  static jfieldID next_offset_fieldID;

  /**
   * Contains a list of stories found by a search.
   */
  foundStories();

  /**
   * Contains a list of stories found by a search.
   *
   * \param[in] total_count_ Approximate total number of stories found.
   * \param[in] stories_ List of stories.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  foundStories(int32 total_count_, array<object_ptr<story>> &&stories_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1678513512;

  static object_ptr<foundStories> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents a list of found users.
 */
class foundUsers final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifiers of the found users.
  array<int53> user_ids_;
  static jfieldID user_ids_fieldID;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;
  static jfieldID next_offset_fieldID;

  /**
   * Represents a list of found users.
   */
  foundUsers();

  /**
   * Represents a list of found users.
   *
   * \param[in] user_ids_ Identifiers of the found users.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  foundUsers(array<int53> &&user_ids_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1150570075;

  static object_ptr<foundUsers> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class webApp;

/**
 * Contains information about a Web App found by its short name.
 */
class foundWebApp final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The Web App.
  object_ptr<webApp> web_app_;
  static jfieldID web_app_fieldID;
  /// True, if the user must be asked for the permission to the bot to send them messages.
  bool request_write_access_;
  static jfieldID request_write_access_fieldID;
  /// True, if there is no need to show an ordinary open URL confirmation before opening the Web App. The field must be ignored and confirmation must be shown anyway if the Web App link was hidden.
  bool skip_confirmation_;
  static jfieldID skip_confirmation_fieldID;

  /**
   * Contains information about a Web App found by its short name.
   */
  foundWebApp();

  /**
   * Contains information about a Web App found by its short name.
   *
   * \param[in] web_app_ The Web App.
   * \param[in] request_write_access_ True, if the user must be asked for the permission to the bot to send them messages.
   * \param[in] skip_confirmation_ True, if there is no need to show an ordinary open URL confirmation before opening the Web App. The field must be ignored and confirmation must be shown anyway if the Web App link was hidden.
   */
  foundWebApp(object_ptr<webApp> &&web_app_, bool request_write_access_, bool skip_confirmation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -290926562;

  static object_ptr<foundWebApp> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class animation;

class formattedText;

class photo;

/**
 * Describes a game. Use getInternalLink with internalLinkTypeGame to share the game.
 */
class game final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique game identifier.
  int64 id_;
  static jfieldID id_fieldID;
  /// Game short name.
  string short_name_;
  static jfieldID short_name_fieldID;
  /// Game title.
  string title_;
  static jfieldID title_fieldID;
  /// Game text, usually containing scoreboards for a game.
  object_ptr<formattedText> text_;
  static jfieldID text_fieldID;
  /// Game description.
  string description_;
  static jfieldID description_fieldID;
  /// Game photo.
  object_ptr<photo> photo_;
  static jfieldID photo_fieldID;
  /// Game animation; may be null.
  object_ptr<animation> animation_;
  static jfieldID animation_fieldID;

  /**
   * Describes a game. Use getInternalLink with internalLinkTypeGame to share the game.
   */
  game();

  /**
   * Describes a game. Use getInternalLink with internalLinkTypeGame to share the game.
   *
   * \param[in] id_ Unique game identifier.
   * \param[in] short_name_ Game short name.
   * \param[in] title_ Game title.
   * \param[in] text_ Game text, usually containing scoreboards for a game.
   * \param[in] description_ Game description.
   * \param[in] photo_ Game photo.
   * \param[in] animation_ Game animation; may be null.
   */
  game(int64 id_, string const &short_name_, string const &title_, object_ptr<formattedText> &&text_, string const &description_, object_ptr<photo> &&photo_, object_ptr<animation> &&animation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1565597752;

  static object_ptr<game> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains one row of the game high score table.
 */
class gameHighScore final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Position in the high score table.
  int32 position_;
  static jfieldID position_fieldID;
  /// User identifier.
  int53 user_id_;
  static jfieldID user_id_fieldID;
  /// User score.
  int32 score_;
  static jfieldID score_fieldID;

  /**
   * Contains one row of the game high score table.
   */
  gameHighScore();

  /**
   * Contains one row of the game high score table.
   *
   * \param[in] position_ Position in the high score table.
   * \param[in] user_id_ User identifier.
   * \param[in] score_ User score.
   */
  gameHighScore(int32 position_, int53 user_id_, int32 score_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 342871838;

  static object_ptr<gameHighScore> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class gameHighScore;

/**
 * Contains a list of game high scores.
 */
class gameHighScores final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A list of game high scores.
  array<object_ptr<gameHighScore>> scores_;
  static jfieldID scores_fieldID;

  /**
   * Contains a list of game high scores.
   */
  gameHighScores();

  /**
   * Contains a list of game high scores.
   *
   * \param[in] scores_ A list of game high scores.
   */
  explicit gameHighScores(array<object_ptr<gameHighScore>> &&scores_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -725770727;

  static object_ptr<gameHighScores> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class sticker;

/**
 * Describes a gift that can be sent to another user or channel chat.
 */
class gift final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the gift.
  int64 id_;
  static jfieldID id_fieldID;
  /// Identifier of the chat that published the gift; 0 if none.
  int53 publisher_chat_id_;
  static jfieldID publisher_chat_id_fieldID;
  /// The sticker representing the gift.
  object_ptr<sticker> sticker_;
  static jfieldID sticker_fieldID;
  /// Number of Telegram Stars that must be paid for the gift.
  int53 star_count_;
  static jfieldID star_count_fieldID;
  /// Number of Telegram Stars that can be claimed by the receiver instead of the regular gift by default. If the gift was paid with just bought Telegram Stars, then full value can be claimed.
  int53 default_sell_star_count_;
  static jfieldID default_sell_star_count_fieldID;
  /// Number of Telegram Stars that must be paid to upgrade the gift; 0 if upgrade isn't possible.
  int53 upgrade_star_count_;
  static jfieldID upgrade_star_count_fieldID;
  /// True, if the gift is a birthday gift.
  bool is_for_birthday_;
  static jfieldID is_for_birthday_fieldID;
  /// Number of remaining times the gift can be purchased; 0 if not limited or the gift was sold out.
  int32 remaining_count_;
  static jfieldID remaining_count_fieldID;
  /// Number of total times the gift can be purchased; 0 if not limited.
  int32 total_count_;
  static jfieldID total_count_fieldID;
  /// Point in time (Unix timestamp) when the gift was send for the first time; for sold out gifts only.
  int32 first_send_date_;
  static jfieldID first_send_date_fieldID;
  /// Point in time (Unix timestamp) when the gift was send for the last time; for sold out gifts only.
  int32 last_send_date_;
  static jfieldID last_send_date_fieldID;

  /**
   * Describes a gift that can be sent to another user or channel chat.
   */
  gift();

  /**
   * Describes a gift that can be sent to another user or channel chat.
   *
   * \param[in] id_ Unique identifier of the gift.
   * \param[in] publisher_chat_id_ Identifier of the chat that published the gift; 0 if none.
   * \param[in] sticker_ The sticker representing the gift.
   * \param[in] star_count_ Number of Telegram Stars that must be paid for the gift.
   * \param[in] default_sell_star_count_ Number of Telegram Stars that can be claimed by the receiver instead of the regular gift by default. If the gift was paid with just bought Telegram Stars, then full value can be claimed.
   * \param[in] upgrade_star_count_ Number of Telegram Stars that must be paid to upgrade the gift; 0 if upgrade isn't possible.
   * \param[in] is_for_birthday_ True, if the gift is a birthday gift.
   * \param[in] remaining_count_ Number of remaining times the gift can be purchased; 0 if not limited or the gift was sold out.
   * \param[in] total_count_ Number of total times the gift can be purchased; 0 if not limited.
   * \param[in] first_send_date_ Point in time (Unix timestamp) when the gift was send for the first time; for sold out gifts only.
   * \param[in] last_send_date_ Point in time (Unix timestamp) when the gift was send for the last time; for sold out gifts only.
   */
  gift(int64 id_, int53 publisher_chat_id_, object_ptr<sticker> &&sticker_, int53 star_count_, int53 default_sell_star_count_, int53 upgrade_star_count_, bool is_for_birthday_, int32 remaining_count_, int32 total_count_, int32 first_send_date_, int32 last_send_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 91025890;

  static object_ptr<gift> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class upgradedGift;

/**
 * Describes a gift available for resale.
 */
class giftForResale final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The gift.
  object_ptr<upgradedGift> gift_;
  static jfieldID gift_fieldID;
  /// Unique identifier of the received gift for the current user; only for the gifts owned by the current user.
  string received_gift_id_;
  static jfieldID received_gift_id_fieldID;

  /**
   * Describes a gift available for resale.
   */
  giftForResale();

  /**
   * Describes a gift available for resale.
   *
   * \param[in] gift_ The gift.
   * \param[in] received_gift_id_ Unique identifier of the received gift for the current user; only for the gifts owned by the current user.
   */
  giftForResale(object_ptr<upgradedGift> &&gift_, string const &received_gift_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1130990515;

  static object_ptr<giftForResale> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes order in which upgraded gifts for resale will be sorted.
 */
class GiftForResaleOrder: public Object {
 public:
  static jclass Class;

  static object_ptr<GiftForResaleOrder> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The gifts will be sorted by their price from the lowest to the highest.
 */
class giftForResaleOrderPrice final : public GiftForResaleOrder {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The gifts will be sorted by their price from the lowest to the highest.
   */
  giftForResaleOrderPrice();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1371740258;

  static object_ptr<GiftForResaleOrder> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The gifts will be sorted by the last date when their price was changed from the newest to the oldest.
 */
class giftForResaleOrderPriceChangeDate final : public GiftForResaleOrder {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The gifts will be sorted by the last date when their price was changed from the newest to the oldest.
   */
  giftForResaleOrderPriceChangeDate();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1694144054;

  static object_ptr<GiftForResaleOrder> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The gifts will be sorted by their number from the smallest to the largest.
 */
class giftForResaleOrderNumber final : public GiftForResaleOrder {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The gifts will be sorted by their number from the smallest to the largest.
   */
  giftForResaleOrderNumber();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1301157632;

  static object_ptr<GiftForResaleOrder> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class acceptedGiftTypes;

/**
 * Contains settings for gift receiving for a user.
 */
class giftSettings final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// True, if a button for sending a gift to the user or by the user must always be shown in the input field.
  bool show_gift_button_;
  static jfieldID show_gift_button_fieldID;
  /// Types of gifts accepted by the user; for Telegram Premium users only.
  object_ptr<acceptedGiftTypes> accepted_gift_types_;
  static jfieldID accepted_gift_types_fieldID;

  /**
   * Contains settings for gift receiving for a user.
   */
  giftSettings();

  /**
   * Contains settings for gift receiving for a user.
   *
   * \param[in] show_gift_button_ True, if a button for sending a gift to the user or by the user must always be shown in the input field.
   * \param[in] accepted_gift_types_ Types of gifts accepted by the user; for Telegram Premium users only.
   */
  giftSettings(bool show_gift_button_, object_ptr<acceptedGiftTypes> &&accepted_gift_types_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 45783168;

  static object_ptr<giftSettings> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class upgradedGiftBackdrop;

class upgradedGiftModel;

class upgradedGiftSymbol;

/**
 * Contains examples of possible upgraded gifts for the given regular gift.
 */
class giftUpgradePreview final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Examples of possible models that can be chosen for the gift after upgrade.
  array<object_ptr<upgradedGiftModel>> models_;
  static jfieldID models_fieldID;
  /// Examples of possible symbols that can be chosen for the gift after upgrade.
  array<object_ptr<upgradedGiftSymbol>> symbols_;
  static jfieldID symbols_fieldID;
  /// Examples of possible backdrops that can be chosen for the gift after upgrade.
  array<object_ptr<upgradedGiftBackdrop>> backdrops_;
  static jfieldID backdrops_fieldID;

  /**
   * Contains examples of possible upgraded gifts for the given regular gift.
   */
  giftUpgradePreview();

  /**
   * Contains examples of possible upgraded gifts for the given regular gift.
   *
   * \param[in] models_ Examples of possible models that can be chosen for the gift after upgrade.
   * \param[in] symbols_ Examples of possible symbols that can be chosen for the gift after upgrade.
   * \param[in] backdrops_ Examples of possible backdrops that can be chosen for the gift after upgrade.
   */
  giftUpgradePreview(array<object_ptr<upgradedGiftModel>> &&models_, array<object_ptr<upgradedGiftSymbol>> &&symbols_, array<object_ptr<upgradedGiftBackdrop>> &&backdrops_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 729908218;

  static object_ptr<giftUpgradePreview> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class giftForResale;

class upgradedGiftBackdropCount;

class upgradedGiftModelCount;

class upgradedGiftSymbolCount;

/**
 * Describes gifts available for resale.
 */
class giftsForResale final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Total number of gifts found.
  int32 total_count_;
  static jfieldID total_count_fieldID;
  /// The gifts.
  array<object_ptr<giftForResale>> gifts_;
  static jfieldID gifts_fieldID;
  /// Available models; for searchGiftsForResale requests without offset and attributes only.
  array<object_ptr<upgradedGiftModelCount>> models_;
  static jfieldID models_fieldID;
  /// Available symbols; for searchGiftsForResale requests without offset and attributes only.
  array<object_ptr<upgradedGiftSymbolCount>> symbols_;
  static jfieldID symbols_fieldID;
  /// Available backdrops; for searchGiftsForResale requests without offset and attributes only.
  array<object_ptr<upgradedGiftBackdropCount>> backdrops_;
  static jfieldID backdrops_fieldID;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;
  static jfieldID next_offset_fieldID;

  /**
   * Describes gifts available for resale.
   */
  giftsForResale();

  /**
   * Describes gifts available for resale.
   *
   * \param[in] total_count_ Total number of gifts found.
   * \param[in] gifts_ The gifts.
   * \param[in] models_ Available models; for searchGiftsForResale requests without offset and attributes only.
   * \param[in] symbols_ Available symbols; for searchGiftsForResale requests without offset and attributes only.
   * \param[in] backdrops_ Available backdrops; for searchGiftsForResale requests without offset and attributes only.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  giftsForResale(int32 total_count_, array<object_ptr<giftForResale>> &&gifts_, array<object_ptr<upgradedGiftModelCount>> &&models_, array<object_ptr<upgradedGiftSymbolCount>> &&symbols_, array<object_ptr<upgradedGiftBackdropCount>> &&backdrops_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 35082425;

  static object_ptr<giftsForResale> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class GiveawayParticipantStatus;

/**
 * This class is an abstract base class.
 * Contains information about a giveaway.
 */
class GiveawayInfo: public Object {
 public:
  static jclass Class;

  static object_ptr<GiveawayInfo> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes an ongoing giveaway.
 */
class giveawayInfoOngoing final : public GiveawayInfo {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Point in time (Unix timestamp) when the giveaway was created.
  int32 creation_date_;
  static jfieldID creation_date_fieldID;
  /// Status of the current user in the giveaway.
  object_ptr<GiveawayParticipantStatus> status_;
  static jfieldID status_fieldID;
  /// True, if the giveaway has ended and results are being prepared.
  bool is_ended_;
  static jfieldID is_ended_fieldID;

  /**
   * Describes an ongoing giveaway.
   */
  giveawayInfoOngoing();

  /**
   * Describes an ongoing giveaway.
   *
   * \param[in] creation_date_ Point in time (Unix timestamp) when the giveaway was created.
   * \param[in] status_ Status of the current user in the giveaway.
   * \param[in] is_ended_ True, if the giveaway has ended and results are being prepared.
   */
  giveawayInfoOngoing(int32 creation_date_, object_ptr<GiveawayParticipantStatus> &&status_, bool is_ended_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1649336400;

  static object_ptr<GiveawayInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes a completed giveaway.
 */
class giveawayInfoCompleted final : public GiveawayInfo {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Point in time (Unix timestamp) when the giveaway was created.
  int32 creation_date_;
  static jfieldID creation_date_fieldID;
  /// Point in time (Unix timestamp) when the winners were selected. May be bigger than winners selection date specified in parameters of the giveaway.
  int32 actual_winners_selection_date_;
  static jfieldID actual_winners_selection_date_fieldID;
  /// True, if the giveaway was canceled and was fully refunded.
  bool was_refunded_;
  static jfieldID was_refunded_fieldID;
  /// True, if the current user is a winner of the giveaway.
  bool is_winner_;
  static jfieldID is_winner_fieldID;
  /// Number of winners in the giveaway.
  int32 winner_count_;
  static jfieldID winner_count_fieldID;
  /// Number of winners, which activated their gift codes; for Telegram Premium giveaways only.
  int32 activation_count_;
  static jfieldID activation_count_fieldID;
  /// Telegram Premium gift code that was received by the current user; empty if the user isn't a winner in the giveaway or the giveaway isn't a Telegram Premium giveaway.
  string gift_code_;
  static jfieldID gift_code_fieldID;
  /// The amount of Telegram Stars won by the current user; 0 if the user isn't a winner in the giveaway or the giveaway isn't a Telegram Star giveaway.
  int53 won_star_count_;
  static jfieldID won_star_count_fieldID;

  /**
   * Describes a completed giveaway.
   */
  giveawayInfoCompleted();

  /**
   * Describes a completed giveaway.
   *
   * \param[in] creation_date_ Point in time (Unix timestamp) when the giveaway was created.
   * \param[in] actual_winners_selection_date_ Point in time (Unix timestamp) when the winners were selected. May be bigger than winners selection date specified in parameters of the giveaway.
   * \param[in] was_refunded_ True, if the giveaway was canceled and was fully refunded.
   * \param[in] is_winner_ True, if the current user is a winner of the giveaway.
   * \param[in] winner_count_ Number of winners in the giveaway.
   * \param[in] activation_count_ Number of winners, which activated their gift codes; for Telegram Premium giveaways only.
   * \param[in] gift_code_ Telegram Premium gift code that was received by the current user; empty if the user isn't a winner in the giveaway or the giveaway isn't a Telegram Premium giveaway.
   * \param[in] won_star_count_ The amount of Telegram Stars won by the current user; 0 if the user isn't a winner in the giveaway or the giveaway isn't a Telegram Star giveaway.
   */
  giveawayInfoCompleted(int32 creation_date_, int32 actual_winners_selection_date_, bool was_refunded_, bool is_winner_, int32 winner_count_, int32 activation_count_, string const &gift_code_, int53 won_star_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 848085852;

  static object_ptr<GiveawayInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes parameters of a giveaway.
 */
class giveawayParameters final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the supergroup or channel chat, which will be automatically boosted by the winners of the giveaway for duration of the Telegram Premium subscription, or for the specified time. If the chat is a channel, then can_post_messages administrator right is required in the channel, otherwise, the user must be an administrator in the supergroup.
  int53 boosted_chat_id_;
  static jfieldID boosted_chat_id_fieldID;
  /// Identifiers of other supergroup or channel chats that must be subscribed by the users to be eligible for the giveaway. There can be up to getOption(&quot;giveaway_additional_chat_count_max&quot;) additional chats.
  array<int53> additional_chat_ids_;
  static jfieldID additional_chat_ids_fieldID;
  /// Point in time (Unix timestamp) when the giveaway is expected to be performed; must be 60-getOption(&quot;giveaway_duration_max&quot;) seconds in the future in scheduled giveaways.
  int32 winners_selection_date_;
  static jfieldID winners_selection_date_fieldID;
  /// True, if only new members of the chats will be eligible for the giveaway.
  bool only_new_members_;
  static jfieldID only_new_members_fieldID;
  /// True, if the list of winners of the giveaway will be available to everyone.
  bool has_public_winners_;
  static jfieldID has_public_winners_fieldID;
  /// The list of two-letter ISO 3166-1 alpha-2 codes of countries, users from which will be eligible for the giveaway. If empty, then all users can participate in the giveaway. There can be up to getOption(&quot;giveaway_country_count_max&quot;) chosen countries. Users with phone number that was bought at https://fragment.com can participate in any giveaway and the country code &quot;FT&quot; must not be specified in the list.
  array<string> country_codes_;
  static jfieldID country_codes_fieldID;
  /// Additional description of the giveaway prize; 0-128 characters.
  string prize_description_;
  static jfieldID prize_description_fieldID;

  /**
   * Describes parameters of a giveaway.
   */
  giveawayParameters();

  /**
   * Describes parameters of a giveaway.
   *
   * \param[in] boosted_chat_id_ Identifier of the supergroup or channel chat, which will be automatically boosted by the winners of the giveaway for duration of the Telegram Premium subscription, or for the specified time. If the chat is a channel, then can_post_messages administrator right is required in the channel, otherwise, the user must be an administrator in the supergroup.
   * \param[in] additional_chat_ids_ Identifiers of other supergroup or channel chats that must be subscribed by the users to be eligible for the giveaway. There can be up to getOption(&quot;giveaway_additional_chat_count_max&quot;) additional chats.
   * \param[in] winners_selection_date_ Point in time (Unix timestamp) when the giveaway is expected to be performed; must be 60-getOption(&quot;giveaway_duration_max&quot;) seconds in the future in scheduled giveaways.
   * \param[in] only_new_members_ True, if only new members of the chats will be eligible for the giveaway.
   * \param[in] has_public_winners_ True, if the list of winners of the giveaway will be available to everyone.
   * \param[in] country_codes_ The list of two-letter ISO 3166-1 alpha-2 codes of countries, users from which will be eligible for the giveaway. If empty, then all users can participate in the giveaway. There can be up to getOption(&quot;giveaway_country_count_max&quot;) chosen countries. Users with phone number that was bought at https://fragment.com can participate in any giveaway and the country code &quot;FT&quot; must not be specified in the list.
   * \param[in] prize_description_ Additional description of the giveaway prize; 0-128 characters.
   */
  giveawayParameters(int53 boosted_chat_id_, array<int53> &&additional_chat_ids_, int32 winners_selection_date_, bool only_new_members_, bool has_public_winners_, array<string> &&country_codes_, string const &prize_description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1171549354;

  static object_ptr<giveawayParameters> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Contains information about status of a user in a giveaway.
 */
class GiveawayParticipantStatus: public Object {
 public:
  static jclass Class;

  static object_ptr<GiveawayParticipantStatus> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user is eligible for the giveaway.
 */
class giveawayParticipantStatusEligible final : public GiveawayParticipantStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user is eligible for the giveaway.
   */
  giveawayParticipantStatusEligible();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 304799383;

  static object_ptr<GiveawayParticipantStatus> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user participates in the giveaway.
 */
class giveawayParticipantStatusParticipating final : public GiveawayParticipantStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The user participates in the giveaway.
   */
  giveawayParticipantStatusParticipating();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 492036975;

  static object_ptr<GiveawayParticipantStatus> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user can't participate in the giveaway, because they have already been member of the chat.
 */
class giveawayParticipantStatusAlreadyWasMember final : public GiveawayParticipantStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Point in time (Unix timestamp) when the user joined the chat.
  int32 joined_chat_date_;
  static jfieldID joined_chat_date_fieldID;

  /**
   * The user can't participate in the giveaway, because they have already been member of the chat.
   */
  giveawayParticipantStatusAlreadyWasMember();

  /**
   * The user can't participate in the giveaway, because they have already been member of the chat.
   *
   * \param[in] joined_chat_date_ Point in time (Unix timestamp) when the user joined the chat.
   */
  explicit giveawayParticipantStatusAlreadyWasMember(int32 joined_chat_date_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 301577632;

  static object_ptr<GiveawayParticipantStatus> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user can't participate in the giveaway, because they are an administrator in one of the chats that created the giveaway.
 */
class giveawayParticipantStatusAdministrator final : public GiveawayParticipantStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the chat administered by the user.
  int53 chat_id_;
  static jfieldID chat_id_fieldID;

  /**
   * The user can't participate in the giveaway, because they are an administrator in one of the chats that created the giveaway.
   */
  giveawayParticipantStatusAdministrator();

  /**
   * The user can't participate in the giveaway, because they are an administrator in one of the chats that created the giveaway.
   *
   * \param[in] chat_id_ Identifier of the chat administered by the user.
   */
  explicit giveawayParticipantStatusAdministrator(int53 chat_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -934593931;

  static object_ptr<GiveawayParticipantStatus> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The user can't participate in the giveaway, because they phone number is from a disallowed country.
 */
class giveawayParticipantStatusDisallowedCountry final : public GiveawayParticipantStatus {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A two-letter ISO 3166-1 alpha-2 country code of the user's country.
  string user_country_code_;
  static jfieldID user_country_code_fieldID;

  /**
   * The user can't participate in the giveaway, because they phone number is from a disallowed country.
   */
  giveawayParticipantStatusDisallowedCountry();

  /**
   * The user can't participate in the giveaway, because they phone number is from a disallowed country.
   *
   * \param[in] user_country_code_ A two-letter ISO 3166-1 alpha-2 country code of the user's country.
   */
  explicit giveawayParticipantStatusDisallowedCountry(string const &user_country_code_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1879794779;

  static object_ptr<GiveawayParticipantStatus> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Contains information about a giveaway prize.
 */
class GiveawayPrize: public Object {
 public:
  static jclass Class;

  static object_ptr<GiveawayPrize> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The giveaway sends Telegram Premium subscriptions to the winners.
 */
class giveawayPrizePremium final : public GiveawayPrize {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Number of months the Telegram Premium subscription will be active after code activation.
  int32 month_count_;
  static jfieldID month_count_fieldID;

  /**
   * The giveaway sends Telegram Premium subscriptions to the winners.
   */
  giveawayPrizePremium();

  /**
   * The giveaway sends Telegram Premium subscriptions to the winners.
   *
   * \param[in] month_count_ Number of months the Telegram Premium subscription will be active after code activation.
   */
  explicit giveawayPrizePremium(int32 month_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 454224248;

  static object_ptr<GiveawayPrize> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The giveaway sends Telegram Stars to the winners.
 */
class giveawayPrizeStars final : public GiveawayPrize {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Number of Telegram Stars that will be shared by all winners.
  int53 star_count_;
  static jfieldID star_count_fieldID;

  /**
   * The giveaway sends Telegram Stars to the winners.
   */
  giveawayPrizeStars();

  /**
   * The giveaway sends Telegram Stars to the winners.
   *
   * \param[in] star_count_ Number of Telegram Stars that will be shared by all winners.
   */
  explicit giveawayPrizeStars(int53 star_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1790173276;

  static object_ptr<GiveawayPrize> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class groupCallRecentSpeaker;

/**
 * Describes a group call.
 */
class groupCall final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Group call identifier.
  int32 id_;
  static jfieldID id_fieldID;
  /// Group call title; for video chats only.
  string title_;
  static jfieldID title_fieldID;
  /// Invite link for the group call; for group calls that aren't bound to a chat. For video chats call getVideoChatInviteLink to get the link.
  string invite_link_;
  static jfieldID invite_link_fieldID;
  /// Point in time (Unix timestamp) when the group call is expected to be started by an administrator; 0 if it is already active or was ended; for video chats only.
  int32 scheduled_start_date_;
  static jfieldID scheduled_start_date_fieldID;
  /// True, if the group call is scheduled and the current user will receive a notification when the group call starts; for video chats only.
  bool enabled_start_notification_;
  static jfieldID enabled_start_notification_fieldID;
  /// True, if the call is active.
  bool is_active_;
  static jfieldID is_active_fieldID;
  /// True, if the call is bound to a chat.
  bool is_video_chat_;
  static jfieldID is_video_chat_fieldID;
  /// True, if the call is an RTMP stream instead of an ordinary video chat; for video chats only.
  bool is_rtmp_stream_;
  static jfieldID is_rtmp_stream_fieldID;
  /// True, if the call is joined.
  bool is_joined_;
  static jfieldID is_joined_fieldID;
  /// True, if user was kicked from the call because of network loss and the call needs to be rejoined.
  bool need_rejoin_;
  static jfieldID need_rejoin_fieldID;
  /// True, if the user is the owner of the call and can end the call, change volume level of other users, or ban users there; for group calls that aren't bound to a chat.
  bool is_owned_;
  static jfieldID is_owned_fieldID;
  /// True, if the current user can manage the group call; for video chats only.
  bool can_be_managed_;
  static jfieldID can_be_managed_fieldID;
  /// Number of participants in the group call.
  int32 participant_count_;
  static jfieldID participant_count_fieldID;
  /// True, if group call participants, which are muted, aren't returned in participant list; for video chats only.
  bool has_hidden_listeners_;
  static jfieldID has_hidden_listeners_fieldID;
  /// True, if all group call participants are loaded.
  bool loaded_all_participants_;
  static jfieldID loaded_all_participants_fieldID;
  /// At most 3 recently speaking users in the group call.
  array<object_ptr<groupCallRecentSpeaker>> recent_speakers_;
  static jfieldID recent_speakers_fieldID;
  /// True, if the current user's video is enabled.
  bool is_my_video_enabled_;
  static jfieldID is_my_video_enabled_fieldID;
  /// True, if the current user's video is paused.
  bool is_my_video_paused_;
  static jfieldID is_my_video_paused_fieldID;
  /// True, if the current user can broadcast video or share screen.
  bool can_enable_video_;
  static jfieldID can_enable_video_fieldID;
  /// True, if only group call administrators can unmute new participants; for video chats only.
  bool mute_new_participants_;
  static jfieldID mute_new_participants_fieldID;
  /// True, if the current user can enable or disable mute_new_participants setting; for video chats only.
  bool can_toggle_mute_new_participants_;
  static jfieldID can_toggle_mute_new_participants_fieldID;
  /// Duration of the ongoing group call recording, in seconds; 0 if none. An updateGroupCall update is not triggered when value of this field changes, but the same recording goes on.
  int32 record_duration_;
  static jfieldID record_duration_fieldID;
  /// True, if a video file is being recorded for the call.
  bool is_video_recorded_;
  static jfieldID is_video_recorded_fieldID;
  /// Call duration, in seconds; for ended calls only.
  int32 duration_;
  static jfieldID duration_fieldID;

  /**
   * Describes a group call.
   */
  groupCall();

  /**
   * Describes a group call.
   *
   * \param[in] id_ Group call identifier.
   * \param[in] title_ Group call title; for video chats only.
   * \param[in] invite_link_ Invite link for the group call; for group calls that aren't bound to a chat. For video chats call getVideoChatInviteLink to get the link.
   * \param[in] scheduled_start_date_ Point in time (Unix timestamp) when the group call is expected to be started by an administrator; 0 if it is already active or was ended; for video chats only.
   * \param[in] enabled_start_notification_ True, if the group call is scheduled and the current user will receive a notification when the group call starts; for video chats only.
   * \param[in] is_active_ True, if the call is active.
   * \param[in] is_video_chat_ True, if the call is bound to a chat.
   * \param[in] is_rtmp_stream_ True, if the call is an RTMP stream instead of an ordinary video chat; for video chats only.
   * \param[in] is_joined_ True, if the call is joined.
   * \param[in] need_rejoin_ True, if user was kicked from the call because of network loss and the call needs to be rejoined.
   * \param[in] is_owned_ True, if the user is the owner of the call and can end the call, change volume level of other users, or ban users there; for group calls that aren't bound to a chat.
   * \param[in] can_be_managed_ True, if the current user can manage the group call; for video chats only.
   * \param[in] participant_count_ Number of participants in the group call.
   * \param[in] has_hidden_listeners_ True, if group call participants, which are muted, aren't returned in participant list; for video chats only.
   * \param[in] loaded_all_participants_ True, if all group call participants are loaded.
   * \param[in] recent_speakers_ At most 3 recently speaking users in the group call.
   * \param[in] is_my_video_enabled_ True, if the current user's video is enabled.
   * \param[in] is_my_video_paused_ True, if the current user's video is paused.
   * \param[in] can_enable_video_ True, if the current user can broadcast video or share screen.
   * \param[in] mute_new_participants_ True, if only group call administrators can unmute new participants; for video chats only.
   * \param[in] can_toggle_mute_new_participants_ True, if the current user can enable or disable mute_new_participants setting; for video chats only.
   * \param[in] record_duration_ Duration of the ongoing group call recording, in seconds; 0 if none. An updateGroupCall update is not triggered when value of this field changes, but the same recording goes on.
   * \param[in] is_video_recorded_ True, if a video file is being recorded for the call.
   * \param[in] duration_ Call duration, in seconds; for ended calls only.
   */
  groupCall(int32 id_, string const &title_, string const &invite_link_, int32 scheduled_start_date_, bool enabled_start_notification_, bool is_active_, bool is_video_chat_, bool is_rtmp_stream_, bool is_joined_, bool need_rejoin_, bool is_owned_, bool can_be_managed_, int32 participant_count_, bool has_hidden_listeners_, bool loaded_all_participants_, array<object_ptr<groupCallRecentSpeaker>> &&recent_speakers_, bool is_my_video_enabled_, bool is_my_video_paused_, bool can_enable_video_, bool mute_new_participants_, bool can_toggle_mute_new_participants_, int32 record_duration_, bool is_video_recorded_, int32 duration_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -144438687;

  static object_ptr<groupCall> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes data channel for a group call.
 */
class GroupCallDataChannel: public Object {
 public:
  static jclass Class;

  static object_ptr<GroupCallDataChannel> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The main data channel for audio and video data.
 */
class groupCallDataChannelMain final : public GroupCallDataChannel {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The main data channel for audio and video data.
   */
  groupCallDataChannelMain();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -32177779;

  static object_ptr<GroupCallDataChannel> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The data channel for screen sharing.
 */
class groupCallDataChannelScreenSharing final : public GroupCallDataChannel {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The data channel for screen sharing.
   */
  groupCallDataChannelScreenSharing();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -601649103;

  static object_ptr<GroupCallDataChannel> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains the group call identifier.
 */
class groupCallId final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Group call identifier.
  int32 id_;
  static jfieldID id_fieldID;

  /**
   * Contains the group call identifier.
   */
  groupCallId();

  /**
   * Contains the group call identifier.
   *
   * \param[in] id_ Group call identifier.
   */
  explicit groupCallId(int32 id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 350534469;

  static object_ptr<groupCallId> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains information about a just created or just joined group call.
 */
class groupCallInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the group call.
  int32 group_call_id_;
  static jfieldID group_call_id_fieldID;
  /// Join response payload for tgcalls; empty if the call isn't joined.
  string join_payload_;
  static jfieldID join_payload_fieldID;

  /**
   * Contains information about a just created or just joined group call.
   */
  groupCallInfo();

  /**
   * Contains information about a just created or just joined group call.
   *
   * \param[in] group_call_id_ Identifier of the group call.
   * \param[in] join_payload_ Join response payload for tgcalls; empty if the call isn't joined.
   */
  groupCallInfo(int32 group_call_id_, string const &join_payload_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 892575956;

  static object_ptr<groupCallInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes parameters used to join a group call.
 */
class groupCallJoinParameters final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Audio channel synchronization source identifier; received from tgcalls.
  int32 audio_source_id_;
  static jfieldID audio_source_id_fieldID;
  /// Group call join payload; received from tgcalls.
  string payload_;
  static jfieldID payload_fieldID;
  /// Pass true to join the call with muted microphone.
  bool is_muted_;
  static jfieldID is_muted_fieldID;
  /// Pass true if the user's video is enabled.
  bool is_my_video_enabled_;
  static jfieldID is_my_video_enabled_fieldID;

  /**
   * Describes parameters used to join a group call.
   */
  groupCallJoinParameters();

  /**
   * Describes parameters used to join a group call.
   *
   * \param[in] audio_source_id_ Audio channel synchronization source identifier; received from tgcalls.
   * \param[in] payload_ Group call join payload; received from tgcalls.
   * \param[in] is_muted_ Pass true to join the call with muted microphone.
   * \param[in] is_my_video_enabled_ Pass true if the user's video is enabled.
   */
  groupCallJoinParameters(int32 audio_source_id_, string const &payload_, bool is_muted_, bool is_my_video_enabled_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1763438054;

  static object_ptr<groupCallJoinParameters> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class MessageSender;

class groupCallParticipantVideoInfo;

/**
 * Represents a group call participant.
 */
class groupCallParticipant final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the group call participant.
  object_ptr<MessageSender> participant_id_;
  static jfieldID participant_id_fieldID;
  /// User's audio channel synchronization source identifier.
  int32 audio_source_id_;
  static jfieldID audio_source_id_fieldID;
  /// User's screen sharing audio channel synchronization source identifier.
  int32 screen_sharing_audio_source_id_;
  static jfieldID screen_sharing_audio_source_id_fieldID;
  /// Information about user's video channel; may be null if there is no active video.
  object_ptr<groupCallParticipantVideoInfo> video_info_;
  static jfieldID video_info_fieldID;
  /// Information about user's screen sharing video channel; may be null if there is no active screen sharing video.
  object_ptr<groupCallParticipantVideoInfo> screen_sharing_video_info_;
  static jfieldID screen_sharing_video_info_fieldID;
  /// The participant user's bio or the participant chat's description.
  string bio_;
  static jfieldID bio_fieldID;
  /// True, if the participant is the current user.
  bool is_current_user_;
  static jfieldID is_current_user_fieldID;
  /// True, if the participant is speaking as set by setGroupCallParticipantIsSpeaking.
  bool is_speaking_;
  static jfieldID is_speaking_fieldID;
  /// True, if the participant hand is raised.
  bool is_hand_raised_;
  static jfieldID is_hand_raised_fieldID;
  /// True, if the current user can mute the participant for all other group call participants.
  bool can_be_muted_for_all_users_;
  static jfieldID can_be_muted_for_all_users_fieldID;
  /// True, if the current user can allow the participant to unmute themselves or unmute the participant (if the participant is the current user).
  bool can_be_unmuted_for_all_users_;
  static jfieldID can_be_unmuted_for_all_users_fieldID;
  /// True, if the current user can mute the participant only for self.
  bool can_be_muted_for_current_user_;
  static jfieldID can_be_muted_for_current_user_fieldID;
  /// True, if the current user can unmute the participant for self.
  bool can_be_unmuted_for_current_user_;
  static jfieldID can_be_unmuted_for_current_user_fieldID;
  /// True, if the participant is muted for all users.
  bool is_muted_for_all_users_;
  static jfieldID is_muted_for_all_users_fieldID;
  /// True, if the participant is muted for the current user.
  bool is_muted_for_current_user_;
  static jfieldID is_muted_for_current_user_fieldID;
  /// True, if the participant is muted for all users, but can unmute themselves.
  bool can_unmute_self_;
  static jfieldID can_unmute_self_fieldID;
  /// Participant's volume level; 1-20000 in hundreds of percents.
  int32 volume_level_;
  static jfieldID volume_level_fieldID;
  /// User's order in the group call participant list. Orders must be compared lexicographically. The bigger is order, the higher is user in the list. If order is empty, the user must be removed from the participant list.
  string order_;
  static jfieldID order_fieldID;

  /**
   * Represents a group call participant.
   */
  groupCallParticipant();

  /**
   * Represents a group call participant.
   *
   * \param[in] participant_id_ Identifier of the group call participant.
   * \param[in] audio_source_id_ User's audio channel synchronization source identifier.
   * \param[in] screen_sharing_audio_source_id_ User's screen sharing audio channel synchronization source identifier.
   * \param[in] video_info_ Information about user's video channel; may be null if there is no active video.
   * \param[in] screen_sharing_video_info_ Information about user's screen sharing video channel; may be null if there is no active screen sharing video.
   * \param[in] bio_ The participant user's bio or the participant chat's description.
   * \param[in] is_current_user_ True, if the participant is the current user.
   * \param[in] is_speaking_ True, if the participant is speaking as set by setGroupCallParticipantIsSpeaking.
   * \param[in] is_hand_raised_ True, if the participant hand is raised.
   * \param[in] can_be_muted_for_all_users_ True, if the current user can mute the participant for all other group call participants.
   * \param[in] can_be_unmuted_for_all_users_ True, if the current user can allow the participant to unmute themselves or unmute the participant (if the participant is the current user).
   * \param[in] can_be_muted_for_current_user_ True, if the current user can mute the participant only for self.
   * \param[in] can_be_unmuted_for_current_user_ True, if the current user can unmute the participant for self.
   * \param[in] is_muted_for_all_users_ True, if the participant is muted for all users.
   * \param[in] is_muted_for_current_user_ True, if the participant is muted for the current user.
   * \param[in] can_unmute_self_ True, if the participant is muted for all users, but can unmute themselves.
   * \param[in] volume_level_ Participant's volume level; 1-20000 in hundreds of percents.
   * \param[in] order_ User's order in the group call participant list. Orders must be compared lexicographically. The bigger is order, the higher is user in the list. If order is empty, the user must be removed from the participant list.
   */
  groupCallParticipant(object_ptr<MessageSender> &&participant_id_, int32 audio_source_id_, int32 screen_sharing_audio_source_id_, object_ptr<groupCallParticipantVideoInfo> &&video_info_, object_ptr<groupCallParticipantVideoInfo> &&screen_sharing_video_info_, string const &bio_, bool is_current_user_, bool is_speaking_, bool is_hand_raised_, bool can_be_muted_for_all_users_, bool can_be_unmuted_for_all_users_, bool can_be_muted_for_current_user_, bool can_be_unmuted_for_current_user_, bool is_muted_for_all_users_, bool is_muted_for_current_user_, bool can_unmute_self_, int32 volume_level_, string const &order_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2059182571;

  static object_ptr<groupCallParticipant> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class groupCallVideoSourceGroup;

/**
 * Contains information about a group call participant's video channel.
 */
class groupCallParticipantVideoInfo final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// List of synchronization source groups of the video.
  array<object_ptr<groupCallVideoSourceGroup>> source_groups_;
  static jfieldID source_groups_fieldID;
  /// Video channel endpoint identifier.
  string endpoint_id_;
  static jfieldID endpoint_id_fieldID;
  /// True, if the video is paused. This flag needs to be ignored, if new video frames are received.
  bool is_paused_;
  static jfieldID is_paused_fieldID;

  /**
   * Contains information about a group call participant's video channel.
   */
  groupCallParticipantVideoInfo();

  /**
   * Contains information about a group call participant's video channel.
   *
   * \param[in] source_groups_ List of synchronization source groups of the video.
   * \param[in] endpoint_id_ Video channel endpoint identifier.
   * \param[in] is_paused_ True, if the video is paused. This flag needs to be ignored, if new video frames are received.
   */
  groupCallParticipantVideoInfo(array<object_ptr<groupCallVideoSourceGroup>> &&source_groups_, string const &endpoint_id_, bool is_paused_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -14294645;

  static object_ptr<groupCallParticipantVideoInfo> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class MessageSender;

/**
 * Contains identifiers of group call participants.
 */
class groupCallParticipants final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Total number of group call participants.
  int32 total_count_;
  static jfieldID total_count_fieldID;
  /// Identifiers of the participants.
  array<object_ptr<MessageSender>> participant_ids_;
  static jfieldID participant_ids_fieldID;

  /**
   * Contains identifiers of group call participants.
   */
  groupCallParticipants();

  /**
   * Contains identifiers of group call participants.
   *
   * \param[in] total_count_ Total number of group call participants.
   * \param[in] participant_ids_ Identifiers of the participants.
   */
  groupCallParticipants(int32 total_count_, array<object_ptr<MessageSender>> &&participant_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1042491570;

  static object_ptr<groupCallParticipants> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class MessageSender;

/**
 * Describes a recently speaking participant in a group call.
 */
class groupCallRecentSpeaker final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Group call participant identifier.
  object_ptr<MessageSender> participant_id_;
  static jfieldID participant_id_fieldID;
  /// True, is the user has spoken recently.
  bool is_speaking_;
  static jfieldID is_speaking_fieldID;

  /**
   * Describes a recently speaking participant in a group call.
   */
  groupCallRecentSpeaker();

  /**
   * Describes a recently speaking participant in a group call.
   *
   * \param[in] participant_id_ Group call participant identifier.
   * \param[in] is_speaking_ True, is the user has spoken recently.
   */
  groupCallRecentSpeaker(object_ptr<MessageSender> &&participant_id_, bool is_speaking_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1819519436;

  static object_ptr<groupCallRecentSpeaker> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Describes the quality of a group call video.
 */
class GroupCallVideoQuality: public Object {
 public:
  static jclass Class;

  static object_ptr<GroupCallVideoQuality> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The worst available video quality.
 */
class groupCallVideoQualityThumbnail final : public GroupCallVideoQuality {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The worst available video quality.
   */
  groupCallVideoQualityThumbnail();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -379186304;

  static object_ptr<GroupCallVideoQuality> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The medium video quality.
 */
class groupCallVideoQualityMedium final : public GroupCallVideoQuality {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The medium video quality.
   */
  groupCallVideoQualityMedium();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 394968234;

  static object_ptr<GroupCallVideoQuality> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * The best available video quality.
 */
class groupCallVideoQualityFull final : public GroupCallVideoQuality {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * The best available video quality.
   */
  groupCallVideoQualityFull();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2125916617;

  static object_ptr<GroupCallVideoQuality> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes a group of video synchronization source identifiers.
 */
class groupCallVideoSourceGroup final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The semantics of sources, one of &quot;SIM&quot; or &quot;FID&quot;.
  string semantics_;
  static jfieldID semantics_fieldID;
  /// The list of synchronization source identifiers.
  array<int32> source_ids_;
  static jfieldID source_ids_fieldID;

  /**
   * Describes a group of video synchronization source identifiers.
   */
  groupCallVideoSourceGroup();

  /**
   * Describes a group of video synchronization source identifiers.
   *
   * \param[in] semantics_ The semantics of sources, one of &quot;SIM&quot; or &quot;FID&quot;.
   * \param[in] source_ids_ The list of synchronization source identifiers.
   */
  groupCallVideoSourceGroup(string const &semantics_, array<int32> &&source_ids_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1190900785;

  static object_ptr<groupCallVideoSourceGroup> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains a list of hashtags.
 */
class hashtags final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// A list of hashtags.
  array<string> hashtags_;
  static jfieldID hashtags_fieldID;

  /**
   * Contains a list of hashtags.
   */
  hashtags();

  /**
   * Contains a list of hashtags.
   *
   * \param[in] hashtags_ A list of hashtags.
   */
  explicit hashtags(array<string> &&hashtags_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 676798885;

  static object_ptr<hashtags> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Contains an HTTP URL.
 */
class httpUrl final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The URL.
  string url_;
  static jfieldID url_fieldID;

  /**
   * Contains an HTTP URL.
   */
  httpUrl();

  /**
   * Contains an HTTP URL.
   *
   * \param[in] url_ The URL.
   */
  explicit httpUrl(string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2018019930;

  static object_ptr<httpUrl> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class date;

class datedFile;

/**
 * An identity document.
 */
class identityDocument final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Document number; 1-24 characters.
  string number_;
  static jfieldID number_fieldID;
  /// Document expiration date; may be null if not applicable.
  object_ptr<date> expiration_date_;
  static jfieldID expiration_date_fieldID;
  /// Front side of the document.
  object_ptr<datedFile> front_side_;
  static jfieldID front_side_fieldID;
  /// Reverse side of the document; only for driver license and identity card; may be null.
  object_ptr<datedFile> reverse_side_;
  static jfieldID reverse_side_fieldID;
  /// Selfie with the document; may be null.
  object_ptr<datedFile> selfie_;
  static jfieldID selfie_fieldID;
  /// List of files containing a certified English translation of the document.
  array<object_ptr<datedFile>> translation_;
  static jfieldID translation_fieldID;

  /**
   * An identity document.
   */
  identityDocument();

  /**
   * An identity document.
   *
   * \param[in] number_ Document number; 1-24 characters.
   * \param[in] expiration_date_ Document expiration date; may be null if not applicable.
   * \param[in] front_side_ Front side of the document.
   * \param[in] reverse_side_ Reverse side of the document; only for driver license and identity card; may be null.
   * \param[in] selfie_ Selfie with the document; may be null.
   * \param[in] translation_ List of files containing a certified English translation of the document.
   */
  identityDocument(string const &number_, object_ptr<date> &&expiration_date_, object_ptr<datedFile> &&front_side_, object_ptr<datedFile> &&reverse_side_, object_ptr<datedFile> &&selfie_, array<object_ptr<datedFile>> &&translation_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1001703606;

  static object_ptr<identityDocument> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents the result of an importContacts request.
 */
class importedContacts final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// User identifiers of the imported contacts in the same order as they were specified in the request; 0 if the contact is not yet a registered user.
  array<int53> user_ids_;
  static jfieldID user_ids_fieldID;
  /// The number of users that imported the corresponding contact; 0 for already registered users or if unavailable.
  array<int32> importer_count_;
  static jfieldID importer_count_fieldID;

  /**
   * Represents the result of an importContacts request.
   */
  importedContacts();

  /**
   * Represents the result of an importContacts request.
   *
   * \param[in] user_ids_ User identifiers of the imported contacts in the same order as they were specified in the request; 0 if the contact is not yet a registered user.
   * \param[in] importer_count_ The number of users that imported the corresponding contact; 0 for already registered users or if unavailable.
   */
  importedContacts(array<int53> &&user_ids_, array<int32> &&importer_count_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2068432290;

  static object_ptr<importedContacts> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class InlineKeyboardButtonType;

/**
 * Represents a single button in an inline keyboard.
 */
class inlineKeyboardButton final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Text of the button.
  string text_;
  static jfieldID text_fieldID;
  /// Type of the button.
  object_ptr<InlineKeyboardButtonType> type_;
  static jfieldID type_fieldID;

  /**
   * Represents a single button in an inline keyboard.
   */
  inlineKeyboardButton();

  /**
   * Represents a single button in an inline keyboard.
   *
   * \param[in] text_ Text of the button.
   * \param[in] type_ Type of the button.
   */
  inlineKeyboardButton(string const &text_, object_ptr<InlineKeyboardButtonType> &&type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -372105704;

  static object_ptr<inlineKeyboardButton> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class TargetChat;

/**
 * This class is an abstract base class.
 * Describes the type of inline keyboard button.
 */
class InlineKeyboardButtonType: public Object {
 public:
  static jclass Class;

  static object_ptr<InlineKeyboardButtonType> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A button that opens a specified URL.
 */
class inlineKeyboardButtonTypeUrl final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// HTTP or tg:// URL to open. If the link is of the type internalLinkTypeWebApp, then the button must be marked as a Web App button.
  string url_;
  static jfieldID url_fieldID;

  /**
   * A button that opens a specified URL.
   */
  inlineKeyboardButtonTypeUrl();

  /**
   * A button that opens a specified URL.
   *
   * \param[in] url_ HTTP or tg:// URL to open. If the link is of the type internalLinkTypeWebApp, then the button must be marked as a Web App button.
   */
  explicit inlineKeyboardButtonTypeUrl(string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1130741420;

  static object_ptr<InlineKeyboardButtonType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A button that opens a specified URL and automatically authorize the current user by calling getLoginUrlInfo.
 */
class inlineKeyboardButtonTypeLoginUrl final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// An HTTP URL to pass to getLoginUrlInfo.
  string url_;
  static jfieldID url_fieldID;
  /// Unique button identifier.
  int53 id_;
  static jfieldID id_fieldID;
  /// If non-empty, new text of the button in forwarded messages.
  string forward_text_;
  static jfieldID forward_text_fieldID;

  /**
   * A button that opens a specified URL and automatically authorize the current user by calling getLoginUrlInfo.
   */
  inlineKeyboardButtonTypeLoginUrl();

  /**
   * A button that opens a specified URL and automatically authorize the current user by calling getLoginUrlInfo.
   *
   * \param[in] url_ An HTTP URL to pass to getLoginUrlInfo.
   * \param[in] id_ Unique button identifier.
   * \param[in] forward_text_ If non-empty, new text of the button in forwarded messages.
   */
  inlineKeyboardButtonTypeLoginUrl(string const &url_, int53 id_, string const &forward_text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1203413081;

  static object_ptr<InlineKeyboardButtonType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A button that opens a Web App by calling openWebApp.
 */
class inlineKeyboardButtonTypeWebApp final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// An HTTP URL to pass to openWebApp.
  string url_;
  static jfieldID url_fieldID;

  /**
   * A button that opens a Web App by calling openWebApp.
   */
  inlineKeyboardButtonTypeWebApp();

  /**
   * A button that opens a Web App by calling openWebApp.
   *
   * \param[in] url_ An HTTP URL to pass to openWebApp.
   */
  explicit inlineKeyboardButtonTypeWebApp(string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1767471672;

  static object_ptr<InlineKeyboardButtonType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A button that sends a callback query to a bot.
 */
class inlineKeyboardButtonTypeCallback final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Data to be sent to the bot via a callback query.
  bytes data_;
  static jfieldID data_fieldID;

  /**
   * A button that sends a callback query to a bot.
   */
  inlineKeyboardButtonTypeCallback();

  /**
   * A button that sends a callback query to a bot.
   *
   * \param[in] data_ Data to be sent to the bot via a callback query.
   */
  explicit inlineKeyboardButtonTypeCallback(bytes const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1127515139;

  static object_ptr<InlineKeyboardButtonType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A button that asks for the 2-step verification password of the current user and then sends a callback query to a bot.
 */
class inlineKeyboardButtonTypeCallbackWithPassword final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Data to be sent to the bot via a callback query.
  bytes data_;
  static jfieldID data_fieldID;

  /**
   * A button that asks for the 2-step verification password of the current user and then sends a callback query to a bot.
   */
  inlineKeyboardButtonTypeCallbackWithPassword();

  /**
   * A button that asks for the 2-step verification password of the current user and then sends a callback query to a bot.
   *
   * \param[in] data_ Data to be sent to the bot via a callback query.
   */
  explicit inlineKeyboardButtonTypeCallbackWithPassword(bytes const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 908018248;

  static object_ptr<InlineKeyboardButtonType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A button with a game that sends a callback query to a bot. This button must be in the first column and row of the keyboard and can be attached only to a message with content of the type messageGame.
 */
class inlineKeyboardButtonTypeCallbackGame final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * A button with a game that sends a callback query to a bot. This button must be in the first column and row of the keyboard and can be attached only to a message with content of the type messageGame.
   */
  inlineKeyboardButtonTypeCallbackGame();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -383429528;

  static object_ptr<InlineKeyboardButtonType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A button that forces an inline query to the bot to be inserted in the input field.
 */
class inlineKeyboardButtonTypeSwitchInline final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Inline query to be sent to the bot.
  string query_;
  static jfieldID query_fieldID;
  /// Target chat from which to send the inline query.
  object_ptr<TargetChat> target_chat_;
  static jfieldID target_chat_fieldID;

  /**
   * A button that forces an inline query to the bot to be inserted in the input field.
   */
  inlineKeyboardButtonTypeSwitchInline();

  /**
   * A button that forces an inline query to the bot to be inserted in the input field.
   *
   * \param[in] query_ Inline query to be sent to the bot.
   * \param[in] target_chat_ Target chat from which to send the inline query.
   */
  inlineKeyboardButtonTypeSwitchInline(string const &query_, object_ptr<TargetChat> &&target_chat_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 544906485;

  static object_ptr<InlineKeyboardButtonType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A button to buy something. This button must be in the first column and row of the keyboard and can be attached only to a message with content of the type messageInvoice.
 */
class inlineKeyboardButtonTypeBuy final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;

  /**
   * A button to buy something. This button must be in the first column and row of the keyboard and can be attached only to a message with content of the type messageInvoice.
   */
  inlineKeyboardButtonTypeBuy();

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1360739440;

  static object_ptr<InlineKeyboardButtonType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A button with a user reference to be handled in the same way as textEntityTypeMentionName entities.
 */
class inlineKeyboardButtonTypeUser final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// User identifier.
  int53 user_id_;
  static jfieldID user_id_fieldID;

  /**
   * A button with a user reference to be handled in the same way as textEntityTypeMentionName entities.
   */
  inlineKeyboardButtonTypeUser();

  /**
   * A button with a user reference to be handled in the same way as textEntityTypeMentionName entities.
   *
   * \param[in] user_id_ User identifier.
   */
  explicit inlineKeyboardButtonTypeUser(int53 user_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1836574114;

  static object_ptr<InlineKeyboardButtonType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A button that copies specified text to clipboard.
 */
class inlineKeyboardButtonTypeCopyText final : public InlineKeyboardButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The text to copy to clipboard.
  string text_;
  static jfieldID text_fieldID;

  /**
   * A button that copies specified text to clipboard.
   */
  inlineKeyboardButtonTypeCopyText();

  /**
   * A button that copies specified text to clipboard.
   *
   * \param[in] text_ The text to copy to clipboard.
   */
  explicit inlineKeyboardButtonTypeCopyText(string const &text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 68883206;

  static object_ptr<InlineKeyboardButtonType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class animation;

class audio;

class contact;

class document;

class game;

class location;

class photo;

class sticker;

class thumbnail;

class venue;

class video;

class voiceNote;

/**
 * This class is an abstract base class.
 * Represents a single result of an inline query.
 */
class InlineQueryResult: public Object {
 public:
  static jclass Class;

  static object_ptr<InlineQueryResult> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents a link to an article or web page.
 */
class inlineQueryResultArticle final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the query result.
  string id_;
  static jfieldID id_fieldID;
  /// URL of the result, if it exists.
  string url_;
  static jfieldID url_fieldID;
  /// Title of the result.
  string title_;
  static jfieldID title_fieldID;
  /// A short description of the result.
  string description_;
  static jfieldID description_fieldID;
  /// Result thumbnail in JPEG format; may be null.
  object_ptr<thumbnail> thumbnail_;
  static jfieldID thumbnail_fieldID;

  /**
   * Represents a link to an article or web page.
   */
  inlineQueryResultArticle();

  /**
   * Represents a link to an article or web page.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] url_ URL of the result, if it exists.
   * \param[in] title_ Title of the result.
   * \param[in] description_ A short description of the result.
   * \param[in] thumbnail_ Result thumbnail in JPEG format; may be null.
   */
  inlineQueryResultArticle(string const &id_, string const &url_, string const &title_, string const &description_, object_ptr<thumbnail> &&thumbnail_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 269930522;

  static object_ptr<InlineQueryResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents a user contact.
 */
class inlineQueryResultContact final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the query result.
  string id_;
  static jfieldID id_fieldID;
  /// A user contact.
  object_ptr<contact> contact_;
  static jfieldID contact_fieldID;
  /// Result thumbnail in JPEG format; may be null.
  object_ptr<thumbnail> thumbnail_;
  static jfieldID thumbnail_fieldID;

  /**
   * Represents a user contact.
   */
  inlineQueryResultContact();

  /**
   * Represents a user contact.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] contact_ A user contact.
   * \param[in] thumbnail_ Result thumbnail in JPEG format; may be null.
   */
  inlineQueryResultContact(string const &id_, object_ptr<contact> &&contact_, object_ptr<thumbnail> &&thumbnail_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -181960174;

  static object_ptr<InlineQueryResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents a point on the map.
 */
class inlineQueryResultLocation final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the query result.
  string id_;
  static jfieldID id_fieldID;
  /// Location result.
  object_ptr<location> location_;
  static jfieldID location_fieldID;
  /// Title of the result.
  string title_;
  static jfieldID title_fieldID;
  /// Result thumbnail in JPEG format; may be null.
  object_ptr<thumbnail> thumbnail_;
  static jfieldID thumbnail_fieldID;

  /**
   * Represents a point on the map.
   */
  inlineQueryResultLocation();

  /**
   * Represents a point on the map.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] location_ Location result.
   * \param[in] title_ Title of the result.
   * \param[in] thumbnail_ Result thumbnail in JPEG format; may be null.
   */
  inlineQueryResultLocation(string const &id_, object_ptr<location> &&location_, string const &title_, object_ptr<thumbnail> &&thumbnail_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 466004752;

  static object_ptr<InlineQueryResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents information about a venue.
 */
class inlineQueryResultVenue final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the query result.
  string id_;
  static jfieldID id_fieldID;
  /// Venue result.
  object_ptr<venue> venue_;
  static jfieldID venue_fieldID;
  /// Result thumbnail in JPEG format; may be null.
  object_ptr<thumbnail> thumbnail_;
  static jfieldID thumbnail_fieldID;

  /**
   * Represents information about a venue.
   */
  inlineQueryResultVenue();

  /**
   * Represents information about a venue.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] venue_ Venue result.
   * \param[in] thumbnail_ Result thumbnail in JPEG format; may be null.
   */
  inlineQueryResultVenue(string const &id_, object_ptr<venue> &&venue_, object_ptr<thumbnail> &&thumbnail_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1281036382;

  static object_ptr<InlineQueryResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents information about a game.
 */
class inlineQueryResultGame final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the query result.
  string id_;
  static jfieldID id_fieldID;
  /// Game result.
  object_ptr<game> game_;
  static jfieldID game_fieldID;

  /**
   * Represents information about a game.
   */
  inlineQueryResultGame();

  /**
   * Represents information about a game.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] game_ Game result.
   */
  inlineQueryResultGame(string const &id_, object_ptr<game> &&game_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1706916987;

  static object_ptr<InlineQueryResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents an animation file.
 */
class inlineQueryResultAnimation final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the query result.
  string id_;
  static jfieldID id_fieldID;
  /// Animation file.
  object_ptr<animation> animation_;
  static jfieldID animation_fieldID;
  /// Animation title.
  string title_;
  static jfieldID title_fieldID;

  /**
   * Represents an animation file.
   */
  inlineQueryResultAnimation();

  /**
   * Represents an animation file.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] animation_ Animation file.
   * \param[in] title_ Animation title.
   */
  inlineQueryResultAnimation(string const &id_, object_ptr<animation> &&animation_, string const &title_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 2009984267;

  static object_ptr<InlineQueryResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents an audio file.
 */
class inlineQueryResultAudio final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the query result.
  string id_;
  static jfieldID id_fieldID;
  /// Audio file.
  object_ptr<audio> audio_;
  static jfieldID audio_fieldID;

  /**
   * Represents an audio file.
   */
  inlineQueryResultAudio();

  /**
   * Represents an audio file.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] audio_ Audio file.
   */
  inlineQueryResultAudio(string const &id_, object_ptr<audio> &&audio_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 842650360;

  static object_ptr<InlineQueryResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents a document.
 */
class inlineQueryResultDocument final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the query result.
  string id_;
  static jfieldID id_fieldID;
  /// Document.
  object_ptr<document> document_;
  static jfieldID document_fieldID;
  /// Document title.
  string title_;
  static jfieldID title_fieldID;
  /// Document description.
  string description_;
  static jfieldID description_fieldID;

  /**
   * Represents a document.
   */
  inlineQueryResultDocument();

  /**
   * Represents a document.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] document_ Document.
   * \param[in] title_ Document title.
   * \param[in] description_ Document description.
   */
  inlineQueryResultDocument(string const &id_, object_ptr<document> &&document_, string const &title_, string const &description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1491268539;

  static object_ptr<InlineQueryResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents a photo.
 */
class inlineQueryResultPhoto final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the query result.
  string id_;
  static jfieldID id_fieldID;
  /// Photo.
  object_ptr<photo> photo_;
  static jfieldID photo_fieldID;
  /// Title of the result, if known.
  string title_;
  static jfieldID title_fieldID;
  /// A short description of the result, if known.
  string description_;
  static jfieldID description_fieldID;

  /**
   * Represents a photo.
   */
  inlineQueryResultPhoto();

  /**
   * Represents a photo.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] photo_ Photo.
   * \param[in] title_ Title of the result, if known.
   * \param[in] description_ A short description of the result, if known.
   */
  inlineQueryResultPhoto(string const &id_, object_ptr<photo> &&photo_, string const &title_, string const &description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1848319440;

  static object_ptr<InlineQueryResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents a sticker.
 */
class inlineQueryResultSticker final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the query result.
  string id_;
  static jfieldID id_fieldID;
  /// Sticker.
  object_ptr<sticker> sticker_;
  static jfieldID sticker_fieldID;

  /**
   * Represents a sticker.
   */
  inlineQueryResultSticker();

  /**
   * Represents a sticker.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] sticker_ Sticker.
   */
  inlineQueryResultSticker(string const &id_, object_ptr<sticker> &&sticker_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1848224245;

  static object_ptr<InlineQueryResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents a video.
 */
class inlineQueryResultVideo final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the query result.
  string id_;
  static jfieldID id_fieldID;
  /// Video.
  object_ptr<video> video_;
  static jfieldID video_fieldID;
  /// Title of the video.
  string title_;
  static jfieldID title_fieldID;
  /// Description of the video.
  string description_;
  static jfieldID description_fieldID;

  /**
   * Represents a video.
   */
  inlineQueryResultVideo();

  /**
   * Represents a video.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] video_ Video.
   * \param[in] title_ Title of the video.
   * \param[in] description_ Description of the video.
   */
  inlineQueryResultVideo(string const &id_, object_ptr<video> &&video_, string const &title_, string const &description_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1373158683;

  static object_ptr<InlineQueryResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Represents a voice note.
 */
class inlineQueryResultVoiceNote final : public InlineQueryResult {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the query result.
  string id_;
  static jfieldID id_fieldID;
  /// Voice note.
  object_ptr<voiceNote> voice_note_;
  static jfieldID voice_note_fieldID;
  /// Title of the voice note.
  string title_;
  static jfieldID title_fieldID;

  /**
   * Represents a voice note.
   */
  inlineQueryResultVoiceNote();

  /**
   * Represents a voice note.
   *
   * \param[in] id_ Unique identifier of the query result.
   * \param[in] voice_note_ Voice note.
   * \param[in] title_ Title of the voice note.
   */
  inlineQueryResultVoiceNote(string const &id_, object_ptr<voiceNote> &&voice_note_, string const &title_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1897393105;

  static object_ptr<InlineQueryResult> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class InlineQueryResult;

class inlineQueryResultsButton;

/**
 * Represents the results of the inline query. Use sendInlineQueryResultMessage to send the result of the query.
 */
class inlineQueryResults final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the inline query.
  int64 inline_query_id_;
  static jfieldID inline_query_id_fieldID;
  /// Button to be shown above inline query results; may be null.
  object_ptr<inlineQueryResultsButton> button_;
  static jfieldID button_fieldID;
  /// Results of the query.
  array<object_ptr<InlineQueryResult>> results_;
  static jfieldID results_fieldID;
  /// The offset for the next request. If empty, then there are no more results.
  string next_offset_;
  static jfieldID next_offset_fieldID;

  /**
   * Represents the results of the inline query. Use sendInlineQueryResultMessage to send the result of the query.
   */
  inlineQueryResults();

  /**
   * Represents the results of the inline query. Use sendInlineQueryResultMessage to send the result of the query.
   *
   * \param[in] inline_query_id_ Unique identifier of the inline query.
   * \param[in] button_ Button to be shown above inline query results; may be null.
   * \param[in] results_ Results of the query.
   * \param[in] next_offset_ The offset for the next request. If empty, then there are no more results.
   */
  inlineQueryResults(int64 inline_query_id_, object_ptr<inlineQueryResultsButton> &&button_, array<object_ptr<InlineQueryResult>> &&results_, string const &next_offset_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1830685615;

  static object_ptr<inlineQueryResults> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class InlineQueryResultsButtonType;

/**
 * Represents a button to be shown above inline query results.
 */
class inlineQueryResultsButton final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The text of the button.
  string text_;
  static jfieldID text_fieldID;
  /// Type of the button.
  object_ptr<InlineQueryResultsButtonType> type_;
  static jfieldID type_fieldID;

  /**
   * Represents a button to be shown above inline query results.
   */
  inlineQueryResultsButton();

  /**
   * Represents a button to be shown above inline query results.
   *
   * \param[in] text_ The text of the button.
   * \param[in] type_ Type of the button.
   */
  inlineQueryResultsButton(string const &text_, object_ptr<InlineQueryResultsButtonType> &&type_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -790689618;

  static object_ptr<inlineQueryResultsButton> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Represents type of button in results of inline query.
 */
class InlineQueryResultsButtonType: public Object {
 public:
  static jclass Class;

  static object_ptr<InlineQueryResultsButtonType> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes the button that opens a private chat with the bot and sends a start message to the bot with the given parameter.
 */
class inlineQueryResultsButtonTypeStartBot final : public InlineQueryResultsButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The parameter for the bot start message.
  string parameter_;
  static jfieldID parameter_fieldID;

  /**
   * Describes the button that opens a private chat with the bot and sends a start message to the bot with the given parameter.
   */
  inlineQueryResultsButtonTypeStartBot();

  /**
   * Describes the button that opens a private chat with the bot and sends a start message to the bot with the given parameter.
   *
   * \param[in] parameter_ The parameter for the bot start message.
   */
  explicit inlineQueryResultsButtonTypeStartBot(string const &parameter_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -23400235;

  static object_ptr<InlineQueryResultsButtonType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Describes the button that opens a Web App by calling getWebAppUrl.
 */
class inlineQueryResultsButtonTypeWebApp final : public InlineQueryResultsButtonType {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// An HTTP URL to pass to getWebAppUrl.
  string url_;
  static jfieldID url_fieldID;

  /**
   * Describes the button that opens a Web App by calling getWebAppUrl.
   */
  inlineQueryResultsButtonTypeWebApp();

  /**
   * Describes the button that opens a Web App by calling getWebAppUrl.
   *
   * \param[in] url_ An HTTP URL to pass to getWebAppUrl.
   */
  explicit inlineQueryResultsButtonTypeWebApp(string const &url_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1197382814;

  static object_ptr<InlineQueryResultsButtonType> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class InputFile;

/**
 * This class is an abstract base class.
 * Contains information about background to set.
 */
class InputBackground: public Object {
 public:
  static jclass Class;

  static object_ptr<InputBackground> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A background from a local file.
 */
class inputBackgroundLocal final : public InputBackground {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Background file to use. Only inputFileLocal and inputFileGenerated are supported. The file must be in JPEG format for wallpapers and in PNG format for patterns.
  object_ptr<InputFile> background_;
  static jfieldID background_fieldID;

  /**
   * A background from a local file.
   */
  inputBackgroundLocal();

  /**
   * A background from a local file.
   *
   * \param[in] background_ Background file to use. Only inputFileLocal and inputFileGenerated are supported. The file must be in JPEG format for wallpapers and in PNG format for patterns.
   */
  explicit inputBackgroundLocal(object_ptr<InputFile> &&background_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1747094364;

  static object_ptr<InputBackground> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A background from the server.
 */
class inputBackgroundRemote final : public InputBackground {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// The background identifier.
  int64 background_id_;
  static jfieldID background_id_fieldID;

  /**
   * A background from the server.
   */
  inputBackgroundRemote();

  /**
   * A background from the server.
   *
   * \param[in] background_id_ The background identifier.
   */
  explicit inputBackgroundRemote(int64 background_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -274976231;

  static object_ptr<InputBackground> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A background previously set in the chat; for chat backgrounds only.
 */
class inputBackgroundPrevious final : public InputBackground {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the message with the background.
  int53 message_id_;
  static jfieldID message_id_fieldID;

  /**
   * A background previously set in the chat; for chat backgrounds only.
   */
  inputBackgroundPrevious();

  /**
   * A background previously set in the chat; for chat backgrounds only.
   *
   * \param[in] message_id_ Identifier of the message with the background.
   */
  explicit inputBackgroundPrevious(int53 message_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -351905954;

  static object_ptr<InputBackground> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class formattedText;

/**
 * Describes a business chat link to create or edit.
 */
class inputBusinessChatLink final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Message draft text that will be added to the input field.
  object_ptr<formattedText> text_;
  static jfieldID text_fieldID;
  /// Link title.
  string title_;
  static jfieldID title_fieldID;

  /**
   * Describes a business chat link to create or edit.
   */
  inputBusinessChatLink();

  /**
   * Describes a business chat link to create or edit.
   *
   * \param[in] text_ Message draft text that will be added to the input field.
   * \param[in] title_ Link title.
   */
  inputBusinessChatLink(object_ptr<formattedText> &&text_, string const &title_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 237858296;

  static object_ptr<inputBusinessChatLink> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class InputFile;

/**
 * Describes settings for a business account start page to set.
 */
class inputBusinessStartPage final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Title text of the start page; 0-getOption(&quot;business_start_page_title_length_max&quot;) characters.
  string title_;
  static jfieldID title_fieldID;
  /// Message text of the start page; 0-getOption(&quot;business_start_page_message_length_max&quot;) characters.
  string message_;
  static jfieldID message_fieldID;
  /// Greeting sticker of the start page; pass null if none. The sticker must belong to a sticker set and must not be a custom emoji.
  object_ptr<InputFile> sticker_;
  static jfieldID sticker_fieldID;

  /**
   * Describes settings for a business account start page to set.
   */
  inputBusinessStartPage();

  /**
   * Describes settings for a business account start page to set.
   *
   * \param[in] title_ Title text of the start page; 0-getOption(&quot;business_start_page_title_length_max&quot;) characters.
   * \param[in] message_ Message text of the start page; 0-getOption(&quot;business_start_page_message_length_max&quot;) characters.
   * \param[in] sticker_ Greeting sticker of the start page; pass null if none. The sticker must belong to a sticker set and must not be a custom emoji.
   */
  inputBusinessStartPage(string const &title_, string const &message_, object_ptr<InputFile> &&sticker_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -327383072;

  static object_ptr<inputBusinessStartPage> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class InputFile;

class chatPhotoSticker;

/**
 * This class is an abstract base class.
 * Describes a photo to be set as a user profile or chat photo.
 */
class InputChatPhoto: public Object {
 public:
  static jclass Class;

  static object_ptr<InputChatPhoto> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A previously used profile photo of the current user.
 */
class inputChatPhotoPrevious final : public InputChatPhoto {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the current user's profile photo to reuse.
  int64 chat_photo_id_;
  static jfieldID chat_photo_id_fieldID;

  /**
   * A previously used profile photo of the current user.
   */
  inputChatPhotoPrevious();

  /**
   * A previously used profile photo of the current user.
   *
   * \param[in] chat_photo_id_ Identifier of the current user's profile photo to reuse.
   */
  explicit inputChatPhotoPrevious(int64 chat_photo_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 23128529;

  static object_ptr<InputChatPhoto> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A static photo in JPEG format.
 */
class inputChatPhotoStatic final : public InputChatPhoto {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Photo to be set as profile photo. Only inputFileLocal and inputFileGenerated are allowed.
  object_ptr<InputFile> photo_;
  static jfieldID photo_fieldID;

  /**
   * A static photo in JPEG format.
   */
  inputChatPhotoStatic();

  /**
   * A static photo in JPEG format.
   *
   * \param[in] photo_ Photo to be set as profile photo. Only inputFileLocal and inputFileGenerated are allowed.
   */
  explicit inputChatPhotoStatic(object_ptr<InputFile> &&photo_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1979179699;

  static object_ptr<InputChatPhoto> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * An animation in MPEG4 format; must be square, at most 10 seconds long, have width between 160 and 1280 and be at most 2MB in size.
 */
class inputChatPhotoAnimation final : public InputChatPhoto {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Animation to be set as profile photo. Only inputFileLocal and inputFileGenerated are allowed.
  object_ptr<InputFile> animation_;
  static jfieldID animation_fieldID;
  /// Timestamp of the frame, which will be used as static chat photo.
  double main_frame_timestamp_;
  static jfieldID main_frame_timestamp_fieldID;

  /**
   * An animation in MPEG4 format; must be square, at most 10 seconds long, have width between 160 and 1280 and be at most 2MB in size.
   */
  inputChatPhotoAnimation();

  /**
   * An animation in MPEG4 format; must be square, at most 10 seconds long, have width between 160 and 1280 and be at most 2MB in size.
   *
   * \param[in] animation_ Animation to be set as profile photo. Only inputFileLocal and inputFileGenerated are allowed.
   * \param[in] main_frame_timestamp_ Timestamp of the frame, which will be used as static chat photo.
   */
  inputChatPhotoAnimation(object_ptr<InputFile> &&animation_, double main_frame_timestamp_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 90846242;

  static object_ptr<InputChatPhoto> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * A sticker on a custom background.
 */
class inputChatPhotoSticker final : public InputChatPhoto {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Information about the sticker.
  object_ptr<chatPhotoSticker> sticker_;
  static jfieldID sticker_fieldID;

  /**
   * A sticker on a custom background.
   */
  inputChatPhotoSticker();

  /**
   * A sticker on a custom background.
   *
   * \param[in] sticker_ Information about the sticker.
   */
  explicit inputChatPhotoSticker(object_ptr<chatPhotoSticker> &&sticker_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1315861341;

  static object_ptr<InputChatPhoto> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class formattedText;

class inputChecklistTask;

/**
 * Describes a checklist to be sent.
 */
class inputChecklist final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Title of the checklist; 1-getOption(&quot;checklist_title_length_max&quot;) characters. May contain only Bold, Italic, Underline, Strikethrough, Spoiler, and CustomEmoji entities.
  object_ptr<formattedText> title_;
  static jfieldID title_fieldID;
  /// List of tasks in the checklist; 1-getOption(&quot;checklist_task_count_max&quot;) tasks.
  array<object_ptr<inputChecklistTask>> tasks_;
  static jfieldID tasks_fieldID;
  /// True, if other users can add tasks to the list.
  bool others_can_add_tasks_;
  static jfieldID others_can_add_tasks_fieldID;
  /// True, if other users can mark tasks as done or not done.
  bool others_can_mark_tasks_as_done_;
  static jfieldID others_can_mark_tasks_as_done_fieldID;

  /**
   * Describes a checklist to be sent.
   */
  inputChecklist();

  /**
   * Describes a checklist to be sent.
   *
   * \param[in] title_ Title of the checklist; 1-getOption(&quot;checklist_title_length_max&quot;) characters. May contain only Bold, Italic, Underline, Strikethrough, Spoiler, and CustomEmoji entities.
   * \param[in] tasks_ List of tasks in the checklist; 1-getOption(&quot;checklist_task_count_max&quot;) tasks.
   * \param[in] others_can_add_tasks_ True, if other users can add tasks to the list.
   * \param[in] others_can_mark_tasks_as_done_ True, if other users can mark tasks as done or not done.
   */
  inputChecklist(object_ptr<formattedText> &&title_, array<object_ptr<inputChecklistTask>> &&tasks_, bool others_can_add_tasks_, bool others_can_mark_tasks_as_done_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -145125739;

  static object_ptr<inputChecklist> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

class formattedText;

/**
 * Describes a task in a checklist to be sent.
 */
class inputChecklistTask final : public Object {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Unique identifier of the task; must be positive.
  int32 id_;
  static jfieldID id_fieldID;
  /// Text of the task; 1-getOption(&quot;checklist_task_text_length_max&quot;) characters without line feeds. May contain only Bold, Italic, Underline, Strikethrough, Spoiler, and CustomEmoji entities.
  object_ptr<formattedText> text_;
  static jfieldID text_fieldID;

  /**
   * Describes a task in a checklist to be sent.
   */
  inputChecklistTask();

  /**
   * Describes a task in a checklist to be sent.
   *
   * \param[in] id_ Unique identifier of the task; must be positive.
   * \param[in] text_ Text of the task; 1-getOption(&quot;checklist_task_text_length_max&quot;) characters without line feeds. May contain only Bold, Italic, Underline, Strikethrough, Spoiler, and CustomEmoji entities.
   */
  inputChecklistTask(int32 id_, object_ptr<formattedText> &&text_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 1633462225;

  static object_ptr<inputChecklistTask> fetch(JNIEnv *env, jobject &p);

  void store(JNIEnv *env, jobject &s) const final;

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base class.
 * Contains information about the payment method chosen by the user.
 */
class InputCredentials: public Object {
 public:
  static jclass Class;

  static object_ptr<InputCredentials> fetch(JNIEnv *env, jobject &p);

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Applies if a user chooses some previously saved payment credentials. To use their previously saved credentials, the user must have a valid temporary password.
 */
class inputCredentialsSaved final : public InputCredentials {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// Identifier of the saved credentials.
  string saved_credentials_id_;
  static jfieldID saved_credentials_id_fieldID;

  /**
   * Applies if a user chooses some previously saved payment credentials. To use their previously saved credentials, the user must have a valid temporary password.
   */
  inputCredentialsSaved();

  /**
   * Applies if a user chooses some previously saved payment credentials. To use their previously saved credentials, the user must have a valid temporary password.
   *
   * \param[in] saved_credentials_id_ Identifier of the saved credentials.
   */
  explicit inputCredentialsSaved(string const &saved_credentials_id_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -2034385364;

  static object_ptr<InputCredentials> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Applies if a user enters new credentials on a payment provider website.
 */
class inputCredentialsNew final : public InputCredentials {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// JSON-encoded data with the credential identifier from the payment provider.
  string data_;
  static jfieldID data_fieldID;
  /// True, if the credential identifier can be saved on the server side.
  bool allow_save_;
  static jfieldID allow_save_fieldID;

  /**
   * Applies if a user enters new credentials on a payment provider website.
   */
  inputCredentialsNew();

  /**
   * Applies if a user enters new credentials on a payment provider website.
   *
   * \param[in] data_ JSON-encoded data with the credential identifier from the payment provider.
   * \param[in] allow_save_ True, if the credential identifier can be saved on the server side.
   */
  inputCredentialsNew(string const &data_, bool allow_save_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -829689558;

  static object_ptr<InputCredentials> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Applies if a user enters new credentials using Apple Pay.
 */
class inputCredentialsApplePay final : public InputCredentials {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// JSON-encoded data with the credential identifier.
  string data_;
  static jfieldID data_fieldID;

  /**
   * Applies if a user enters new credentials using Apple Pay.
   */
  inputCredentialsApplePay();

  /**
   * Applies if a user enters new credentials using Apple Pay.
   *
   * \param[in] data_ JSON-encoded data with the credential identifier.
   */
  explicit inputCredentialsApplePay(string const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = -1246570799;

  static object_ptr<InputCredentials> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * Applies if a user enters new credentials using Google Pay.
 */
class inputCredentialsGooglePay final : public InputCredentials {
  /**
   * Returns identifier uniquely determining a type of the object.
   * \return this->ID.
   */
  std::int32_t get_id() const final {
    return ID;
  }

 public:
  static jclass Class;
  /// JSON-encoded data with the credential identifier.
  string data_;
  static jfieldID data_fieldID;

  /**
   * Applies if a user enters new credentials using Google Pay.
   */
  inputCredentialsGooglePay();

  /**
   * Applies if a user enters new credentials using Google Pay.
   *
   * \param[in] data_ JSON-encoded data with the credential identifier.
   */
  explicit inputCredentialsGooglePay(string const &data_);

  /// Identifier uniquely determining a type of the object.
  static const std::int32_t ID = 844384100;

  static object_ptr<InputCredentials> fetch(JNIEnv *env, jobject &p);

  /**
   * Helper function for to_string method. Appends string representation of the object to the storer.
   * \param[in] s Storer to which object string representation will be appended.
   * \param[in] field_name Object field_name if applicable.
   */
  void store(TlStorerToString &s, const char *field_name) const final;

  static void init_jni_vars(JNIEnv *env);
};

/**
 * This class is an abstract base cl