//
// Copyright Aliaksei Levin (levlam@telegram.org), Arseny Smirnov (arseny30@gmail.com) 2014-2021
//
// Distributed under the Boost Software License, Version 1.0. (See accompanying
// file LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)
//
#include "td/telegram/MessagesManager.h"

#include "td/telegram/AuthManager.h"
#include "td/telegram/ChatId.h"
#include "td/telegram/ConfigShared.h"
#include "td/telegram/ContactsManager.h"
#include "td/telegram/Dependencies.h"
#include "td/telegram/DialogDb.h"
#include "td/telegram/DialogFilter.h"
#include "td/telegram/DialogFilter.hpp"
#include "td/telegram/DialogInviteLink.h"
#include "td/telegram/DialogLocation.h"
#include "td/telegram/DraftMessage.h"
#include "td/telegram/DraftMessage.hpp"
#include "td/telegram/FileReferenceManager.h"
#include "td/telegram/files/FileId.hpp"
#include "td/telegram/files/FileLocation.h"
#include "td/telegram/files/FileManager.h"
#include "td/telegram/files/FileType.h"
#include "td/telegram/Global.h"
#include "td/telegram/GroupCallManager.h"
#include "td/telegram/GroupCallParticipant.h"
#include "td/telegram/InlineQueriesManager.h"
#include "td/telegram/InputMessageText.h"
#include "td/telegram/LinkManager.h"
#include "td/telegram/Location.h"
#include "td/telegram/logevent/LogEvent.h"
#include "td/telegram/MessageContent.h"
#include "td/telegram/MessageEntity.h"
#include "td/telegram/MessageEntity.hpp"
#include "td/telegram/MessagesDb.h"
#include "td/telegram/misc.h"
#include "td/telegram/net/DcId.h"
#include "td/telegram/net/NetActor.h"
#include "td/telegram/net/NetQuery.h"
#include "td/telegram/NotificationGroupType.h"
#include "td/telegram/NotificationManager.h"
#include "td/telegram/NotificationSettings.hpp"
#include "td/telegram/NotificationType.h"
#include "td/telegram/ReplyMarkup.h"
#include "td/telegram/ReplyMarkup.hpp"
#include "td/telegram/SecretChatsManager.h"
#include "td/telegram/SequenceDispatcher.h"
#include "td/telegram/Td.h"
#include "td/telegram/TdDb.h"
#include "td/telegram/TdParameters.h"
#include "td/telegram/TopDialogCategory.h"
#include "td/telegram/UpdatesManager.h"
#include "td/telegram/Version.h"
#include "td/telegram/WebPageId.h"

#include "td/db/binlog/BinlogEvent.h"
#include "td/db/binlog/BinlogHelper.h"
#include "td/db/SqliteKeyValue.h"
#include "td/db/SqliteKeyValueAsync.h"

#include "td/actor/PromiseFuture.h"
#include "td/actor/SleepActor.h"

#include "td/utils/algorithm.h"
#include "td/utils/format.h"
#include "td/utils/misc.h"
#include "td/utils/PathView.h"
#include "td/utils/Random.h"
#include "td/utils/Slice.h"
#include "td/utils/SliceBuilder.h"
#include "td/utils/Time.h"
#include "td/utils/tl_helpers.h"
#include "td/utils/utf8.h"

#include <algorithm>
#include <cstring>
#include <limits>
#include <tuple>
#include <type_traits>
#include <unordered_map>
#include <unordered_set>
#include <utility>

namespace td {

class GetDialogFiltersQuery final : public Td::ResultHandler {
  Promise<vector<tl_object_ptr<telegram_api::dialogFilter>>> promise_;

 public:
  explicit GetDialogFiltersQuery(Promise<vector<tl_object_ptr<telegram_api::dialogFilter>>> &&promise)
      : promise_(std::move(promise)) {
  }

  void send() {
    send_query(G()->net_query_creator().create(telegram_api::messages_getDialogFilters()));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getDialogFilters>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    promise_.set_value(result_ptr.move_as_ok());
  }

  void on_error(uint64 id, Status status) final {
    promise_.set_error(std::move(status));
  }
};

class UpdateDialogFilterQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit UpdateDialogFilterQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogFilterId dialog_filter_id, tl_object_ptr<telegram_api::dialogFilter> filter) {
    int32 flags = 0;
    if (filter != nullptr) {
      flags |= telegram_api::messages_updateDialogFilter::FILTER_MASK;
    }
    send_query(G()->net_query_creator().create(
        telegram_api::messages_updateDialogFilter(flags, dialog_filter_id.get(), std::move(filter))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_updateDialogFilter>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    LOG(INFO) << "Receive result for UpdateDialogFilterQuery: " << result_ptr.ok();
    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    LOG(ERROR) << "Receive error for UpdateDialogFilterQuery: " << status;
    promise_.set_error(std::move(status));
  }
};

class UpdateDialogFiltersOrderQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit UpdateDialogFiltersOrderQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(const vector<DialogFilterId> &dialog_filter_ids) {
    send_query(G()->net_query_creator().create(telegram_api::messages_updateDialogFiltersOrder(
        transform(dialog_filter_ids, [](auto dialog_filter_id) { return dialog_filter_id.get(); }))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_updateDialogFiltersOrder>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    LOG(INFO) << "Receive result for UpdateDialogFiltersOrderQuery: " << result_ptr.ok();
    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    promise_.set_error(std::move(status));
  }
};

class GetSuggestedDialogFiltersQuery final : public Td::ResultHandler {
  Promise<vector<tl_object_ptr<telegram_api::dialogFilterSuggested>>> promise_;

 public:
  explicit GetSuggestedDialogFiltersQuery(Promise<vector<tl_object_ptr<telegram_api::dialogFilterSuggested>>> &&promise)
      : promise_(std::move(promise)) {
  }

  void send() {
    send_query(G()->net_query_creator().create(telegram_api::messages_getSuggestedDialogFilters()));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getSuggestedDialogFilters>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    promise_.set_value(result_ptr.move_as_ok());
  }

  void on_error(uint64 id, Status status) final {
    promise_.set_error(std::move(status));
  }
};

class GetOnlinesQuery final : public Td::ResultHandler {
  DialogId dialog_id_;

 public:
  void send(DialogId dialog_id) {
    dialog_id_ = dialog_id;
    CHECK(dialog_id.get_type() == DialogType::Channel);
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return on_error(0, Status::Error(400, "Can't access the chat"));
    }

    send_query(G()->net_query_creator().create(telegram_api::messages_getOnlines(std::move(input_peer))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getOnlines>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto result = result_ptr.move_as_ok();
    td->messages_manager_->on_update_dialog_online_member_count(dialog_id_, result->onlines_, true);
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetOnlinesQuery");
    td->messages_manager_->on_update_dialog_online_member_count(dialog_id_, 0, true);
  }
};

class GetAllDraftsQuery final : public Td::ResultHandler {
 public:
  void send() {
    send_query(G()->net_query_creator().create(telegram_api::messages_getAllDrafts()));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getAllDrafts>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for GetAllDraftsQuery: " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr), Promise<Unit>());
  }

  void on_error(uint64 id, Status status) final {
    if (!G()->is_expected_error(status)) {
      LOG(ERROR) << "Receive error for GetAllDraftsQuery: " << status;
    }
    status.ignore();
  }
};

class GetDialogQuery final : public Td::ResultHandler {
  DialogId dialog_id_;

 public:
  void send(DialogId dialog_id) {
    dialog_id_ = dialog_id;
    send_query(G()->net_query_creator().create(telegram_api::messages_getPeerDialogs(
        td->messages_manager_->get_input_dialog_peers({dialog_id}, AccessRights::Read))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getPeerDialogs>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto result = result_ptr.move_as_ok();
    LOG(INFO) << "Receive chat: " << to_string(result);

    td->contacts_manager_->on_get_users(std::move(result->users_), "GetDialogQuery");
    td->contacts_manager_->on_get_chats(std::move(result->chats_), "GetDialogQuery");
    td->messages_manager_->on_get_dialogs(
        FolderId(), std::move(result->dialogs_), -1, std::move(result->messages_),
        PromiseCreator::lambda([actor_id = td->messages_manager_actor_.get(), dialog_id = dialog_id_](Result<> result) {
          send_closure(actor_id, &MessagesManager::on_get_dialog_query_finished, dialog_id,
                       result.is_error() ? result.move_as_error() : Status::OK());
        }));
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetDialogQuery");
    td->messages_manager_->on_get_dialog_query_finished(dialog_id_, std::move(status));
  }
};

class GetDialogsQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit GetDialogsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(vector<InputDialogId> input_dialog_ids) {
    LOG(INFO) << "Send GetDialogsQuery to get " << input_dialog_ids;
    CHECK(!input_dialog_ids.empty());
    CHECK(input_dialog_ids.size() <= 100);
    auto input_dialog_peers = InputDialogId::get_input_dialog_peers(input_dialog_ids);
    CHECK(input_dialog_peers.size() == input_dialog_ids.size());
    send_query(G()->net_query_creator().create(telegram_api::messages_getPeerDialogs(std::move(input_dialog_peers))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getPeerDialogs>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto result = result_ptr.move_as_ok();
    LOG(INFO) << "Receive chats: " << to_string(result);

    td->contacts_manager_->on_get_users(std::move(result->users_), "GetDialogsQuery");
    td->contacts_manager_->on_get_chats(std::move(result->chats_), "GetDialogsQuery");
    td->messages_manager_->on_get_dialogs(FolderId(), std::move(result->dialogs_), -1, std::move(result->messages_),
                                          std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    promise_.set_error(std::move(status));
  }
};

class GetPinnedDialogsActor final : public NetActorOnce {
  FolderId folder_id_;
  Promise<Unit> promise_;

 public:
  explicit GetPinnedDialogsActor(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  NetQueryRef send(FolderId folder_id, uint64 sequence_id) {
    folder_id_ = folder_id;
    auto query = G()->net_query_creator().create(telegram_api::messages_getPinnedDialogs(folder_id.get()));
    auto result = query.get_weak();
    send_closure(td->messages_manager_->sequence_dispatcher_, &MultiSequenceDispatcher::send_with_callback,
                 std::move(query), actor_shared(this), sequence_id);
    return result;
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getPinnedDialogs>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto result = result_ptr.move_as_ok();
    LOG(INFO) << "Receive pinned chats in " << folder_id_ << ": " << to_string(result);

    td->contacts_manager_->on_get_users(std::move(result->users_), "GetPinnedDialogsActor");
    td->contacts_manager_->on_get_chats(std::move(result->chats_), "GetPinnedDialogsActor");
    td->messages_manager_->on_get_dialogs(folder_id_, std::move(result->dialogs_), -2, std::move(result->messages_),
                                          std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    promise_.set_error(std::move(status));
  }
};

class GetDialogUnreadMarksQuery final : public Td::ResultHandler {
 public:
  void send() {
    send_query(G()->net_query_creator().create(telegram_api::messages_getDialogUnreadMarks()));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getDialogUnreadMarks>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto results = result_ptr.move_as_ok();
    for (auto &result : results) {
      td->messages_manager_->on_update_dialog_is_marked_as_unread(DialogId(result), true);
    }

    G()->td_db()->get_binlog_pmc()->set("fetched_marks_as_unread", "1");
  }

  void on_error(uint64 id, Status status) final {
    if (!G()->is_expected_error(status)) {
      LOG(ERROR) << "Receive error for GetDialogUnreadMarksQuery: " << status;
    }
    status.ignore();
  }
};

class GetDiscussionMessageQuery final : public Td::ResultHandler {
  Promise<MessageThreadInfo> promise_;
  DialogId dialog_id_;
  MessageId message_id_;
  DialogId expected_dialog_id_;
  MessageId expected_message_id_;

 public:
  explicit GetDiscussionMessageQuery(Promise<MessageThreadInfo> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId message_id, DialogId expected_dialog_id, MessageId expected_message_id) {
    dialog_id_ = dialog_id;
    message_id_ = message_id;
    expected_dialog_id_ = expected_dialog_id;
    expected_message_id_ = expected_message_id;
    CHECK(expected_dialog_id_.is_valid());
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(
        telegram_api::messages_getDiscussionMessage(std::move(input_peer), message_id.get_server_message_id().get())));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getDiscussionMessage>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    td->messages_manager_->process_discussion_message(result_ptr.move_as_ok(), dialog_id_, message_id_,
                                                      expected_dialog_id_, expected_message_id_, std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    if (expected_dialog_id_ == dialog_id_) {
      td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetDiscussionMessageQuery");
    }
    promise_.set_error(std::move(status));
  }
};

class GetMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit GetMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(vector<tl_object_ptr<telegram_api::InputMessage>> &&message_ids) {
    send_query(G()->net_query_creator().create(telegram_api::messages_getMessages(std::move(message_ids))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto info = td->messages_manager_->get_messages_info(result_ptr.move_as_ok(), "GetMessagesQuery");
    LOG_IF(ERROR, info.is_channel_messages) << "Receive channel messages in GetMessagesQuery";
    td->messages_manager_->on_get_messages(std::move(info.messages), info.is_channel_messages, false,
                                           std::move(promise_), "GetMessagesQuery");
  }

  void on_error(uint64 id, Status status) final {
    if (status.message() == "MESSAGE_IDS_EMPTY") {
      promise_.set_value(Unit());
      return;
    }
    promise_.set_error(std::move(status));
  }
};

class GetChannelMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;
  MessageId last_new_message_id_;

 public:
  explicit GetChannelMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, tl_object_ptr<telegram_api::InputChannel> &&input_channel,
            vector<tl_object_ptr<telegram_api::InputMessage>> &&message_ids, MessageId last_new_message_id) {
    channel_id_ = channel_id;
    last_new_message_id_ = last_new_message_id;
    CHECK(input_channel != nullptr);
    send_query(G()->net_query_creator().create(
        telegram_api::channels_getMessages(std::move(input_channel), std::move(message_ids))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::channels_getMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto info = td->messages_manager_->get_messages_info(result_ptr.move_as_ok(), "GetChannelMessagesQuery");
    LOG_IF(ERROR, !info.is_channel_messages) << "Receive ordinary messages in GetChannelMessagesQuery";
    // messages with invalid big identifiers can be received as messageEmpty
    // bots can receive messageEmpty because of their privacy mode
    if (last_new_message_id_.is_valid() && !td->auth_manager_->is_bot()) {
      vector<MessageId> empty_message_ids;
      for (auto &message : info.messages) {
        if (message->get_id() == telegram_api::messageEmpty::ID) {
          auto message_id = MessagesManager::get_message_id(message, false);
          if (message_id.is_valid() && message_id <= last_new_message_id_) {
            empty_message_ids.push_back(message_id);
          }
        }
      }
      td->messages_manager_->on_get_empty_messages(DialogId(channel_id_), empty_message_ids);
    }
    td->messages_manager_->get_channel_difference_if_needed(
        DialogId(channel_id_), std::move(info),
        PromiseCreator::lambda([actor_id = td->messages_manager_actor_.get(),
                                promise = std::move(promise_)](Result<MessagesManager::MessagesInfo> &&result) mutable {
          if (result.is_error()) {
            promise.set_error(result.move_as_error());
          } else {
            auto info = result.move_as_ok();
            send_closure(actor_id, &MessagesManager::on_get_messages, std::move(info.messages),
                         info.is_channel_messages, false, std::move(promise), "GetChannelMessagesQuery");
          }
        }));
  }

  void on_error(uint64 id, Status status) final {
    if (status.message() == "MESSAGE_IDS_EMPTY") {
      promise_.set_value(Unit());
      return;
    }
    td->contacts_manager_->on_get_channel_error(channel_id_, status, "GetChannelMessagesQuery");
    promise_.set_error(std::move(status));
  }
};

class GetScheduledMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit GetScheduledMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, tl_object_ptr<telegram_api::InputPeer> &&input_peer, vector<int32> &&message_ids) {
    dialog_id_ = dialog_id;
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(
        telegram_api::messages_getScheduledMessages(std::move(input_peer), std::move(message_ids))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getScheduledMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto info = td->messages_manager_->get_messages_info(result_ptr.move_as_ok(), "GetScheduledMessagesQuery");
    LOG_IF(ERROR, info.is_channel_messages != (dialog_id_.get_type() == DialogType::Channel))
        << "Receive wrong messages constructor in GetScheduledMessagesQuery";
    td->messages_manager_->on_get_messages(std::move(info.messages), info.is_channel_messages, true,
                                           std::move(promise_), "GetScheduledMessagesQuery");
  }

  void on_error(uint64 id, Status status) final {
    if (status.message() == "MESSAGE_IDS_EMPTY") {
      promise_.set_value(Unit());
      return;
    }
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetScheduledMessagesQuery");
    promise_.set_error(std::move(status));
  }
};

class UpdateDialogPinnedMessageQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit UpdateDialogPinnedMessageQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId message_id, bool is_unpin, bool disable_notification, bool only_for_self) {
    dialog_id_ = dialog_id;
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      LOG(INFO) << "Can't update pinned message in " << dialog_id;
      return on_error(0, Status::Error(400, "Can't update pinned message"));
    }

    int32 flags = 0;
    if (disable_notification) {
      flags |= telegram_api::messages_updatePinnedMessage::SILENT_MASK;
    }
    if (is_unpin) {
      flags |= telegram_api::messages_updatePinnedMessage::UNPIN_MASK;
    }
    if (only_for_self) {
      flags |= telegram_api::messages_updatePinnedMessage::PM_ONESIDE_MASK;
    }
    send_query(G()->net_query_creator().create(
        telegram_api::messages_updatePinnedMessage(flags, false /*ignored*/, false /*ignored*/, false /*ignored*/,
                                                   std::move(input_peer), message_id.get_server_message_id().get())));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_updatePinnedMessage>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for UpdateDialogPinnedMessageQuery: " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "UpdateDialogPinnedMessageQuery");
    promise_.set_error(std::move(status));
  }
};

class UnpinAllMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

  void send_request() {
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id_, AccessRights::Write);
    if (input_peer == nullptr) {
      LOG(INFO) << "Can't unpin all messages in " << dialog_id_;
      return on_error(0, Status::Error(400, "Can't unpin all messages"));
    }

    send_query(G()->net_query_creator().create(telegram_api::messages_unpinAllMessages(std::move(input_peer))));
  }

 public:
  explicit UnpinAllMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id) {
    dialog_id_ = dialog_id;

    send_request();
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_unpinAllMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto affected_history = result_ptr.move_as_ok();
    CHECK(affected_history->get_id() == telegram_api::messages_affectedHistory::ID);

    if (affected_history->pts_count_ > 0) {
      affected_history->pts_count_ = 0;  // force receiving real updates from the server
      auto promise = affected_history->offset_ > 0 ? Promise<Unit>() : std::move(promise_);
      if (dialog_id_.get_type() == DialogType::Channel) {
        td->messages_manager_->add_pending_channel_update(dialog_id_, make_tl_object<dummyUpdate>(),
                                                          affected_history->pts_, affected_history->pts_count_,
                                                          std::move(promise), "unpin all messages");
      } else {
        td->updates_manager_->add_pending_pts_update(make_tl_object<dummyUpdate>(), affected_history->pts_,
                                                     affected_history->pts_count_, Time::now(), std::move(promise),
                                                     "unpin all messages");
      }
    } else if (affected_history->offset_ <= 0) {
      promise_.set_value(Unit());
    }

    if (affected_history->offset_ > 0) {
      send_request();
      return;
    }
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "UnpinAllMessagesQuery");
    promise_.set_error(std::move(status));
  }
};

class GetMessageReadParticipantsQuery final : public Td::ResultHandler {
  Promise<vector<UserId>> promise_;
  DialogId dialog_id_;

 public:
  explicit GetMessageReadParticipantsQuery(Promise<vector<UserId>> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId message_id) {
    dialog_id_ = dialog_id;
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(telegram_api::messages_getMessageReadParticipants(
        std::move(input_peer), message_id.get_server_message_id().get())));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getMessageReadParticipants>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    promise_.set_value(UserId::get_user_ids(result_ptr.ok()));
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetMessageReadParticipantsQuery");
    promise_.set_error(std::move(status));
  }
};

class ExportChannelMessageLinkQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;
  MessageId message_id_;
  bool for_group_ = false;
  bool ignore_result_ = false;

 public:
  explicit ExportChannelMessageLinkQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, MessageId message_id, bool for_group, bool ignore_result) {
    channel_id_ = channel_id;
    message_id_ = message_id;
    for_group_ = for_group;
    ignore_result_ = ignore_result;
    auto input_channel = td->contacts_manager_->get_input_channel(channel_id);
    if (input_channel == nullptr) {
      return on_error(0, Status::Error(400, "Can't access the chat"));
    }
    int32 flags = 0;
    if (for_group) {
      flags |= telegram_api::channels_exportMessageLink::GROUPED_MASK;
    }
    send_query(G()->net_query_creator().create(
        telegram_api::channels_exportMessageLink(flags, false /*ignored*/, false /*ignored*/, std::move(input_channel),
                                                 message_id.get_server_message_id().get())));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::channels_exportMessageLink>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(DEBUG) << "Receive result for ExportChannelMessageLinkQuery: " << to_string(ptr);
    if (!ignore_result_) {
      td->messages_manager_->on_get_public_message_link({DialogId(channel_id_), message_id_}, for_group_,
                                                        std::move(ptr->link_), std::move(ptr->html_));
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    if (!ignore_result_) {
      td->contacts_manager_->on_get_channel_error(channel_id_, status, "ExportChannelMessageLinkQuery");
    }
    promise_.set_error(std::move(status));
  }
};

class GetDialogListActor final : public NetActorOnce {
  FolderId folder_id_;
  Promise<Unit> promise_;

 public:
  explicit GetDialogListActor(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(FolderId folder_id, int32 offset_date, ServerMessageId offset_message_id, DialogId offset_dialog_id,
            int32 limit, uint64 sequence_id) {
    folder_id_ = folder_id;
    auto input_peer = MessagesManager::get_input_peer_force(offset_dialog_id);
    CHECK(input_peer != nullptr);

    int32 flags =
        telegram_api::messages_getDialogs::EXCLUDE_PINNED_MASK | telegram_api::messages_getDialogs::FOLDER_ID_MASK;
    auto query = G()->net_query_creator().create(
        telegram_api::messages_getDialogs(flags, false /*ignored*/, folder_id.get(), offset_date,
                                          offset_message_id.get(), std::move(input_peer), limit, 0));
    send_closure(td->messages_manager_->sequence_dispatcher_, &MultiSequenceDispatcher::send_with_callback,
                 std::move(query), actor_shared(this), sequence_id);
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getDialogs>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive chats from chat list of " << folder_id_ << ": " << to_string(ptr);
    switch (ptr->get_id()) {
      case telegram_api::messages_dialogs::ID: {
        auto dialogs = move_tl_object_as<telegram_api::messages_dialogs>(ptr);
        td->contacts_manager_->on_get_users(std::move(dialogs->users_), "GetDialogListActor");
        td->contacts_manager_->on_get_chats(std::move(dialogs->chats_), "GetDialogListActor");
        td->messages_manager_->on_get_dialogs(folder_id_, std::move(dialogs->dialogs_),
                                              narrow_cast<int32>(dialogs->dialogs_.size()),
                                              std::move(dialogs->messages_), std::move(promise_));
        break;
      }
      case telegram_api::messages_dialogsSlice::ID: {
        auto dialogs = move_tl_object_as<telegram_api::messages_dialogsSlice>(ptr);
        td->contacts_manager_->on_get_users(std::move(dialogs->users_), "GetDialogListActor");
        td->contacts_manager_->on_get_chats(std::move(dialogs->chats_), "GetDialogListActor");
        td->messages_manager_->on_get_dialogs(folder_id_, std::move(dialogs->dialogs_), max(dialogs->count_, 0),
                                              std::move(dialogs->messages_), std::move(promise_));
        break;
      }
      case telegram_api::messages_dialogsNotModified::ID:
        LOG(ERROR) << "Receive " << to_string(ptr);
        return on_error(id, Status::Error(500, "Receive wrong server response messages.dialogsNotModified"));
      default:
        UNREACHABLE();
    }
  }

  void on_error(uint64 id, Status status) final {
    promise_.set_error(std::move(status));
  }
};

class SearchPublicDialogsQuery final : public Td::ResultHandler {
  string query_;

 public:
  void send(const string &query) {
    query_ = query;
    send_query(G()->net_query_creator().create(telegram_api::contacts_search(query, 3 /* ignored server-side */)));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::contacts_search>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto dialogs = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SearchPublicDialogsQuery: " << to_string(dialogs);
    td->contacts_manager_->on_get_users(std::move(dialogs->users_), "SearchPublicDialogsQuery");
    td->contacts_manager_->on_get_chats(std::move(dialogs->chats_), "SearchPublicDialogsQuery");
    td->messages_manager_->on_get_public_dialogs_search_result(query_, std::move(dialogs->my_results_),
                                                               std::move(dialogs->results_));
  }

  void on_error(uint64 id, Status status) final {
    if (!G()->is_expected_error(status)) {
      LOG(ERROR) << "Receive error for SearchPublicDialogsQuery: " << status;
    }
    td->messages_manager_->on_failed_public_dialogs_search(query_, std::move(status));
  }
};

class GetCommonDialogsQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  UserId user_id_;
  int64 offset_chat_id_ = 0;

 public:
  explicit GetCommonDialogsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(UserId user_id, int64 offset_chat_id, int32 limit) {
    user_id_ = user_id;
    offset_chat_id_ = offset_chat_id;
    LOG(INFO) << "Get common dialogs with " << user_id << " from " << offset_chat_id << " with limit " << limit;

    auto input_user = td->contacts_manager_->get_input_user(user_id);
    CHECK(input_user != nullptr);

    send_query(G()->net_query_creator().create(
        telegram_api::messages_getCommonChats(std::move(input_user), offset_chat_id, limit)));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getCommonChats>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto chats_ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for GetCommonDialogsQuery: " << to_string(chats_ptr);
    switch (chats_ptr->get_id()) {
      case telegram_api::messages_chats::ID: {
        auto chats = move_tl_object_as<telegram_api::messages_chats>(chats_ptr);
        td->messages_manager_->on_get_common_dialogs(user_id_, offset_chat_id_, std::move(chats->chats_),
                                                     narrow_cast<int32>(chats->chats_.size()));
        break;
      }
      case telegram_api::messages_chatsSlice::ID: {
        auto chats = move_tl_object_as<telegram_api::messages_chatsSlice>(chats_ptr);
        td->messages_manager_->on_get_common_dialogs(user_id_, offset_chat_id_, std::move(chats->chats_),
                                                     chats->count_);
        break;
      }
      default:
        UNREACHABLE();
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    promise_.set_error(std::move(status));
  }
};

class GetBlockedDialogsQuery final : public Td::ResultHandler {
  Promise<td_api::object_ptr<td_api::messageSenders>> promise_;
  int32 offset_;
  int32 limit_;

 public:
  explicit GetBlockedDialogsQuery(Promise<td_api::object_ptr<td_api::messageSenders>> &&promise)
      : promise_(std::move(promise)) {
  }

  void send(int32 offset, int32 limit) {
    offset_ = offset;
    limit_ = limit;

    send_query(G()->net_query_creator().create(telegram_api::contacts_getBlocked(offset, limit)));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::contacts_getBlocked>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for GetBlockedDialogsQuery: " << to_string(ptr);

    switch (ptr->get_id()) {
      case telegram_api::contacts_blocked::ID: {
        auto blocked_peers = move_tl_object_as<telegram_api::contacts_blocked>(ptr);

        td->contacts_manager_->on_get_users(std::move(blocked_peers->users_), "GetBlockedDialogsQuery");
        td->contacts_manager_->on_get_chats(std::move(blocked_peers->chats_), "GetBlockedDialogsQuery");
        td->messages_manager_->on_get_blocked_dialogs(offset_, limit_,
                                                      narrow_cast<int32>(blocked_peers->blocked_.size()),
                                                      std::move(blocked_peers->blocked_), std::move(promise_));
        break;
      }
      case telegram_api::contacts_blockedSlice::ID: {
        auto blocked_peers = move_tl_object_as<telegram_api::contacts_blockedSlice>(ptr);

        td->contacts_manager_->on_get_users(std::move(blocked_peers->users_), "GetBlockedDialogsQuery");
        td->contacts_manager_->on_get_chats(std::move(blocked_peers->chats_), "GetBlockedDialogsQuery");
        td->messages_manager_->on_get_blocked_dialogs(offset_, limit_, blocked_peers->count_,
                                                      std::move(blocked_peers->blocked_), std::move(promise_));
        break;
      }
      default:
        UNREACHABLE();
    }
  }

  void on_error(uint64 id, Status status) final {
    promise_.set_error(std::move(status));
  }
};

class CreateChatQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  int64 random_id_;

 public:
  explicit CreateChatQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(vector<tl_object_ptr<telegram_api::InputUser>> &&input_users, const string &title, int64 random_id) {
    random_id_ = random_id;
    send_query(G()->net_query_creator().create(telegram_api::messages_createChat(std::move(input_users), title)));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_createChat>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for CreateChatQuery: " << to_string(ptr);
    td->messages_manager_->on_create_new_dialog_success(random_id_, std::move(ptr), DialogType::Chat,
                                                        std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_create_new_dialog_fail(random_id_, std::move(status), std::move(promise_));
  }
};

class CreateChannelQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  int64 random_id_;

 public:
  explicit CreateChannelQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(const string &title, bool is_megagroup, const string &about, const DialogLocation &location,
            bool for_import, int64 random_id) {
    int32 flags = 0;
    if (is_megagroup) {
      flags |= telegram_api::channels_createChannel::MEGAGROUP_MASK;
    } else {
      flags |= telegram_api::channels_createChannel::BROADCAST_MASK;
    }
    if (!location.empty()) {
      flags |= telegram_api::channels_createChannel::GEO_POINT_MASK;
    }
    if (for_import) {
      flags |= telegram_api::channels_createChannel::FOR_IMPORT_MASK;
    }

    random_id_ = random_id;
    send_query(G()->net_query_creator().create(
        telegram_api::channels_createChannel(flags, false /*ignored*/, false /*ignored*/, false /*ignored*/, title,
                                             about, location.get_input_geo_point(), location.get_address())));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::channels_createChannel>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for CreateChannelQuery: " << to_string(ptr);
    td->messages_manager_->on_create_new_dialog_success(random_id_, std::move(ptr), DialogType::Channel,
                                                        std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_create_new_dialog_fail(random_id_, std::move(status), std::move(promise_));
  }
};

class CheckHistoryImportQuery final : public Td::ResultHandler {
  Promise<tl_object_ptr<td_api::MessageFileType>> promise_;

 public:
  explicit CheckHistoryImportQuery(Promise<tl_object_ptr<td_api::MessageFileType>> &&promise)
      : promise_(std::move(promise)) {
  }

  void send(const string &message_file_head) {
    send_query(G()->net_query_creator().create(telegram_api::messages_checkHistoryImport(message_file_head)));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_checkHistoryImport>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for CheckHistoryImportQuery: " << to_string(ptr);
    auto file_type = [&]() -> td_api::object_ptr<td_api::MessageFileType> {
      if (ptr->pm_) {
        return td_api::make_object<td_api::messageFileTypePrivate>(ptr->title_);
      } else if (ptr->group_) {
        return td_api::make_object<td_api::messageFileTypeGroup>(ptr->title_);
      } else {
        return td_api::make_object<td_api::messageFileTypeUnknown>();
      }
    }();
    promise_.set_value(std::move(file_type));
  }

  void on_error(uint64 id, Status status) final {
    promise_.set_error(std::move(status));
  }
};

class CheckHistoryImportPeerQuery final : public Td::ResultHandler {
  Promise<string> promise_;
  DialogId dialog_id_;

 public:
  explicit CheckHistoryImportPeerQuery(Promise<string> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id) {
    dialog_id_ = dialog_id;
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(telegram_api::messages_checkHistoryImportPeer(std::move(input_peer))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_checkHistoryImportPeer>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for CheckHistoryImportPeerQuery: " << to_string(ptr);
    promise_.set_value(std::move(ptr->confirm_text_));
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "CheckHistoryImportPeerQuery");
    promise_.set_error(std::move(status));
  }
};

class InitHistoryImportQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  FileId file_id_;
  DialogId dialog_id_;
  vector<FileId> attached_file_ids_;

 public:
  explicit InitHistoryImportQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, FileId file_id, tl_object_ptr<telegram_api::InputFile> &&input_file,
            vector<FileId> attached_file_ids) {
    CHECK(input_file != nullptr);
    file_id_ = file_id;
    dialog_id_ = dialog_id;
    attached_file_ids_ = std::move(attached_file_ids);

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(telegram_api::messages_initHistoryImport(
        std::move(input_peer), std::move(input_file), narrow_cast<int32>(attached_file_ids_.size()))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_initHistoryImport>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    td->file_manager_->delete_partial_remote_location(file_id_);

    auto ptr = result_ptr.move_as_ok();
    td->messages_manager_->start_import_messages(dialog_id_, ptr->id_, std::move(attached_file_ids_),
                                                 std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    if (FileReferenceManager::is_file_reference_error(status)) {
      LOG(ERROR) << "Receive file reference error " << status;
    }
    if (begins_with(status.message(), "FILE_PART_") && ends_with(status.message(), "_MISSING")) {
      // TODO support FILE_PART_*_MISSING
    }

    td->file_manager_->delete_partial_remote_location(file_id_);

    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "InitHistoryImportQuery");
    promise_.set_error(std::move(status));
  }
};

class UploadImportedMediaQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  int64 import_id_;
  FileId file_id_;

 public:
  explicit UploadImportedMediaQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, int64 import_id, const string &file_name, FileId file_id,
            tl_object_ptr<telegram_api::InputMedia> &&input_media) {
    CHECK(input_media != nullptr);
    dialog_id_ = dialog_id;
    import_id_ = import_id;
    file_id_ = file_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      return on_error(0, Status::Error(400, "Can't access the chat"));
    }

    send_query(G()->net_query_creator().create(telegram_api::messages_uploadImportedMedia(
        std::move(input_peer), import_id, file_name, std::move(input_media))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_uploadImportedMedia>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    td->file_manager_->delete_partial_remote_location(file_id_);

    // ignore response

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    if (FileReferenceManager::is_file_reference_error(status)) {
      LOG(ERROR) << "Receive file reference error " << status;
    }
    if (begins_with(status.message(), "FILE_PART_") && ends_with(status.message(), "_MISSING")) {
      // TODO support FILE_PART_*_MISSING
    }

    td->file_manager_->delete_partial_remote_location(file_id_);
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "UploadImportedMediaQuery");
    promise_.set_error(std::move(status));
  }
};

class StartImportHistoryQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit StartImportHistoryQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, int64 import_id) {
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    CHECK(input_peer != nullptr);

    send_query(
        G()->net_query_creator().create(telegram_api::messages_startHistoryImport(std::move(input_peer), import_id)));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_startHistoryImport>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    if (!result_ptr.ok()) {
      return on_error(id, Status::Error(500, "Import history returned false"));
    }
    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "StartImportHistoryQuery");
    promise_.set_error(std::move(status));
  }
};

class EditDialogPhotoQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  FileId file_id_;
  bool was_uploaded_ = false;
  string file_reference_;
  DialogId dialog_id_;

 public:
  explicit EditDialogPhotoQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, FileId file_id, tl_object_ptr<telegram_api::InputChatPhoto> &&input_chat_photo) {
    CHECK(input_chat_photo != nullptr);
    file_id_ = file_id;
    was_uploaded_ = FileManager::extract_was_uploaded(input_chat_photo);
    file_reference_ = FileManager::extract_file_reference(input_chat_photo);
    dialog_id_ = dialog_id;

    switch (dialog_id.get_type()) {
      case DialogType::Chat:
        send_query(G()->net_query_creator().create(
            telegram_api::messages_editChatPhoto(dialog_id.get_chat_id().get(), std::move(input_chat_photo))));
        break;
      case DialogType::Channel: {
        auto channel_id = dialog_id.get_channel_id();
        auto input_channel = td->contacts_manager_->get_input_channel(channel_id);
        CHECK(input_channel != nullptr);
        send_query(G()->net_query_creator().create(
            telegram_api::channels_editPhoto(std::move(input_channel), std::move(input_chat_photo))));
        break;
      }
      default:
        UNREACHABLE();
    }
  }

  void on_result(uint64 id, BufferSlice packet) final {
    static_assert(std::is_same<telegram_api::messages_editChatPhoto::ReturnType,
                               telegram_api::channels_editPhoto::ReturnType>::value,
                  "");
    auto result_ptr = fetch_result<telegram_api::messages_editChatPhoto>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for EditDialogPhotoQuery: " << to_string(ptr);

    if (file_id_.is_valid() && was_uploaded_) {
      td->file_manager_->delete_partial_remote_location(file_id_);
    }

    td->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    if (file_id_.is_valid() && was_uploaded_) {
      td->file_manager_->delete_partial_remote_location(file_id_);
    }
    if (!td->auth_manager_->is_bot() && FileReferenceManager::is_file_reference_error(status)) {
      if (file_id_.is_valid() && !was_uploaded_) {
        VLOG(file_references) << "Receive " << status << " for " << file_id_;
        td->file_manager_->delete_file_reference(file_id_, file_reference_);
        td->messages_manager_->upload_dialog_photo(dialog_id_, file_id_, false, 0.0, false, std::move(promise_), {-1});
        return;
      } else {
        LOG(ERROR) << "Receive file reference error, but file_id = " << file_id_
                   << ", was_uploaded = " << was_uploaded_;
      }
    }

    if (status.message() == "CHAT_NOT_MODIFIED") {
      if (!td->auth_manager_->is_bot()) {
        promise_.set_value(Unit());
        return;
      }
    } else {
      td->messages_manager_->on_get_dialog_error(dialog_id_, status, "EditDialogPhotoQuery");
    }
    td->updates_manager_->get_difference("EditDialogPhotoQuery");
    promise_.set_error(std::move(status));
  }
};

class EditDialogTitleQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit EditDialogTitleQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, const string &title) {
    dialog_id_ = dialog_id;
    switch (dialog_id.get_type()) {
      case DialogType::Chat:
        send_query(G()->net_query_creator().create(
            telegram_api::messages_editChatTitle(dialog_id.get_chat_id().get(), title)));
        break;
      case DialogType::Channel: {
        auto channel_id = dialog_id.get_channel_id();
        auto input_channel = td->contacts_manager_->get_input_channel(channel_id);
        CHECK(input_channel != nullptr);
        send_query(G()->net_query_creator().create(telegram_api::channels_editTitle(std::move(input_channel), title)));
        break;
      }
      default:
        UNREACHABLE();
    }
  }

  void on_result(uint64 id, BufferSlice packet) final {
    static_assert(std::is_same<telegram_api::messages_editChatTitle::ReturnType,
                               telegram_api::channels_editTitle::ReturnType>::value,
                  "");
    auto result_ptr = fetch_result<telegram_api::messages_editChatTitle>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for EditDialogTitleQuery: " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    td->updates_manager_->get_difference("EditDialogTitleQuery");

    if (status.message() == "CHAT_NOT_MODIFIED") {
      if (!td->auth_manager_->is_bot()) {
        promise_.set_value(Unit());
        return;
      }
    } else {
      td->messages_manager_->on_get_dialog_error(dialog_id_, status, "EditDialogTitleQuery");
    }
    promise_.set_error(std::move(status));
  }
};

class SetChatThemeQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit SetChatThemeQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, const string &theme_name) {
    dialog_id_ = dialog_id;
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(telegram_api::messages_setChatTheme(std::move(input_peer), theme_name)));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_setChatTheme>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SetChatThemeQuery: " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    if (status.message() == "CHAT_NOT_MODIFIED") {
      if (!td->auth_manager_->is_bot()) {
        promise_.set_value(Unit());
        return;
      }
    } else {
      td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SetChatThemeQuery");
    }
    promise_.set_error(std::move(status));
  }
};

class SetHistoryTtlQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit SetHistoryTtlQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, int32 period) {
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    CHECK(input_peer != nullptr);

    send_query(G()->net_query_creator().create(telegram_api::messages_setHistoryTTL(std::move(input_peer), period)));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_setHistoryTTL>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SetHistoryTtlQuery: " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    if (status.message() == "CHAT_NOT_MODIFIED") {
      if (!td->auth_manager_->is_bot()) {
        promise_.set_value(Unit());
        return;
      }
    } else {
      td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SetHistoryTtlQuery");
    }
    promise_.set_error(std::move(status));
  }
};

class EditChatDefaultBannedRightsQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit EditChatDefaultBannedRightsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, RestrictedRights permissions) {
    dialog_id_ = dialog_id;
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(telegram_api::messages_editChatDefaultBannedRights(
        std::move(input_peer), permissions.get_chat_banned_rights())));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_editChatDefaultBannedRights>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for EditChatDefaultBannedRightsQuery: " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    if (status.message() == "CHAT_NOT_MODIFIED") {
      if (!td->auth_manager_->is_bot()) {
        promise_.set_value(Unit());
        return;
      }
    } else {
      td->messages_manager_->on_get_dialog_error(dialog_id_, status, "EditChatDefaultBannedRightsQuery");
    }
    promise_.set_error(std::move(status));
  }
};

class SaveDraftMessageQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit SaveDraftMessageQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, const unique_ptr<DraftMessage> &draft_message) {
    LOG(INFO) << "Save draft in " << dialog_id;
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      LOG(INFO) << "Can't update draft message because have no write access to " << dialog_id;
      return on_error(0, Status::Error(400, "Can't save draft message"));
    }

    int32 flags = 0;
    ServerMessageId reply_to_message_id;
    if (draft_message != nullptr) {
      if (draft_message->reply_to_message_id.is_valid() && draft_message->reply_to_message_id.is_server()) {
        reply_to_message_id = draft_message->reply_to_message_id.get_server_message_id();
        flags |= MessagesManager::SEND_MESSAGE_FLAG_IS_REPLY;
      }
      if (draft_message->input_message_text.disable_web_page_preview) {
        flags |= MessagesManager::SEND_MESSAGE_FLAG_DISABLE_WEB_PAGE_PREVIEW;
      }
      if (!draft_message->input_message_text.text.entities.empty()) {
        flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_ENTITIES;
      }
    }

    send_query(G()->net_query_creator().create(telegram_api::messages_saveDraft(
        flags, false /*ignored*/, reply_to_message_id.get(), std::move(input_peer),
        draft_message == nullptr ? "" : draft_message->input_message_text.text.text,
        draft_message == nullptr
            ? vector<tl_object_ptr<telegram_api::MessageEntity>>()
            : get_input_message_entities(td->contacts_manager_.get(), draft_message->input_message_text.text.entities,
                                         "SaveDraftMessageQuery"))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_saveDraft>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    if (!result) {
      on_error(id, Status::Error(400, "Save draft failed"));
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SaveDraftMessageQuery")) {
      LOG(ERROR) << "Receive error for SaveDraftMessageQuery: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class ClearAllDraftsQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit ClearAllDraftsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send() {
    send_query(G()->net_query_creator().create(telegram_api::messages_clearAllDrafts()));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_clearAllDrafts>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.move_as_ok();
    if (!result) {
      LOG(INFO) << "Receive false for ClearAllDraftsQuery";
    } else {
      LOG(INFO) << "All draft messages has been cleared";
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    if (!G()->is_expected_error(status)) {
      LOG(ERROR) << "Receive error for ClearAllDraftsQuery: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class ToggleDialogPinQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  bool is_pinned_;

 public:
  explicit ToggleDialogPinQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, bool is_pinned) {
    dialog_id_ = dialog_id;
    is_pinned_ = is_pinned;

    auto input_peer = td->messages_manager_->get_input_dialog_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return on_error(0, Status::Error(400, "Can't access the chat"));
    }

    int32 flags = 0;
    if (is_pinned) {
      flags |= telegram_api::messages_toggleDialogPin::PINNED_MASK;
    }
    send_query(G()->net_query_creator().create(
        telegram_api::messages_toggleDialogPin(flags, false /*ignored*/, std::move(input_peer))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_toggleDialogPin>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    if (!result) {
      on_error(id, Status::Error(400, "Toggle dialog pin failed"));
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "ToggleDialogPinQuery")) {
      LOG(ERROR) << "Receive error for ToggleDialogPinQuery: " << status;
    }
    td->messages_manager_->on_update_pinned_dialogs(FolderId::main());
    td->messages_manager_->on_update_pinned_dialogs(FolderId::archive());
    promise_.set_error(std::move(status));
  }
};

class ReorderPinnedDialogsQuery final : public Td::ResultHandler {
  FolderId folder_id_;
  Promise<Unit> promise_;

 public:
  explicit ReorderPinnedDialogsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(FolderId folder_id, const vector<DialogId> &dialog_ids) {
    folder_id_ = folder_id;
    int32 flags = telegram_api::messages_reorderPinnedDialogs::FORCE_MASK;
    send_query(G()->net_query_creator().create(telegram_api::messages_reorderPinnedDialogs(
        flags, true /*ignored*/, folder_id.get(),
        td->messages_manager_->get_input_dialog_peers(dialog_ids, AccessRights::Read))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_reorderPinnedDialogs>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.move_as_ok();
    if (!result) {
      return on_error(id, Status::Error(400, "Result is false"));
    }
    LOG(INFO) << "Pinned chats reordered in " << folder_id_;

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    if (!G()->is_expected_error(status)) {
      LOG(ERROR) << "Receive error for ReorderPinnedDialogsQuery: " << status;
    }
    td->messages_manager_->on_update_pinned_dialogs(folder_id_);
    promise_.set_error(std::move(status));
  }
};

class ToggleDialogUnreadMarkQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  bool is_marked_as_unread_;

 public:
  explicit ToggleDialogUnreadMarkQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, bool is_marked_as_unread) {
    dialog_id_ = dialog_id;
    is_marked_as_unread_ = is_marked_as_unread;

    auto input_peer = td->messages_manager_->get_input_dialog_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return on_error(0, Status::Error(400, "Can't access the chat"));
    }

    int32 flags = 0;
    if (is_marked_as_unread) {
      flags |= telegram_api::messages_markDialogUnread::UNREAD_MASK;
    }
    send_query(G()->net_query_creator().create(
        telegram_api::messages_markDialogUnread(flags, false /*ignored*/, std::move(input_peer))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_markDialogUnread>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    if (!result) {
      on_error(id, Status::Error(400, "Toggle dialog mark failed"));
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "ToggleDialogUnreadMarkQuery")) {
      LOG(ERROR) << "Receive error for ToggleDialogUnreadMarkQuery: " << status;
    }
    if (!G()->close_flag()) {
      td->messages_manager_->on_update_dialog_is_marked_as_unread(dialog_id_, !is_marked_as_unread_);
    }
    promise_.set_error(std::move(status));
  }
};

class ToggleDialogIsBlockedActor final : public NetActorOnce {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  bool is_blocked_;

 public:
  explicit ToggleDialogIsBlockedActor(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, bool is_blocked, uint64 sequence_dispatcher_id) {
    dialog_id_ = dialog_id;
    is_blocked_ = is_blocked;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Know);
    CHECK(input_peer != nullptr && input_peer->get_id() != telegram_api::inputPeerEmpty::ID);
    auto query = is_blocked ? G()->net_query_creator().create(telegram_api::contacts_block(std::move(input_peer)))
                            : G()->net_query_creator().create(telegram_api::contacts_unblock(std::move(input_peer)));
    send_closure(td->messages_manager_->sequence_dispatcher_, &MultiSequenceDispatcher::send_with_callback,
                 std::move(query), actor_shared(this), sequence_dispatcher_id);
  }

  void on_result(uint64 id, BufferSlice packet) final {
    static_assert(
        std::is_same<telegram_api::contacts_block::ReturnType, telegram_api::contacts_unblock::ReturnType>::value, "");
    auto result_ptr = fetch_result<telegram_api::contacts_block>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    LOG_IF(WARNING, !result) << "Block/Unblock " << dialog_id_ << " has failed";

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "ToggleDialogIsBlockedActor")) {
      LOG(ERROR) << "Receive error for ToggleDialogIsBlockedActor: " << status;
    }
    if (!G()->close_flag()) {
      td->messages_manager_->on_update_dialog_is_blocked(dialog_id_, !is_blocked_);
      td->messages_manager_->get_dialog_info_full(dialog_id_, Auto(), "ToggleDialogIsBlockedActor");
      td->messages_manager_->reget_dialog_action_bar(dialog_id_, "ToggleDialogIsBlockedActor");
    }
    promise_.set_error(std::move(status));
  }
};

class GetMessagesViewsQuery final : public Td::ResultHandler {
  DialogId dialog_id_;
  vector<MessageId> message_ids_;

 public:
  void send(DialogId dialog_id, vector<MessageId> &&message_ids, bool increment_view_counter) {
    dialog_id_ = dialog_id;
    message_ids_ = std::move(message_ids);

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return on_error(0, Status::Error(400, "Can't access the chat"));
    }

    LOG(INFO) << "View " << message_ids_.size() << " messages in " << dialog_id
              << ", increment = " << increment_view_counter;
    send_query(G()->net_query_creator().create(telegram_api::messages_getMessagesViews(
        std::move(input_peer), MessagesManager::get_server_message_ids(message_ids_), increment_view_counter)));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getMessagesViews>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto result = result_ptr.move_as_ok();
    auto interaction_infos = std::move(result->views_);
    if (message_ids_.size() != interaction_infos.size()) {
      return on_error(id, Status::Error(500, "Wrong number of message views returned"));
    }
    td->contacts_manager_->on_get_users(std::move(result->users_), "GetMessagesViewsQuery");
    td->contacts_manager_->on_get_chats(std::move(result->chats_), "GetMessagesViewsQuery");
    for (size_t i = 0; i < message_ids_.size(); i++) {
      FullMessageId full_message_id{dialog_id_, message_ids_[i]};

      auto *info = interaction_infos[i].get();
      auto flags = info->flags_;
      auto view_count = (flags & telegram_api::messageViews::VIEWS_MASK) != 0 ? info->views_ : 0;
      auto forward_count = (flags & telegram_api::messageViews::FORWARDS_MASK) != 0 ? info->forwards_ : 0;
      td->messages_manager_->on_update_message_interaction_info(full_message_id, view_count, forward_count, true,
                                                                std::move(info->replies_));
    }
  }

  void on_error(uint64 id, Status status) final {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetMessagesViewsQuery")) {
      LOG(ERROR) << "Receive error for GetMessagesViewsQuery: " << status;
    }
  }
};

class ReadMessagesContentsQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit ReadMessagesContentsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(vector<MessageId> &&message_ids) {
    LOG(INFO) << "Receive ReadMessagesContentsQuery for messages " << format::as_array(message_ids);

    send_query(G()->net_query_creator().create(
        telegram_api::messages_readMessageContents(MessagesManager::get_server_message_ids(message_ids))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_readMessageContents>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto affected_messages = result_ptr.move_as_ok();
    CHECK(affected_messages->get_id() == telegram_api::messages_affectedMessages::ID);

    if (affected_messages->pts_count_ > 0) {
      td->updates_manager_->add_pending_pts_update(make_tl_object<dummyUpdate>(), affected_messages->pts_,
                                                   affected_messages->pts_count_, Time::now(), Promise<Unit>(),
                                                   "read messages content query");
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    if (!G()->is_expected_error(status)) {
      LOG(ERROR) << "Receive error for read message contents: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class ReadChannelMessagesContentsQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;

 public:
  explicit ReadChannelMessagesContentsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, vector<MessageId> &&message_ids) {
    channel_id_ = channel_id;

    auto input_channel = td->contacts_manager_->get_input_channel(channel_id);
    if (input_channel == nullptr) {
      LOG(ERROR) << "Have no input channel for " << channel_id;
      return on_error(0, Status::Error(500, "Can't read channel message contents"));
    }

    LOG(INFO) << "Receive ReadChannelMessagesContentsQuery for messages " << format::as_array(message_ids) << " in "
              << channel_id;

    send_query(G()->net_query_creator().create(telegram_api::channels_readMessageContents(
        std::move(input_channel), MessagesManager::get_server_message_ids(message_ids))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::channels_readMessageContents>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    LOG_IF(ERROR, !result) << "Read channel messages contents failed";

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    if (!td->contacts_manager_->on_get_channel_error(channel_id_, status, "ReadChannelMessagesContentsQuery")) {
      LOG(ERROR) << "Receive error for read messages contents in " << channel_id_ << ": " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class GetDialogMessageByDateQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  int32 date_;
  int64 random_id_;

 public:
  explicit GetDialogMessageByDateQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, int32 date, int64 random_id) {
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(400, "Can't access the chat"));
    }

    dialog_id_ = dialog_id;
    date_ = date;
    random_id_ = random_id;

    send_query(G()->net_query_creator().create(
        telegram_api::messages_getHistory(std::move(input_peer), 0, date, -3, 5, 0, 0, 0)));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto info = td->messages_manager_->get_messages_info(result_ptr.move_as_ok(), "GetDialogMessageByDateQuery");
    td->messages_manager_->get_channel_difference_if_needed(
        dialog_id_, std::move(info),
        PromiseCreator::lambda([actor_id = td->messages_manager_actor_.get(), dialog_id = dialog_id_, date = date_,
                                random_id = random_id_,
                                promise = std::move(promise_)](Result<MessagesManager::MessagesInfo> &&result) mutable {
          if (result.is_error()) {
            promise.set_error(result.move_as_error());
          } else {
            auto info = result.move_as_ok();
            send_closure(actor_id, &MessagesManager::on_get_dialog_message_by_date_success, dialog_id, date, random_id,
                         std::move(info.messages), std::move(promise));
          }
        }));
  }

  void on_error(uint64 id, Status status) final {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetDialogMessageByDateQuery")) {
      LOG(ERROR) << "Receive error for GetDialogMessageByDateQuery in " << dialog_id_ << ": " << status;
    }
    promise_.set_error(std::move(status));
    td->messages_manager_->on_get_dialog_message_by_date_fail(random_id_);
  }
};

class GetHistoryQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  MessageId from_message_id_;
  MessageId old_last_new_message_id_;
  int32 offset_;
  int32 limit_;
  bool from_the_end_;

 public:
  explicit GetHistoryQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId from_message_id, MessageId old_last_new_message_id, int32 offset,
            int32 limit) {
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(400, "Can't access the chat"));
    }
    CHECK(offset < 0);

    dialog_id_ = dialog_id;
    from_message_id_ = from_message_id;
    old_last_new_message_id_ = old_last_new_message_id;
    offset_ = offset;
    limit_ = limit;
    from_the_end_ = false;
    send_query(G()->net_query_creator().create(telegram_api::messages_getHistory(
        std::move(input_peer), from_message_id.get_server_message_id().get(), 0, offset, limit, 0, 0, 0)));
  }

  void send_get_from_the_end(DialogId dialog_id, MessageId old_last_new_message_id, int32 limit) {
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(400, "Can't access the chat"));
    }

    dialog_id_ = dialog_id;
    old_last_new_message_id_ = old_last_new_message_id;
    offset_ = 0;
    limit_ = limit;
    from_the_end_ = true;
    send_query(G()->net_query_creator().create(
        telegram_api::messages_getHistory(std::move(input_peer), 0, 0, 0, limit, 0, 0, 0)));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto info = td->messages_manager_->get_messages_info(result_ptr.move_as_ok(), "GetHistoryQuery");
    td->messages_manager_->get_channel_difference_if_needed(
        dialog_id_, std::move(info),
        PromiseCreator::lambda([actor_id = td->messages_manager_actor_.get(), dialog_id = dialog_id_,
                                from_message_id = from_message_id_, old_last_new_message_id = old_last_new_message_id_,
                                offset = offset_, limit = limit_, from_the_end = from_the_end_,
                                promise = std::move(promise_)](Result<MessagesManager::MessagesInfo> &&result) mutable {
          if (result.is_error()) {
            promise.set_error(result.move_as_error());
          } else {
            auto info = result.move_as_ok();
            // TODO use info.total_count, info.pts
            send_closure(actor_id, &MessagesManager::on_get_history, dialog_id, from_message_id,
                         old_last_new_message_id, offset, limit, from_the_end, std::move(info.messages),
                         std::move(promise));
          }
        }));
  }

  void on_error(uint64 id, Status status) final {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetHistoryQuery")) {
      LOG(ERROR) << "Receive error for GetHistoryQuery in " << dialog_id_ << ": " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class ReadHistoryQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit ReadHistoryQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId max_message_id) {
    dialog_id_ = dialog_id;
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(
        telegram_api::messages_readHistory(std::move(input_peer), max_message_id.get_server_message_id().get())));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_readHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto affected_messages = result_ptr.move_as_ok();
    CHECK(affected_messages->get_id() == telegram_api::messages_affectedMessages::ID);
    LOG(INFO) << "Receive result for ReadHistoryQuery: " << to_string(affected_messages);

    if (affected_messages->pts_count_ > 0) {
      td->updates_manager_->add_pending_pts_update(make_tl_object<dummyUpdate>(), affected_messages->pts_,
                                                   affected_messages->pts_count_, Time::now(), Promise<Unit>(),
                                                   "read history query");
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "ReadHistoryQuery")) {
      LOG(ERROR) << "Receive error for ReadHistoryQuery: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class ReadChannelHistoryQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;

 public:
  explicit ReadChannelHistoryQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, MessageId max_message_id) {
    channel_id_ = channel_id;
    auto input_channel = td->contacts_manager_->get_input_channel(channel_id);
    CHECK(input_channel != nullptr);

    send_query(G()->net_query_creator().create(
        telegram_api::channels_readHistory(std::move(input_channel), max_message_id.get_server_message_id().get())));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::channels_readHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    if (!td->contacts_manager_->on_get_channel_error(channel_id_, status, "ReadChannelHistoryQuery")) {
      LOG(ERROR) << "Receive error for ReadChannelHistoryQuery: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class ReadDiscussionQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit ReadDiscussionQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId top_thread_message_id, MessageId max_message_id) {
    dialog_id_ = dialog_id;
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(telegram_api::messages_readDiscussion(
        std::move(input_peer), top_thread_message_id.get_server_message_id().get(),
        max_message_id.get_server_message_id().get())));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_readDiscussion>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "ReadDiscussionQuery");
    promise_.set_error(std::move(status));
  }
};

class GetSearchResultCalendarQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  MessageId from_message_id_;
  MessageSearchFilter filter_;
  int64 random_id_;

 public:
  explicit GetSearchResultCalendarQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId from_message_id, MessageSearchFilter filter, int64 random_id) {
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);

    dialog_id_ = dialog_id;
    from_message_id_ = from_message_id;
    filter_ = filter;
    random_id_ = random_id;

    send_query(G()->net_query_creator().create(telegram_api::messages_getSearchResultsCalendar(
        std::move(input_peer), get_input_messages_filter(filter), from_message_id.get_server_message_id().get(), 0)));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getSearchResultsCalendar>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto result = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for GetSearchResultCalendarQuery: " << to_string(result);
    td->contacts_manager_->on_get_users(std::move(result->users_), "GetSearchResultCalendarQuery");
    td->contacts_manager_->on_get_chats(std::move(result->chats_), "GetSearchResultCalendarQuery");

    MessagesManager::MessagesInfo info;
    info.messages = std::move(result->messages_);
    info.total_count = result->count_;
    info.is_channel_messages = dialog_id_.get_type() == DialogType::Channel;

    td->messages_manager_->get_channel_difference_if_needed(
        dialog_id_, std::move(info),
        PromiseCreator::lambda([actor_id = td->messages_manager_actor_.get(), dialog_id = dialog_id_,
                                from_message_id = from_message_id_, filter = filter_, random_id = random_id_,
                                periods = std::move(result->periods_),
                                promise = std::move(promise_)](Result<MessagesManager::MessagesInfo> &&result) mutable {
          if (result.is_error()) {
            promise.set_error(result.move_as_error());
          } else {
            auto info = result.move_as_ok();
            send_closure(actor_id, &MessagesManager::on_get_message_search_result_calendar, dialog_id, from_message_id,
                         filter, random_id, info.total_count, std::move(info.messages), std::move(periods),
                         std::move(promise));
          }
        }));
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SearchMessagesQuery");
    td->messages_manager_->on_failed_get_message_search_result_calendar(dialog_id_, random_id_);
    promise_.set_error(std::move(status));
  }
};

class SearchMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  string query_;
  DialogId sender_dialog_id_;
  MessageId from_message_id_;
  int32 offset_;
  int32 limit_;
  MessageSearchFilter filter_;
  MessageId top_thread_message_id_;
  int64 random_id_;
  bool handle_errors_ = true;

 public:
  explicit SearchMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, const string &query, DialogId sender_dialog_id, MessageId from_message_id, int32 offset,
            int32 limit, MessageSearchFilter filter, MessageId top_thread_message_id, int64 random_id) {
    auto input_peer = dialog_id.is_valid() ? td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read)
                                           : make_tl_object<telegram_api::inputPeerEmpty>();
    CHECK(input_peer != nullptr);

    dialog_id_ = dialog_id;
    query_ = query;
    sender_dialog_id_ = sender_dialog_id;
    from_message_id_ = from_message_id;
    offset_ = offset;
    limit_ = limit;
    filter_ = filter;
    top_thread_message_id_ = top_thread_message_id;
    random_id_ = random_id;

    if (filter == MessageSearchFilter::UnreadMention) {
      send_query(G()->net_query_creator().create(
          telegram_api::messages_getUnreadMentions(std::move(input_peer), from_message_id.get_server_message_id().get(),
                                                   offset, limit, std::numeric_limits<int32>::max(), 0)));
    } else if (top_thread_message_id.is_valid() && query.empty() && !sender_dialog_id.is_valid() &&
               filter == MessageSearchFilter::Empty) {
      handle_errors_ = dialog_id.get_type() != DialogType::Channel ||
                       td->contacts_manager_->get_channel_type(dialog_id.get_channel_id()) !=
                           ContactsManager::ChannelType::Broadcast;
      send_query(G()->net_query_creator().create(telegram_api::messages_getReplies(
          std::move(input_peer), top_thread_message_id.get_server_message_id().get(),
          from_message_id.get_server_message_id().get(), 0, offset, limit, std::numeric_limits<int32>::max(), 0, 0)));
    } else {
      int32 flags = 0;
      tl_object_ptr<telegram_api::InputPeer> sender_input_peer;
      if (sender_dialog_id.is_valid()) {
        flags |= telegram_api::messages_search::FROM_ID_MASK;
        sender_input_peer = td->messages_manager_->get_input_peer(sender_dialog_id, AccessRights::Read);
        CHECK(sender_input_peer != nullptr);
      }
      if (top_thread_message_id.is_valid()) {
        flags |= telegram_api::messages_search::TOP_MSG_ID_MASK;
      }

      send_query(G()->net_query_creator().create(telegram_api::messages_search(
          flags, std::move(input_peer), query, std::move(sender_input_peer),
          top_thread_message_id.get_server_message_id().get(), get_input_messages_filter(filter), 0,
          std::numeric_limits<int32>::max(), from_message_id.get_server_message_id().get(), offset, limit,
          std::numeric_limits<int32>::max(), 0, 0)));
    }
  }

  void on_result(uint64 id, BufferSlice packet) final {
    static_assert(std::is_same<telegram_api::messages_getUnreadMentions::ReturnType,
                               telegram_api::messages_search::ReturnType>::value,
                  "");
    static_assert(
        std::is_same<telegram_api::messages_getReplies::ReturnType, telegram_api::messages_search::ReturnType>::value,
        "");
    auto result_ptr = fetch_result<telegram_api::messages_search>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto info = td->messages_manager_->get_messages_info(result_ptr.move_as_ok(), "SearchMessagesQuery");
    td->messages_manager_->get_channel_difference_if_needed(
        dialog_id_, std::move(info),
        PromiseCreator::lambda([actor_id = td->messages_manager_actor_.get(), dialog_id = dialog_id_,
                                query = std::move(query_), sender_dialog_id = sender_dialog_id_,
                                from_message_id = from_message_id_, offset = offset_, limit = limit_, filter = filter_,
                                top_thread_message_id = top_thread_message_id_, random_id = random_id_,
                                promise = std::move(promise_)](Result<MessagesManager::MessagesInfo> &&result) mutable {
          if (result.is_error()) {
            promise.set_error(result.move_as_error());
          } else {
            auto info = result.move_as_ok();
            send_closure(actor_id, &MessagesManager::on_get_dialog_messages_search_result, dialog_id, query,
                         sender_dialog_id, from_message_id, offset, limit, filter, top_thread_message_id, random_id,
                         info.total_count, std::move(info.messages), std::move(promise));
          }
        }));
  }

  void on_error(uint64 id, Status status) final {
    if (handle_errors_) {
      td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SearchMessagesQuery");
    }
    td->messages_manager_->on_failed_dialog_messages_search(dialog_id_, random_id_);
    promise_.set_error(std::move(status));
  }
};

class GetSearchResultPositionsQuery final : public Td::ResultHandler {
  Promise<td_api::object_ptr<td_api::messagePositions>> promise_;
  DialogId dialog_id_;
  MessageSearchFilter filter_;

 public:
  explicit GetSearchResultPositionsQuery(Promise<td_api::object_ptr<td_api::messagePositions>> &&promise)
      : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageSearchFilter filter, MessageId from_message_id, int32 limit) {
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(400, "Can't access the chat"));
    }
    dialog_id_ = dialog_id;
    filter_ = filter;

    send_query(G()->net_query_creator().create(
        telegram_api::messages_getSearchResultsPositions(std::move(input_peer), get_input_messages_filter(filter),
                                                         from_message_id.get_server_message_id().get(), limit)));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getSearchResultsPositions>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    td->messages_manager_->on_get_dialog_sparse_message_positions(dialog_id_, filter_, result_ptr.move_as_ok(),
                                                                  std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetSearchResultPositionsQuery");
    promise_.set_error(std::move(status));
  }
};

class GetSearchCountersQuery final : public Td::ResultHandler {
  Promise<int32> promise_;
  DialogId dialog_id_;
  MessageSearchFilter filter_;

 public:
  explicit GetSearchCountersQuery(Promise<int32> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageSearchFilter filter) {
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(400, "Can't access the chat"));
    }

    dialog_id_ = dialog_id;
    filter_ = filter;

    CHECK(filter != MessageSearchFilter::Empty);
    CHECK(filter != MessageSearchFilter::UnreadMention);
    CHECK(filter != MessageSearchFilter::FailedToSend);
    vector<telegram_api::object_ptr<telegram_api::MessagesFilter>> filters;
    filters.push_back(get_input_messages_filter(filter));
    send_query(G()->net_query_creator().create(
        telegram_api::messages_getSearchCounters(std::move(input_peer), std::move(filters))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getSearchCounters>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto result = result_ptr.move_as_ok();
    if (result.size() != 1 || result[0]->filter_->get_id() != get_input_messages_filter(filter_)->get_id()) {
      LOG(ERROR) << "Receive unexpected response for get message count in " << dialog_id_ << " with filter " << filter_
                 << ": " << to_string(result);
      return on_error(id, Status::Error(500, "Receive wrong response"));
    }

    td->messages_manager_->on_get_dialog_message_count(dialog_id_, filter_, result[0]->count_, std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetSearchCountersQuery");
    promise_.set_error(std::move(status));
  }
};

class SearchMessagesGlobalQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  string query_;
  int32 offset_date_;
  DialogId offset_dialog_id_;
  MessageId offset_message_id_;
  int32 limit_;
  MessageSearchFilter filter_;
  int32 min_date_;
  int32 max_date_;
  int64 random_id_;

 public:
  explicit SearchMessagesGlobalQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(FolderId folder_id, bool ignore_folder_id, const string &query, int32 offset_date,
            DialogId offset_dialog_id, MessageId offset_message_id, int32 limit, MessageSearchFilter filter,
            int32 min_date, int32 max_date, int64 random_id) {
    query_ = query;
    offset_date_ = offset_date;
    offset_dialog_id_ = offset_dialog_id;
    offset_message_id_ = offset_message_id;
    limit_ = limit;
    random_id_ = random_id;
    filter_ = filter;
    min_date_ = min_date;
    max_date_ = max_date;

    auto input_peer = MessagesManager::get_input_peer_force(offset_dialog_id);
    CHECK(input_peer != nullptr);

    int32 flags = 0;
    if (!ignore_folder_id) {
      flags |= telegram_api::messages_searchGlobal::FOLDER_ID_MASK;
    }
    send_query(G()->net_query_creator().create(telegram_api::messages_searchGlobal(
        flags, folder_id.get(), query, get_input_messages_filter(filter), min_date_, max_date_, offset_date_,
        std::move(input_peer), offset_message_id.get_server_message_id().get(), limit)));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_searchGlobal>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto info = td->messages_manager_->get_messages_info(result_ptr.move_as_ok(), "SearchMessagesGlobalQuery");
    td->messages_manager_->get_channel_differences_if_needed(
        std::move(info),
        PromiseCreator::lambda([actor_id = td->messages_manager_actor_.get(), query = std::move(query_),
                                offset_date = offset_date_, offset_dialog_id = offset_dialog_id_,
                                offset_message_id = offset_message_id_, limit = limit_, filter = std::move(filter_),
                                min_date = min_date_, max_date = max_date_, random_id = random_id_,
                                promise = std::move(promise_)](Result<MessagesManager::MessagesInfo> &&result) mutable {
          if (result.is_error()) {
            promise.set_error(result.move_as_error());
          } else {
            auto info = result.move_as_ok();
            send_closure(actor_id, &MessagesManager::on_get_messages_search_result, query, offset_date,
                         offset_dialog_id, offset_message_id, limit, filter, min_date, max_date, random_id,
                         info.total_count, std::move(info.messages), std::move(promise));
          }
        }));
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_failed_messages_search(random_id_);
    promise_.set_error(std::move(status));
  }
};

class GetAllScheduledMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  uint32 generation_;

 public:
  explicit GetAllScheduledMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, int64 hash, uint32 generation) {
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);

    dialog_id_ = dialog_id;
    generation_ = generation;

    send_query(
        G()->net_query_creator().create(telegram_api::messages_getScheduledHistory(std::move(input_peer), hash)));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getScheduledHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    if (result_ptr.ok()->get_id() == telegram_api::messages_messagesNotModified::ID) {
      td->messages_manager_->on_get_scheduled_server_messages(dialog_id_, generation_, Auto(), true);
    } else {
      auto info = td->messages_manager_->get_messages_info(result_ptr.move_as_ok(), "GetAllScheduledMessagesQuery");
      td->messages_manager_->on_get_scheduled_server_messages(dialog_id_, generation_, std::move(info.messages), false);
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetAllScheduledMessagesQuery");
    promise_.set_error(std::move(status));
  }
};

class GetRecentLocationsQuery final : public Td::ResultHandler {
  Promise<td_api::object_ptr<td_api::messages>> promise_;
  DialogId dialog_id_;
  int32 limit_;

 public:
  explicit GetRecentLocationsQuery(Promise<td_api::object_ptr<td_api::messages>> &&promise)
      : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, int32 limit) {
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return on_error(0, Status::Error(400, "Have no info about the chat"));
    }

    dialog_id_ = dialog_id;
    limit_ = limit;

    send_query(
        G()->net_query_creator().create(telegram_api::messages_getRecentLocations(std::move(input_peer), limit, 0)));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getRecentLocations>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto info = td->messages_manager_->get_messages_info(result_ptr.move_as_ok(), "GetRecentLocationsQuery");
    td->messages_manager_->get_channel_difference_if_needed(
        dialog_id_, std::move(info),
        PromiseCreator::lambda([actor_id = td->messages_manager_actor_.get(), dialog_id = dialog_id_, limit = limit_,
                                promise = std::move(promise_)](Result<MessagesManager::MessagesInfo> &&result) mutable {
          if (result.is_error()) {
            promise.set_error(result.move_as_error());
          } else {
            auto info = result.move_as_ok();
            send_closure(actor_id, &MessagesManager::on_get_recent_locations, dialog_id, limit, info.total_count,
                         std::move(info.messages), std::move(promise));
          }
        }));
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetRecentLocationsQuery");
    promise_.set_error(std::move(status));
  }
};

class GetMessagePublicForwardsQuery final : public Td::ResultHandler {
  Promise<td_api::object_ptr<td_api::foundMessages>> promise_;
  DialogId dialog_id_;
  int32 limit_;

 public:
  explicit GetMessagePublicForwardsQuery(Promise<td_api::object_ptr<td_api::foundMessages>> &&promise)
      : promise_(std::move(promise)) {
  }

  void send(DcId dc_id, FullMessageId full_message_id, int32 offset_date, DialogId offset_dialog_id,
            ServerMessageId offset_message_id, int32 limit) {
    dialog_id_ = full_message_id.get_dialog_id();
    limit_ = limit;

    auto input_peer = MessagesManager::get_input_peer_force(offset_dialog_id);
    CHECK(input_peer != nullptr);

    send_query(
        G()->net_query_creator().create(telegram_api::stats_getMessagePublicForwards(
                                            td->contacts_manager_->get_input_channel(dialog_id_.get_channel_id()),
                                            full_message_id.get_message_id().get_server_message_id().get(), offset_date,
                                            std::move(input_peer), offset_message_id.get(), limit),
                                        dc_id));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::stats_getMessagePublicForwards>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto info = td->messages_manager_->get_messages_info(result_ptr.move_as_ok(), "GetMessagePublicForwardsQuery");
    td->messages_manager_->get_channel_differences_if_needed(
        std::move(info),
        PromiseCreator::lambda([actor_id = td->messages_manager_actor_.get(),
                                promise = std::move(promise_)](Result<MessagesManager::MessagesInfo> &&result) mutable {
          if (result.is_error()) {
            promise.set_error(result.move_as_error());
          } else {
            auto info = result.move_as_ok();
            send_closure(actor_id, &MessagesManager::on_get_message_public_forwards, info.total_count,
                         std::move(info.messages), std::move(promise));
          }
        }));
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetMessagePublicForwardsQuery");
    promise_.set_error(std::move(status));
  }
};

class HidePromoDataQuery final : public Td::ResultHandler {
  DialogId dialog_id_;

 public:
  void send(DialogId dialog_id) {
    dialog_id_ = dialog_id;
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(telegram_api::help_hidePromoData(std::move(input_peer))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::help_hidePromoData>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    // we are not interested in the result
  }

  void on_error(uint64 id, Status status) final {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "HidePromoDataQuery")) {
      LOG(ERROR) << "Receive error for sponsored chat hiding: " << status;
    }
  }
};

class DeleteHistoryQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  MessageId max_message_id_;
  bool remove_from_dialog_list_;
  bool revoke_;

  void send_request() {
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id_, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(400, "Chat is not accessible"));
    }

    int32 flags = 0;
    if (!remove_from_dialog_list_) {
      flags |= telegram_api::messages_deleteHistory::JUST_CLEAR_MASK;
    }
    if (revoke_) {
      flags |= telegram_api::messages_deleteHistory::REVOKE_MASK;
    }
    LOG(INFO) << "Delete " << dialog_id_ << " history up to " << max_message_id_ << " with flags " << flags;

    send_query(G()->net_query_creator().create(
        telegram_api::messages_deleteHistory(flags, false /*ignored*/, false /*ignored*/, std::move(input_peer),
                                             max_message_id_.get_server_message_id().get(), 0, 0)));
  }

 public:
  explicit DeleteHistoryQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId max_message_id, bool remove_from_dialog_list, bool revoke) {
    dialog_id_ = dialog_id;
    max_message_id_ = max_message_id;
    remove_from_dialog_list_ = remove_from_dialog_list;
    revoke_ = revoke;

    send_request();
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_deleteHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto affected_history = result_ptr.move_as_ok();
    CHECK(affected_history->get_id() == telegram_api::messages_affectedHistory::ID);

    if (affected_history->pts_count_ > 0) {
      td->updates_manager_->add_pending_pts_update(make_tl_object<dummyUpdate>(), affected_history->pts_,
                                                   affected_history->pts_count_, Time::now(), Promise<Unit>(),
                                                   "delete history query");
    }

    if (affected_history->offset_ > 0) {
      send_request();
      return;
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "DeleteHistoryQuery");
    promise_.set_error(std::move(status));
  }
};

class DeleteChannelHistoryQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;
  MessageId max_message_id_;
  bool allow_error_;

 public:
  explicit DeleteChannelHistoryQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, MessageId max_message_id, bool allow_error) {
    channel_id_ = channel_id;
    max_message_id_ = max_message_id;
    allow_error_ = allow_error;
    auto input_channel = td->contacts_manager_->get_input_channel(channel_id);
    CHECK(input_channel != nullptr);

    send_query(G()->net_query_creator().create(
        telegram_api::channels_deleteHistory(std::move(input_channel), max_message_id.get_server_message_id().get())));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::channels_deleteHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    LOG_IF(ERROR, !allow_error_ && !result)
        << "Delete history in " << channel_id_ << " up to " << max_message_id_ << " failed";

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    if (!td->contacts_manager_->on_get_channel_error(channel_id_, status, "DeleteChannelHistoryQuery")) {
      LOG(ERROR) << "Receive error for DeleteChannelHistoryQuery: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class DeleteMessagesByDateQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  int32 min_date_;
  int32 max_date_;
  bool revoke_;

  void send_request() {
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id_, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(400, "Chat is not accessible"));
    }

    int32 flags = telegram_api::messages_deleteHistory::JUST_CLEAR_MASK |
                  telegram_api::messages_deleteHistory::MIN_DATE_MASK |
                  telegram_api::messages_deleteHistory::MAX_DATE_MASK;
    if (revoke_) {
      flags |= telegram_api::messages_deleteHistory::REVOKE_MASK;
    }

    send_query(G()->net_query_creator().create(telegram_api::messages_deleteHistory(
        flags, false /*ignored*/, false /*ignored*/, std::move(input_peer), 0, min_date_, max_date_)));
  }

 public:
  explicit DeleteMessagesByDateQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, int32 min_date, int32 max_date, bool revoke) {
    dialog_id_ = dialog_id;
    min_date_ = min_date;
    max_date_ = max_date;
    revoke_ = revoke;

    send_request();
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_deleteHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto affected_history = result_ptr.move_as_ok();
    CHECK(affected_history->get_id() == telegram_api::messages_affectedHistory::ID);

    if (affected_history->pts_count_ > 0) {
      affected_history->pts_count_ = 0;  // force receiving real updates from the server
      auto promise = affected_history->offset_ > 0 ? Promise<Unit>() : std::move(promise_);
      td->updates_manager_->add_pending_pts_update(make_tl_object<dummyUpdate>(), affected_history->pts_,
                                                   affected_history->pts_count_, Time::now(), std::move(promise),
                                                   "DeleteMessagesByDateQuery");
    } else if (affected_history->offset_ <= 0) {
      promise_.set_value(Unit());
    }

    if (affected_history->offset_ > 0) {
      send_request();
      return;
    }
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "DeleteMessagesByDateQuery");
    promise_.set_error(std::move(status));
  }
};

class DeletePhoneCallHistoryQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  bool revoke_;

  void send_request() {
    int32 flags = 0;
    if (revoke_) {
      flags |= telegram_api::messages_deletePhoneCallHistory::REVOKE_MASK;
    }
    send_query(
        G()->net_query_creator().create(telegram_api::messages_deletePhoneCallHistory(flags, false /*ignored*/)));
  }

 public:
  explicit DeletePhoneCallHistoryQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(bool revoke) {
    revoke_ = revoke;

    send_request();
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_deletePhoneCallHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto affected_messages = result_ptr.move_as_ok();
    CHECK(affected_messages->get_id() == telegram_api::messages_affectedFoundMessages::ID);

    if (affected_messages->pts_count_ > 0) {
      auto promise = affected_messages->offset_ > 0 ? Promise<Unit>() : std::move(promise_);
      auto pts = affected_messages->pts_;
      auto pts_count = affected_messages->pts_count_;
      auto update =
          make_tl_object<telegram_api::updateDeleteMessages>(std::move(affected_messages->messages_), pts, pts_count);
      td->updates_manager_->add_pending_pts_update(std::move(update), pts, pts_count, Time::now(), std::move(promise),
                                                   "delete phone call history query");
    } else if (affected_messages->offset_ <= 0) {
      promise_.set_value(Unit());
    }

    if (affected_messages->offset_ > 0) {
      send_request();
      return;
    }
  }

  void on_error(uint64 id, Status status) final {
    promise_.set_error(std::move(status));
  }
};

class BlockFromRepliesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit BlockFromRepliesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(MessageId message_id, bool need_delete_message, bool need_delete_all_messages, bool report_spam) {
    int32 flags = 0;
    if (need_delete_message) {
      flags |= telegram_api::contacts_blockFromReplies::DELETE_MESSAGE_MASK;
    }
    if (need_delete_all_messages) {
      flags |= telegram_api::contacts_blockFromReplies::DELETE_HISTORY_MASK;
    }
    if (report_spam) {
      flags |= telegram_api::contacts_blockFromReplies::REPORT_SPAM_MASK;
    }
    send_query(G()->net_query_creator().create(telegram_api::contacts_blockFromReplies(
        flags, false /*ignored*/, false /*ignored*/, false /*ignored*/, message_id.get_server_message_id().get())));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::contacts_blockFromReplies>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for BlockFromRepliesQuery: " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    promise_.set_error(std::move(status));
  }
};

class DeleteUserHistoryQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;
  UserId user_id_;

  void send_request() {
    auto input_channel = td->contacts_manager_->get_input_channel(channel_id_);
    if (input_channel == nullptr) {
      return promise_.set_error(Status::Error(400, "Chat is not accessible"));
    }
    auto input_user = td->contacts_manager_->get_input_user(user_id_);
    if (input_user == nullptr) {
      return promise_.set_error(Status::Error(400, "Usat is not accessible"));
    }

    LOG(INFO) << "Delete all messages from " << user_id_ << " in " << channel_id_;

    send_query(G()->net_query_creator().create(
        telegram_api::channels_deleteUserHistory(std::move(input_channel), std::move(input_user))));
  }

 public:
  explicit DeleteUserHistoryQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, UserId user_id) {
    channel_id_ = channel_id;
    user_id_ = user_id;

    send_request();
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::channels_deleteUserHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto affected_history = result_ptr.move_as_ok();
    CHECK(affected_history->get_id() == telegram_api::messages_affectedHistory::ID);

    if (affected_history->pts_count_ > 0) {
      td->messages_manager_->add_pending_channel_update(
          DialogId(channel_id_), make_tl_object<dummyUpdate>(), affected_history->pts_, affected_history->pts_count_,
          affected_history->offset_ > 0 ? Promise<Unit>() : std::move(promise_), "delete user history query");
    } else if (affected_history->offset_ <= 0) {
      promise_.set_value(Unit());
    }

    if (affected_history->offset_ > 0) {
      send_request();
      return;
    }
  }

  void on_error(uint64 id, Status status) final {
    td->contacts_manager_->on_get_channel_error(channel_id_, status, "DeleteUserHistoryQuery");
    promise_.set_error(std::move(status));
  }
};

class ReadMentionsQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

  void send_request() {
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id_, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(400, "Chat is not accessible"));
    }

    LOG(INFO) << "Read all mentions in " << dialog_id_;

    send_query(G()->net_query_creator().create(telegram_api::messages_readMentions(std::move(input_peer))));
  }

 public:
  explicit ReadMentionsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id) {
    dialog_id_ = dialog_id;

    send_request();
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_readMentions>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto affected_history = result_ptr.move_as_ok();
    CHECK(affected_history->get_id() == telegram_api::messages_affectedHistory::ID);

    if (affected_history->pts_count_ > 0) {
      if (dialog_id_.get_type() == DialogType::Channel) {
        LOG(ERROR) << "Receive pts_count " << affected_history->pts_count_ << " in result of ReadMentionsQuery in "
                   << dialog_id_;
        td->updates_manager_->get_difference("Wrong messages_readMentions result");
      } else {
        td->updates_manager_->add_pending_pts_update(make_tl_object<dummyUpdate>(), affected_history->pts_,
                                                     affected_history->pts_count_, Time::now(), Promise<Unit>(),
                                                     "read all mentions query");
      }
    }

    if (affected_history->offset_ > 0) {
      send_request();
      return;
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "ReadMentionsQuery");
    promise_.set_error(std::move(status));
  }
};

class SendSecretMessageActor final : public NetActor {
  int64 random_id_;

 public:
  void send(DialogId dialog_id, int64 reply_to_random_id, int32 ttl, const string &text, SecretInputMedia media,
            vector<tl_object_ptr<secret_api::MessageEntity>> &&entities, UserId via_bot_user_id, int64 media_album_id,
            bool disable_notification, int64 random_id) {
    if (false && !media.empty()) {
      td->messages_manager_->on_send_secret_message_error(random_id, Status::Error(400, "FILE_PART_1_MISSING"), Auto());
      stop();
      return;
    }

    CHECK(dialog_id.get_type() == DialogType::SecretChat);
    random_id_ = random_id;

    int32 flags = 0;
    if (reply_to_random_id != 0) {
      flags |= secret_api::decryptedMessage::REPLY_TO_RANDOM_ID_MASK;
    }
    if (via_bot_user_id.is_valid()) {
      flags |= secret_api::decryptedMessage::VIA_BOT_NAME_MASK;
    }
    if (!media.empty()) {
      flags |= secret_api::decryptedMessage::MEDIA_MASK;
    }
    if (!entities.empty()) {
      flags |= secret_api::decryptedMessage::ENTITIES_MASK;
    }
    if (media_album_id != 0) {
      CHECK(media_album_id < 0);
      flags |= secret_api::decryptedMessage::GROUPED_ID_MASK;
    }
    if (disable_notification) {
      flags |= secret_api::decryptedMessage::SILENT_MASK;
    }

    send_closure(
        G()->secret_chats_manager(), &SecretChatsManager::send_message, dialog_id.get_secret_chat_id(),
        make_tl_object<secret_api::decryptedMessage>(
            flags, false /*ignored*/, random_id, ttl, text, std::move(media.decrypted_media_), std::move(entities),
            td->contacts_manager_->get_user_username(via_bot_user_id), reply_to_random_id, -media_album_id),
        std::move(media.input_file_), PromiseCreator::event(self_closure(this, &SendSecretMessageActor::done)));
  }

  void done() {
    stop();
  }
};

class SendMessageActor final : public NetActorOnce {
  int64 random_id_;
  DialogId dialog_id_;

 public:
  void send(int32 flags, DialogId dialog_id, MessageId reply_to_message_id, int32 schedule_date,
            tl_object_ptr<telegram_api::ReplyMarkup> &&reply_markup,
            vector<tl_object_ptr<telegram_api::MessageEntity>> &&entities, const string &text, int64 random_id,
            NetQueryRef *send_query_ref, uint64 sequence_dispatcher_id) {
    random_id_ = random_id;
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      on_error(0, Status::Error(400, "Have no write access to the chat"));
      stop();
      return;
    }

    if (!entities.empty()) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_ENTITIES;
    }

    auto query = G()->net_query_creator().create(telegram_api::messages_sendMessage(
        flags, false /*ignored*/, false /*ignored*/, false /*ignored*/, false /*ignored*/, std::move(input_peer),
        reply_to_message_id.get_server_message_id().get(), text, random_id, std::move(reply_markup),
        std::move(entities), schedule_date));
    if (G()->shared_config().get_option_boolean("use_quick_ack")) {
      query->quick_ack_promise_ = PromiseCreator::lambda(
          [random_id](Unit) {
            send_closure(G()->messages_manager(), &MessagesManager::on_send_message_get_quick_ack, random_id);
          },
          PromiseCreator::Ignore());
    }
    *send_query_ref = query.get_weak();
    query->debug("send to MessagesManager::MultiSequenceDispatcher");
    send_closure(td->messages_manager_->sequence_dispatcher_, &MultiSequenceDispatcher::send_with_callback,
                 std::move(query), actor_shared(this), sequence_dispatcher_id);
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_sendMessage>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SendMessage for " << random_id_ << ": " << to_string(ptr);

    auto constructor_id = ptr->get_id();
    if (constructor_id != telegram_api::updateShortSentMessage::ID) {
      td->messages_manager_->check_send_message_result(random_id_, dialog_id_, ptr.get(), "SendMessage");
      return td->updates_manager_->on_get_updates(std::move(ptr), Promise<Unit>());
    }
    auto sent_message = move_tl_object_as<telegram_api::updateShortSentMessage>(ptr);
    td->messages_manager_->on_update_sent_text_message(random_id_, std::move(sent_message->media_),
                                                       std::move(sent_message->entities_));

    auto message_id = MessageId(ServerMessageId(sent_message->id_));
    auto ttl_period = (sent_message->flags_ & telegram_api::updateShortSentMessage::TTL_PERIOD_MASK) != 0
                          ? sent_message->ttl_period_
                          : 0;
    auto update = make_tl_object<updateSentMessage>(random_id_, message_id, sent_message->date_, ttl_period);
    if (dialog_id_.get_type() == DialogType::Channel) {
      td->messages_manager_->add_pending_channel_update(dialog_id_, std::move(update), sent_message->pts_,
                                                        sent_message->pts_count_, Promise<Unit>(),
                                                        "send message actor");
      return;
    }

    td->updates_manager_->add_pending_pts_update(std::move(update), sent_message->pts_, sent_message->pts_count_,
                                                 Time::now(), Promise<Unit>(), "send message actor");
  }

  void on_error(uint64 id, Status status) final {
    LOG(INFO) << "Receive error for SendMessage: " << status;
    if (G()->close_flag() && G()->parameters().use_message_db) {
      // do not send error, message will be re-sent
      return;
    }
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SendMessageActor");
    td->messages_manager_->on_send_message_fail(random_id_, std::move(status));
  }
};

class StartBotQuery final : public Td::ResultHandler {
  int64 random_id_;
  DialogId dialog_id_;

 public:
  NetQueryRef send(tl_object_ptr<telegram_api::InputUser> bot_input_user, DialogId dialog_id,
                   tl_object_ptr<telegram_api::InputPeer> input_peer, const string &parameter, int64 random_id) {
    CHECK(bot_input_user != nullptr);
    CHECK(input_peer != nullptr);
    random_id_ = random_id;
    dialog_id_ = dialog_id;

    auto query = G()->net_query_creator().create(
        telegram_api::messages_startBot(std::move(bot_input_user), std::move(input_peer), random_id, parameter));
    if (G()->shared_config().get_option_boolean("use_quick_ack")) {
      query->quick_ack_promise_ = PromiseCreator::lambda(
          [random_id](Unit) {
            send_closure(G()->messages_manager(), &MessagesManager::on_send_message_get_quick_ack, random_id);
          },
          PromiseCreator::Ignore());
    }
    auto send_query_ref = query.get_weak();
    send_query(std::move(query));
    return send_query_ref;
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_startBot>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for StartBotQuery for " << random_id_ << ": " << to_string(ptr);
    // Result may contain messageActionChatAddUser
    // td->messages_manager_->check_send_message_result(random_id_, dialog_id_, ptr.get(), "StartBot");
    td->updates_manager_->on_get_updates(std::move(ptr), Promise<Unit>());
  }

  void on_error(uint64 id, Status status) final {
    LOG(INFO) << "Receive error for StartBotQuery: " << status;
    if (G()->close_flag() && G()->parameters().use_message_db) {
      // do not send error, message should be re-sent
      return;
    }
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "StartBotQuery");
    td->messages_manager_->on_send_message_fail(random_id_, std::move(status));
  }
};

class SendInlineBotResultQuery final : public Td::ResultHandler {
  int64 random_id_;
  DialogId dialog_id_;

 public:
  NetQueryRef send(int32 flags, DialogId dialog_id, MessageId reply_to_message_id, int32 schedule_date, int64 random_id,
                   int64 query_id, const string &result_id) {
    random_id_ = random_id;
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    CHECK(input_peer != nullptr);

    auto query = G()->net_query_creator().create(telegram_api::messages_sendInlineBotResult(
        flags, false /*ignored*/, false /*ignored*/, false /*ignored*/, false /*ignored*/, std::move(input_peer),
        reply_to_message_id.get_server_message_id().get(), random_id, query_id, result_id, schedule_date));
    auto send_query_ref = query.get_weak();
    send_query(std::move(query));
    return send_query_ref;
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_sendInlineBotResult>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SendInlineBotResultQuery for " << random_id_ << ": " << to_string(ptr);
    td->messages_manager_->check_send_message_result(random_id_, dialog_id_, ptr.get(), "SendInlineBotResult");
    td->updates_manager_->on_get_updates(std::move(ptr), Promise<Unit>());
  }

  void on_error(uint64 id, Status status) final {
    LOG(INFO) << "Receive error for SendInlineBotResultQuery: " << status;
    if (G()->close_flag() && G()->parameters().use_message_db) {
      // do not send error, message will be re-sent
      return;
    }
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SendInlineBotResultQuery");
    td->messages_manager_->on_send_message_fail(random_id_, std::move(status));
  }
};

class SendMultiMediaActor final : public NetActorOnce {
  vector<FileId> file_ids_;
  vector<string> file_references_;
  vector<int64> random_ids_;
  DialogId dialog_id_;

 public:
  void send(int32 flags, DialogId dialog_id, MessageId reply_to_message_id, int32 schedule_date,
            vector<FileId> &&file_ids, vector<tl_object_ptr<telegram_api::inputSingleMedia>> &&input_single_media,
            uint64 sequence_dispatcher_id) {
    for (auto &single_media : input_single_media) {
      random_ids_.push_back(single_media->random_id_);
      CHECK(FileManager::extract_was_uploaded(single_media->media_) == false);
      file_references_.push_back(FileManager::extract_file_reference(single_media->media_));
    }
    dialog_id_ = dialog_id;
    file_ids_ = std::move(file_ids);
    CHECK(file_ids_.size() == random_ids_.size());

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      on_error(0, Status::Error(400, "Have no write access to the chat"));
      stop();
      return;
    }

    auto query = G()->net_query_creator().create(telegram_api::messages_sendMultiMedia(
        flags, false /*ignored*/, false /*ignored*/, false /*ignored*/, std::move(input_peer),
        reply_to_message_id.get_server_message_id().get(), std::move(input_single_media), schedule_date));
    // no quick ack, because file reference errors are very likely to happen
    query->debug("send to MessagesManager::MultiSequenceDispatcher");
    send_closure(td->messages_manager_->sequence_dispatcher_, &MultiSequenceDispatcher::send_with_callback,
                 std::move(query), actor_shared(this), sequence_dispatcher_id);
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_sendMultiMedia>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SendMultiMedia for " << format::as_array(random_ids_) << ": " << to_string(ptr);

    auto sent_random_ids = UpdatesManager::get_sent_messages_random_ids(ptr.get());
    bool is_result_wrong = false;
    auto sent_random_ids_size = sent_random_ids.size();
    for (auto &random_id : random_ids_) {
      auto it = sent_random_ids.find(random_id);
      if (it == sent_random_ids.end()) {
        if (random_ids_.size() == 1) {
          is_result_wrong = true;
        }
        td->messages_manager_->on_send_message_fail(random_id, Status::Error(400, "Message was not sent"));
      } else {
        sent_random_ids.erase(it);
      }
    }
    if (!sent_random_ids.empty()) {
      is_result_wrong = true;
    }
    if (!is_result_wrong) {
      auto sent_messages = UpdatesManager::get_new_messages(ptr.get());
      if (sent_random_ids_size != sent_messages.size()) {
        is_result_wrong = true;
      }
      for (auto &sent_message : sent_messages) {
        if (MessagesManager::get_message_dialog_id(*sent_message) != dialog_id_) {
          is_result_wrong = true;
        }
      }
    }
    if (is_result_wrong) {
      LOG(ERROR) << "Receive wrong result for SendMultiMedia with random_ids " << format::as_array(random_ids_)
                 << " to " << dialog_id_ << ": " << oneline(to_string(ptr));
      td->updates_manager_->schedule_get_difference("Wrong sendMultiMedia result");
    }

    td->updates_manager_->on_get_updates(std::move(ptr), Promise<Unit>());
  }

  void on_error(uint64 id, Status status) final {
    LOG(INFO) << "Receive error for SendMultiMedia: " << status;
    if (G()->close_flag() && G()->parameters().use_message_db) {
      // do not send error, message will be re-sent
      return;
    }
    if (!td->auth_manager_->is_bot() && FileReferenceManager::is_file_reference_error(status)) {
      auto pos = FileReferenceManager::get_file_reference_error_pos(status);
      if (1 <= pos && pos <= file_ids_.size() && file_ids_[pos - 1].is_valid()) {
        VLOG(file_references) << "Receive " << status << " for " << file_ids_[pos - 1];
        td->file_manager_->delete_file_reference(file_ids_[pos - 1], file_references_[pos - 1]);
        td->messages_manager_->on_send_media_group_file_reference_error(dialog_id_, std::move(random_ids_));
        return;
      } else {
        LOG(ERROR) << "Receive file reference error " << status << ", but file_ids = " << file_ids_
                   << ", message_count = " << file_ids_.size();
      }
    }
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SendMultiMediaActor");
    for (auto &random_id : random_ids_) {
      td->messages_manager_->on_send_message_fail(random_id, status.clone());
    }
  }
};

class SendMediaActor final : public NetActorOnce {
  int64 random_id_ = 0;
  FileId file_id_;
  FileId thumbnail_file_id_;
  DialogId dialog_id_;
  string file_reference_;
  bool was_uploaded_ = false;
  bool was_thumbnail_uploaded_ = false;

 public:
  void send(FileId file_id, FileId thumbnail_file_id, int32 flags, DialogId dialog_id, MessageId reply_to_message_id,
            int32 schedule_date, tl_object_ptr<telegram_api::ReplyMarkup> &&reply_markup,
            vector<tl_object_ptr<telegram_api::MessageEntity>> &&entities, const string &text,
            tl_object_ptr<telegram_api::InputMedia> &&input_media, int64 random_id, NetQueryRef *send_query_ref,
            uint64 sequence_dispatcher_id) {
    random_id_ = random_id;
    file_id_ = file_id;
    thumbnail_file_id_ = thumbnail_file_id;
    dialog_id_ = dialog_id;
    file_reference_ = FileManager::extract_file_reference(input_media);
    was_uploaded_ = FileManager::extract_was_uploaded(input_media);
    was_thumbnail_uploaded_ = FileManager::extract_was_thumbnail_uploaded(input_media);

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      on_error(0, Status::Error(400, "Have no write access to the chat"));
      stop();
      return;
    }
    if (!entities.empty()) {
      flags |= telegram_api::messages_sendMedia::ENTITIES_MASK;
    }

    auto query = G()->net_query_creator().create(telegram_api::messages_sendMedia(
        flags, false /*ignored*/, false /*ignored*/, false /*ignored*/, std::move(input_peer),
        reply_to_message_id.get_server_message_id().get(), std::move(input_media), text, random_id,
        std::move(reply_markup), std::move(entities), schedule_date));
    if (G()->shared_config().get_option_boolean("use_quick_ack") && was_uploaded_) {
      query->quick_ack_promise_ = PromiseCreator::lambda(
          [random_id](Unit) {
            send_closure(G()->messages_manager(), &MessagesManager::on_send_message_get_quick_ack, random_id);
          },
          PromiseCreator::Ignore());
    }
    *send_query_ref = query.get_weak();
    query->debug("send to MessagesManager::MultiSequenceDispatcher");
    send_closure(td->messages_manager_->sequence_dispatcher_, &MultiSequenceDispatcher::send_with_callback,
                 std::move(query), actor_shared(this), sequence_dispatcher_id);
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_sendMedia>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    if (was_thumbnail_uploaded_) {
      CHECK(thumbnail_file_id_.is_valid());
      // always delete partial remote location for the thumbnail, because it can't be reused anyway
      // TODO delete it only in the case it can't be merged with file thumbnail
      td->file_manager_->delete_partial_remote_location(thumbnail_file_id_);
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SendMedia for " << random_id_ << ": " << to_string(ptr);
    td->messages_manager_->check_send_message_result(random_id_, dialog_id_, ptr.get(), "SendMedia");
    td->updates_manager_->on_get_updates(std::move(ptr), Promise<Unit>());
  }

  void on_error(uint64 id, Status status) final {
    LOG(INFO) << "Receive error for SendMedia: " << status;
    if (G()->close_flag() && G()->parameters().use_message_db) {
      // do not send error, message will be re-sent
      return;
    }
    if (was_uploaded_) {
      if (was_thumbnail_uploaded_) {
        CHECK(thumbnail_file_id_.is_valid());
        // always delete partial remote location for the thumbnail, because it can't be reused anyway
        td->file_manager_->delete_partial_remote_location(thumbnail_file_id_);
      }

      CHECK(file_id_.is_valid());
      if (begins_with(status.message(), "FILE_PART_") && ends_with(status.message(), "_MISSING")) {
        td->messages_manager_->on_send_message_file_part_missing(random_id_,
                                                                 to_integer<int32>(status.message().substr(10)));
        return;
      } else {
        if (status.code() != 429 && status.code() < 500 && !G()->close_flag()) {
          td->file_manager_->delete_partial_remote_location(file_id_);
        }
      }
    } else if (!td->auth_manager_->is_bot() && FileReferenceManager::is_file_reference_error(status)) {
      if (file_id_.is_valid() && !was_uploaded_) {
        VLOG(file_references) << "Receive " << status << " for " << file_id_;
        td->file_manager_->delete_file_reference(file_id_, file_reference_);
        td->messages_manager_->on_send_message_file_reference_error(random_id_);
        return;
      } else {
        LOG(ERROR) << "Receive file reference error, but file_id = " << file_id_
                   << ", was_uploaded = " << was_uploaded_;
      }
    }

    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SendMediaActor");
    td->messages_manager_->on_send_message_fail(random_id_, std::move(status));
  }
};

class UploadMediaQuery final : public Td::ResultHandler {
  DialogId dialog_id_;
  MessageId message_id_;
  FileId file_id_;
  FileId thumbnail_file_id_;
  string file_reference_;
  bool was_uploaded_ = false;
  bool was_thumbnail_uploaded_ = false;

 public:
  void send(DialogId dialog_id, MessageId message_id, FileId file_id, FileId thumbnail_file_id,
            tl_object_ptr<telegram_api::InputMedia> &&input_media) {
    CHECK(input_media != nullptr);
    dialog_id_ = dialog_id;
    message_id_ = message_id;
    file_id_ = file_id;
    thumbnail_file_id_ = thumbnail_file_id;
    file_reference_ = FileManager::extract_file_reference(input_media);
    was_uploaded_ = FileManager::extract_was_uploaded(input_media);
    was_thumbnail_uploaded_ = FileManager::extract_was_thumbnail_uploaded(input_media);

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      return on_error(0, Status::Error(400, "Have no write access to the chat"));
    }

    send_query(G()->net_query_creator().create(
        telegram_api::messages_uploadMedia(std::move(input_peer), std::move(input_media))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_uploadMedia>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    if (was_thumbnail_uploaded_) {
      CHECK(thumbnail_file_id_.is_valid());
      // always delete partial remote location for the thumbnail, because it can't be reused anyway
      td->file_manager_->delete_partial_remote_location(thumbnail_file_id_);
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for UploadMediaQuery for " << message_id_ << " in " << dialog_id_ << ": "
              << to_string(ptr);
    td->messages_manager_->on_upload_message_media_success(dialog_id_, message_id_, std::move(ptr));
  }

  void on_error(uint64 id, Status status) final {
    LOG(INFO) << "Receive error for UploadMediaQuery for " << message_id_ << " in " << dialog_id_ << ": " << status;
    if (G()->close_flag() && G()->parameters().use_message_db) {
      // do not send error, message will be re-sent
      return;
    }
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "UploadMediaQuery");
    if (was_uploaded_) {
      if (was_thumbnail_uploaded_) {
        CHECK(thumbnail_file_id_.is_valid());
        // always delete partial remote location for the thumbnail, because it can't be reused anyway
        td->file_manager_->delete_partial_remote_location(thumbnail_file_id_);
      }

      CHECK(file_id_.is_valid());
      if (begins_with(status.message(), "FILE_PART_") && ends_with(status.message(), "_MISSING")) {
        td->messages_manager_->on_upload_message_media_file_part_missing(
            dialog_id_, message_id_, to_integer<int32>(status.message().substr(10)));
        return;
      } else {
        if (status.code() != 429 && status.code() < 500 && !G()->close_flag()) {
          td->file_manager_->delete_partial_remote_location(file_id_);
        }
      }
    } else if (FileReferenceManager::is_file_reference_error(status)) {
      LOG(ERROR) << "Receive file reference error for UploadMediaQuery";
    }
    td->messages_manager_->on_upload_message_media_fail(dialog_id_, message_id_, std::move(status));
  }
};

class SendScheduledMessageActor final : public NetActorOnce {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit SendScheduledMessageActor(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId message_id, uint64 sequence_dispatcher_id) {
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Edit);
    if (input_peer == nullptr) {
      on_error(0, Status::Error(400, "Can't access the chat"));
      stop();
      return;
    }

    LOG(DEBUG) << "Send " << FullMessageId{dialog_id, message_id};

    int32 server_message_id = message_id.get_scheduled_server_message_id().get();
    auto query = G()->net_query_creator().create(
        telegram_api::messages_sendScheduledMessages(std::move(input_peer), {server_message_id}));

    query->debug("send to MessagesManager::MultiSequenceDispatcher");
    send_closure(td->messages_manager_->sequence_dispatcher_, &MultiSequenceDispatcher::send_with_callback,
                 std::move(query), actor_shared(this), sequence_dispatcher_id);
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_sendScheduledMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SendScheduledMessageActor: " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    LOG(INFO) << "Receive error for SendScheduledMessageActor: " << status;
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SendScheduledMessageActor");
    promise_.set_error(std::move(status));
  }
};

class EditMessageActor final : public NetActorOnce {
  Promise<int32> promise_;
  DialogId dialog_id_;

 public:
  explicit EditMessageActor(Promise<Unit> &&promise) {
    promise_ = PromiseCreator::lambda([promise = std::move(promise)](Result<int32> result) mutable {
      if (result.is_error()) {
        promise.set_error(result.move_as_error());
      } else {
        promise.set_value(Unit());
      }
    });
  }
  explicit EditMessageActor(Promise<int32> &&promise) : promise_(std::move(promise)) {
  }

  void send(int32 flags, DialogId dialog_id, MessageId message_id, const string &text,
            vector<tl_object_ptr<telegram_api::MessageEntity>> &&entities,
            tl_object_ptr<telegram_api::InputMedia> &&input_media,
            tl_object_ptr<telegram_api::ReplyMarkup> &&reply_markup, int32 schedule_date,
            uint64 sequence_dispatcher_id) {
    dialog_id_ = dialog_id;

    if (false && input_media != nullptr) {
      on_error(0, Status::Error(400, "FILE_PART_1_MISSING"));
      stop();
      return;
    }

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Edit);
    if (input_peer == nullptr) {
      on_error(0, Status::Error(400, "Can't access the chat"));
      stop();
      return;
    }

    if (reply_markup != nullptr) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_REPLY_MARKUP;
    }
    if (!entities.empty()) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_ENTITIES;
    }
    if (!text.empty()) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_MESSAGE;
    }
    if (input_media != nullptr) {
      flags |= telegram_api::messages_editMessage::MEDIA_MASK;
    }
    if (schedule_date != 0) {
      flags |= telegram_api::messages_editMessage::SCHEDULE_DATE_MASK;
    }
    LOG(DEBUG) << "Edit message with flags " << flags;

    int32 server_message_id = schedule_date != 0 ? message_id.get_scheduled_server_message_id().get()
                                                 : message_id.get_server_message_id().get();
    auto query = G()->net_query_creator().create(telegram_api::messages_editMessage(
        flags, false /*ignored*/, std::move(input_peer), server_message_id, text, std::move(input_media),
        std::move(reply_markup), std::move(entities), schedule_date));

    query->debug("send to MessagesManager::MultiSequenceDispatcher");
    send_closure(td->messages_manager_->sequence_dispatcher_, &MultiSequenceDispatcher::send_with_callback,
                 std::move(query), actor_shared(this), sequence_dispatcher_id);
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_editMessage>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for EditMessageActor: " << to_string(ptr);
    auto pts = td->updates_manager_->get_update_edit_message_pts(ptr.get());
    auto promise = PromiseCreator::lambda(
        [promise = std::move(promise_), pts](Result<Unit> result) mutable { promise.set_value(std::move(pts)); });
    td->updates_manager_->on_get_updates(std::move(ptr), std::move(promise));
  }

  void on_error(uint64 id, Status status) final {
    LOG(INFO) << "Receive error for EditMessage: " << status;
    if (!td->auth_manager_->is_bot() && status.message() == "MESSAGE_NOT_MODIFIED") {
      return promise_.set_value(0);
    }
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "EditMessageActor");
    promise_.set_error(std::move(status));
  }
};

class EditInlineMessageQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit EditInlineMessageQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(int32 flags, tl_object_ptr<telegram_api::InputBotInlineMessageID> input_bot_inline_message_id,
            const string &text, vector<tl_object_ptr<telegram_api::MessageEntity>> &&entities,
            tl_object_ptr<telegram_api::InputMedia> &&input_media,
            tl_object_ptr<telegram_api::ReplyMarkup> &&reply_markup) {
    CHECK(input_bot_inline_message_id != nullptr);

    // file in an inline message can't be uploaded to another datacenter,
    // so only previously uploaded files or URLs can be used in the InputMedia
    CHECK(!FileManager::extract_was_uploaded(input_media));

    if (reply_markup != nullptr) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_REPLY_MARKUP;
    }
    if (!entities.empty()) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_ENTITIES;
    }
    if (!text.empty()) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_MESSAGE;
    }
    if (input_media != nullptr) {
      flags |= telegram_api::messages_editInlineBotMessage::MEDIA_MASK;
    }
    LOG(DEBUG) << "Edit inline message with flags " << flags;

    auto dc_id = DcId::internal(InlineQueriesManager::get_inline_message_dc_id(input_bot_inline_message_id));
    send_query(G()->net_query_creator().create(
        telegram_api::messages_editInlineBotMessage(flags, false /*ignored*/, std::move(input_bot_inline_message_id),
                                                    text, std::move(input_media), std::move(reply_markup),
                                                    std::move(entities)),
        dc_id));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_editInlineBotMessage>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    LOG_IF(ERROR, !result_ptr.ok()) << "Receive false in result of editInlineMessage";

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    LOG(INFO) << "Receive error for EditInlineMessageQuery: " << status;
    promise_.set_error(std::move(status));
  }
};

class ForwardMessagesActor final : public NetActorOnce {
  Promise<Unit> promise_;
  vector<int64> random_ids_;
  DialogId to_dialog_id_;

 public:
  explicit ForwardMessagesActor(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(int32 flags, DialogId to_dialog_id, DialogId from_dialog_id, const vector<MessageId> &message_ids,
            vector<int64> &&random_ids, int32 schedule_date, uint64 sequence_dispatcher_id) {
    LOG(INFO) << "Forward " << format::as_array(message_ids) << " from " << from_dialog_id << " to " << to_dialog_id;

    random_ids_ = random_ids;
    to_dialog_id_ = to_dialog_id;

    auto to_input_peer = td->messages_manager_->get_input_peer(to_dialog_id, AccessRights::Write);
    if (to_input_peer == nullptr) {
      on_error(0, Status::Error(400, "Have no write access to the chat"));
      stop();
      return;
    }

    auto from_input_peer = td->messages_manager_->get_input_peer(from_dialog_id, AccessRights::Read);
    if (from_input_peer == nullptr) {
      on_error(0, Status::Error(400, "Can't access the chat to forward messages from"));
      stop();
      return;
    }

    auto query = G()->net_query_creator().create(telegram_api::messages_forwardMessages(
        flags, false /*ignored*/, false /*ignored*/, false /*ignored*/, false /*ignored*/, false /*ignored*/,
        std::move(from_input_peer), MessagesManager::get_server_message_ids(message_ids), std::move(random_ids),
        std::move(to_input_peer), schedule_date));
    if (G()->shared_config().get_option_boolean("use_quick_ack")) {
      query->quick_ack_promise_ = PromiseCreator::lambda(
          [random_ids = random_ids_](Unit) {
            for (auto random_id : random_ids) {
              send_closure(G()->messages_manager(), &MessagesManager::on_send_message_get_quick_ack, random_id);
            }
          },
          PromiseCreator::Ignore());
    }
    send_closure(td->messages_manager_->sequence_dispatcher_, &MultiSequenceDispatcher::send_with_callback,
                 std::move(query), actor_shared(this), sequence_dispatcher_id);
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_forwardMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for ForwardMessages for " << format::as_array(random_ids_) << ": " << to_string(ptr);
    auto sent_random_ids = UpdatesManager::get_sent_messages_random_ids(ptr.get());
    bool is_result_wrong = false;
    auto sent_random_ids_size = sent_random_ids.size();
    for (auto &random_id : random_ids_) {
      auto it = sent_random_ids.find(random_id);
      if (it == sent_random_ids.end()) {
        if (random_ids_.size() == 1) {
          is_result_wrong = true;
        }
        td->messages_manager_->on_send_message_fail(random_id, Status::Error(400, "Message was not forwarded"));
      } else {
        sent_random_ids.erase(it);
      }
    }
    if (!sent_random_ids.empty()) {
      is_result_wrong = true;
    }
    if (!is_result_wrong) {
      auto sent_messages = UpdatesManager::get_new_messages(ptr.get());
      if (sent_random_ids_size != sent_messages.size()) {
        is_result_wrong = true;
      }
      for (auto &sent_message : sent_messages) {
        if (MessagesManager::get_message_dialog_id(*sent_message) != to_dialog_id_) {
          is_result_wrong = true;
        }
      }
    }
    if (is_result_wrong) {
      LOG(ERROR) << "Receive wrong result for forwarding messages with random_ids " << format::as_array(random_ids_)
                 << " to " << to_dialog_id_ << ": " << oneline(to_string(ptr));
      td->updates_manager_->schedule_get_difference("Wrong forwardMessages result");
    }

    td->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    LOG(INFO) << "Receive error for forward messages: " << status;
    if (G()->close_flag() && G()->parameters().use_message_db) {
      // do not send error, messages should be re-sent
      return;
    }
    // no on_get_dialog_error call, because two dialogs are involved
    for (auto &random_id : random_ids_) {
      td->messages_manager_->on_send_message_fail(random_id, status.clone());
    }
    promise_.set_error(std::move(status));
  }
};

class SendScreenshotNotificationQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  int64 random_id_;
  DialogId dialog_id_;

 public:
  explicit SendScreenshotNotificationQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, int64 random_id) {
    random_id_ = random_id;
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    CHECK(input_peer != nullptr);

    auto query = G()->net_query_creator().create(
        telegram_api::messages_sendScreenshotNotification(std::move(input_peer), 0, random_id));
    send_query(std::move(query));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_sendScreenshotNotification>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SendScreenshotNotificationQuery for " << random_id_ << ": " << to_string(ptr);
    td->messages_manager_->check_send_message_result(random_id_, dialog_id_, ptr.get(),
                                                     "SendScreenshotNotificationQuery");
    td->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    LOG(INFO) << "Receive error for SendScreenshotNotificationQuery: " << status;
    if (G()->close_flag() && G()->parameters().use_message_db) {
      // do not send error, messages should be re-sent
      return;
    }
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SendScreenshotNotificationQuery");
    td->messages_manager_->on_send_message_fail(random_id_, status.clone());
    promise_.set_error(std::move(status));
  }
};

class SetTypingQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  int32 generation_ = 0;

 public:
  explicit SetTypingQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  NetQueryRef send(DialogId dialog_id, tl_object_ptr<telegram_api::InputPeer> &&input_peer, MessageId message_id,
                   tl_object_ptr<telegram_api::SendMessageAction> &&action) {
    dialog_id_ = dialog_id;
    CHECK(input_peer != nullptr);

    int32 flags = 0;
    if (message_id.is_valid()) {
      flags |= telegram_api::messages_setTyping::TOP_MSG_ID_MASK;
    }
    auto net_query = G()->net_query_creator().create(telegram_api::messages_setTyping(
        flags, std::move(input_peer), message_id.get_server_message_id().get(), std::move(action)));
    auto result = net_query.get_weak();
    generation_ = result.generation();
    send_query(std::move(net_query));
    return result;
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_setTyping>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    // ignore result
    promise_.set_value(Unit());

    send_closure_later(G()->messages_manager(), &MessagesManager::after_set_typing_query, dialog_id_, generation_);
  }

  void on_error(uint64 id, Status status) final {
    if (status.code() == NetQuery::Canceled) {
      return promise_.set_value(Unit());
    }

    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SetTypingQuery")) {
      LOG(INFO) << "Receive error for set typing: " << status;
    }
    promise_.set_error(std::move(status));

    send_closure_later(G()->messages_manager(), &MessagesManager::after_set_typing_query, dialog_id_, generation_);
  }
};

class DeleteMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  int32 query_count_;

 public:
  explicit DeleteMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, vector<MessageId> &&message_ids, bool revoke) {
    LOG(INFO) << "Send deleteMessagesQuery to delete " << format::as_array(message_ids) << " in " << dialog_id;
    dialog_id_ = dialog_id;
    int32 flags = 0;
    if (revoke) {
      flags |= telegram_api::messages_deleteMessages::REVOKE_MASK;
    }

    query_count_ = 0;
    auto server_message_ids = MessagesManager::get_server_message_ids(message_ids);
    const size_t MAX_SLICE_SIZE = 100;
    for (size_t i = 0; i < server_message_ids.size(); i += MAX_SLICE_SIZE) {
      auto end_i = i + MAX_SLICE_SIZE;
      auto end = end_i < server_message_ids.size() ? server_message_ids.begin() + end_i : server_message_ids.end();
      vector<int32> slice(server_message_ids.begin() + i, end);

      query_count_++;
      send_query(G()->net_query_creator().create(
          telegram_api::messages_deleteMessages(flags, false /*ignored*/, std::move(slice))));
    }
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_deleteMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto affected_messages = result_ptr.move_as_ok();
    CHECK(affected_messages->get_id() == telegram_api::messages_affectedMessages::ID);

    if (affected_messages->pts_count_ > 0) {
      td->updates_manager_->add_pending_pts_update(make_tl_object<dummyUpdate>(), affected_messages->pts_,
                                                   affected_messages->pts_count_, Time::now(), Promise<Unit>(),
                                                   "delete messages query");
    }
    if (--query_count_ == 0) {
      promise_.set_value(Unit());
    }
  }

  void on_error(uint64 id, Status status) final {
    if (!G()->is_expected_error(status)) {
      // MESSAGE_DELETE_FORBIDDEN can be returned in group chats when administrator rights was removed
      // MESSAGE_DELETE_FORBIDDEN can be returned in private chats for bots when revoke time limit exceeded
      if (status.message() != "MESSAGE_DELETE_FORBIDDEN" ||
          (dialog_id_.get_type() == DialogType::User && !td->auth_manager_->is_bot())) {
        LOG(ERROR) << "Receive error for delete messages: " << status;
      }
    }
    promise_.set_error(std::move(status));
  }
};

class DeleteChannelMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  int32 query_count_;
  ChannelId channel_id_;

 public:
  explicit DeleteChannelMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, vector<MessageId> &&message_ids) {
    channel_id_ = channel_id;
    LOG(INFO) << "Send deleteChannelMessagesQuery to delete " << format::as_array(message_ids) << " in the "
              << channel_id;

    query_count_ = 0;
    auto server_message_ids = MessagesManager::get_server_message_ids(message_ids);
    const size_t MAX_SLICE_SIZE = 100;
    for (size_t i = 0; i < server_message_ids.size(); i += MAX_SLICE_SIZE) {
      auto end_i = i + MAX_SLICE_SIZE;
      auto end = end_i < server_message_ids.size() ? server_message_ids.begin() + end_i : server_message_ids.end();
      vector<int32> slice(server_message_ids.begin() + i, end);

      query_count_++;
      auto input_channel = td->contacts_manager_->get_input_channel(channel_id);
      CHECK(input_channel != nullptr);
      send_query(G()->net_query_creator().create(
          telegram_api::channels_deleteMessages(std::move(input_channel), std::move(slice))));
    }
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::channels_deleteMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto affected_messages = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for DeleteChannelMessagesQuery: " << to_string(affected_messages);
    CHECK(affected_messages->get_id() == telegram_api::messages_affectedMessages::ID);

    if (affected_messages->pts_count_ > 0) {
      td->messages_manager_->add_pending_channel_update(DialogId(channel_id_), make_tl_object<dummyUpdate>(),
                                                        affected_messages->pts_, affected_messages->pts_count_,
                                                        Promise<Unit>(), "DeleteChannelMessagesQuery");
    }
    if (--query_count_ == 0) {
      promise_.set_value(Unit());
    }
  }

  void on_error(uint64 id, Status status) final {
    if (!td->contacts_manager_->on_get_channel_error(channel_id_, status, "DeleteChannelMessagesQuery") &&
        status.message() != "MESSAGE_DELETE_FORBIDDEN") {
      LOG(ERROR) << "Receive error for delete channel messages: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class DeleteScheduledMessagesQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit DeleteScheduledMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, vector<MessageId> &&message_ids) {
    dialog_id_ = dialog_id;
    LOG(INFO) << "Send deleteScheduledMessagesQuery to delete " << format::as_array(message_ids);

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return on_error(0, Status::Error(400, "Can't access the chat"));
    }
    send_query(G()->net_query_creator().create(telegram_api::messages_deleteScheduledMessages(
        std::move(input_peer), MessagesManager::get_scheduled_server_message_ids(message_ids))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_deleteScheduledMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for DeleteScheduledMessagesQuery: " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "DeleteScheduledMessagesQuery")) {
      LOG(ERROR) << "Receive error for delete scheduled messages: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class GetDialogNotifySettingsQuery final : public Td::ResultHandler {
  DialogId dialog_id_;

 public:
  void send(DialogId dialog_id) {
    dialog_id_ = dialog_id;
    auto input_notify_peer = td->messages_manager_->get_input_notify_peer(dialog_id);
    CHECK(input_notify_peer != nullptr);
    send_query(G()->net_query_creator().create(telegram_api::account_getNotifySettings(std::move(input_notify_peer))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::account_getNotifySettings>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    td->messages_manager_->on_update_dialog_notify_settings(dialog_id_, std::move(ptr), "GetDialogNotifySettingsQuery");
    td->messages_manager_->on_get_dialog_notification_settings_query_finished(dialog_id_, Status::OK());
  }

  void on_error(uint64 id, Status status) final {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetDialogNotifySettingsQuery");
    td->messages_manager_->on_get_dialog_notification_settings_query_finished(dialog_id_, std::move(status));
  }
};

class GetNotifySettingsExceptionsQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit GetNotifySettingsExceptionsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(NotificationSettingsScope scope, bool filter_scope, bool compare_sound) {
    int32 flags = 0;
    tl_object_ptr<telegram_api::InputNotifyPeer> input_notify_peer;
    if (filter_scope) {
      flags |= telegram_api::account_getNotifyExceptions::PEER_MASK;
      input_notify_peer = get_input_notify_peer(scope);
    }
    if (compare_sound) {
      flags |= telegram_api::account_getNotifyExceptions::COMPARE_SOUND_MASK;
    }
    send_query(G()->net_query_creator().create(
        telegram_api::account_getNotifyExceptions(flags, false /* ignored */, std::move(input_notify_peer))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::account_getNotifyExceptions>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto updates_ptr = result_ptr.move_as_ok();
    auto dialog_ids = UpdatesManager::get_update_notify_settings_dialog_ids(updates_ptr.get());
    vector<tl_object_ptr<telegram_api::User>> users;
    vector<tl_object_ptr<telegram_api::Chat>> chats;
    switch (updates_ptr->get_id()) {
      case telegram_api::updatesCombined::ID: {
        auto updates = static_cast<telegram_api::updatesCombined *>(updates_ptr.get());
        users = std::move(updates->users_);
        chats = std::move(updates->chats_);
        reset_to_empty(updates->users_);
        reset_to_empty(updates->chats_);
        break;
      }
      case telegram_api::updates::ID: {
        auto updates = static_cast<telegram_api::updates *>(updates_ptr.get());
        users = std::move(updates->users_);
        chats = std::move(updates->chats_);
        reset_to_empty(updates->users_);
        reset_to_empty(updates->chats_);
        break;
      }
    }
    td->contacts_manager_->on_get_users(std::move(users), "GetNotifySettingsExceptionsQuery");
    td->contacts_manager_->on_get_chats(std::move(chats), "GetNotifySettingsExceptionsQuery");
    for (auto &dialog_id : dialog_ids) {
      td->messages_manager_->force_create_dialog(dialog_id, "GetNotifySettingsExceptionsQuery");
    }
    td->updates_manager_->on_get_updates(std::move(updates_ptr), std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    promise_.set_error(std::move(status));
  }
};

class GetScopeNotifySettingsQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  NotificationSettingsScope scope_;

 public:
  explicit GetScopeNotifySettingsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(NotificationSettingsScope scope) {
    scope_ = scope;
    auto input_notify_peer = get_input_notify_peer(scope);
    CHECK(input_notify_peer != nullptr);
    send_query(G()->net_query_creator().create(telegram_api::account_getNotifySettings(std::move(input_notify_peer))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::account_getNotifySettings>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    td->messages_manager_->on_update_scope_notify_settings(scope_, std::move(ptr));

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    promise_.set_error(std::move(status));
  }
};

class UpdateDialogNotifySettingsQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit UpdateDialogNotifySettingsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, const DialogNotificationSettings &new_settings) {
    dialog_id_ = dialog_id;

    auto input_notify_peer = td->messages_manager_->get_input_notify_peer(dialog_id);
    if (input_notify_peer == nullptr) {
      return on_error(0, Status::Error(500, "Can't update chat notification settings"));
    }

    int32 flags = 0;
    if (!new_settings.use_default_mute_until) {
      flags |= telegram_api::inputPeerNotifySettings::MUTE_UNTIL_MASK;
    }
    if (!new_settings.use_default_sound) {
      flags |= telegram_api::inputPeerNotifySettings::SOUND_MASK;
    }
    if (!new_settings.use_default_show_preview) {
      flags |= telegram_api::inputPeerNotifySettings::SHOW_PREVIEWS_MASK;
    }
    if (new_settings.silent_send_message) {
      flags |= telegram_api::inputPeerNotifySettings::SILENT_MASK;
    }
    send_query(G()->net_query_creator().create(telegram_api::account_updateNotifySettings(
        std::move(input_notify_peer), make_tl_object<telegram_api::inputPeerNotifySettings>(
                                          flags, new_settings.show_preview, new_settings.silent_send_message,
                                          new_settings.mute_until, new_settings.sound))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::account_updateNotifySettings>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    if (!result) {
      return on_error(id, Status::Error(400, "Receive false as result"));
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "UpdateDialogNotifySettingsQuery")) {
      LOG(INFO) << "Receive error for set chat notification settings: " << status;
    }

    if (!td->auth_manager_->is_bot() && td->messages_manager_->get_input_notify_peer(dialog_id_) != nullptr) {
      // trying to repair notification settings for this dialog
      td->messages_manager_->send_get_dialog_notification_settings_query(dialog_id_, Promise<>());
    }

    promise_.set_error(std::move(status));
  }
};

class UpdateScopeNotifySettingsQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  NotificationSettingsScope scope_;

 public:
  explicit UpdateScopeNotifySettingsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(NotificationSettingsScope scope, const ScopeNotificationSettings &new_settings) {
    auto input_notify_peer = get_input_notify_peer(scope);
    CHECK(input_notify_peer != nullptr);
    int32 flags = telegram_api::inputPeerNotifySettings::MUTE_UNTIL_MASK |
                  telegram_api::inputPeerNotifySettings::SOUND_MASK |
                  telegram_api::inputPeerNotifySettings::SHOW_PREVIEWS_MASK;
    send_query(G()->net_query_creator().create(telegram_api::account_updateNotifySettings(
        std::move(input_notify_peer),
        make_tl_object<telegram_api::inputPeerNotifySettings>(flags, new_settings.show_preview, false,
                                                              new_settings.mute_until, new_settings.sound))));
    scope_ = scope;
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::account_updateNotifySettings>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    if (!result) {
      return on_error(id, Status::Error(400, "Receive false as result"));
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    LOG(INFO) << "Receive error for set notification settings: " << status;

    if (!td->auth_manager_->is_bot()) {
      // trying to repair notification settings for this scope
      td->messages_manager_->send_get_scope_notification_settings_query(scope_, Promise<>());
    }

    promise_.set_error(std::move(status));
  }
};

class ResetNotifySettingsQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit ResetNotifySettingsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send() {
    send_query(G()->net_query_creator().create(telegram_api::account_resetNotifySettings()));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::account_resetNotifySettings>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    if (!result) {
      return on_error(id, Status::Error(400, "Receive false as result"));
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    if (!G()->is_expected_error(status)) {
      LOG(ERROR) << "Receive error for reset notification settings: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class GetPeerSettingsQuery final : public Td::ResultHandler {
  DialogId dialog_id_;

 public:
  void send(DialogId dialog_id) {
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);

    send_query(G()->net_query_creator().create(telegram_api::messages_getPeerSettings(std::move(input_peer))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_getPeerSettings>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    td->messages_manager_->on_get_peer_settings(dialog_id_, result_ptr.move_as_ok());
  }

  void on_error(uint64 id, Status status) final {
    LOG(INFO) << "Receive error for get peer settings: " << status;
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetPeerSettingsQuery");
  }
};

class UpdatePeerSettingsQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit UpdatePeerSettingsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, bool is_spam_dialog) {
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_value(Unit());
    }

    if (is_spam_dialog) {
      send_query(G()->net_query_creator().create(telegram_api::messages_reportSpam(std::move(input_peer))));
    } else {
      send_query(G()->net_query_creator().create(telegram_api::messages_hidePeerSettingsBar(std::move(input_peer))));
    }
  }

  void on_result(uint64 id, BufferSlice packet) final {
    static_assert(std::is_same<telegram_api::messages_reportSpam::ReturnType,
                               telegram_api::messages_hidePeerSettingsBar::ReturnType>::value,
                  "");
    auto result_ptr = fetch_result<telegram_api::messages_reportSpam>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    td->messages_manager_->on_get_peer_settings(dialog_id_, make_tl_object<telegram_api::peerSettings>(), true);

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    LOG(INFO) << "Receive error for update peer settings: " << status;
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "UpdatePeerSettingsQuery");
    td->messages_manager_->reget_dialog_action_bar(dialog_id_, "UpdatePeerSettingsQuery");
    promise_.set_error(std::move(status));
  }
};

class ReportEncryptedSpamQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit ReportEncryptedSpamQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id) {
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_encrypted_chat(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);
    LOG(INFO) << "Report spam in " << to_string(input_peer);

    send_query(G()->net_query_creator().create(telegram_api::messages_reportEncryptedSpam(std::move(input_peer))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::messages_reportEncryptedSpam>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    td->messages_manager_->on_get_peer_settings(dialog_id_, make_tl_object<telegram_api::peerSettings>(), true);

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    LOG(INFO) << "Receive error for report encrypted spam: " << status;
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "ReportEncryptedSpamQuery");
    td->messages_manager_->reget_dialog_action_bar(
        DialogId(td->contacts_manager_->get_secret_chat_user_id(dialog_id_.get_secret_chat_id())),
        "ReportEncryptedSpamQuery");
    promise_.set_error(std::move(status));
  }
};

class ReportPeerQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit ReportPeerQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, const vector<MessageId> &message_ids, ReportReason &&report_reason) {
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);

    if (message_ids.empty()) {
      send_query(G()->net_query_creator().create(telegram_api::account_reportPeer(
          std::move(input_peer), report_reason.get_input_report_reason(), report_reason.get_message())));
    } else {
      send_query(G()->net_query_creator().create(
          telegram_api::messages_report(std::move(input_peer), MessagesManager::get_server_message_ids(message_ids),
                                        report_reason.get_input_report_reason(), report_reason.get_message())));
    }
  }

  void on_result(uint64 id, BufferSlice packet) final {
    static_assert(
        std::is_same<telegram_api::account_reportPeer::ReturnType, telegram_api::messages_report::ReturnType>::value,
        "");
    auto result_ptr = fetch_result<telegram_api::account_reportPeer>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    if (!result) {
      return on_error(id, Status::Error(400, "Receive false as result"));
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    LOG(INFO) << "Receive error for report peer: " << status;
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "ReportPeerQuery");
    td->messages_manager_->reget_dialog_action_bar(dialog_id_, "ReportPeerQuery");
    promise_.set_error(std::move(status));
  }
};

class ReportProfilePhotoQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  FileId file_id_;
  string file_reference_;
  ReportReason report_reason_;

 public:
  explicit ReportProfilePhotoQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, FileId file_id, tl_object_ptr<telegram_api::InputPhoto> &&input_photo,
            ReportReason &&report_reason) {
    dialog_id_ = dialog_id;
    file_id_ = file_id;
    file_reference_ = FileManager::extract_file_reference(input_photo);
    report_reason_ = std::move(report_reason);

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);

    send_query(G()->net_query_creator().create(telegram_api::account_reportProfilePhoto(
        std::move(input_peer), std::move(input_photo), report_reason_.get_input_report_reason(),
        report_reason_.get_message())));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::account_reportProfilePhoto>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    if (!result) {
      return on_error(id, Status::Error(400, "Receive false as result"));
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    LOG(INFO) << "Receive error for report chat photo: " << status;
    if (!td->auth_manager_->is_bot() && FileReferenceManager::is_file_reference_error(status)) {
      VLOG(file_references) << "Receive " << status << " for " << file_id_;
      td->file_manager_->delete_file_reference(file_id_, file_reference_);
      td->file_reference_manager_->repair_file_reference(
          file_id_,
          PromiseCreator::lambda([dialog_id = dialog_id_, file_id = file_id_, report_reason = std::move(report_reason_),
                                  promise = std::move(promise_)](Result<Unit> result) mutable {
            if (result.is_error()) {
              LOG(INFO) << "Reported photo " << file_id << " is likely to be deleted";
              return promise.set_value(Unit());
            }
            send_closure(G()->messages_manager(), &MessagesManager::report_dialog_photo, dialog_id, file_id,
                         std::move(report_reason), std::move(promise));
          }));
      return;
    }

    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "ReportProfilePhotoQuery");
    promise_.set_error(std::move(status));
  }
};

class EditPeerFoldersQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit EditPeerFoldersQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, FolderId folder_id) {
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);

    vector<telegram_api::object_ptr<telegram_api::inputFolderPeer>> input_folder_peers;
    input_folder_peers.push_back(
        telegram_api::make_object<telegram_api::inputFolderPeer>(std::move(input_peer), folder_id.get()));
    send_query(G()->net_query_creator().create(telegram_api::folders_editPeerFolders(std::move(input_folder_peers))));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::folders_editPeerFolders>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for EditPeerFoldersQuery: " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr), std::move(promise_));
  }

  void on_error(uint64 id, Status status) final {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "EditPeerFoldersQuery")) {
      LOG(INFO) << "Receive error for EditPeerFoldersQuery: " << status;
    }

    // trying to repair folder ID for this dialog
    td->messages_manager_->get_dialog_info_full(dialog_id_, Auto(), "EditPeerFoldersQuery");

    promise_.set_error(std::move(status));
  }
};

class GetChannelDifferenceQuery final : public Td::ResultHandler {
  DialogId dialog_id_;
  int32 pts_;
  int32 limit_;

 public:
  void send(DialogId dialog_id, tl_object_ptr<telegram_api::InputChannel> &&input_channel, int32 pts, int32 limit,
            bool force) {
    CHECK(pts >= 0);
    dialog_id_ = dialog_id;
    pts_ = pts;
    limit_ = limit;
    CHECK(input_channel != nullptr);

    int32 flags = 0;
    if (force) {
      flags |= telegram_api::updates_getChannelDifference::FORCE_MASK;
    }
    send_query(G()->net_query_creator().create(telegram_api::updates_getChannelDifference(
        flags, false /*ignored*/, std::move(input_channel), make_tl_object<telegram_api::channelMessagesFilterEmpty>(),
        pts, limit)));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::updates_getChannelDifference>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    td->messages_manager_->on_get_channel_difference(dialog_id_, pts_, limit_, result_ptr.move_as_ok());
  }

  void on_error(uint64 id, Status status) final {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetChannelDifferenceQuery")) {
      LOG(ERROR) << "Receive updates.getChannelDifference error for " << dialog_id_ << " with pts " << pts_
                 << " and limit " << limit_ << ": " << status;
    }
    td->messages_manager_->on_get_channel_difference(dialog_id_, pts_, limit_, nullptr);
  }
};

class ResolveUsernameQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  string username_;

 public:
  explicit ResolveUsernameQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(const string &username) {
    username_ = username;

    LOG(INFO) << "Send ResolveUsernameQuery with username = " << username;
    send_query(G()->net_query_creator().create(telegram_api::contacts_resolveUsername(username)));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::contacts_resolveUsername>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(DEBUG) << "Receive result for ResolveUsernameQuery: " << to_string(ptr);
    td->contacts_manager_->on_get_users(std::move(ptr->users_), "ResolveUsernameQuery");
    td->contacts_manager_->on_get_chats(std::move(ptr->chats_), "ResolveUsernameQuery");

    td->messages_manager_->on_resolved_username(username_, DialogId(ptr->peer_));

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    if (status.message() == Slice("USERNAME_NOT_OCCUPIED")) {
      td->messages_manager_->drop_username(username_);
    }
    promise_.set_error(std::move(status));
  }
};

class GetChannelAdminLogQuery final : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;
  int64 random_id_;

 public:
  explicit GetChannelAdminLogQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, const string &query, int64 from_event_id, int32 limit,
            tl_object_ptr<telegram_api::channelAdminLogEventsFilter> filter,
            vector<tl_object_ptr<telegram_api::InputUser>> input_users, int64 random_id) {
    channel_id_ = channel_id;
    random_id_ = random_id;

    auto input_channel = td->contacts_manager_->get_input_channel(channel_id);
    CHECK(input_channel != nullptr);

    int32 flags = 0;
    if (filter != nullptr) {
      flags |= telegram_api::channels_getAdminLog::EVENTS_FILTER_MASK;
    }
    if (!input_users.empty()) {
      flags |= telegram_api::channels_getAdminLog::ADMINS_MASK;
    }

    send_query(G()->net_query_creator().create(telegram_api::channels_getAdminLog(
        flags, std::move(input_channel), query, std::move(filter), std::move(input_users), from_event_id, 0, limit)));
  }

  void on_result(uint64 id, BufferSlice packet) final {
    auto result_ptr = fetch_result<telegram_api::channels_getAdminLog>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    td->messages_manager_->on_get_event_log(channel_id_, random_id_, result_ptr.move_as_ok());
    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) final {
    td->contacts_manager_->on_get_channel_error(channel_id_, status, "GetChannelAdminLogQuery");
    td->messages_manager_->on_get_event_log(channel_id_, random_id_, nullptr);
    promise_.set_error(std::move(status));
  }
};

class MessagesManager::UploadMediaCallback final : public FileManager::UploadCallback {
 public:
  void on_progress(FileId file_id) final {
  }
  void on_upload_ok(FileId file_id, tl_object_ptr<telegram_api::InputFile> input_file) final {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_media, file_id, std::move(input_file),
                       nullptr);
  }
  void on_upload_encrypted_ok(FileId file_id, tl_object_ptr<telegram_api::InputEncryptedFile> input_file) final {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_media, file_id, nullptr,
                       std::move(input_file));
  }
  void on_upload_secure_ok(FileId file_id, tl_object_ptr<telegram_api::InputSecureFile> input_file) final {
    UNREACHABLE();
  }
  void on_upload_error(FileId file_id, Status error) final {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_media_error, file_id, std::move(error));
  }
};

class MessagesManager::UploadThumbnailCallback final : public FileManager::UploadCallback {
 public:
  void on_upload_ok(FileId file_id, tl_object_ptr<telegram_api::InputFile> input_file) final {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_thumbnail, file_id, std::move(input_file));
  }
  void on_upload_encrypted_ok(FileId file_id, tl_object_ptr<telegram_api::InputEncryptedFile> input_file) final {
    UNREACHABLE();
  }
  void on_upload_secure_ok(FileId file_id, tl_object_ptr<telegram_api::InputSecureFile> input_file) final {
    UNREACHABLE();
  }
  void on_upload_error(FileId file_id, Status error) final {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_thumbnail, file_id, nullptr);
  }
};

class MessagesManager::UploadDialogPhotoCallback final : public FileManager::UploadCallback {
 public:
  void on_upload_ok(FileId file_id, tl_object_ptr<telegram_api::InputFile> input_file) final {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_dialog_photo, file_id,
                       std::move(input_file));
  }
  void on_upload_encrypted_ok(FileId file_id, tl_object_ptr<telegram_api::InputEncryptedFile> input_file) final {
    UNREACHABLE();
  }
  void on_upload_secure_ok(FileId file_id, tl_object_ptr<telegram_api::InputSecureFile> input_file) final {
    UNREACHABLE();
  }
  void on_upload_error(FileId file_id, Status error) final {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_dialog_photo_error, file_id,
                       std::move(error));
  }
};

class MessagesManager::UploadImportedMessagesCallback final : public FileManager::UploadCallback {
 public:
  void on_upload_ok(FileId file_id, tl_object_ptr<telegram_api::InputFile> input_file) final {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_imported_messages, file_id,
                       std::move(input_file));
  }
  void on_upload_encrypted_ok(FileId file_id, tl_object_ptr<telegram_api::InputEncryptedFile> input_file) final {
    UNREACHABLE();
  }
  void on_upload_secure_ok(FileId file_id, tl_object_ptr<telegram_api::InputSecureFile> input_file) final {
    UNREACHABLE();
  }
  void on_upload_error(FileId file_id, Status error) final {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_imported_messages_error, file_id,
                       std::move(error));
  }
};

class MessagesManager::UploadImportedMessageAttachmentCallback final : public FileManager::UploadCallback {
 public:
  void on_upload_ok(FileId file_id, tl_object_ptr<telegram_api::InputFile> input_file) final {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_imported_message_attachment, file_id,
                       std::move(input_file));
  }
  void on_upload_encrypted_ok(FileId file_id, tl_object_ptr<telegram_api::InputEncryptedFile> input_file) final {
    UNREACHABLE();
  }
  void on_upload_secure_ok(FileId file_id, tl_object_ptr<telegram_api::InputSecureFile> input_file) final {
    UNREACHABLE();
  }
  void on_upload_error(FileId file_id, Status error) final {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_imported_message_attachment_error, file_id,
                       std::move(error));
  }
};

template <class StorerT>
void MessagesManager::Message::store(StorerT &storer) const {
  using td::store;
  bool has_sender = sender_user_id.is_valid();
  bool has_edit_date = edit_date > 0;
  bool has_random_id = random_id != 0;
  bool is_forwarded = forward_info != nullptr;
  bool is_reply = reply_to_message_id.is_valid();
  bool is_reply_to_random_id = reply_to_random_id != 0;
  bool is_via_bot = via_bot_user_id.is_valid();
  bool has_view_count = view_count > 0;
  bool has_reply_markup = reply_markup != nullptr;
  bool has_ttl = ttl != 0;
  bool has_author_signature = !author_signature.empty();
  bool has_forward_author_signature = is_forwarded && !forward_info->author_signature.empty();
  bool has_media_album_id = media_album_id != 0;
  bool has_forward_from =
      is_forwarded && (forward_info->from_dialog_id.is_valid() || forward_info->from_message_id.is_valid());
  bool has_send_date = message_id.is_yet_unsent() && send_date != 0;
  bool has_flags2 = true;
  bool has_notification_id = notification_id.is_valid();
  bool has_forward_sender_name = is_forwarded && !forward_info->sender_name.empty();
  bool has_send_error_code = send_error_code != 0;
  bool has_real_forward_from = real_forward_from_dialog_id.is_valid() && real_forward_from_message_id.is_valid();
  bool has_legacy_layer = legacy_layer != 0;
  bool has_restriction_reasons = !restriction_reasons.empty();
  bool has_forward_psa_type = is_forwarded && !forward_info->psa_type.empty();
  bool has_forward_count = forward_count > 0;
  bool has_reply_info = !reply_info.is_empty();
  bool has_sender_dialog_id = sender_dialog_id.is_valid();
  bool has_reply_in_dialog_id = is_reply && reply_in_dialog_id.is_valid();
  bool has_top_thread_message_id = top_thread_message_id.is_valid();
  bool has_thread_draft_message = thread_draft_message != nullptr;
  bool has_local_thread_message_ids = !local_thread_message_ids.empty();
  bool has_linked_top_thread_message_id = linked_top_thread_message_id.is_valid();
  bool has_interaction_info_update_date = interaction_info_update_date != 0;
  bool has_send_emoji = !send_emoji.empty();
  bool is_imported = is_forwarded && forward_info->is_imported;
  bool has_ttl_period = ttl_period != 0;
  bool has_max_reply_media_timestamp = max_reply_media_timestamp >= 0;
  bool are_message_media_timestamp_entities_found = true;
  bool has_flags3 = false;
  BEGIN_STORE_FLAGS();
  STORE_FLAG(is_channel_post);
  STORE_FLAG(is_outgoing);
  STORE_FLAG(is_failed_to_send);
  STORE_FLAG(disable_notification);
  STORE_FLAG(contains_mention);
  STORE_FLAG(from_background);
  STORE_FLAG(disable_web_page_preview);
  STORE_FLAG(clear_draft);
  STORE_FLAG(have_previous);
  STORE_FLAG(have_next);
  STORE_FLAG(has_sender);
  STORE_FLAG(has_edit_date);
  STORE_FLAG(has_random_id);
  STORE_FLAG(is_forwarded);
  STORE_FLAG(is_reply);
  STORE_FLAG(is_reply_to_random_id);
  STORE_FLAG(is_via_bot);
  STORE_FLAG(has_view_count);
  STORE_FLAG(has_reply_markup);
  STORE_FLAG(has_ttl);
  STORE_FLAG(has_author_signature);
  STORE_FLAG(has_forward_author_signature);
  STORE_FLAG(had_reply_markup);
  STORE_FLAG(contains_unread_mention);
  STORE_FLAG(has_media_album_id);
  STORE_FLAG(has_forward_from);
  STORE_FLAG(in_game_share);
  STORE_FLAG(is_content_secret);
  STORE_FLAG(has_send_date);
  STORE_FLAG(has_flags2);
  END_STORE_FLAGS();
  if (has_flags2) {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(has_notification_id);
    STORE_FLAG(is_mention_notification_disabled);
    STORE_FLAG(had_forward_info);
    STORE_FLAG(has_forward_sender_name);
    STORE_FLAG(has_send_error_code);
    STORE_FLAG(hide_via_bot);
    STORE_FLAG(is_bot_start_message);
    STORE_FLAG(has_real_forward_from);
    STORE_FLAG(has_legacy_layer);
    STORE_FLAG(hide_edit_date);
    STORE_FLAG(has_restriction_reasons);
    STORE_FLAG(is_from_scheduled);
    STORE_FLAG(is_copy);
    STORE_FLAG(has_forward_psa_type);
    STORE_FLAG(has_forward_count);
    STORE_FLAG(has_reply_info);
    STORE_FLAG(has_sender_dialog_id);
    STORE_FLAG(has_reply_in_dialog_id);
    STORE_FLAG(has_top_thread_message_id);
    STORE_FLAG(has_thread_draft_message);
    STORE_FLAG(has_local_thread_message_ids);
    STORE_FLAG(has_linked_top_thread_message_id);
    STORE_FLAG(is_pinned);
    STORE_FLAG(has_interaction_info_update_date);
    STORE_FLAG(has_send_emoji);
    STORE_FLAG(is_imported);
    STORE_FLAG(has_ttl_period);
    STORE_FLAG(has_max_reply_media_timestamp);
    STORE_FLAG(are_message_media_timestamp_entities_found);
    STORE_FLAG(has_flags3);
    END_STORE_FLAGS();
  }

  store(message_id, storer);
  if (has_sender) {
    store(sender_user_id, storer);
  }
  store(date, storer);
  if (has_edit_date) {
    store(edit_date, storer);
  }
  if (has_send_date) {
    store(send_date, storer);
  }
  if (has_random_id) {
    store(random_id, storer);
  }
  if (is_forwarded) {
    store(forward_info->sender_user_id, storer);
    store(forward_info->date, storer);
    store(forward_info->sender_dialog_id, storer);
    store(forward_info->message_id, storer);
    if (has_forward_author_signature) {
      store(forward_info->author_signature, storer);
    }
    if (has_forward_sender_name) {
      store(forward_info->sender_name, storer);
    }
    if (has_forward_from) {
      store(forward_info->from_dialog_id, storer);
      store(forward_info->from_message_id, storer);
    }
    if (has_forward_psa_type) {
      store(forward_info->psa_type, storer);
    }
  }
  if (has_real_forward_from) {
    store(real_forward_from_dialog_id, storer);
    store(real_forward_from_message_id, storer);
  }
  if (is_reply) {
    store(reply_to_message_id, storer);
  }
  if (is_reply_to_random_id) {
    store(reply_to_random_id, storer);
  }
  if (is_via_bot) {
    store(via_bot_user_id, storer);
  }
  if (has_view_count) {
    store(view_count, storer);
  }
  if (has_forward_count) {
    store(forward_count, storer);
  }
  if (has_reply_info) {
    store(reply_info, storer);
  }
  if (has_ttl) {
    store(ttl, storer);
    store_time(ttl_expires_at, storer);
  }
  if (has_send_error_code) {
    store(send_error_code, storer);
    store(send_error_message, storer);
    if (send_error_code == 429) {
      store_time(try_resend_at, storer);
    }
  }
  if (has_author_signature) {
    store(author_signature, storer);
  }
  if (has_media_album_id) {
    store(media_album_id, storer);
  }
  if (has_notification_id) {
    store(notification_id, storer);
  }
  if (has_legacy_layer) {
    store(legacy_layer, storer);
  }
  if (has_restriction_reasons) {
    store(restriction_reasons, storer);
  }
  if (has_sender_dialog_id) {
    store(sender_dialog_id, storer);
  }
  if (has_reply_in_dialog_id) {
    store(reply_in_dialog_id, storer);
  }
  if (has_top_thread_message_id) {
    store(top_thread_message_id, storer);
  }
  if (has_thread_draft_message) {
    store(thread_draft_message, storer);
  }
  if (has_local_thread_message_ids) {
    store(local_thread_message_ids, storer);
  }
  if (has_linked_top_thread_message_id) {
    store(linked_top_thread_message_id, storer);
  }
  if (has_interaction_info_update_date) {
    store(interaction_info_update_date, storer);
  }
  if (has_send_emoji) {
    store(send_emoji, storer);
  }
  store_message_content(content.get(), storer);
  if (has_reply_markup) {
    store(reply_markup, storer);
  }
  if (has_ttl_period) {
    store(ttl_period, storer);
  }
  if (has_max_reply_media_timestamp) {
    store(max_reply_media_timestamp, storer);
  }
}

// do not forget to resolve message dependencies
template <class ParserT>
void MessagesManager::Message::parse(ParserT &parser) {
  using td::parse;
  bool has_sender;
  bool has_edit_date;
  bool has_random_id;
  bool is_forwarded;
  bool is_reply;
  bool is_reply_to_random_id;
  bool is_via_bot;
  bool has_view_count;
  bool has_reply_markup;
  bool has_ttl;
  bool has_author_signature;
  bool has_forward_author_signature;
  bool has_media_album_id;
  bool has_forward_from;
  bool has_send_date;
  bool has_flags2;
  bool has_notification_id = false;
  bool has_forward_sender_name = false;
  bool has_send_error_code = false;
  bool has_real_forward_from = false;
  bool has_legacy_layer = false;
  bool has_restriction_reasons = false;
  bool has_forward_psa_type = false;
  bool has_forward_count = false;
  bool has_reply_info = false;
  bool has_sender_dialog_id = false;
  bool has_reply_in_dialog_id = false;
  bool has_top_thread_message_id = false;
  bool has_thread_draft_message = false;
  bool has_local_thread_message_ids = false;
  bool has_linked_top_thread_message_id = false;
  bool has_interaction_info_update_date = false;
  bool has_send_emoji = false;
  bool is_imported = false;
  bool has_ttl_period = false;
  bool has_max_reply_media_timestamp = false;
  bool has_flags3 = false;
  BEGIN_PARSE_FLAGS();
  PARSE_FLAG(is_channel_post);
  PARSE_FLAG(is_outgoing);
  PARSE_FLAG(is_failed_to_send);
  PARSE_FLAG(disable_notification);
  PARSE_FLAG(contains_mention);
  PARSE_FLAG(from_background);
  PARSE_FLAG(disable_web_page_preview);
  PARSE_FLAG(clear_draft);
  PARSE_FLAG(have_previous);
  PARSE_FLAG(have_next);
  PARSE_FLAG(has_sender);
  PARSE_FLAG(has_edit_date);
  PARSE_FLAG(has_random_id);
  PARSE_FLAG(is_forwarded);
  PARSE_FLAG(is_reply);
  PARSE_FLAG(is_reply_to_random_id);
  PARSE_FLAG(is_via_bot);
  PARSE_FLAG(has_view_count);
  PARSE_FLAG(has_reply_markup);
  PARSE_FLAG(has_ttl);
  PARSE_FLAG(has_author_signature);
  PARSE_FLAG(has_forward_author_signature);
  PARSE_FLAG(had_reply_markup);
  PARSE_FLAG(contains_unread_mention);
  PARSE_FLAG(has_media_album_id);
  PARSE_FLAG(has_forward_from);
  PARSE_FLAG(in_game_share);
  PARSE_FLAG(is_content_secret);
  PARSE_FLAG(has_send_date);
  PARSE_FLAG(has_flags2);
  END_PARSE_FLAGS();
  if (has_flags2) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(has_notification_id);
    PARSE_FLAG(is_mention_notification_disabled);
    PARSE_FLAG(had_forward_info);
    PARSE_FLAG(has_forward_sender_name);
    PARSE_FLAG(has_send_error_code);
    PARSE_FLAG(hide_via_bot);
    PARSE_FLAG(is_bot_start_message);
    PARSE_FLAG(has_real_forward_from);
    PARSE_FLAG(has_legacy_layer);
    PARSE_FLAG(hide_edit_date);
    PARSE_FLAG(has_restriction_reasons);
    PARSE_FLAG(is_from_scheduled);
    PARSE_FLAG(is_copy);
    PARSE_FLAG(has_forward_psa_type);
    PARSE_FLAG(has_forward_count);
    PARSE_FLAG(has_reply_info);
    PARSE_FLAG(has_sender_dialog_id);
    PARSE_FLAG(has_reply_in_dialog_id);
    PARSE_FLAG(has_top_thread_message_id);
    PARSE_FLAG(has_thread_draft_message);
    PARSE_FLAG(has_local_thread_message_ids);
    PARSE_FLAG(has_linked_top_thread_message_id);
    PARSE_FLAG(is_pinned);
    PARSE_FLAG(has_interaction_info_update_date);
    PARSE_FLAG(has_send_emoji);
    PARSE_FLAG(is_imported);
    PARSE_FLAG(has_ttl_period);
    PARSE_FLAG(has_max_reply_media_timestamp);
    PARSE_FLAG(are_media_timestamp_entities_found);
    PARSE_FLAG(has_flags3);
    END_PARSE_FLAGS();
  }
  if (has_flags3) {
    BEGIN_PARSE_FLAGS();
    END_PARSE_FLAGS();
  }

  parse(message_id, parser);
  random_y = get_random_y(message_id);
  if (has_sender) {
    parse(sender_user_id, parser);
  }
  parse(date, parser);
  if (has_edit_date) {
    parse(edit_date, parser);
  }
  if (has_send_date) {
    CHECK(message_id.is_valid() || message_id.is_valid_scheduled());
    CHECK(message_id.is_yet_unsent());
    parse(send_date, parser);
  } else if (message_id.is_valid() && message_id.is_yet_unsent()) {
    send_date = date;  // for backward compatibility
  }
  if (has_random_id) {
    parse(random_id, parser);
  }
  if (is_forwarded) {
    forward_info = make_unique<MessageForwardInfo>();
    parse(forward_info->sender_user_id, parser);
    parse(forward_info->date, parser);
    parse(forward_info->sender_dialog_id, parser);
    parse(forward_info->message_id, parser);
    if (has_forward_author_signature) {
      parse(forward_info->author_signature, parser);
    }
    if (has_forward_sender_name) {
      parse(forward_info->sender_name, parser);
    }
    if (has_forward_from) {
      parse(forward_info->from_dialog_id, parser);
      parse(forward_info->from_message_id, parser);
    }
    if (has_forward_psa_type) {
      parse(forward_info->psa_type, parser);
    }
    forward_info->is_imported = is_imported;
  }
  if (has_real_forward_from) {
    parse(real_forward_from_dialog_id, parser);
    parse(real_forward_from_message_id, parser);
  }
  if (is_reply) {
    parse(reply_to_message_id, parser);
  }
  if (is_reply_to_random_id) {
    parse(reply_to_random_id, parser);
  }
  if (is_via_bot) {
    parse(via_bot_user_id, parser);
  }
  if (has_view_count) {
    parse(view_count, parser);
  }
  if (has_forward_count) {
    parse(forward_count, parser);
  }
  if (has_reply_info) {
    parse(reply_info, parser);
  }
  if (has_ttl) {
    parse(ttl, parser);
    parse_time(ttl_expires_at, parser);
  }
  if (has_send_error_code) {
    parse(send_error_code, parser);
    parse(send_error_message, parser);
    if (send_error_code == 429) {
      parse_time(try_resend_at, parser);
    }
  }
  if (has_author_signature) {
    parse(author_signature, parser);
  }
  if (has_media_album_id) {
    parse(media_album_id, parser);
  }
  if (has_notification_id) {
    parse(notification_id, parser);
  }
  if (has_legacy_layer) {
    parse(legacy_layer, parser);
  }
  if (has_restriction_reasons) {
    parse(restriction_reasons, parser);
  }
  if (has_sender_dialog_id) {
    parse(sender_dialog_id, parser);
  }
  if (has_reply_in_dialog_id) {
    parse(reply_in_dialog_id, parser);
  }
  if (has_top_thread_message_id) {
    parse(top_thread_message_id, parser);
  }
  if (has_thread_draft_message) {
    parse(thread_draft_message, parser);
  }
  if (has_local_thread_message_ids) {
    parse(local_thread_message_ids, parser);
  }
  if (has_linked_top_thread_message_id) {
    parse(linked_top_thread_message_id, parser);
  }
  if (has_interaction_info_update_date) {
    parse(interaction_info_update_date, parser);
  }
  if (has_send_emoji) {
    parse(send_emoji, parser);
  }
  parse_message_content(content, parser);
  if (has_reply_markup) {
    parse(reply_markup, parser);
  }
  if (has_ttl_period) {
    parse(ttl_period, parser);
  }
  if (has_max_reply_media_timestamp) {
    parse(max_reply_media_timestamp, parser);
  }

  CHECK(content != nullptr);
  is_content_secret |=
      is_secret_message_content(ttl, content->get_type());  // repair is_content_secret for old messages
  if (hide_edit_date && content->get_type() == MessageContentType::LiveLocation) {
    hide_edit_date = false;
  }
}

template <class StorerT>
void MessagesManager::NotificationGroupInfo::store(StorerT &storer) const {
  using td::store;
  store(group_id, storer);
  store(last_notification_date, storer);
  store(last_notification_id, storer);
  store(max_removed_notification_id, storer);
  store(max_removed_message_id, storer);
}

template <class ParserT>
void MessagesManager::NotificationGroupInfo::parse(ParserT &parser) {
  using td::parse;
  parse(group_id, parser);
  parse(last_notification_date, parser);
  parse(last_notification_id, parser);
  parse(max_removed_notification_id, parser);
  if (parser.version() >= static_cast<int32>(Version::AddNotificationGroupInfoMaxRemovedMessageId)) {
    parse(max_removed_message_id, parser);
  }
}

template <class StorerT>
void MessagesManager::Dialog::store(StorerT &storer) const {
  using td::store;
  const Message *last_database_message = nullptr;
  if (last_database_message_id.is_valid()) {
    last_database_message = get_message(this, last_database_message_id);
  }

  auto dialog_type = dialog_id.get_type();
  bool has_draft_message = draft_message != nullptr;
  bool has_last_database_message = last_database_message != nullptr;
  bool has_first_database_message_id = first_database_message_id.is_valid();
  bool has_first_database_message_id_by_index = true;
  bool has_message_count_by_index = true;
  bool has_client_data = !client_data.empty();
  bool has_last_read_all_mentions_message_id = last_read_all_mentions_message_id.is_valid();
  bool has_max_unavailable_message_id = max_unavailable_message_id.is_valid();
  bool has_local_unread_count = local_unread_count != 0;
  bool has_deleted_last_message = delete_last_message_date > 0;
  bool has_last_clear_history_message_id = last_clear_history_message_id.is_valid();
  bool has_last_database_message_id = !has_last_database_message && last_database_message_id.is_valid();
  bool has_message_notification_group =
      message_notification_group.group_id.is_valid() && !message_notification_group.try_reuse;
  bool has_mention_notification_group =
      mention_notification_group.group_id.is_valid() && !mention_notification_group.try_reuse;
  bool has_new_secret_chat_notification_id = new_secret_chat_notification_id.is_valid();
  bool has_pinned_message_notification = pinned_message_notification_message_id.is_valid();
  bool has_last_pinned_message_id = last_pinned_message_id.is_valid();
  bool has_flags2 = true;
  bool has_max_notification_message_id =
      max_notification_message_id.is_valid() && max_notification_message_id > last_new_message_id;
  bool has_folder_id = folder_id != FolderId();
  bool has_pending_read_channel_inbox = pending_read_channel_inbox_pts != 0;
  bool has_distance = distance >= 0;
  bool has_last_yet_unsent_message = last_message_id.is_valid() && last_message_id.is_yet_unsent();
  bool has_active_group_call_id = active_group_call_id.is_valid();
  bool has_message_ttl_setting = !message_ttl_setting.is_empty();
  bool has_default_join_group_call_as_dialog_id = default_join_group_call_as_dialog_id.is_valid();
  bool store_has_bots = dialog_type == DialogType::Chat || dialog_type == DialogType::Channel;
  bool has_theme_name = !theme_name.empty();
  bool has_flags3 = true;
  bool has_pending_join_requests = pending_join_request_count != 0;
  BEGIN_STORE_FLAGS();
  STORE_FLAG(has_draft_message);
  STORE_FLAG(has_last_database_message);
  STORE_FLAG(know_can_report_spam);
  STORE_FLAG(can_report_spam);
  STORE_FLAG(has_first_database_message_id);
  STORE_FLAG(false);
  STORE_FLAG(has_first_database_message_id_by_index);
  STORE_FLAG(has_message_count_by_index);
  STORE_FLAG(has_client_data);
  STORE_FLAG(need_restore_reply_markup);
  STORE_FLAG(have_full_history);
  STORE_FLAG(has_last_read_all_mentions_message_id);
  STORE_FLAG(has_max_unavailable_message_id);
  STORE_FLAG(is_last_read_inbox_message_id_inited);
  STORE_FLAG(is_last_read_outbox_message_id_inited);
  STORE_FLAG(has_local_unread_count);
  STORE_FLAG(has_deleted_last_message);
  STORE_FLAG(has_last_clear_history_message_id);
  STORE_FLAG(is_last_message_deleted_locally);
  STORE_FLAG(has_contact_registered_message);
  STORE_FLAG(has_last_database_message_id);
  STORE_FLAG(need_repair_server_unread_count);
  STORE_FLAG(is_marked_as_unread);
  STORE_FLAG(has_message_notification_group);
  STORE_FLAG(has_mention_notification_group);
  STORE_FLAG(has_new_secret_chat_notification_id);
  STORE_FLAG(has_pinned_message_notification);
  STORE_FLAG(has_last_pinned_message_id);
  STORE_FLAG(is_last_pinned_message_id_inited);
  STORE_FLAG(has_flags2);
  END_STORE_FLAGS();

  store(dialog_id, storer);  // must be stored at offset 4

  if (has_flags2) {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(has_max_notification_message_id);
    STORE_FLAG(has_folder_id);
    STORE_FLAG(is_folder_id_inited);
    STORE_FLAG(has_pending_read_channel_inbox);
    STORE_FLAG(know_action_bar);
    STORE_FLAG(can_add_contact);
    STORE_FLAG(can_block_user);
    STORE_FLAG(can_share_phone_number);
    STORE_FLAG(can_report_location);
    STORE_FLAG(has_scheduled_server_messages);
    STORE_FLAG(has_scheduled_database_messages);
    STORE_FLAG(need_repair_channel_server_unread_count);
    STORE_FLAG(can_unarchive);
    STORE_FLAG(has_distance);
    STORE_FLAG(hide_distance);
    STORE_FLAG(has_last_yet_unsent_message);
    STORE_FLAG(is_blocked);
    STORE_FLAG(is_is_blocked_inited);
    STORE_FLAG(has_active_group_call);
    STORE_FLAG(is_group_call_empty);
    STORE_FLAG(has_active_group_call_id);
    STORE_FLAG(can_invite_members);
    STORE_FLAG(has_message_ttl_setting);
    STORE_FLAG(is_message_ttl_setting_inited);
    STORE_FLAG(has_default_join_group_call_as_dialog_id);
    STORE_FLAG(store_has_bots ? has_bots : false);
    STORE_FLAG(store_has_bots ? is_has_bots_inited : false);
    STORE_FLAG(is_theme_name_inited);
    STORE_FLAG(has_theme_name);
    STORE_FLAG(has_flags3);
    END_STORE_FLAGS();
  }
  if (has_flags3) {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(has_pending_join_requests);
    END_STORE_FLAGS();
  }

  store(last_new_message_id, storer);
  store(server_unread_count, storer);
  if (has_local_unread_count) {
    store(local_unread_count, storer);
  }
  store(last_read_inbox_message_id, storer);
  store(last_read_outbox_message_id, storer);
  store(reply_markup_message_id, storer);
  store(notification_settings, storer);
  if (has_draft_message) {
    store(draft_message, storer);
  }
  store(last_clear_history_date, storer);
  store(order, storer);
  if (has_last_database_message) {
    store(*last_database_message, storer);
  }
  if (has_first_database_message_id) {
    store(first_database_message_id, storer);
  }
  if (has_deleted_last_message) {
    store(delete_last_message_date, storer);
    store(deleted_last_message_id, storer);
  }
  if (has_last_clear_history_message_id) {
    store(last_clear_history_message_id, storer);
  }

  if (has_first_database_message_id_by_index) {
    store(static_cast<int32>(first_database_message_id_by_index.size()), storer);
    for (auto first_message_id : first_database_message_id_by_index) {
      store(first_message_id, storer);
    }
  }
  if (has_message_count_by_index) {
    store(static_cast<int32>(message_count_by_index.size()), storer);
    for (auto message_count : message_count_by_index) {
      store(message_count, storer);
    }
  }
  if (has_client_data) {
    store(client_data, storer);
  }
  if (has_last_read_all_mentions_message_id) {
    store(last_read_all_mentions_message_id, storer);
  }
  if (has_max_unavailable_message_id) {
    store(max_unavailable_message_id, storer);
  }
  if (has_last_database_message_id) {
    store(last_database_message_id, storer);
  }
  if (has_message_notification_group) {
    store(message_notification_group, storer);
  }
  if (has_mention_notification_group) {
    store(mention_notification_group, storer);
  }
  if (has_new_secret_chat_notification_id) {
    store(new_secret_chat_notification_id, storer);
  }
  if (has_pinned_message_notification) {
    store(pinned_message_notification_message_id, storer);
  }
  if (has_last_pinned_message_id) {
    store(last_pinned_message_id, storer);
  }
  if (has_max_notification_message_id) {
    store(max_notification_message_id, storer);
  }
  if (has_folder_id) {
    store(folder_id, storer);
  }
  if (has_pending_read_channel_inbox) {
    store(pending_read_channel_inbox_pts, storer);
    store(pending_read_channel_inbox_max_message_id, storer);
    store(pending_read_channel_inbox_server_unread_count, storer);
  }
  if (has_distance) {
    store(distance, storer);
  }
  if (has_active_group_call_id) {
    store(active_group_call_id, storer);
  }
  if (has_message_ttl_setting) {
    store(message_ttl_setting, storer);
  }
  if (has_default_join_group_call_as_dialog_id) {
    store(default_join_group_call_as_dialog_id, storer);
  }
  if (has_theme_name) {
    store(theme_name, storer);
  }
  if (has_pending_join_requests) {
    store(pending_join_request_count, storer);
    store(pending_join_request_user_ids, storer);
  }
}

// do not forget to resolve dialog dependencies including dependencies of last_message
template <class ParserT>
void MessagesManager::Dialog::parse(ParserT &parser) {
  using td::parse;
  bool has_draft_message;
  bool has_last_database_message;
  bool has_first_database_message_id;
  bool legacy_is_pinned;
  bool has_first_database_message_id_by_index;
  bool has_message_count_by_index;
  bool has_client_data;
  bool has_last_read_all_mentions_message_id;
  bool has_max_unavailable_message_id;
  bool has_local_unread_count;
  bool has_deleted_last_message;
  bool has_last_clear_history_message_id;
  bool has_last_database_message_id;
  bool has_message_notification_group;
  bool has_mention_notification_group;
  bool has_new_secret_chat_notification_id;
  bool has_pinned_message_notification;
  bool has_last_pinned_message_id;
  bool has_flags2;
  bool has_max_notification_message_id = false;
  bool has_folder_id = false;
  bool has_pending_read_channel_inbox = false;
  bool has_distance = false;
  bool has_active_group_call_id = false;
  bool has_message_ttl_setting = false;
  bool has_default_join_group_call_as_dialog_id = false;
  bool has_theme_name = false;
  bool has_flags3 = false;
  bool has_pending_join_requests = false;
  BEGIN_PARSE_FLAGS();
  PARSE_FLAG(has_draft_message);
  PARSE_FLAG(has_last_database_message);
  PARSE_FLAG(know_can_report_spam);
  PARSE_FLAG(can_report_spam);
  PARSE_FLAG(has_first_database_message_id);
  PARSE_FLAG(legacy_is_pinned);
  PARSE_FLAG(has_first_database_message_id_by_index);
  PARSE_FLAG(has_message_count_by_index);
  PARSE_FLAG(has_client_data);
  PARSE_FLAG(need_restore_reply_markup);
  PARSE_FLAG(have_full_history);
  PARSE_FLAG(has_last_read_all_mentions_message_id);
  PARSE_FLAG(has_max_unavailable_message_id);
  PARSE_FLAG(is_last_read_inbox_message_id_inited);
  PARSE_FLAG(is_last_read_outbox_message_id_inited);
  PARSE_FLAG(has_local_unread_count);
  PARSE_FLAG(has_deleted_last_message);
  PARSE_FLAG(has_last_clear_history_message_id);
  PARSE_FLAG(is_last_message_deleted_locally);
  PARSE_FLAG(has_contact_registered_message);
  PARSE_FLAG(has_last_database_message_id);
  PARSE_FLAG(need_repair_server_unread_count);
  PARSE_FLAG(is_marked_as_unread);
  PARSE_FLAG(has_message_notification_group);
  PARSE_FLAG(has_mention_notification_group);
  PARSE_FLAG(has_new_secret_chat_notification_id);
  PARSE_FLAG(has_pinned_message_notification);
  PARSE_FLAG(has_last_pinned_message_id);
  PARSE_FLAG(is_last_pinned_message_id_inited);
  PARSE_FLAG(has_flags2);
  END_PARSE_FLAGS();

  parse(dialog_id, parser);  // must be stored at offset 4

  if (has_flags2) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(has_max_notification_message_id);
    PARSE_FLAG(has_folder_id);
    PARSE_FLAG(is_folder_id_inited);
    PARSE_FLAG(has_pending_read_channel_inbox);
    PARSE_FLAG(know_action_bar);
    PARSE_FLAG(can_add_contact);
    PARSE_FLAG(can_block_user);
    PARSE_FLAG(can_share_phone_number);
    PARSE_FLAG(can_report_location);
    PARSE_FLAG(has_scheduled_server_messages);
    PARSE_FLAG(has_scheduled_database_messages);
    PARSE_FLAG(need_repair_channel_server_unread_count);
    PARSE_FLAG(can_unarchive);
    PARSE_FLAG(has_distance);
    PARSE_FLAG(hide_distance);
    PARSE_FLAG(had_last_yet_unsent_message);
    PARSE_FLAG(is_blocked);
    PARSE_FLAG(is_is_blocked_inited);
    PARSE_FLAG(has_active_group_call);
    PARSE_FLAG(is_group_call_empty);
    PARSE_FLAG(has_active_group_call_id);
    PARSE_FLAG(can_invite_members);
    PARSE_FLAG(has_message_ttl_setting);
    PARSE_FLAG(is_message_ttl_setting_inited);
    PARSE_FLAG(has_default_join_group_call_as_dialog_id);
    PARSE_FLAG(has_bots);
    PARSE_FLAG(is_has_bots_inited);
    PARSE_FLAG(is_theme_name_inited);
    PARSE_FLAG(has_theme_name);
    PARSE_FLAG(has_flags3);
    END_PARSE_FLAGS();
  } else {
    is_folder_id_inited = false;
    know_action_bar = false;
    can_add_contact = false;
    can_block_user = false;
    can_share_phone_number = false;
    can_report_location = false;
    has_scheduled_server_messages = false;
    has_scheduled_database_messages = false;
    need_repair_channel_server_unread_count = false;
    can_unarchive = false;
    hide_distance = false;
    had_last_yet_unsent_message = false;
    is_blocked = false;
    is_is_blocked_inited = false;
    has_active_group_call = false;
    is_group_call_empty = false;
    can_invite_members = false;
    is_message_ttl_setting_inited = false;
    has_bots = false;
    is_has_bots_inited = false;
    is_theme_name_inited = false;
  }
  if (has_flags3) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(has_pending_join_requests);
    END_PARSE_FLAGS();
  }

  parse(last_new_message_id, parser);
  parse(server_unread_count, parser);
  if (has_local_unread_count) {
    parse(local_unread_count, parser);
  }
  parse(last_read_inbox_message_id, parser);
  if (last_read_inbox_message_id.is_valid()) {
    is_last_read_inbox_message_id_inited = true;
  }
  parse(last_read_outbox_message_id, parser);
  if (last_read_outbox_message_id.is_valid()) {
    is_last_read_outbox_message_id_inited = true;
  }
  parse(reply_markup_message_id, parser);
  parse(notification_settings, parser);
  if (has_draft_message) {
    parse(draft_message, parser);
  }
  parse(last_clear_history_date, parser);
  parse(order, parser);
  if (has_last_database_message) {
    parse(messages, parser);
  }
  if (has_first_database_message_id) {
    parse(first_database_message_id, parser);
  }
  if (legacy_is_pinned) {
    int64 legacy_pinned_order;
    parse(legacy_pinned_order, parser);
  }
  if (has_deleted_last_message) {
    parse(delete_last_message_date, parser);
    parse(deleted_last_message_id, parser);
  }
  if (has_last_clear_history_message_id) {
    parse(last_clear_history_message_id, parser);
  }

  if (has_first_database_message_id_by_index) {
    int32 size;
    parse(size, parser);
    if (size < 0) {
      // the log event is broken
      // it should be impossible, but has happenned at least once
      parser.set_error("Wrong first_database_message_id_by_index table size");
      return;
    }
    LOG_CHECK(static_cast<size_t>(size) <= first_database_message_id_by_index.size())
        << size << " " << first_database_message_id_by_index.size();
    for (int32 i = 0; i < size; i++) {
      parse(first_database_message_id_by_index[i], parser);
    }
  }
  if (has_message_count_by_index) {
    int32 size;
    parse(size, parser);
    if (size < 0) {
      // the log event is broken
      // it should be impossible, but has happenned at least once
      parser.set_error("Wrong message_count_by_index table size");
      return;
    }
    LOG_CHECK(static_cast<size_t>(size) <= message_count_by_index.size())
        << size << " " << message_count_by_index.size();
    for (int32 i = 0; i < size; i++) {
      parse(message_count_by_index[i], parser);
    }
  }
  unread_mention_count = message_count_by_index[message_search_filter_index(MessageSearchFilter::UnreadMention)];
  LOG(INFO) << "Set unread mention message count in " << dialog_id << " to " << unread_mention_count;
  if (unread_mention_count < 0) {
    unread_mention_count = 0;
  }
  if (has_client_data) {
    parse(client_data, parser);
  }
  if (has_last_read_all_mentions_message_id) {
    parse(last_read_all_mentions_message_id, parser);
  }
  if (has_max_unavailable_message_id) {
    parse(max_unavailable_message_id, parser);
  }
  if (has_last_database_message_id) {
    parse(last_database_message_id, parser);
  }
  if (has_message_notification_group) {
    parse(message_notification_group, parser);
  }
  if (has_mention_notification_group) {
    parse(mention_notification_group, parser);
  }
  if (has_new_secret_chat_notification_id) {
    parse(new_secret_chat_notification_id, parser);
  }
  if (has_pinned_message_notification) {
    parse(pinned_message_notification_message_id, parser);
  }
  if (has_last_pinned_message_id) {
    parse(last_pinned_message_id, parser);
  }
  if (has_max_notification_message_id) {
    parse(max_notification_message_id, parser);
  }
  if (has_folder_id) {
    parse(folder_id, parser);
  }
  if (has_pending_read_channel_inbox) {
    parse(pending_read_channel_inbox_pts, parser);
    parse(pending_read_channel_inbox_max_message_id, parser);
    parse(pending_read_channel_inbox_server_unread_count, parser);
  }
  if (has_distance) {
    parse(distance, parser);
  }
  if (has_active_group_call_id) {
    parse(active_group_call_id, parser);
  }
  if (has_message_ttl_setting) {
    parse(message_ttl_setting, parser);
  }
  if (has_default_join_group_call_as_dialog_id) {
    parse(default_join_group_call_as_dialog_id, parser);
  }
  if (has_theme_name) {
    parse(theme_name, parser);
  }
  if (has_pending_join_requests) {
    parse(pending_join_request_count, parser);
    parse(pending_join_request_user_ids, parser);
  }
}

template <class StorerT>
void MessagesManager::CallsDbState::store(StorerT &storer) const {
  using td::store;
  store(static_cast<int32>(first_calls_database_message_id_by_index.size()), storer);
  for (auto first_message_id : first_calls_database_message_id_by_index) {
    store(first_message_id, storer);
  }
  store(static_cast<int32>(message_count_by_index.size()), storer);
  for (auto message_count : message_count_by_index) {
    store(message_count, storer);
  }
}

template <class ParserT>
void MessagesManager::CallsDbState::parse(ParserT &parser) {
  using td::parse;
  int32 size;
  parse(size, parser);
  LOG_CHECK(static_cast<size_t>(size) <= first_calls_database_message_id_by_index.size())
      << size << " " << first_calls_database_message_id_by_index.size();
  for (int32 i = 0; i < size; i++) {
    parse(first_calls_database_message_id_by_index[i], parser);
  }
  parse(size, parser);
  LOG_CHECK(static_cast<size_t>(size) <= message_count_by_index.size()) << size << " " << message_count_by_index.size();
  for (int32 i = 0; i < size; i++) {
    parse(message_count_by_index[i], parser);
  }
}

void MessagesManager::load_calls_db_state() {
  if (!G()->parameters().use_message_db) {
    return;
  }
  std::fill(calls_db_state_.message_count_by_index.begin(), calls_db_state_.message_count_by_index.end(), -1);
  auto value = G()->td_db()->get_sqlite_sync_pmc()->get("calls_db_state");
  if (value.empty()) {
    return;
  }
  log_event_parse(calls_db_state_, value).ensure();
  LOG(INFO) << "Save calls database state " << calls_db_state_.first_calls_database_message_id_by_index[0] << " ("
            << calls_db_state_.message_count_by_index[0] << ") "
            << calls_db_state_.first_calls_database_message_id_by_index[1] << " ("
            << calls_db_state_.message_count_by_index[1] << ")";
}

void MessagesManager::save_calls_db_state() {
  if (!G()->parameters().use_message_db) {
    return;
  }

  LOG(INFO) << "Save calls database state " << calls_db_state_.first_calls_database_message_id_by_index[0] << " ("
            << calls_db_state_.message_count_by_index[0] << ") "
            << calls_db_state_.first_calls_database_message_id_by_index[1] << " ("
            << calls_db_state_.message_count_by_index[1] << ")";
  G()->td_db()->get_sqlite_pmc()->set("calls_db_state", log_event_store(calls_db_state_).as_slice().str(), Auto());
}

MessagesManager::Dialog::~Dialog() {
  if (!G()->close_flag()) {
    LOG(ERROR) << "Destroy " << dialog_id;
  }
}

MessagesManager::MessagesManager(Td *td, ActorShared<> parent)
    : recently_found_dialogs_{td, "recently_found", MAX_RECENT_DIALOGS}
    , recently_opened_dialogs_{td, "recently_opened", MAX_RECENT_DIALOGS}
    , td_(td)
    , parent_(std::move(parent)) {
  upload_media_callback_ = std::make_shared<UploadMediaCallback>();
  upload_thumbnail_callback_ = std::make_shared<UploadThumbnailCallback>();
  upload_dialog_photo_callback_ = std::make_shared<UploadDialogPhotoCallback>();
  upload_imported_messages_callback_ = std::make_shared<UploadImportedMessagesCallback>();
  upload_imported_message_attachment_callback_ = std::make_shared<UploadImportedMessageAttachmentCallback>();

  channel_get_difference_timeout_.set_callback(on_channel_get_difference_timeout_callback);
  channel_get_difference_timeout_.set_callback_data(static_cast<void *>(this));

  channel_get_difference_retry_timeout_.set_callback(on_channel_get_difference_timeout_callback);
  channel_get_difference_retry_timeout_.set_callback_data(static_cast<void *>(this));

  pending_message_views_timeout_.set_callback(on_pending_message_views_timeout_callback);
  pending_message_views_timeout_.set_callback_data(static_cast<void *>(this));

  pending_message_live_location_view_timeout_.set_callback(on_pending_message_live_location_view_timeout_callback);
  pending_message_live_location_view_timeout_.set_callback_data(static_cast<void *>(this));

  pending_draft_message_timeout_.set_callback(on_pending_draft_message_timeout_callback);
  pending_draft_message_timeout_.set_callback_data(static_cast<void *>(this));

  pending_read_history_timeout_.set_callback(on_pending_read_history_timeout_callback);
  pending_read_history_timeout_.set_callback_data(static_cast<void *>(this));

  pending_updated_dialog_timeout_.set_callback(on_pending_updated_dialog_timeout_callback);
  pending_updated_dialog_timeout_.set_callback_data(static_cast<void *>(this));

  pending_unload_dialog_timeout_.set_callback(on_pending_unload_dialog_timeout_callback);
  pending_unload_dialog_timeout_.set_callback_data(static_cast<void *>(this));

  dialog_unmute_timeout_.set_callback(on_dialog_unmute_timeout_callback);
  dialog_unmute_timeout_.set_callback_data(static_cast<void *>(this));

  pending_send_dialog_action_timeout_.set_callback(on_pending_send_dialog_action_timeout_callback);
  pending_send_dialog_action_timeout_.set_callback_data(static_cast<void *>(this));

  active_dialog_action_timeout_.set_callback(on_active_dialog_action_timeout_callback);
  active_dialog_action_timeout_.set_callback_data(static_cast<void *>(this));

  update_dialog_online_member_count_timeout_.set_callback(on_update_dialog_online_member_count_timeout_callback);
  update_dialog_online_member_count_timeout_.set_callback_data(static_cast<void *>(this));

  preload_folder_dialog_list_timeout_.set_callback(on_preload_folder_dialog_list_timeout_callback);
  preload_folder_dialog_list_timeout_.set_callback_data(static_cast<void *>(this));

  sequence_dispatcher_ = create_actor<MultiSequenceDispatcher>("multi sequence dispatcher");
}

MessagesManager::~MessagesManager() = default;

void MessagesManager::on_channel_get_difference_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::on_channel_get_difference_timeout,
                     DialogId(dialog_id_int));
}

void MessagesManager::on_pending_message_views_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::on_pending_message_views_timeout,
                     DialogId(dialog_id_int));
}

void MessagesManager::on_pending_message_live_location_view_timeout_callback(void *messages_manager_ptr,
                                                                             int64 task_id) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager),
                     &MessagesManager::view_message_live_location_on_server, task_id);
}

void MessagesManager::on_pending_draft_message_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager),
                     &MessagesManager::save_dialog_draft_message_on_server, DialogId(dialog_id_int));
}

void MessagesManager::on_pending_read_history_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::do_read_history_on_server,
                     DialogId(dialog_id_int));
}

void MessagesManager::on_pending_updated_dialog_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  // no check for G()->close_flag() to save dialogs even while closing

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  // TODO it is unsafe to save dialog to database before binlog is flushed

  // no send_closure_later, because messages_manager can be not an actor while closing
  messages_manager->save_dialog_to_database(DialogId(dialog_id_int));
}

void MessagesManager::on_pending_unload_dialog_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::unload_dialog,
                     DialogId(dialog_id_int));
}

void MessagesManager::on_dialog_unmute_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  if (1 <= dialog_id_int && dialog_id_int <= 3) {
    send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::on_scope_unmute,
                       static_cast<NotificationSettingsScope>(dialog_id_int - 1));
  } else {
    send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::on_dialog_unmute,
                       DialogId(dialog_id_int));
  }
}

void MessagesManager::on_pending_send_dialog_action_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::on_send_dialog_action_timeout,
                     DialogId(dialog_id_int));
}

void MessagesManager::on_active_dialog_action_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::on_active_dialog_action_timeout,
                     DialogId(dialog_id_int));
}

void MessagesManager::on_update_dialog_online_member_count_timeout_callback(void *messages_manager_ptr,
                                                                            int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager),
                     &MessagesManager::on_update_dialog_online_member_count_timeout, DialogId(dialog_id_int));
}

void MessagesManager::on_preload_folder_dialog_list_timeout_callback(void *messages_manager_ptr, int64 folder_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::preload_folder_dialog_list,
                     FolderId(narrow_cast<int32>(folder_id_int)));
}

td_api::object_ptr<td_api::MessageSender> MessagesManager::get_message_sender_object_const(UserId user_id,
                                                                                           DialogId dialog_id,
                                                                                           const char *source) const {
  if (dialog_id.is_valid() && have_dialog(dialog_id)) {
    return td_api::make_object<td_api::messageSenderChat>(dialog_id.get());
  }
  if (!user_id.is_valid()) {
    // can happen only if the server sends a message with wrong sender
    LOG(ERROR) << "Receive message with wrong sender " << user_id << '/' << dialog_id << " from " << source;
    user_id = td_->contacts_manager_->add_service_notifications_user();
  }
  return td_api::make_object<td_api::messageSenderUser>(td_->contacts_manager_->get_user_id_object(user_id, source));
}

td_api::object_ptr<td_api::MessageSender> MessagesManager::get_message_sender_object(UserId user_id, DialogId dialog_id,
                                                                                     const char *source) {
  if (dialog_id.is_valid() && !have_dialog(dialog_id)) {
    LOG(ERROR) << "Failed to find " << dialog_id;
    force_create_dialog(dialog_id, source);
  }
  if (!user_id.is_valid() && td_->auth_manager_->is_bot()) {
    td_->contacts_manager_->add_anonymous_bot_user();
    td_->contacts_manager_->add_service_notifications_user();
  }
  return get_message_sender_object_const(user_id, dialog_id, source);
}

td_api::object_ptr<td_api::MessageSender> MessagesManager::get_message_sender_object_const(DialogId dialog_id,
                                                                                           const char *source) const {
  if (dialog_id.get_type() == DialogType::User) {
    return get_message_sender_object_const(dialog_id.get_user_id(), DialogId(), source);
  }
  return get_message_sender_object_const(UserId(), dialog_id, source);
}

td_api::object_ptr<td_api::MessageSender> MessagesManager::get_message_sender_object(DialogId dialog_id,
                                                                                     const char *source) {
  if (dialog_id.get_type() == DialogType::User) {
    return get_message_sender_object(dialog_id.get_user_id(), DialogId(), source);
  }
  return get_message_sender_object(UserId(), dialog_id, source);
}

BufferSlice MessagesManager::get_dialog_database_value(const Dialog *d) {
  // can't use log_event_store, because it tries to parse stored Dialog
  LogEventStorerCalcLength storer_calc_length;
  store(*d, storer_calc_length);

  BufferSlice value_buffer{storer_calc_length.get_length()};
  auto value = value_buffer.as_slice();

  LogEventStorerUnsafe storer_unsafe(value.ubegin());
  store(*d, storer_unsafe);
  return value_buffer;
}

void MessagesManager::save_dialog_to_database(DialogId dialog_id) {
  CHECK(G()->parameters().use_message_db);
  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  LOG(INFO) << "Save " << dialog_id << " to database";
  vector<NotificationGroupKey> changed_group_keys;
  bool can_reuse_notification_group = false;
  auto add_group_key = [&](auto &group_info) {
    if (group_info.is_changed) {
      can_reuse_notification_group |= group_info.try_reuse;
      changed_group_keys.emplace_back(group_info.group_id, group_info.try_reuse ? DialogId() : dialog_id,
                                      group_info.last_notification_date);
      group_info.is_changed = false;
    }
  };
  add_group_key(d->message_notification_group);
  add_group_key(d->mention_notification_group);
  auto fixed_folder_id = d->folder_id == FolderId::archive() ? FolderId::archive() : FolderId::main();
  G()->td_db()->get_dialog_db_async()->add_dialog(
      dialog_id, fixed_folder_id, d->is_folder_id_inited ? d->order : 0, get_dialog_database_value(d),
      std::move(changed_group_keys), PromiseCreator::lambda([dialog_id, can_reuse_notification_group](Result<> result) {
        send_closure(G()->messages_manager(), &MessagesManager::on_save_dialog_to_database, dialog_id,
                     can_reuse_notification_group, result.is_ok());
      }));
}

void MessagesManager::on_save_dialog_to_database(DialogId dialog_id, bool can_reuse_notification_group, bool success) {
  LOG(INFO) << "Successfully saved " << dialog_id << " to database";

  if (success && can_reuse_notification_group && !G()->close_flag()) {
    auto d = get_dialog(dialog_id);
    CHECK(d != nullptr);
    try_reuse_notification_group(d->message_notification_group);
    try_reuse_notification_group(d->mention_notification_group);
  }

  // TODO erase some events from binlog
}

void MessagesManager::try_reuse_notification_group(NotificationGroupInfo &group_info) {
  if (!group_info.try_reuse) {
    return;
  }
  if (group_info.is_changed) {
    LOG(ERROR) << "Failed to reuse changed " << group_info.group_id;
    return;
  }
  group_info.try_reuse = false;
  if (!group_info.group_id.is_valid()) {
    LOG(ERROR) << "Failed to reuse invalid " << group_info.group_id;
    return;
  }
  CHECK(group_info.last_notification_id == NotificationId());
  CHECK(group_info.last_notification_date == 0);
  send_closure_later(G()->notification_manager(), &NotificationManager::try_reuse_notification_group_id,
                     group_info.group_id);
  notification_group_id_to_dialog_id_.erase(group_info.group_id);
  group_info.group_id = NotificationGroupId();
  group_info.max_removed_notification_id = NotificationId();
  group_info.max_removed_message_id = MessageId();
}

void MessagesManager::invalidate_message_indexes(Dialog *d) {
  CHECK(d != nullptr);
  bool is_secret = d->dialog_id.get_type() == DialogType::SecretChat;
  for (size_t i = 0; i < d->message_count_by_index.size(); i++) {
    if (is_secret || i == static_cast<size_t>(message_search_filter_index(MessageSearchFilter::FailedToSend))) {
      // always know all messages
      d->first_database_message_id_by_index[i] = MessageId::min();
      // keep the count
    } else {
      // some messages are unknown; drop first_database_message_id and count
      d->first_database_message_id_by_index[i] = MessageId();
      d->message_count_by_index[i] = -1;
    }
  }
}

void MessagesManager::update_message_count_by_index(Dialog *d, int diff, const Message *m) {
  auto index_mask = get_message_index_mask(d->dialog_id, m);
  index_mask &= ~message_search_filter_index_mask(
      MessageSearchFilter::UnreadMention);  // unread mention count has been already manually updated

  update_message_count_by_index(d, diff, index_mask);
}

void MessagesManager::update_message_count_by_index(Dialog *d, int diff, int32 index_mask) {
  if (index_mask == 0) {
    return;
  }

  LOG(INFO) << "Update message count by " << diff << " in index mask " << index_mask;
  int i = 0;
  for (auto &message_count : d->message_count_by_index) {
    if (((index_mask >> i) & 1) != 0 && message_count != -1) {
      message_count += diff;
      if (message_count < 0) {
        if (d->dialog_id.get_type() == DialogType::SecretChat ||
            i == message_search_filter_index(MessageSearchFilter::FailedToSend)) {
          message_count = 0;
        } else {
          message_count = -1;
        }
      }
      on_dialog_updated(d->dialog_id, "update_message_count_by_index");
    }
    i++;
  }

  i = static_cast<int>(MessageSearchFilter::Call) - 1;
  for (auto &message_count : calls_db_state_.message_count_by_index) {
    if (((index_mask >> i) & 1) != 0 && message_count != -1) {
      message_count += diff;
      if (message_count < 0) {
        if (d->dialog_id.get_type() == DialogType::SecretChat) {
          message_count = 0;
        } else {
          message_count = -1;
        }
      }
      save_calls_db_state();
    }
    i++;
  }
}

int32 MessagesManager::get_message_index_mask(DialogId dialog_id, const Message *m) const {
  CHECK(m != nullptr);
  if (m->message_id.is_scheduled() || m->message_id.is_yet_unsent()) {
    return 0;
  }
  if (m->is_failed_to_send) {
    return message_search_filter_index_mask(MessageSearchFilter::FailedToSend);
  }
  bool is_secret = dialog_id.get_type() == DialogType::SecretChat;
  if (!m->message_id.is_server() && !is_secret) {
    return 0;
  }

  int32 index_mask = 0;
  if (m->is_pinned) {
    index_mask |= message_search_filter_index_mask(MessageSearchFilter::Pinned);
  }
  // retain second condition just in case
  if (m->is_content_secret || (m->ttl > 0 && !is_secret)) {
    return index_mask;
  }
  index_mask |= get_message_content_index_mask(m->content.get(), td_, m->is_outgoing);
  if (m->contains_mention) {
    index_mask |= message_search_filter_index_mask(MessageSearchFilter::Mention);
    if (m->contains_unread_mention) {
      index_mask |= message_search_filter_index_mask(MessageSearchFilter::UnreadMention);
    }
  }
  LOG(INFO) << "Have index mask " << index_mask << " for " << m->message_id << " in " << dialog_id;
  return index_mask;
}

void MessagesManager::update_reply_count_by_message(Dialog *d, int diff, const Message *m) {
  if (td_->auth_manager_->is_bot() || !m->top_thread_message_id.is_valid() ||
      m->top_thread_message_id == m->message_id || !m->message_id.is_valid() || !m->message_id.is_server()) {
    return;
  }

  auto replier_dialog_id =
      has_message_sender_user_id(d->dialog_id, m) ? DialogId(m->sender_user_id) : m->sender_dialog_id;
  update_message_reply_count(d, m->top_thread_message_id, replier_dialog_id, m->message_id,
                             diff < 0 ? G()->unix_time() : m->date, diff);
}

void MessagesManager::update_message_reply_count(Dialog *d, MessageId message_id, DialogId replier_dialog_id,
                                                 MessageId reply_message_id, int32 update_date, int diff,
                                                 bool is_recursive) {
  if (d == nullptr) {
    return;
  }

  Message *m = get_message(d, message_id);
  if (m == nullptr || !is_active_message_reply_info(d->dialog_id, m->reply_info)) {
    return;
  }
  LOG(INFO) << "Update reply count to " << message_id << " in " << d->dialog_id << " by " << diff << " from "
            << reply_message_id << " sent by " << replier_dialog_id;
  if (m->interaction_info_update_date < update_date &&
      m->reply_info.add_reply(replier_dialog_id, reply_message_id, diff)) {
    on_message_reply_info_changed(d->dialog_id, m);
    on_message_changed(d, m, true, "update_message_reply_count_by_message");
  }

  if (!is_recursive && is_discussion_message(d->dialog_id, m)) {
    update_message_reply_count(get_dialog(m->forward_info->from_dialog_id), m->forward_info->from_message_id,
                               replier_dialog_id, reply_message_id, update_date, diff, true);
  }
}

vector<MessageId> MessagesManager::get_message_ids(const vector<int64> &input_message_ids) {
  vector<MessageId> message_ids;
  message_ids.reserve(input_message_ids.size());
  for (auto &input_message_id : input_message_ids) {
    message_ids.push_back(MessageId(input_message_id));
  }
  return message_ids;
}

vector<int32> MessagesManager::get_server_message_ids(const vector<MessageId> &message_ids) {
  return transform(message_ids, [](MessageId message_id) { return message_id.get_server_message_id().get(); });
}

vector<int32> MessagesManager::get_scheduled_server_message_ids(const vector<MessageId> &message_ids) {
  return transform(message_ids,
                   [](MessageId message_id) { return message_id.get_scheduled_server_message_id().get(); });
}

tl_object_ptr<telegram_api::InputPeer> MessagesManager::get_input_peer(DialogId dialog_id,
                                                                       AccessRights access_rights) const {
  switch (dialog_id.get_type()) {
    case DialogType::User:
      return td_->contacts_manager_->get_input_peer_user(dialog_id.get_user_id(), access_rights);
    case DialogType::Chat:
      return td_->contacts_manager_->get_input_peer_chat(dialog_id.get_chat_id(), access_rights);
    case DialogType::Channel:
      return td_->contacts_manager_->get_input_peer_channel(dialog_id.get_channel_id(), access_rights);
    case DialogType::SecretChat:
      return nullptr;
    case DialogType::None:
      return make_tl_object<telegram_api::inputPeerEmpty>();
    default:
      UNREACHABLE();
      return nullptr;
  }
}

tl_object_ptr<telegram_api::InputPeer> MessagesManager::get_input_peer_force(DialogId dialog_id) {
  switch (dialog_id.get_type()) {
    case DialogType::User: {
      UserId user_id = dialog_id.get_user_id();
      return make_tl_object<telegram_api::inputPeerUser>(user_id.get(), 0);
    }
    case DialogType::Chat: {
      ChatId chat_id = dialog_id.get_chat_id();
      return make_tl_object<telegram_api::inputPeerChat>(chat_id.get());
    }
    case DialogType::Channel: {
      ChannelId channel_id = dialog_id.get_channel_id();
      return make_tl_object<telegram_api::inputPeerChannel>(channel_id.get(), 0);
    }
    case DialogType::SecretChat:
    case DialogType::None:
      return make_tl_object<telegram_api::inputPeerEmpty>();
    default:
      UNREACHABLE();
      return nullptr;
  }
}

vector<tl_object_ptr<telegram_api::InputPeer>> MessagesManager::get_input_peers(const vector<DialogId> &dialog_ids,
                                                                                AccessRights access_rights) const {
  vector<tl_object_ptr<telegram_api::InputPeer>> input_peers;
  input_peers.reserve(dialog_ids.size());
  for (auto &dialog_id : dialog_ids) {
    auto input_peer = get_input_peer(dialog_id, access_rights);
    if (input_peer == nullptr) {
      LOG(ERROR) << "Have no access to " << dialog_id;
      continue;
    }
    input_peers.push_back(std::move(input_peer));
  }
  return input_peers;
}

tl_object_ptr<telegram_api::InputDialogPeer> MessagesManager::get_input_dialog_peer(DialogId dialog_id,
                                                                                    AccessRights access_rights) const {
  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
    case DialogType::None:
      return make_tl_object<telegram_api::inputDialogPeer>(get_input_peer(dialog_id, access_rights));
    case DialogType::SecretChat:
      return nullptr;
    default:
      UNREACHABLE();
      return nullptr;
  }
}

vector<tl_object_ptr<telegram_api::InputDialogPeer>> MessagesManager::get_input_dialog_peers(
    const vector<DialogId> &dialog_ids, AccessRights access_rights) const {
  vector<tl_object_ptr<telegram_api::InputDialogPeer>> input_dialog_peers;
  input_dialog_peers.reserve(dialog_ids.size());
  for (auto &dialog_id : dialog_ids) {
    auto input_dialog_peer = get_input_dialog_peer(dialog_id, access_rights);
    if (input_dialog_peer == nullptr) {
      LOG(ERROR) << "Have no access to " << dialog_id;
      continue;
    }
    input_dialog_peers.push_back(std::move(input_dialog_peer));
  }
  return input_dialog_peers;
}

bool MessagesManager::have_input_peer(DialogId dialog_id, AccessRights access_rights) const {
  switch (dialog_id.get_type()) {
    case DialogType::User: {
      UserId user_id = dialog_id.get_user_id();
      return td_->contacts_manager_->have_input_peer_user(user_id, access_rights);
    }
    case DialogType::Chat: {
      ChatId chat_id = dialog_id.get_chat_id();
      return td_->contacts_manager_->have_input_peer_chat(chat_id, access_rights);
    }
    case DialogType::Channel: {
      ChannelId channel_id = dialog_id.get_channel_id();
      return td_->contacts_manager_->have_input_peer_channel(channel_id, access_rights);
    }
    case DialogType::SecretChat: {
      SecretChatId secret_chat_id = dialog_id.get_secret_chat_id();
      return td_->contacts_manager_->have_input_encrypted_peer(secret_chat_id, access_rights);
    }
    case DialogType::None:
      return false;
    default:
      UNREACHABLE();
      return false;
  }
}

bool MessagesManager::have_dialog_info(DialogId dialog_id) const {
  switch (dialog_id.get_type()) {
    case DialogType::User: {
      UserId user_id = dialog_id.get_user_id();
      return td_->contacts_manager_->have_user(user_id);
    }
    case DialogType::Chat: {
      ChatId chat_id = dialog_id.get_chat_id();
      return td_->contacts_manager_->have_chat(chat_id);
    }
    case DialogType::Channel: {
      ChannelId channel_id = dialog_id.get_channel_id();
      return td_->contacts_manager_->have_channel(channel_id);
    }
    case DialogType::SecretChat: {
      SecretChatId secret_chat_id = dialog_id.get_secret_chat_id();
      return td_->contacts_manager_->have_secret_chat(secret_chat_id);
    }
    case DialogType::None:
    default:
      return false;
  }
}

bool MessagesManager::have_dialog_info_force(DialogId dialog_id) const {
  switch (dialog_id.get_type()) {
    case DialogType::User: {
      UserId user_id = dialog_id.get_user_id();
      return td_->contacts_manager_->have_user_force(user_id);
    }
    case DialogType::Chat: {
      ChatId chat_id = dialog_id.get_chat_id();
      return td_->contacts_manager_->have_chat_force(chat_id);
    }
    case DialogType::Channel: {
      ChannelId channel_id = dialog_id.get_channel_id();
      return td_->contacts_manager_->have_channel_force(channel_id);
    }
    case DialogType::SecretChat: {
      SecretChatId secret_chat_id = dialog_id.get_secret_chat_id();
      return td_->contacts_manager_->have_secret_chat_force(secret_chat_id);
    }
    case DialogType::None:
    default:
      return false;
  }
}

tl_object_ptr<telegram_api::inputEncryptedChat> MessagesManager::get_input_encrypted_chat(
    DialogId dialog_id, AccessRights access_rights) const {
  switch (dialog_id.get_type()) {
    case DialogType::SecretChat: {
      SecretChatId secret_chat_id = dialog_id.get_secret_chat_id();
      return td_->contacts_manager_->get_input_encrypted_chat(secret_chat_id, access_rights);
    }
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
    case DialogType::None:
    default:
      UNREACHABLE();
      return nullptr;
  }
}

bool MessagesManager::is_allowed_useless_update(const tl_object_ptr<telegram_api::Update> &update) {
  auto constructor_id = update->get_id();
  if (constructor_id == dummyUpdate::ID) {
    // allow dummyUpdate just in case
    return true;
  }
  if (constructor_id == telegram_api::updateNewMessage::ID ||
      constructor_id == telegram_api::updateNewChannelMessage::ID) {
    // new outgoing messages are received again if random_id coincide
    return true;
  }

  return false;
}

void MessagesManager::skip_old_pending_pts_update(tl_object_ptr<telegram_api::Update> &&update, int32 new_pts,
                                                  int32 old_pts, int32 pts_count, const char *source) {
  if (update->get_id() == telegram_api::updateNewMessage::ID) {
    auto update_new_message = static_cast<telegram_api::updateNewMessage *>(update.get());
    auto full_message_id = get_full_message_id(update_new_message->message_, false);
    if (update_message_ids_.find(full_message_id) != update_message_ids_.end()) {
      if (new_pts == old_pts) {  // otherwise message can be already deleted
        // apply sent message anyway
        on_get_message(std::move(update_new_message->message_), true, false, false, true, true,
                       "updateNewMessage with an awaited message");
        return;
      } else {
        LOG(ERROR) << "Receive awaited sent " << full_message_id << " from " << source << " with pts " << new_pts
                   << " and pts_count " << pts_count << ", but current pts is " << old_pts;
        dump_debug_message_op(get_dialog(full_message_id.get_dialog_id()), 3);
      }
    }
  }
  if (update->get_id() == updateSentMessage::ID) {
    auto update_sent_message = static_cast<updateSentMessage *>(update.get());
    if (being_sent_messages_.count(update_sent_message->random_id_) > 0) {
      if (new_pts == old_pts) {  // otherwise message can be already deleted
        // apply sent message anyway
        on_send_message_success(update_sent_message->random_id_, update_sent_message->message_id_,
                                update_sent_message->date_, update_sent_message->ttl_period_, FileId(),
                                "process old updateSentMessage");
        return;
      } else {
        LOG(ERROR) << "Receive awaited sent " << update_sent_message->message_id_ << " from " << source << " with pts "
                   << new_pts << " and pts_count " << pts_count << ", but current pts is " << old_pts;
        dump_debug_message_op(get_dialog(being_sent_messages_[update_sent_message->random_id_].get_dialog_id()), 3);
      }
    }
    return;
  }

  // very old or unuseful update
  LOG_IF(WARNING, new_pts == old_pts && pts_count == 0 && !is_allowed_useless_update(update))
      << "Receive useless update " << oneline(to_string(update)) << " from " << source;
}

MessagesManager::Dialog *MessagesManager::get_service_notifications_dialog() {
  UserId service_notifications_user_id = td_->contacts_manager_->add_service_notifications_user();
  DialogId service_notifications_dialog_id(service_notifications_user_id);
  force_create_dialog(service_notifications_dialog_id, "get_service_notifications_dialog");
  return get_dialog(service_notifications_dialog_id);
}

void MessagesManager::save_auth_notification_ids() {
  auto min_date = G()->unix_time() - AUTH_NOTIFICATION_ID_CACHE_TIME;
  vector<string> ids;
  for (auto &it : auth_notification_id_date_) {
    auto date = it.second;
    if (date < min_date) {
      continue;
    }
    ids.push_back(it.first);
    ids.push_back(to_string(date));
  }

  if (ids.empty()) {
    G()->td_db()->get_binlog_pmc()->erase("auth_notification_ids");
    return;
  }

  G()->td_db()->get_binlog_pmc()->set("auth_notification_ids", implode(ids, ','));
}

void MessagesManager::on_update_service_notification(tl_object_ptr<telegram_api::updateServiceNotification> &&update,
                                                     bool skip_new_entities, Promise<Unit> &&promise) {
  bool has_date = (update->flags_ & telegram_api::updateServiceNotification::INBOX_DATE_MASK) != 0;
  auto date = has_date ? update->inbox_date_ : G()->unix_time();
  if (date <= 0) {
    LOG(ERROR) << "Receive message date " << date << " in " << to_string(update);
    return;
  }
  bool is_auth_notification = begins_with(update->type_, "auth");
  if (is_auth_notification) {
    auto &old_date = auth_notification_id_date_[update->type_.substr(4)];
    if (date <= old_date) {
      LOG(INFO) << "Skip already applied " << to_string(update);
      return;
    }
    old_date = date;
  }

  bool is_authorized = td_->auth_manager_->is_authorized();
  bool is_user = is_authorized && !td_->auth_manager_->is_bot();
  auto contacts_manager = is_authorized ? td_->contacts_manager_.get() : nullptr;
  auto message_text = get_message_text(contacts_manager, std::move(update->message_), std::move(update->entities_),
                                       skip_new_entities, !is_user, date, false, "on_update_service_notification");
  DialogId owner_dialog_id = is_user ? get_service_notifications_dialog()->dialog_id : DialogId();
  int32 ttl = 0;
  bool disable_web_page_preview = false;
  auto content = get_message_content(td_, std::move(message_text), std::move(update->media_), owner_dialog_id, false,
                                     UserId(), &ttl, &disable_web_page_preview);
  bool is_content_secret = is_secret_message_content(ttl, content->get_type());

  if (update->popup_) {
    send_closure(G()->td(), &Td::send_update,
                 td_api::make_object<td_api::updateServiceNotification>(
                     update->type_, get_message_content_object(content.get(), td_, owner_dialog_id, date,
                                                               is_content_secret, true, -1)));
  }
  if (has_date && is_user) {
    Dialog *d = get_service_notifications_dialog();
    CHECK(d != nullptr);
    auto dialog_id = d->dialog_id;
    CHECK(dialog_id.get_type() == DialogType::User);

    auto new_message = make_unique<Message>();
    set_message_id(new_message, get_next_local_message_id(d));
    new_message->sender_user_id = dialog_id.get_user_id();
    new_message->date = date;
    new_message->ttl = ttl;
    new_message->disable_web_page_preview = disable_web_page_preview;
    new_message->is_content_secret = is_content_secret;
    new_message->content = std::move(content);
    new_message->have_previous = true;
    new_message->have_next = true;

    bool need_update = true;
    bool need_update_dialog_pos = false;

    const Message *m = add_message_to_dialog(d, std::move(new_message), true, &need_update, &need_update_dialog_pos,
                                             "on_update_service_notification");
    if (m != nullptr && need_update) {
      send_update_new_message(d, m);
    }
    register_new_local_message_id(d, m);

    if (need_update_dialog_pos) {
      send_update_chat_last_message(d, "on_update_service_notification");
    }
  }
  promise.set_value(Unit());

  if (is_auth_notification) {
    save_auth_notification_ids();
  }
}

void MessagesManager::on_update_read_channel_inbox(tl_object_ptr<telegram_api::updateReadChannelInbox> &&update) {
  ChannelId channel_id(update->channel_id_);
  if (!channel_id.is_valid()) {
    LOG(ERROR) << "Receive invalid " << channel_id << " in updateReadChannelInbox";
    return;
  }

  FolderId folder_id;
  if ((update->flags_ & telegram_api::updateReadChannelInbox::FOLDER_ID_MASK) != 0) {
    folder_id = FolderId(update->folder_id_);
  }
  on_update_dialog_folder_id(DialogId(channel_id), folder_id);
  on_read_channel_inbox(channel_id, MessageId(ServerMessageId(update->max_id_)), update->still_unread_count_,
                        update->pts_, "updateReadChannelInbox");
}

void MessagesManager::on_update_read_channel_outbox(tl_object_ptr<telegram_api::updateReadChannelOutbox> &&update) {
  ChannelId channel_id(update->channel_id_);
  if (!channel_id.is_valid()) {
    LOG(ERROR) << "Receive invalid " << channel_id << " in updateReadChannelOutbox";
    return;
  }

  DialogId dialog_id = DialogId(channel_id);
  read_history_outbox(dialog_id, MessageId(ServerMessageId(update->max_id_)));
}

void MessagesManager::on_update_read_channel_messages_contents(
    tl_object_ptr<telegram_api::updateChannelReadMessagesContents> &&update) {
  ChannelId channel_id(update->channel_id_);
  if (!channel_id.is_valid()) {
    LOG(ERROR) << "Receive invalid " << channel_id << " in updateChannelReadMessagesContents";
    return;
  }

  DialogId dialog_id = DialogId(channel_id);

  Dialog *d = get_dialog_force(dialog_id, "on_update_read_channel_messages_contents");
  if (d == nullptr) {
    LOG(INFO) << "Receive read channel messages contents update in unknown " << dialog_id;
    return;
  }

  for (auto &server_message_id : update->messages_) {
    read_channel_message_content_from_updates(d, MessageId(ServerMessageId(server_message_id)));
  }
}

void MessagesManager::on_update_read_message_comments(DialogId dialog_id, MessageId message_id,
                                                      MessageId max_message_id, MessageId last_read_inbox_message_id,
                                                      MessageId last_read_outbox_message_id) {
  Dialog *d = get_dialog_force(dialog_id, "on_update_read_message_comments");
  if (d == nullptr) {
    LOG(INFO) << "Ignore update of read message comments in unknown " << dialog_id << " in updateReadDiscussion";
    return;
  }

  auto m = get_message_force(d, message_id, "on_update_read_message_comments");
  if (m == nullptr || !m->message_id.is_server() || m->top_thread_message_id != m->message_id ||
      !is_active_message_reply_info(dialog_id, m->reply_info)) {
    return;
  }
  if (m->reply_info.update_max_message_ids(max_message_id, last_read_inbox_message_id, last_read_outbox_message_id)) {
    on_message_reply_info_changed(dialog_id, m);
    on_message_changed(d, m, true, "on_update_read_message_comments");
  }
}

void MessagesManager::on_update_channel_too_long(tl_object_ptr<telegram_api::updateChannelTooLong> &&update,
                                                 bool force_apply) {
  ChannelId channel_id(update->channel_id_);
  if (!channel_id.is_valid()) {
    LOG(ERROR) << "Receive invalid " << channel_id << " in updateChannelTooLong";
    return;
  }

  DialogId dialog_id = DialogId(channel_id);
  auto d = get_dialog_force(dialog_id, "on_update_channel_too_long 4");
  if (d == nullptr) {
    auto pts = load_channel_pts(dialog_id);
    if (pts > 0) {
      d = add_dialog(dialog_id, "on_update_channel_too_long 5");
      CHECK(d != nullptr);
      CHECK(d->pts == pts);
      update_dialog_pos(d, "on_update_channel_too_long 6");
    }
  }

  int32 update_pts = (update->flags_ & UPDATE_CHANNEL_TO_LONG_FLAG_HAS_PTS) ? update->pts_ : 0;

  if (d != nullptr) {
    if (update_pts == 0 || update_pts > d->pts) {
      get_channel_difference(dialog_id, d->pts, true, "on_update_channel_too_long 1");
    }
  } else {
    if (force_apply) {
      get_channel_difference(dialog_id, -1, true, "on_update_channel_too_long 2");
    } else {
      td_->updates_manager_->schedule_get_difference("on_update_channel_too_long 3");
    }
  }
}

void MessagesManager::on_update_message_view_count(FullMessageId full_message_id, int32 view_count) {
  if (view_count < 0) {
    LOG(ERROR) << "Receive " << view_count << " views in updateChannelMessageViews for " << full_message_id;
    return;
  }
  update_message_interaction_info(full_message_id, view_count, -1, false, nullptr);
}

void MessagesManager::on_update_message_forward_count(FullMessageId full_message_id, int32 forward_count) {
  if (forward_count < 0) {
    LOG(ERROR) << "Receive " << forward_count << " forwards in updateChannelMessageForwards for " << full_message_id;
    return;
  }
  update_message_interaction_info(full_message_id, -1, forward_count, false, nullptr);
}

void MessagesManager::on_update_message_interaction_info(FullMessageId full_message_id, int32 view_count,
                                                         int32 forward_count, bool has_reply_info,
                                                         tl_object_ptr<telegram_api::messageReplies> &&reply_info) {
  if (view_count < 0 || forward_count < 0) {
    LOG(ERROR) << "Receive " << view_count << "/" << forward_count << " interaction counters for " << full_message_id;
    return;
  }
  update_message_interaction_info(full_message_id, view_count, forward_count, has_reply_info, std::move(reply_info));
}

void MessagesManager::on_pending_message_views_timeout(DialogId dialog_id) {
  if (G()->close_flag()) {
    return;
  }

  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  const size_t MAX_MESSAGE_VIEWS = 100;  // server side limit
  vector<MessageId> message_ids;
  message_ids.reserve(min(d->pending_viewed_message_ids.size(), MAX_MESSAGE_VIEWS));
  for (auto message_id : d->pending_viewed_message_ids) {
    message_ids.push_back(message_id);
    if (message_ids.size() >= MAX_MESSAGE_VIEWS) {
      td_->create_handler<GetMessagesViewsQuery>()->send(dialog_id, std::move(message_ids), d->increment_view_counter);
      message_ids.clear();
    }
  }
  if (!message_ids.empty()) {
    td_->create_handler<GetMessagesViewsQuery>()->send(dialog_id, std::move(message_ids), d->increment_view_counter);
  }
  d->pending_viewed_message_ids.clear();
  d->increment_view_counter = false;
}

void MessagesManager::update_message_interaction_info(FullMessageId full_message_id, int32 view_count,
                                                      int32 forward_count, bool has_reply_info,
                                                      tl_object_ptr<telegram_api::messageReplies> &&reply_info) {
  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id, "update_message_interaction_info");
  if (d == nullptr) {
    return;
  }
  auto message_id = full_message_id.get_message_id();
  Message *m = get_message_force(d, message_id, "update_message_interaction_info");
  if (m == nullptr) {
    LOG(INFO) << "Ignore message interaction info about unknown " << full_message_id;
    if (!message_id.is_scheduled() && message_id > d->last_new_message_id &&
        dialog_id.get_type() == DialogType::Channel) {
      get_channel_difference(dialog_id, d->pts, true, "update_message_interaction_info");
    }
    return;
  }

  if (view_count < 0) {
    view_count = m->view_count;
  }
  if (forward_count < 0) {
    forward_count = m->forward_count;
  }
  bool is_empty_reply_info = reply_info == nullptr;
  MessageReplyInfo new_reply_info(std::move(reply_info), td_->auth_manager_->is_bot());
  if (new_reply_info.is_empty() && !is_empty_reply_info) {
    has_reply_info = false;
  }

  if (update_message_interaction_info(dialog_id, m, view_count, forward_count, has_reply_info,
                                      std::move(new_reply_info), "update_message_interaction_info")) {
    on_message_changed(d, m, true, "update_message_interaction_info");
  }
}

bool MessagesManager::is_active_message_reply_info(DialogId dialog_id, const MessageReplyInfo &info) const {
  if (info.is_empty()) {
    return false;
  }
  if (dialog_id.get_type() != DialogType::Channel) {
    return false;
  }

  if (!info.is_comment) {
    return true;
  }
  if (!is_broadcast_channel(dialog_id)) {
    return true;
  }

  auto channel_id = dialog_id.get_channel_id();
  if (!td_->contacts_manager_->get_channel_has_linked_channel(channel_id)) {
    return false;
  }

  auto linked_channel_id = td_->contacts_manager_->get_channel_linked_channel_id(channel_id);
  if (!linked_channel_id.is_valid()) {
    // keep the comment button while linked channel is unknown
    send_closure_later(G()->contacts_manager(), &ContactsManager::load_channel_full, channel_id, false, Promise<Unit>(),
                       "is_active_message_reply_info");
    return true;
  }

  return linked_channel_id == info.channel_id;
}

bool MessagesManager::is_visible_message_reply_info(DialogId dialog_id, const Message *m) const {
  CHECK(m != nullptr);
  if (!m->message_id.is_valid()) {
    return false;
  }
  bool is_broadcast = is_broadcast_channel(dialog_id);
  if (!m->message_id.is_server() && !(is_broadcast && m->message_id.is_yet_unsent())) {
    return false;
  }
  if (is_broadcast && (m->had_reply_markup || m->reply_markup != nullptr)) {
    return false;
  }
  return is_active_message_reply_info(dialog_id, m->reply_info);
}

void MessagesManager::on_message_reply_info_changed(DialogId dialog_id, const Message *m) const {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  if (is_visible_message_reply_info(dialog_id, m)) {
    send_update_message_interaction_info(dialog_id, m);
  }
}

td_api::object_ptr<td_api::messageInteractionInfo> MessagesManager::get_message_interaction_info_object(
    DialogId dialog_id, const Message *m) const {
  bool is_visible_reply_info = is_visible_message_reply_info(dialog_id, m);
  if (m->view_count == 0 && m->forward_count == 0 && !is_visible_reply_info) {
    return nullptr;
  }
  if (m->message_id.is_scheduled() && (m->forward_info == nullptr || is_broadcast_channel(dialog_id))) {
    return nullptr;
  }
  if (m->message_id.is_local() && m->forward_info == nullptr) {
    return nullptr;
  }

  td_api::object_ptr<td_api::messageReplyInfo> reply_info;
  if (is_visible_reply_info) {
    reply_info = m->reply_info.get_message_reply_info_object(td_->contacts_manager_.get(), this);
    CHECK(reply_info != nullptr);
  }

  return td_api::make_object<td_api::messageInteractionInfo>(m->view_count, m->forward_count, std::move(reply_info));
}

bool MessagesManager::update_message_interaction_info(DialogId dialog_id, Message *m, int32 view_count,
                                                      int32 forward_count, bool has_reply_info,
                                                      MessageReplyInfo &&reply_info, const char *source) {
  CHECK(m != nullptr);
  m->interaction_info_update_date = G()->unix_time();  // doesn't force message saving
  if (m->message_id.is_valid_scheduled()) {
    has_reply_info = false;
  }
  bool need_update_reply_info = has_reply_info && m->reply_info.need_update_to(reply_info);
  if (has_reply_info && m->reply_info.channel_id == reply_info.channel_id) {
    if (need_update_reply_info) {
      reply_info.update_max_message_ids(m->reply_info);
    } else {
      if (m->reply_info.update_max_message_ids(reply_info) && view_count <= m->view_count &&
          forward_count <= m->forward_count) {
        on_message_reply_info_changed(dialog_id, m);
        on_message_changed(get_dialog(dialog_id), m, true, "update_message_interaction_info");
      }
    }
  }
  if (view_count > m->view_count || forward_count > m->forward_count || need_update_reply_info) {
    LOG(DEBUG) << "Update interaction info of " << FullMessageId{dialog_id, m->message_id} << " from " << m->view_count
               << '/' << m->forward_count << "/" << m->reply_info << " to " << view_count << '/' << forward_count << "/"
               << reply_info;
    bool need_update = false;
    if (view_count > m->view_count) {
      m->view_count = view_count;
      need_update = true;
    }
    if (forward_count > m->forward_count) {
      m->forward_count = forward_count;
      need_update = true;
    }
    if (need_update_reply_info) {
      if (m->reply_info.channel_id != reply_info.channel_id) {
        if (m->reply_info.channel_id.is_valid() && reply_info.channel_id.is_valid() && m->message_id.is_server()) {
          LOG(ERROR) << "Reply info of " << FullMessageId{dialog_id, m->message_id} << " changed from " << m->reply_info
                     << " to " << reply_info << " from " << source;
        }
      }
      m->reply_info = std::move(reply_info);
      if (!m->top_thread_message_id.is_valid() && !is_broadcast_channel(dialog_id) &&
          is_visible_message_reply_info(dialog_id, m)) {
        m->top_thread_message_id = m->message_id;
      }
      need_update |= is_visible_message_reply_info(dialog_id, m);
    }
    if (need_update) {
      send_update_message_interaction_info(dialog_id, m);
    }
    return true;
  }
  return false;
}

void MessagesManager::on_update_live_location_viewed(FullMessageId full_message_id) {
  LOG(DEBUG) << "Live location was viewed in " << full_message_id;
  if (!are_active_live_location_messages_loaded_) {
    get_active_live_location_messages(PromiseCreator::lambda([actor_id = actor_id(this), full_message_id](Unit result) {
      send_closure(actor_id, &MessagesManager::on_update_live_location_viewed, full_message_id);
    }));
    return;
  }

  auto active_live_location_message_ids = get_active_live_location_messages(Auto());
  if (!td::contains(active_live_location_message_ids, full_message_id)) {
    LOG(DEBUG) << "Can't find " << full_message_id << " in " << active_live_location_message_ids;
    return;
  }

  send_update_message_live_location_viewed(full_message_id);
}

void MessagesManager::on_update_some_live_location_viewed(Promise<Unit> &&promise) {
  LOG(DEBUG) << "Some live location was viewed";
  if (!are_active_live_location_messages_loaded_) {
    get_active_live_location_messages(
        PromiseCreator::lambda([actor_id = actor_id(this), promise = std::move(promise)](Unit result) mutable {
          send_closure(actor_id, &MessagesManager::on_update_some_live_location_viewed, std::move(promise));
        }));
    return;
  }

  // update all live locations, because it is unknown, which exactly was viewed
  auto active_live_location_message_ids = get_active_live_location_messages(Auto());
  for (auto full_message_id : active_live_location_message_ids) {
    send_update_message_live_location_viewed(full_message_id);
  }

  promise.set_value(Unit());
}

bool MessagesManager::need_skip_bot_commands(DialogId dialog_id, const Message *m) const {
  if (td_->auth_manager_->is_bot()) {
    return false;
  }

  if (m != nullptr && m->message_id.is_scheduled()) {
    return true;
  }

  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  return (d->is_has_bots_inited && !d->has_bots) || is_broadcast_channel(dialog_id);
}

void MessagesManager::on_external_update_message_content(FullMessageId full_message_id) {
  Dialog *d = get_dialog(full_message_id.get_dialog_id());
  CHECK(d != nullptr);
  Message *m = get_message(d, full_message_id.get_message_id());
  CHECK(m != nullptr);
  send_update_message_content(d, m, true, "on_external_update_message_content");
  if (m->message_id == d->last_message_id) {
    send_update_chat_last_message_impl(d, "on_external_update_message_content");
  }
}

bool MessagesManager::update_message_contains_unread_mention(Dialog *d, Message *m, bool contains_unread_mention,
                                                             const char *source) {
  LOG_CHECK(m != nullptr) << source;
  CHECK(!m->message_id.is_scheduled());
  if (!contains_unread_mention && m->contains_unread_mention) {
    remove_message_notification_id(d, m, true, true);  // should be called before contains_unread_mention is updated

    m->contains_unread_mention = false;
    if (d->unread_mention_count == 0) {
      if (is_dialog_inited(d)) {
        LOG(ERROR) << "Unread mention count of " << d->dialog_id << " became negative from " << source;
      }
    } else {
      set_dialog_unread_mention_count(d, d->unread_mention_count - 1);
      on_dialog_updated(d->dialog_id, "update_message_contains_unread_mention");
    }
    LOG(INFO) << "Update unread mention message count in " << d->dialog_id << " to " << d->unread_mention_count
              << " by reading " << m->message_id << " from " << source;

    send_closure(G()->td(), &Td::send_update,
                 make_tl_object<td_api::updateMessageMentionRead>(d->dialog_id.get(), m->message_id.get(),
                                                                  d->unread_mention_count));
    return true;
  }
  return false;
}

void MessagesManager::on_read_channel_inbox(ChannelId channel_id, MessageId max_message_id, int32 server_unread_count,
                                            int32 pts, const char *source) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(!max_message_id.is_scheduled());
  if (!max_message_id.is_valid() && server_unread_count <= 0) {
    return;
  }

  DialogId dialog_id(channel_id);
  Dialog *d = get_dialog_force(dialog_id, source);
  if (d == nullptr) {
    LOG(INFO) << "Receive read inbox in unknown " << dialog_id << " from " << source;
    return;
  }

  /*
  // dropping unread count can make things worse, so don't drop it
  if (server_unread_count > 0 && G()->parameters().use_message_db && d->is_last_read_inbox_message_id_inited) {
    server_unread_count = -1;
  }
  */

  if (d->pts == pts) {
    read_history_inbox(dialog_id, max_message_id, server_unread_count, source);
  } else if (d->pts > pts) {
    // outdated update, need to repair server_unread_count from the server
    repair_channel_server_unread_count(d);
  } else {
    // update from the future, keep it until it can be applied
    if (pts >= d->pending_read_channel_inbox_pts) {
      if (d->pending_read_channel_inbox_pts == 0) {
        channel_get_difference_retry_timeout_.add_timeout_in(dialog_id.get(), 0.001);
      }
      d->pending_read_channel_inbox_pts = pts;
      d->pending_read_channel_inbox_max_message_id = max_message_id;
      d->pending_read_channel_inbox_server_unread_count = server_unread_count;
      on_dialog_updated(dialog_id, "on_read_channel_inbox");
    }
  }
}

void MessagesManager::on_read_channel_outbox(ChannelId channel_id, MessageId max_message_id) {
  DialogId dialog_id(channel_id);
  CHECK(!max_message_id.is_scheduled());
  if (max_message_id.is_valid()) {
    read_history_outbox(dialog_id, max_message_id);
  }
}

void MessagesManager::on_update_channel_max_unavailable_message_id(ChannelId channel_id,
                                                                   MessageId max_unavailable_message_id) {
  if (!channel_id.is_valid()) {
    LOG(ERROR) << "Receive max_unavailable_message_id in invalid " << channel_id;
    return;
  }

  DialogId dialog_id(channel_id);
  CHECK(!max_unavailable_message_id.is_scheduled());
  if (!max_unavailable_message_id.is_valid() && max_unavailable_message_id != MessageId()) {
    LOG(ERROR) << "Receive wrong max_unavailable_message_id: " << max_unavailable_message_id;
    max_unavailable_message_id = MessageId();
  }
  set_dialog_max_unavailable_message_id(dialog_id, max_unavailable_message_id, true,
                                        "on_update_channel_max_unavailable_message_id");
}

void MessagesManager::on_update_dialog_online_member_count(DialogId dialog_id, int32 online_member_count,
                                                           bool is_from_server) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  if (!dialog_id.is_valid()) {
    LOG(ERROR) << "Receive number of online members in invalid " << dialog_id;
    return;
  }

  if (is_broadcast_channel(dialog_id)) {
    LOG_IF(ERROR, online_member_count != 0)
        << "Receive " << online_member_count << " as a number of online members in a channel " << dialog_id;
    return;
  }

  if (online_member_count < 0) {
    LOG(ERROR) << "Receive " << online_member_count << " as a number of online members in a " << dialog_id;
    return;
  }

  set_dialog_online_member_count(dialog_id, online_member_count, is_from_server,
                                 "on_update_channel_online_member_count");
}

void MessagesManager::on_update_delete_scheduled_messages(DialogId dialog_id,
                                                          vector<ScheduledServerMessageId> &&server_message_ids) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  if (!dialog_id.is_valid()) {
    LOG(ERROR) << "Receive deleted scheduled messages in invalid " << dialog_id;
    return;
  }

  Dialog *d = get_dialog_force(dialog_id, "on_update_delete_scheduled_messages");
  if (d == nullptr) {
    LOG(INFO) << "Skip updateDeleteScheduledMessages in unknown " << dialog_id;
    return;
  }

  vector<int64> deleted_message_ids;
  for (auto server_message_id : server_message_ids) {
    if (!server_message_id.is_valid()) {
      LOG(ERROR) << "Incoming update tries to delete scheduled message " << server_message_id.get();
      continue;
    }

    auto message = do_delete_scheduled_message(d, MessageId(server_message_id, std::numeric_limits<int32>::max()), true,
                                               "on_update_delete_scheduled_messages");
    if (message != nullptr) {
      deleted_message_ids.push_back(message->message_id.get());
    }
  }

  send_update_delete_messages(dialog_id, std::move(deleted_message_ids), true, false);

  send_update_chat_has_scheduled_messages(d, true);
}

void MessagesManager::on_user_dialog_action(DialogId dialog_id, MessageId top_thread_message_id,
                                            DialogId typing_dialog_id, DialogAction action, int32 date,
                                            MessageContentType message_content_type) {
  if (td_->auth_manager_->is_bot() || !typing_dialog_id.is_valid() || is_broadcast_channel(dialog_id)) {
    return;
  }
  if (top_thread_message_id != MessageId() && !top_thread_message_id.is_valid()) {
    LOG(ERROR) << "Ignore " << action << " in the message thread of " << top_thread_message_id;
    return;
  }

  auto dialog_type = dialog_id.get_type();
  if (action == DialogAction::get_speaking_action()) {
    if ((dialog_type != DialogType::Chat && dialog_type != DialogType::Channel) || top_thread_message_id.is_valid()) {
      LOG(ERROR) << "Receive " << action << " in thread of " << top_thread_message_id << " in " << dialog_id;
      return;
    }
    const Dialog *d = get_dialog_force(dialog_id, "on_user_dialog_action");
    if (d != nullptr && d->active_group_call_id.is_valid()) {
      auto group_call_id = td_->group_call_manager_->get_group_call_id(d->active_group_call_id, dialog_id);
      td_->group_call_manager_->on_user_speaking_in_group_call(group_call_id, typing_dialog_id, date);
    }
    return;
  }

  if (typing_dialog_id.get_type() != DialogType::User) {
    LOG(ERROR) << "Ignore " << action << " of " << typing_dialog_id << " in " << dialog_id;
    return;
  }
  auto user_id = typing_dialog_id.get_user_id();

  {
    auto message_import_progress = action.get_importing_messages_action_progress();
    if (message_import_progress >= 0) {
      // TODO
      return;
    }
  }

  {
    auto clicking_info = action.get_clicking_animated_emoji_action_info();
    if (!clicking_info.data.empty()) {
      auto message_id = MessageId(ServerMessageId(clicking_info.message_id));
      CHECK(message_id.is_valid());
      if (date > G()->unix_time() - 10) {
        on_animated_emoji_message_clicked({dialog_id, message_id}, user_id, clicking_info.emoji,
                                          std::move(clicking_info.data));
      }
      return;
    }
  }

  if (is_unsent_animated_emoji_click(td_, dialog_id, action)) {
    LOG(DEBUG) << "Ignore unsent " << action;
    return;
  }

  if (!td_->contacts_manager_->have_min_user(user_id)) {
    LOG(DEBUG) << "Ignore " << action << " of unknown " << user_id;
    return;
  }
  if (!have_dialog(dialog_id)) {
    LOG(DEBUG) << "Ignore " << action << " in unknown " << dialog_id;
    return;
  }

  bool is_canceled = action == DialogAction();
  if (!is_canceled || message_content_type != MessageContentType::None) {
    td_->contacts_manager_->on_update_user_local_was_online(user_id, date);
  }

  if (dialog_type == DialogType::User || dialog_type == DialogType::SecretChat) {
    if (!td_->contacts_manager_->is_user_bot(user_id) && !td_->contacts_manager_->is_user_status_exact(user_id) &&
        !get_dialog(dialog_id)->is_opened && !is_canceled) {
      return;
    }
  }

  if (is_canceled) {
    auto actions_it = active_dialog_actions_.find(dialog_id);
    if (actions_it == active_dialog_actions_.end()) {
      return;
    }

    auto &active_actions = actions_it->second;
    auto it = std::find_if(active_actions.begin(), active_actions.end(),
                           [user_id](const ActiveDialogAction &action) { return action.user_id == user_id; });
    if (it == active_actions.end()) {
      return;
    }

    if (!td_->contacts_manager_->is_user_bot(user_id) &&
        !it->action.is_canceled_by_message_of_type(message_content_type)) {
      return;
    }

    LOG(DEBUG) << "Cancel action of " << user_id << " in " << dialog_id;
    top_thread_message_id = it->top_thread_message_id;
    active_actions.erase(it);
    if (active_actions.empty()) {
      active_dialog_actions_.erase(dialog_id);
      LOG(DEBUG) << "Cancel action timeout in " << dialog_id;
      active_dialog_action_timeout_.cancel_timeout(dialog_id.get());
    }
  } else {
    if (date < G()->unix_time_cached() - DIALOG_ACTION_TIMEOUT - 60) {
      LOG(DEBUG) << "Ignore too old action of " << user_id << " in " << dialog_id << " sent at " << date;
      return;
    }
    auto &active_actions = active_dialog_actions_[dialog_id];
    auto it = std::find_if(active_actions.begin(), active_actions.end(),
                           [user_id](const ActiveDialogAction &action) { return action.user_id == user_id; });
    MessageId prev_top_thread_message_id;
    DialogAction prev_action;
    if (it != active_actions.end()) {
      LOG(DEBUG) << "Re-add action of " << user_id << " in " << dialog_id;
      prev_top_thread_message_id = it->top_thread_message_id;
      prev_action = it->action;
      active_actions.erase(it);
    } else {
      LOG(DEBUG) << "Add action of " << user_id << " in " << dialog_id;
    }

    active_actions.emplace_back(top_thread_message_id, user_id, action, Time::now());
    if (top_thread_message_id == prev_top_thread_message_id && action == prev_action) {
      return;
    }
    if (top_thread_message_id != prev_top_thread_message_id && prev_top_thread_message_id.is_valid()) {
      send_update_user_chat_action(dialog_id, prev_top_thread_message_id, user_id, DialogAction());
    }
    if (active_actions.size() == 1u) {
      LOG(DEBUG) << "Set action timeout in " << dialog_id;
      active_dialog_action_timeout_.set_timeout_in(dialog_id.get(), DIALOG_ACTION_TIMEOUT);
    }
  }

  if (top_thread_message_id.is_valid()) {
    send_update_user_chat_action(dialog_id, MessageId(), user_id, action);
  }
  send_update_user_chat_action(dialog_id, top_thread_message_id, user_id, action);
}

void MessagesManager::cancel_user_dialog_action(DialogId dialog_id, const Message *m) {
  CHECK(m != nullptr);
  if (td_->auth_manager_->is_bot() || m->forward_info != nullptr || m->had_forward_info ||
      m->via_bot_user_id.is_valid() || m->hide_via_bot || m->is_channel_post || m->message_id.is_scheduled()) {
    return;
  }

  on_user_dialog_action(dialog_id, MessageId(), DialogId(m->sender_user_id), DialogAction(), m->date,
                        m->content->get_type());
}

void MessagesManager::add_postponed_channel_update(DialogId dialog_id, tl_object_ptr<telegram_api::Update> &&update,
                                                   int32 new_pts, int32 pts_count, Promise<Unit> &&promise) {
  postponed_channel_updates_[dialog_id].emplace(
      new_pts, PendingPtsUpdate(std::move(update), new_pts, pts_count, std::move(promise)));
}

void MessagesManager::add_pending_channel_update(DialogId dialog_id, tl_object_ptr<telegram_api::Update> &&update,
                                                 int32 new_pts, int32 pts_count, Promise<Unit> &&promise,
                                                 const char *source, bool is_postponed_update) {
  LOG(INFO) << "Receive from " << source << " pending " << to_string(update);
  CHECK(update != nullptr);
  if (dialog_id.get_type() != DialogType::Channel) {
    LOG(ERROR) << "Receive channel update in invalid " << dialog_id << " from " << source << ": "
               << oneline(to_string(update));
    promise.set_value(Unit());
    return;
  }
  if (pts_count < 0 || new_pts <= pts_count) {
    LOG(ERROR) << "Receive channel update from " << source << " with wrong pts = " << new_pts
               << " or pts_count = " << pts_count << ": " << oneline(to_string(update));
    promise.set_value(Unit());
    return;
  }

  auto channel_id = dialog_id.get_channel_id();
  if (!td_->contacts_manager_->have_channel(channel_id) && td_->contacts_manager_->have_min_channel(channel_id)) {
    td_->updates_manager_->schedule_get_difference("add_pending_channel_update 1");
    promise.set_value(Unit());
    return;
  }

  // TODO need to save all updates that can change result of running queries not associated with pts (for example
  // getHistory) and apply them to result of these queries

  Dialog *d = get_dialog_force(dialog_id, "add_pending_channel_update 2");
  if (d == nullptr) {
    auto pts = load_channel_pts(dialog_id);
    if (pts > 0) {
      if (!td_->contacts_manager_->have_channel(channel_id)) {
        // do not create dialog if there is no info about the channel
        LOG(INFO) << "There is no info about " << channel_id << ", so ignore " << oneline(to_string(update));
        promise.set_value(Unit());
        return;
      }

      if (new_pts <= pts && new_pts >= pts - 19999) {
        LOG(INFO) << "There is no need to process an update with pts " << new_pts << " in " << dialog_id << " with pts "
                  << pts;
        promise.set_value(Unit());
        return;
      }

      if (new_pts > pts && pts != new_pts - pts_count) {
        LOG(INFO) << "Found a gap in unknown " << dialog_id << " with pts = " << pts << ". new_pts = " << new_pts
                  << ", pts_count = " << pts_count << " in update from " << source;
        add_postponed_channel_update(dialog_id, std::move(update), new_pts, pts_count, std::move(promise));
        get_channel_difference(dialog_id, pts, true, "add_pending_channel_update 3");
        return;
      }

      d = add_dialog(dialog_id, "add_pending_channel_update 4");
      CHECK(d != nullptr);
      CHECK(d->pts == pts);
      update_dialog_pos(d, "add_pending_channel_update 5");
    }
  }
  if (d == nullptr) {
    // if there is no dialog, it can be created by the update
    LOG(INFO) << "Receive pending update from " << source << " about unknown " << dialog_id;
    if (running_get_channel_difference(dialog_id)) {
      add_postponed_channel_update(dialog_id, std::move(update), new_pts, pts_count, std::move(promise));
      return;
    }
  } else {
    int32 old_pts = d->pts;
    if (new_pts <= old_pts) {  // very old or unuseful update
      if (new_pts < old_pts - 19999 && !is_postponed_update) {
        // restore channel pts after delete_first_messages
        auto now = Time::now();
        if (now > last_channel_pts_jump_warning_time_ + 1) {
          LOG(ERROR) << "Restore pts in " << d->dialog_id << " from " << source << " after delete_first_messages from "
                     << old_pts << " to " << new_pts << " is temporarily disabled, pts_count = " << pts_count
                     << ", update is from " << source << ": " << oneline(to_string(update));
          last_channel_pts_jump_warning_time_ = now;
        }
        get_channel_difference(dialog_id, old_pts, true, "add_pending_channel_update old");
      }

      if (update->get_id() == telegram_api::updateNewChannelMessage::ID) {
        auto update_new_channel_message = static_cast<telegram_api::updateNewChannelMessage *>(update.get());
        auto message_id = get_message_id(update_new_channel_message->message_, false);
        FullMessageId full_message_id(dialog_id, message_id);
        if (update_message_ids_.find(full_message_id) != update_message_ids_.end()) {
          // apply sent channel message
          on_get_message(std::move(update_new_channel_message->message_), true, true, false, true, true,
                         "updateNewChannelMessage with an awaited message");
          promise.set_value(Unit());
          return;
        }
      }
      if (update->get_id() == updateSentMessage::ID) {
        auto update_sent_message = static_cast<updateSentMessage *>(update.get());
        if (being_sent_messages_.count(update_sent_message->random_id_) > 0) {
          // apply sent channel message
          on_send_message_success(update_sent_message->random_id_, update_sent_message->message_id_,
                                  update_sent_message->date_, update_sent_message->ttl_period_, FileId(),
                                  "process old updateSentChannelMessage");
          promise.set_value(Unit());
          return;
        }
      }

      LOG_IF(WARNING, new_pts == old_pts && pts_count == 0)
          << "Receive from " << source << " useless channel update " << oneline(to_string(update));
      LOG(INFO) << "Skip already applied channel update";
      promise.set_value(Unit());
      return;
    }

    if (running_get_channel_difference(dialog_id)) {
      LOG(INFO) << "Postpone channel update, because getChannelDifference is run";
      add_postponed_channel_update(dialog_id, std::move(update), new_pts, pts_count, std::move(promise));
      return;
    }

    if (old_pts != new_pts - pts_count) {
      LOG(INFO) << "Found a gap in the " << dialog_id << " with pts = " << old_pts << ". new_pts = " << new_pts
                << ", pts_count = " << pts_count << " in update from " << source;
      if (d->was_opened || td_->contacts_manager_->get_channel_status(channel_id).is_member() ||
          is_dialog_sponsored(d)) {
        add_postponed_channel_update(dialog_id, std::move(update), new_pts, pts_count, std::move(promise));
        get_channel_difference(dialog_id, old_pts, true, "add_pending_channel_update pts mismatch");
      } else {
        promise.set_value(Unit());
      }
      return;
    }
  }

  if (d == nullptr || pts_count > 0) {
    process_channel_update(std::move(update));
    LOG_CHECK(!running_get_channel_difference(dialog_id)) << '"' << active_get_channel_differencies_[dialog_id] << '"';
  } else {
    LOG_IF(INFO, update->get_id() != dummyUpdate::ID)
        << "Skip useless channel update from " << source << ": " << to_string(update);
  }

  if (d == nullptr) {
    d = get_dialog(dialog_id);
    if (d == nullptr) {
      LOG(INFO) << "Update didn't created " << dialog_id;
      promise.set_value(Unit());
      return;
    }
  }

  CHECK(new_pts > d->pts);
  set_channel_pts(d, new_pts, source);
  promise.set_value(Unit());
}

bool MessagesManager::is_old_channel_update(DialogId dialog_id, int32 new_pts) {
  CHECK(dialog_id.get_type() == DialogType::Channel);

  const Dialog *d = get_dialog_force(dialog_id, "is_old_channel_update");
  return new_pts <= (d == nullptr ? load_channel_pts(dialog_id) : d->pts);
}

void MessagesManager::process_pts_update(tl_object_ptr<telegram_api::Update> &&update) {
  switch (update->get_id()) {
    case dummyUpdate::ID:
      LOG(INFO) << "Process dummyUpdate";
      break;
    case telegram_api::updateNewMessage::ID: {
      auto update_new_message = move_tl_object_as<telegram_api::updateNewMessage>(update);
      LOG(INFO) << "Process updateNewMessage";
      on_get_message(std::move(update_new_message->message_), true, false, false, true, true, "updateNewMessage");
      break;
    }
    case updateSentMessage::ID: {
      auto update_sent_message = move_tl_object_as<updateSentMessage>(update);
      LOG(INFO) << "Process updateSentMessage " << update_sent_message->random_id_;
      on_send_message_success(update_sent_message->random_id_, update_sent_message->message_id_,
                              update_sent_message->date_, update_sent_message->ttl_period_, FileId(),
                              "process updateSentMessage");
      break;
    }
    case telegram_api::updateReadMessagesContents::ID: {
      auto read_contents_update = move_tl_object_as<telegram_api::updateReadMessagesContents>(update);
      LOG(INFO) << "Process updateReadMessageContents";
      for (auto &message_id : read_contents_update->messages_) {
        read_message_content_from_updates(MessageId(ServerMessageId(message_id)));
      }
      break;
    }
    case telegram_api::updateEditMessage::ID: {
      auto update_edit_message = move_tl_object_as<telegram_api::updateEditMessage>(update);
      auto full_message_id = on_get_message(std::move(update_edit_message->message_), false, false, false, false, false,
                                            "updateEditMessage");
      LOG(INFO) << "Process updateEditMessage";
      on_message_edited(full_message_id, update_edit_message->pts_);
      break;
    }
    case telegram_api::updateDeleteMessages::ID: {
      auto delete_update = move_tl_object_as<telegram_api::updateDeleteMessages>(update);
      LOG(INFO) << "Process updateDeleteMessages";
      vector<MessageId> message_ids;
      for (auto message : delete_update->messages_) {
        message_ids.push_back(MessageId(ServerMessageId(message)));
      }
      delete_messages_from_updates(message_ids);
      break;
    }
    case telegram_api::updateReadHistoryInbox::ID: {
      auto read_update = move_tl_object_as<telegram_api::updateReadHistoryInbox>(update);
      LOG(INFO) << "Process updateReadHistoryInbox";
      DialogId dialog_id(read_update->peer_);
      FolderId folder_id;
      if ((read_update->flags_ & telegram_api::updateReadHistoryInbox::FOLDER_ID_MASK) != 0) {
        folder_id = FolderId(read_update->folder_id_);
      }
      on_update_dialog_folder_id(dialog_id, folder_id);
      read_history_inbox(dialog_id, MessageId(ServerMessageId(read_update->max_id_)),
                         -1 /*read_update->still_unread_count*/, "updateReadHistoryInbox");
      break;
    }
    case telegram_api::updateReadHistoryOutbox::ID: {
      auto read_update = move_tl_object_as<telegram_api::updateReadHistoryOutbox>(update);
      LOG(INFO) << "Process updateReadHistoryOutbox";
      read_history_outbox(DialogId(read_update->peer_), MessageId(ServerMessageId(read_update->max_id_)));
      break;
    }
    case telegram_api::updatePinnedMessages::ID: {
      auto pinned_messages_update = move_tl_object_as<telegram_api::updatePinnedMessages>(update);
      LOG(INFO) << "Process updatePinnedMessages";
      vector<MessageId> message_ids;
      for (auto message : pinned_messages_update->messages_) {
        message_ids.push_back(MessageId(ServerMessageId(message)));
      }
      update_dialog_pinned_messages_from_updates(DialogId(pinned_messages_update->peer_), message_ids,
                                                 pinned_messages_update->pinned_);
      break;
    }
    default:
      UNREACHABLE();
  }
  CHECK(!td_->updates_manager_->running_get_difference());
}

void MessagesManager::process_channel_update(tl_object_ptr<telegram_api::Update> &&update) {
  switch (update->get_id()) {
    case dummyUpdate::ID:
      LOG(INFO) << "Process dummyUpdate";
      break;
    case updateSentMessage::ID: {
      auto update_sent_message = move_tl_object_as<updateSentMessage>(update);
      LOG(INFO) << "Process updateSentMessage " << update_sent_message->random_id_;
      on_send_message_success(update_sent_message->random_id_, update_sent_message->message_id_,
                              update_sent_message->date_, update_sent_message->ttl_period_, FileId(),
                              "process updateSentChannelMessage");
      break;
    }
    case telegram_api::updateNewChannelMessage::ID: {
      auto update_new_channel_message = move_tl_object_as<telegram_api::updateNewChannelMessage>(update);
      LOG(INFO) << "Process updateNewChannelMessage";
      on_get_message(std::move(update_new_channel_message->message_), true, true, false, true, true,
                     "updateNewChannelMessage");
      break;
    }
    case telegram_api::updateDeleteChannelMessages::ID: {
      auto delete_channel_messages_update = move_tl_object_as<telegram_api::updateDeleteChannelMessages>(update);
      LOG(INFO) << "Process updateDeleteChannelMessages";
      ChannelId channel_id(delete_channel_messages_update->channel_id_);
      if (!channel_id.is_valid()) {
        LOG(ERROR) << "Receive invalid " << channel_id;
        break;
      }

      vector<MessageId> message_ids;
      for (auto &message : delete_channel_messages_update->messages_) {
        message_ids.push_back(MessageId(ServerMessageId(message)));
      }

      auto dialog_id = DialogId(channel_id);
      delete_dialog_messages(dialog_id, message_ids, true, false, "updateDeleteChannelMessages");
      break;
    }
    case telegram_api::updateEditChannelMessage::ID: {
      auto update_edit_channel_message = move_tl_object_as<telegram_api::updateEditChannelMessage>(update);
      LOG(INFO) << "Process updateEditChannelMessage";
      auto full_message_id = on_get_message(std::move(update_edit_channel_message->message_), false, true, false, false,
                                            false, "updateEditChannelMessage");
      on_message_edited(full_message_id, update_edit_channel_message->pts_);
      break;
    }
    case telegram_api::updatePinnedChannelMessages::ID: {
      auto pinned_channel_messages_update = move_tl_object_as<telegram_api::updatePinnedChannelMessages>(update);
      LOG(INFO) << "Process updatePinnedChannelMessages";
      ChannelId channel_id(pinned_channel_messages_update->channel_id_);
      if (!channel_id.is_valid()) {
        LOG(ERROR) << "Receive invalid " << channel_id;
        break;
      }

      vector<MessageId> message_ids;
      for (auto &message : pinned_channel_messages_update->messages_) {
        message_ids.push_back(MessageId(ServerMessageId(message)));
      }

      update_dialog_pinned_messages_from_updates(DialogId(channel_id), message_ids,
                                                 pinned_channel_messages_update->pinned_);
      break;
    }
    default:
      UNREACHABLE();
  }
}

void MessagesManager::on_message_edited(FullMessageId full_message_id, int32 pts) {
  if (full_message_id == FullMessageId()) {
    return;
  }

  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog(dialog_id);
  Message *m = get_message(d, full_message_id.get_message_id());
  CHECK(m != nullptr);
  m->last_edit_pts = pts;
  if (td_->auth_manager_->is_bot()) {
    d->last_edited_message_id = m->message_id;
    send_update_message_edited(dialog_id, m);
  }
  update_used_hashtags(dialog_id, m);
}

string MessagesManager::get_notification_settings_scope_database_key(NotificationSettingsScope scope) {
  switch (scope) {
    case NotificationSettingsScope::Private:
      return "nsfpc";
    case NotificationSettingsScope::Group:
      return "nsfgc";
    case NotificationSettingsScope::Channel:
      return "nsfcc";
    default:
      UNREACHABLE();
      return "";
  }
}

void MessagesManager::save_scope_notification_settings(NotificationSettingsScope scope,
                                                       const ScopeNotificationSettings &new_settings) {
  string key = get_notification_settings_scope_database_key(scope);
  G()->td_db()->get_binlog_pmc()->set(key, log_event_store(new_settings).as_slice().str());
}

bool MessagesManager::update_dialog_notification_settings(DialogId dialog_id,
                                                          DialogNotificationSettings *current_settings,
                                                          const DialogNotificationSettings &new_settings) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return false;
  }

  bool need_update_server = current_settings->mute_until != new_settings.mute_until ||
                            current_settings->sound != new_settings.sound ||
                            current_settings->show_preview != new_settings.show_preview ||
                            current_settings->use_default_mute_until != new_settings.use_default_mute_until ||
                            current_settings->use_default_sound != new_settings.use_default_sound ||
                            current_settings->use_default_show_preview != new_settings.use_default_show_preview;
  bool need_update_local =
      current_settings->use_default_disable_pinned_message_notifications !=
          new_settings.use_default_disable_pinned_message_notifications ||
      current_settings->disable_pinned_message_notifications != new_settings.disable_pinned_message_notifications ||
      current_settings->use_default_disable_mention_notifications !=
          new_settings.use_default_disable_mention_notifications ||
      current_settings->disable_mention_notifications != new_settings.disable_mention_notifications;

  bool is_changed = need_update_server || need_update_local ||
                    current_settings->is_synchronized != new_settings.is_synchronized ||
                    current_settings->is_use_default_fixed != new_settings.is_use_default_fixed;

  if (is_changed) {
    Dialog *d = get_dialog(dialog_id);
    LOG_CHECK(d != nullptr) << "Wrong " << dialog_id << " in update_dialog_notification_settings";
    bool was_dialog_mentions_disabled = is_dialog_mention_notifications_disabled(d);

    VLOG(notifications) << "Update notification settings in " << dialog_id << " from " << *current_settings << " to "
                        << new_settings;

    update_dialog_unmute_timeout(d, current_settings->use_default_mute_until, current_settings->mute_until,
                                 new_settings.use_default_mute_until, new_settings.mute_until);

    *current_settings = new_settings;
    on_dialog_updated(dialog_id, "update_dialog_notification_settings");

    if (is_dialog_muted(d)) {
      // no check for was_muted to clean pending message notifications in chats with unsynchronized settings
      remove_all_dialog_notifications(d, false, "update_dialog_notification_settings 2");
    }
    if (is_dialog_pinned_message_notifications_disabled(d) && d->mention_notification_group.group_id.is_valid() &&
        d->pinned_message_notification_message_id.is_valid()) {
      remove_dialog_pinned_message_notification(d, "update_dialog_notification_settings 3");
    }
    if (was_dialog_mentions_disabled != is_dialog_mention_notifications_disabled(d)) {
      if (was_dialog_mentions_disabled) {
        update_dialog_mention_notification_count(d);
      } else {
        remove_dialog_mention_notifications(d);
      }
    }

    if (need_update_server || need_update_local) {
      send_closure(G()->td(), &Td::send_update,
                   make_tl_object<td_api::updateChatNotificationSettings>(
                       dialog_id.get(), get_chat_notification_settings_object(current_settings)));
    }
  }
  return need_update_server;
}

bool MessagesManager::update_scope_notification_settings(NotificationSettingsScope scope,
                                                         ScopeNotificationSettings *current_settings,
                                                         const ScopeNotificationSettings &new_settings) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return false;
  }

  bool need_update_server = current_settings->mute_until != new_settings.mute_until ||
                            current_settings->sound != new_settings.sound ||
                            current_settings->show_preview != new_settings.show_preview;
  bool need_update_local =
      current_settings->disable_pinned_message_notifications != new_settings.disable_pinned_message_notifications ||
      current_settings->disable_mention_notifications != new_settings.disable_mention_notifications;
  bool was_inited = current_settings->is_synchronized;
  bool is_inited = new_settings.is_synchronized;
  if (was_inited && !is_inited) {
    return false;  // just in case
  }
  bool is_changed = need_update_server || need_update_local || was_inited != is_inited;
  if (is_changed) {
    save_scope_notification_settings(scope, new_settings);

    VLOG(notifications) << "Update notification settings in " << scope << " from " << *current_settings << " to "
                        << new_settings;

    update_scope_unmute_timeout(scope, current_settings->mute_until, new_settings.mute_until);

    if (!current_settings->disable_pinned_message_notifications && new_settings.disable_pinned_message_notifications) {
      VLOG(notifications) << "Remove pinned message notifications in " << scope;
      for (auto &dialog : dialogs_) {
        Dialog *d = dialog.second.get();
        if (d->notification_settings.use_default_disable_pinned_message_notifications &&
            d->mention_notification_group.group_id.is_valid() && d->pinned_message_notification_message_id.is_valid() &&
            get_dialog_notification_setting_scope(d->dialog_id) == scope) {
          remove_dialog_pinned_message_notification(d, "update_scope_notification_settings");
        }
      }
    }
    if (current_settings->disable_mention_notifications != new_settings.disable_mention_notifications) {
      VLOG(notifications) << "Remove mention notifications in " << scope;
      for (auto &dialog : dialogs_) {
        Dialog *d = dialog.second.get();
        if (d->notification_settings.use_default_disable_mention_notifications &&
            get_dialog_notification_setting_scope(d->dialog_id) == scope) {
          if (current_settings->disable_mention_notifications) {
            update_dialog_mention_notification_count(d);
          } else {
            remove_dialog_mention_notifications(d);
          }
        }
      }
    }

    *current_settings = new_settings;

    send_closure(G()->td(), &Td::send_update, get_update_scope_notification_settings_object(scope));
  }
  return need_update_server;
}

void MessagesManager::schedule_dialog_unmute(DialogId dialog_id, bool use_default, int32 mute_until) {
  auto now = G()->unix_time_cached();
  if (!use_default && mute_until >= now && mute_until < now + 366 * 86400) {
    dialog_unmute_timeout_.set_timeout_in(dialog_id.get(), mute_until - now + 1);
  } else {
    dialog_unmute_timeout_.cancel_timeout(dialog_id.get());
  }
}

void MessagesManager::update_dialog_unmute_timeout(Dialog *d, bool &old_use_default, int32 &old_mute_until,
                                                   bool new_use_default, int32 new_mute_until) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  if (old_use_default == new_use_default && old_mute_until == new_mute_until) {
    return;
  }
  CHECK(d != nullptr);
  CHECK(old_mute_until >= 0);

  schedule_dialog_unmute(d->dialog_id, new_use_default, new_mute_until);

  bool was_muted = (old_use_default ? get_scope_mute_until(d->dialog_id) : old_mute_until) != 0;
  bool is_muted = (new_use_default ? get_scope_mute_until(d->dialog_id) : new_mute_until) != 0;
  if (was_muted != is_muted && need_unread_counter(d->order)) {
    auto unread_count = d->server_unread_count + d->local_unread_count;
    if (unread_count != 0 || d->is_marked_as_unread) {
      for (auto &list : get_dialog_lists(d)) {
        if (unread_count != 0 && list.is_message_unread_count_inited_) {
          int32 delta = was_muted ? -unread_count : unread_count;
          list.unread_message_muted_count_ += delta;
          send_update_unread_message_count(list, d->dialog_id, true, "update_dialog_unmute_timeout");
        }
        if (list.is_dialog_unread_count_inited_) {
          int32 delta = was_muted ? -1 : 1;
          list.unread_dialog_muted_count_ += delta;
          if (unread_count == 0 && d->is_marked_as_unread) {
            list.unread_dialog_muted_marked_count_ += delta;
          }
          send_update_unread_chat_count(list, d->dialog_id, true, "update_dialog_unmute_timeout");
        }
      }
    }
  }

  old_use_default = new_use_default;
  old_mute_until = new_mute_until;

  if (was_muted != is_muted && !dialog_filters_.empty()) {
    update_dialog_lists(d, get_dialog_positions(d), true, false, "update_dialog_unmute_timeout");
  }
}

void MessagesManager::schedule_scope_unmute(NotificationSettingsScope scope, int32 mute_until) {
  auto now = G()->unix_time_cached();
  if (mute_until >= now && mute_until < now + 366 * 86400) {
    dialog_unmute_timeout_.set_timeout_in(static_cast<int64>(scope) + 1, mute_until - now + 1);
  } else {
    dialog_unmute_timeout_.cancel_timeout(static_cast<int64>(scope) + 1);
  }
}

void MessagesManager::update_scope_unmute_timeout(NotificationSettingsScope scope, int32 &old_mute_until,
                                                  int32 new_mute_until) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  LOG(INFO) << "Update " << scope << " unmute timeout from " << old_mute_until << " to " << new_mute_until;
  if (old_mute_until == new_mute_until) {
    return;
  }
  CHECK(old_mute_until >= 0);

  schedule_scope_unmute(scope, new_mute_until);

  auto was_muted = old_mute_until != 0;
  auto is_muted = new_mute_until != 0;
  if (was_muted != is_muted) {
    if (G()->parameters().use_message_db) {
      std::unordered_map<DialogListId, int32, DialogListIdHash> delta;
      std::unordered_map<DialogListId, int32, DialogListIdHash> total_count;
      std::unordered_map<DialogListId, int32, DialogListIdHash> marked_count;
      std::unordered_set<DialogListId, DialogListIdHash> dialog_list_ids;
      for (auto &dialog : dialogs_) {
        Dialog *d = dialog.second.get();
        if (need_unread_counter(d->order) && d->notification_settings.use_default_mute_until &&
            get_dialog_notification_setting_scope(d->dialog_id) == scope) {
          int32 unread_count = d->server_unread_count + d->local_unread_count;
          if (unread_count != 0) {
            for (auto dialog_list_id : get_dialog_list_ids(d)) {
              delta[dialog_list_id] += unread_count;
              total_count[dialog_list_id]++;
              dialog_list_ids.insert(dialog_list_id);
            }
          } else if (d->is_marked_as_unread) {
            for (auto dialog_list_id : get_dialog_list_ids(d)) {
              total_count[dialog_list_id]++;
              marked_count[dialog_list_id]++;
              dialog_list_ids.insert(dialog_list_id);
            }
          }
        }
      }
      for (auto dialog_list_id : dialog_list_ids) {
        auto *list = get_dialog_list(dialog_list_id);
        CHECK(list != nullptr);
        if (delta[dialog_list_id] != 0 && list->is_message_unread_count_inited_) {
          if (was_muted) {
            list->unread_message_muted_count_ -= delta[dialog_list_id];
          } else {
            list->unread_message_muted_count_ += delta[dialog_list_id];
          }
          send_update_unread_message_count(*list, DialogId(), true, "update_scope_unmute_timeout");
        }
        if (total_count[dialog_list_id] != 0 && list->is_dialog_unread_count_inited_) {
          if (was_muted) {
            list->unread_dialog_muted_count_ -= total_count[dialog_list_id];
            list->unread_dialog_muted_marked_count_ -= marked_count[dialog_list_id];
          } else {
            list->unread_dialog_muted_count_ += total_count[dialog_list_id];
            list->unread_dialog_muted_marked_count_ += marked_count[dialog_list_id];
          }
          send_update_unread_chat_count(*list, DialogId(), true, "update_scope_unmute_timeout");
        }
      }
    }
  }

  old_mute_until = new_mute_until;

  if (was_muted != is_muted && !dialog_filters_.empty()) {
    for (auto &dialog : dialogs_) {
      Dialog *d = dialog.second.get();
      if (d->order != DEFAULT_ORDER && d->notification_settings.use_default_mute_until &&
          get_dialog_notification_setting_scope(d->dialog_id) == scope) {
        update_dialog_lists(d, get_dialog_positions(d), true, false, "update_scope_unmute_timeout");
      }
    }
  }
  if (!was_muted && is_muted) {
    for (auto &dialog : dialogs_) {
      Dialog *d = dialog.second.get();
      if (d->order != DEFAULT_ORDER && d->notification_settings.use_default_mute_until &&
          get_dialog_notification_setting_scope(d->dialog_id) == scope) {
        remove_all_dialog_notifications(d, false, "update_scope_unmute_timeout");
      }
    }
  }
}

void MessagesManager::on_dialog_unmute(DialogId dialog_id) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  if (d->notification_settings.use_default_mute_until) {
    return;
  }
  if (d->notification_settings.mute_until == 0) {
    return;
  }

  auto now = G()->unix_time();
  if (d->notification_settings.mute_until > now) {
    LOG(ERROR) << "Failed to unmute " << dialog_id << " in " << now << ", will be unmuted in "
               << d->notification_settings.mute_until;
    schedule_dialog_unmute(dialog_id, false, d->notification_settings.mute_until);
    return;
  }

  LOG(INFO) << "Unmute " << dialog_id;
  update_dialog_unmute_timeout(d, d->notification_settings.use_default_mute_until, d->notification_settings.mute_until,
                               false, 0);
  send_closure(G()->td(), &Td::send_update,
               make_tl_object<td_api::updateChatNotificationSettings>(
                   dialog_id.get(), get_chat_notification_settings_object(&d->notification_settings)));
  on_dialog_updated(dialog_id, "on_dialog_unmute");
}

void MessagesManager::on_scope_unmute(NotificationSettingsScope scope) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  auto notification_settings = get_scope_notification_settings(scope);
  CHECK(notification_settings != nullptr);

  if (notification_settings->mute_until == 0) {
    return;
  }

  auto now = G()->unix_time();
  if (notification_settings->mute_until > now) {
    LOG(ERROR) << "Failed to unmute " << scope << " in " << now << ", will be unmuted in "
               << notification_settings->mute_until;
    schedule_scope_unmute(scope, notification_settings->mute_until);
    return;
  }

  LOG(INFO) << "Unmute " << scope;
  update_scope_unmute_timeout(scope, notification_settings->mute_until, 0);
  send_closure(G()->td(), &Td::send_update, get_update_scope_notification_settings_object(scope));
  save_scope_notification_settings(scope, *notification_settings);
}

void MessagesManager::on_update_dialog_notify_settings(
    DialogId dialog_id, tl_object_ptr<telegram_api::peerNotifySettings> &&peer_notify_settings, const char *source) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  VLOG(notifications) << "Receive notification settings for " << dialog_id << " from " << source << ": "
                      << to_string(peer_notify_settings);

  DialogNotificationSettings *current_settings = get_dialog_notification_settings(dialog_id, true);
  if (current_settings == nullptr) {
    return;
  }

  const DialogNotificationSettings notification_settings = ::td::get_dialog_notification_settings(
      std::move(peer_notify_settings), current_settings->use_default_disable_pinned_message_notifications,
      current_settings->disable_pinned_message_notifications,
      current_settings->use_default_disable_mention_notifications, current_settings->disable_mention_notifications);
  if (!notification_settings.is_synchronized) {
    return;
  }

  update_dialog_notification_settings(dialog_id, current_settings, notification_settings);
}

void MessagesManager::on_update_scope_notify_settings(
    NotificationSettingsScope scope, tl_object_ptr<telegram_api::peerNotifySettings> &&peer_notify_settings) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  auto old_notification_settings = get_scope_notification_settings(scope);
  CHECK(old_notification_settings != nullptr);

  const ScopeNotificationSettings notification_settings = ::td::get_scope_notification_settings(
      std::move(peer_notify_settings), old_notification_settings->disable_pinned_message_notifications,
      old_notification_settings->disable_mention_notifications);
  if (!notification_settings.is_synchronized) {
    return;
  }

  update_scope_notification_settings(scope, old_notification_settings, notification_settings);
}

bool MessagesManager::update_dialog_silent_send_message(Dialog *d, bool silent_send_message) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return false;
  }

  CHECK(d != nullptr);
  LOG_IF(WARNING, !d->notification_settings.is_synchronized)
      << "Have unknown notification settings in " << d->dialog_id;
  if (d->notification_settings.silent_send_message == silent_send_message) {
    return false;
  }

  LOG(INFO) << "Update silent send message in " << d->dialog_id << " to " << silent_send_message;
  d->notification_settings.silent_send_message = silent_send_message;

  on_dialog_updated(d->dialog_id, "update_dialog_silent_send_message");

  send_closure(G()->td(), &Td::send_update,
               make_tl_object<td_api::updateChatDefaultDisableNotification>(d->dialog_id.get(), silent_send_message));
  return true;
}

void MessagesManager::reget_dialog_action_bar(DialogId dialog_id, const char *source) {
  if (G()->close_flag() || !dialog_id.is_valid() || td_->auth_manager_->is_bot()) {
    return;
  }

  LOG(INFO) << "Reget action bar in " << dialog_id << " from " << source;
  switch (dialog_id.get_type()) {
    case DialogType::User:
      td_->contacts_manager_->reload_user_full(dialog_id.get_user_id());
      return;
    case DialogType::Chat:
    case DialogType::Channel:
      if (!have_input_peer(dialog_id, AccessRights::Read)) {
        return;
      }

      return td_->create_handler<GetPeerSettingsQuery>()->send(dialog_id);
    case DialogType::SecretChat:
    case DialogType::None:
    default:
      UNREACHABLE();
  }
}

void MessagesManager::repair_dialog_action_bar(Dialog *d, const char *source) {
  CHECK(d != nullptr);
  auto dialog_id = d->dialog_id;
  d->know_action_bar = false;
  if (have_input_peer(dialog_id, AccessRights::Read)) {
    create_actor<SleepActor>(
        "RepairChatActionBarActor", 1.0,
        PromiseCreator::lambda([actor_id = actor_id(this), dialog_id, source](Result<Unit> result) {
          send_closure(actor_id, &MessagesManager::reget_dialog_action_bar, dialog_id, source);
        }))
        .release();
  }
  // there is no need to change action bar
  on_dialog_updated(dialog_id, source);
}

void MessagesManager::hide_dialog_action_bar(DialogId dialog_id) {
  Dialog *d = get_dialog_force(dialog_id, "hide_dialog_action_bar");
  if (d == nullptr) {
    return;
  }
  hide_dialog_action_bar(d);
}

void MessagesManager::hide_dialog_action_bar(Dialog *d) {
  CHECK(d->dialog_id.get_type() != DialogType::SecretChat);
  if (!d->know_can_report_spam) {
    return;
  }
  if (!d->can_report_spam && !d->can_add_contact && !d->can_block_user && !d->can_share_phone_number &&
      !d->can_report_location && !d->can_unarchive && d->distance < 0 && !d->can_invite_members) {
    return;
  }

  d->can_report_spam = false;
  d->can_add_contact = false;
  d->can_block_user = false;
  d->can_share_phone_number = false;
  d->can_report_location = false;
  d->can_unarchive = false;
  d->distance = -1;
  d->can_invite_members = false;
  send_update_chat_action_bar(d);
}

void MessagesManager::remove_dialog_action_bar(DialogId dialog_id, Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(dialog_id, "remove_dialog_action_bar");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return promise.set_error(Status::Error(400, "Can't access the chat"));
  }

  if (dialog_id.get_type() == DialogType::SecretChat) {
    dialog_id = DialogId(td_->contacts_manager_->get_secret_chat_user_id(dialog_id.get_secret_chat_id()));
    d = get_dialog_force(dialog_id, "remove_dialog_action_bar 2");
    if (d == nullptr) {
      return promise.set_error(Status::Error(400, "Chat with the user not found"));
    }
    if (!have_input_peer(dialog_id, AccessRights::Read)) {
      return promise.set_error(Status::Error(400, "Can't access the chat"));
    }
  }

  if (!d->know_can_report_spam) {
    return promise.set_error(Status::Error(400, "Can't update chat action bar"));
  }

  if (!d->can_report_spam && !d->can_add_contact && !d->can_block_user && !d->can_share_phone_number &&
      !d->can_report_location && !d->can_unarchive && d->distance < 0 && !d->can_invite_members) {
    return promise.set_value(Unit());
  }

  hide_dialog_action_bar(d);

  toggle_dialog_report_spam_state_on_server(dialog_id, false, 0, std::move(promise));
}

void MessagesManager::repair_dialog_active_group_call_id(DialogId dialog_id) {
  if (have_input_peer(dialog_id, AccessRights::Read)) {
    LOG(INFO) << "Repair active voice chat ID in " << dialog_id;
    create_actor<SleepActor>("RepairChatActiveVoiceChatId", 1.0,
                             PromiseCreator::lambda([actor_id = actor_id(this), dialog_id](Result<Unit> result) {
                               send_closure(actor_id, &MessagesManager::do_repair_dialog_active_group_call_id,
                                            dialog_id);
                             }))
        .release();
  }
}

void MessagesManager::do_repair_dialog_active_group_call_id(DialogId dialog_id) {
  if (G()->close_flag()) {
    return;
  }

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  bool need_repair_active_group_call_id = d->has_active_group_call && !d->active_group_call_id.is_valid();
  bool need_repair_expected_group_call_id =
      d->has_expected_active_group_call_id && d->active_group_call_id != d->expected_active_group_call_id;
  d->has_expected_active_group_call_id = false;
  if (!need_repair_active_group_call_id && !need_repair_expected_group_call_id) {
    return;
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return;
  }

  reload_dialog_info_full(dialog_id);
}

class MessagesManager::ToggleDialogReportSpamStateOnServerLogEvent {
 public:
  DialogId dialog_id_;
  bool is_spam_dialog_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
    td::store(is_spam_dialog_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
    td::parse(is_spam_dialog_, parser);
  }
};

uint64 MessagesManager::save_toggle_dialog_report_spam_state_on_server_log_event(DialogId dialog_id,
                                                                                 bool is_spam_dialog) {
  ToggleDialogReportSpamStateOnServerLogEvent log_event{dialog_id, is_spam_dialog};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ToggleDialogReportSpamStateOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::toggle_dialog_report_spam_state_on_server(DialogId dialog_id, bool is_spam_dialog,
                                                                uint64 log_event_id, Promise<Unit> &&promise) {
  if (log_event_id == 0 && G()->parameters().use_message_db) {
    log_event_id = save_toggle_dialog_report_spam_state_on_server_log_event(dialog_id, is_spam_dialog);
  }

  auto new_promise = get_erase_log_event_promise(log_event_id, std::move(promise));
  promise = std::move(new_promise);  // to prevent self-move

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      return td_->create_handler<UpdatePeerSettingsQuery>(std::move(promise))->send(dialog_id, is_spam_dialog);
    case DialogType::SecretChat:
      if (is_spam_dialog) {
        return td_->create_handler<ReportEncryptedSpamQuery>(std::move(promise))->send(dialog_id);
      } else {
        auto user_id = td_->contacts_manager_->get_secret_chat_user_id(dialog_id.get_secret_chat_id());
        if (!user_id.is_valid()) {
          return promise.set_error(Status::Error(400, "Peer user not found"));
        }
        return td_->create_handler<UpdatePeerSettingsQuery>(std::move(promise))->send(DialogId(user_id), false);
      }
    case DialogType::None:
    default:
      UNREACHABLE();
      return;
  }
}

bool MessagesManager::can_report_dialog(DialogId dialog_id) const {
  // doesn't include possibility of report from action bar
  switch (dialog_id.get_type()) {
    case DialogType::User:
      return td_->contacts_manager_->can_report_user(dialog_id.get_user_id());
    case DialogType::Chat:
      return false;
    case DialogType::Channel:
      return !td_->contacts_manager_->get_channel_status(dialog_id.get_channel_id()).is_creator();
    case DialogType::SecretChat:
      return false;
    case DialogType::None:
    default:
      UNREACHABLE();
      return false;
  }
}

void MessagesManager::report_dialog(DialogId dialog_id, const vector<MessageId> &message_ids, ReportReason &&reason,
                                    Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(dialog_id, "report_dialog");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return promise.set_error(Status::Error(400, "Can't access the chat"));
  }

  Dialog *user_d = d;
  bool is_dialog_spam_report = false;
  bool can_report_spam = d->can_report_spam;
  if (reason.is_spam() && message_ids.empty()) {
    // report from action bar
    if (dialog_id.get_type() == DialogType::SecretChat) {
      auto user_dialog_id = DialogId(td_->contacts_manager_->get_secret_chat_user_id(dialog_id.get_secret_chat_id()));
      user_d = get_dialog_force(user_dialog_id, "report_dialog 2");
      if (user_d == nullptr) {
        return promise.set_error(Status::Error(400, "Chat with the user not found"));
      }

      is_dialog_spam_report = user_d->know_can_report_spam;
      can_report_spam = user_d->can_report_spam;
    } else {
      is_dialog_spam_report = d->know_can_report_spam;
    }
  }

  if (is_dialog_spam_report && can_report_spam) {
    hide_dialog_action_bar(user_d);
    return toggle_dialog_report_spam_state_on_server(dialog_id, true, 0, std::move(promise));
  }

  if (!can_report_dialog(dialog_id)) {
    if (is_dialog_spam_report) {
      return promise.set_value(Unit());
    }

    return promise.set_error(Status::Error(400, "Chat can't be reported"));
  }

  vector<MessageId> server_message_ids;
  for (auto message_id : message_ids) {
    if (message_id.is_scheduled()) {
      return promise.set_error(Status::Error(400, "Can't report scheduled messages"));
    }
    if (message_id.is_valid() && message_id.is_server()) {
      server_message_ids.push_back(message_id);
    }
  }

  if (dialog_id.get_type() == DialogType::Channel && reason.is_unrelated_location()) {
    hide_dialog_action_bar(d);
  }

  td_->create_handler<ReportPeerQuery>(std::move(promise))->send(dialog_id, server_message_ids, std::move(reason));
}

void MessagesManager::report_dialog_photo(DialogId dialog_id, FileId file_id, ReportReason &&reason,
                                          Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(dialog_id, "report_dialog_photo");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return promise.set_error(Status::Error(400, "Can't access the chat"));
  }

  if (!can_report_dialog(dialog_id)) {
    return promise.set_error(Status::Error(400, "Chat photo can't be reported"));
  }

  auto file_view = td_->file_manager_->get_file_view(file_id);
  if (file_view.empty()) {
    return promise.set_error(Status::Error(400, "Unknown file ID"));
  }
  if (file_view.get_type() != FileType::Photo || !file_view.has_remote_location() ||
      !file_view.remote_location().is_photo()) {
    return promise.set_error(Status::Error(400, "Only full chat photos can be reported"));
  }

  td_->create_handler<ReportProfilePhotoQuery>(std::move(promise))
      ->send(dialog_id, file_id, file_view.remote_location().as_input_photo(), std::move(reason));
}

void MessagesManager::on_get_peer_settings(DialogId dialog_id,
                                           tl_object_ptr<telegram_api::peerSettings> &&peer_settings,
                                           bool ignore_privacy_exception) {
  CHECK(peer_settings != nullptr);
  if (dialog_id.get_type() == DialogType::User && !ignore_privacy_exception) {
    td_->contacts_manager_->on_update_user_need_phone_number_privacy_exception(dialog_id.get_user_id(),
                                                                               peer_settings->need_contacts_exception_);
  }

  Dialog *d = get_dialog_force(dialog_id, "on_get_peer_settings");
  if (d == nullptr) {
    return;
  }

  auto can_report_spam = peer_settings->report_spam_;
  auto can_add_contact = peer_settings->add_contact_;
  auto can_block_user = peer_settings->block_contact_;
  auto can_share_phone_number = peer_settings->share_contact_;
  auto can_report_location = peer_settings->report_geo_;
  auto can_unarchive = peer_settings->autoarchived_;
  auto distance =
      (peer_settings->flags_ & telegram_api::peerSettings::GEO_DISTANCE_MASK) != 0 ? peer_settings->geo_distance_ : -1;
  auto can_invite_members = peer_settings->invite_members_;
  if (d->can_report_spam == can_report_spam && d->can_add_contact == can_add_contact &&
      d->can_block_user == can_block_user && d->can_share_phone_number == can_share_phone_number &&
      d->can_report_location == can_report_location && d->can_unarchive == can_unarchive && d->distance == distance &&
      d->can_invite_members == can_invite_members) {
    if (!d->know_action_bar || !d->know_can_report_spam) {
      d->know_can_report_spam = true;
      d->know_action_bar = true;
      on_dialog_updated(d->dialog_id, "on_get_peer_settings");
    }
    return;
  }

  d->know_can_report_spam = true;
  d->know_action_bar = true;
  d->can_report_spam = can_report_spam;
  d->can_add_contact = can_add_contact;
  d->can_block_user = can_block_user;
  d->can_share_phone_number = can_share_phone_number;
  d->can_report_location = can_report_location;
  d->can_unarchive = can_unarchive;
  d->distance = distance < 0 ? -1 : distance;
  d->can_invite_members = can_invite_members;

  fix_dialog_action_bar(d);

  send_update_chat_action_bar(d);
}

void MessagesManager::fix_dialog_action_bar(Dialog *d) {
  CHECK(d != nullptr);
  if (!d->know_action_bar) {
    return;
  }

  auto dialog_type = d->dialog_id.get_type();
  if (d->distance >= 0 && dialog_type != DialogType::User) {
    LOG(ERROR) << "Receive distance " << d->distance << " to " << d->dialog_id;
    d->distance = -1;
  }

  if (d->can_report_location) {
    if (dialog_type != DialogType::Channel) {
      LOG(ERROR) << "Receive can_report_location in " << d->dialog_id;
      d->can_report_location = false;
    } else if (d->can_report_spam || d->can_add_contact || d->can_block_user || d->can_share_phone_number ||
               d->can_unarchive || d->can_invite_members) {
      LOG(ERROR) << "Receive action bar " << d->can_report_spam << "/" << d->can_add_contact << "/" << d->can_block_user
                 << "/" << d->can_share_phone_number << "/" << d->can_report_location << "/" << d->can_unarchive << "/"
                 << d->can_invite_members;
      d->can_report_spam = false;
      d->can_add_contact = false;
      d->can_block_user = false;
      d->can_share_phone_number = false;
      d->can_unarchive = false;
      d->can_invite_members = false;
      CHECK(d->distance == -1);
    }
  }
  if (d->can_invite_members) {
    if (dialog_type != DialogType::Chat && (dialog_type != DialogType::Channel || is_broadcast_channel(d->dialog_id))) {
      LOG(ERROR) << "Receive can_invite_members in " << d->dialog_id;
      d->can_invite_members = false;
    } else if (d->can_report_spam || d->can_add_contact || d->can_block_user || d->can_share_phone_number ||
               d->can_unarchive) {
      LOG(ERROR) << "Receive action bar " << d->can_report_spam << "/" << d->can_add_contact << "/" << d->can_block_user
                 << "/" << d->can_share_phone_number << "/" << d->can_unarchive << "/" << d->can_invite_members;
      d->can_report_spam = false;
      d->can_add_contact = false;
      d->can_block_user = false;
      d->can_share_phone_number = false;
      d->can_unarchive = false;
      CHECK(d->distance == -1);
    }
  }
  if (dialog_type == DialogType::User) {
    auto user_id = d->dialog_id.get_user_id();
    bool is_me = user_id == td_->contacts_manager_->get_my_id();
    bool is_blocked = d->is_blocked;
    bool is_deleted = td_->contacts_manager_->is_user_deleted(user_id);
    bool is_contact = td_->contacts_manager_->is_user_contact(user_id);
    if (is_me || is_blocked) {
      d->can_report_spam = false;
      d->can_unarchive = false;
    }
    if (is_me || is_blocked || is_deleted) {
      d->can_share_phone_number = false;
    }
    if (is_me || is_blocked || is_deleted || is_contact) {
      d->can_block_user = false;
      d->can_add_contact = false;
    }
  }
  if (d->folder_id != FolderId::archive()) {
    d->can_unarchive = false;
  }
  if (d->can_share_phone_number) {
    CHECK(!d->can_report_location);
    CHECK(!d->can_invite_members);
    if (dialog_type != DialogType::User) {
      LOG(ERROR) << "Receive can_share_phone_number in " << d->dialog_id;
      d->can_share_phone_number = false;
    } else if (d->can_report_spam || d->can_add_contact || d->can_block_user || d->can_unarchive || d->distance >= 0) {
      LOG(ERROR) << "Receive action bar " << d->can_report_spam << "/" << d->can_add_contact << "/" << d->can_block_user
                 << "/" << d->can_share_phone_number << "/" << d->can_unarchive << "/" << d->distance;
      d->can_report_spam = false;
      d->can_add_contact = false;
      d->can_block_user = false;
      d->can_unarchive = false;
    }
  }
  if (d->can_block_user) {
    CHECK(!d->can_report_location);
    CHECK(!d->can_invite_members);
    CHECK(!d->can_share_phone_number);
    if (dialog_type != DialogType::User) {
      LOG(ERROR) << "Receive can_block_user in " << d->dialog_id;
      d->can_block_user = false;
    } else if (!d->can_report_spam || !d->can_add_contact) {
      LOG(ERROR) << "Receive action bar " << d->can_report_spam << "/" << d->can_add_contact << "/"
                 << d->can_block_user;
      d->can_report_spam = true;
      d->can_add_contact = true;
    }
  }
  if (d->can_add_contact) {
    CHECK(!d->can_report_location);
    CHECK(!d->can_invite_members);
    CHECK(!d->can_share_phone_number);
    if (dialog_type != DialogType::User) {
      LOG(ERROR) << "Receive can_add_contact in " << d->dialog_id;
      d->can_add_contact = false;
    } else if (d->can_report_spam != d->can_block_user) {
      LOG(ERROR) << "Receive action bar " << d->can_report_spam << "/" << d->can_add_contact << "/"
                 << d->can_block_user;
      d->can_report_spam = false;
      d->can_block_user = false;
      d->can_unarchive = false;
    }
  }
  if (!d->can_block_user) {
    d->distance = -1;
  }
  if (!d->can_report_spam) {
    d->can_unarchive = false;
  }
}

Result<string> MessagesManager::get_login_button_url(FullMessageId full_message_id, int64 button_id) {
  Dialog *d = get_dialog_force(full_message_id.get_dialog_id(), "get_login_button_url");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }
  if (!have_input_peer(d->dialog_id, AccessRights::Read)) {
    return Status::Error(400, "Can't access the chat");
  }

  auto m = get_message_force(d, full_message_id.get_message_id(), "get_login_button_url");
  if (m == nullptr) {
    return Status::Error(400, "Message not found");
  }
  if (m->reply_markup == nullptr || m->reply_markup->type != ReplyMarkup::Type::InlineKeyboard) {
    return Status::Error(400, "Message has no inline keyboard");
  }
  if (m->message_id.is_scheduled()) {
    return Status::Error(400, "Can't use login buttons from scheduled messages");
  }
  if (!m->message_id.is_server()) {
    // it shouldn't have UrlAuth buttons anyway
    return Status::Error(400, "Message is not server");
  }
  if (d->dialog_id.get_type() == DialogType::SecretChat) {
    // secret chat messages can't have reply markup, so this shouldn't happen now
    return Status::Error(400, "Message is in a secret chat");
  }
  if (button_id < std::numeric_limits<int32>::min() || button_id > std::numeric_limits<int32>::max()) {
    return Status::Error(400, "Invalid button identifier specified");
  }

  for (auto &row : m->reply_markup->inline_keyboard) {
    for (auto &button : row) {
      if (button.type == InlineKeyboardButton::Type::UrlAuth && button.id == button_id) {
        return button.data;
      }
    }
  }

  return Status::Error(400, "Button not found");
}

void MessagesManager::load_secret_thumbnail(FileId thumbnail_file_id) {
  class Callback final : public FileManager::DownloadCallback {
   public:
    explicit Callback(Promise<> download_promise) : download_promise_(std::move(download_promise)) {
    }

    void on_download_ok(FileId file_id) final {
      download_promise_.set_value(Unit());
    }
    void on_download_error(FileId file_id, Status error) final {
      download_promise_.set_error(std::move(error));
    }

   private:
    Promise<> download_promise_;
  };

  auto thumbnail_promise = PromiseCreator::lambda([actor_id = actor_id(this),
                                                   thumbnail_file_id](Result<BufferSlice> r_thumbnail) {
    BufferSlice thumbnail_slice;
    if (r_thumbnail.is_ok()) {
      thumbnail_slice = r_thumbnail.move_as_ok();
    }
    send_closure(actor_id, &MessagesManager::on_load_secret_thumbnail, thumbnail_file_id, std::move(thumbnail_slice));
  });

  auto download_promise = PromiseCreator::lambda(
      [thumbnail_file_id, thumbnail_promise = std::move(thumbnail_promise)](Result<Unit> r_download) mutable {
        if (r_download.is_error()) {
          thumbnail_promise.set_error(r_download.move_as_error());
          return;
        }
        send_closure(G()->file_manager(), &FileManager::get_content, thumbnail_file_id, std::move(thumbnail_promise));
      });

  send_closure(G()->file_manager(), &FileManager::download, thumbnail_file_id,
               std::make_shared<Callback>(std::move(download_promise)), 1, -1, -1);
}

void MessagesManager::on_upload_media(FileId file_id, tl_object_ptr<telegram_api::InputFile> input_file,
                                      tl_object_ptr<telegram_api::InputEncryptedFile> input_encrypted_file) {
  LOG(INFO) << "File " << file_id << " has been uploaded";

  auto it = being_uploaded_files_.find(file_id);
  if (it == being_uploaded_files_.end()) {
    // callback may be called just before the file upload was canceled
    return;
  }

  auto full_message_id = it->second.first;
  auto thumbnail_file_id = it->second.second;

  being_uploaded_files_.erase(it);

  Message *m = get_message(full_message_id);
  if (m == nullptr) {
    // message has already been deleted by the user or sent to inaccessible channel, do not need to send or edit it
    // file upload should be already canceled in cancel_send_message_query, it shouldn't happen
    LOG(ERROR) << "Message with a media has already been deleted";
    return;
  }

  bool is_edit = m->message_id.is_any_server();
  auto dialog_id = full_message_id.get_dialog_id();
  auto can_send_status = can_send_message(dialog_id);
  if (!is_edit && can_send_status.is_error()) {
    // user has left the chat during upload of the file or lost their privileges
    LOG(INFO) << "Can't send a message to " << dialog_id << ": " << can_send_status.error();

    fail_send_message(full_message_id, can_send_status.move_as_error());
    return;
  }

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      if (input_file && thumbnail_file_id.is_valid()) {
        // TODO: download thumbnail if needed (like in secret chats)
        LOG(INFO) << "Ask to upload thumbnail " << thumbnail_file_id;
        CHECK(being_uploaded_thumbnails_.find(thumbnail_file_id) == being_uploaded_thumbnails_.end());
        being_uploaded_thumbnails_[thumbnail_file_id] = {full_message_id, file_id, std::move(input_file)};
        td_->file_manager_->upload(thumbnail_file_id, upload_thumbnail_callback_, 32, m->message_id.get());
      } else {
        do_send_media(dialog_id, m, file_id, thumbnail_file_id, std::move(input_file), nullptr);
      }
      break;
    case DialogType::SecretChat:
      if (thumbnail_file_id.is_valid()) {
        LOG(INFO) << "Ask to load thumbnail " << thumbnail_file_id;
        CHECK(being_loaded_secret_thumbnails_.find(thumbnail_file_id) == being_loaded_secret_thumbnails_.end());
        being_loaded_secret_thumbnails_[thumbnail_file_id] = {full_message_id, file_id,
                                                              std::move(input_encrypted_file)};

        load_secret_thumbnail(thumbnail_file_id);
      } else {
        do_send_secret_media(dialog_id, m, file_id, thumbnail_file_id, std::move(input_encrypted_file), BufferSlice());
      }
      break;
    case DialogType::None:
    default:
      UNREACHABLE();
      break;
  }
}

void MessagesManager::do_send_media(DialogId dialog_id, Message *m, FileId file_id, FileId thumbnail_file_id,
                                    tl_object_ptr<telegram_api::InputFile> input_file,
                                    tl_object_ptr<telegram_api::InputFile> input_thumbnail) {
  CHECK(m != nullptr);

  bool have_input_file = input_file != nullptr;
  bool have_input_thumbnail = input_thumbnail != nullptr;
  LOG(INFO) << "Do send media file " << file_id << " with thumbnail " << thumbnail_file_id
            << ", have_input_file = " << have_input_file << ", have_input_thumbnail = " << have_input_thumbnail
            << ", TTL = " << m->ttl;

  MessageContent *content = nullptr;
  if (m->message_id.is_any_server()) {
    content = m->edited_content.get();
    if (content == nullptr) {
      LOG(ERROR) << "Message has no edited content";
      return;
    }
  } else {
    content = m->content.get();
  }

  auto input_media = get_input_media(content, td_, std::move(input_file), std::move(input_thumbnail), file_id,
                                     thumbnail_file_id, m->ttl, true);
  LOG_CHECK(input_media != nullptr) << to_string(get_message_object(dialog_id, m, "do_send_media")) << ' '
                                    << have_input_file << ' ' << have_input_thumbnail << ' ' << file_id << ' '
                                    << thumbnail_file_id << ' ' << m->ttl;

  on_message_media_uploaded(dialog_id, m, std::move(input_media), file_id, thumbnail_file_id);
}

void MessagesManager::do_send_secret_media(DialogId dialog_id, Message *m, FileId file_id, FileId thumbnail_file_id,
                                           tl_object_ptr<telegram_api::InputEncryptedFile> input_encrypted_file,
                                           BufferSlice thumbnail) {
  CHECK(dialog_id.get_type() == DialogType::SecretChat);
  CHECK(m != nullptr);
  CHECK(m->message_id.is_valid());
  CHECK(m->message_id.is_yet_unsent());

  bool have_input_file = input_encrypted_file != nullptr;
  LOG(INFO) << "Do send secret media file " << file_id << " with thumbnail " << thumbnail_file_id
            << ", have_input_file = " << have_input_file;

  on_secret_message_media_uploaded(
      dialog_id, m,
      get_secret_input_media(m->content.get(), td_, std::move(input_encrypted_file), std::move(thumbnail)), file_id,
      thumbnail_file_id);
}

void MessagesManager::on_upload_media_error(FileId file_id, Status status) {
  if (G()->close_flag()) {
    // do not fail upload if closing
    return;
  }

  LOG(WARNING) << "File " << file_id << " has upload error " << status;
  CHECK(status.is_error());

  auto it = being_uploaded_files_.find(file_id);
  if (it == being_uploaded_files_.end()) {
    // callback may be called just before the file upload was canceled
    return;
  }

  auto full_message_id = it->second.first;

  being_uploaded_files_.erase(it);

  bool is_edit = full_message_id.get_message_id().is_any_server();
  if (is_edit) {
    fail_edit_message_media(full_message_id, Status::Error(status.code() > 0 ? status.code() : 500, status.message()));
  } else {
    fail_send_message(full_message_id, std::move(status));
  }
}

void MessagesManager::on_load_secret_thumbnail(FileId thumbnail_file_id, BufferSlice thumbnail) {
  if (G()->close_flag()) {
    // do not send secret media if closing, thumbnail may be wrong
    return;
  }

  LOG(INFO) << "SecretThumbnail " << thumbnail_file_id << " has been loaded with size " << thumbnail.size();

  auto it = being_loaded_secret_thumbnails_.find(thumbnail_file_id);
  if (it == being_loaded_secret_thumbnails_.end()) {
    // just in case, as in on_upload_thumbnail
    return;
  }

  auto full_message_id = it->second.full_message_id;
  auto file_id = it->second.file_id;
  auto input_file = std::move(it->second.input_file);

  being_loaded_secret_thumbnails_.erase(it);

  Message *m = get_message(full_message_id);
  if (m == nullptr) {
    // message has already been deleted by the user, do not need to send it
    // cancel file upload of the main file to allow next upload with the same file to succeed
    LOG(INFO) << "Message with a media has already been deleted";
    cancel_upload_file(file_id);
    return;
  }
  CHECK(m->message_id.is_yet_unsent());

  if (thumbnail.empty()) {
    delete_message_content_thumbnail(m->content.get(), td_);
  }

  auto dialog_id = full_message_id.get_dialog_id();
  auto can_send_status = can_send_message(dialog_id);
  if (can_send_status.is_error()) {
    // secret chat was closed during load of the file
    LOG(INFO) << "Can't send a message to " << dialog_id << ": " << can_send_status.error();

    fail_send_message(full_message_id, can_send_status.move_as_error());
    return;
  }

  do_send_secret_media(dialog_id, m, file_id, thumbnail_file_id, std::move(input_file), std::move(thumbnail));
}

void MessagesManager::on_upload_thumbnail(FileId thumbnail_file_id,
                                          tl_object_ptr<telegram_api::InputFile> thumbnail_input_file) {
  if (G()->close_flag()) {
    // do not fail upload if closing
    return;
  }

  LOG(INFO) << "Thumbnail " << thumbnail_file_id << " has been uploaded as " << to_string(thumbnail_input_file);

  auto it = being_uploaded_thumbnails_.find(thumbnail_file_id);
  if (it == being_uploaded_thumbnails_.end()) {
    // callback may be called just before the thumbnail upload was canceled
    return;
  }

  auto full_message_id = it->second.full_message_id;
  auto file_id = it->second.file_id;
  auto input_file = std::move(it->second.input_file);

  being_uploaded_thumbnails_.erase(it);

  Message *m = get_message(full_message_id);
  if (m == nullptr) {
    // message has already been deleted by the user or sent to inaccessible channel, do not need to send or edit it
    // thumbnail file upload should be already canceled in cancel_send_message_query
    LOG(ERROR) << "Message with a media has already been deleted";
    return;
  }

  bool is_edit = m->message_id.is_any_server();

  if (thumbnail_input_file == nullptr) {
    delete_message_content_thumbnail(is_edit ? m->edited_content.get() : m->content.get(), td_);
  }

  auto dialog_id = full_message_id.get_dialog_id();
  auto can_send_status = can_send_message(dialog_id);
  if (!is_edit && can_send_status.is_error()) {
    // user has left the chat during upload of the thumbnail or lost their privileges
    LOG(INFO) << "Can't send a message to " << dialog_id << ": " << can_send_status.error();

    fail_send_message(full_message_id, can_send_status.move_as_error());
    return;
  }

  do_send_media(dialog_id, m, file_id, thumbnail_file_id, std::move(input_file), std::move(thumbnail_input_file));
}

void MessagesManager::on_upload_dialog_photo(FileId file_id, tl_object_ptr<telegram_api::InputFile> input_file) {
  LOG(INFO) << "File " << file_id << " has been uploaded";

  auto it = being_uploaded_dialog_photos_.find(file_id);
  if (it == being_uploaded_dialog_photos_.end()) {
    // just in case, as in on_upload_media
    return;
  }

  DialogId dialog_id = it->second.dialog_id;
  double main_frame_timestamp = it->second.main_frame_timestamp;
  bool is_animation = it->second.is_animation;
  bool is_reupload = it->second.is_reupload;
  Promise<Unit> promise = std::move(it->second.promise);

  being_uploaded_dialog_photos_.erase(it);

  FileView file_view = td_->file_manager_->get_file_view(file_id);
  CHECK(!file_view.is_encrypted());
  if (input_file == nullptr && file_view.has_remote_location()) {
    if (file_view.main_remote_location().is_web()) {
      return promise.set_error(Status::Error(400, "Can't use web photo as profile photo"));
    }
    if (is_reupload) {
      return promise.set_error(Status::Error(400, "Failed to reupload the file"));
    }

    if (is_animation) {
      CHECK(file_view.get_type() == FileType::Animation);
      // delete file reference and forcely reupload the file
      auto file_reference = FileManager::extract_file_reference(file_view.main_remote_location().as_input_document());
      td_->file_manager_->delete_file_reference(file_id, file_reference);
      upload_dialog_photo(dialog_id, file_id, is_animation, main_frame_timestamp, true, std::move(promise), {-1});
    } else {
      CHECK(file_view.get_type() == FileType::Photo);
      auto input_photo = file_view.main_remote_location().as_input_photo();
      auto input_chat_photo = make_tl_object<telegram_api::inputChatPhoto>(std::move(input_photo));
      send_edit_dialog_photo_query(dialog_id, file_id, std::move(input_chat_photo), std::move(promise));
    }
    return;
  }
  CHECK(input_file != nullptr);

  int32 flags = 0;
  tl_object_ptr<telegram_api::InputFile> photo_input_file;
  tl_object_ptr<telegram_api::InputFile> video_input_file;
  if (is_animation) {
    flags |= telegram_api::inputChatUploadedPhoto::VIDEO_MASK;
    video_input_file = std::move(input_file);

    if (main_frame_timestamp != 0.0) {
      flags |= telegram_api::inputChatUploadedPhoto::VIDEO_START_TS_MASK;
    }
  } else {
    flags |= telegram_api::inputChatUploadedPhoto::FILE_MASK;
    photo_input_file = std::move(input_file);
  }

  auto input_chat_photo = make_tl_object<telegram_api::inputChatUploadedPhoto>(
      flags, std::move(photo_input_file), std::move(video_input_file), main_frame_timestamp);
  send_edit_dialog_photo_query(dialog_id, file_id, std::move(input_chat_photo), std::move(promise));
}

void MessagesManager::on_upload_dialog_photo_error(FileId file_id, Status status) {
  if (G()->close_flag()) {
    // do not fail upload if closing
    return;
  }

  LOG(INFO) << "File " << file_id << " has upload error " << status;
  CHECK(status.is_error());

  auto it = being_uploaded_dialog_photos_.find(file_id);
  if (it == being_uploaded_dialog_photos_.end()) {
    // just in case, as in on_upload_media_error
    return;
  }

  Promise<Unit> promise = std::move(it->second.promise);

  being_uploaded_dialog_photos_.erase(it);

  promise.set_error(std::move(status));
}

void MessagesManager::on_upload_imported_messages(FileId file_id, tl_object_ptr<telegram_api::InputFile> input_file) {
  LOG(INFO) << "File " << file_id << " has been uploaded";

  auto it = being_uploaded_imported_messages_.find(file_id);
  if (it == being_uploaded_imported_messages_.end()) {
    // just in case, as in on_upload_media
    return;
  }

  CHECK(it->second != nullptr);
  DialogId dialog_id = it->second->dialog_id;
  vector<FileId> attached_file_ids = std::move(it->second->attached_file_ids);
  bool is_reupload = it->second->is_reupload;
  Promise<Unit> promise = std::move(it->second->promise);

  being_uploaded_imported_messages_.erase(it);

  TRY_STATUS_PROMISE(promise, can_send_message(dialog_id));

  FileView file_view = td_->file_manager_->get_file_view(file_id);
  CHECK(!file_view.is_encrypted());
  if (input_file == nullptr && file_view.has_remote_location()) {
    if (file_view.main_remote_location().is_web()) {
      return promise.set_error(Status::Error(400, "Can't use web file"));
    }
    if (is_reupload) {
      return promise.set_error(Status::Error(400, "Failed to reupload the file"));
    }

    CHECK(file_view.get_type() == FileType::Document);
    // delete file reference and forcely reupload the file
    auto file_reference = FileManager::extract_file_reference(file_view.main_remote_location().as_input_document());
    td_->file_manager_->delete_file_reference(file_id, file_reference);
    upload_imported_messages(dialog_id, file_id, std::move(attached_file_ids), true, std::move(promise), {-1});
    return;
  }
  CHECK(input_file != nullptr);

  td_->create_handler<InitHistoryImportQuery>(std::move(promise))
      ->send(dialog_id, file_id, std::move(input_file), std::move(attached_file_ids));
}

void MessagesManager::on_upload_imported_messages_error(FileId file_id, Status status) {
  if (G()->close_flag()) {
    // do not fail upload if closing
    return;
  }

  LOG(INFO) << "File " << file_id << " has upload error " << status;
  CHECK(status.is_error());

  auto it = being_uploaded_imported_messages_.find(file_id);
  if (it == being_uploaded_imported_messages_.end()) {
    // just in case, as in on_upload_media_error
    return;
  }

  Promise<Unit> promise = std::move(it->second->promise);

  being_uploaded_imported_messages_.erase(it);

  promise.set_error(std::move(status));
}

void MessagesManager::on_upload_imported_message_attachment(FileId file_id,
                                                            tl_object_ptr<telegram_api::InputFile> input_file) {
  LOG(INFO) << "File " << file_id << " has been uploaded";

  auto it = being_uploaded_imported_message_attachments_.find(file_id);
  if (it == being_uploaded_imported_message_attachments_.end()) {
    // just in case, as in on_upload_media
    return;
  }

  CHECK(it->second != nullptr);
  DialogId dialog_id = it->second->dialog_id;
  int64 import_id = it->second->import_id;
  bool is_reupload = it->second->is_reupload;
  Promise<Unit> promise = std::move(it->second->promise);

  being_uploaded_imported_message_attachments_.erase(it);

  FileView file_view = td_->file_manager_->get_file_view(file_id);
  CHECK(!file_view.is_encrypted());
  if (input_file == nullptr && file_view.has_remote_location()) {
    if (file_view.main_remote_location().is_web()) {
      return promise.set_error(Status::Error(400, "Can't use web file"));
    }
    if (is_reupload) {
      return promise.set_error(Status::Error(400, "Failed to reupload the file"));
    }

    // delete file reference and forcely reupload the file
    auto file_reference =
        file_view.get_type() == FileType::Photo
            ? FileManager::extract_file_reference(file_view.main_remote_location().as_input_photo())
            : FileManager::extract_file_reference(file_view.main_remote_location().as_input_document());
    td_->file_manager_->delete_file_reference(file_id, file_reference);
    upload_imported_message_attachment(dialog_id, import_id, file_id, true, std::move(promise), {-1});
    return;
  }
  CHECK(input_file != nullptr);

  auto suggested_path = file_view.suggested_path();
  const PathView path_view(suggested_path);
  td_->create_handler<UploadImportedMediaQuery>(std::move(promise))
      ->send(dialog_id, import_id, path_view.file_name().str(), file_id,
             get_fake_input_media(td_, std::move(input_file), file_id));
}

void MessagesManager::on_upload_imported_message_attachment_error(FileId file_id, Status status) {
  if (G()->close_flag()) {
    // do not fail upload if closing
    return;
  }

  LOG(INFO) << "File " << file_id << " has upload error " << status;
  CHECK(status.is_error());

  auto it = being_uploaded_imported_message_attachments_.find(file_id);
  if (it == being_uploaded_imported_message_attachments_.end()) {
    // just in case, as in on_upload_media_error
    return;
  }

  Promise<Unit> promise = std::move(it->second->promise);

  being_uploaded_imported_message_attachments_.erase(it);

  promise.set_error(std::move(status));
}

void MessagesManager::before_get_difference() {
  running_get_difference_ = true;

  // scheduled messages are not returned in getDifference, so we must always reget them after it
  scheduled_messages_sync_generation_++;
}

void MessagesManager::after_get_difference() {
  CHECK(!td_->updates_manager_->running_get_difference());

  running_get_difference_ = false;

  if (!pending_on_get_dialogs_.empty()) {
    LOG(INFO) << "Apply postponed results of getDialogs";
    for (auto &res : pending_on_get_dialogs_) {
      on_get_dialogs(res.folder_id, std::move(res.dialogs), res.total_count, std::move(res.messages),
                     std::move(res.promise));
    }
    pending_on_get_dialogs_.clear();
  }

  if (!postponed_chat_read_inbox_updates_.empty()) {
    LOG(INFO) << "Send postponed chat read inbox updates";
    auto dialog_ids = std::move(postponed_chat_read_inbox_updates_);
    for (auto dialog_id : dialog_ids) {
      send_update_chat_read_inbox(get_dialog(dialog_id), false, "after_get_difference");
    }
  }
  while (!postponed_unread_message_count_updates_.empty()) {
    auto *list = get_dialog_list(*postponed_unread_message_count_updates_.begin());
    CHECK(list != nullptr);
    send_update_unread_message_count(*list, DialogId(), true, "after_get_difference");
  }
  while (!postponed_unread_chat_count_updates_.empty()) {
    auto *list = get_dialog_list(*postponed_unread_chat_count_updates_.begin());
    CHECK(list != nullptr);
    send_update_unread_chat_count(*list, DialogId(), true, "after_get_difference");
  }

  vector<FullMessageId> update_message_ids_to_delete;
  for (auto &it : update_message_ids_) {
    // there can be unhandled updateMessageId updates after getDifference even for ordinary chats,
    // because despite updates coming during getDifference have already been applied,
    // some of them could be postponed because of pts gap
    auto full_message_id = it.first;
    auto dialog_id = full_message_id.get_dialog_id();
    auto message_id = full_message_id.get_message_id();
    CHECK(message_id.is_valid());
    CHECK(message_id.is_server());
    switch (dialog_id.get_type()) {
      case DialogType::Channel:
        // get channel difference may prevent updates from being applied
        if (running_get_channel_difference(dialog_id)) {
          break;
        }
      // fallthrough
      case DialogType::User:
      case DialogType::Chat: {
        if (!have_message_force({dialog_id, it.second}, "after get difference")) {
          // The sent message has already been deleted by the user or sent to inaccessible channel.
          // The sent message may never be received, but we will need updateMessageId in case the message is received
          // to delete it from the server and not add to the chat.
          // But if the chat is inaccessible or the message is in an inaccessible chat part, then we will not be able to
          // add the message or delete it from the server. In this case we forget updateMessageId for such messages in
          // order to not check them over and over.
          const Dialog *d = get_dialog(dialog_id);
          if (!have_input_peer(dialog_id, AccessRights::Read) ||
              (d != nullptr &&
               message_id <= td::max(d->last_clear_history_message_id, d->max_unavailable_message_id))) {
            update_message_ids_to_delete.push_back(it.first);
          }
          break;
        }

        const Dialog *d = get_dialog(dialog_id);
        CHECK(d != nullptr);
        if (dialog_id.get_type() == DialogType::Channel || message_id <= d->last_new_message_id) {
          LOG(ERROR) << "Receive updateMessageId from " << it.second << " to " << full_message_id
                     << " but not receive corresponding message, last_new_message_id = " << d->last_new_message_id;
        }
        if (dialog_id.get_type() != DialogType::Channel && message_id <= d->last_new_message_id) {
          dump_debug_message_op(get_dialog(dialog_id));
        }
        if (message_id <= d->last_new_message_id) {
          get_message_from_server(it.first, PromiseCreator::lambda([full_message_id](Result<Unit> result) {
                                    if (result.is_error()) {
                                      LOG(WARNING)
                                          << "Failed to get missing " << full_message_id << ": " << result.error();
                                    } else {
                                      LOG(WARNING) << "Successfully get missing " << full_message_id;
                                    }
                                  }),
                                  "get missing");
        } else if (dialog_id.get_type() == DialogType::Channel) {
          LOG(INFO) << "Schedule getDifference in " << dialog_id.get_channel_id();
          channel_get_difference_retry_timeout_.add_timeout_in(dialog_id.get(), 0.001);
        }
        break;
      }
      case DialogType::SecretChat:
        break;
      case DialogType::None:
      default:
        UNREACHABLE();
        break;
    }
  }
  for (auto full_message_id : update_message_ids_to_delete) {
    update_message_ids_.erase(full_message_id);
  }

  if (!td_->auth_manager_->is_bot()) {
    if (!G()->td_db()->get_binlog_pmc()->isset("fetched_marks_as_unread")) {
      td_->create_handler<GetDialogUnreadMarksQuery>()->send();
    }

    load_notification_settings();

    auto dialog_list_id = DialogListId(FolderId::archive());
    auto *list = get_dialog_list(dialog_list_id);
    CHECK(list != nullptr);
    if (!list->is_dialog_unread_count_inited_) {
      int32 limit = list->are_pinned_dialogs_inited_ ? static_cast<int32>(list->pinned_dialogs_.size())
                                                     : get_pinned_dialogs_limit(dialog_list_id);
      LOG(INFO) << "Loading chat list in " << dialog_list_id << " to init total unread count";
      get_dialogs_from_list(dialog_list_id, limit + 2, Auto());
    }
  }
}

void MessagesManager::on_get_empty_messages(DialogId dialog_id, const vector<MessageId> &empty_message_ids) {
  if (!empty_message_ids.empty()) {
    delete_dialog_messages(dialog_id, empty_message_ids, true, true, "on_get_empty_messages");
  }
}

MessagesManager::MessagesInfo MessagesManager::get_messages_info(
    tl_object_ptr<telegram_api::messages_Messages> &&messages_ptr, const char *source) {
  CHECK(messages_ptr != nullptr);
  LOG(DEBUG) << "Receive result for " << source << ": " << to_string(messages_ptr);

  vector<tl_object_ptr<telegram_api::User>> users;
  vector<tl_object_ptr<telegram_api::Chat>> chats;
  MessagesInfo result;
  switch (messages_ptr->get_id()) {
    case telegram_api::messages_messages::ID: {
      auto messages = move_tl_object_as<telegram_api::messages_messages>(messages_ptr);

      users = std::move(messages->users_);
      chats = std::move(messages->chats_);
      result.total_count = narrow_cast<int32>(messages->messages_.size());
      result.messages = std::move(messages->messages_);
      break;
    }
    case telegram_api::messages_messagesSlice::ID: {
      auto messages = move_tl_object_as<telegram_api::messages_messagesSlice>(messages_ptr);

      users = std::move(messages->users_);
      chats = std::move(messages->chats_);
      result.total_count = messages->count_;
      result.messages = std::move(messages->messages_);
      break;
    }
    case telegram_api::messages_channelMessages::ID: {
      auto messages = move_tl_object_as<telegram_api::messages_channelMessages>(messages_ptr);

      users = std::move(messages->users_);
      chats = std::move(messages->chats_);
      result.total_count = messages->count_;
      result.messages = std::move(messages->messages_);
      result.is_channel_messages = true;
      break;
    }
    case telegram_api::messages_messagesNotModified::ID:
      LOG(ERROR) << "Server returned messagesNotModified in response to " << source;
      break;
    default:
      UNREACHABLE();
      break;
  }

  td_->contacts_manager_->on_get_users(std::move(users), source);
  td_->contacts_manager_->on_get_chats(std::move(chats), source);

  return result;
}

void MessagesManager::get_channel_difference_if_needed(DialogId dialog_id, MessagesInfo &&messages_info,
                                                       Promise<MessagesInfo> &&promise) {
  for (auto &message : messages_info.messages) {
    if (need_channel_difference_to_add_message(dialog_id, message)) {
      return run_after_channel_difference(
          dialog_id,
          PromiseCreator::lambda([messages_info = std::move(messages_info), promise = std::move(promise)](
                                     Unit ignored) mutable { promise.set_value(std::move(messages_info)); }));
    }
  }
  promise.set_value(std::move(messages_info));
}

void MessagesManager::get_channel_differences_if_needed(MessagesInfo &&messages_info, Promise<MessagesInfo> &&promise) {
  MultiPromiseActorSafe mpas{"GetChannelDifferencesIfNeededMultiPromiseActor"};
  mpas.add_promise(Promise<Unit>());
  mpas.set_ignore_errors(true);
  auto lock = mpas.get_promise();
  for (auto &message : messages_info.messages) {
    if (message == nullptr) {
      continue;
    }

    auto dialog_id = get_message_dialog_id(message);
    if (need_channel_difference_to_add_message(dialog_id, message)) {
      run_after_channel_difference(dialog_id, mpas.get_promise());
    }
  }
  // must be added after messages_info is checked
  mpas.add_promise(PromiseCreator::lambda([messages_info = std::move(messages_info), promise = std::move(promise)](
                                              Unit ignored) mutable { promise.set_value(std::move(messages_info)); }));
  lock.set_value(Unit());
}

void MessagesManager::on_get_messages(vector<tl_object_ptr<telegram_api::Message>> &&messages, bool is_channel_message,
                                      bool is_scheduled, Promise<Unit> &&promise, const char *source) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  LOG(DEBUG) << "Receive " << messages.size() << " messages";
  for (auto &message : messages) {
    on_get_message(std::move(message), false, is_channel_message, is_scheduled, false, false, source);
  }
  promise.set_value(Unit());
}

void MessagesManager::on_get_history(DialogId dialog_id, MessageId from_message_id, MessageId old_last_new_message_id,
                                     int32 offset, int32 limit, bool from_the_end,
                                     vector<tl_object_ptr<telegram_api::Message>> &&messages, Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  LOG(INFO) << "Receive " << messages.size() << " history messages " << (from_the_end ? "from the end " : "") << "in "
            << dialog_id << " from " << from_message_id << " with offset " << offset << " and limit " << limit;
  CHECK(-limit < offset && offset <= 0);
  CHECK(offset < 0 || from_the_end);
  CHECK(!from_message_id.is_scheduled());

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  MessageId last_received_message_id = messages.empty() ? MessageId() : get_message_id(messages[0], false);
  if (old_last_new_message_id < d->last_new_message_id && (from_the_end || old_last_new_message_id < from_message_id) &&
      last_received_message_id < d->last_new_message_id) {
    // new server messages were added to the dialog since the request was sent, but weren't received
    // they should have been received, so we must repeat the request to get them
    if (from_the_end) {
      get_history_from_the_end_impl(d, false, false, std::move(promise));
    } else {
      get_history_impl(d, from_message_id, offset, limit, false, false, std::move(promise));
    }
    return;
  }

  // the server can return less messages than requested if some of messages are deleted during request
  // but if it happens, it is likely that there are no more messages on the server
  bool have_full_history = from_the_end && narrow_cast<int32>(messages.size()) < limit && messages.size() <= 1;

  if (messages.empty()) {
    if (have_full_history) {
      d->have_full_history = true;
      on_dialog_updated(dialog_id, "set have_full_history");
    }

    if (from_the_end && d->have_full_history && d->messages == nullptr) {
      if (!d->last_database_message_id.is_valid()) {
        set_dialog_is_empty(d, "on_get_history empty");
      } else {
        LOG(INFO) << "Skip marking " << dialog_id << " as empty, because it probably has messages from "
                  << d->first_database_message_id << " to " << d->last_database_message_id << " in the database";
      }
    }

    // be aware that in some cases an empty answer may be returned, because of the race of getHistory and deleteMessages
    // and not because there are no more messages
    promise.set_value(Unit());
    return;
  }

  if (messages.size() > 1) {
    // check that messages are received in decreasing message_id order
    MessageId cur_message_id = MessageId::max();
    for (const auto &message : messages) {
      MessageId message_id = get_message_id(message, false);
      if (message_id >= cur_message_id) {
        string error = PSTRING() << "Receive messages in the wrong order in history of " << dialog_id << " from "
                                 << from_message_id << " with offset " << offset << ", limit " << limit
                                 << ", from_the_end = " << from_the_end << ": ";
        for (const auto &debug_message : messages) {
          error += to_string(debug_message);
        }
        // TODO move to ERROR
        LOG(FATAL) << error;
        promise.set_value(Unit());
        return;
      }
      cur_message_id = message_id;
    }
  }

  // be aware that returned messages are guaranteed to be consecutive messages, but if !from_the_end they
  // may be older (if some messages was deleted) or newer (if some messages were added) than an expected answer
  // be aware that any subset of the returned messages may be already deleted and returned as MessageEmpty
  bool is_channel_message = dialog_id.get_type() == DialogType::Channel;
  MessageId first_added_message_id;
  MessageId last_added_message_id;
  bool have_next = false;

  if (narrow_cast<int32>(messages.size()) < limit + offset && messages.size() <= 1) {
    MessageId first_received_message_id = get_message_id(messages.back(), false);
    if (first_received_message_id >= from_message_id && d->first_database_message_id.is_valid() &&
        first_received_message_id >= d->first_database_message_id) {
      // it is likely that there are no more history messages on the server
      have_full_history = true;
    }
  }

  bool prev_have_full_history = d->have_full_history;
  MessageId prev_last_new_message_id = d->last_new_message_id;
  MessageId prev_first_database_message_id = d->first_database_message_id;
  MessageId prev_last_database_message_id = d->last_database_message_id;
  MessageId prev_last_message_id = d->last_message_id;
  if (from_the_end) {
    // delete all server messages with ID > last_received_message_id
    // there were no new messages received after the getHistory request was sent, so they are already deleted message
    vector<MessageId> message_ids;
    find_newer_messages(d->messages.get(), last_received_message_id, message_ids);
    if (!message_ids.empty()) {
      bool need_update_dialog_pos = false;
      vector<int64> deleted_message_ids;
      for (auto message_id : message_ids) {
        CHECK(message_id > last_received_message_id);
        if (message_id.is_server()) {
          auto message = delete_message(d, message_id, true, &need_update_dialog_pos, "on_get_gistory 1");
          CHECK(message != nullptr);
          deleted_message_ids.push_back(message->message_id.get());
        }
      }
      if (need_update_dialog_pos) {
        send_update_chat_last_message(d, "on_get_gistory 2");
      }

      if (!deleted_message_ids.empty()) {
        send_update_delete_messages(dialog_id, std::move(deleted_message_ids), true, false);

        message_ids.clear();
        find_newer_messages(d->messages.get(), last_received_message_id, message_ids);
      }

      // connect all messages with ID > last_received_message_id
      for (size_t i = 0; i + 1 < message_ids.size(); i++) {
        auto m = get_message(d, message_ids[i]);
        CHECK(m != nullptr);
        if (!m->have_next) {
          m->have_next = true;
          attach_message_to_next(d, message_ids[i], "on_get_history 3");
        }
      }

      if (!message_ids.empty()) {
        have_next = true;
      }
    }
  }

  for (auto &message : messages) {
    if (!have_next && from_the_end && get_message_id(message, false) < d->last_message_id) {
      // last message in the dialog should be attached to the next message if there is some
      have_next = true;
    }

    auto message_dialog_id = get_message_dialog_id(message);
    if (message_dialog_id != dialog_id) {
      LOG(ERROR) << "Receive " << get_message_id(message, false) << " in wrong " << message_dialog_id << " instead of "
                 << dialog_id << ": " << oneline(to_string(message));
      continue;
    }

    auto full_message_id =
        on_get_message(std::move(message), false, is_channel_message, false, false, have_next, "get history");
    auto message_id = full_message_id.get_message_id();
    if (message_id.is_valid()) {
      if (!last_added_message_id.is_valid()) {
        last_added_message_id = message_id;
      }

      if (!have_next) {
        have_next = true;
      } else if (first_added_message_id.is_valid()) {
        Message *next_message = get_message(d, first_added_message_id);
        CHECK(next_message != nullptr);
        if (!next_message->have_previous) {
          LOG(INFO) << "Fix have_previous for " << first_added_message_id;
          next_message->have_previous = true;
          attach_message_to_previous(d, first_added_message_id, "on_get_history");
        }
      }
      first_added_message_id = message_id;
      if (!message_id.is_yet_unsent()) {
        // message should be already saved to database in on_get_message
        // add_message_to_database(d, get_message(d, message_id), "on_get_history");
      }
    }
  }

  if (have_full_history) {
    d->have_full_history = true;
    on_dialog_updated(dialog_id, "set have_full_history 2");
  }

  //  LOG_IF(ERROR, d->first_message_id.is_valid() && d->first_message_id > first_received_message_id)
  //      << "Receive " << first_received_message_id << ", but first chat message is " << d->first_message_id;

  bool intersect_last_database_message_ids =
      last_added_message_id >= d->first_database_message_id && d->last_database_message_id >= first_added_message_id;
  bool need_update_database_message_ids =
      last_added_message_id.is_valid() && (from_the_end || intersect_last_database_message_ids);
  if (from_the_end) {
    if (!d->last_new_message_id.is_valid()) {
      set_dialog_last_new_message_id(
          d, last_added_message_id.is_valid() ? last_added_message_id : last_received_message_id, "on_get_history");
    }
    if (last_added_message_id.is_valid() && last_added_message_id > d->last_message_id) {
      CHECK(d->last_new_message_id.is_valid());
      set_dialog_last_message_id(d, last_added_message_id, "on_get_history");
      send_update_chat_last_message(d, "on_get_history");
    }
  }

  if (need_update_database_message_ids) {
    if (from_the_end && !intersect_last_database_message_ids && d->last_database_message_id.is_valid()) {
      if (d->last_database_message_id < first_added_message_id || last_added_message_id == d->last_message_id) {
        set_dialog_first_database_message_id(d, MessageId(), "on_get_history 1");
        set_dialog_last_database_message_id(d, MessageId(), "on_get_history 1");
      } else {
        auto min_message_id = td::min(d->first_database_message_id, d->last_message_id);
        CHECK(last_added_message_id < min_message_id);
        if (min_message_id <= last_added_message_id.get_next_message_id(MessageType::Server)) {
          // connect local messages with last received server message
          set_dialog_first_database_message_id(d, last_added_message_id, "on_get_history 2");
        } else {
          LOG(WARNING) << "Have last " << d->last_message_id << " and first database " << d->first_database_message_id
                       << " in " << dialog_id << ", but received history from the end only up to "
                       << last_added_message_id;
          // can't connect messages, because there can be unknown server messages after last_added_message_id
        }
      }
    }
    if (!d->last_database_message_id.is_valid()) {
      CHECK(d->last_message_id.is_valid());
      MessagesConstIterator it(d, d->last_message_id);
      MessageId new_first_database_message_id;
      while (*it != nullptr) {
        auto message_id = (*it)->message_id;
        if (message_id.is_server() || message_id.is_local()) {
          if (!d->last_database_message_id.is_valid()) {
            set_dialog_last_database_message_id(d, message_id, "on_get_history");
          }
          new_first_database_message_id = message_id;
          try_restore_dialog_reply_markup(d, *it);
        }
        --it;
      }
      if (new_first_database_message_id.is_valid()) {
        set_dialog_first_database_message_id(d, new_first_database_message_id, "on_get_history");
      }
    } else {
      LOG_CHECK(d->last_new_message_id.is_valid())
          << dialog_id << " " << from_the_end << " " << d->first_database_message_id << " "
          << d->last_database_message_id << " " << first_added_message_id << " " << last_added_message_id << " "
          << d->last_message_id << " " << d->last_new_message_id << " " << d->have_full_history << " "
          << prev_last_new_message_id << " " << prev_first_database_message_id << " " << prev_last_database_message_id
          << " " << prev_last_message_id << " " << prev_have_full_history << " " << d->debug_last_new_message_id << " "
          << d->debug_first_database_message_id << " " << d->debug_last_database_message_id << " " << from_message_id
          << " " << offset << " " << limit << " " << messages.size() << " " << last_received_message_id << " "
          << d->debug_set_dialog_last_database_message_id;
      CHECK(d->first_database_message_id.is_valid());
      {
        MessagesConstIterator it(d, d->first_database_message_id);
        if (*it != nullptr && ((*it)->message_id == d->first_database_message_id || (*it)->have_next)) {
          MessageId new_first_database_message_id = d->first_database_message_id;
          while (*it != nullptr) {
            auto message_id = (*it)->message_id;
            if ((message_id.is_server() || message_id.is_local()) && message_id < new_first_database_message_id) {
              new_first_database_message_id = message_id;
              try_restore_dialog_reply_markup(d, *it);
            }
            --it;
          }
          if (new_first_database_message_id != d->first_database_message_id) {
            set_dialog_first_database_message_id(d, new_first_database_message_id, "on_get_history 2");
          }
        }
      }
      {
        MessagesConstIterator it(d, d->last_database_message_id);
        if (*it != nullptr && ((*it)->message_id == d->last_database_message_id || (*it)->have_next)) {
          MessageId new_last_database_message_id = d->last_database_message_id;
          while (*it != nullptr) {
            auto message_id = (*it)->message_id;
            if ((message_id.is_server() || message_id.is_local()) && message_id > new_last_database_message_id) {
              new_last_database_message_id = message_id;
            }
            ++it;
          }
          if (new_last_database_message_id != d->last_database_message_id) {
            set_dialog_last_database_message_id(d, new_last_database_message_id, "on_get_history 2");
          }
        }
      }
    }
    LOG_CHECK(d->first_database_message_id.is_valid())
        << dialog_id << " " << from_the_end << " " << d->first_database_message_id << " " << d->last_database_message_id
        << " " << first_added_message_id << " " << last_added_message_id << " " << d->last_message_id << " "
        << d->last_new_message_id << " " << d->have_full_history << " " << prev_last_new_message_id << " "
        << prev_first_database_message_id << " " << prev_last_database_message_id << " " << prev_last_message_id << " "
        << prev_have_full_history << " " << d->debug_last_new_message_id << " " << d->debug_first_database_message_id
        << " " << d->debug_last_database_message_id << " " << from_message_id << " " << offset << " " << limit << " "
        << messages.size() << " " << last_received_message_id << " " << d->debug_set_dialog_last_database_message_id;

    CHECK(d->last_database_message_id.is_valid());

    for (auto &first_message_id : d->first_database_message_id_by_index) {
      if (first_added_message_id < first_message_id && first_message_id <= last_added_message_id) {
        first_message_id = first_added_message_id;
      }
    }
  }
  promise.set_value(Unit());
}

vector<DialogId> MessagesManager::get_peers_dialog_ids(vector<tl_object_ptr<telegram_api::Peer>> &&peers) {
  vector<DialogId> result;
  result.reserve(peers.size());
  for (auto &peer : peers) {
    DialogId dialog_id(peer);
    if (dialog_id.is_valid()) {
      force_create_dialog(dialog_id, "get_peers_dialog_ids");
      result.push_back(dialog_id);
    }
  }
  return result;
}

void MessagesManager::on_get_public_dialogs_search_result(const string &query,
                                                          vector<tl_object_ptr<telegram_api::Peer>> &&my_peers,
                                                          vector<tl_object_ptr<telegram_api::Peer>> &&peers) {
  auto it = search_public_dialogs_queries_.find(query);
  CHECK(it != search_public_dialogs_queries_.end());
  CHECK(!it->second.empty());
  auto promises = std::move(it->second);
  search_public_dialogs_queries_.erase(it);

  found_public_dialogs_[query] = get_peers_dialog_ids(std::move(peers));
  found_on_server_dialogs_[query] = get_peers_dialog_ids(std::move(my_peers));

  for (auto &promise : promises) {
    promise.set_value(Unit());
  }
}

void MessagesManager::on_failed_public_dialogs_search(const string &query, Status &&error) {
  auto it = search_public_dialogs_queries_.find(query);
  CHECK(it != search_public_dialogs_queries_.end());
  CHECK(!it->second.empty());
  auto promises = std::move(it->second);
  search_public_dialogs_queries_.erase(it);

  found_public_dialogs_[query];     // negative cache
  found_on_server_dialogs_[query];  // negative cache

  for (auto &promise : promises) {
    promise.set_error(error.clone());
  }
}

void MessagesManager::on_get_message_search_result_calendar(
    DialogId dialog_id, MessageId from_message_id, MessageSearchFilter filter, int64 random_id, int32 total_count,
    vector<tl_object_ptr<telegram_api::Message>> &&messages,
    vector<tl_object_ptr<telegram_api::searchResultsCalendarPeriod>> &&periods, Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  auto it = found_dialog_message_calendars_.find(random_id);
  CHECK(it != found_dialog_message_calendars_.end());

  int32 received_message_count = 0;
  for (auto &message : messages) {
    auto new_full_message_id = on_get_message(std::move(message), false, dialog_id.get_type() == DialogType::Channel,
                                              false, false, false, "on_get_message_search_result_calendar");
    if (new_full_message_id == FullMessageId()) {
      total_count--;
      continue;
    }

    if (new_full_message_id.get_dialog_id() != dialog_id) {
      LOG(ERROR) << "Receive " << new_full_message_id << " instead of a message in " << dialog_id;
      total_count--;
      continue;
    }

    received_message_count++;
  }
  if (total_count < received_message_count) {
    LOG(ERROR) << "Receive " << received_message_count << " valid messages out of " << total_count << " in "
               << messages.size() << " messages";
    total_count = received_message_count;
  }

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  auto &old_message_count = d->message_count_by_index[message_search_filter_index(filter)];
  if (old_message_count != total_count) {
    old_message_count = total_count;
    on_dialog_updated(dialog_id, "on_get_message_search_result_calendar");
  }

  vector<td_api::object_ptr<td_api::messageCalendarDay>> days;
  for (auto &period : periods) {
    auto message_id = MessageId(ServerMessageId(period->min_msg_id_));
    const auto *m = get_message(d, message_id);
    if (m == nullptr) {
      LOG(ERROR) << "Failed to find " << message_id;
      continue;
    }
    if (period->count_ <= 0) {
      LOG(ERROR) << "Receive " << to_string(period);
      continue;
    }
    days.push_back(td_api::make_object<td_api::messageCalendarDay>(
        period->count_, get_message_object(dialog_id, m, "on_get_message_search_result_calendar")));
  }
  it->second = td_api::make_object<td_api::messageCalendar>(total_count, std::move(days));
  promise.set_value(Unit());
}

void MessagesManager::on_failed_get_message_search_result_calendar(DialogId dialog_id, int64 random_id) {
  auto it = found_dialog_message_calendars_.find(random_id);
  CHECK(it != found_dialog_message_calendars_.end());
  found_dialog_message_calendars_.erase(it);
}

void MessagesManager::on_get_dialog_messages_search_result(
    DialogId dialog_id, const string &query, DialogId sender_dialog_id, MessageId from_message_id, int32 offset,
    int32 limit, MessageSearchFilter filter, MessageId top_thread_message_id, int64 random_id, int32 total_count,
    vector<tl_object_ptr<telegram_api::Message>> &&messages, Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  LOG(INFO) << "Receive " << messages.size() << " found messages in " << dialog_id;
  if (!dialog_id.is_valid()) {
    CHECK(query.empty());
    CHECK(!sender_dialog_id.is_valid());
    CHECK(!top_thread_message_id.is_valid());
    auto it = found_call_messages_.find(random_id);
    CHECK(it != found_call_messages_.end());

    MessageId first_added_message_id;
    if (messages.empty()) {
      // messages may be empty because there are no more messages or they can't be found due to global limit
      // anyway pretend that there are no more messages
      first_added_message_id = MessageId::min();
    }

    auto &result = it->second.second;
    CHECK(result.empty());
    int32 added_message_count = 0;
    for (auto &message : messages) {
      auto new_full_message_id =
          on_get_message(std::move(message), false, false, false, false, false, "search call messages");
      if (new_full_message_id != FullMessageId()) {
        result.push_back(new_full_message_id);
        added_message_count++;
      }

      auto message_id = new_full_message_id.get_message_id();
      if (message_id < first_added_message_id || !first_added_message_id.is_valid()) {
        first_added_message_id = message_id;
      }
    }
    if (total_count < added_message_count) {
      LOG(ERROR) << "Receive total_count = " << total_count << ", but added " << added_message_count
                 << " messages out of " << messages.size();
      total_count = added_message_count;
    }
    if (G()->parameters().use_message_db) {
      bool update_state = false;

      auto &old_message_count = calls_db_state_.message_count_by_index[call_message_search_filter_index(filter)];
      if (old_message_count != total_count) {
        LOG(INFO) << "Update calls database message count to " << total_count;
        old_message_count = total_count;
        update_state = true;
      }

      auto &old_first_db_message_id =
          calls_db_state_.first_calls_database_message_id_by_index[call_message_search_filter_index(filter)];
      bool from_the_end = !from_message_id.is_valid() || from_message_id >= MessageId::max();
      LOG(INFO) << "Have from_the_end = " << from_the_end << ", old_first_db_message_id = " << old_first_db_message_id
                << ", first_added_message_id = " << first_added_message_id << ", from_message_id = " << from_message_id;
      if ((from_the_end || (old_first_db_message_id.is_valid() && old_first_db_message_id <= from_message_id)) &&
          (!old_first_db_message_id.is_valid() || first_added_message_id < old_first_db_message_id)) {
        LOG(INFO) << "Update calls database first message to " << first_added_message_id;
        old_first_db_message_id = first_added_message_id;
        update_state = true;
      }
      if (update_state) {
        save_calls_db_state();
      }
    }
    it->second.first = total_count;
    promise.set_value(Unit());
    return;
  }

  auto it = found_dialog_messages_.find(random_id);
  CHECK(it != found_dialog_messages_.end());

  auto &result = it->second.second;
  CHECK(result.empty());
  MessageId first_added_message_id;
  if (messages.empty()) {
    // messages may be empty because there are no more messages or they can't be found due to global limit
    // anyway pretend that there are no more messages
    first_added_message_id = MessageId::min();
  }
  bool can_be_in_different_dialog = top_thread_message_id.is_valid() && is_broadcast_channel(dialog_id);
  DialogId real_dialog_id;
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  for (auto &message : messages) {
    auto new_full_message_id = on_get_message(std::move(message), false, dialog_id.get_type() == DialogType::Channel,
                                              false, false, false, "on_get_dialog_messages_search_result");
    if (new_full_message_id == FullMessageId()) {
      total_count--;
      continue;
    }

    if (new_full_message_id.get_dialog_id() != dialog_id) {
      if (!can_be_in_different_dialog) {
        LOG(ERROR) << "Receive " << new_full_message_id << " instead of a message in " << dialog_id;
        total_count--;
        continue;
      } else {
        if (!real_dialog_id.is_valid()) {
          real_dialog_id = new_full_message_id.get_dialog_id();
          found_dialog_messages_dialog_id_[random_id] = real_dialog_id;
        } else if (new_full_message_id.get_dialog_id() != real_dialog_id) {
          LOG(ERROR) << "Receive " << new_full_message_id << " instead of a message in " << real_dialog_id << " or "
                     << dialog_id;
          total_count--;
          continue;
        }
      }
    }

    auto message_id = new_full_message_id.get_message_id();
    if (filter == MessageSearchFilter::UnreadMention && message_id <= d->last_read_all_mentions_message_id &&
        !real_dialog_id.is_valid()) {
      total_count--;
      continue;
    }

    // TODO check that messages are returned in decreasing message_id order
    if (message_id < first_added_message_id || !first_added_message_id.is_valid()) {
      first_added_message_id = message_id;
    }
    result.push_back(message_id);
  }
  if (total_count < static_cast<int32>(result.size())) {
    LOG(ERROR) << "Receive " << result.size() << " valid messages out of " << total_count << " in " << messages.size()
               << " messages";
    total_count = static_cast<int32>(result.size());
  }
  if (query.empty() && !sender_dialog_id.is_valid() && filter != MessageSearchFilter::Empty &&
      !top_thread_message_id.is_valid()) {
    bool from_the_end = !from_message_id.is_valid() ||
                        (d->last_message_id != MessageId() && from_message_id > d->last_message_id) ||
                        from_message_id >= MessageId::max();
    bool update_dialog = false;

    auto &old_message_count = d->message_count_by_index[message_search_filter_index(filter)];
    if (old_message_count != total_count) {
      old_message_count = total_count;
      if (filter == MessageSearchFilter::UnreadMention) {
        d->unread_mention_count = old_message_count;
        update_dialog_mention_notification_count(d);
        send_update_chat_unread_mention_count(d);
      }
      update_dialog = true;
    }

    auto &old_first_database_message_id = d->first_database_message_id_by_index[message_search_filter_index(filter)];
    if ((from_the_end ||
         (old_first_database_message_id.is_valid() && old_first_database_message_id <= from_message_id)) &&
        (!old_first_database_message_id.is_valid() || first_added_message_id < old_first_database_message_id)) {
      old_first_database_message_id = first_added_message_id;
      update_dialog = true;
    }
    if (update_dialog) {
      on_dialog_updated(dialog_id, "search results");
    }

    if (from_the_end && filter == MessageSearchFilter::Pinned) {
      set_dialog_last_pinned_message_id(d, result.empty() ? MessageId() : result[0]);
    }
  }

  it->second.first = total_count;
  promise.set_value(Unit());
}

void MessagesManager::on_failed_dialog_messages_search(DialogId dialog_id, int64 random_id) {
  if (!dialog_id.is_valid()) {
    auto it = found_call_messages_.find(random_id);
    CHECK(it != found_call_messages_.end());
    found_call_messages_.erase(it);
    return;
  }

  auto it = found_dialog_messages_.find(random_id);
  CHECK(it != found_dialog_messages_.end());
  found_dialog_messages_.erase(it);
}

void MessagesManager::on_get_dialog_message_count(DialogId dialog_id, MessageSearchFilter filter, int32 total_count,
                                                  Promise<int32> &&promise) {
  LOG(INFO) << "Receive " << total_count << " message count in " << dialog_id << " with filter " << filter;
  if (total_count < 0) {
    LOG(ERROR) << "Receive total message count = " << total_count << " in " << dialog_id << " with filter " << filter;
    total_count = 0;
  }

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  CHECK(filter != MessageSearchFilter::Empty);
  CHECK(filter != MessageSearchFilter::UnreadMention);
  CHECK(filter != MessageSearchFilter::FailedToSend);

  auto &old_message_count = d->message_count_by_index[message_search_filter_index(filter)];
  if (old_message_count != total_count) {
    old_message_count = total_count;
    on_dialog_updated(dialog_id, "on_get_dialog_message_count");
  }

  if (total_count == 0) {
    auto &old_first_database_message_id = d->first_database_message_id_by_index[message_search_filter_index(filter)];
    if (old_first_database_message_id != MessageId::min()) {
      old_first_database_message_id = MessageId::min();
      on_dialog_updated(dialog_id, "on_get_dialog_message_count");
    }
    if (filter == MessageSearchFilter::Pinned) {
      set_dialog_last_pinned_message_id(d, MessageId());
    }
  }
  promise.set_value(std::move(total_count));
}

void MessagesManager::on_get_messages_search_result(const string &query, int32 offset_date, DialogId offset_dialog_id,
                                                    MessageId offset_message_id, int32 limit,
                                                    MessageSearchFilter filter, int32 min_date, int32 max_date,
                                                    int64 random_id, int32 total_count,
                                                    vector<tl_object_ptr<telegram_api::Message>> &&messages,
                                                    Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  LOG(INFO) << "Receive " << messages.size() << " found messages";
  auto it = found_messages_.find(random_id);
  CHECK(it != found_messages_.end());

  auto &result = it->second.second;
  CHECK(result.empty());
  for (auto &message : messages) {
    auto dialog_id = get_message_dialog_id(message);
    auto new_full_message_id = on_get_message(std::move(message), false, dialog_id.get_type() == DialogType::Channel,
                                              false, false, false, "search messages");
    if (new_full_message_id != FullMessageId()) {
      CHECK(dialog_id == new_full_message_id.get_dialog_id());
      result.push_back(new_full_message_id);
    } else {
      total_count--;
    }
  }
  if (total_count < static_cast<int32>(result.size())) {
    LOG(ERROR) << "Receive " << result.size() << " valid messages out of " << total_count << " in " << messages.size()
               << " messages";
    total_count = static_cast<int32>(result.size());
  }
  it->second.first = total_count;
  promise.set_value(Unit());
}

void MessagesManager::on_failed_messages_search(int64 random_id) {
  auto it = found_messages_.find(random_id);
  CHECK(it != found_messages_.end());
  found_messages_.erase(it);
}

void MessagesManager::on_get_scheduled_server_messages(DialogId dialog_id, uint32 generation,
                                                       vector<tl_object_ptr<telegram_api::Message>> &&messages,
                                                       bool is_not_modified) {
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  if (generation < d->scheduled_messages_sync_generation) {
    LOG(INFO) << "Ignore scheduled messages with old generation " << generation << " instead of "
              << d->scheduled_messages_sync_generation << " in " << dialog_id;
    return;
  }
  d->scheduled_messages_sync_generation = generation;

  if (is_not_modified) {
    LOG(INFO) << "Scheduled messages are mot modified in " << dialog_id;
    return;
  }

  vector<MessageId> old_message_ids;
  find_old_messages(d->scheduled_messages.get(),
                    MessageId(ScheduledServerMessageId(), std::numeric_limits<int32>::max(), true), old_message_ids);
  std::unordered_map<ScheduledServerMessageId, MessageId, ScheduledServerMessageIdHash> old_server_message_ids;
  for (auto &message_id : old_message_ids) {
    if (message_id.is_scheduled_server()) {
      old_server_message_ids[message_id.get_scheduled_server_message_id()] = message_id;
    }
  }

  bool is_channel_message = dialog_id.get_type() == DialogType::Channel;
  bool has_scheduled_server_messages = false;
  for (auto &message : messages) {
    auto message_dialog_id = get_message_dialog_id(message);
    if (message_dialog_id != dialog_id) {
      LOG(ERROR) << "Receive " << get_message_id(message, true) << " in wrong " << message_dialog_id << " instead of "
                 << dialog_id << ": " << oneline(to_string(message));
      continue;
    }

    auto full_message_id = on_get_message(std::move(message), d->sent_scheduled_messages, is_channel_message, true,
                                          false, false, "on_get_scheduled_server_messages");
    auto message_id = full_message_id.get_message_id();
    if (message_id.is_valid_scheduled()) {
      CHECK(message_id.is_scheduled_server());
      old_server_message_ids.erase(message_id.get_scheduled_server_message_id());
      has_scheduled_server_messages = true;
    }
  }
  on_update_dialog_has_scheduled_server_messages(dialog_id, has_scheduled_server_messages);

  for (auto it : old_server_message_ids) {
    auto message_id = it.second;
    auto message = do_delete_scheduled_message(d, message_id, true, "on_get_scheduled_server_messages");
    CHECK(message != nullptr);
    send_update_delete_messages(dialog_id, {message->message_id.get()}, true, false);
  }

  send_update_chat_has_scheduled_messages(d, false);
}

void MessagesManager::on_get_recent_locations(DialogId dialog_id, int32 limit, int32 total_count,
                                              vector<tl_object_ptr<telegram_api::Message>> &&messages,
                                              Promise<td_api::object_ptr<td_api::messages>> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  LOG(INFO) << "Receive " << messages.size() << " recent locations in " << dialog_id;
  vector<MessageId> result;
  for (auto &message : messages) {
    auto new_full_message_id = on_get_message(std::move(message), false, dialog_id.get_type() == DialogType::Channel,
                                              false, false, false, "get recent locations");
    if (new_full_message_id != FullMessageId()) {
      if (new_full_message_id.get_dialog_id() != dialog_id) {
        LOG(ERROR) << "Receive " << new_full_message_id << " instead of a message in " << dialog_id;
        total_count--;
        continue;
      }
      auto m = get_message(new_full_message_id);
      CHECK(m != nullptr);
      if (m->content->get_type() != MessageContentType::LiveLocation) {
        LOG(ERROR) << "Receive a message of wrong type " << m->content->get_type() << " in on_get_recent_locations in "
                   << dialog_id;
        total_count--;
        continue;
      }

      result.push_back(m->message_id);
    } else {
      total_count--;
    }
  }
  if (total_count < static_cast<int32>(result.size())) {
    LOG(ERROR) << "Receive " << result.size() << " valid messages out of " << total_count << " in " << messages.size()
               << " messages";
    total_count = static_cast<int32>(result.size());
  }
  promise.set_value(get_messages_object(total_count, dialog_id, result, true, "on_get_recent_locations"));
}

void MessagesManager::on_get_message_public_forwards(int32 total_count,
                                                     vector<tl_object_ptr<telegram_api::Message>> &&messages,
                                                     Promise<td_api::object_ptr<td_api::foundMessages>> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  LOG(INFO) << "Receive " << messages.size() << " forwarded messages";
  vector<td_api::object_ptr<td_api::message>> result;
  FullMessageId last_full_message_id;
  for (auto &message : messages) {
    auto dialog_id = get_message_dialog_id(message);
    auto new_full_message_id = on_get_message(std::move(message), false, dialog_id.get_type() == DialogType::Channel,
                                              false, false, false, "get message public forwards");
    if (new_full_message_id != FullMessageId()) {
      CHECK(dialog_id == new_full_message_id.get_dialog_id());
      result.push_back(get_message_object(new_full_message_id, "on_get_message_public_forwards"));
      CHECK(result.back() != nullptr);
      last_full_message_id = new_full_message_id;
    } else {
      total_count--;
    }
  }
  if (total_count < static_cast<int32>(result.size())) {
    LOG(ERROR) << "Receive " << result.size() << " valid messages out of " << total_count << " in " << messages.size()
               << " messages";
    total_count = static_cast<int32>(result.size());
  }
  string next_offset;
  if (!result.empty()) {
    auto m = get_message(last_full_message_id);
    CHECK(m != nullptr);
    next_offset = PSTRING() << m->date << "," << last_full_message_id.get_dialog_id().get() << ","
                            << m->message_id.get_server_message_id().get();
  }

  promise.set_value(td_api::make_object<td_api::foundMessages>(total_count, std::move(result), next_offset));
}

void MessagesManager::delete_messages_from_updates(const vector<MessageId> &message_ids) {
  std::unordered_map<DialogId, vector<int64>, DialogIdHash> deleted_message_ids;
  std::unordered_map<DialogId, bool, DialogIdHash> need_update_dialog_pos;
  for (auto message_id : message_ids) {
    if (!message_id.is_valid() || !message_id.is_server()) {
      LOG(ERROR) << "Incoming update tries to delete " << message_id;
      continue;
    }

    Dialog *d = get_dialog_by_message_id(message_id);
    if (d != nullptr) {
      auto message = delete_message(d, message_id, true, &need_update_dialog_pos[d->dialog_id], "updates");
      CHECK(message != nullptr);
      LOG_CHECK(message->message_id == message_id) << message_id << " " << message->message_id << " " << d->dialog_id;
      deleted_message_ids[d->dialog_id].push_back(message->message_id.get());
    }
    if (last_clear_history_message_id_to_dialog_id_.count(message_id)) {
      d = get_dialog(last_clear_history_message_id_to_dialog_id_[message_id]);
      CHECK(d != nullptr);
      auto message = delete_message(d, message_id, true, &need_update_dialog_pos[d->dialog_id], "updates");
      CHECK(message == nullptr);
    }
  }
  for (auto &it : need_update_dialog_pos) {
    if (it.second) {
      auto dialog_id = it.first;
      Dialog *d = get_dialog(dialog_id);
      CHECK(d != nullptr);
      send_update_chat_last_message(d, "delete_messages_from_updates");
    }
  }
  for (auto &it : deleted_message_ids) {
    auto dialog_id = it.first;
    send_update_delete_messages(dialog_id, std::move(it.second), true, false);
  }
}

void MessagesManager::delete_dialog_messages(DialogId dialog_id, const vector<MessageId> &message_ids,
                                             bool from_updates, bool skip_update_for_not_found_messages,
                                             const char *source) {
  Dialog *d = get_dialog_force(dialog_id, "delete_dialog_messages");
  if (d == nullptr) {
    LOG(INFO) << "Ignore deleteChannelMessages for unknown " << dialog_id << " from " << source;
    CHECK(from_updates);
    CHECK(dialog_id.get_type() == DialogType::Channel);
    return;
  }

  vector<int64> deleted_message_ids;
  bool need_update_dialog_pos = false;
  for (auto message_id : message_ids) {
    CHECK(!message_id.is_scheduled());
    if (from_updates) {
      if (!message_id.is_valid() || (!message_id.is_server() && dialog_id.get_type() != DialogType::SecretChat)) {
        LOG(ERROR) << "Tried to delete " << message_id << " in " << dialog_id << " from " << source;
        continue;
      }
    } else {
      CHECK(message_id.is_valid());
    }

    bool was_already_deleted = d->deleted_message_ids.count(message_id) != 0;
    auto message = delete_message(d, message_id, true, &need_update_dialog_pos, source);
    if (message == nullptr) {
      if (!skip_update_for_not_found_messages && !was_already_deleted) {
        deleted_message_ids.push_back(message_id.get());
      }
    } else {
      deleted_message_ids.push_back(message->message_id.get());
    }
  }
  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, source);
  }
  send_update_delete_messages(dialog_id, std::move(deleted_message_ids), true, false);
}

void MessagesManager::update_dialog_pinned_messages_from_updates(DialogId dialog_id,
                                                                 const vector<MessageId> &message_ids, bool is_pin) {
  Dialog *d = get_dialog_force(dialog_id, "update_dialog_pinned_messages_from_updates");
  if (d == nullptr) {
    LOG(INFO) << "Ignore updatePinnedMessages for unknown " << dialog_id;
    return;
  }

  for (auto message_id : message_ids) {
    if (!message_id.is_valid() || (!message_id.is_server() && dialog_id.get_type() != DialogType::SecretChat)) {
      LOG(ERROR) << "Incoming update tries to pin/unpin " << message_id << " in " << dialog_id;
      continue;
    }

    Message *m = get_message_force(d, message_id, "update_dialog_pinned_messages_from_updates");
    if (m != nullptr && update_message_is_pinned(d, m, is_pin, "update_dialog_pinned_messages_from_updates")) {
      on_message_changed(d, m, true, "update_dialog_pinned_messages_from_updates");
    }
  }
}

bool MessagesManager::update_message_is_pinned(Dialog *d, Message *m, bool is_pinned, const char *source) {
  CHECK(m != nullptr);
  CHECK(!m->message_id.is_scheduled());
  if (m->is_pinned == is_pinned) {
    return false;
  }

  LOG(INFO) << "Update message is_pinned of " << m->message_id << " in " << d->dialog_id << " to " << is_pinned
            << " from " << source;
  auto old_index_mask = get_message_index_mask(d->dialog_id, m);
  m->is_pinned = is_pinned;
  auto new_index_mask = get_message_index_mask(d->dialog_id, m);
  update_message_count_by_index(d, -1, old_index_mask & ~new_index_mask);
  update_message_count_by_index(d, +1, new_index_mask & ~old_index_mask);

  send_closure(G()->td(), &Td::send_update,
               make_tl_object<td_api::updateMessageIsPinned>(d->dialog_id.get(), m->message_id.get(), is_pinned));
  if (is_pinned) {
    if (d->is_last_pinned_message_id_inited && m->message_id > d->last_pinned_message_id) {
      set_dialog_last_pinned_message_id(d, m->message_id);
    }
  } else {
    if (d->is_last_pinned_message_id_inited && m->message_id == d->last_pinned_message_id) {
      if (d->message_count_by_index[message_search_filter_index(MessageSearchFilter::Pinned)] == 0) {
        set_dialog_last_pinned_message_id(d, MessageId());
      } else {
        drop_dialog_last_pinned_message_id(d);
      }
    }
  }
  return true;
}

string MessagesManager::get_message_search_text(const Message *m) const {
  if (m->is_content_secret) {
    return string();
  }
  return get_message_content_search_text(td_, m->content.get());
}

bool MessagesManager::can_forward_message(DialogId from_dialog_id, const Message *m) {
  if (m == nullptr) {
    return false;
  }
  if (m->ttl > 0) {
    return false;
  }
  if (m->message_id.is_scheduled()) {
    return false;
  }
  switch (from_dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      // ok
      break;
    case DialogType::SecretChat:
      return false;
    case DialogType::None:
    default:
      UNREACHABLE();
      return false;
  }

  return can_forward_message_content(m->content.get());
}

bool MessagesManager::can_get_message_statistics(FullMessageId full_message_id) {
  return can_get_message_statistics(full_message_id.get_dialog_id(),
                                    get_message_force(full_message_id, "can_get_message_statistics"));
}

bool MessagesManager::can_get_message_statistics(DialogId dialog_id, const Message *m) const {
  if (td_->auth_manager_->is_bot()) {
    return false;
  }
  if (m == nullptr || m->message_id.is_scheduled() || !m->message_id.is_server() || m->view_count == 0 ||
      m->had_forward_info || (m->forward_info != nullptr && m->forward_info->message_id.is_valid())) {
    return false;
  }
  return td_->contacts_manager_->can_get_channel_message_statistics(dialog_id);
}

bool MessagesManager::can_delete_channel_message(const DialogParticipantStatus &status, const Message *m, bool is_bot) {
  if (m == nullptr) {
    return true;
  }
  if (m->message_id.is_local() || m->message_id.is_yet_unsent()) {
    return true;
  }
  if (m->message_id.is_scheduled()) {
    if (m->is_channel_post) {
      return status.can_post_messages();
    }
    return true;
  }

  if (is_bot && G()->unix_time_cached() >= m->date + 2 * 86400) {
    // bots can't delete messages older than 2 days
    return false;
  }

  CHECK(m->message_id.is_server());
  if (m->message_id.get_server_message_id().get() == 1) {
    return false;
  }
  auto content_type = m->content->get_type();
  if (content_type == MessageContentType::ChannelMigrateFrom || content_type == MessageContentType::ChannelCreate) {
    return false;
  }

  if (status.can_delete_messages()) {
    return true;
  }

  if (!m->is_outgoing) {
    return false;
  }

  if (m->is_channel_post || is_service_message_content(content_type)) {
    return status.can_post_messages();
  }

  return true;
}

bool MessagesManager::can_delete_message(DialogId dialog_id, const Message *m) const {
  if (m == nullptr) {
    return true;
  }
  if (m->message_id.is_local() || m->message_id.is_yet_unsent()) {
    return true;
  }
  switch (dialog_id.get_type()) {
    case DialogType::User:
      return true;
    case DialogType::Chat:
      return true;
    case DialogType::Channel: {
      auto dialog_status = td_->contacts_manager_->get_channel_permissions(dialog_id.get_channel_id());
      return can_delete_channel_message(dialog_status, m, td_->auth_manager_->is_bot());
    }
    case DialogType::SecretChat:
      return true;
    case DialogType::None:
    default:
      UNREACHABLE();
      return false;
  }
}

bool MessagesManager::can_revoke_message(DialogId dialog_id, const Message *m) const {
  if (m == nullptr) {
    return true;
  }
  if (m->message_id.is_local()) {
    return false;
  }
  if (dialog_id == get_my_dialog_id()) {
    return false;
  }
  if (m->message_id.is_scheduled()) {
    return false;
  }
  if (m->message_id.is_yet_unsent()) {
    return true;
  }
  CHECK(m->message_id.is_server());

  const int32 DEFAULT_REVOKE_TIME_LIMIT = td_->auth_manager_->is_bot() ? 2 * 86400 : std::numeric_limits<int32>::max();
  auto content_type = m->content->get_type();
  switch (dialog_id.get_type()) {
    case DialogType::User: {
      bool can_revoke_incoming = G()->shared_config().get_option_boolean("revoke_pm_inbox", true);
      int64 revoke_time_limit =
          G()->shared_config().get_option_integer("revoke_pm_time_limit", DEFAULT_REVOKE_TIME_LIMIT);

      if (G()->unix_time_cached() - m->date < 86400 && content_type == MessageContentType::Dice) {
        return false;
      }
      return ((m->is_outgoing && !is_service_message_content(content_type)) ||
              (can_revoke_incoming && content_type != MessageContentType::ScreenshotTaken)) &&
             G()->unix_time_cached() - m->date <= revoke_time_limit;
    }
    case DialogType::Chat: {
      bool is_appointed_administrator =
          td_->contacts_manager_->is_appointed_chat_administrator(dialog_id.get_chat_id());
      int64 revoke_time_limit = G()->shared_config().get_option_integer("revoke_time_limit", DEFAULT_REVOKE_TIME_LIMIT);

      return ((m->is_outgoing && !is_service_message_content(content_type)) || is_appointed_administrator) &&
             G()->unix_time_cached() - m->date <= revoke_time_limit;
    }
    case DialogType::Channel:
      return true;  // any server message that can be deleted will be deleted for all participants
    case DialogType::SecretChat:
      // all non-service messages will be deleted for everyone if secret chat is active
      return td_->contacts_manager_->get_secret_chat_state(dialog_id.get_secret_chat_id()) == SecretChatState::Active &&
             !is_service_message_content(content_type);
    case DialogType::None:
    default:
      UNREACHABLE();
      return false;
  }
}

void MessagesManager::delete_messages(DialogId dialog_id, const vector<MessageId> &input_message_ids, bool revoke,
                                      Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());
  Dialog *d = get_dialog_force(dialog_id, "delete_messages");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat is not found"));
  }

  if (input_message_ids.empty()) {
    return promise.set_value(Unit());
  }

  auto dialog_type = dialog_id.get_type();
  bool is_secret = dialog_type == DialogType::SecretChat;

  vector<MessageId> message_ids;
  message_ids.reserve(input_message_ids.size());
  vector<MessageId> deleted_server_message_ids;

  vector<MessageId> deleted_scheduled_server_message_ids;
  for (auto message_id : input_message_ids) {
    if (!message_id.is_valid() && !message_id.is_valid_scheduled()) {
      return promise.set_error(Status::Error(400, "Invalid message identifier"));
    }

    message_id = get_persistent_message_id(d, message_id);
    message_ids.push_back(message_id);
    auto m = get_message_force(d, message_id, "delete_messages");
    if (m != nullptr) {
      if (m->message_id.is_scheduled()) {
        if (m->message_id.is_scheduled_server()) {
          deleted_scheduled_server_message_ids.push_back(m->message_id);
        }
      } else {
        if (m->message_id.is_server() || is_secret) {
          deleted_server_message_ids.push_back(m->message_id);
        }
      }
    }
  }

  bool is_bot = td_->auth_manager_->is_bot();
  for (auto message_id : message_ids) {
    auto m = get_message(d, message_id);
    if (!can_delete_message(dialog_id, m)) {
      return promise.set_error(Status::Error(400, "Message can't be deleted"));
    }
    if (is_bot && !message_id.is_scheduled() && message_id.is_server() && !can_revoke_message(dialog_id, m)) {
      return promise.set_error(Status::Error(400, "Message can't be deleted for everyone"));
    }
  }

  MultiPromiseActorSafe mpas{"DeleteMessagesOnServerMultiPromiseActor"};
  mpas.add_promise(std::move(promise));

  auto lock = mpas.get_promise();
  delete_messages_on_server(dialog_id, std::move(deleted_server_message_ids), revoke, 0, mpas.get_promise());
  delete_scheduled_messages_on_server(dialog_id, std::move(deleted_scheduled_server_message_ids), 0,
                                      mpas.get_promise());
  lock.set_value(Unit());

  bool need_update_dialog_pos = false;
  bool need_update_chat_has_scheduled_messages = false;
  vector<int64> deleted_message_ids;
  for (auto message_id : message_ids) {
    auto m = delete_message(d, message_id, true, &need_update_dialog_pos, DELETE_MESSAGE_USER_REQUEST_SOURCE);
    if (m == nullptr) {
      LOG(INFO) << "Can't delete " << message_id << " because it is not found";
    } else {
      need_update_chat_has_scheduled_messages |= m->message_id.is_scheduled();
      deleted_message_ids.push_back(m->message_id.get());
    }
  }

  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "delete_messages");
  }
  send_update_delete_messages(dialog_id, std::move(deleted_message_ids), true, false);

  if (need_update_chat_has_scheduled_messages) {
    send_update_chat_has_scheduled_messages(d, true);
  }
}

void MessagesManager::delete_sent_message_on_server(DialogId dialog_id, MessageId message_id) {
  // being sent message was deleted by the user or is in an inaccessible channel
  // don't need to send an update to the user, because the message has already been deleted
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    LOG(INFO) << "Ignore sent " << message_id << " in inaccessible " << dialog_id;
    return;
  }

  LOG(INFO) << "Delete already deleted sent " << message_id << " in " << dialog_id << " from server";
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  if (get_message_force(d, message_id, "delete_sent_message_on_server") != nullptr) {
    delete_messages(dialog_id, {message_id}, true, Auto());
  } else {
    if (message_id.is_valid()) {
      CHECK(message_id.is_server());
      delete_messages_on_server(dialog_id, {message_id}, true, 0, Auto());
    } else {
      CHECK(message_id.is_scheduled_server());
      delete_scheduled_messages_on_server(dialog_id, {message_id}, 0, Auto());
    }

    bool need_update_dialog_pos = false;
    auto m = delete_message(d, message_id, true, &need_update_dialog_pos, "delete_sent_message_on_server");
    CHECK(m == nullptr);
    CHECK(need_update_dialog_pos == false);
  }
}

class MessagesManager::DeleteMessagesOnServerLogEvent {
 public:
  DialogId dialog_id_;
  vector<MessageId> message_ids_;
  bool revoke_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(revoke_);
    END_STORE_FLAGS();

    td::store(dialog_id_, storer);
    td::store(message_ids_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(revoke_);
    END_PARSE_FLAGS();

    td::parse(dialog_id_, parser);
    td::parse(message_ids_, parser);
  }
};

uint64 MessagesManager::save_delete_messages_on_server_log_event(DialogId dialog_id,
                                                                 const vector<MessageId> &message_ids, bool revoke) {
  DeleteMessagesOnServerLogEvent log_event{dialog_id, message_ids, revoke};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::DeleteMessagesOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::delete_messages_on_server(DialogId dialog_id, vector<MessageId> message_ids, bool revoke,
                                                uint64 log_event_id, Promise<Unit> &&promise) {
  if (message_ids.empty()) {
    return promise.set_value(Unit());
  }
  LOG(INFO) << (revoke ? "Revoke " : "Delete ") << format::as_array(message_ids) << " in " << dialog_id
            << " from server";

  if (log_event_id == 0 && G()->parameters().use_message_db) {
    log_event_id = save_delete_messages_on_server_log_event(dialog_id, message_ids, revoke);
  }

  auto new_promise = get_erase_log_event_promise(log_event_id, std::move(promise));
  promise = std::move(new_promise);  // to prevent self-move

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
      td_->create_handler<DeleteMessagesQuery>(std::move(promise))->send(dialog_id, std::move(message_ids), revoke);
      break;
    case DialogType::Channel:
      td_->create_handler<DeleteChannelMessagesQuery>(std::move(promise))
          ->send(dialog_id.get_channel_id(), std::move(message_ids));
      break;
    case DialogType::SecretChat: {
      vector<int64> random_ids;
      auto d = get_dialog_force(dialog_id, "delete_messages_on_server");
      CHECK(d != nullptr);
      for (auto &message_id : message_ids) {
        auto *m = get_message(d, message_id);
        if (m != nullptr) {
          random_ids.push_back(m->random_id);
        }
      }
      if (!random_ids.empty()) {
        send_closure(G()->secret_chats_manager(), &SecretChatsManager::delete_messages, dialog_id.get_secret_chat_id(),
                     std::move(random_ids), std::move(promise));
      } else {
        promise.set_value(Unit());
      }
      break;
    }
    case DialogType::None:
    default:
      UNREACHABLE();
  }
}

class MessagesManager::DeleteScheduledMessagesOnServerLogEvent {
 public:
  DialogId dialog_id_;
  vector<MessageId> message_ids_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
    td::store(message_ids_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
    td::parse(message_ids_, parser);
  }
};

uint64 MessagesManager::save_delete_scheduled_messages_on_server_log_event(DialogId dialog_id,
                                                                           const vector<MessageId> &message_ids) {
  DeleteScheduledMessagesOnServerLogEvent log_event{dialog_id, message_ids};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::DeleteScheduledMessagesOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::delete_scheduled_messages_on_server(DialogId dialog_id, vector<MessageId> message_ids,
                                                          uint64 log_event_id, Promise<Unit> &&promise) {
  if (message_ids.empty()) {
    return promise.set_value(Unit());
  }
  LOG(INFO) << "Delete " << format::as_array(message_ids) << " in " << dialog_id << " from server";

  if (log_event_id == 0 && G()->parameters().use_message_db) {
    log_event_id = save_delete_scheduled_messages_on_server_log_event(dialog_id, message_ids);
  }

  auto new_promise = get_erase_log_event_promise(log_event_id, std::move(promise));
  promise = std::move(new_promise);  // to prevent self-move

  td_->create_handler<DeleteScheduledMessagesQuery>(std::move(promise))->send(dialog_id, std::move(message_ids));
}

void MessagesManager::delete_dialog_history(DialogId dialog_id, bool remove_from_dialog_list, bool revoke,
                                            Promise<Unit> &&promise) {
  LOG(INFO) << "Receive deleteChatHistory request to delete all messages in " << dialog_id
            << ", remove_from_chat_list is " << remove_from_dialog_list << ", revoke is " << revoke;

  Dialog *d = get_dialog_force(dialog_id, "delete_dialog_history");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return promise.set_error(Status::Error(400, "Chat info not found"));
  }

  if (is_dialog_sponsored(d)) {
    auto chat_source = sponsored_dialog_source_.get_chat_source_object();
    if (chat_source == nullptr || chat_source->get_id() != td_api::chatSourcePublicServiceAnnouncement::ID) {
      return promise.set_error(Status::Error(400, "Can't delete the chat"));
    }
    if (!remove_from_dialog_list) {
      return promise.set_error(
          Status::Error(400, "Can't delete only chat history without removing the chat from the chat list"));
    }

    removed_sponsored_dialog_id_ = dialog_id;
    remove_sponsored_dialog();

    td_->create_handler<HidePromoDataQuery>()->send(dialog_id);
    promise.set_value(Unit());
    return;
  }

  auto dialog_type = dialog_id.get_type();
  switch (dialog_type) {
    case DialogType::User:
    case DialogType::Chat:
      // ok
      break;
    case DialogType::Channel:
      if (is_broadcast_channel(dialog_id)) {
        return promise.set_error(Status::Error(400, "Can't delete chat history in a channel"));
      }
      if (td_->contacts_manager_->is_channel_public(dialog_id.get_channel_id())) {
        return promise.set_error(Status::Error(400, "Can't delete chat history in a public supergroup"));
      }
      break;
    case DialogType::SecretChat:
      // ok
      break;
    case DialogType::None:
    default:
      UNREACHABLE();
      break;
  }

  auto last_new_message_id = d->last_new_message_id;
  if (dialog_type != DialogType::SecretChat && last_new_message_id == MessageId()) {
    // TODO get dialog from the server and delete history from last message identifier
  }

  bool allow_error = d->messages == nullptr;
  auto old_order = d->order;

  delete_all_dialog_messages(d, remove_from_dialog_list, true);

  if (last_new_message_id.is_valid() && last_new_message_id == d->max_unavailable_message_id && !revoke &&
      !(old_order != DEFAULT_ORDER && remove_from_dialog_list)) {
    // history has already been cleared, nothing to do
    promise.set_value(Unit());
    return;
  }

  set_dialog_max_unavailable_message_id(dialog_id, last_new_message_id, false, "delete_dialog_history");

  delete_dialog_history_on_server(dialog_id, last_new_message_id, remove_from_dialog_list, revoke, allow_error, 0,
                                  std::move(promise));
}

class MessagesManager::DeleteDialogHistoryOnServerLogEvent {
 public:
  DialogId dialog_id_;
  MessageId max_message_id_;
  bool remove_from_dialog_list_;
  bool revoke_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(remove_from_dialog_list_);
    STORE_FLAG(revoke_);
    END_STORE_FLAGS();

    td::store(dialog_id_, storer);
    td::store(max_message_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(remove_from_dialog_list_);
    PARSE_FLAG(revoke_);
    END_PARSE_FLAGS();

    td::parse(dialog_id_, parser);
    td::parse(max_message_id_, parser);
  }
};

uint64 MessagesManager::save_delete_dialog_history_on_server_log_event(DialogId dialog_id, MessageId max_message_id,
                                                                       bool remove_from_dialog_list, bool revoke) {
  DeleteDialogHistoryOnServerLogEvent log_event{dialog_id, max_message_id, remove_from_dialog_list, revoke};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::DeleteDialogHistoryOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::delete_dialog_history_on_server(DialogId dialog_id, MessageId max_message_id,
                                                      bool remove_from_dialog_list, bool revoke, bool allow_error,
                                                      uint64 log_event_id, Promise<Unit> &&promise) {
  LOG(INFO) << "Delete history in " << dialog_id << " up to " << max_message_id << " from server";

  if (log_event_id == 0 && G()->parameters().use_message_db) {
    log_event_id =
        save_delete_dialog_history_on_server_log_event(dialog_id, max_message_id, remove_from_dialog_list, revoke);
  }

  auto new_promise = get_erase_log_event_promise(log_event_id, std::move(promise));
  promise = std::move(new_promise);  // to prevent self-move

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
      td_->create_handler<DeleteHistoryQuery>(std::move(promise))
          ->send(dialog_id, max_message_id, remove_from_dialog_list, revoke);
      break;
    case DialogType::Channel:
      td_->create_handler<DeleteChannelHistoryQuery>(std::move(promise))
          ->send(dialog_id.get_channel_id(), max_message_id, allow_error);
      break;
    case DialogType::SecretChat:
      send_closure(G()->secret_chats_manager(), &SecretChatsManager::delete_all_messages,
                   dialog_id.get_secret_chat_id(), std::move(promise));
      break;
    case DialogType::None:
    default:
      UNREACHABLE();
      break;
  }
}

void MessagesManager::delete_all_call_messages(bool revoke, Promise<Unit> &&promise) {
  delete_all_call_messages_on_server(revoke, 0, std::move(promise));
}

class MessagesManager::DeleteAllCallMessagesOnServerLogEvent {
 public:
  bool revoke_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(revoke_);
    END_STORE_FLAGS();
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(revoke_);
    END_PARSE_FLAGS();
  }
};

uint64 MessagesManager::save_delete_all_call_messages_on_server_log_event(bool revoke) {
  DeleteAllCallMessagesOnServerLogEvent log_event{revoke};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::DeleteAllCallMessagesOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::delete_all_call_messages_on_server(bool revoke, uint64 log_event_id, Promise<Unit> &&promise) {
  if (log_event_id == 0) {
    log_event_id = save_delete_all_call_messages_on_server_log_event(revoke);
  }

  auto new_promise = get_erase_log_event_promise(log_event_id, std::move(promise));
  promise = std::move(new_promise);  // to prevent self-move

  td_->create_handler<DeletePhoneCallHistoryQuery>(std::move(promise))->send(revoke);
}

void MessagesManager::find_messages(const Message *m, vector<MessageId> &message_ids,
                                    const std::function<bool(const Message *)> &condition) {
  if (m == nullptr) {
    return;
  }

  find_messages(m->left.get(), message_ids, condition);

  if (condition(m)) {
    message_ids.push_back(m->message_id);
  }

  find_messages(m->right.get(), message_ids, condition);
}

void MessagesManager::find_old_messages(const Message *m, MessageId max_message_id, vector<MessageId> &message_ids) {
  if (m == nullptr) {
    return;
  }

  find_old_messages(m->left.get(), max_message_id, message_ids);

  if (m->message_id <= max_message_id) {
    message_ids.push_back(m->message_id);

    find_old_messages(m->right.get(), max_message_id, message_ids);
  }
}

void MessagesManager::find_newer_messages(const Message *m, MessageId min_message_id, vector<MessageId> &message_ids) {
  if (m == nullptr) {
    return;
  }

  if (m->message_id > min_message_id) {
    find_newer_messages(m->left.get(), min_message_id, message_ids);

    message_ids.push_back(m->message_id);
  }

  find_newer_messages(m->right.get(), min_message_id, message_ids);
}

void MessagesManager::find_unloadable_messages(const Dialog *d, int32 unload_before_date, const Message *m,
                                               vector<MessageId> &message_ids, int32 &left_to_unload) const {
  if (m == nullptr) {
    return;
  }

  find_unloadable_messages(d, unload_before_date, m->left.get(), message_ids, left_to_unload);

  if (can_unload_message(d, m)) {
    if (m->last_access_date <= unload_before_date) {
      message_ids.push_back(m->message_id);
    } else {
      left_to_unload++;
    }
  }

  find_unloadable_messages(d, unload_before_date, m->right.get(), message_ids, left_to_unload);
}

void MessagesManager::delete_dialog_messages_from_user(DialogId dialog_id, UserId user_id, Promise<Unit> &&promise) {
  bool is_bot = td_->auth_manager_->is_bot();
  CHECK(!is_bot);

  Dialog *d = get_dialog_force(dialog_id, "delete_dialog_messages_from_user");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }

  if (!have_input_peer(dialog_id, AccessRights::Write)) {
    return promise.set_error(Status::Error(400, "Not enough rights"));
  }

  if (!td_->contacts_manager_->have_input_user(user_id)) {
    return promise.set_error(Status::Error(400, "User not found"));
  }

  ChannelId channel_id;
  DialogParticipantStatus channel_status = DialogParticipantStatus::Left();
  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::SecretChat:
      return promise.set_error(Status::Error(400, "All messages from a user can be deleted only in supergroup chats"));
    case DialogType::Channel: {
      channel_id = dialog_id.get_channel_id();
      auto channel_type = td_->contacts_manager_->get_channel_type(channel_id);
      if (channel_type != ContactsManager::ChannelType::Megagroup) {
        return promise.set_error(Status::Error(400, "The method is available only for supergroup chats"));
      }
      channel_status = td_->contacts_manager_->get_channel_permissions(channel_id);
      if (!channel_status.can_delete_messages()) {
        return promise.set_error(Status::Error(400, "Need delete messages administator right in the supergroup chat"));
      }
      channel_id = dialog_id.get_channel_id();
      break;
    }
    case DialogType::None:
    default:
      UNREACHABLE();
      break;
  }
  CHECK(channel_id.is_valid());

  if (G()->parameters().use_message_db) {
    LOG(INFO) << "Delete all messages from " << user_id << " in " << dialog_id << " from database";
    G()->td_db()->get_messages_db_async()->delete_dialog_messages_from_user(dialog_id, user_id,
                                                                            Auto());  // TODO Promise
  }

  vector<MessageId> message_ids;
  find_messages(d->messages.get(), message_ids, [user_id](const Message *m) { return m->sender_user_id == user_id; });

  vector<int64> deleted_message_ids;
  bool need_update_dialog_pos = false;
  for (auto message_id : message_ids) {
    auto m = get_message(d, message_id);
    CHECK(m != nullptr);
    CHECK(m->sender_user_id == user_id);
    CHECK(m->message_id == message_id);
    if (can_delete_channel_message(channel_status, m, is_bot)) {
      auto p = delete_message(d, message_id, true, &need_update_dialog_pos, "delete messages from user");
      CHECK(p.get() == m);
      deleted_message_ids.push_back(p->message_id.get());
    }
  }

  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "delete_messages_from_user");
  }

  send_update_delete_messages(dialog_id, std::move(deleted_message_ids), true, false);

  delete_all_channel_messages_from_user_on_server(channel_id, user_id, 0, std::move(promise));
}

class MessagesManager::DeleteAllChannelMessagesFromUserOnServerLogEvent {
 public:
  ChannelId channel_id_;
  UserId user_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(channel_id_, storer);
    td::store(user_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(channel_id_, parser);
    td::parse(user_id_, parser);
  }
};

uint64 MessagesManager::save_delete_all_channel_messages_from_user_on_server_log_event(ChannelId channel_id,
                                                                                       UserId user_id) {
  DeleteAllChannelMessagesFromUserOnServerLogEvent log_event{channel_id, user_id};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::DeleteAllChannelMessagesFromUserOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::delete_all_channel_messages_from_user_on_server(ChannelId channel_id, UserId user_id,
                                                                      uint64 log_event_id, Promise<Unit> &&promise) {
  if (log_event_id == 0 && G()->parameters().use_chat_info_db) {
    log_event_id = save_delete_all_channel_messages_from_user_on_server_log_event(channel_id, user_id);
  }

  td_->create_handler<DeleteUserHistoryQuery>(get_erase_log_event_promise(log_event_id, std::move(promise)))
      ->send(channel_id, user_id);
}

void MessagesManager::delete_dialog_messages_by_date(DialogId dialog_id, int32 min_date, int32 max_date, bool revoke,
                                                     Promise<Unit> &&promise) {
  bool is_bot = td_->auth_manager_->is_bot();
  CHECK(!is_bot);

  Dialog *d = get_dialog_force(dialog_id, "delete_dialog_messages_by_date");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return promise.set_error(Status::Error(400, "Can't access the chat"));
  }

  if (min_date > max_date) {
    return promise.set_error(Status::Error(400, "Wrong date interval specified"));
  }

  const int32 telegram_launch_date = 1376438400;
  if (max_date < telegram_launch_date) {
    return promise.set_value(Unit());
  }
  if (min_date < telegram_launch_date) {
    min_date = telegram_launch_date;
  }

  auto current_date = max(G()->unix_time(), 1635000000);
  if (min_date >= current_date - 30) {
    return promise.set_value(Unit());
  }
  if (max_date >= current_date - 30) {
    max_date = current_date - 31;
  }
  CHECK(min_date <= max_date);

  switch (dialog_id.get_type()) {
    case DialogType::User:
      break;
    case DialogType::Chat:
      if (revoke) {
        return promise.set_error(Status::Error(400, "Bulk message revocation is unsupported in basic group chats"));
      }
      break;
    case DialogType::Channel:
      return promise.set_error(Status::Error(400, "Bulk message deletion is unsupported in supergroup chats"));
    case DialogType::SecretChat:
      return promise.set_error(Status::Error(400, "Bulk message deletion is unsupported in secret chats"));
    case DialogType::None:
    default:
      UNREACHABLE();
      break;
  }

  // TODO delete in database by dates

  vector<MessageId> message_ids;
  find_messages_by_date(d->messages.get(), min_date, max_date, message_ids);

  bool need_update_dialog_pos = false;
  vector<int64> deleted_message_ids;
  for (auto message_id : message_ids) {
    auto m = delete_message(d, message_id, true, &need_update_dialog_pos, DELETE_MESSAGE_USER_REQUEST_SOURCE);
    CHECK(m != nullptr);
    deleted_message_ids.push_back(m->message_id.get());
  }

  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "delete_dialog_messages_by_date");
  }
  send_update_delete_messages(dialog_id, std::move(deleted_message_ids), true, false);

  delete_dialog_messages_by_date_on_server(dialog_id, min_date, max_date, revoke, 0, std::move(promise));
}

class MessagesManager::DeleteDialogMessagesByDateOnServerLogEvent {
 public:
  DialogId dialog_id_;
  int32 min_date_;
  int32 max_date_;
  bool revoke_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(revoke_);
    END_STORE_FLAGS();
    td::store(dialog_id_, storer);
    td::store(min_date_, storer);
    td::store(max_date_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(revoke_);
    END_PARSE_FLAGS();
    td::parse(dialog_id_, parser);
    td::parse(min_date_, parser);
    td::parse(max_date_, parser);
  }
};

uint64 MessagesManager::save_delete_dialog_messages_by_date_on_server_log_event(DialogId dialog_id, int32 min_date,
                                                                                int32 max_date, bool revoke) {
  DeleteDialogMessagesByDateOnServerLogEvent log_event{dialog_id, min_date, max_date, revoke};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::DeleteDialogMessagesByDateOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::delete_dialog_messages_by_date_on_server(DialogId dialog_id, int32 min_date, int32 max_date,
                                                               bool revoke, uint64 log_event_id,
                                                               Promise<Unit> &&promise) {
  if (log_event_id == 0 && G()->parameters().use_chat_info_db) {
    log_event_id = save_delete_dialog_messages_by_date_on_server_log_event(dialog_id, min_date, max_date, revoke);
  }

  td_->create_handler<DeleteMessagesByDateQuery>(get_erase_log_event_promise(log_event_id, std::move(promise)))
      ->send(dialog_id, min_date, max_date, revoke);
}

int32 MessagesManager::get_unload_dialog_delay() const {
  constexpr int32 DIALOG_UNLOAD_DELAY = 60;        // seconds
  constexpr int32 DIALOG_UNLOAD_BOT_DELAY = 1800;  // seconds

  CHECK(is_message_unload_enabled());
  auto default_unload_delay = td_->auth_manager_->is_bot() ? DIALOG_UNLOAD_BOT_DELAY : DIALOG_UNLOAD_DELAY;
  return narrow_cast<int32>(G()->shared_config().get_option_integer("message_unload_delay", default_unload_delay));
}

void MessagesManager::unload_dialog(DialogId dialog_id) {
  if (G()->close_flag()) {
    return;
  }

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  if (!d->has_unload_timeout) {
    LOG(INFO) << "Don't need to unload " << dialog_id;
    // possible right after the dialog was opened
    return;
  }

  if (!is_message_unload_enabled()) {
    // just in case
    LOG(INFO) << "Message unload is disabled in " << dialog_id;
    d->has_unload_timeout = false;
    return;
  }

  vector<MessageId> to_unload_message_ids;
  int32 left_to_unload = 0;
  find_unloadable_messages(d, G()->unix_time_cached() - get_unload_dialog_delay() + 2, d->messages.get(),
                           to_unload_message_ids, left_to_unload);

  vector<int64> unloaded_message_ids;
  for (auto message_id : to_unload_message_ids) {
    unload_message(d, message_id);
    unloaded_message_ids.push_back(message_id.get());
  }

  if (!unloaded_message_ids.empty()) {
    if (!G()->parameters().use_message_db && !d->is_empty) {
      d->have_full_history = false;
    }

    send_closure_later(
        G()->td(), &Td::send_update,
        make_tl_object<td_api::updateDeleteMessages>(dialog_id.get(), std::move(unloaded_message_ids), false, true));
  }

  if (left_to_unload > 0) {
    LOG(DEBUG) << "Need to unload " << left_to_unload << " messages more in " << dialog_id;
    pending_unload_dialog_timeout_.add_timeout_in(d->dialog_id.get(), get_unload_dialog_delay());
  } else {
    d->has_unload_timeout = false;
  }
}

void MessagesManager::delete_all_dialog_messages(Dialog *d, bool remove_from_dialog_list, bool is_permanently_deleted) {
  CHECK(d != nullptr);
  LOG(INFO) << "Delete all messages in " << d->dialog_id
            << " with remove_from_dialog_list = " << remove_from_dialog_list
            << " and is_permanently_deleted = " << is_permanently_deleted;
  if (is_debug_message_op_enabled()) {
    d->debug_message_op.emplace_back(Dialog::MessageOp::DeleteAll, MessageId(), MessageContentType::None,
                                     remove_from_dialog_list, false, false, "");
  }

  if (d->server_unread_count + d->local_unread_count > 0) {
    MessageId max_message_id =
        d->last_database_message_id.is_valid() ? d->last_database_message_id : d->last_new_message_id;
    if (max_message_id.is_valid()) {
      read_history_inbox(d->dialog_id, max_message_id, -1, "delete_all_dialog_messages 1");
    }
    if (d->server_unread_count != 0 || d->local_unread_count != 0) {
      set_dialog_last_read_inbox_message_id(d, MessageId::min(), 0, 0, true, "delete_all_dialog_messages 2");
    }
  }

  if (d->unread_mention_count > 0) {
    set_dialog_unread_mention_count(d, 0);
    send_update_chat_unread_mention_count(d);
  }

  bool has_last_message_id = d->last_message_id != MessageId();
  int32 last_message_date = 0;
  MessageId last_clear_history_message_id;
  if (!remove_from_dialog_list) {
    if (has_last_message_id) {
      auto m = get_message(d, d->last_message_id);
      CHECK(m != nullptr);
      last_message_date = m->date;
      last_clear_history_message_id = d->last_message_id;
    } else {
      last_message_date = d->last_clear_history_date;
      last_clear_history_message_id = d->last_clear_history_message_id;
    }
  }

  vector<int64> deleted_message_ids;
  do_delete_all_dialog_messages(d, d->messages, is_permanently_deleted, deleted_message_ids);
  delete_all_dialog_messages_from_database(d, MessageId::max(), "delete_all_dialog_messages 3");
  if (is_permanently_deleted) {
    for (auto id : deleted_message_ids) {
      d->deleted_message_ids.insert(MessageId{id});
    }
  }

  if (d->reply_markup_message_id != MessageId()) {
    set_dialog_reply_markup(d, MessageId());
  }

  set_dialog_first_database_message_id(d, MessageId(), "delete_all_dialog_messages 4");
  set_dialog_last_database_message_id(d, MessageId(), "delete_all_dialog_messages 5");
  set_dialog_last_clear_history_date(d, last_message_date, last_clear_history_message_id,
                                     "delete_all_dialog_messages 6");
  d->last_read_all_mentions_message_id = MessageId();                            // it is not needed anymore
  d->message_notification_group.max_removed_notification_id = NotificationId();  // it is not needed anymore
  d->message_notification_group.max_removed_message_id = MessageId();            // it is not needed anymore
  d->mention_notification_group.max_removed_notification_id = NotificationId();  // it is not needed anymore
  d->mention_notification_group.max_removed_message_id = MessageId();            // it is not needed anymore
  std::fill(d->message_count_by_index.begin(), d->message_count_by_index.end(), 0);
  d->notification_id_to_message_id.clear();

  if (has_last_message_id) {
    set_dialog_last_message_id(d, MessageId(), "delete_all_dialog_messages 7");
    send_update_chat_last_message(d, "delete_all_dialog_messages 8");
  }
  if (remove_from_dialog_list) {
    set_dialog_order(d, DEFAULT_ORDER, true, false, "delete_all_dialog_messages 9");
  } else {
    update_dialog_pos(d, "delete_all_dialog_messages 10");
  }

  on_dialog_updated(d->dialog_id, "delete_all_dialog_messages 11");

  send_update_delete_messages(d->dialog_id, std::move(deleted_message_ids), is_permanently_deleted, false);
}

void MessagesManager::on_dialog_deleted(DialogId dialog_id, Promise<Unit> &&promise) {
  LOG(INFO) << "Delete " << dialog_id;
  Dialog *d = get_dialog_force(dialog_id, "on_dialog_deleted");
  if (d == nullptr) {
    return promise.set_value(Unit());
  }

  delete_all_dialog_messages(d, true, false);
  if (dialog_id.get_type() != DialogType::SecretChat) {
    d->have_full_history = false;
    d->is_empty = false;
    d->need_restore_reply_markup = true;
  }
  recently_found_dialogs_.remove_dialog(dialog_id);
  recently_opened_dialogs_.remove_dialog(dialog_id);
  if (dialog_id.get_type() == DialogType::Channel) {
    G()->td_db()->get_binlog_pmc()->erase(get_channel_pts_key(dialog_id));
  }

  close_dialog(d);
  promise.set_value(Unit());
}

void MessagesManager::on_update_dialog_group_call_rights(DialogId dialog_id) {
  auto d = get_dialog(dialog_id);
  if (d == nullptr) {
    // nothing to do
    return;
  }

  if (d->active_group_call_id.is_valid()) {
    td_->group_call_manager_->on_update_group_call_rights(d->active_group_call_id);
  }
}

void MessagesManager::read_all_dialog_mentions(DialogId dialog_id, Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(dialog_id, "read_all_dialog_mentions");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }

  LOG(INFO) << "Receive readAllChatMentions request in " << dialog_id << " with " << d->unread_mention_count
            << " unread mentions";
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return promise.set_error(Status::Error(400, "Chat is not accessible"));
  }
  if (dialog_id.get_type() == DialogType::SecretChat) {
    CHECK(d->unread_mention_count == 0);
    return promise.set_value(Unit());
  }

  if (d->last_new_message_id > d->last_read_all_mentions_message_id) {
    d->last_read_all_mentions_message_id = d->last_new_message_id;
    on_dialog_updated(dialog_id, "read_all_dialog_mentions");
  }

  vector<MessageId> message_ids;
  find_messages(d->messages.get(), message_ids, [](const Message *m) { return m->contains_unread_mention; });

  LOG(INFO) << "Found " << message_ids.size() << " messages with unread mentions in memory";
  bool is_update_sent = false;
  for (auto message_id : message_ids) {
    auto m = get_message(d, message_id);
    CHECK(m != nullptr);
    CHECK(m->contains_unread_mention);
    CHECK(m->message_id == message_id);
    CHECK(m->message_id.is_valid());
    remove_message_notification_id(d, m, true, false);  // should be called before contains_unread_mention is updated
    m->contains_unread_mention = false;

    send_closure(G()->td(), &Td::send_update,
                 make_tl_object<td_api::updateMessageMentionRead>(dialog_id.get(), m->message_id.get(), 0));
    is_update_sent = true;
    on_message_changed(d, m, true, "read_all_dialog_mentions");
  }

  if (d->unread_mention_count != 0) {
    set_dialog_unread_mention_count(d, 0);
    if (!is_update_sent) {
      send_update_chat_unread_mention_count(d);
    } else {
      LOG(INFO) << "Update unread mention message count in " << dialog_id << " to " << d->unread_mention_count;
      on_dialog_updated(dialog_id, "read_all_dialog_mentions");
    }
  }
  remove_message_dialog_notifications(d, MessageId::max(), true, "read_all_dialog_mentions");

  read_all_dialog_mentions_on_server(dialog_id, 0, std::move(promise));
}

class MessagesManager::ReadAllDialogMentionsOnServerLogEvent {
 public:
  DialogId dialog_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
  }
};

uint64 MessagesManager::save_read_all_dialog_mentions_on_server_log_event(DialogId dialog_id) {
  ReadAllDialogMentionsOnServerLogEvent log_event{dialog_id};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ReadAllDialogMentionsOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::read_all_dialog_mentions_on_server(DialogId dialog_id, uint64 log_event_id,
                                                         Promise<Unit> &&promise) {
  if (log_event_id == 0 && G()->parameters().use_message_db) {
    log_event_id = save_read_all_dialog_mentions_on_server_log_event(dialog_id);
  }

  LOG(INFO) << "Read all mentions on server in " << dialog_id;
  td_->create_handler<ReadMentionsQuery>(get_erase_log_event_promise(log_event_id, std::move(promise)))
      ->send(dialog_id);
}

void MessagesManager::read_message_content_from_updates(MessageId message_id) {
  if (!message_id.is_valid() || !message_id.is_server()) {
    LOG(ERROR) << "Incoming update tries to read content of " << message_id;
    return;
  }

  Dialog *d = get_dialog_by_message_id(message_id);
  if (d != nullptr) {
    Message *m = get_message(d, message_id);
    CHECK(m != nullptr);
    read_message_content(d, m, false, "read_message_content_from_updates");
  }
}

void MessagesManager::read_channel_message_content_from_updates(Dialog *d, MessageId message_id) {
  CHECK(d != nullptr);
  if (!message_id.is_valid() || !message_id.is_server()) {
    LOG(ERROR) << "Incoming update tries to read content of " << message_id << " in " << d->dialog_id;
    return;
  }

  Message *m = get_message_force(d, message_id, "read_channel_message_content_from_updates");
  if (m != nullptr) {
    read_message_content(d, m, false, "read_channel_message_content_from_updates");
  } else if (message_id > d->last_new_message_id) {
    get_channel_difference(d->dialog_id, d->pts, true, "read_channel_message_content_from_updates");
  }
}

bool MessagesManager::read_message_content(Dialog *d, Message *m, bool is_local_read, const char *source) {
  LOG_CHECK(m != nullptr) << source;
  CHECK(!m->message_id.is_scheduled());
  bool is_mention_read = update_message_contains_unread_mention(d, m, false, "read_message_content");
  bool is_content_read =
      update_opened_message_content(m->content.get()) | ttl_on_open(d, m, Time::now(), is_local_read);

  LOG(INFO) << "Read message content of " << m->message_id << " in " << d->dialog_id
            << ": is_mention_read = " << is_mention_read << ", is_content_read = " << is_content_read;
  if (is_mention_read || is_content_read) {
    on_message_changed(d, m, true, "read_message_content");
    if (is_content_read) {
      send_closure(G()->td(), &Td::send_update,
                   make_tl_object<td_api::updateMessageContentOpened>(d->dialog_id.get(), m->message_id.get()));
    }
    return true;
  }
  return false;
}

bool MessagesManager::has_incoming_notification(DialogId dialog_id, const Message *m) const {
  if (m->is_from_scheduled) {
    return true;
  }
  return !m->is_outgoing && dialog_id != get_my_dialog_id();
}

int32 MessagesManager::calc_new_unread_count_from_last_unread(Dialog *d, MessageId max_message_id,
                                                              MessageType type) const {
  CHECK(!max_message_id.is_scheduled());
  MessagesConstIterator it(d, max_message_id);
  if (*it == nullptr || (*it)->message_id != max_message_id) {
    return -1;
  }

  int32 unread_count = type == MessageType::Server ? d->server_unread_count : d->local_unread_count;
  while (*it != nullptr && (*it)->message_id > d->last_read_inbox_message_id) {
    if (has_incoming_notification(d->dialog_id, *it) && (*it)->message_id.get_type() == type) {
      unread_count--;
    }
    --it;
  }
  if (*it == nullptr || (*it)->message_id != d->last_read_inbox_message_id) {
    return -1;
  }

  LOG(INFO) << "Found " << unread_count << " unread messages in " << d->dialog_id << " from last unread message";
  return unread_count;
}

int32 MessagesManager::calc_new_unread_count_from_the_end(Dialog *d, MessageId max_message_id, MessageType type,
                                                          int32 hint_unread_count) const {
  CHECK(!max_message_id.is_scheduled());
  int32 unread_count = 0;
  MessagesConstIterator it(d, MessageId::max());
  while (*it != nullptr && (*it)->message_id > max_message_id) {
    if (has_incoming_notification(d->dialog_id, *it) && (*it)->message_id.get_type() == type) {
      unread_count++;
    }
    --it;
  }

  bool is_count_exact = d->last_message_id.is_valid() && *it != nullptr;
  if (hint_unread_count >= 0) {
    if (is_count_exact) {
      if (hint_unread_count == unread_count) {
        return hint_unread_count;
      }
    } else {
      if (hint_unread_count >= unread_count) {
        return hint_unread_count;
      }
    }

    // hint_unread_count is definitely wrong, ignore it

    if (need_unread_counter(d->order)) {
      LOG(ERROR) << "Receive hint_unread_count = " << hint_unread_count << ", but found " << unread_count
                 << " unread messages in " << d->dialog_id;
    }
  }

  if (!is_count_exact) {
    // unread count is likely to be calculated wrong, so ignore it
    return -1;
  }

  LOG(INFO) << "Found " << unread_count << " unread messages in " << d->dialog_id << " from the end";
  return unread_count;
}

int32 MessagesManager::calc_new_unread_count(Dialog *d, MessageId max_message_id, MessageType type,
                                             int32 hint_unread_count) const {
  CHECK(!max_message_id.is_scheduled());
  if (d->is_empty) {
    return 0;
  }

  if (!d->last_read_inbox_message_id.is_valid()) {
    return calc_new_unread_count_from_the_end(d, max_message_id, type, hint_unread_count);
  }

  if (!d->last_message_id.is_valid() ||
      (d->last_message_id.get() - max_message_id.get() > max_message_id.get() - d->last_read_inbox_message_id.get())) {
    int32 unread_count = calc_new_unread_count_from_last_unread(d, max_message_id, type);
    return unread_count >= 0 ? unread_count
                             : calc_new_unread_count_from_the_end(d, max_message_id, type, hint_unread_count);
  } else {
    int32 unread_count = calc_new_unread_count_from_the_end(d, max_message_id, type, hint_unread_count);
    return unread_count >= 0 ? unread_count : calc_new_unread_count_from_last_unread(d, max_message_id, type);
  }
}

void MessagesManager::repair_server_unread_count(DialogId dialog_id, int32 unread_count) {
  if (td_->auth_manager_->is_bot() || !have_input_peer(dialog_id, AccessRights::Read)) {
    return;
  }
  if (pending_read_history_timeout_.has_timeout(dialog_id.get())) {
    return;  // postpone until read history request is sent
  }

  LOG(INFO) << "Repair server unread count in " << dialog_id << " from " << unread_count;
  create_actor<SleepActor>("RepairServerUnreadCountSleepActor", 0.2,
                           PromiseCreator::lambda([actor_id = actor_id(this), dialog_id](Result<Unit> result) {
                             send_closure(actor_id, &MessagesManager::send_get_dialog_query, dialog_id, Promise<Unit>(),
                                          0, "repair_server_unread_count");
                           }))
      .release();
}

void MessagesManager::repair_channel_server_unread_count(Dialog *d) {
  CHECK(d != nullptr);
  CHECK(d->dialog_id.get_type() == DialogType::Channel);

  if (td_->auth_manager_->is_bot()) {
    return;
  }
  if (d->last_read_inbox_message_id >= d->last_new_message_id) {
    // all messages are already read
    return;
  }
  if (!need_unread_counter(d->order)) {
    // there are no unread counters in left channels
    return;
  }
  if (!d->need_repair_channel_server_unread_count) {
    d->need_repair_channel_server_unread_count = true;
    on_dialog_updated(d->dialog_id, "repair_channel_server_unread_count");
  }

  LOG(INFO) << "Reload ChannelFull for " << d->dialog_id << " to repair unread message counts";
  get_dialog_info_full(d->dialog_id, Auto(), "repair_channel_server_unread_count");
}

void MessagesManager::read_history_inbox(DialogId dialog_id, MessageId max_message_id, int32 unread_count,
                                         const char *source) {
  CHECK(!max_message_id.is_scheduled());

  if (td_->auth_manager_->is_bot()) {
    return;
  }

  Dialog *d = get_dialog_force(dialog_id, "read_history_inbox");
  if (d != nullptr) {
    if (d->need_repair_channel_server_unread_count) {
      d->need_repair_channel_server_unread_count = false;
      on_dialog_updated(dialog_id, "read_history_inbox");
    }

    // there can be updateReadHistoryInbox up to message 0, if messages where read and then all messages where deleted
    if (!max_message_id.is_valid() && max_message_id != MessageId()) {
      LOG(ERROR) << "Receive read inbox update in " << dialog_id << " up to " << max_message_id << " from " << source;
      return;
    }
    if (d->is_last_read_inbox_message_id_inited && max_message_id <= d->last_read_inbox_message_id) {
      LOG(INFO) << "Receive read inbox update in " << dialog_id << " up to " << max_message_id << " from " << source
                << ", but all messages have already been read up to " << d->last_read_inbox_message_id;
      if (max_message_id == d->last_read_inbox_message_id && unread_count >= 0 &&
          unread_count != d->server_unread_count) {
        set_dialog_last_read_inbox_message_id(d, MessageId::min(), unread_count, d->local_unread_count, true, source);
      }
      return;
    }

    if (max_message_id != MessageId() && max_message_id.is_yet_unsent()) {
      LOG(ERROR) << "Tried to update last read inbox message in " << dialog_id << " with " << max_message_id << " from "
                 << source;
      return;
    }

    if (max_message_id != MessageId() && unread_count > 0 && max_message_id >= d->last_new_message_id &&
        max_message_id >= d->last_message_id && max_message_id >= d->last_database_message_id) {
      if (d->last_new_message_id.is_valid()) {
        LOG(ERROR) << "Have unknown " << unread_count << " unread messages up to " << max_message_id << " in "
                   << dialog_id << " with last_new_message_id = " << d->last_new_message_id
                   << ", last_message_id = " << d->last_message_id
                   << ", last_database_message_id = " << d->last_database_message_id << " from " << source;
      }
      unread_count = 0;
    }

    LOG_IF(INFO, d->last_new_message_id.is_valid() && max_message_id > d->last_new_message_id &&
                     max_message_id > d->max_notification_message_id && max_message_id.is_server() &&
                     dialog_id.get_type() != DialogType::Channel && !running_get_difference_)
        << "Receive read inbox update up to unknown " << max_message_id << " in " << dialog_id << " from " << source
        << ". Last new is " << d->last_new_message_id << ", unread_count = " << unread_count
        << ". Possible only for deleted incoming message";

    if (dialog_id.get_type() == DialogType::SecretChat) {
      ttl_read_history(d, false, max_message_id, d->last_read_inbox_message_id, Time::now());
    }

    if (max_message_id > d->last_new_message_id && dialog_id.get_type() == DialogType::Channel) {
      LOG(INFO) << "Schedule getDifference in " << dialog_id.get_channel_id();
      channel_get_difference_retry_timeout_.add_timeout_in(dialog_id.get(), 0.001);
    }

    int32 server_unread_count = calc_new_unread_count(d, max_message_id, MessageType::Server, unread_count);
    int32 local_unread_count =
        d->local_unread_count == 0 ? 0 : calc_new_unread_count(d, max_message_id, MessageType::Local, -1);

    if (server_unread_count < 0) {
      server_unread_count = unread_count >= 0 ? unread_count : d->server_unread_count;
      if (dialog_id.get_type() != DialogType::SecretChat && have_input_peer(dialog_id, AccessRights::Read) &&
          need_unread_counter(d->order)) {
        d->need_repair_server_unread_count = true;
        repair_server_unread_count(dialog_id, server_unread_count);
      }
    }
    if (local_unread_count < 0) {
      // TODO repair local unread count
      local_unread_count = d->local_unread_count;
    }

    set_dialog_last_read_inbox_message_id(d, max_message_id, server_unread_count, local_unread_count, true, source);

    if (d->is_marked_as_unread && max_message_id != MessageId()) {
      set_dialog_is_marked_as_unread(d, false);
    }
  } else {
    LOG(INFO) << "Receive read inbox about unknown " << dialog_id << " from " << source;
  }
}

void MessagesManager::read_history_outbox(DialogId dialog_id, MessageId max_message_id, int32 read_date) {
  CHECK(!max_message_id.is_scheduled());

  if (td_->auth_manager_->is_bot()) {
    return;
  }

  Dialog *d = get_dialog_force(dialog_id, "read_history_outbox");
  if (d != nullptr) {
    if (!max_message_id.is_valid()) {
      LOG(ERROR) << "Receive read outbox update in " << dialog_id << " with " << max_message_id;
      return;
    }
    if (max_message_id <= d->last_read_outbox_message_id) {
      LOG(INFO) << "Receive read outbox update up to " << max_message_id
                << ", but all messages have already been read up to " << d->last_read_outbox_message_id;
      return;
    }

    if (max_message_id.is_yet_unsent()) {
      LOG(ERROR) << "Tried to update last read outbox message with " << max_message_id;
      return;
    }

    // it is impossible for just sent outgoing messages because updates are ordered by pts
    LOG_IF(INFO, d->last_new_message_id.is_valid() && max_message_id > d->last_new_message_id &&
                     dialog_id.get_type() != DialogType::Channel)
        << "Receive read outbox update about unknown " << max_message_id << " in " << dialog_id << " with last new "
        << d->last_new_message_id << ". Possible only for deleted outgoing message";

    if (dialog_id.get_type() == DialogType::SecretChat) {
      double server_time = Time::now();
      double read_time = server_time;
      if (read_date <= 0) {
        LOG(ERROR) << "Receive wrong read date " << read_date << " in " << dialog_id;
      } else if (read_date < server_time) {
        read_time = read_date;
      }
      ttl_read_history(d, true, max_message_id, d->last_read_outbox_message_id, read_time);
    }

    set_dialog_last_read_outbox_message_id(d, max_message_id);
  } else {
    LOG(INFO) << "Receive read outbox update about unknown " << dialog_id;
  }
}

bool MessagesManager::need_unread_counter(int64 dialog_order) {
  return dialog_order != DEFAULT_ORDER;
}

int32 MessagesManager::get_dialog_total_count(const DialogList &list) const {
  int32 sponsored_dialog_count = 0;
  if (sponsored_dialog_id_.is_valid() && list.dialog_list_id == DialogListId(FolderId::main())) {
    auto d = get_dialog(sponsored_dialog_id_);
    CHECK(d != nullptr);
    if (is_dialog_sponsored(d)) {
      sponsored_dialog_count = 1;
    }
  }
  if (list.server_dialog_total_count_ != -1 && list.secret_chat_total_count_ != -1) {
    return std::max(list.server_dialog_total_count_ + list.secret_chat_total_count_,
                    list.in_memory_dialog_total_count_) +
           sponsored_dialog_count;
  }
  if (list.list_last_dialog_date_ == MAX_DIALOG_DATE) {
    return list.in_memory_dialog_total_count_ + sponsored_dialog_count;
  }
  return list.in_memory_dialog_total_count_ + sponsored_dialog_count + 1;
}

void MessagesManager::repair_server_dialog_total_count(DialogListId dialog_list_id) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }
  if (!dialog_list_id.is_folder()) {
    // can repair total count only in folders
    return;
  }

  LOG(INFO) << "Repair total chat count in " << dialog_list_id;
  send_closure(td_->create_net_actor<GetDialogListActor>(Promise<Unit>()), &GetDialogListActor::send,
               dialog_list_id.get_folder_id(), 2147483647, ServerMessageId(), DialogId(), 1,
               get_sequence_dispatcher_id(DialogId(), MessageContentType::None));
}

void MessagesManager::repair_secret_chat_total_count(DialogListId dialog_list_id) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  if (G()->parameters().use_message_db && dialog_list_id.is_folder()) {
    // race-prone
    G()->td_db()->get_dialog_db_async()->get_secret_chat_count(
        dialog_list_id.get_folder_id(),
        PromiseCreator::lambda([actor_id = actor_id(this), dialog_list_id](Result<int32> result) {
          if (result.is_error()) {
            return;
          }
          send_closure(actor_id, &MessagesManager::on_get_secret_chat_total_count, dialog_list_id, result.move_as_ok());
        }));
  } else {
    int32 total_count = 0;
    auto *list = get_dialog_list(dialog_list_id);
    CHECK(list != nullptr);
    for (auto &folder_id : get_dialog_list_folder_ids(*list)) {
      const auto *folder_list = get_dialog_list(DialogListId(folder_id));
      CHECK(folder_list != nullptr);
      if (folder_list->need_unread_count_recalc_) {
        // can't repair total secret chat count yet
        return;
      }

      const auto *folder = get_dialog_folder(folder_id);
      CHECK(folder != nullptr);
      for (const auto &dialog_date : folder->ordered_dialogs_) {
        auto dialog_id = dialog_date.get_dialog_id();
        if (dialog_id.get_type() == DialogType::SecretChat && dialog_date.get_order() != DEFAULT_ORDER) {
          total_count++;
        }
      }
    }
    on_get_secret_chat_total_count(dialog_list_id, total_count);
  }
}

void MessagesManager::on_get_secret_chat_total_count(DialogListId dialog_list_id, int32 total_count) {
  if (G()->close_flag()) {
    return;
  }

  CHECK(!td_->auth_manager_->is_bot());
  auto *list = get_dialog_list(dialog_list_id);
  if (list == nullptr) {
    // just in case
    return;
  }
  CHECK(total_count >= 0);
  if (list->secret_chat_total_count_ != total_count) {
    auto old_dialog_total_count = get_dialog_total_count(*list);
    list->secret_chat_total_count_ = total_count;
    if (list->is_dialog_unread_count_inited_) {
      if (old_dialog_total_count != get_dialog_total_count(*list)) {
        send_update_unread_chat_count(*list, DialogId(), true, "on_get_secret_chat_total_count");
      } else {
        save_unread_chat_count(*list);
      }
    }
  }
}

void MessagesManager::recalc_unread_count(DialogListId dialog_list_id, int32 old_dialog_total_count, bool force) {
  if (G()->close_flag() || td_->auth_manager_->is_bot() || !G()->parameters().use_message_db) {
    return;
  }

  auto *list_ptr = get_dialog_list(dialog_list_id);
  CHECK(list_ptr != nullptr);
  auto &list = *list_ptr;
  if (!list.need_unread_count_recalc_ && !force) {
    return;
  }
  LOG(INFO) << "Recalculate unread counts in " << dialog_list_id;
  list.need_unread_count_recalc_ = false;
  list.is_message_unread_count_inited_ = true;
  list.is_dialog_unread_count_inited_ = true;

  int32 message_total_count = 0;
  int32 message_muted_count = 0;
  int32 dialog_total_count = 0;
  int32 dialog_muted_count = 0;
  int32 dialog_marked_count = 0;
  int32 dialog_muted_marked_count = 0;
  int32 server_dialog_total_count = 0;
  int32 secret_chat_total_count = 0;
  for (auto folder_id : get_dialog_list_folder_ids(list)) {
    const auto &folder = *get_dialog_folder(folder_id);
    for (const auto &dialog_date : folder.ordered_dialogs_) {
      if (dialog_date.get_order() == DEFAULT_ORDER) {
        break;
      }

      auto dialog_id = dialog_date.get_dialog_id();
      Dialog *d = get_dialog(dialog_id);
      CHECK(d != nullptr);
      if (!is_dialog_in_list(d, dialog_list_id)) {
        continue;
      }

      int unread_count = d->server_unread_count + d->local_unread_count;
      if (need_unread_counter(d->order) && (unread_count > 0 || d->is_marked_as_unread)) {
        message_total_count += unread_count;
        dialog_total_count++;
        if (unread_count == 0 && d->is_marked_as_unread) {
          dialog_marked_count++;
        }

        LOG(DEBUG) << "Have " << unread_count << " messages in " << dialog_id;
        if (is_dialog_muted(d)) {
          message_muted_count += unread_count;
          dialog_muted_count++;
          if (unread_count == 0 && d->is_marked_as_unread) {
            dialog_muted_marked_count++;
          }
        }
      }
      if (d->order != DEFAULT_ORDER) {  // must not count sponsored dialog, which is added independently
        if (dialog_id.get_type() == DialogType::SecretChat) {
          secret_chat_total_count++;
        } else {
          server_dialog_total_count++;
        }
      }
    }
  }

  if (list.unread_message_total_count_ != message_total_count ||
      list.unread_message_muted_count_ != message_muted_count) {
    list.unread_message_total_count_ = message_total_count;
    list.unread_message_muted_count_ = message_muted_count;
    send_update_unread_message_count(list, DialogId(), true, "recalc_unread_count");
  }

  if (old_dialog_total_count == -1) {
    old_dialog_total_count = get_dialog_total_count(list);
  }
  bool need_save = false;
  if (list.list_last_dialog_date_ == MAX_DIALOG_DATE) {
    if (server_dialog_total_count != list.server_dialog_total_count_ ||
        secret_chat_total_count != list.secret_chat_total_count_) {
      list.server_dialog_total_count_ = server_dialog_total_count;
      list.secret_chat_total_count_ = secret_chat_total_count;
      need_save = true;
    }
  } else {
    if (list.server_dialog_total_count_ == -1) {
      // recalc_unread_count is called only after getDialogs request; it is unneeded to call getDialogs again
      repair_server_dialog_total_count(dialog_list_id);
    }

    if (list.secret_chat_total_count_ == -1) {
      repair_secret_chat_total_count(dialog_list_id);
    }
  }
  if (list.unread_dialog_total_count_ != dialog_total_count || list.unread_dialog_muted_count_ != dialog_muted_count ||
      list.unread_dialog_marked_count_ != dialog_marked_count ||
      list.unread_dialog_muted_marked_count_ != dialog_muted_marked_count ||
      old_dialog_total_count != get_dialog_total_count(list)) {
    list.unread_dialog_total_count_ = dialog_total_count;
    list.unread_dialog_muted_count_ = dialog_muted_count;
    list.unread_dialog_marked_count_ = dialog_marked_count;
    list.unread_dialog_muted_marked_count_ = dialog_muted_marked_count;
    send_update_unread_chat_count(list, DialogId(), true, "recalc_unread_count");
  } else if (need_save) {
    save_unread_chat_count(list);
  }
}

void MessagesManager::set_dialog_last_read_inbox_message_id(Dialog *d, MessageId message_id, int32 server_unread_count,
                                                            int32 local_unread_count, bool force_update,
                                                            const char *source) {
  CHECK(!message_id.is_scheduled());

  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(d != nullptr);
  LOG(INFO) << "Update last read inbox message in " << d->dialog_id << " from " << d->last_read_inbox_message_id
            << " to " << message_id << " and update unread message count from " << d->server_unread_count << " + "
            << d->local_unread_count << " to " << server_unread_count << " + " << local_unread_count << " from "
            << source;
  if (message_id != MessageId::min()) {
    d->last_read_inbox_message_id = message_id;
    d->is_last_read_inbox_message_id_inited = true;
  }
  int32 old_unread_count = d->server_unread_count + d->local_unread_count;
  d->server_unread_count = server_unread_count;
  d->local_unread_count = local_unread_count;

  if (need_unread_counter(d->order)) {
    const int32 new_unread_count = d->server_unread_count + d->local_unread_count;
    for (auto &list : get_dialog_lists(d)) {
      int32 delta = new_unread_count - old_unread_count;
      if (delta != 0 && list.is_message_unread_count_inited_) {
        list.unread_message_total_count_ += delta;
        if (is_dialog_muted(d)) {
          list.unread_message_muted_count_ += delta;
        }
        send_update_unread_message_count(list, d->dialog_id, force_update, source);
      }
      delta = static_cast<int32>(new_unread_count != 0) - static_cast<int32>(old_unread_count != 0);
      if (delta != 0 && list.is_dialog_unread_count_inited_) {
        if (d->is_marked_as_unread) {
          list.unread_dialog_marked_count_ -= delta;
        } else {
          list.unread_dialog_total_count_ += delta;
        }
        if (is_dialog_muted(d)) {
          if (d->is_marked_as_unread) {
            list.unread_dialog_muted_marked_count_ -= delta;
          } else {
            list.unread_dialog_muted_count_ += delta;
          }
        }
        send_update_unread_chat_count(list, d->dialog_id, force_update, source);
      }
    }

    bool was_unread = old_unread_count != 0 || d->is_marked_as_unread;
    bool is_unread = new_unread_count != 0 || d->is_marked_as_unread;
    if (!dialog_filters_.empty() && was_unread != is_unread) {
      update_dialog_lists(d, get_dialog_positions(d), true, false, "set_dialog_last_read_inbox_message_id");
    }
  }

  if (message_id != MessageId::min() && d->last_read_inbox_message_id.is_valid() &&
      (d->order != DEFAULT_ORDER || is_dialog_sponsored(d))) {
    VLOG(notifications) << "Remove some notifications in " << d->dialog_id
                        << " after updating last read inbox message to " << message_id
                        << " and unread message count to " << server_unread_count << " + " << local_unread_count
                        << " from " << source;
    if (d->message_notification_group.group_id.is_valid()) {
      auto total_count = get_dialog_pending_notification_count(d, false);
      if (total_count == 0) {
        set_dialog_last_notification(d->dialog_id, d->message_notification_group, 0, NotificationId(), source);
      }
      if (!d->pending_new_message_notifications.empty()) {
        for (auto &it : d->pending_new_message_notifications) {
          if (it.second <= message_id) {
            it.first = DialogId();
          }
        }
        flush_pending_new_message_notifications(d->dialog_id, false, DialogId(UserId(static_cast<int64>(1))));
      }
      total_count -= static_cast<int32>(d->pending_new_message_notifications.size());
      if (total_count < 0) {
        LOG(ERROR) << "Total message notification count is " << total_count << " in " << d->dialog_id
                   << " with old unread_count = " << old_unread_count << " and " << d->pending_new_message_notifications
                   << " pending new message notifications after reading history up to " << message_id;
        total_count = 0;
      }
      send_closure_later(G()->notification_manager(), &NotificationManager::remove_notification_group,
                         d->message_notification_group.group_id, NotificationId(), d->last_read_inbox_message_id,
                         total_count, Slice(source) == Slice("view_messages"), Promise<Unit>());
    }

    if (d->mention_notification_group.group_id.is_valid() && d->pinned_message_notification_message_id.is_valid() &&
        d->pinned_message_notification_message_id <= d->last_read_inbox_message_id) {
      // remove pinned message notification when it is read
      remove_dialog_pinned_message_notification(d, source);
    }
  }

  send_update_chat_read_inbox(d, force_update, source);
}

void MessagesManager::set_dialog_last_read_outbox_message_id(Dialog *d, MessageId message_id) {
  CHECK(!message_id.is_scheduled());

  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(d != nullptr);
  LOG(INFO) << "Update last read outbox message in " << d->dialog_id << " from " << d->last_read_outbox_message_id
            << " to " << message_id;
  d->last_read_outbox_message_id = message_id;
  d->is_last_read_outbox_message_id_inited = true;
  send_update_chat_read_outbox(d);
}

void MessagesManager::set_dialog_max_unavailable_message_id(DialogId dialog_id, MessageId max_unavailable_message_id,
                                                            bool from_update, const char *source) {
  CHECK(!max_unavailable_message_id.is_scheduled());

  Dialog *d = get_dialog_force(dialog_id, source);
  if (d != nullptr) {
    if (d->last_new_message_id.is_valid() && max_unavailable_message_id > d->last_new_message_id && from_update) {
      if (!td_->auth_manager_->is_bot()) {
        LOG(ERROR) << "Tried to set " << dialog_id << " max unavailable message to " << max_unavailable_message_id
                   << " from " << source << ", but last new message is " << d->last_new_message_id;
      }
      max_unavailable_message_id = d->last_new_message_id;
    }

    if (d->max_unavailable_message_id == max_unavailable_message_id) {
      return;
    }

    if (max_unavailable_message_id.is_valid() && max_unavailable_message_id.is_yet_unsent()) {
      LOG(ERROR) << "Tried to update " << dialog_id << " last read outbox message with " << max_unavailable_message_id
                 << " from " << source;
      return;
    }
    LOG(INFO) << "Set max unavailable message to " << max_unavailable_message_id << " in " << dialog_id << " from "
              << source;

    on_dialog_updated(dialog_id, "set_dialog_max_unavailable_message_id");

    if (d->max_unavailable_message_id > max_unavailable_message_id) {
      d->max_unavailable_message_id = max_unavailable_message_id;
      return;
    }

    d->max_unavailable_message_id = max_unavailable_message_id;

    vector<MessageId> message_ids;
    find_old_messages(d->messages.get(), max_unavailable_message_id, message_ids);

    vector<int64> deleted_message_ids;
    bool need_update_dialog_pos = false;
    for (auto message_id : message_ids) {
      if (message_id.is_yet_unsent()) {
        continue;
      }

      auto m = get_message(d, message_id);
      CHECK(m != nullptr);
      CHECK(m->message_id <= max_unavailable_message_id);
      CHECK(m->message_id == message_id);
      auto p =
          delete_message(d, message_id, !from_update, &need_update_dialog_pos, "set_dialog_max_unavailable_message_id");
      CHECK(p.get() == m);
      deleted_message_ids.push_back(p->message_id.get());
    }

    if (need_update_dialog_pos) {
      send_update_chat_last_message(d, "set_dialog_max_unavailable_message_id");
    }

    send_update_delete_messages(dialog_id, std::move(deleted_message_ids), !from_update, false);

    if (d->server_unread_count + d->local_unread_count > 0) {
      read_history_inbox(dialog_id, max_unavailable_message_id, -1, "set_dialog_max_unavailable_message_id");
    }
  } else {
    LOG(INFO) << "Receive max unavailable message in unknown " << dialog_id << " from " << source;
  }
}

void MessagesManager::set_dialog_online_member_count(DialogId dialog_id, int32 online_member_count, bool is_from_server,
                                                     const char *source) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  Dialog *d = get_dialog(dialog_id);
  if (d == nullptr) {
    return;
  }

  auto &info = dialog_online_member_counts_[dialog_id];
  LOG(INFO) << "Change number of online members from " << info.online_member_count << " to " << online_member_count
            << " in " << dialog_id << " from " << source;
  bool need_update = d->is_opened && (!info.is_update_sent || info.online_member_count != online_member_count);
  info.online_member_count = online_member_count;
  info.updated_time = Time::now();

  if (need_update) {
    info.is_update_sent = true;
    send_update_chat_online_member_count(dialog_id, online_member_count);
  }
  if (d->is_opened) {
    if (is_from_server) {
      update_dialog_online_member_count_timeout_.set_timeout_in(dialog_id.get(), ONLINE_MEMBER_COUNT_UPDATE_TIME);
    } else {
      update_dialog_online_member_count_timeout_.add_timeout_in(dialog_id.get(), ONLINE_MEMBER_COUNT_UPDATE_TIME);
    }
  }
}

void MessagesManager::on_update_dialog_online_member_count_timeout(DialogId dialog_id) {
  if (G()->close_flag()) {
    return;
  }

  LOG(INFO) << "Expired timeout for number of online members in " << dialog_id;
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  if (!d->is_opened) {
    send_update_chat_online_member_count(dialog_id, 0);
    return;
  }

  if (dialog_id.get_type() == DialogType::Channel && !is_broadcast_channel(dialog_id)) {
    auto participant_count = td_->contacts_manager_->get_channel_participant_count(dialog_id.get_channel_id());
    if (participant_count == 0 || participant_count >= 195) {
      td_->create_handler<GetOnlinesQuery>()->send(dialog_id);
    } else {
      td_->contacts_manager_->get_channel_participants(dialog_id.get_channel_id(),
                                                       td_api::make_object<td_api::supergroupMembersFilterRecent>(),
                                                       string(), 0, 200, 200, Auto());
    }
    return;
  }
  if (dialog_id.get_type() == DialogType::Chat) {
    // we need actual online status state, so we need to reget chat participants
    td_->contacts_manager_->repair_chat_participants(dialog_id.get_chat_id());
    return;
  }
}

MessageId MessagesManager::get_message_id(const tl_object_ptr<telegram_api::Message> &message_ptr, bool is_scheduled) {
  switch (message_ptr->get_id()) {
    case telegram_api::messageEmpty::ID: {
      auto message = static_cast<const telegram_api::messageEmpty *>(message_ptr.get());
      return is_scheduled ? MessageId() : MessageId(ServerMessageId(message->id_));
    }
    case telegram_api::message::ID: {
      auto message = static_cast<const telegram_api::message *>(message_ptr.get());
      return is_scheduled ? MessageId(ScheduledServerMessageId(message->id_), message->date_)
                          : MessageId(ServerMessageId(message->id_));
    }
    case telegram_api::messageService::ID: {
      auto message = static_cast<const telegram_api::messageService *>(message_ptr.get());
      return is_scheduled ? MessageId(ScheduledServerMessageId(message->id_), message->date_)
                          : MessageId(ServerMessageId(message->id_));
    }
    default:
      UNREACHABLE();
      return MessageId();
  }
}

DialogId MessagesManager::get_message_dialog_id(const tl_object_ptr<telegram_api::Message> &message_ptr) {
  CHECK(message_ptr != nullptr);
  switch (message_ptr->get_id()) {
    case telegram_api::messageEmpty::ID: {
      auto message = static_cast<const telegram_api::messageEmpty *>(message_ptr.get());
      return message->peer_id_ == nullptr ? DialogId() : DialogId(message->peer_id_);
    }
    case telegram_api::message::ID: {
      auto message = static_cast<const telegram_api::message *>(message_ptr.get());
      return DialogId(message->peer_id_);
    }
    case telegram_api::messageService::ID: {
      auto message = static_cast<const telegram_api::messageService *>(message_ptr.get());
      return DialogId(message->peer_id_);
    }
    default:
      UNREACHABLE();
      return DialogId();
  }
}

FullMessageId MessagesManager::get_full_message_id(const tl_object_ptr<telegram_api::Message> &message_ptr,
                                                   bool is_scheduled) {
  return {get_message_dialog_id(message_ptr), get_message_id(message_ptr, is_scheduled)};
}

int32 MessagesManager::get_message_date(const tl_object_ptr<telegram_api::Message> &message_ptr) {
  switch (message_ptr->get_id()) {
    case telegram_api::messageEmpty::ID:
      return 0;
    case telegram_api::message::ID: {
      auto message = static_cast<const telegram_api::message *>(message_ptr.get());
      return message->date_;
    }
    case telegram_api::messageService::ID: {
      auto message = static_cast<const telegram_api::messageService *>(message_ptr.get());
      return message->date_;
    }
    default:
      UNREACHABLE();
      return 0;
  }
}

void MessagesManager::ttl_read_history(Dialog *d, bool is_outgoing, MessageId from_message_id,
                                       MessageId till_message_id, double view_date) {
  CHECK(!from_message_id.is_scheduled());
  CHECK(!till_message_id.is_scheduled());

  // TODO: protect with log event
  suffix_load_till_message_id(d, till_message_id,
                              PromiseCreator::lambda([actor_id = actor_id(this), dialog_id = d->dialog_id, is_outgoing,
                                                      from_message_id, till_message_id, view_date](Result<Unit>) {
                                send_closure(actor_id, &MessagesManager::ttl_read_history_impl, dialog_id, is_outgoing,
                                             from_message_id, till_message_id, view_date);
                              }));
}

void MessagesManager::ttl_read_history_impl(DialogId dialog_id, bool is_outgoing, MessageId from_message_id,
                                            MessageId till_message_id, double view_date) {
  CHECK(!from_message_id.is_scheduled());
  CHECK(!till_message_id.is_scheduled());

  auto *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  auto now = Time::now();
  for (auto it = MessagesIterator(d, from_message_id); *it && (*it)->message_id >= till_message_id; --it) {
    auto *m = *it;
    if (m->is_outgoing == is_outgoing) {
      ttl_on_view(d, m, view_date, now);
    }
  }
}

void MessagesManager::ttl_on_view(const Dialog *d, Message *m, double view_date, double now) {
  if (m->ttl > 0 && m->ttl_expires_at == 0 && !m->message_id.is_scheduled() && !m->message_id.is_yet_unsent() &&
      !m->is_failed_to_send && !m->is_content_secret) {
    m->ttl_expires_at = m->ttl + view_date;
    ttl_register_message(d->dialog_id, m, now);
    on_message_changed(d, m, true, "ttl_on_view");
  }
}

bool MessagesManager::ttl_on_open(Dialog *d, Message *m, double now, bool is_local_read) {
  CHECK(!m->message_id.is_scheduled());
  if (m->ttl > 0 && m->ttl_expires_at == 0) {
    if (!is_local_read && d->dialog_id.get_type() != DialogType::SecretChat) {
      on_message_ttl_expired(d, m);
    } else {
      m->ttl_expires_at = m->ttl + now;
      ttl_register_message(d->dialog_id, m, now);
    }
    return true;
  }
  return false;
}

void MessagesManager::ttl_register_message(DialogId dialog_id, const Message *m, double now) {
  CHECK(m != nullptr);
  CHECK(m->ttl_expires_at != 0);
  CHECK(!m->message_id.is_scheduled());

  auto it_flag = ttl_nodes_.emplace(dialog_id, m->message_id, false);
  CHECK(it_flag.second);
  auto it = it_flag.first;

  ttl_heap_.insert(m->ttl_expires_at, it->as_heap_node());
  ttl_update_timeout(now);
}

void MessagesManager::ttl_period_register_message(DialogId dialog_id, const Message *m, double server_time) {
  CHECK(m != nullptr);
  CHECK(m->ttl_period != 0);
  CHECK(!m->message_id.is_scheduled());

  auto it_flag = ttl_nodes_.emplace(dialog_id, m->message_id, true);
  CHECK(it_flag.second);
  auto it = it_flag.first;

  auto now = Time::now();
  ttl_heap_.insert(now + (m->date + m->ttl_period - server_time), it->as_heap_node());
  ttl_update_timeout(now);
}

void MessagesManager::ttl_unregister_message(DialogId dialog_id, const Message *m, const char *source) {
  if (m->ttl_expires_at == 0) {
    return;
  }
  CHECK(!m->message_id.is_scheduled());

  auto it = ttl_nodes_.find(TtlNode(dialog_id, m->message_id, false));

  // expect m->ttl == 0, but m->ttl_expires_at > 0 from binlog
  LOG_CHECK(it != ttl_nodes_.end()) << dialog_id << " " << m->message_id << " " << source << " " << G()->close_flag()
                                    << " " << m->ttl << " " << m->ttl_expires_at << " " << Time::now() << " "
                                    << m->from_database;

  auto *heap_node = it->as_heap_node();
  if (heap_node->in_heap()) {
    ttl_heap_.erase(heap_node);
  }
  ttl_nodes_.erase(it);
  ttl_update_timeout(Time::now());
}

void MessagesManager::ttl_period_unregister_message(DialogId dialog_id, const Message *m) {
  if (m->ttl_period == 0) {
    return;
  }
  CHECK(!m->message_id.is_scheduled());

  auto it = ttl_nodes_.find(TtlNode(dialog_id, m->message_id, true));

  CHECK(it != ttl_nodes_.end());

  auto *heap_node = it->as_heap_node();
  if (heap_node->in_heap()) {
    ttl_heap_.erase(heap_node);
  }
  ttl_nodes_.erase(it);
  ttl_update_timeout(Time::now());
}

void MessagesManager::ttl_loop(double now) {
  std::unordered_map<DialogId, std::vector<MessageId>, DialogIdHash> to_delete;
  while (!ttl_heap_.empty() && ttl_heap_.top_key() < now) {
    TtlNode *ttl_node = TtlNode::from_heap_node(ttl_heap_.pop());
    auto full_message_id = ttl_node->full_message_id_;
    auto dialog_id = full_message_id.get_dialog_id();
    if (dialog_id.get_type() == DialogType::SecretChat || ttl_node->by_ttl_period_) {
      to_delete[dialog_id].push_back(full_message_id.get_message_id());
    } else {
      auto d = get_dialog(dialog_id);
      CHECK(d != nullptr);
      auto m = get_message(d, full_message_id.get_message_id());
      CHECK(m != nullptr);
      on_message_ttl_expired(d, m);
      on_message_changed(d, m, true, "ttl_loop");
    }
  }
  for (auto &it : to_delete) {
    delete_dialog_messages(it.first, it.second, false, true, "ttl_loop");
  }
  ttl_update_timeout(now);
}

void MessagesManager::ttl_update_timeout(double now) {
  if (ttl_heap_.empty()) {
    if (!ttl_slot_.empty()) {
      ttl_slot_.cancel_timeout();
    }
    return;
  }
  ttl_slot_.set_event(EventCreator::yield(actor_id()));
  ttl_slot_.set_timeout_in(ttl_heap_.top_key() - now);
}

void MessagesManager::on_message_ttl_expired(Dialog *d, Message *m) {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  CHECK(m->ttl > 0);
  CHECK(d->dialog_id.get_type() != DialogType::SecretChat);
  ttl_unregister_message(d->dialog_id, m, "on_message_ttl_expired");
  unregister_message_content(td_, m->content.get(), {d->dialog_id, m->message_id}, "on_message_ttl_expired");
  remove_message_file_sources(d->dialog_id, m);
  on_message_ttl_expired_impl(d, m);
  register_message_content(td_, m->content.get(), {d->dialog_id, m->message_id}, "on_message_ttl_expired");
  send_update_message_content(d, m, true, "on_message_ttl_expired");
  // the caller must call on_message_changed
}

void MessagesManager::on_message_ttl_expired_impl(Dialog *d, Message *m) {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  CHECK(m->message_id.is_valid());
  CHECK(m->ttl > 0);
  CHECK(d->dialog_id.get_type() != DialogType::SecretChat);
  delete_message_files(d->dialog_id, m);
  update_expired_message_content(m->content);
  m->ttl = 0;
  m->ttl_expires_at = 0;
  if (m->reply_markup != nullptr) {
    if (m->reply_markup->type != ReplyMarkup::Type::InlineKeyboard) {
      if (!td_->auth_manager_->is_bot()) {
        if (d->reply_markup_message_id == m->message_id) {
          set_dialog_reply_markup(d, MessageId());
        }
      }
      m->had_reply_markup = true;
    }
    m->reply_markup = nullptr;
  }
  remove_message_notification_id(d, m, true, true);
  update_message_contains_unread_mention(d, m, false, "on_message_ttl_expired_impl");
  unregister_message_reply(d, m);
  m->contains_mention = false;
  m->reply_to_message_id = MessageId();
  m->max_reply_media_timestamp = -1;
  m->reply_in_dialog_id = DialogId();
  m->top_thread_message_id = MessageId();
  m->linked_top_thread_message_id = MessageId();
  m->is_content_secret = false;
}

void MessagesManager::loop() {
  auto token = get_link_token();
  if (token == YieldType::TtlDb) {
    ttl_db_loop(G()->server_time());
  } else {
    ttl_loop(Time::now());
  }
}

class MessagesManager::DialogFiltersLogEvent {
 public:
  int32 updated_date = 0;
  const vector<unique_ptr<DialogFilter>> *server_dialog_filters_in;
  const vector<unique_ptr<DialogFilter>> *dialog_filters_in;
  vector<unique_ptr<DialogFilter>> server_dialog_filters_out;
  vector<unique_ptr<DialogFilter>> dialog_filters_out;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(updated_date, storer);
    td::store(*server_dialog_filters_in, storer);
    td::store(*dialog_filters_in, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(updated_date, parser);
    td::parse(server_dialog_filters_out, parser);
    td::parse(dialog_filters_out, parser);
  }
};

void MessagesManager::tear_down() {
  parent_.reset();
}

void MessagesManager::start_up() {
  init();
}

void MessagesManager::create_folders() {
  LOG(INFO) << "Create folders";
  dialog_folders_[FolderId::main()].folder_id = FolderId::main();
  dialog_folders_[FolderId::archive()].folder_id = FolderId::archive();

  add_dialog_list(DialogListId(FolderId::main()));
  add_dialog_list(DialogListId(FolderId::archive()));
}

void MessagesManager::init() {
  if (is_inited_) {
    return;
  }
  is_inited_ = true;

  always_wait_for_mailbox();

  start_time_ = Time::now();
  last_channel_pts_jump_warning_time_ = start_time_ - 3600;

  bool is_authorized = td_->auth_manager_->is_authorized();
  bool was_authorized_user = td_->auth_manager_->was_authorized() && !td_->auth_manager_->is_bot();
  if (was_authorized_user) {
    create_folders();  // ensure that Main and Archive dialog lists are created
  }
  if (is_authorized && td_->auth_manager_->is_bot()) {
    disable_get_dialog_filter_ = true;
  }
  authorization_date_ = G()->shared_config().get_option_integer("authorization_date");

  if (was_authorized_user) {
    vector<NotificationSettingsScope> scopes{NotificationSettingsScope::Private, NotificationSettingsScope::Group,
                                             NotificationSettingsScope::Channel};
    for (auto scope : scopes) {
      auto notification_settings_string =
          G()->td_db()->get_binlog_pmc()->get(get_notification_settings_scope_database_key(scope));
      if (!notification_settings_string.empty()) {
        auto current_settings = get_scope_notification_settings(scope);
        CHECK(current_settings != nullptr);
        log_event_parse(*current_settings, notification_settings_string).ensure();

        VLOG(notifications) << "Loaded notification settings in " << scope << ": " << *current_settings;

        schedule_scope_unmute(scope, current_settings->mute_until);

        send_closure(G()->td(), &Td::send_update, get_update_scope_notification_settings_object(scope));
      }
    }
    if (!channels_notification_settings_.is_synchronized && is_authorized) {
      channels_notification_settings_ = chats_notification_settings_;
      channels_notification_settings_.disable_pinned_message_notifications = false;
      channels_notification_settings_.disable_mention_notifications = false;
      channels_notification_settings_.is_synchronized = false;
      send_get_scope_notification_settings_query(NotificationSettingsScope::Channel, Promise<>());
    }
  }
  G()->td_db()->get_binlog_pmc()->erase("nsfac");

  if (was_authorized_user) {
    auto dialog_filters = G()->td_db()->get_binlog_pmc()->get("dialog_filters");
    if (!dialog_filters.empty()) {
      DialogFiltersLogEvent log_event;
      if (log_event_parse(log_event, dialog_filters).is_ok()) {
        dialog_filters_updated_date_ = G()->ignore_background_updates() ? 0 : log_event.updated_date;
        std::unordered_set<DialogFilterId, DialogFilterIdHash> server_dialog_filter_ids;
        for (auto &dialog_filter : log_event.server_dialog_filters_out) {
          if (server_dialog_filter_ids.insert(dialog_filter->dialog_filter_id).second) {
            server_dialog_filters_.push_back(std::move(dialog_filter));
          }
        }
        for (auto &dialog_filter : log_event.dialog_filters_out) {
          add_dialog_filter(std::move(dialog_filter), false, "binlog");
        }
        LOG(INFO) << "Loaded server chat filters " << get_dialog_filter_ids(server_dialog_filters_)
                  << " and local chat filters " << get_dialog_filter_ids(dialog_filters_);
      } else {
        LOG(ERROR) << "Failed to parse chat filters from binlog";
      }
    }
    send_update_chat_filters();  // always send updateChatFilters
  }

  if (G()->parameters().use_message_db && was_authorized_user) {
    // erase old keys
    G()->td_db()->get_binlog_pmc()->erase("last_server_dialog_date");
    G()->td_db()->get_binlog_pmc()->erase("unread_message_count");
    G()->td_db()->get_binlog_pmc()->erase("unread_dialog_count");

    auto last_database_server_dialog_dates = G()->td_db()->get_binlog_pmc()->prefix_get("last_server_dialog_date");
    for (auto &it : last_database_server_dialog_dates) {
      auto r_folder_id = to_integer_safe<int32>(it.first);
      if (r_folder_id.is_error()) {
        LOG(ERROR) << "Can't parse folder ID from " << it.first;
        continue;
      }

      string order_str;
      string dialog_id_str;
      std::tie(order_str, dialog_id_str) = split(it.second);

      auto r_order = to_integer_safe<int64>(order_str);
      auto r_dialog_id = to_integer_safe<int64>(dialog_id_str);
      if (r_order.is_error() || r_dialog_id.is_error()) {
        LOG(ERROR) << "Can't parse " << it.second;
      } else {
        FolderId folder_id(r_folder_id.ok());
        auto *folder = get_dialog_folder(folder_id);
        CHECK(folder != nullptr);
        DialogDate dialog_date(r_order.ok(), DialogId(r_dialog_id.ok()));
        if (folder->last_database_server_dialog_date_ < dialog_date) {
          folder->last_database_server_dialog_date_ = dialog_date;
        }
        LOG(INFO) << "Loaded last_database_server_dialog_date_ " << folder->last_database_server_dialog_date_ << " in "
                  << folder_id;
      }
    }

    auto sponsored_dialog_id_string = G()->td_db()->get_binlog_pmc()->get("sponsored_dialog_id");
    if (!sponsored_dialog_id_string.empty()) {
      auto dialog_id_source = split(Slice(sponsored_dialog_id_string));
      auto r_dialog_id = to_integer_safe<int64>(dialog_id_source.first);
      auto r_source = DialogSource::unserialize(dialog_id_source.second);
      if (r_dialog_id.is_error() || r_source.is_error()) {
        LOG(ERROR) << "Can't parse " << sponsored_dialog_id_string;
      } else {
        DialogId dialog_id(r_dialog_id.ok());

        const Dialog *d = get_dialog_force(dialog_id, "init");
        if (d != nullptr) {
          LOG(INFO) << "Loaded sponsored " << dialog_id;
          add_sponsored_dialog(d, r_source.move_as_ok());
        } else {
          LOG(ERROR) << "Can't load " << dialog_id;
        }
      }
    }

    auto pinned_dialog_ids = G()->td_db()->get_binlog_pmc()->prefix_get("pinned_dialog_ids");
    for (auto &it : pinned_dialog_ids) {
      auto r_folder_id = to_integer_safe<int32>(it.first);
      if (r_folder_id.is_error()) {
        LOG(ERROR) << "Can't parse folder ID from " << it.first;
        continue;
      }
      FolderId folder_id(r_folder_id.ok());

      auto r_dialog_ids = transform(full_split(Slice(it.second), ','), [](Slice str) -> Result<DialogId> {
        TRY_RESULT(dialog_id_int, to_integer_safe<int64>(str));
        DialogId dialog_id(dialog_id_int);
        if (!dialog_id.is_valid()) {
          return Status::Error("Have invalid dialog ID");
        }
        return dialog_id;
      });
      if (std::any_of(r_dialog_ids.begin(), r_dialog_ids.end(),
                      [](auto &r_dialog_id) { return r_dialog_id.is_error(); })) {
        LOG(ERROR) << "Can't parse " << it.second;
        reload_pinned_dialogs(DialogListId(folder_id), Auto());
      } else {
        auto *list = get_dialog_list(DialogListId(folder_id));
        CHECK(list != nullptr);
        CHECK(list->pinned_dialogs_.empty());
        for (auto &r_dialog_id : reversed(r_dialog_ids)) {
          auto dialog_id = r_dialog_id.move_as_ok();
          auto order = get_next_pinned_dialog_order();
          list->pinned_dialogs_.emplace_back(order, dialog_id);
          list->pinned_dialog_id_orders_.emplace(dialog_id, order);
        }
        std::reverse(list->pinned_dialogs_.begin(), list->pinned_dialogs_.end());
        list->are_pinned_dialogs_inited_ = true;
        update_list_last_pinned_dialog_date(*list);

        LOG(INFO) << "Loaded pinned chats " << list->pinned_dialogs_ << " in " << folder_id;
      }
    }

    auto unread_message_counts = G()->td_db()->get_binlog_pmc()->prefix_get("unread_message_count");
    for (auto &it : unread_message_counts) {
      auto r_dialog_list_id = to_integer_safe<int64>(it.first);
      if (r_dialog_list_id.is_error()) {
        LOG(ERROR) << "Can't parse dialog list ID from " << it.first;
        continue;
      }
      string total_count;
      string muted_count;
      std::tie(total_count, muted_count) = split(it.second);

      auto r_total_count = to_integer_safe<int32>(total_count);
      auto r_muted_count = to_integer_safe<int32>(muted_count);
      if (r_total_count.is_error() || r_muted_count.is_error()) {
        LOG(ERROR) << "Can't parse " << it.second;
      } else {
        DialogListId dialog_list_id(r_dialog_list_id.ok());
        auto *list = get_dialog_list(dialog_list_id);
        if (list != nullptr) {
          list->unread_message_total_count_ = r_total_count.ok();
          list->unread_message_muted_count_ = r_muted_count.ok();
          list->is_message_unread_count_inited_ = true;
          send_update_unread_message_count(*list, DialogId(), true, "load unread_message_count", true);
        } else {
          G()->td_db()->get_binlog_pmc()->erase("unread_message_count" + it.first);
        }
      }
    }

    auto unread_dialog_counts = G()->td_db()->get_binlog_pmc()->prefix_get("unread_dialog_count");
    for (auto &it : unread_dialog_counts) {
      auto r_dialog_list_id = to_integer_safe<int64>(it.first);
      if (r_dialog_list_id.is_error()) {
        LOG(ERROR) << "Can't parse dialog list ID from " << it.first;
        continue;
      }

      auto counts = transform(full_split(Slice(it.second)), [](Slice str) { return to_integer_safe<int32>(str); });
      if ((counts.size() != 4 && counts.size() != 6) ||
          std::any_of(counts.begin(), counts.end(), [](auto &c) { return c.is_error(); })) {
        LOG(ERROR) << "Can't parse " << it.second;
      } else {
        DialogListId dialog_list_id(r_dialog_list_id.ok());
        auto *list = get_dialog_list(dialog_list_id);
        if (list != nullptr) {
          list->unread_dialog_total_count_ = counts[0].ok();
          list->unread_dialog_muted_count_ = counts[1].ok();
          list->unread_dialog_marked_count_ = counts[2].ok();
          list->unread_dialog_muted_marked_count_ = counts[3].ok();
          if (counts.size() == 6) {
            list->server_dialog_total_count_ = counts[4].ok();
            list->secret_chat_total_count_ = counts[5].ok();
          }
          if (list->server_dialog_total_count_ == -1) {
            repair_server_dialog_total_count(dialog_list_id);
          }
          if (list->secret_chat_total_count_ == -1) {
            repair_secret_chat_total_count(dialog_list_id);
          }
          list->is_dialog_unread_count_inited_ = true;
          send_update_unread_chat_count(*list, DialogId(), true, "load unread_dialog_count", true);
        } else {
          G()->td_db()->get_binlog_pmc()->erase("unread_dialog_count" + it.first);
        }
      }
    }
  } else {
    G()->td_db()->get_binlog_pmc()->erase_by_prefix("pinned_dialog_ids");
    G()->td_db()->get_binlog_pmc()->erase_by_prefix("last_server_dialog_date");
    G()->td_db()->get_binlog_pmc()->erase_by_prefix("unread_message_count");
    G()->td_db()->get_binlog_pmc()->erase_by_prefix("unread_dialog_count");
    G()->td_db()->get_binlog_pmc()->erase("sponsored_dialog_id");
  }
  G()->td_db()->get_binlog_pmc()->erase("dialog_pinned_current_order");

  if (G()->parameters().use_message_db) {
    ttl_db_loop_start(G()->server_time());
  }

  load_calls_db_state();

  if (was_authorized_user && is_authorized) {
    if (need_synchronize_dialog_filters()) {
      reload_dialog_filters();
    } else {
      auto cache_time = get_dialog_filters_cache_time();
      schedule_dialog_filters_reload(cache_time - max(0, G()->unix_time() - dialog_filters_updated_date_));
    }
  }

  auto auth_notification_ids_string = G()->td_db()->get_binlog_pmc()->get("auth_notification_ids");
  if (!auth_notification_ids_string.empty()) {
    VLOG(notifications) << "Loaded auth_notification_ids = " << auth_notification_ids_string;
    auto ids = full_split(auth_notification_ids_string, ',');
    CHECK(ids.size() % 2 == 0);
    bool is_changed = false;
    auto min_date = G()->unix_time() - AUTH_NOTIFICATION_ID_CACHE_TIME;
    for (size_t i = 0; i < ids.size(); i += 2) {
      auto date = to_integer_safe<int32>(ids[i + 1]).ok();
      if (date < min_date) {
        is_changed = true;
        continue;
      }
      auth_notification_id_date_.emplace(std::move(ids[i]), date);
    }
    if (is_changed) {
      save_auth_notification_ids();
    }
  }

  /*
  FI LE *f = std::f open("error.txt", "r");
  if (f != nullptr) {
    DialogId dialog_id(ChannelId(123456));
    force_create_dialog(dialog_id, "test");
    Dialog *d = get_dialog(dialog_id);
    CHECK(d != nullptr);

    delete_all_dialog_messages(d, true, false);

    d->last_new_message_id = MessageId();
    d->last_read_inbox_message_id = MessageId();
    d->last_read_outbox_message_id = MessageId();
    d->is_last_read_inbox_message_id_inited = false;
    d->is_last_read_outbox_message_id_inited = false;

    struct MessageBasicInfo {
      MessageId message_id;
      bool have_previous;
      bool have_next;
    };
    vector<MessageBasicInfo> messages_info;
    std::function<void(Message *m)> get_messages_info = [&](Message *m) {
      if (m == nullptr) {
        return;
      }
      get_messages_info(m->left.get());
      messages_info.push_back(MessageBasicInfo{m->message_id, m->have_previous, m->have_next});
      get_messages_info(m->right.get());
    };

    char buf[1280];
    while (std::f gets(buf, sizeof(buf), f) != nullptr) {
      Slice log_string(buf, std::strlen(buf));
      Slice op = log_string.substr(0, log_string.find(' '));
      if (op != "MessageOpAdd" && op != "MessageOpDelete") {
        LOG(ERROR) << "Unsupported op " << op;
        continue;
      }
      log_string.remove_prefix(log_string.find(' ') + 1);

      if (!begins_with(log_string, "at ")) {
        LOG(ERROR) << "Date expected, found " << log_string;
        continue;
      }
      log_string.remove_prefix(3);
      auto date_slice = log_string.substr(0, log_string.find(' '));
      log_string.remove_prefix(date_slice.size());

      bool is_server = false;
      if (begins_with(log_string, " server message ")) {
        log_string.remove_prefix(16);
        is_server = true;
      } else if (begins_with(log_string, " yet unsent message ")) {
        log_string.remove_prefix(20);
      } else if (begins_with(log_string, " local message ")) {
        log_string.remove_prefix(15);
      } else {
        LOG(ERROR) << "Message identifier expected, found " << log_string;
        continue;
      }

      auto server_message_id = to_integer<int32>(log_string);
      auto add = 0;
      if (!is_server) {
        log_string.remove_prefix(log_string.find('.') + 1);
        add = to_integer<int32>(log_string);
      }
      log_string.remove_prefix(log_string.find(' ') + 1);

      auto message_id = MessageId(MessageId(ServerMessageId(server_message_id)).get() + add);

      auto content_type = log_string.substr(0, log_string.find(' '));
      log_string.remove_prefix(log_string.find(' ') + 1);

      auto read_bool = [](Slice &str) {
        if (begins_with(str, "true ")) {
          str.remove_prefix(5);
          return true;
        }
        if (begins_with(str, "false ")) {
          str.remove_prefix(6);
          return false;
        }
        LOG(ERROR) << "Bool expected, found " << str;
        return false;
      };

      bool from_update = read_bool(log_string);
      bool have_previous = read_bool(log_string);
      bool have_next = read_bool(log_string);

      if (op == "MessageOpAdd") {
        auto m = make_unique<Message>();
        set_message_id(m, message_id);
        m->date = G()->unix_time();
        m->content = create_text_message_content("text", {}, {});

        m->have_previous = have_previous;
        m->have_next = have_next;

        bool need_update = from_update;
        bool need_update_dialog_pos = false;
        if (add_message_to_dialog(dialog_id, std::move(m), from_update, &need_update, &need_update_dialog_pos,
                                  "Unknown source") == nullptr) {
          LOG(ERROR) << "Can't add message " << message_id;
        }
      } else {
        bool need_update_dialog_pos = false;
        auto m = delete_message(d, message_id, true, &need_update_dialog_pos, "Unknown source");
        CHECK(m != nullptr);
      }

      messages_info.clear();
      get_messages_info(d->messages.get());

      for (size_t i = 0; i + 1 < messages_info.size(); i++) {
        if (messages_info[i].have_next != messages_info[i + 1].have_previous) {
          LOG(ERROR) << messages_info[i].message_id << " has have_next = " << messages_info[i].have_next << ", but "
                     << messages_info[i + 1].message_id
                     << " has have_previous = " << messages_info[i + 1].have_previous;
        }
      }
      if (!messages_info.empty()) {
        if (messages_info.back().have_next != false) {
          LOG(ERROR) << messages_info.back().message_id << " has have_next = true, but there is no next message";
        }
        if (messages_info[0].have_previous != false) {
          LOG(ERROR) << messages_info[0].message_id << " has have_previous = true, but there is no previous message";
        }
      }
    }

    messages_info.clear();
    get_messages_info(d->messages.get());
    for (auto &info : messages_info) {
      bool need_update_dialog_pos = false;
      auto m = delete_message(d, info.message_id, true, &need_update_dialog_pos, "Unknown source");
      CHECK(m != nullptr);
    }

    std::f close(f);
  }
  */
}

void MessagesManager::on_authorization_success() {
  CHECK(td_->auth_manager_->is_authorized());
  authorization_date_ = G()->shared_config().get_option_integer("authorization_date");

  if (td_->auth_manager_->is_bot()) {
    disable_get_dialog_filter_ = true;
    return;
  }

  create_folders();

  reload_dialog_filters();
}

void MessagesManager::ttl_db_loop_start(double server_now) {
  ttl_db_expires_from_ = 0;
  ttl_db_expires_till_ = static_cast<int32>(server_now) + 15 /* 15 seconds */;
  ttl_db_has_query_ = false;

  ttl_db_loop(server_now);
}

void MessagesManager::ttl_db_loop(double server_now) {
  LOG(INFO) << "Begin ttl_db loop: " << tag("expires_from", ttl_db_expires_from_)
            << tag("expires_till", ttl_db_expires_till_) << tag("has_query", ttl_db_has_query_);
  if (ttl_db_has_query_) {
    return;
  }

  auto now = static_cast<int32>(server_now);

  if (ttl_db_expires_till_ < 0) {
    LOG(INFO) << "Finish ttl_db loop";
    return;
  }

  if (now < ttl_db_expires_from_) {
    ttl_db_slot_.set_event(EventCreator::yield(actor_shared(this, YieldType::TtlDb)));
    auto wakeup_in = ttl_db_expires_from_ - server_now;
    ttl_db_slot_.set_timeout_in(wakeup_in);
    LOG(INFO) << "Set ttl_db timeout in " << wakeup_in;
    return;
  }

  ttl_db_has_query_ = true;
  int32 limit = 50;
  LOG(INFO) << "Send ttl_db query " << tag("expires_from", ttl_db_expires_from_)
            << tag("expires_till", ttl_db_expires_till_) << tag("limit", limit);
  G()->td_db()->get_messages_db_async()->get_expiring_messages(
      ttl_db_expires_from_, ttl_db_expires_till_, limit,
      PromiseCreator::lambda(
          [actor_id = actor_id(this)](Result<std::pair<std::vector<MessagesDbMessage>, int32>> result) {
            send_closure(actor_id, &MessagesManager::ttl_db_on_result, std::move(result), false);
          }));
}

void MessagesManager::ttl_db_on_result(Result<std::pair<std::vector<MessagesDbMessage>, int32>> r_result, bool dummy) {
  if (G()->close_flag()) {
    return;
  }

  auto result = r_result.move_as_ok();
  ttl_db_has_query_ = false;
  ttl_db_expires_from_ = ttl_db_expires_till_;
  ttl_db_expires_till_ = result.second;

  LOG(INFO) << "Receive ttl_db query result " << tag("new expires_till", ttl_db_expires_till_)
            << tag("got messages", result.first.size());
  for (auto &dialog_message : result.first) {
    on_get_message_from_database(dialog_message, false, "ttl_db_on_result");
  }
  ttl_db_loop(G()->server_time());
}

void MessagesManager::on_send_secret_message_error(int64 random_id, Status error, Promise<> promise) {
  promise.set_value(Unit());  // TODO: set after error is saved

  auto it = being_sent_messages_.find(random_id);
  if (it != being_sent_messages_.end()) {
    auto full_message_id = it->second;
    auto *m = get_message(full_message_id);
    if (m != nullptr) {
      auto file_id = get_message_content_upload_file_id(m->content.get());
      if (file_id.is_valid()) {
        if (G()->close_flag() && G()->parameters().use_message_db) {
          // do not send error, message will be re-sent
          return;
        }
        if (begins_with(error.message(), "FILE_PART_") && ends_with(error.message(), "_MISSING")) {
          on_send_message_file_part_missing(random_id, to_integer<int32>(error.message().substr(10)));
          return;
        }

        if (error.code() != 429 && error.code() < 500 && !G()->close_flag()) {
          td_->file_manager_->delete_partial_remote_location(file_id);
        }
      }
    }
  }

  on_send_message_fail(random_id, std::move(error));
}

void MessagesManager::on_send_secret_message_success(int64 random_id, MessageId message_id, int32 date,
                                                     unique_ptr<EncryptedFile> file, Promise<> promise) {
  promise.set_value(Unit());  // TODO: set after message is saved

  FileId new_file_id;
  if (file != nullptr) {
    if (!DcId::is_valid(file->dc_id_)) {
      LOG(ERROR) << "Wrong dc_id = " << file->dc_id_ << " in file " << *file;
    } else {
      DialogId owner_dialog_id;
      auto it = being_sent_messages_.find(random_id);
      if (it != being_sent_messages_.end()) {
        owner_dialog_id = it->second.get_dialog_id();
      }

      new_file_id = td_->file_manager_->register_remote(
          FullRemoteFileLocation(FileType::Encrypted, file->id_, file->access_hash_, DcId::internal(file->dc_id_), ""),
          FileLocationSource::FromServer, owner_dialog_id, 0, file->size_, to_string(static_cast<uint64>(file->id_)));
    }
  }

  on_send_message_success(random_id, message_id, date, 0, new_file_id, "process send_secret_message_success");
}

void MessagesManager::delete_secret_messages(SecretChatId secret_chat_id, std::vector<int64> random_ids,
                                             Promise<> promise) {
  LOG(DEBUG) << "On delete messages in " << secret_chat_id << " with random_ids " << random_ids;
  CHECK(secret_chat_id.is_valid());

  DialogId dialog_id(secret_chat_id);
  if (!have_dialog_force(dialog_id, "delete_secret_messages")) {
    LOG(ERROR) << "Ignore delete secret messages in unknown " << dialog_id;
    promise.set_value(Unit());
    return;
  }

  auto pending_secret_message = make_unique<PendingSecretMessage>();
  pending_secret_message->success_promise = std::move(promise);
  pending_secret_message->type = PendingSecretMessage::Type::DeleteMessages;
  pending_secret_message->dialog_id = dialog_id;
  pending_secret_message->random_ids = std::move(random_ids);

  add_secret_message(std::move(pending_secret_message));
}

void MessagesManager::finish_delete_secret_messages(DialogId dialog_id, std::vector<int64> random_ids,
                                                    Promise<> promise) {
  LOG(INFO) << "Delete messages with random_ids " << random_ids << " in " << dialog_id;
  promise.set_value(Unit());  // TODO: set after event is saved

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  vector<MessageId> to_delete_message_ids;
  for (auto &random_id : random_ids) {
    auto message_id = get_message_id_by_random_id(d, random_id, "delete_secret_messages");
    if (!message_id.is_valid()) {
      LOG(INFO) << "Can't find message with random_id " << random_id;
      continue;
    }
    const Message *m = get_message(d, message_id);
    CHECK(m != nullptr);
    if (!is_service_message_content(m->content->get_type())) {
      to_delete_message_ids.push_back(message_id);
    } else {
      LOG(INFO) << "Skip deletion of service " << message_id;
    }
  }
  delete_dialog_messages(dialog_id, to_delete_message_ids, true, false, "finish_delete_secret_messages");
}

void MessagesManager::delete_secret_chat_history(SecretChatId secret_chat_id, bool remove_from_dialog_list,
                                                 MessageId last_message_id, Promise<> promise) {
  LOG(DEBUG) << "Delete history in " << secret_chat_id << " up to " << last_message_id;
  CHECK(secret_chat_id.is_valid());
  CHECK(!last_message_id.is_scheduled());

  DialogId dialog_id(secret_chat_id);
  if (!have_dialog_force(dialog_id, "delete_secret_chat_history")) {
    LOG(ERROR) << "Ignore delete history in unknown " << dialog_id;
    promise.set_value(Unit());
    return;
  }

  auto pending_secret_message = make_unique<PendingSecretMessage>();
  pending_secret_message->success_promise = std::move(promise);
  pending_secret_message->type = PendingSecretMessage::Type::DeleteHistory;
  pending_secret_message->dialog_id = dialog_id;
  pending_secret_message->last_message_id = last_message_id;
  pending_secret_message->remove_from_dialog_list = remove_from_dialog_list;

  add_secret_message(std::move(pending_secret_message));
}

void MessagesManager::finish_delete_secret_chat_history(DialogId dialog_id, bool remove_from_dialog_list,
                                                        MessageId last_message_id, Promise<> promise) {
  LOG(DEBUG) << "Delete history in " << dialog_id << " up to " << last_message_id;
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  // TODO: probably last_message_id is not needed
  delete_all_dialog_messages(d, remove_from_dialog_list, true);
  promise.set_value(Unit());  // TODO: set after event is saved
}

void MessagesManager::read_secret_chat_outbox(SecretChatId secret_chat_id, int32 up_to_date, int32 read_date) {
  if (!secret_chat_id.is_valid()) {
    LOG(ERROR) << "Receive read secret chat outbox in the invalid " << secret_chat_id;
    return;
  }
  auto dialog_id = DialogId(secret_chat_id);
  Dialog *d = get_dialog_force(dialog_id, "read_secret_chat_outbox");
  if (d == nullptr) {
    return;
  }

  if (read_date > 0) {
    auto user_id = td_->contacts_manager_->get_secret_chat_user_id(secret_chat_id);
    if (user_id.is_valid()) {
      td_->contacts_manager_->on_update_user_local_was_online(user_id, read_date);
    }
  }

  // TODO: protect with log event
  suffix_load_till_date(
      d, up_to_date,
      PromiseCreator::lambda([actor_id = actor_id(this), dialog_id, up_to_date, read_date](Result<Unit> result) {
        send_closure(actor_id, &MessagesManager::read_secret_chat_outbox_inner, dialog_id, up_to_date, read_date);
      }));
}

void MessagesManager::read_secret_chat_outbox_inner(DialogId dialog_id, int32 up_to_date, int32 read_date) {
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  auto end = MessagesConstIterator(d, MessageId::max());
  while (*end && (*end)->date > up_to_date) {
    --end;
  }
  if (!*end) {
    LOG(INFO) << "Ignore read_secret_chat_outbox in " << dialog_id << " at " << up_to_date
              << ": no messages with such date are known";
    return;
  }
  auto max_message_id = (*end)->message_id;
  read_history_outbox(dialog_id, max_message_id, read_date);
}

void MessagesManager::open_secret_message(SecretChatId secret_chat_id, int64 random_id, Promise<> promise) {
  promise.set_value(Unit());  // TODO: set after event is saved
  DialogId dialog_id(secret_chat_id);
  Dialog *d = get_dialog_force(dialog_id, "open_secret_message");
  if (d == nullptr) {
    LOG(ERROR) << "Ignore opening secret chat message in unknown " << dialog_id;
    return;
  }

  auto message_id = get_message_id_by_random_id(d, random_id, "open_secret_message");
  if (!message_id.is_valid()) {
    return;
  }
  Message *m = get_message(d, message_id);
  CHECK(m != nullptr);
  if (m->message_id.is_yet_unsent() || m->is_failed_to_send || !m->is_outgoing) {
    LOG(ERROR) << "Peer has opened wrong " << message_id << " in " << dialog_id;
    return;
  }

  read_message_content(d, m, false, "open_secret_message");
}

void MessagesManager::on_update_secret_chat_state(SecretChatId secret_chat_id, SecretChatState state) {
  if (state == SecretChatState::Closed && !td_->auth_manager_->is_bot()) {
    DialogId dialog_id(secret_chat_id);
    Dialog *d = get_dialog_force(dialog_id, "on_update_secret_chat_state");
    if (d != nullptr) {
      if (d->new_secret_chat_notification_id.is_valid()) {
        remove_new_secret_chat_notification(d, true);
      }
      if (d->message_notification_group.group_id.is_valid() && get_dialog_pending_notification_count(d, false) == 0 &&
          !d->message_notification_group.last_notification_id.is_valid()) {
        CHECK(d->message_notification_group.last_notification_date == 0);
        d->message_notification_group.try_reuse = true;
        d->message_notification_group.is_changed = true;
        on_dialog_updated(d->dialog_id, "on_update_secret_chat_state");
      }
      CHECK(!d->mention_notification_group.group_id.is_valid());  // there can't be unread mentions in secret chats
    }
  }
}

void MessagesManager::on_get_secret_message(SecretChatId secret_chat_id, UserId user_id, MessageId message_id,
                                            int32 date, unique_ptr<EncryptedFile> file,
                                            tl_object_ptr<secret_api::decryptedMessage> message, Promise<> promise) {
  LOG(DEBUG) << "On get " << to_string(message);
  CHECK(message != nullptr);
  CHECK(secret_chat_id.is_valid());
  CHECK(user_id.is_valid());
  CHECK(message_id.is_valid());
  CHECK(date > 0);

  auto pending_secret_message = make_unique<PendingSecretMessage>();
  pending_secret_message->success_promise = std::move(promise);
  MessageInfo &message_info = pending_secret_message->message_info;
  message_info.dialog_id = DialogId(secret_chat_id);
  message_info.message_id = message_id;
  message_info.sender_user_id = user_id;
  message_info.date = date;
  message_info.random_id = message->random_id_;
  message_info.ttl = message->ttl_;

  Dialog *d = get_dialog_force(message_info.dialog_id, "on_get_secret_message");
  if (d == nullptr && have_dialog_info_force(message_info.dialog_id)) {
    force_create_dialog(message_info.dialog_id, "on_get_secret_message", true, true);
    d = get_dialog(message_info.dialog_id);
  }
  if (d == nullptr) {
    LOG(ERROR) << "Ignore secret message in unknown " << message_info.dialog_id;
    pending_secret_message->success_promise.set_error(Status::Error(500, "Chat not found"));
    return;
  }

  pending_secret_message->load_data_multipromise.add_promise(Auto());
  auto lock_promise = pending_secret_message->load_data_multipromise.get_promise();

  int32 flags = MESSAGE_FLAG_HAS_UNREAD_CONTENT | MESSAGE_FLAG_HAS_FROM_ID;
  if ((message->flags_ & secret_api::decryptedMessage::REPLY_TO_RANDOM_ID_MASK) != 0) {
    message_info.reply_to_message_id = get_message_id_by_random_id(
        get_dialog(message_info.dialog_id), message->reply_to_random_id_, "on_get_secret_message");
    if (message_info.reply_to_message_id.is_valid()) {
      flags |= MESSAGE_FLAG_IS_REPLY;
    }
  }
  if ((message->flags_ & secret_api::decryptedMessage::ENTITIES_MASK) != 0) {
    flags |= MESSAGE_FLAG_HAS_ENTITIES;
  }
  if ((message->flags_ & secret_api::decryptedMessage::MEDIA_MASK) != 0) {
    flags |= MESSAGE_FLAG_HAS_MEDIA;
  }
  if ((message->flags_ & secret_api::decryptedMessage::SILENT_MASK) != 0) {
    flags |= MESSAGE_FLAG_IS_SILENT;
  }

  if (!clean_input_string(message->via_bot_name_)) {
    LOG(WARNING) << "Receive invalid bot username " << message->via_bot_name_;
    message->via_bot_name_.clear();
  }
  if (!message->via_bot_name_.empty()) {
    auto request_promise = PromiseCreator::lambda(
        [actor_id = actor_id(this), via_bot_username = message->via_bot_name_, message_info_ptr = &message_info,
         promise = pending_secret_message->load_data_multipromise.get_promise()](Unit) mutable {
          send_closure(actor_id, &MessagesManager::on_resolve_secret_chat_message_via_bot_username, via_bot_username,
                       message_info_ptr, std::move(promise));
        });
    search_public_dialog(message->via_bot_name_, false, std::move(request_promise));
  }
  if ((message->flags_ & secret_api::decryptedMessage::GROUPED_ID_MASK) != 0 && message->grouped_id_ != 0) {
    message_info.media_album_id = message->grouped_id_;
    flags |= MESSAGE_FLAG_HAS_MEDIA_ALBUM_ID;
  }

  message_info.flags = flags;
  message_info.content = get_secret_message_content(
      td_, std::move(message->message_), std::move(file), std::move(message->media_), std::move(message->entities_),
      message_info.dialog_id, pending_secret_message->load_data_multipromise);

  add_secret_message(std::move(pending_secret_message), std::move(lock_promise));
}

void MessagesManager::on_resolve_secret_chat_message_via_bot_username(const string &via_bot_username,
                                                                      MessageInfo *message_info_ptr,
                                                                      Promise<Unit> &&promise) {
  if (!G()->close_flag()) {
    auto dialog_id = resolve_dialog_username(via_bot_username);
    if (dialog_id.is_valid() && dialog_id.get_type() == DialogType::User) {
      auto user_id = dialog_id.get_user_id();
      auto r_bot_data = td_->contacts_manager_->get_bot_data(user_id);
      if (r_bot_data.is_ok() && r_bot_data.ok().is_inline) {
        message_info_ptr->flags |= MESSAGE_FLAG_IS_SENT_VIA_BOT;
        message_info_ptr->via_bot_user_id = user_id;
      }
    }
  }
  promise.set_value(Unit());
}

void MessagesManager::on_secret_chat_screenshot_taken(SecretChatId secret_chat_id, UserId user_id, MessageId message_id,
                                                      int32 date, int64 random_id, Promise<> promise) {
  LOG(DEBUG) << "On screenshot taken in " << secret_chat_id;
  CHECK(secret_chat_id.is_valid());
  CHECK(user_id.is_valid());
  CHECK(message_id.is_valid());
  CHECK(date > 0);

  auto pending_secret_message = make_unique<PendingSecretMessage>();
  pending_secret_message->success_promise = std::move(promise);
  MessageInfo &message_info = pending_secret_message->message_info;
  message_info.dialog_id = DialogId(secret_chat_id);
  message_info.message_id = message_id;
  message_info.sender_user_id = user_id;
  message_info.date = date;
  message_info.random_id = random_id;
  message_info.flags = MESSAGE_FLAG_HAS_FROM_ID;
  message_info.content = create_screenshot_taken_message_content();

  Dialog *d = get_dialog_force(message_info.dialog_id, "on_secret_chat_screenshot_taken");
  if (d == nullptr && have_dialog_info_force(message_info.dialog_id)) {
    force_create_dialog(message_info.dialog_id, "on_get_secret_message", true, true);
    d = get_dialog(message_info.dialog_id);
  }
  if (d == nullptr) {
    LOG(ERROR) << "Ignore secret message in unknown " << message_info.dialog_id;
    pending_secret_message->success_promise.set_error(Status::Error(500, "Chat not found"));
    return;
  }

  add_secret_message(std::move(pending_secret_message));
}

void MessagesManager::on_secret_chat_ttl_changed(SecretChatId secret_chat_id, UserId user_id, MessageId message_id,
                                                 int32 date, int32 ttl, int64 random_id, Promise<> promise) {
  LOG(DEBUG) << "On TTL set in " << secret_chat_id << " to " << ttl;
  CHECK(secret_chat_id.is_valid());
  CHECK(user_id.is_valid());
  CHECK(message_id.is_valid());
  CHECK(date > 0);
  if (ttl < 0) {
    LOG(WARNING) << "Receive wrong TTL = " << ttl;
    promise.set_value(Unit());
    return;
  }

  auto pending_secret_message = make_unique<PendingSecretMessage>();
  pending_secret_message->success_promise = std::move(promise);
  MessageInfo &message_info = pending_secret_message->message_info;
  message_info.dialog_id = DialogId(secret_chat_id);
  message_info.message_id = message_id;
  message_info.sender_user_id = user_id;
  message_info.date = date;
  message_info.random_id = random_id;
  message_info.flags = MESSAGE_FLAG_HAS_FROM_ID;
  message_info.content = create_chat_set_ttl_message_content(ttl);

  Dialog *d = get_dialog_force(message_info.dialog_id, "on_secret_chat_ttl_changed");
  if (d == nullptr && have_dialog_info_force(message_info.dialog_id)) {
    force_create_dialog(message_info.dialog_id, "on_get_secret_message", true, true);
    d = get_dialog(message_info.dialog_id);
  }
  if (d == nullptr) {
    LOG(ERROR) << "Ignore secret message in unknown " << message_info.dialog_id;
    pending_secret_message->success_promise.set_error(Status::Error(500, "Chat not found"));
    return;
  }

  add_secret_message(std::move(pending_secret_message));
}

void MessagesManager::add_secret_message(unique_ptr<PendingSecretMessage> pending_secret_message,
                                         Promise<Unit> lock_promise) {
  auto &multipromise = pending_secret_message->load_data_multipromise;
  multipromise.set_ignore_errors(true);
  int64 token = pending_secret_messages_.add(std::move(pending_secret_message));

  multipromise.add_promise(PromiseCreator::lambda([actor_id = actor_id(this), token](Result<Unit> result) {
    if (result.is_ok()) {
      send_closure(actor_id, &MessagesManager::on_add_secret_message_ready, token);
    }
  }));

  if (!lock_promise) {
    lock_promise = multipromise.get_promise();
  }
  lock_promise.set_value(Unit());
}

void MessagesManager::on_add_secret_message_ready(int64 token) {
  if (G()->close_flag()) {
    return;
  }

  pending_secret_messages_.finish(
      token, [actor_id = actor_id(this)](unique_ptr<PendingSecretMessage> pending_secret_message) {
        send_closure_later(actor_id, &MessagesManager::finish_add_secret_message, std::move(pending_secret_message));
      });
}

void MessagesManager::finish_add_secret_message(unique_ptr<PendingSecretMessage> pending_secret_message) {
  if (G()->close_flag()) {
    return;
  }

  if (pending_secret_message->type == PendingSecretMessage::Type::DeleteMessages) {
    return finish_delete_secret_messages(pending_secret_message->dialog_id,
                                         std::move(pending_secret_message->random_ids),
                                         std::move(pending_secret_message->success_promise));
  }
  if (pending_secret_message->type == PendingSecretMessage::Type::DeleteHistory) {
    return finish_delete_secret_chat_history(
        pending_secret_message->dialog_id, pending_secret_message->remove_from_dialog_list,
        pending_secret_message->last_message_id, std::move(pending_secret_message->success_promise));
  }

  auto d = get_dialog(pending_secret_message->message_info.dialog_id);
  CHECK(d != nullptr);
  auto random_id = pending_secret_message->message_info.random_id;
  auto message_id = get_message_id_by_random_id(d, random_id, "finish_add_secret_message");
  if (message_id.is_valid()) {
    if (message_id != pending_secret_message->message_info.message_id) {
      LOG(WARNING) << "Ignore duplicate " << pending_secret_message->message_info.message_id
                   << " received earlier with " << message_id << " and random_id " << random_id;
    }
  } else {
    on_get_message(std::move(pending_secret_message->message_info), true, false, true, true,
                   "finish add secret message");
  }
  pending_secret_message->success_promise.set_value(Unit());  // TODO: set after message is saved
}

MessagesManager::MessageInfo MessagesManager::parse_telegram_api_message(
    tl_object_ptr<telegram_api::Message> message_ptr, bool is_scheduled, const char *source) const {
  LOG(DEBUG) << "Receive from " << source << " " << to_string(message_ptr);
  LOG_CHECK(message_ptr != nullptr) << source;

  MessageInfo message_info;
  message_info.message_id = get_message_id(message_ptr, is_scheduled);
  switch (message_ptr->get_id()) {
    case telegram_api::messageEmpty::ID:
      message_info.message_id = MessageId();
      break;
    case telegram_api::message::ID: {
      auto message = move_tl_object_as<telegram_api::message>(message_ptr);

      message_info.dialog_id = DialogId(message->peer_id_);
      if (message->from_id_ != nullptr) {
        message_info.sender_dialog_id = DialogId(message->from_id_);
      } else {
        message_info.sender_dialog_id = message_info.dialog_id;
      }
      message_info.date = message->date_;
      message_info.forward_header = std::move(message->fwd_from_);
      message_info.reply_header = std::move(message->reply_to_);
      if (message->flags_ & MESSAGE_FLAG_IS_SENT_VIA_BOT) {
        message_info.via_bot_user_id = UserId(message->via_bot_id_);
        if (!message_info.via_bot_user_id.is_valid()) {
          LOG(ERROR) << "Receive invalid " << message_info.via_bot_user_id << " from " << source;
          message_info.via_bot_user_id = UserId();
        }
      }
      if (message->flags_ & MESSAGE_FLAG_HAS_INTERACTION_INFO) {
        message_info.view_count = message->views_;
        message_info.forward_count = message->forwards_;
      }
      if (message->flags_ & MESSAGE_FLAG_HAS_REPLY_INFO) {
        message_info.reply_info = std::move(message->replies_);
      }
      if (message->flags_ & MESSAGE_FLAG_HAS_EDIT_DATE) {
        message_info.edit_date = message->edit_date_;
      }
      if (message->flags_ & MESSAGE_FLAG_HAS_MEDIA_ALBUM_ID) {
        message_info.media_album_id = message->grouped_id_;
      }
      if (message->flags_ & MESSAGE_FLAG_HAS_TTL_PERIOD) {
        message_info.ttl_period = message->ttl_period_;
      }
      message_info.flags = message->flags_;
      bool is_content_read = (message->flags_ & MESSAGE_FLAG_HAS_UNREAD_CONTENT) == 0;
      if (is_message_auto_read(message_info.dialog_id, (message->flags_ & MESSAGE_FLAG_IS_OUT) != 0)) {
        is_content_read = true;
      }
      if (is_scheduled) {
        is_content_read = false;
      }
      auto new_source = PSTRING() << FullMessageId(message_info.dialog_id, message_info.message_id) << " from "
                                  << source;
      message_info.content = get_message_content(
          td_,
          get_message_text(td_->contacts_manager_.get(), std::move(message->message_), std::move(message->entities_),
                           true, td_->auth_manager_->is_bot(),
                           message_info.forward_header ? message_info.forward_header->date_ : message_info.date,
                           message_info.media_album_id != 0, new_source.c_str()),
          std::move(message->media_), message_info.dialog_id, is_content_read, message_info.via_bot_user_id,
          &message_info.ttl, &message_info.disable_web_page_preview);
      message_info.reply_markup =
          message->flags_ & MESSAGE_FLAG_HAS_REPLY_MARKUP ? std::move(message->reply_markup_) : nullptr;
      message_info.restriction_reasons = get_restriction_reasons(std::move(message->restriction_reason_));
      message_info.author_signature = std::move(message->post_author_);
      break;
    }
    case telegram_api::messageService::ID: {
      auto message = move_tl_object_as<telegram_api::messageService>(message_ptr);

      message_info.dialog_id = DialogId(message->peer_id_);
      if (message->from_id_ != nullptr) {
        message_info.sender_dialog_id = DialogId(message->from_id_);
      } else {
        message_info.sender_dialog_id = message_info.dialog_id;
      }
      message_info.date = message->date_;
      if (message->flags_ & MESSAGE_FLAG_HAS_TTL_PERIOD) {
        message_info.ttl_period = message->ttl_period_;
      }
      message_info.flags = message->flags_;

      DialogId reply_in_dialog_id;
      MessageId reply_to_message_id;
      if (message->reply_to_ != nullptr) {
        reply_to_message_id = MessageId(ServerMessageId(message->reply_to_->reply_to_msg_id_));
        auto reply_to_peer_id = std::move(message->reply_to_->reply_to_peer_id_);
        if (reply_to_peer_id != nullptr) {
          reply_in_dialog_id = DialogId(reply_to_peer_id);
          if (!reply_in_dialog_id.is_valid()) {
            LOG(ERROR) << "Receive reply in invalid " << to_string(reply_to_peer_id);
            reply_to_message_id = MessageId();
            reply_in_dialog_id = DialogId();
          }
        }
      }
      message_info.content = get_action_message_content(td_, std::move(message->action_), message_info.dialog_id,
                                                        reply_in_dialog_id, reply_to_message_id);
      break;
    }
    default:
      UNREACHABLE();
      break;
  }
  if (message_info.sender_dialog_id.is_valid() && message_info.sender_dialog_id.get_type() == DialogType::User) {
    message_info.sender_user_id = message_info.sender_dialog_id.get_user_id();
    message_info.sender_dialog_id = DialogId();
  }
  return message_info;
}

std::pair<DialogId, unique_ptr<MessagesManager::Message>> MessagesManager::create_message(MessageInfo &&message_info,
                                                                                          bool is_channel_message) {
  DialogId dialog_id = message_info.dialog_id;
  MessageId message_id = message_info.message_id;
  if ((!message_id.is_valid() && !message_id.is_valid_scheduled()) || !dialog_id.is_valid()) {
    if (message_id != MessageId() || dialog_id != DialogId()) {
      LOG(ERROR) << "Receive " << message_id << " in " << dialog_id;
    }
    return {DialogId(), nullptr};
  }
  if (message_id.is_yet_unsent() || message_id.is_local()) {
    LOG(ERROR) << "Receive " << message_id;
    return {DialogId(), nullptr};
  }

  CHECK(message_info.content != nullptr);

  auto dialog_type = dialog_id.get_type();
  UserId sender_user_id = message_info.sender_user_id;
  DialogId sender_dialog_id = message_info.sender_dialog_id;
  if (!sender_user_id.is_valid()) {
    if (sender_user_id != UserId()) {
      LOG(ERROR) << "Receive invalid " << sender_user_id;
      sender_user_id = UserId();
    }
    if (!is_broadcast_channel(dialog_id) && td_->auth_manager_->is_bot()) {
      if (dialog_id == sender_dialog_id) {
        td_->contacts_manager_->add_anonymous_bot_user();
      } else {
        td_->contacts_manager_->add_service_notifications_user();
      }
    }
  }
  if (sender_dialog_id.is_valid()) {
    if (dialog_type == DialogType::User || dialog_type == DialogType::SecretChat) {
      LOG(ERROR) << "Receive " << message_id << " sent by " << sender_dialog_id << " in " << dialog_id;
      return {DialogId(), nullptr};
    }
  } else if (sender_dialog_id != DialogId()) {
    LOG(ERROR) << "Receive invalid " << sender_dialog_id;
    sender_dialog_id = DialogId();
  }
  if (message_id.is_scheduled()) {
    is_channel_message = (dialog_type == DialogType::Channel);
  }

  int32 flags = message_info.flags;
  bool is_outgoing = (flags & MESSAGE_FLAG_IS_OUT) != 0;
  bool is_silent = (flags & MESSAGE_FLAG_IS_SILENT) != 0;
  bool is_channel_post = (flags & MESSAGE_FLAG_IS_POST) != 0;
  bool is_legacy = (flags & MESSAGE_FLAG_IS_LEGACY) != 0;
  bool hide_edit_date = (flags & MESSAGE_FLAG_HIDE_EDIT_DATE) != 0;
  bool is_from_scheduled = (flags & MESSAGE_FLAG_IS_FROM_SCHEDULED) != 0;
  bool is_pinned = (flags & MESSAGE_FLAG_IS_PINNED) != 0;

  LOG_IF(ERROR, is_channel_message != (dialog_type == DialogType::Channel))
      << "is_channel_message is wrong for " << message_id << " received in the " << dialog_id;
  LOG_IF(ERROR, is_channel_post && !is_broadcast_channel(dialog_id))
      << "is_channel_post is true for " << message_id << " received in the " << dialog_id;

  UserId my_id = td_->contacts_manager_->get_my_id();
  DialogId my_dialog_id = DialogId(my_id);
  if (dialog_id == my_dialog_id && (sender_user_id != my_id || sender_dialog_id.is_valid())) {
    LOG(ERROR) << "Receive " << sender_user_id << "/" << sender_dialog_id << " as a sender of " << message_id
               << " instead of self";
    sender_user_id = my_id;
    sender_dialog_id = DialogId();
  }

  bool supposed_to_be_outgoing = sender_user_id == my_id && !(dialog_id == my_dialog_id && !message_id.is_scheduled());
  if (sender_user_id.is_valid() && supposed_to_be_outgoing != is_outgoing) {
    LOG(ERROR) << "Receive wrong message out flag: me is " << my_id << ", message is from " << sender_user_id
               << ", flags = " << flags << " for " << message_id << " in " << dialog_id;
    is_outgoing = supposed_to_be_outgoing;

    /*
    // it is useless to call getChannelsDifference, because the channel pts will be increased already
    if (dialog_type == DialogType::Channel && !running_get_difference_ && !running_get_channel_difference(dialog_id) &&
        get_channel_difference_to_log_event_id_.count(dialog_id) == 0) {
      // it is safer to completely ignore the message and re-get it through getChannelsDifference
      Dialog *d = get_dialog(dialog_id);
      if (d != nullptr) {
        channel_get_difference_retry_timeout_.add_timeout_in(dialog_id.get(), 0.001);
        return {DialogId(), nullptr};
      }
    }
    */
  }

  MessageId reply_to_message_id = message_info.reply_to_message_id;  // for secret messages
  DialogId reply_in_dialog_id;
  MessageId top_thread_message_id;
  if (message_info.reply_header != nullptr) {
    reply_to_message_id = MessageId(ServerMessageId(message_info.reply_header->reply_to_msg_id_));
    auto reply_to_peer_id = std::move(message_info.reply_header->reply_to_peer_id_);
    if (reply_to_peer_id != nullptr) {
      reply_in_dialog_id = DialogId(reply_to_peer_id);
      if (!reply_in_dialog_id.is_valid()) {
        LOG(ERROR) << "Receive reply in invalid " << to_string(reply_to_peer_id);
        reply_to_message_id = MessageId();
        reply_in_dialog_id = DialogId();
      }
      if (reply_in_dialog_id == dialog_id) {
        reply_in_dialog_id = DialogId();  // just in case
      }
    }
    if (reply_to_message_id.is_valid() && !td_->auth_manager_->is_bot() && !message_id.is_scheduled() &&
        !reply_in_dialog_id.is_valid()) {
      if ((message_info.reply_header->flags_ & telegram_api::messageReplyHeader::REPLY_TO_TOP_ID_MASK) != 0) {
        top_thread_message_id = MessageId(ServerMessageId(message_info.reply_header->reply_to_top_id_));
      } else if (!is_broadcast_channel(dialog_id)) {
        top_thread_message_id = reply_to_message_id;
      }
    }
  }
  fix_server_reply_to_message_id(dialog_id, message_id, reply_in_dialog_id, reply_to_message_id);
  fix_server_reply_to_message_id(dialog_id, message_id, reply_in_dialog_id, top_thread_message_id);

  UserId via_bot_user_id = message_info.via_bot_user_id;
  if (!via_bot_user_id.is_valid()) {
    via_bot_user_id = UserId();
  }

  int32 date = message_info.date;
  if (date <= 0) {
    LOG(ERROR) << "Wrong date = " << date << " received in " << message_id << " in " << dialog_id;
    date = 1;
  }

  int32 edit_date = message_info.edit_date;
  if (edit_date < 0) {
    LOG(ERROR) << "Wrong edit_date = " << edit_date << " received in " << message_id << " in " << dialog_id;
    edit_date = 0;
  }

  auto content_type = message_info.content->get_type();
  if (hide_edit_date && td_->auth_manager_->is_bot()) {
    hide_edit_date = false;
  }
  if (hide_edit_date && content_type == MessageContentType::LiveLocation) {
    hide_edit_date = false;
  }

  int32 ttl_period = message_info.ttl_period;
  if (ttl_period < 0 || (message_id.is_scheduled() && ttl_period != 0)) {
    LOG(ERROR) << "Wrong TTL period = " << ttl_period << " received in " << message_id << " in " << dialog_id;
    ttl_period = 0;
  }

  int32 ttl = message_info.ttl;
  bool is_content_secret = is_secret_message_content(ttl, content_type);  // must be calculated before TTL is adjusted
  if (ttl < 0 || (message_id.is_scheduled() && ttl != 0)) {
    LOG(ERROR) << "Wrong TTL = " << ttl << " received in " << message_id << " in " << dialog_id;
    ttl = 0;
  } else if (ttl > 0) {
    ttl = max(ttl, get_message_content_duration(message_info.content.get(), td_) + 1);
  }

  int32 view_count = message_info.view_count;
  if (view_count < 0) {
    LOG(ERROR) << "Wrong view_count = " << view_count << " received in " << message_id << " in " << dialog_id;
    view_count = 0;
  }
  int32 forward_count = message_info.forward_count;
  if (forward_count < 0) {
    LOG(ERROR) << "Wrong forward_count = " << forward_count << " received in " << message_id << " in " << dialog_id;
    forward_count = 0;
  }
  MessageReplyInfo reply_info(std::move(message_info.reply_info), td_->auth_manager_->is_bot());
  if (!top_thread_message_id.is_valid() && !is_broadcast_channel(dialog_id) &&
      is_active_message_reply_info(dialog_id, reply_info) && !message_id.is_scheduled()) {
    top_thread_message_id = message_id;
  }
  if (top_thread_message_id.is_valid() && dialog_type != DialogType::Channel) {
    top_thread_message_id = MessageId();
  }

  bool has_forward_info = message_info.forward_header != nullptr;

  if (sender_dialog_id.is_valid() && sender_dialog_id != dialog_id && have_dialog_info_force(sender_dialog_id)) {
    force_create_dialog(sender_dialog_id, "create_message", true);
  }

  LOG(INFO) << "Receive " << message_id << " in " << dialog_id << " from " << sender_user_id << "/" << sender_dialog_id;

  auto message = make_unique<Message>();
  set_message_id(message, message_id);
  message->sender_user_id = sender_user_id;
  message->sender_dialog_id = sender_dialog_id;
  message->date = date;
  message->ttl_period = ttl_period;
  message->ttl = ttl;
  message->disable_web_page_preview = message_info.disable_web_page_preview;
  message->edit_date = edit_date;
  message->random_id = message_info.random_id;
  message->forward_info = get_message_forward_info(std::move(message_info.forward_header));
  message->reply_to_message_id = reply_to_message_id;
  message->reply_in_dialog_id = reply_in_dialog_id;
  message->top_thread_message_id = top_thread_message_id;
  message->via_bot_user_id = via_bot_user_id;
  message->restriction_reasons = std::move(message_info.restriction_reasons);
  message->author_signature = std::move(message_info.author_signature);
  message->is_outgoing = is_outgoing;
  message->is_channel_post = is_channel_post;
  message->contains_mention =
      !is_outgoing && dialog_type != DialogType::User &&
      ((flags & MESSAGE_FLAG_HAS_MENTION) != 0 || content_type == MessageContentType::PinMessage);
  message->contains_unread_mention =
      !message_id.is_scheduled() && message_id.is_server() && message->contains_mention &&
      (flags & MESSAGE_FLAG_HAS_UNREAD_CONTENT) != 0 &&
      (dialog_type == DialogType::Chat || (dialog_type == DialogType::Channel && !is_broadcast_channel(dialog_id)));
  message->disable_notification = is_silent;
  message->is_content_secret = is_content_secret;
  message->hide_edit_date = hide_edit_date;
  message->is_from_scheduled = is_from_scheduled;
  message->is_pinned = is_pinned;
  message->view_count = view_count;
  message->forward_count = forward_count;
  message->reply_info = std::move(reply_info);
  message->legacy_layer = (is_legacy ? MTPROTO_LAYER : 0);
  message->content = std::move(message_info.content);
  message->reply_markup = get_reply_markup(std::move(message_info.reply_markup), td_->auth_manager_->is_bot(), false,
                                           message->contains_mention || dialog_type == DialogType::User);

  if (content_type == MessageContentType::ExpiredPhoto || content_type == MessageContentType::ExpiredVideo) {
    CHECK(message->ttl == 0);  // TTL is ignored/set to 0 if the message has already been expired
    if (message->reply_markup != nullptr) {
      if (message->reply_markup->type != ReplyMarkup::Type::InlineKeyboard) {
        message->had_reply_markup = true;
      }
      message->reply_markup = nullptr;
    }
    message->reply_to_message_id = MessageId();
    message->reply_in_dialog_id = DialogId();
    message->top_thread_message_id = MessageId();
    message->linked_top_thread_message_id = MessageId();
  }

  if (message_info.media_album_id != 0) {
    if (!is_allowed_media_group_content(content_type)) {
      LOG(ERROR) << "Receive media group identifier " << message_info.media_album_id << " in " << message_id << " from "
                 << dialog_id << " with content "
                 << oneline(to_string(get_message_content_object(message->content.get(), td_, dialog_id, message->date,
                                                                 is_content_secret, false, -1)));
    } else {
      message->media_album_id = message_info.media_album_id;
    }
  }

  if (message->forward_info == nullptr && has_forward_info) {
    message->had_forward_info = true;
  }

  return {dialog_id, std::move(message)};
}

MessageId MessagesManager::find_old_message_id(DialogId dialog_id, MessageId message_id) const {
  if (message_id.is_scheduled()) {
    CHECK(message_id.is_scheduled_server());
    auto dialog_it = update_scheduled_message_ids_.find(dialog_id);
    if (dialog_it != update_scheduled_message_ids_.end()) {
      auto it = dialog_it->second.find(message_id.get_scheduled_server_message_id());
      if (it != dialog_it->second.end()) {
        return it->second;
      }
    }
  } else {
    CHECK(message_id.is_server());
    auto it = update_message_ids_.find(FullMessageId(dialog_id, message_id));
    if (it != update_message_ids_.end()) {
      return it->second;
    }
  }
  return MessageId();
}

void MessagesManager::delete_update_message_id(DialogId dialog_id, MessageId message_id) {
  if (message_id.is_scheduled()) {
    CHECK(message_id.is_scheduled_server());
    auto dialog_it = update_scheduled_message_ids_.find(dialog_id);
    CHECK(dialog_it != update_scheduled_message_ids_.end());
    auto erased_count = dialog_it->second.erase(message_id.get_scheduled_server_message_id());
    CHECK(erased_count > 0);
    if (dialog_it->second.empty()) {
      update_scheduled_message_ids_.erase(dialog_it);
    }
  } else {
    CHECK(message_id.is_server());
    auto erased_count = update_message_ids_.erase(FullMessageId(dialog_id, message_id));
    CHECK(erased_count > 0);
  }
}

FullMessageId MessagesManager::on_get_message(tl_object_ptr<telegram_api::Message> message_ptr, bool from_update,
                                              bool is_channel_message, bool is_scheduled, bool have_previous,
                                              bool have_next, const char *source) {
  return on_get_message(parse_telegram_api_message(std::move(message_ptr), is_scheduled, source), from_update,
                        is_channel_message, have_previous, have_next, source);
}

FullMessageId MessagesManager::on_get_message(MessageInfo &&message_info, bool from_update, bool is_channel_message,
                                              bool have_previous, bool have_next, const char *source) {
  DialogId dialog_id;
  unique_ptr<Message> new_message;
  std::tie(dialog_id, new_message) = create_message(std::move(message_info), is_channel_message);
  if (new_message == nullptr) {
    return FullMessageId();
  }
  MessageId message_id = new_message->message_id;

  new_message->have_previous = have_previous;
  new_message->have_next = have_next;

  bool need_update = from_update;
  bool need_update_dialog_pos = false;

  MessageId old_message_id = find_old_message_id(dialog_id, message_id);
  bool is_sent_message = false;
  if (old_message_id.is_valid() || old_message_id.is_valid_scheduled()) {
    LOG(INFO) << "Found temporary " << old_message_id << " for " << FullMessageId{dialog_id, message_id};
    Dialog *d = get_dialog(dialog_id);
    CHECK(d != nullptr);

    if (!from_update && !message_id.is_scheduled()) {
      if (message_id <= d->last_new_message_id) {
        if (get_message_force(d, message_id, "receive missed unsent message not from update") != nullptr) {
          LOG(ERROR) << "New " << old_message_id << "/" << message_id << " in " << dialog_id << " from " << source
                     << " has identifier less than last_new_message_id = " << d->last_new_message_id;
          return FullMessageId();
        }
        // if there is no message yet, then it is likely was missed because of a server bug and is being repaired via
        // get_message_from_server from after_get_difference
        // TODO move to INFO
        LOG(ERROR) << "Receive " << old_message_id << "/" << message_id << " in " << dialog_id << " from " << source
                   << " with identifier less than last_new_message_id = " << d->last_new_message_id
                   << " and trying to add it anyway";
      } else {
        // TODO move to INFO
        LOG(ERROR) << "Ignore " << old_message_id << "/" << message_id << " received not through update from " << source
                   << ": " << oneline(to_string(get_message_object(dialog_id, new_message.get(), "on_get_message")));
        if (dialog_id.get_type() == DialogType::Channel && have_input_peer(dialog_id, AccessRights::Read)) {
          channel_get_difference_retry_timeout_.add_timeout_in(dialog_id.get(), 0.001);
        }
        return FullMessageId();
      }
    }

    delete_update_message_id(dialog_id, message_id);

    if (!new_message->is_outgoing && dialog_id != get_my_dialog_id()) {
      // sent message is not from me
      LOG(ERROR) << "Sent in " << dialog_id << " " << message_id << " is sent by " << new_message->sender_user_id << "/"
                 << new_message->sender_dialog_id;
      return FullMessageId();
    }

    being_readded_message_id_ = {dialog_id, old_message_id};
    unique_ptr<Message> old_message =
        delete_message(d, old_message_id, false, &need_update_dialog_pos, "add sent message");
    if (old_message == nullptr) {
      delete_sent_message_on_server(dialog_id, new_message->message_id);
      being_readded_message_id_ = FullMessageId();
      return FullMessageId();
    }
    old_message_id = old_message->message_id;

    need_update = false;

    if (old_message_id.is_valid() && message_id.is_valid() && message_id < old_message_id &&
        !can_overflow_message_id(dialog_id)) {
      LOG(ERROR) << "Sent " << old_message_id << " to " << dialog_id << " as " << message_id;
    }

    set_message_id(new_message, old_message_id);
    new_message->from_database = false;
    new_message->have_previous = false;
    new_message->have_next = false;
    update_message(d, old_message.get(), std::move(new_message), &need_update_dialog_pos, false);
    new_message = std::move(old_message);

    set_message_id(new_message, message_id);
    send_update_message_send_succeeded(d, old_message_id, new_message.get());

    if (!message_id.is_scheduled()) {
      is_sent_message = true;
    }

    if (!from_update) {
      new_message->have_previous = have_previous;
      new_message->have_next = have_next;
    } else {
      new_message->have_previous = true;
      new_message->have_next = true;
    }
  }

  const Message *m = add_message_to_dialog(dialog_id, std::move(new_message), from_update, &need_update,
                                           &need_update_dialog_pos, source);
  being_readded_message_id_ = FullMessageId();
  Dialog *d = get_dialog(dialog_id);
  if (m == nullptr) {
    if (need_update_dialog_pos && d != nullptr) {
      send_update_chat_last_message(d, "on_get_message");
    }
    if (old_message_id.is_valid() || old_message_id.is_valid_scheduled()) {
      CHECK(d != nullptr);
      if (!old_message_id.is_valid() || !message_id.is_valid() || old_message_id <= message_id) {
        LOG(ERROR) << "Failed to add just sent " << old_message_id << " to " << dialog_id << " as " << message_id
                   << " from " << source << ": " << debug_add_message_to_dialog_fail_reason_;
      }
      send_update_delete_messages(dialog_id, {message_id.get()}, true, false);
    }

    return FullMessageId();
  }

  CHECK(d != nullptr);

  auto pcc_it = pending_created_dialogs_.find(dialog_id);
  if (from_update && pcc_it != pending_created_dialogs_.end()) {
    pcc_it->second.set_value(Unit());

    pending_created_dialogs_.erase(pcc_it);
  }

  if (need_update) {
    send_update_new_message(d, m);
  }

  if (is_sent_message) {
    try_add_active_live_location(dialog_id, m);

    // add_message_to_dialog will not update counts, because need_update == false
    update_message_count_by_index(d, +1, m);
  }

  if (is_sent_message || (need_update && !message_id.is_scheduled())) {
    update_reply_count_by_message(d, +1, m);
    update_forward_count(dialog_id, m);
  }

  if (dialog_id.get_type() == DialogType::Channel && !have_input_peer(dialog_id, AccessRights::Read)) {
    auto p = delete_message(d, message_id, false, &need_update_dialog_pos, "get a message in inaccessible chat");
    CHECK(p.get() == m);
    // CHECK(d->messages == nullptr);
    send_update_delete_messages(dialog_id, {p->message_id.get()}, false, false);
    // don't need to update dialog pos
    return FullMessageId();
  }

  if (m->message_id.is_scheduled()) {
    send_update_chat_has_scheduled_messages(d, false);
  }

  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "on_get_message");
  }

  // set dialog reply markup only after updateNewMessage and updateChatLastMessage are sent
  if (need_update && m->reply_markup != nullptr && !m->message_id.is_scheduled() &&
      m->reply_markup->type != ReplyMarkup::Type::InlineKeyboard && m->reply_markup->is_personal &&
      !td_->auth_manager_->is_bot()) {
    set_dialog_reply_markup(d, message_id);
  }

  return FullMessageId(dialog_id, message_id);
}

void MessagesManager::set_dialog_last_message_id(Dialog *d, MessageId last_message_id, const char *source) {
  CHECK(!last_message_id.is_scheduled());

  LOG(INFO) << "Set " << d->dialog_id << " last message to " << last_message_id << " from " << source;
  d->last_message_id = last_message_id;

  if (!last_message_id.is_valid()) {
    d->suffix_load_first_message_id_ = MessageId();
    d->suffix_load_done_ = false;
  }
  if (last_message_id.is_valid() && d->delete_last_message_date != 0) {
    d->delete_last_message_date = 0;
    d->deleted_last_message_id = MessageId();
    d->is_last_message_deleted_locally = false;
    on_dialog_updated(d->dialog_id, "update_delete_last_message_date");
  }
  if (d->pending_last_message_date != 0) {
    d->pending_last_message_date = 0;
    d->pending_last_message_id = MessageId();
  }
}

void MessagesManager::set_dialog_first_database_message_id(Dialog *d, MessageId first_database_message_id,
                                                           const char *source) {
  CHECK(!first_database_message_id.is_scheduled());
  if (first_database_message_id == d->first_database_message_id) {
    return;
  }

  LOG(INFO) << "Set " << d->dialog_id << " first database message to " << first_database_message_id << " from "
            << source;
  d->first_database_message_id = first_database_message_id;
  on_dialog_updated(d->dialog_id, "set_dialog_first_database_message_id");
}

void MessagesManager::set_dialog_last_database_message_id(Dialog *d, MessageId last_database_message_id,
                                                          const char *source, bool is_loaded_from_database) {
  CHECK(!last_database_message_id.is_scheduled());
  if (last_database_message_id == d->last_database_message_id) {
    return;
  }

  LOG(INFO) << "Set " << d->dialog_id << " last database message to " << last_database_message_id << " from " << source;
  d->debug_set_dialog_last_database_message_id = source;
  d->last_database_message_id = last_database_message_id;
  if (!is_loaded_from_database) {
    on_dialog_updated(d->dialog_id, "set_dialog_last_database_message_id");
  }
}

void MessagesManager::remove_dialog_newer_messages(Dialog *d, MessageId from_message_id, const char *source) {
  LOG(INFO) << "Remove messages in " << d->dialog_id << " newer than " << from_message_id << " from " << source;
  CHECK(!d->last_new_message_id.is_valid());

  delete_all_dialog_messages_from_database(d, MessageId::max(), "remove_dialog_newer_messages");
  set_dialog_first_database_message_id(d, MessageId(), "remove_dialog_newer_messages");
  set_dialog_last_database_message_id(d, MessageId(), source);
  if (d->dialog_id.get_type() != DialogType::SecretChat && !d->is_empty) {
    d->have_full_history = false;
  }
  invalidate_message_indexes(d);

  vector<MessageId> to_delete_message_ids;
  find_newer_messages(d->messages.get(), from_message_id, to_delete_message_ids);
  td::remove_if(to_delete_message_ids, [](MessageId message_id) { return message_id.is_yet_unsent(); });
  if (!to_delete_message_ids.empty()) {
    LOG(INFO) << "Delete " << format::as_array(to_delete_message_ids) << " newer than " << from_message_id << " in "
              << d->dialog_id << " from " << source;

    vector<int64> deleted_message_ids;
    bool need_update_dialog_pos = false;
    for (auto message_id : to_delete_message_ids) {
      auto message = delete_message(d, message_id, false, &need_update_dialog_pos, "remove_dialog_newer_messages");
      if (message != nullptr) {
        deleted_message_ids.push_back(message->message_id.get());
      }
    }
    if (need_update_dialog_pos) {
      send_update_chat_last_message(d, "remove_dialog_newer_messages");
    }
    send_update_delete_messages(d->dialog_id, std::move(deleted_message_ids), false, false);
  }
}

void MessagesManager::set_dialog_last_new_message_id(Dialog *d, MessageId last_new_message_id, const char *source) {
  CHECK(!last_new_message_id.is_scheduled());

  LOG_CHECK(last_new_message_id > d->last_new_message_id)
      << last_new_message_id << " " << d->last_new_message_id << " " << source;
  CHECK(d->dialog_id.get_type() == DialogType::SecretChat || last_new_message_id.is_server());
  if (!d->last_new_message_id.is_valid()) {
    remove_dialog_newer_messages(d, last_new_message_id, source);

    auto last_new_message = get_message(d, last_new_message_id);
    if (last_new_message != nullptr) {
      add_message_to_database(d, last_new_message, "set_dialog_last_new_message_id");
      set_dialog_first_database_message_id(d, last_new_message_id, "set_dialog_last_new_message_id");
      set_dialog_last_database_message_id(d, last_new_message_id, "set_dialog_last_new_message_id");
      try_restore_dialog_reply_markup(d, last_new_message);
    }
  }

  LOG(INFO) << "Set " << d->dialog_id << " last new message to " << last_new_message_id << " from " << source;
  d->last_new_message_id = last_new_message_id;
  on_dialog_updated(d->dialog_id, source);
}

void MessagesManager::set_dialog_last_clear_history_date(Dialog *d, int32 date, MessageId last_clear_history_message_id,
                                                         const char *source, bool is_loaded_from_database) {
  CHECK(!last_clear_history_message_id.is_scheduled());

  if (d->last_clear_history_message_id == last_clear_history_message_id && d->last_clear_history_date == date) {
    return;
  }

  LOG(INFO) << "Set " << d->dialog_id << " last clear history date to " << date << " of "
            << last_clear_history_message_id << " from " << source;
  if (d->last_clear_history_message_id.is_valid()) {
    switch (d->dialog_id.get_type()) {
      case DialogType::User:
      case DialogType::Chat:
        last_clear_history_message_id_to_dialog_id_.erase(d->last_clear_history_message_id);
        break;
      case DialogType::Channel:
      case DialogType::SecretChat:
        // nothing to do
        break;
      case DialogType::None:
      default:
        UNREACHABLE();
    }
  }

  d->last_clear_history_date = date;
  d->last_clear_history_message_id = last_clear_history_message_id;
  if (!is_loaded_from_database) {
    on_dialog_updated(d->dialog_id, "set_dialog_last_clear_history_date");
  }

  if (d->last_clear_history_message_id.is_valid()) {
    switch (d->dialog_id.get_type()) {
      case DialogType::User:
      case DialogType::Chat:
        last_clear_history_message_id_to_dialog_id_[d->last_clear_history_message_id] = d->dialog_id;
        break;
      case DialogType::Channel:
      case DialogType::SecretChat:
        // nothing to do
        break;
      case DialogType::None:
      default:
        UNREACHABLE();
    }
  }
}

void MessagesManager::set_dialog_unread_mention_count(Dialog *d, int32 unread_mention_count) {
  CHECK(d->unread_mention_count != unread_mention_count);
  CHECK(unread_mention_count >= 0);

  d->unread_mention_count = unread_mention_count;
  d->message_count_by_index[message_search_filter_index(MessageSearchFilter::UnreadMention)] = unread_mention_count;
}

void MessagesManager::set_dialog_is_empty(Dialog *d, const char *source) {
  LOG(INFO) << "Set " << d->dialog_id << " is_empty to true from " << source;
  CHECK(d->have_full_history);
  d->is_empty = true;

  if (d->server_unread_count + d->local_unread_count > 0) {
    MessageId max_message_id =
        d->last_database_message_id.is_valid() ? d->last_database_message_id : d->last_new_message_id;
    if (max_message_id.is_valid()) {
      read_history_inbox(d->dialog_id, max_message_id, -1, "set_dialog_is_empty");
    }
    if (d->server_unread_count != 0 || d->local_unread_count != 0) {
      set_dialog_last_read_inbox_message_id(d, MessageId::min(), 0, 0, true, "set_dialog_is_empty");
    }
  }
  if (d->unread_mention_count > 0) {
    set_dialog_unread_mention_count(d, 0);
    send_update_chat_unread_mention_count(d);
  }
  if (d->reply_markup_message_id != MessageId()) {
    set_dialog_reply_markup(d, MessageId());
  }
  std::fill(d->message_count_by_index.begin(), d->message_count_by_index.end(), 0);
  d->notification_id_to_message_id.clear();

  if (d->delete_last_message_date != 0) {
    if (d->is_last_message_deleted_locally && d->last_clear_history_date == 0) {
      set_dialog_last_clear_history_date(d, d->delete_last_message_date, d->deleted_last_message_id,
                                         "set_dialog_is_empty");
    }
    d->delete_last_message_date = 0;
    d->deleted_last_message_id = MessageId();
    d->is_last_message_deleted_locally = false;

    on_dialog_updated(d->dialog_id, "set_dialog_is_empty");
  }
  if (d->pending_last_message_date != 0) {
    d->pending_last_message_date = 0;
    d->pending_last_message_id = MessageId();
  }
  if (d->last_database_message_id.is_valid()) {
    set_dialog_first_database_message_id(d, MessageId(), "set_dialog_is_empty");
    set_dialog_last_database_message_id(d, MessageId(), "set_dialog_is_empty");
  }

  update_dialog_pos(d, source);
}

bool MessagesManager::is_dialog_pinned(DialogListId dialog_list_id, DialogId dialog_id) const {
  if (get_dialog_pinned_order(dialog_list_id, dialog_id) != DEFAULT_ORDER) {
    return true;
  }
  if (dialog_list_id.is_filter()) {
    const auto *filter = get_dialog_filter(dialog_list_id.get_filter_id());
    if (filter != nullptr) {
      for (const auto &input_dialog_id : filter->pinned_dialog_ids) {
        if (input_dialog_id.get_dialog_id() == dialog_id) {
          return true;
        }
      }
    }
  }
  return false;
}

int64 MessagesManager::get_dialog_pinned_order(DialogListId dialog_list_id, DialogId dialog_id) const {
  return get_dialog_pinned_order(get_dialog_list(dialog_list_id), dialog_id);
}

int64 MessagesManager::get_dialog_pinned_order(const DialogList *list, DialogId dialog_id) {
  if (list != nullptr && !list->pinned_dialog_id_orders_.empty()) {
    auto it = list->pinned_dialog_id_orders_.find(dialog_id);
    if (it != list->pinned_dialog_id_orders_.end()) {
      return it->second;
    }
  }
  return DEFAULT_ORDER;
}

bool MessagesManager::set_dialog_is_pinned(DialogId dialog_id, bool is_pinned) {
  if (td_->auth_manager_->is_bot()) {
    return false;
  }

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  return set_dialog_is_pinned(DialogListId(d->folder_id), d, is_pinned);
}

// only removes the Dialog from the dialog list, but changes nothing in the corresponding DialogFilter
bool MessagesManager::set_dialog_is_pinned(DialogListId dialog_list_id, Dialog *d, bool is_pinned,
                                           bool need_update_dialog_lists) {
  if (td_->auth_manager_->is_bot()) {
    return false;
  }

  CHECK(d != nullptr);
  if (d->order == DEFAULT_ORDER && is_pinned) {
    // the chat can't be pinned
    return false;
  }

  auto positions = get_dialog_positions(d);
  auto *list = get_dialog_list(dialog_list_id);
  if (list == nullptr) {
    return false;
  }
  if (!list->are_pinned_dialogs_inited_) {
    return false;
  }
  bool was_pinned = false;
  for (size_t pos = 0; pos < list->pinned_dialogs_.size(); pos++) {
    auto &pinned_dialog = list->pinned_dialogs_[pos];
    if (pinned_dialog.get_dialog_id() == d->dialog_id) {
      // the dialog was already pinned
      if (is_pinned) {
        if (pos == 0) {
          return false;
        }
        auto order = get_next_pinned_dialog_order();
        pinned_dialog = DialogDate(order, d->dialog_id);
        std::rotate(list->pinned_dialogs_.begin(), list->pinned_dialogs_.begin() + pos,
                    list->pinned_dialogs_.begin() + pos + 1);
        list->pinned_dialog_id_orders_[d->dialog_id] = order;
      } else {
        list->pinned_dialogs_.erase(list->pinned_dialogs_.begin() + pos);
        list->pinned_dialog_id_orders_.erase(d->dialog_id);
      }
      was_pinned = true;
      break;
    }
  }
  if (!was_pinned) {
    if (!is_pinned) {
      return false;
    }
    auto order = get_next_pinned_dialog_order();
    list->pinned_dialogs_.insert(list->pinned_dialogs_.begin(), {order, d->dialog_id});
    list->pinned_dialog_id_orders_.emplace(d->dialog_id, order);
  }

  LOG(INFO) << "Set " << d->dialog_id << " is pinned in " << dialog_list_id << " to " << is_pinned;
  if (dialog_list_id.is_folder() && G()->parameters().use_message_db) {
    G()->td_db()->get_binlog_pmc()->set(
        PSTRING() << "pinned_dialog_ids" << dialog_list_id.get_folder_id().get(),
        implode(transform(list->pinned_dialogs_,
                          [](auto &pinned_dialog) { return PSTRING() << pinned_dialog.get_dialog_id().get(); }),
                ','));
  }

  if (need_update_dialog_lists) {
    update_dialog_lists(d, std::move(positions), true, false, "set_dialog_is_pinned");
  }
  return true;
}

void MessagesManager::set_dialog_reply_markup(Dialog *d, MessageId message_id) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(!message_id.is_scheduled());

  if (d->reply_markup_message_id != message_id) {
    on_dialog_updated(d->dialog_id, "set_dialog_reply_markup");
  }

  d->need_restore_reply_markup = false;

  if (d->reply_markup_message_id.is_valid() || message_id.is_valid()) {
    LOG_CHECK(d->is_update_new_chat_sent) << "Wrong " << d->dialog_id << " in set_dialog_reply_markup";
    d->reply_markup_message_id = message_id;
    send_closure(G()->td(), &Td::send_update,
                 make_tl_object<td_api::updateChatReplyMarkup>(d->dialog_id.get(), message_id.get()));
  }
}

void MessagesManager::try_restore_dialog_reply_markup(Dialog *d, const Message *m) {
  if (!d->need_restore_reply_markup || td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(!m->message_id.is_scheduled());
  if (m->had_reply_markup) {
    LOG(INFO) << "Restore deleted reply markup in " << d->dialog_id;
    set_dialog_reply_markup(d, MessageId());
  } else if (m->reply_markup != nullptr && m->reply_markup->type != ReplyMarkup::Type::InlineKeyboard &&
             m->reply_markup->is_personal) {
    LOG(INFO) << "Restore reply markup in " << d->dialog_id << " to " << m->message_id;
    set_dialog_reply_markup(d, m->message_id);
  }
}

void MessagesManager::set_dialog_pinned_message_notification(Dialog *d, MessageId message_id, const char *source) {
  CHECK(d != nullptr);
  CHECK(!message_id.is_scheduled());
  auto old_message_id = d->pinned_message_notification_message_id;
  if (old_message_id == message_id) {
    return;
  }
  VLOG(notifications) << "Change pinned message notification in " << d->dialog_id << " from " << old_message_id
                      << " to " << message_id;
  if (old_message_id.is_valid()) {
    auto m = get_message_force(d, old_message_id, source);
    if (m != nullptr && m->notification_id.is_valid() && is_message_notification_active(d, m)) {
      // Can't remove pinned_message_notification_message_id  before the call,
      // because the notification needs to be still active inside remove_message_notification_id
      remove_message_notification_id(d, m, true, false, true);
      on_message_changed(d, m, false, source);
    } else {
      send_closure_later(G()->notification_manager(), &NotificationManager::remove_temporary_notification_by_message_id,
                         d->mention_notification_group.group_id, old_message_id, false, source);
    }
  }
  d->pinned_message_notification_message_id = message_id;
  on_dialog_updated(d->dialog_id, source);
}

void MessagesManager::remove_dialog_pinned_message_notification(Dialog *d, const char *source) {
  set_dialog_pinned_message_notification(d, MessageId(), source);
}

void MessagesManager::remove_dialog_mention_notifications(Dialog *d) {
  auto notification_group_id = d->mention_notification_group.group_id;
  if (!notification_group_id.is_valid()) {
    return;
  }
  if (d->unread_mention_count == 0) {
    return;
  }
  CHECK(!d->being_added_message_id.is_valid());

  VLOG(notifications) << "Remove mention notifications in " << d->dialog_id;

  vector<MessageId> message_ids;
  std::unordered_set<NotificationId, NotificationIdHash> removed_notification_ids_set;
  find_messages(d->messages.get(), message_ids, [](const Message *m) { return m->contains_unread_mention; });
  VLOG(notifications) << "Found unread mentions in " << message_ids;
  for (auto &message_id : message_ids) {
    auto m = get_message(d, message_id);
    CHECK(m != nullptr);
    CHECK(m->message_id.is_valid());
    if (m->notification_id.is_valid() && is_message_notification_active(d, m) &&
        is_from_mention_notification_group(d, m)) {
      removed_notification_ids_set.insert(m->notification_id);
    }
  }

  message_ids = td_->notification_manager_->get_notification_group_message_ids(notification_group_id);
  VLOG(notifications) << "Found active mention notifications in " << message_ids;
  for (auto &message_id : message_ids) {
    CHECK(!message_id.is_scheduled());
    if (message_id != d->pinned_message_notification_message_id) {
      auto m = get_message_force(d, message_id, "remove_dialog_mention_notifications");
      if (m != nullptr && m->notification_id.is_valid() && is_message_notification_active(d, m)) {
        CHECK(is_from_mention_notification_group(d, m));
        removed_notification_ids_set.insert(m->notification_id);
      }
    }
  }

  vector<NotificationId> removed_notification_ids(removed_notification_ids_set.begin(),
                                                  removed_notification_ids_set.end());
  for (size_t i = 0; i < removed_notification_ids.size(); i++) {
    send_closure_later(G()->notification_manager(), &NotificationManager::remove_notification, notification_group_id,
                       removed_notification_ids[i], false, i + 1 == removed_notification_ids.size(), Promise<Unit>(),
                       "remove_dialog_mention_notifications");
  }
}

bool MessagesManager::set_dialog_last_notification(DialogId dialog_id, NotificationGroupInfo &group_info,
                                                   int32 last_notification_date, NotificationId last_notification_id,
                                                   const char *source) {
  if (group_info.last_notification_date != last_notification_date ||
      group_info.last_notification_id != last_notification_id) {
    VLOG(notifications) << "Set " << group_info.group_id << '/' << dialog_id << " last notification to "
                        << last_notification_id << " sent at " << last_notification_date << " from " << source;
    group_info.last_notification_date = last_notification_date;
    group_info.last_notification_id = last_notification_id;
    group_info.is_changed = true;
    on_dialog_updated(dialog_id, "set_dialog_last_notification");
    return true;
  }
  return false;
}

void MessagesManager::on_update_sent_text_message(int64 random_id,
                                                  tl_object_ptr<telegram_api::MessageMedia> message_media,
                                                  vector<tl_object_ptr<telegram_api::MessageEntity>> &&entities) {
  int32 message_media_id = message_media == nullptr ? telegram_api::messageMediaEmpty::ID : message_media->get_id();
  LOG_IF(ERROR, message_media_id != telegram_api::messageMediaWebPage::ID &&
                    message_media_id != telegram_api::messageMediaEmpty::ID)
      << "Receive non web-page media for text message: " << oneline(to_string(message_media));

  auto it = being_sent_messages_.find(random_id);
  if (it == being_sent_messages_.end()) {
    // result of sending message has already been received through getDifference
    return;
  }

  auto full_message_id = it->second;
  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog(dialog_id);
  auto m = get_message_force(d, full_message_id.get_message_id(), "on_update_sent_text_message");
  if (m == nullptr) {
    // message has already been deleted
    return;
  }
  CHECK(m->message_id.is_yet_unsent());
  full_message_id = FullMessageId(dialog_id, m->message_id);

  if (m->content->get_type() != MessageContentType::Text) {
    LOG(ERROR) << "Text message content has been already changed to " << m->content->get_type();
    return;
  }

  const FormattedText *old_message_text = get_message_content_text(m->content.get());
  CHECK(old_message_text != nullptr);
  FormattedText new_message_text = get_message_text(
      td_->contacts_manager_.get(), old_message_text->text, std::move(entities), true, td_->auth_manager_->is_bot(),
      m->forward_info ? m->forward_info->date : m->date, m->media_album_id != 0, "on_update_sent_text_message");
  auto new_content =
      get_message_content(td_, std::move(new_message_text), std::move(message_media), dialog_id,
                          true /*likely ignored*/, UserId() /*likely ignored*/, nullptr /*ignored*/, nullptr);
  if (new_content->get_type() != MessageContentType::Text) {
    LOG(ERROR) << "Text message content has changed to " << new_content->get_type();
    return;
  }

  bool need_update = false;
  bool is_content_changed = false;
  merge_message_contents(td_, m->content.get(), new_content.get(), need_message_changed_warning(m), dialog_id, false,
                         is_content_changed, need_update);

  if (is_content_changed || need_update) {
    reregister_message_content(td_, m->content.get(), new_content.get(), full_message_id,
                               "on_update_sent_text_message");
    m->content = std::move(new_content);
    m->is_content_secret = is_secret_message_content(m->ttl, MessageContentType::Text);
  }
  if (need_update) {
    send_update_message_content(dialog_id, m, true, "on_update_sent_text_message");
    if (m->message_id == d->last_message_id) {
      send_update_chat_last_message_impl(d, "on_update_sent_text_message");
    }
  }
}

void MessagesManager::delete_pending_message_web_page(FullMessageId full_message_id) {
  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  Message *m = get_message(d, full_message_id.get_message_id());
  CHECK(m != nullptr);

  MessageContent *content = m->content.get();
  CHECK(has_message_content_web_page(content));
  unregister_message_content(td_, content, full_message_id, "delete_pending_message_web_page");
  remove_message_content_web_page(content);
  register_message_content(td_, content, full_message_id, "delete_pending_message_web_page");

  // don't need to send an updateMessageContent, because the web page was pending

  on_message_changed(d, m, false, "delete_pending_message_web_page");
}

void MessagesManager::on_get_dialogs(FolderId folder_id, vector<tl_object_ptr<telegram_api::Dialog>> &&dialog_folders,
                                     int32 total_count, vector<tl_object_ptr<telegram_api::Message>> &&messages,
                                     Promise<Unit> &&promise) {
  if (td_->updates_manager_->running_get_difference()) {
    LOG(INFO) << "Postpone result of getDialogs";
    pending_on_get_dialogs_.push_back(PendingOnGetDialogs{folder_id, std::move(dialog_folders), total_count,
                                                          std::move(messages), std::move(promise)});
    return;
  }
  bool from_dialog_list = total_count >= 0;
  bool from_get_dialog = total_count == -1;
  bool from_pinned_dialog_list = total_count == -2;

  if (from_get_dialog && dialog_folders.size() == 1 && dialog_folders[0]->get_id() == telegram_api::dialog::ID) {
    DialogId dialog_id(static_cast<const telegram_api::dialog *>(dialog_folders[0].get())->peer_);
    if (running_get_channel_difference(dialog_id)) {
      LOG(INFO) << "Postpone result of channels getDialogs for " << dialog_id;
      pending_channel_on_get_dialogs_.emplace(
          dialog_id, PendingOnGetDialogs{folder_id, std::move(dialog_folders), total_count, std::move(messages),
                                         std::move(promise)});
      return;
    }
  }

  vector<tl_object_ptr<telegram_api::dialog>> dialogs;
  for (auto &dialog_folder : dialog_folders) {
    switch (dialog_folder->get_id()) {
      case telegram_api::dialog::ID:
        dialogs.push_back(telegram_api::move_object_as<telegram_api::dialog>(dialog_folder));
        break;
      case telegram_api::dialogFolder::ID: {
        auto folder = telegram_api::move_object_as<telegram_api::dialogFolder>(dialog_folder);
        if (from_pinned_dialog_list) {
          // TODO update unread_muted_peers_count:int unread_unmuted_peers_count:int
          // unread_muted_messages_count:int unread_unmuted_messages_count:int
          FolderId folder_folder_id(folder->folder_->id_);
          if (folder_folder_id == FolderId::archive()) {
            // archive is expected in pinned dialogs list
            break;
          }
        }
        LOG(ERROR) << "Receive unexpected " << to_string(folder);
        break;
      }
      default:
        UNREACHABLE();
    }
  }

  if (from_get_dialog) {
    LOG(INFO) << "Process " << dialogs.size() << " chats";
  } else if (from_pinned_dialog_list) {
    LOG(INFO) << "Process " << dialogs.size() << " pinned chats in " << folder_id;
  } else {
    LOG(INFO) << "Process " << dialogs.size() << " chats out of " << total_count << " in " << folder_id;
  }
  std::unordered_map<FullMessageId, DialogDate, FullMessageIdHash> full_message_id_to_dialog_date;
  std::unordered_map<FullMessageId, tl_object_ptr<telegram_api::Message>, FullMessageIdHash> full_message_id_to_message;
  for (auto &message : messages) {
    auto full_message_id = get_full_message_id(message, false);
    if (from_dialog_list) {
      auto message_date = get_message_date(message);
      int64 order = get_dialog_order(full_message_id.get_message_id(), message_date);
      full_message_id_to_dialog_date.emplace(full_message_id, DialogDate(order, full_message_id.get_dialog_id()));
    }
    full_message_id_to_message[full_message_id] = std::move(message);
  }

  DialogDate max_dialog_date = MIN_DIALOG_DATE;
  for (auto &dialog : dialogs) {
    //    LOG(INFO) << to_string(dialog);
    DialogId dialog_id(dialog->peer_);
    bool has_pts = (dialog->flags_ & DIALOG_FLAG_HAS_PTS) != 0;

    if (!dialog_id.is_valid()) {
      LOG(ERROR) << "Receive wrong " << dialog_id;
      return promise.set_error(Status::Error(500, "Wrong query result returned: receive wrong chat identifier"));
    }
    switch (dialog_id.get_type()) {
      case DialogType::User:
      case DialogType::Chat:
        if (has_pts) {
          LOG(ERROR) << "Receive user or group " << dialog_id << " with pts";
          return promise.set_error(
              Status::Error(500, "Wrong query result returned: receive user or basic group chat with pts"));
        }
        break;
      case DialogType::Channel:
        if (!has_pts) {
          LOG(ERROR) << "Receive channel " << dialog_id << " without pts";
          return promise.set_error(
              Status::Error(500, "Wrong query result returned: receive supergroup chat without pts"));
        }
        break;
      case DialogType::SecretChat:
      case DialogType::None:
      default:
        UNREACHABLE();
        return promise.set_error(Status::Error(500, "UNREACHABLE"));
    }

    if (from_dialog_list) {
      MessageId last_message_id(ServerMessageId(dialog->top_message_));
      if (last_message_id.is_valid()) {
        FullMessageId full_message_id(dialog_id, last_message_id);
        auto it = full_message_id_to_dialog_date.find(full_message_id);
        if (it == full_message_id_to_dialog_date.end()) {
          LOG(ERROR) << "Last " << last_message_id << " in " << dialog_id << " not found";
          return promise.set_error(Status::Error(500, "Wrong query result returned: last message not found"));
        }
        FolderId dialog_folder_id((dialog->flags_ & DIALOG_FLAG_HAS_FOLDER_ID) != 0 ? dialog->folder_id_ : 0);
        if (dialog_folder_id != folder_id) {
          LOG(ERROR) << "Receive " << dialog_id << " in " << dialog_folder_id << " instead of " << folder_id;
          continue;
        }

        DialogDate dialog_date = it->second;
        CHECK(dialog_date.get_dialog_id() == dialog_id);

        if (max_dialog_date < dialog_date) {
          max_dialog_date = dialog_date;
        }
      } else {
        LOG(ERROR) << "Receive " << last_message_id << " as last chat message";
        continue;
      }
    }
  }

  if (from_dialog_list && total_count < narrow_cast<int32>(dialogs.size())) {
    LOG(ERROR) << "Receive chat total_count = " << total_count << ", but " << dialogs.size() << " chats";
    total_count = narrow_cast<int32>(dialogs.size());
  }

  vector<DialogId> added_dialog_ids;
  for (auto &dialog : dialogs) {
    MessageId last_message_id(ServerMessageId(dialog->top_message_));
    if (!last_message_id.is_valid() && from_dialog_list) {
      // skip dialogs without messages
      total_count--;
      continue;
    }

    DialogId dialog_id(dialog->peer_);
    if (td::contains(added_dialog_ids, dialog_id)) {
      LOG(ERROR) << "Receive " << dialog_id << " twice in result of getChats with total_count = " << total_count;
      continue;
    }
    added_dialog_ids.push_back(dialog_id);
    Dialog *d = get_dialog_force(dialog_id, "on_get_dialogs");
    bool need_update_dialog_pos = false;
    CHECK(!being_added_dialog_id_.is_valid());
    being_added_dialog_id_ = dialog_id;
    if (d == nullptr) {
      d = add_dialog(dialog_id, "on_get_dialogs");
      need_update_dialog_pos = true;
    } else {
      LOG(INFO) << "Receive already created " << dialog_id;
      CHECK(d->dialog_id == dialog_id);
    }
    bool is_new = d->last_new_message_id == MessageId();
    auto positions = get_dialog_positions(d);

    set_dialog_folder_id(d, FolderId((dialog->flags_ & DIALOG_FLAG_HAS_FOLDER_ID) != 0 ? dialog->folder_id_ : 0));

    on_update_dialog_notify_settings(dialog_id, std::move(dialog->notify_settings_), "on_get_dialogs");
    if (!d->notification_settings.is_synchronized && !td_->auth_manager_->is_bot()) {
      LOG(ERROR) << "Failed to synchronize settings in " << dialog_id;
      d->notification_settings.is_synchronized = true;
      on_dialog_updated(dialog_id, "set notification_settings.is_synchronized");
    }

    if (dialog->unread_count_ < 0) {
      LOG(ERROR) << "Receive " << dialog->unread_count_ << " as number of unread messages in " << dialog_id;
      dialog->unread_count_ = 0;
    }
    MessageId read_inbox_max_message_id = MessageId(ServerMessageId(dialog->read_inbox_max_id_));
    if (!read_inbox_max_message_id.is_valid() && read_inbox_max_message_id != MessageId()) {
      LOG(ERROR) << "Receive " << read_inbox_max_message_id << " as last read inbox message in " << dialog_id;
      read_inbox_max_message_id = MessageId();
    }
    MessageId read_outbox_max_message_id = MessageId(ServerMessageId(dialog->read_outbox_max_id_));
    if (!read_outbox_max_message_id.is_valid() && read_outbox_max_message_id != MessageId()) {
      LOG(ERROR) << "Receive " << read_outbox_max_message_id << " as last read outbox message in " << dialog_id;
      read_outbox_max_message_id = MessageId();
    }
    if (dialog->unread_mentions_count_ < 0) {
      LOG(ERROR) << "Receive " << dialog->unread_mentions_count_ << " as number of unread mention messages in "
                 << dialog_id;
      dialog->unread_mentions_count_ = 0;
    }
    if (!d->is_is_blocked_inited && !td_->auth_manager_->is_bot()) {
      // asynchronously get is_blocked from the server
      // TODO add is_blocked to telegram_api::dialog
      get_dialog_info_full(dialog_id, Auto(), "on_get_dialogs init is_blocked");
    } else if (!d->is_has_bots_inited && !td_->auth_manager_->is_bot()) {
      // asynchronously get has_bots from the server
      // TODO add has_bots to telegram_api::dialog
      get_dialog_info_full(dialog_id, Auto(), "on_get_dialogs init has_bots");
    } else if (!d->is_theme_name_inited && !td_->auth_manager_->is_bot()) {
      // asynchronously get theme_name from the server
      // TODO add theme_name to telegram_api::dialog
      get_dialog_info_full(dialog_id, Auto(), "on_get_dialogs init theme_name");
    } else if (!d->is_last_pinned_message_id_inited && !td_->auth_manager_->is_bot()) {
      // asynchronously get dialog pinned message from the server
      get_dialog_pinned_message(dialog_id, Auto());
    }

    need_update_dialog_pos |= update_dialog_draft_message(
        d, get_draft_message(td_->contacts_manager_.get(), std::move(dialog->draft_)), true, false);
    if (is_new) {
      bool has_pts = (dialog->flags_ & DIALOG_FLAG_HAS_PTS) != 0;
      if (last_message_id.is_valid()) {
        FullMessageId full_message_id(dialog_id, last_message_id);
        auto last_message = std::move(full_message_id_to_message[full_message_id]);
        if (last_message == nullptr) {
          LOG(ERROR) << "Last " << full_message_id << " not found";
        } else if (!has_pts || d->pts == 0 || dialog->pts_ <= d->pts || d->is_channel_difference_finished) {
          auto added_full_message_id =
              on_get_message(std::move(last_message), false, has_pts, false, false, false, "get chats");
          CHECK(d->last_new_message_id == MessageId());
          set_dialog_last_new_message_id(d, last_message_id, "on_get_dialogs");
          if (d->last_new_message_id > d->last_message_id && added_full_message_id.get_message_id().is_valid()) {
            CHECK(added_full_message_id.get_message_id() == d->last_new_message_id);
            set_dialog_last_message_id(d, d->last_new_message_id, "on_get_dialogs");
            send_update_chat_last_message(d, "on_get_dialogs");
          }
        } else {
          get_channel_difference(dialog_id, d->pts, true, "on_get_dialogs");
        }
      }

      if (has_pts && !running_get_channel_difference(dialog_id)) {
        set_channel_pts(d, dialog->pts_, "get channel");
      }
    }
    bool is_marked_as_unread = dialog->unread_mark_;
    if (is_marked_as_unread != d->is_marked_as_unread) {
      set_dialog_is_marked_as_unread(d, is_marked_as_unread);
    }

    if (need_update_dialog_pos) {
      update_dialog_pos(d, "on_get_dialogs");
    }

    if (!td_->auth_manager_->is_bot() && !from_pinned_dialog_list) {
      // set is_pinned only after updating dialog pos to ensure that order is initialized
      bool is_pinned = (dialog->flags_ & DIALOG_FLAG_IS_PINNED) != 0;
      bool was_pinned = is_dialog_pinned(DialogListId(d->folder_id), dialog_id);
      if (is_pinned != was_pinned) {
        set_dialog_is_pinned(DialogListId(d->folder_id), d, is_pinned);
      }
    }

    if (!G()->parameters().use_message_db || is_new || !d->is_last_read_inbox_message_id_inited ||
        d->need_repair_server_unread_count) {
      if (d->last_read_inbox_message_id.is_valid() && !d->last_read_inbox_message_id.is_server() &&
          read_inbox_max_message_id == d->last_read_inbox_message_id.get_prev_server_message_id()) {
        read_inbox_max_message_id = d->last_read_inbox_message_id;
      }
      if (d->need_repair_server_unread_count &&
          (d->last_read_inbox_message_id <= read_inbox_max_message_id || !need_unread_counter(d->order) ||
           !have_input_peer(dialog_id, AccessRights::Read))) {
        LOG(INFO) << "Repaired server unread count in " << dialog_id << " from " << d->last_read_inbox_message_id << "/"
                  << d->server_unread_count << " to " << read_inbox_max_message_id << "/" << dialog->unread_count_;
        d->need_repair_server_unread_count = false;
        on_dialog_updated(dialog_id, "repaired dialog server unread count");
      }
      if (d->need_repair_server_unread_count) {
        auto &previous_message_id = previous_repaired_read_inbox_max_message_id_[dialog_id];
        if (previous_message_id >= read_inbox_max_message_id) {
          // protect from sending the request in a loop
          LOG(ERROR) << "Failed to repair server unread count in " << dialog_id
                     << ", because receive read_inbox_max_message_id = " << read_inbox_max_message_id << " after "
                     << previous_message_id << ", but messages are read up to " << d->last_read_inbox_message_id;
          d->need_repair_server_unread_count = false;
          on_dialog_updated(dialog_id, "failed to repair dialog server unread count");
        } else {
          LOG(INFO) << "Have last_read_inbox_message_id = " << d->last_read_inbox_message_id << ", but received only "
                    << read_inbox_max_message_id << " from the server, trying to repair server unread count again";
          previous_message_id = read_inbox_max_message_id;
          repair_server_unread_count(dialog_id, d->server_unread_count);
        }
      }
      if (!d->need_repair_server_unread_count) {
        previous_repaired_read_inbox_max_message_id_.erase(dialog_id);
      }
      if ((d->server_unread_count != dialog->unread_count_ &&
           d->last_read_inbox_message_id == read_inbox_max_message_id) ||
          d->last_read_inbox_message_id < read_inbox_max_message_id) {
        set_dialog_last_read_inbox_message_id(d, read_inbox_max_message_id, dialog->unread_count_,
                                              d->local_unread_count, true, "on_get_dialogs");
      }
      if (!d->is_last_read_inbox_message_id_inited) {
        d->is_last_read_inbox_message_id_inited = true;
        on_dialog_updated(dialog_id, "set is_last_read_inbox_message_id_inited");
      }
    }

    if (!G()->parameters().use_message_db || is_new || !d->is_last_read_outbox_message_id_inited) {
      if (d->last_read_outbox_message_id < read_outbox_max_message_id) {
        set_dialog_last_read_outbox_message_id(d, read_outbox_max_message_id);
      }
      if (!d->is_last_read_outbox_message_id_inited) {
        d->is_last_read_outbox_message_id_inited = true;
        on_dialog_updated(dialog_id, "set is_last_read_outbox_message_id_inited");
      }
    }

    if (!G()->parameters().use_message_db || is_new) {
      if (d->unread_mention_count != dialog->unread_mentions_count_) {
        set_dialog_unread_mention_count(d, dialog->unread_mentions_count_);
        update_dialog_mention_notification_count(d);
        send_update_chat_unread_mention_count(d);
      }
    }

    being_added_dialog_id_ = DialogId();

    update_dialog_lists(d, std::move(positions), true, false, "on_get_dialogs");
  }

  if (from_dialog_list) {
    CHECK(!td_->auth_manager_->is_bot());
    CHECK(total_count >= 0);

    auto &folder_list = add_dialog_list(DialogListId(folder_id));
    if (folder_list.server_dialog_total_count_ != total_count) {
      auto old_dialog_total_count = get_dialog_total_count(folder_list);
      folder_list.server_dialog_total_count_ = total_count;
      if (folder_list.is_dialog_unread_count_inited_) {
        if (old_dialog_total_count != get_dialog_total_count(folder_list)) {
          send_update_unread_chat_count(folder_list, DialogId(), true, "on_get_dialogs");
        } else {
          save_unread_chat_count(folder_list);
        }
      }
    }

    auto *folder = get_dialog_folder(folder_id);
    CHECK(folder != nullptr);
    if (dialogs.empty()) {
      // if there are no more dialogs on the server
      max_dialog_date = MAX_DIALOG_DATE;
    }
    if (folder->last_server_dialog_date_ < max_dialog_date) {
      folder->last_server_dialog_date_ = max_dialog_date;
      update_last_dialog_date(folder_id);
    } else if (promise) {
      LOG(ERROR) << "Last server dialog date didn't increased from " << folder->last_server_dialog_date_ << " to "
                 << max_dialog_date << " after receiving " << dialogs.size() << " chats " << added_dialog_ids
                 << " from " << total_count << " in " << folder_id
                 << ". last_dialog_date = " << folder->folder_last_dialog_date_
                 << ", last_loaded_database_dialog_date = " << folder->last_loaded_database_dialog_date_;
    }
  }
  if (from_pinned_dialog_list) {
    CHECK(!td_->auth_manager_->is_bot());
    auto *folder_list = get_dialog_list(DialogListId(folder_id));
    CHECK(folder_list != nullptr);
    auto pinned_dialog_ids = remove_secret_chat_dialog_ids(get_pinned_dialog_ids(DialogListId(folder_id)));
    bool are_pinned_dialogs_saved = folder_list->are_pinned_dialogs_inited_;
    folder_list->are_pinned_dialogs_inited_ = true;
    if (pinned_dialog_ids != added_dialog_ids) {
      LOG(INFO) << "Update pinned chats order from " << format::as_array(pinned_dialog_ids) << " to "
                << format::as_array(added_dialog_ids);
      std::unordered_set<DialogId, DialogIdHash> old_pinned_dialog_ids(pinned_dialog_ids.begin(),
                                                                       pinned_dialog_ids.end());

      std::reverse(pinned_dialog_ids.begin(), pinned_dialog_ids.end());
      std::reverse(added_dialog_ids.begin(), added_dialog_ids.end());
      auto old_it = pinned_dialog_ids.begin();
      for (auto dialog_id : added_dialog_ids) {
        old_pinned_dialog_ids.erase(dialog_id);
        while (old_it < pinned_dialog_ids.end()) {
          if (*old_it == dialog_id) {
            break;
          }
          ++old_it;
        }
        if (old_it < pinned_dialog_ids.end()) {
          // leave dialog where it is
          ++old_it;
          continue;
        }
        if (set_dialog_is_pinned(dialog_id, true)) {
          are_pinned_dialogs_saved = true;
        }
      }
      for (auto dialog_id : old_pinned_dialog_ids) {
        Dialog *d = get_dialog_force(dialog_id, "on_get_dialogs pinned");
        if (d == nullptr) {
          LOG(ERROR) << "Failed to find " << dialog_id << " to unpin in " << folder_id;
          force_create_dialog(dialog_id, "from_pinned_dialog_list", true);
          d = get_dialog_force(dialog_id, "on_get_dialogs pinned 2");
        }
        if (d != nullptr && set_dialog_is_pinned(DialogListId(folder_id), d, false)) {
          are_pinned_dialogs_saved = true;
        }
      }
    } else {
      LOG(INFO) << "Pinned chats are not changed";
    }
    update_list_last_pinned_dialog_date(*folder_list);

    if (!are_pinned_dialogs_saved && G()->parameters().use_message_db) {
      LOG(INFO) << "Save empty pinned chat list in " << folder_id;
      G()->td_db()->get_binlog_pmc()->set(PSTRING() << "pinned_dialog_ids" << folder_id.get(), "");
    }
  }
  promise.set_value(Unit());
}

void MessagesManager::dump_debug_message_op(const Dialog *d, int priority) {
  if (!is_debug_message_op_enabled()) {
    return;
  }
  if (d == nullptr) {
    LOG(ERROR) << "Chat not found";
    return;
  }
  static int last_dumped_priority = -1;
  if (priority <= last_dumped_priority) {
    LOG(ERROR) << "Skip dump " << d->dialog_id;
    return;
  }
  last_dumped_priority = priority;

  for (auto &op : d->debug_message_op) {
    switch (op.type) {
      case Dialog::MessageOp::Add: {
        LOG(ERROR) << "MessageOpAdd at " << op.date << " " << op.message_id << " " << op.content_type << " "
                   << op.from_update << " " << op.have_previous << " " << op.have_next << " " << op.source;
        break;
      }
      case Dialog::MessageOp::SetPts: {
        LOG(ERROR) << "MessageOpSetPts at " << op.date << " " << op.pts << " " << op.source;
        break;
      }
      case Dialog::MessageOp::Delete: {
        LOG(ERROR) << "MessageOpDelete at " << op.date << " " << op.message_id << " " << op.content_type << " "
                   << op.from_update << " " << op.have_previous << " " << op.have_next << " " << op.source;
        break;
      }
      case Dialog::MessageOp::DeleteAll: {
        LOG(ERROR) << "MessageOpDeleteAll at " << op.date << " " << op.from_update;
        break;
      }
      default:
        UNREACHABLE();
    }
  }
}

bool MessagesManager::is_message_unload_enabled() const {
  return G()->parameters().use_message_db || td_->auth_manager_->is_bot();
}

bool MessagesManager::can_unload_message(const Dialog *d, const Message *m) const {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  CHECK(m->message_id.is_valid());
  // don't want to unload messages from opened dialogs
  // don't want to unload messages to which there are replies in yet unsent messages
  // don't want to unload message with active reply markup
  // don't want to unload the newest pinned message
  // don't want to unload last edited message, because server can send updateEditChannelMessage again
  // can't unload from memory last dialog, last database messages, yet unsent messages, being edited media messages and active live locations
  // can't unload messages in dialog with active suffix load query
  FullMessageId full_message_id{d->dialog_id, m->message_id};
  return !d->is_opened && m->message_id != d->last_message_id && m->message_id != d->last_database_message_id &&
         !m->message_id.is_yet_unsent() && active_live_location_full_message_ids_.count(full_message_id) == 0 &&
         replied_by_yet_unsent_messages_.count(full_message_id) == 0 && m->edited_content == nullptr &&
         d->suffix_load_queries_.empty() && m->message_id != d->reply_markup_message_id &&
         m->message_id != d->last_pinned_message_id && m->message_id != d->last_edited_message_id;
}

void MessagesManager::unload_message(Dialog *d, MessageId message_id) {
  CHECK(d != nullptr);
  CHECK(message_id.is_valid());
  bool need_update_dialog_pos = false;
  auto m = do_delete_message(d, message_id, false, true, &need_update_dialog_pos, "unload_message");
  CHECK(!need_update_dialog_pos);
}

unique_ptr<MessagesManager::Message> MessagesManager::delete_message(Dialog *d, MessageId message_id,
                                                                     bool is_permanently_deleted,
                                                                     bool *need_update_dialog_pos, const char *source) {
  return do_delete_message(d, message_id, is_permanently_deleted, false, need_update_dialog_pos, source);
}

void MessagesManager::add_random_id_to_message_id_correspondence(Dialog *d, int64 random_id, MessageId message_id) {
  CHECK(d != nullptr);
  CHECK(d->dialog_id.get_type() == DialogType::SecretChat);
  CHECK(message_id.is_valid());
  auto it = d->random_id_to_message_id.find(random_id);
  if (it == d->random_id_to_message_id.end() || it->second < message_id) {
    LOG(INFO) << "Add correspondence from random_id " << random_id << " to " << message_id << " in " << d->dialog_id;
    d->random_id_to_message_id[random_id] = message_id;
  }
}

void MessagesManager::delete_random_id_to_message_id_correspondence(Dialog *d, int64 random_id, MessageId message_id) {
  CHECK(d != nullptr);
  CHECK(d->dialog_id.get_type() == DialogType::SecretChat);
  CHECK(message_id.is_valid());
  auto it = d->random_id_to_message_id.find(random_id);
  if (it != d->random_id_to_message_id.end() && it->second == message_id) {
    LOG(INFO) << "Delete correspondence from random_id " << random_id << " to " << message_id << " in " << d->dialog_id;
    d->random_id_to_message_id.erase(it);
  }
}

void MessagesManager::add_notification_id_to_message_id_correspondence(Dialog *d, NotificationId notification_id,
                                                                       MessageId message_id) {
  CHECK(d != nullptr);
  CHECK(notification_id.is_valid());
  CHECK(message_id.is_valid());
  auto it = d->notification_id_to_message_id.find(notification_id);
  if (it == d->notification_id_to_message_id.end()) {
    VLOG(notifications) << "Add correspondence from " << notification_id << " to " << message_id << " in "
                        << d->dialog_id;
    d->notification_id_to_message_id.emplace(notification_id, message_id);
  } else if (it->second != message_id) {
    LOG(ERROR) << "Have duplicated " << notification_id << " in " << d->dialog_id << " in " << message_id << " and "
               << it->second;
    if (it->second < message_id) {
      it->second = message_id;
    }
  }
}

void MessagesManager::delete_notification_id_to_message_id_correspondence(Dialog *d, NotificationId notification_id,
                                                                          MessageId message_id) {
  CHECK(d != nullptr);
  CHECK(notification_id.is_valid());
  CHECK(message_id.is_valid());
  auto it = d->notification_id_to_message_id.find(notification_id);
  if (it != d->notification_id_to_message_id.end() && it->second == message_id) {
    VLOG(notifications) << "Delete correspondence from " << notification_id << " to " << message_id << " in "
                        << d->dialog_id;
    d->notification_id_to_message_id.erase(it);
  } else {
    LOG(ERROR) << "Can't find " << notification_id << " in " << d->dialog_id << " with " << message_id;
  }
}

void MessagesManager::remove_message_notification_id(Dialog *d, Message *m, bool is_permanent, bool force_update,
                                                     bool ignore_pinned_message_notification_removal) {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  CHECK(m->message_id.is_valid());
  if (!m->notification_id.is_valid()) {
    return;
  }

  auto from_mentions = is_from_mention_notification_group(d, m);
  auto &group_info = get_notification_group_info(d, m);
  if (!group_info.group_id.is_valid()) {
    return;
  }

  bool had_active_notification = is_message_notification_active(d, m);

  auto notification_id = m->notification_id;
  VLOG(notifications) << "Remove " << notification_id << " from " << m->message_id << " in " << group_info.group_id
                      << '/' << d->dialog_id << " from database, was_active = " << had_active_notification
                      << ", is_permanent = " << is_permanent;
  delete_notification_id_to_message_id_correspondence(d, notification_id, m->message_id);
  m->removed_notification_id = m->notification_id;
  m->notification_id = NotificationId();
  if (d->pinned_message_notification_message_id == m->message_id && is_permanent &&
      !ignore_pinned_message_notification_removal) {
    remove_dialog_pinned_message_notification(
        d, "remove_message_notification_id");  // must be called after notification_id is removed
  }
  if (group_info.last_notification_id == notification_id) {
    // last notification is deleted, need to find new last notification
    fix_dialog_last_notification_id(d, from_mentions, m->message_id);
  }

  if (is_permanent) {
    if (had_active_notification) {
      send_closure_later(G()->notification_manager(), &NotificationManager::remove_notification, group_info.group_id,
                         notification_id, is_permanent, force_update, Promise<Unit>(),
                         "remove_message_notification_id");
    }

    // on_message_changed will be called by the caller
    // don't need to call there to not save twice/or to save just deleted message
  } else {
    on_message_changed(d, m, false, "remove_message_notification_id");
  }
}

void MessagesManager::remove_new_secret_chat_notification(Dialog *d, bool is_permanent) {
  CHECK(d != nullptr);
  auto notification_id = d->new_secret_chat_notification_id;
  CHECK(notification_id.is_valid());
  VLOG(notifications) << "Remove " << notification_id << " about new secret " << d->dialog_id << " from "
                      << d->message_notification_group.group_id;
  d->new_secret_chat_notification_id = NotificationId();
  bool is_fixed = set_dialog_last_notification(d->dialog_id, d->message_notification_group, 0, NotificationId(),
                                               "remove_new_secret_chat_notification");
  CHECK(is_fixed);
  if (is_permanent) {
    CHECK(d->message_notification_group.group_id.is_valid());
    send_closure_later(G()->notification_manager(), &NotificationManager::remove_notification,
                       d->message_notification_group.group_id, notification_id, true, true, Promise<Unit>(),
                       "remove_new_secret_chat_notification");
  }
}

void MessagesManager::fix_dialog_last_notification_id(Dialog *d, bool from_mentions, MessageId message_id) {
  CHECK(d != nullptr);
  CHECK(!message_id.is_scheduled());
  MessagesConstIterator it(d, message_id);
  auto &group_info = from_mentions ? d->mention_notification_group : d->message_notification_group;
  VLOG(notifications) << "Trying to fix last notification identifier in " << group_info.group_id << " from "
                      << d->dialog_id << " from " << message_id << "/" << group_info.last_notification_id;
  if (*it != nullptr && ((*it)->message_id == message_id || (*it)->have_next)) {
    while (*it != nullptr) {
      const Message *m = *it;
      if (is_from_mention_notification_group(d, m) == from_mentions && m->notification_id.is_valid() &&
          is_message_notification_active(d, m) && m->message_id != message_id) {
        bool is_fixed = set_dialog_last_notification(d->dialog_id, group_info, m->date, m->notification_id,
                                                     "fix_dialog_last_notification_id");
        CHECK(is_fixed);
        return;
      }
      --it;
    }
  }
  if (G()->parameters().use_message_db) {
    get_message_notifications_from_database(
        d->dialog_id, group_info.group_id, group_info.last_notification_id, message_id, 1,
        PromiseCreator::lambda(
            [actor_id = actor_id(this), dialog_id = d->dialog_id, from_mentions,
             prev_last_notification_id = group_info.last_notification_id](Result<vector<Notification>> result) {
              send_closure(actor_id, &MessagesManager::do_fix_dialog_last_notification_id, dialog_id, from_mentions,
                           prev_last_notification_id, std::move(result));
            }));
  }
}

void MessagesManager::do_fix_dialog_last_notification_id(DialogId dialog_id, bool from_mentions,
                                                         NotificationId prev_last_notification_id,
                                                         Result<vector<Notification>> result) {
  if (result.is_error()) {
    return;
  }

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  auto &group_info = from_mentions ? d->mention_notification_group : d->message_notification_group;
  VLOG(notifications) << "Receive " << result.ok().size() << " message notifications in " << group_info.group_id << '/'
                      << dialog_id << " from " << prev_last_notification_id;
  if (group_info.last_notification_id != prev_last_notification_id) {
    // last_notification_id was changed
    return;
  }

  auto notifications = result.move_as_ok();
  CHECK(notifications.size() <= 1);

  int32 last_notification_date = 0;
  NotificationId last_notification_id;
  if (!notifications.empty()) {
    last_notification_date = notifications[0].date;
    last_notification_id = notifications[0].notification_id;
  }

  bool is_fixed = set_dialog_last_notification(dialog_id, group_info, last_notification_date, last_notification_id,
                                               "do_fix_dialog_last_notification_id");
  CHECK(is_fixed);
}

// DO NOT FORGET TO ADD ALL CHANGES OF THIS FUNCTION AS WELL TO do_delete_all_dialog_messages
unique_ptr<MessagesManager::Message> MessagesManager::do_delete_message(Dialog *d, MessageId message_id,
                                                                        bool is_permanently_deleted,
                                                                        bool only_from_memory,
                                                                        bool *need_update_dialog_pos,
                                                                        const char *source) {
  CHECK(d != nullptr);
  if (!message_id.is_valid()) {
    if (message_id.is_valid_scheduled()) {
      return do_delete_scheduled_message(d, message_id, is_permanently_deleted, source);
    }

    LOG(ERROR) << "Trying to delete " << message_id << " in " << d->dialog_id << " from " << source;
    return nullptr;
  }

  FullMessageId full_message_id(d->dialog_id, message_id);
  unique_ptr<Message> *v = treap_find_message(&d->messages, message_id);
  if (*v == nullptr) {
    LOG(INFO) << message_id << " is not found in " << d->dialog_id << " to be deleted from " << source;
    if (only_from_memory) {
      return nullptr;
    }

    if (get_message_force(d, message_id, "do_delete_message") == nullptr) {
      // currently there may be a race between add_message_to_database and get_message_force,
      // so delete a message from database just in case
      delete_message_from_database(d, message_id, nullptr, is_permanently_deleted);

      if (is_permanently_deleted && d->last_clear_history_message_id == message_id) {
        set_dialog_last_clear_history_date(d, 0, MessageId(), "do_delete_message");
        *need_update_dialog_pos = true;
      }

      /*
      can't do this because the message may be never received in the dialog, unread count will became negative
      // if last_read_inbox_message_id is not known, we can't be sure whether unread_count should be decreased or not
      if (message_id.is_valid() && !message_id.is_yet_unsent() && d->is_last_read_inbox_message_id_inited &&
          message_id > d->last_read_inbox_message_id && !td_->auth_manager_->is_bot()) {
        int32 server_unread_count = d->server_unread_count;
        int32 local_unread_count = d->local_unread_count;
        int32 &unread_count = message_id.is_server() ? server_unread_count : local_unread_count;
        if (unread_count == 0) {
          LOG(ERROR) << "Unread count became negative in " << d->dialog_id << " after deletion of " << message_id
                     << ". Last read is " << d->last_read_inbox_message_id;
          dump_debug_message_op(d, 3);
        } else {
          unread_count--;
          set_dialog_last_read_inbox_message_id(d, MessageId::min(), server_unread_count, local_unread_count, false,
                                                source);
        }
      }
      */
      return nullptr;
    }
    v = treap_find_message(&d->messages, message_id);
    CHECK(*v != nullptr);
  }

  const Message *m = v->get();
  CHECK(m->message_id == message_id);

  if (only_from_memory && !can_unload_message(d, m)) {
    return nullptr;
  }

  LOG_CHECK(!d->being_deleted_message_id.is_valid())
      << d->being_deleted_message_id << " " << message_id << " " << source;
  d->being_deleted_message_id = message_id;

  if (is_debug_message_op_enabled()) {
    d->debug_message_op.emplace_back(Dialog::MessageOp::Delete, m->message_id, m->content->get_type(), false,
                                     m->have_previous, m->have_next, source);
  }

  bool need_get_history = false;
  if (!only_from_memory) {
    LOG(INFO) << "Deleting " << full_message_id << " with have_previous = " << m->have_previous
              << " and have_next = " << m->have_next << " from " << source;

    delete_message_from_database(d, message_id, m, is_permanently_deleted);

    delete_active_live_location(d->dialog_id, m);
    remove_message_file_sources(d->dialog_id, m);

    if (message_id == d->last_message_id) {
      MessagesConstIterator it(d, message_id);
      CHECK(*it == m);
      if ((*it)->have_previous) {
        --it;
        if (*it != nullptr) {
          set_dialog_last_message_id(d, (*it)->message_id, "do_delete_message");
        } else {
          LOG(ERROR) << "Have have_previous is true, but there is no previous for " << full_message_id << " from "
                     << source;
          dump_debug_message_op(d);
          set_dialog_last_message_id(d, MessageId(), "do_delete_message");
        }
      } else {
        need_get_history = true;
        set_dialog_last_message_id(d, MessageId(), "do_delete_message");
        d->delete_last_message_date = m->date;
        d->deleted_last_message_id = message_id;
        d->is_last_message_deleted_locally = Slice(source) == Slice(DELETE_MESSAGE_USER_REQUEST_SOURCE);
        on_dialog_updated(d->dialog_id, "do delete last message");
      }
      *need_update_dialog_pos = true;
    }

    if (message_id == d->last_database_message_id) {
      MessagesConstIterator it(d, message_id);
      CHECK(*it == m);
      while ((*it)->have_previous) {
        --it;
        if (*it == nullptr || !(*it)->message_id.is_yet_unsent()) {
          break;
        }
      }

      if (*it != nullptr) {
        if (!(*it)->message_id.is_yet_unsent() && (*it)->message_id != d->last_database_message_id) {
          if ((*it)->message_id < d->first_database_message_id && d->dialog_id.get_type() == DialogType::Channel) {
            // possible if messages was deleted from database, but not from memory after updateChannelTooLong
            set_dialog_last_database_message_id(d, MessageId(), "do_delete_message");
          } else {
            set_dialog_last_database_message_id(d, (*it)->message_id, "do_delete_message");
            if (d->last_database_message_id < d->first_database_message_id) {
              LOG(ERROR) << "Last database " << d->last_database_message_id << " became less than first database "
                         << d->first_database_message_id << " after deletion of " << full_message_id;
              set_dialog_first_database_message_id(d, d->last_database_message_id, "do_delete_message 2");
            }
          }
        } else {
          need_get_history = true;
        }
      } else {
        LOG(ERROR) << "Have have_previous is true, but there is no previous";
        dump_debug_message_op(d);
      }
    }
    if (d->last_database_message_id.is_valid()) {
      CHECK(d->first_database_message_id.is_valid());
    } else {
      set_dialog_first_database_message_id(d, MessageId(), "do_delete_message");
    }

    if (message_id == d->suffix_load_first_message_id_) {
      MessagesConstIterator it(d, message_id);
      CHECK(*it == m);
      if ((*it)->have_previous) {
        --it;
        if (*it != nullptr) {
          d->suffix_load_first_message_id_ = (*it)->message_id;
        } else {
          LOG(ERROR) << "Have have_previous is true, but there is no previous for " << full_message_id << " from "
                     << source;
          dump_debug_message_op(d);
          d->suffix_load_first_message_id_ = MessageId();
          d->suffix_load_done_ = false;
        }
      } else {
        d->suffix_load_first_message_id_ = MessageId();
        d->suffix_load_done_ = false;
      }
    }
  }
  if (only_from_memory && message_id >= d->suffix_load_first_message_id_) {
    d->suffix_load_first_message_id_ = MessageId();
    d->suffix_load_done_ = false;
  }

  if (m->have_previous && (only_from_memory || !m->have_next)) {
    MessagesIterator it(d, message_id);
    CHECK(*it == m);
    --it;
    Message *prev_m = *it;
    if (prev_m != nullptr) {
      prev_m->have_next = false;
    } else {
      LOG(ERROR) << "Have have_previous is true, but there is no previous for " << full_message_id << " from "
                 << source;
      dump_debug_message_op(d);
    }
  }
  if ((*v)->have_next && (only_from_memory || !(*v)->have_previous)) {
    MessagesIterator it(d, message_id);
    CHECK(*it == m);
    ++it;
    Message *next_m = *it;
    if (next_m != nullptr) {
      next_m->have_previous = false;
    } else {
      LOG(ERROR) << "Have have_next is true, but there is no next for " << full_message_id << " from " << source;
      dump_debug_message_op(d);
    }
  }

  auto result = treap_delete_message(v);

  d->being_deleted_message_id = MessageId();

  if (!only_from_memory) {
    if (need_get_history && !td_->auth_manager_->is_bot() && have_input_peer(d->dialog_id, AccessRights::Read)) {
      send_closure_later(actor_id(this), &MessagesManager::get_history_from_the_end, d->dialog_id, true, false,
                         Promise<Unit>());
    }

    if (d->reply_markup_message_id == message_id) {
      set_dialog_reply_markup(d, MessageId());
    }
    // if last_read_inbox_message_id is not known, we can't be sure whether unread_count should be decreased or not
    if (has_incoming_notification(d->dialog_id, result.get()) && message_id > d->last_read_inbox_message_id &&
        d->is_last_read_inbox_message_id_inited && !td_->auth_manager_->is_bot()) {
      int32 server_unread_count = d->server_unread_count;
      int32 local_unread_count = d->local_unread_count;
      int32 &unread_count = message_id.is_server() ? server_unread_count : local_unread_count;
      if (unread_count == 0) {
        if (need_unread_counter(d->order)) {
          LOG(ERROR) << "Unread count became negative in " << d->dialog_id << " after deletion of " << message_id
                     << ". Last read is " << d->last_read_inbox_message_id;
          dump_debug_message_op(d, 3);
        }
      } else {
        unread_count--;
        set_dialog_last_read_inbox_message_id(d, MessageId::min(), server_unread_count, local_unread_count, false,
                                              source);
      }
    }
    if (result->contains_unread_mention) {
      if (d->unread_mention_count == 0) {
        if (is_dialog_inited(d)) {
          LOG(ERROR) << "Unread mention count became negative in " << d->dialog_id << " after deletion of "
                     << message_id;
        }
      } else {
        set_dialog_unread_mention_count(d, d->unread_mention_count - 1);
        send_update_chat_unread_mention_count(d);
      }
    }

    update_message_count_by_index(d, -1, result.get());
    update_reply_count_by_message(d, -1, result.get());
  }

  on_message_deleted(d, result.get(), is_permanently_deleted, source);

  return result;
}

void MessagesManager::on_message_deleted(Dialog *d, Message *m, bool is_permanently_deleted, const char *source) {
  // also called for unloaded messages

  if (m->message_id.is_yet_unsent() && m->top_thread_message_id.is_valid()) {
    auto it = d->yet_unsent_thread_message_ids.find(m->top_thread_message_id);
    CHECK(it != d->yet_unsent_thread_message_ids.end());
    auto is_deleted = it->second.erase(m->message_id) > 0;
    CHECK(is_deleted);
    if (it->second.empty()) {
      d->yet_unsent_thread_message_ids.erase(it);
    }
  }

  cancel_send_deleted_message(d->dialog_id, m, is_permanently_deleted);

  CHECK(m->message_id.is_valid());
  switch (d->dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
      if (m->message_id.is_server()) {
        message_id_to_dialog_id_.erase(m->message_id);
      }
      break;
    case DialogType::Channel:
      // nothing to do
      break;
    case DialogType::SecretChat:
      delete_random_id_to_message_id_correspondence(d, m->random_id, m->message_id);
      break;
    case DialogType::None:
    default:
      UNREACHABLE();
  }
  ttl_unregister_message(d->dialog_id, m, source);
  ttl_period_unregister_message(d->dialog_id, m);
  delete_bot_command_message_id(d->dialog_id, m->message_id);
  unregister_message_content(td_, m->content.get(), {d->dialog_id, m->message_id}, "on_message_deleted");
  unregister_message_reply(d, m);
  if (m->notification_id.is_valid()) {
    delete_notification_id_to_message_id_correspondence(d, m->notification_id, m->message_id);
  }
}

unique_ptr<MessagesManager::Message> MessagesManager::do_delete_scheduled_message(Dialog *d, MessageId message_id,
                                                                                  bool is_permanently_deleted,
                                                                                  const char *source) {
  CHECK(d != nullptr);
  LOG_CHECK(message_id.is_valid_scheduled()) << d->dialog_id << ' ' << message_id << ' ' << source;

  unique_ptr<Message> *v = treap_find_message(&d->scheduled_messages, message_id);
  if (*v == nullptr) {
    LOG(INFO) << message_id << " is not found in " << d->dialog_id << " to be deleted from " << source;
    auto message = get_message_force(d, message_id, "do_delete_scheduled_message");
    if (message == nullptr) {
      // currently there may be a race between add_message_to_database and get_message_force,
      // so delete a message from database just in case
      delete_message_from_database(d, message_id, nullptr, is_permanently_deleted);
      return nullptr;
    }

    message_id = message->message_id;
    v = treap_find_message(&d->scheduled_messages, message_id);
    CHECK(*v != nullptr);
  }

  const Message *m = v->get();
  CHECK(m->message_id == message_id);

  LOG(INFO) << "Deleting " << FullMessageId{d->dialog_id, message_id} << " from " << source;

  delete_message_from_database(d, message_id, m, is_permanently_deleted);

  remove_message_file_sources(d->dialog_id, m);

  auto result = treap_delete_message(v);

  if (message_id.is_scheduled_server()) {
    size_t erased_count = d->scheduled_message_date.erase(message_id.get_scheduled_server_message_id());
    CHECK(erased_count != 0);
  }

  cancel_send_deleted_message(d->dialog_id, result.get(), is_permanently_deleted);

  unregister_message_content(td_, result->content.get(), {d->dialog_id, message_id}, "do_delete_scheduled_message");
  unregister_message_reply(d, m);

  return result;
}

void MessagesManager::do_delete_all_dialog_messages(Dialog *d, unique_ptr<Message> &message,
                                                    bool is_permanently_deleted, vector<int64> &deleted_message_ids) {
  if (message == nullptr) {
    return;
  }
  const Message *m = message.get();
  MessageId message_id = m->message_id;

  if (is_debug_message_op_enabled()) {
    d->debug_message_op.emplace_back(Dialog::MessageOp::Delete, m->message_id, m->content->get_type(), false,
                                     m->have_previous, m->have_next, "delete all messages");
  }

  LOG(INFO) << "Delete " << message_id;
  deleted_message_ids.push_back(message_id.get());

  do_delete_all_dialog_messages(d, message->right, is_permanently_deleted, deleted_message_ids);
  do_delete_all_dialog_messages(d, message->left, is_permanently_deleted, deleted_message_ids);

  delete_active_live_location(d->dialog_id, m);
  remove_message_file_sources(d->dialog_id, m);

  on_message_deleted(d, message.get(), is_permanently_deleted, "do_delete_all_dialog_messages");

  message = nullptr;
}

bool MessagesManager::have_dialog(DialogId dialog_id) const {
  return dialogs_.count(dialog_id) > 0;
}

void MessagesManager::load_dialogs(vector<DialogId> dialog_ids, Promise<vector<DialogId>> &&promise) {
  LOG(INFO) << "Load chats " << format::as_array(dialog_ids);

  Dependencies dependencies;
  for (auto dialog_id : dialog_ids) {
    if (dialog_id.is_valid() && !have_dialog(dialog_id)) {
      add_dialog_dependencies(dependencies, dialog_id);
    }
  }
  resolve_dependencies_force(td_, dependencies, "load_dialogs");

  td::remove_if(dialog_ids, [this](DialogId dialog_id) { return !have_dialog_info(dialog_id); });

  for (auto dialog_id : dialog_ids) {
    force_create_dialog(dialog_id, "load_dialogs");
  }

  LOG(INFO) << "Loaded chats " << format::as_array(dialog_ids);
  promise.set_value(std::move(dialog_ids));
}

bool MessagesManager::load_dialog(DialogId dialog_id, int left_tries, Promise<Unit> &&promise) {
  if (!dialog_id.is_valid()) {
    promise.set_error(Status::Error(400, "Invalid chat identifier specified"));
    return false;
  }

  if (!have_dialog_force(dialog_id, "load_dialog")) {  // TODO remove _force
    if (G()->parameters().use_message_db) {
      //      TODO load dialog from database, DialogLoader
      //      send_closure_later(actor_id(this), &MessagesManager::load_dialog_from_database, dialog_id,
      //      std::move(promise));
      //      return false;
    }
    if (td_->auth_manager_->is_bot()) {
      switch (dialog_id.get_type()) {
        case DialogType::User: {
          auto user_id = dialog_id.get_user_id();
          auto have_user = td_->contacts_manager_->get_user(user_id, left_tries, std::move(promise));
          if (!have_user) {
            return false;
          }
          break;
        }
        case DialogType::Chat: {
          auto have_chat = td_->contacts_manager_->get_chat(dialog_id.get_chat_id(), left_tries, std::move(promise));
          if (!have_chat) {
            return false;
          }
          break;
        }
        case DialogType::Channel: {
          auto have_channel =
              td_->contacts_manager_->get_channel(dialog_id.get_channel_id(), left_tries, std::move(promise));
          if (!have_channel) {
            return false;
          }
          break;
        }
        case DialogType::SecretChat:
          promise.set_error(Status::Error(400, "Chat not found"));
          return false;
        case DialogType::None:
        default:
          UNREACHABLE();
      }
      if (!have_input_peer(dialog_id, AccessRights::Read)) {
        return false;
      }

      add_dialog(dialog_id, "load_dialog");
      return true;
    }

    promise.set_error(Status::Error(400, "Chat not found"));
    return false;
  }

  promise.set_value(Unit());
  return true;
}

void MessagesManager::load_dialog_filter(DialogFilterId dialog_filter_id, bool force, Promise<Unit> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());
  if (!dialog_filter_id.is_valid()) {
    return promise.set_error(Status::Error(400, "Invalid chat filter identifier specified"));
  }

  auto filter = get_dialog_filter(dialog_filter_id);
  if (filter == nullptr) {
    return promise.set_value(Unit());
  }

  load_dialog_filter(filter, force, std::move(promise));
}

void MessagesManager::load_dialog_filter(const DialogFilter *filter, bool force, Promise<Unit> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());
  vector<InputDialogId> needed_dialog_ids;
  for (auto input_dialog_ids :
       {&filter->pinned_dialog_ids, &filter->excluded_dialog_ids, &filter->included_dialog_ids}) {
    for (auto input_dialog_id : *input_dialog_ids) {
      if (!have_dialog(input_dialog_id.get_dialog_id())) {
        needed_dialog_ids.push_back(input_dialog_id);
      }
    }
  }

  vector<InputDialogId> input_dialog_ids;
  for (auto &input_dialog_id : needed_dialog_ids) {
    auto dialog_id = input_dialog_id.get_dialog_id();
    // TODO load dialogs asynchronously
    if (!have_dialog_force(dialog_id, "load_dialog_filter")) {
      if (dialog_id.get_type() == DialogType::SecretChat) {
        if (have_dialog_info_force(dialog_id)) {
          force_create_dialog(dialog_id, "load_dialog_filter");
        }
      } else {
        input_dialog_ids.push_back(input_dialog_id);
      }
    }
  }

  if (!input_dialog_ids.empty() && !force) {
    const size_t MAX_SLICE_SIZE = 100;
    MultiPromiseActorSafe mpas{"GetFilterDialogsOnServerMultiPromiseActor"};
    mpas.add_promise(std::move(promise));
    mpas.set_ignore_errors(true);
    auto lock = mpas.get_promise();

    for (size_t i = 0; i < input_dialog_ids.size(); i += MAX_SLICE_SIZE) {
      auto end_i = i + MAX_SLICE_SIZE;
      auto end = end_i < input_dialog_ids.size() ? input_dialog_ids.begin() + end_i : input_dialog_ids.end();
      td_->create_handler<GetDialogsQuery>(mpas.get_promise())->send({input_dialog_ids.begin() + i, end});
    }

    lock.set_value(Unit());
    return;
  }

  promise.set_value(Unit());
}

void MessagesManager::get_recommended_dialog_filters(
    Promise<td_api::object_ptr<td_api::recommendedChatFilters>> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());
  auto query_promise =
      PromiseCreator::lambda([actor_id = actor_id(this), promise = std::move(promise)](
                                 Result<vector<tl_object_ptr<telegram_api::dialogFilterSuggested>>> result) mutable {
        send_closure(actor_id, &MessagesManager::on_get_recommended_dialog_filters, std::move(result),
                     std::move(promise));
      });
  td_->create_handler<GetSuggestedDialogFiltersQuery>(std::move(query_promise))->send();
}

void MessagesManager::on_get_recommended_dialog_filters(
    Result<vector<tl_object_ptr<telegram_api::dialogFilterSuggested>>> result,
    Promise<td_api::object_ptr<td_api::recommendedChatFilters>> &&promise) {
  if (result.is_error()) {
    return promise.set_error(result.move_as_error());
  }
  CHECK(!td_->auth_manager_->is_bot());
  auto suggested_filters = result.move_as_ok();

  MultiPromiseActorSafe mpas{"LoadRecommendedFiltersMultiPromiseActor"};
  mpas.add_promise(Promise<Unit>());
  auto lock = mpas.get_promise();

  vector<RecommendedDialogFilter> filters;
  for (auto &suggested_filter : suggested_filters) {
    RecommendedDialogFilter filter;
    filter.dialog_filter = DialogFilter::get_dialog_filter(std::move(suggested_filter->filter_), false);
    CHECK(filter.dialog_filter != nullptr);
    filter.dialog_filter->dialog_filter_id = DialogFilterId();  // just in case
    load_dialog_filter(filter.dialog_filter.get(), false, mpas.get_promise());

    filter.description = std::move(suggested_filter->description_);
    filters.push_back(std::move(filter));
  }

  mpas.add_promise(PromiseCreator::lambda([actor_id = actor_id(this), filters = std::move(filters),
                                           promise = std::move(promise)](Result<Unit> &&result) mutable {
    send_closure(actor_id, &MessagesManager::on_load_recommended_dialog_filters, std::move(result), std::move(filters),
                 std::move(promise));
  }));
  lock.set_value(Unit());
}

void MessagesManager::on_load_recommended_dialog_filters(
    Result<Unit> &&result, vector<RecommendedDialogFilter> &&filters,
    Promise<td_api::object_ptr<td_api::recommendedChatFilters>> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());
  if (result.is_error()) {
    return promise.set_error(result.move_as_error());
  }
  CHECK(!td_->auth_manager_->is_bot());

  auto chat_filters = transform(filters, [this](const RecommendedDialogFilter &filter) {
    return td_api::make_object<td_api::recommendedChatFilter>(get_chat_filter_object(filter.dialog_filter.get()),
                                                              filter.description);
  });
  recommended_dialog_filters_ = std::move(filters);
  promise.set_value(td_api::make_object<td_api::recommendedChatFilters>(std::move(chat_filters)));
}

Result<DialogDate> MessagesManager::get_dialog_list_last_date(DialogListId dialog_list_id) {
  CHECK(!td_->auth_manager_->is_bot());

  auto *list_ptr = get_dialog_list(dialog_list_id);
  if (list_ptr == nullptr) {
    return Status::Error(400, "Chat list not found");
  }
  return list_ptr->list_last_dialog_date_;
}

vector<DialogId> MessagesManager::get_dialogs(DialogListId dialog_list_id, DialogDate offset, int32 limit,
                                              bool exact_limit, bool force, Promise<Unit> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());

  auto *list_ptr = get_dialog_list(dialog_list_id);
  if (list_ptr == nullptr) {
    promise.set_error(Status::Error(400, "Chat list not found"));
    return {};
  }
  auto &list = *list_ptr;

  LOG(INFO) << "Get chats in " << dialog_list_id << " with offset " << offset << " and limit " << limit
            << ". last_dialog_date = " << list.list_last_dialog_date_
            << ", last_pinned_dialog_date_ = " << list.last_pinned_dialog_date_
            << ", are_pinned_dialogs_inited_ = " << list.are_pinned_dialogs_inited_;

  if (limit <= 0) {
    promise.set_error(Status::Error(400, "Parameter limit must be positive"));
    return {};
  }

  vector<DialogId> result;
  if (dialog_list_id == DialogListId(FolderId::main()) && sponsored_dialog_id_.is_valid()) {
    auto d = get_dialog(sponsored_dialog_id_);
    CHECK(d != nullptr);
    if (is_dialog_sponsored(d)) {
      DialogDate date(get_dialog_private_order(&list, d), d->dialog_id);
      if (offset < date) {
        result.push_back(sponsored_dialog_id_);
        offset = date;
        limit--;
      }
    }
  }

  if (!list.are_pinned_dialogs_inited_) {
    if (limit == 0 || force) {
      promise.set_value(Unit());
      return result;
    } else {
      if (dialog_list_id.is_folder()) {
        auto &folder = *get_dialog_folder(dialog_list_id.get_folder_id());
        if (folder.last_loaded_database_dialog_date_ == folder.last_database_server_dialog_date_ &&
            folder.folder_last_dialog_date_ != MAX_DIALOG_DATE) {
          load_dialog_list(list, limit, std::move(promise));
          return {};
        }
      }
      reload_pinned_dialogs(dialog_list_id, std::move(promise));
      return {};
    }
  }
  if (dialog_list_id.is_filter()) {
    auto *filter = get_dialog_filter(dialog_list_id.get_filter_id());
    CHECK(filter != nullptr);
    vector<InputDialogId> input_dialog_ids;
    for (const auto &input_dialog_id : filter->pinned_dialog_ids) {
      auto dialog_id = input_dialog_id.get_dialog_id();
      if (!have_dialog_force(dialog_id, "get_dialogs")) {
        if (dialog_id.get_type() == DialogType::SecretChat) {
          if (have_dialog_info_force(dialog_id)) {
            force_create_dialog(dialog_id, "get_dialogs");
          }
        } else {
          input_dialog_ids.push_back(input_dialog_id);
        }
      }
    }

    if (!input_dialog_ids.empty()) {
      if (limit == 0 || force) {
        promise.set_value(Unit());
        return result;
      } else {
        td_->create_handler<GetDialogsQuery>(std::move(promise))->send(std::move(input_dialog_ids));
        return {};
      }
    }
  }

  bool need_reload_pinned_dialogs = false;
  if (!list.pinned_dialogs_.empty() && offset < list.pinned_dialogs_.back() && limit > 0) {
    for (auto &pinned_dialog : list.pinned_dialogs_) {
      if (offset < pinned_dialog) {
        auto dialog_id = pinned_dialog.get_dialog_id();
        auto d = get_dialog_force(dialog_id, "get_dialogs");
        if (d == nullptr) {
          LOG(ERROR) << "Failed to load pinned " << dialog_id << " from " << dialog_list_id;
          if (dialog_id.get_type() != DialogType::SecretChat) {
            need_reload_pinned_dialogs = true;
          }
          continue;
        }
        if (d->order == DEFAULT_ORDER) {
          LOG(INFO) << "Loaded pinned " << dialog_id << " with default order in " << dialog_list_id;
          continue;
        }
        result.push_back(dialog_id);
        offset = pinned_dialog;
        limit--;
        if (limit == 0) {
          break;
        }
      }
    }
  }
  if (need_reload_pinned_dialogs) {
    reload_pinned_dialogs(dialog_list_id, Auto());
  }
  update_list_last_pinned_dialog_date(list);

  vector<const DialogFolder *> folders;
  vector<std::set<DialogDate>::const_iterator> folder_iterators;
  for (auto folder_id : get_dialog_list_folder_ids(list)) {
    folders.push_back(get_dialog_folder(folder_id));
    folder_iterators.push_back(folders.back()->ordered_dialogs_.upper_bound(offset));
  }
  while (limit > 0) {
    size_t best_pos = 0;
    DialogDate best_dialog_date = MAX_DIALOG_DATE;
    for (size_t i = 0; i < folders.size(); i++) {
      while (folder_iterators[i] != folders[i]->ordered_dialogs_.end() &&
             *folder_iterators[i] <= list.list_last_dialog_date_ &&
             (!is_dialog_in_list(get_dialog(folder_iterators[i]->get_dialog_id()), dialog_list_id) ||
              get_dialog_pinned_order(&list, folder_iterators[i]->get_dialog_id()) != DEFAULT_ORDER)) {
        ++folder_iterators[i];
      }
      if (folder_iterators[i] != folders[i]->ordered_dialogs_.end() &&
          *folder_iterators[i] <= list.list_last_dialog_date_ && *folder_iterators[i] < best_dialog_date) {
        best_pos = i;
        best_dialog_date = *folder_iterators[i];
      }
    }
    if (best_dialog_date == MAX_DIALOG_DATE || best_dialog_date.get_order() == DEFAULT_ORDER) {
      break;
    }

    limit--;
    result.push_back(folder_iterators[best_pos]->get_dialog_id());
    ++folder_iterators[best_pos];
  }

  if ((!result.empty() && (!exact_limit || limit == 0)) || force || list.list_last_dialog_date_ == MAX_DIALOG_DATE) {
    if (limit > 0 && list.list_last_dialog_date_ != MAX_DIALOG_DATE) {
      load_dialog_list(list, limit, Promise<Unit>());
    }

    promise.set_value(Unit());
    return result;
  } else {
    if (!result.empty()) {
      LOG(INFO) << "Have only " << result.size() << " chats, but " << limit << " chats more are needed";
    }
    load_dialog_list(list, limit, std::move(promise));
    return {};
  }
}

void MessagesManager::load_dialog_list(DialogList &list, int32 limit, Promise<Unit> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());
  if (limit > MAX_GET_DIALOGS + 2) {
    limit = MAX_GET_DIALOGS + 2;
  }
  bool is_request_sent = false;
  for (auto folder_id : get_dialog_list_folder_ids(list)) {
    const auto &folder = *get_dialog_folder(folder_id);
    if (folder.folder_last_dialog_date_ != MAX_DIALOG_DATE) {
      load_folder_dialog_list(folder_id, limit, false);
      is_request_sent = true;
    }
  }
  if (is_request_sent) {
    LOG(INFO) << "Wait for loading of " << limit << " chats in " << list.dialog_list_id;
    list.load_list_queries_.push_back(std::move(promise));
  } else {
    LOG(ERROR) << "There is nothing to load for " << list.dialog_list_id << " with folders "
               << get_dialog_list_folder_ids(list);
    promise.set_value(Unit());
  }
}

void MessagesManager::load_folder_dialog_list(FolderId folder_id, int32 limit, bool only_local) {
  if (G()->close_flag()) {
    return;
  }

  CHECK(!td_->auth_manager_->is_bot());
  auto &folder = *get_dialog_folder(folder_id);
  if (folder.folder_last_dialog_date_ == MAX_DIALOG_DATE) {
    return;
  }

  bool use_database = G()->parameters().use_message_db &&
                      folder.last_loaded_database_dialog_date_ < folder.last_database_server_dialog_date_;
  if (only_local && !use_database) {
    return;
  }

  auto &multipromise = folder.load_folder_dialog_list_multipromise_;
  if (multipromise.promise_count() != 0) {
    // queries have already been sent, just wait for the result
    LOG(INFO) << "Skip loading of dialog list in " << folder_id << " with limit " << limit
              << ", because it is already being loaded";
    if (use_database && folder.load_dialog_list_limit_max_ != 0) {
      folder.load_dialog_list_limit_max_ = max(folder.load_dialog_list_limit_max_, limit);
    }
    return;
  }
  LOG(INFO) << "Load dialog list in " << folder_id << " with limit " << limit;
  multipromise.add_promise(PromiseCreator::lambda([actor_id = actor_id(this), folder_id](Result<Unit> result) {
    send_closure_later(actor_id, &MessagesManager::on_load_folder_dialog_list, folder_id, std::move(result));
  }));

  bool is_query_sent = false;
  if (use_database) {
    load_folder_dialog_list_from_database(folder_id, limit, multipromise.get_promise());
    is_query_sent = true;
  } else {
    LOG(INFO) << "Get chats from " << folder.last_server_dialog_date_;
    multipromise.add_promise(PromiseCreator::lambda([actor_id = actor_id(this), folder_id](Result<Unit> result) {
      if (result.is_ok()) {
        send_closure(actor_id, &MessagesManager::recalc_unread_count, DialogListId(folder_id), -1, true);
      }
    }));
    auto lock = multipromise.get_promise();
    reload_pinned_dialogs(DialogListId(folder_id), multipromise.get_promise());
    if (folder.folder_last_dialog_date_ == folder.last_server_dialog_date_) {
      send_closure(
          td_->create_net_actor<GetDialogListActor>(multipromise.get_promise()), &GetDialogListActor::send, folder_id,
          folder.last_server_dialog_date_.get_date(),
          folder.last_server_dialog_date_.get_message_id().get_next_server_message_id().get_server_message_id(),
          folder.last_server_dialog_date_.get_dialog_id(), int32{MAX_GET_DIALOGS},
          get_sequence_dispatcher_id(DialogId(), MessageContentType::None));
      is_query_sent = true;
    }
    if (folder_id == FolderId::main() && folder.last_server_dialog_date_ == MIN_DIALOG_DATE) {
      // do not pass promise to not wait for drafts before showing chat list
      td_->create_handler<GetAllDraftsQuery>()->send();
    }
    lock.set_value(Unit());
  }
  CHECK(is_query_sent);
}

void MessagesManager::on_load_folder_dialog_list(FolderId folder_id, Result<Unit> &&result) {
  if (G()->close_flag()) {
    return;
  }
  CHECK(!td_->auth_manager_->is_bot());

  const auto &folder = *get_dialog_folder(folder_id);
  if (result.is_ok()) {
    LOG(INFO) << "Successfully loaded chats in " << folder_id;
    if (folder.last_server_dialog_date_ == MAX_DIALOG_DATE) {
      return;
    }

    bool need_new_get_dialog_list = false;
    for (const auto &list_it : dialog_lists_) {
      auto &list = list_it.second;
      if (!list.load_list_queries_.empty() && has_dialogs_from_folder(list, folder)) {
        LOG(INFO) << "Need to load more chats in " << folder_id << " for " << list_it.first;
        need_new_get_dialog_list = true;
      }
    }
    if (need_new_get_dialog_list) {
      load_folder_dialog_list(folder_id, int32{MAX_GET_DIALOGS}, false);
    }
    return;
  }

  LOG(WARNING) << "Failed to load chats in " << folder_id << ": " << result.error();
  vector<Promise<Unit>> promises;
  for (auto &list_it : dialog_lists_) {
    auto &list = list_it.second;
    if (!list.load_list_queries_.empty() && has_dialogs_from_folder(list, folder)) {
      append(promises, std::move(list.load_list_queries_));
      list.load_list_queries_.clear();
    }
  }

  for (auto &promise : promises) {
    promise.set_error(result.error().clone());
  }
}

void MessagesManager::load_folder_dialog_list_from_database(FolderId folder_id, int32 limit, Promise<Unit> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());
  auto &folder = *get_dialog_folder(folder_id);
  LOG(INFO) << "Load " << limit << " chats in " << folder_id << " from database from "
            << folder.last_loaded_database_dialog_date_
            << ", last database server dialog date = " << folder.last_database_server_dialog_date_;

  CHECK(folder.load_dialog_list_limit_max_ == 0);
  folder.load_dialog_list_limit_max_ = limit;
  G()->td_db()->get_dialog_db_async()->get_dialogs(
      folder_id, folder.last_loaded_database_dialog_date_.get_order(),
      folder.last_loaded_database_dialog_date_.get_dialog_id(), limit,
      PromiseCreator::lambda([actor_id = actor_id(this), folder_id, limit,
                              promise = std::move(promise)](DialogDbGetDialogsResult result) mutable {
        send_closure(actor_id, &MessagesManager::on_get_dialogs_from_database, folder_id, limit, std::move(result),
                     std::move(promise));
      }));
}

void MessagesManager::on_get_dialogs_from_database(FolderId folder_id, int32 limit, DialogDbGetDialogsResult &&dialogs,
                                                   Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());
  CHECK(!td_->auth_manager_->is_bot());
  auto &folder = *get_dialog_folder(folder_id);
  LOG(INFO) << "Receive " << dialogs.dialogs.size() << " from expected " << limit << " chats in " << folder_id
            << " in from database with next order " << dialogs.next_order << " and next " << dialogs.next_dialog_id;
  int32 new_get_dialogs_limit = 0;
  int32 have_more_dialogs_in_database = (limit == static_cast<int32>(dialogs.dialogs.size()));
  if (have_more_dialogs_in_database && limit < folder.load_dialog_list_limit_max_) {
    new_get_dialogs_limit = folder.load_dialog_list_limit_max_ - limit;
  }
  folder.load_dialog_list_limit_max_ = 0;

  size_t dialogs_skipped = 0;
  for (auto &dialog : dialogs.dialogs) {
    Dialog *d = on_load_dialog_from_database(DialogId(), std::move(dialog), "on_get_dialogs_from_database");
    if (d == nullptr) {
      dialogs_skipped++;
      continue;
    }
    if (d->folder_id != folder_id) {
      LOG(WARNING) << "Skip " << d->dialog_id << " received from database, because it is in " << d->folder_id
                   << " instead of " << folder_id;
      dialogs_skipped++;
      continue;
    }

    LOG(INFO) << "Loaded from database " << d->dialog_id << " with order " << d->order;
  }

  DialogDate max_dialog_date(dialogs.next_order, dialogs.next_dialog_id);
  if (!have_more_dialogs_in_database) {
    folder.last_loaded_database_dialog_date_ = MAX_DIALOG_DATE;
    LOG(INFO) << "Set last loaded database dialog date to " << folder.last_loaded_database_dialog_date_;
    folder.last_server_dialog_date_ = max(folder.last_server_dialog_date_, folder.last_database_server_dialog_date_);
    LOG(INFO) << "Set last server dialog date to " << folder.last_server_dialog_date_;
    update_last_dialog_date(folder_id);
  } else if (folder.last_loaded_database_dialog_date_ < max_dialog_date) {
    folder.last_loaded_database_dialog_date_ = min(max_dialog_date, folder.last_database_server_dialog_date_);
    LOG(INFO) << "Set last loaded database dialog date to " << folder.last_loaded_database_dialog_date_;
    folder.last_server_dialog_date_ = max(folder.last_server_dialog_date_, folder.last_loaded_database_dialog_date_);
    LOG(INFO) << "Set last server dialog date to " << folder.last_server_dialog_date_;
    update_last_dialog_date(folder_id);

    for (const auto &list_it : dialog_lists_) {
      auto &list = list_it.second;
      if (!list.load_list_queries_.empty() && has_dialogs_from_folder(list, folder) && new_get_dialogs_limit < limit) {
        new_get_dialogs_limit = limit;
      }
    }
  } else {
    LOG(ERROR) << "Last loaded database dialog date didn't increased, skipped " << dialogs_skipped << " chats out of "
               << dialogs.dialogs.size();
  }

  if (!(folder.last_loaded_database_dialog_date_ < folder.last_database_server_dialog_date_)) {
    // have_more_dialogs_in_database = false;
    new_get_dialogs_limit = 0;
  }

  if (new_get_dialogs_limit == 0) {
    preload_folder_dialog_list_timeout_.add_timeout_in(folder_id.get(), 0.2);
    promise.set_value(Unit());
  } else {
    load_folder_dialog_list_from_database(folder_id, new_get_dialogs_limit, std::move(promise));
  }
}

void MessagesManager::preload_folder_dialog_list(FolderId folder_id) {
  if (G()->close_flag()) {
    LOG(INFO) << "Skip chat list preload in " << folder_id << " because of closing";
    return;
  }
  CHECK(!td_->auth_manager_->is_bot());

  auto &folder = *get_dialog_folder(folder_id);
  CHECK(G()->parameters().use_message_db);
  if (folder.load_folder_dialog_list_multipromise_.promise_count() != 0) {
    LOG(INFO) << "Skip chat list preload in " << folder_id << ", because there is a pending load chat list request";
    return;
  }

  if (folder.last_loaded_database_dialog_date_ < folder.last_database_server_dialog_date_) {
    // if there are some dialogs in database, preload some of them
    load_folder_dialog_list(folder_id, 20, true);
  } else if (folder.folder_last_dialog_date_ != MAX_DIALOG_DATE) {
    // otherwise load more dialogs from the server
    load_folder_dialog_list(folder_id, MAX_GET_DIALOGS, false);
  } else {
    recalc_unread_count(DialogListId(folder_id), -1, false);
  }
}

void MessagesManager::get_dialogs_from_list(DialogListId dialog_list_id, int32 limit,
                                            Promise<td_api::object_ptr<td_api::chats>> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());

  if (get_dialog_list(dialog_list_id) == nullptr) {
    return promise.set_error(Status::Error(400, "Chat list not found"));
  }

  if (limit <= 0) {
    return promise.set_error(Status::Error(400, "Parameter limit must be positive"));
  }

  auto task_id = ++current_get_dialogs_task_id_;
  auto &task = get_dialogs_tasks_[task_id];
  task.dialog_list_id = dialog_list_id;
  task.retry_count = 5;
  task.limit = limit;
  task.promise = std::move(promise);
  get_dialogs_from_list_impl(task_id);
}

void MessagesManager::get_dialogs_from_list_impl(int64 task_id) {
  auto task_it = get_dialogs_tasks_.find(task_id);
  CHECK(task_it != get_dialogs_tasks_.end());
  auto &task = task_it->second;
  auto promise = PromiseCreator::lambda([actor_id = actor_id(this), task_id](Result<Unit> &&result) {
    // on_get_dialogs_from_list can delete get_dialogs_tasks_[task_id], so it must be called later
    send_closure_later(actor_id, &MessagesManager::on_get_dialogs_from_list, task_id, std::move(result));
  });
  auto dialog_ids = get_dialogs(task.dialog_list_id, MIN_DIALOG_DATE, task.limit, true, false, std::move(promise));
  auto &list = *get_dialog_list(task.dialog_list_id);
  auto total_count = get_dialog_total_count(list);
  LOG(INFO) << "Receive " << dialog_ids.size() << " chats instead of " << task.limit << " out of " << total_count
            << " in " << task.dialog_list_id;
  CHECK(dialog_ids.size() <= static_cast<size_t>(total_count));
  CHECK(dialog_ids.size() <= static_cast<size_t>(task.limit));
  if (dialog_ids.size() == static_cast<size_t>(min(total_count, task.limit)) ||
      list.list_last_dialog_date_ == MAX_DIALOG_DATE || task.retry_count == 0) {
    auto task_promise = std::move(task.promise);
    get_dialogs_tasks_.erase(task_it);
    if (!task_promise) {
      dialog_ids.clear();
    }
    return task_promise.set_value(get_chats_object(total_count, dialog_ids));
  }
  // nor the limit, nor the end of the list were reached; wait for the promise
}

void MessagesManager::on_get_dialogs_from_list(int64 task_id, Result<Unit> &&result) {
  auto task_it = get_dialogs_tasks_.find(task_id);
  if (task_it == get_dialogs_tasks_.end()) {
    // the task has already been completed
    LOG(INFO) << "Chat list load task " << task_id << " has already been completed";
    return;
  }
  auto &task = task_it->second;
  if (result.is_error()) {
    LOG(INFO) << "Chat list load task " << task_id << " failed with the error " << result.error();
    auto task_promise = std::move(task.promise);
    get_dialogs_tasks_.erase(task_it);
    return task_promise.set_error(result.move_as_error());
  }

  auto list_ptr = get_dialog_list(task.dialog_list_id);
  CHECK(list_ptr != nullptr);
  auto &list = *list_ptr;
  if (task.last_dialog_date == list.list_last_dialog_date_) {
    // no new chats were loaded
    task.retry_count--;
  } else {
    CHECK(task.last_dialog_date < list.list_last_dialog_date_);
    task.last_dialog_date = list.list_last_dialog_date_;
    task.retry_count = 5;
  }
  get_dialogs_from_list_impl(task_id);
}

vector<DialogId> MessagesManager::get_pinned_dialog_ids(DialogListId dialog_list_id) const {
  CHECK(!td_->auth_manager_->is_bot());
  if (dialog_list_id.is_filter()) {
    const auto *filter = get_dialog_filter(dialog_list_id.get_filter_id());
    if (filter == nullptr) {
      return {};
    }
    return transform(filter->pinned_dialog_ids, [](auto &input_dialog) { return input_dialog.get_dialog_id(); });
  }

  auto *list = get_dialog_list(dialog_list_id);
  if (list == nullptr || !list->are_pinned_dialogs_inited_) {
    return {};
  }
  return transform(list->pinned_dialogs_, [](auto &pinned_dialog) { return pinned_dialog.get_dialog_id(); });
}

void MessagesManager::reload_pinned_dialogs(DialogListId dialog_list_id, Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());
  CHECK(!td_->auth_manager_->is_bot());

  if (dialog_list_id.is_folder()) {
    send_closure(td_->create_net_actor<GetPinnedDialogsActor>(std::move(promise)), &GetPinnedDialogsActor::send,
                 dialog_list_id.get_folder_id(), get_sequence_dispatcher_id(DialogId(), MessageContentType::None));
  } else if (dialog_list_id.is_filter()) {
    schedule_dialog_filters_reload(0.0);
    dialog_filter_reload_queries_.push_back(std::move(promise));
  }
}

double MessagesManager::get_dialog_filters_cache_time() {
  return DIALOG_FILTERS_CACHE_TIME * 0.0001 * Random::fast(9000, 11000);
}

void MessagesManager::schedule_dialog_filters_reload(double timeout) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }
  if (timeout <= 0) {
    timeout = 0.0;
    if (dialog_filters_updated_date_ != 0) {
      dialog_filters_updated_date_ = 0;
      save_dialog_filters();
    }
  }
  LOG(INFO) << "Schedule reload of chat filters in " << timeout;
  reload_dialog_filters_timeout_.set_callback(std::move(MessagesManager::on_reload_dialog_filters_timeout));
  reload_dialog_filters_timeout_.set_callback_data(static_cast<void *>(this));
  reload_dialog_filters_timeout_.set_timeout_in(timeout);
}

void MessagesManager::on_reload_dialog_filters_timeout(void *messages_manager_ptr) {
  if (G()->close_flag()) {
    return;
  }
  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::reload_dialog_filters);
}

void MessagesManager::reload_dialog_filters() {
  if (G()->close_flag()) {
    return;
  }
  CHECK(!td_->auth_manager_->is_bot());
  if (are_dialog_filters_being_synchronized_ || are_dialog_filters_being_reloaded_) {
    need_dialog_filters_reload_ = true;
    return;
  }
  LOG(INFO) << "Reload chat filters from server";
  are_dialog_filters_being_reloaded_ = true;
  need_dialog_filters_reload_ = false;
  auto promise = PromiseCreator::lambda(
      [actor_id = actor_id(this)](Result<vector<tl_object_ptr<telegram_api::dialogFilter>>> r_filters) {
        send_closure(actor_id, &MessagesManager::on_get_dialog_filters, std::move(r_filters), false);
      });
  td_->create_handler<GetDialogFiltersQuery>(std::move(promise))->send();
}

void MessagesManager::on_get_dialog_filters(Result<vector<tl_object_ptr<telegram_api::dialogFilter>>> r_filters,
                                            bool dummy) {
  are_dialog_filters_being_reloaded_ = false;
  if (G()->close_flag()) {
    return;
  }
  CHECK(!td_->auth_manager_->is_bot());
  auto promises = std::move(dialog_filter_reload_queries_);
  dialog_filter_reload_queries_.clear();
  if (r_filters.is_error()) {
    for (auto &promise : promises) {
      promise.set_error(r_filters.error().clone());
    }
    LOG(WARNING) << "Receive error " << r_filters.error() << " for GetDialogFiltersQuery";
    need_dialog_filters_reload_ = false;
    schedule_dialog_filters_reload(Random::fast(60, 5 * 60));
    return;
  }

  auto filters = r_filters.move_as_ok();
  vector<unique_ptr<DialogFilter>> new_server_dialog_filters;
  LOG(INFO) << "Receive " << filters.size() << " chat filters from server";
  std::unordered_set<DialogFilterId, DialogFilterIdHash> new_dialog_filter_ids;
  for (auto &filter : filters) {
    auto dialog_filter = DialogFilter::get_dialog_filter(std::move(filter), true);
    if (dialog_filter == nullptr) {
      continue;
    }
    if (!new_dialog_filter_ids.insert(dialog_filter->dialog_filter_id).second) {
      LOG(ERROR) << "Receive duplicate " << dialog_filter->dialog_filter_id;
      continue;
    }

    sort_dialog_filter_input_dialog_ids(dialog_filter.get(), "on_get_dialog_filters 1");
    new_server_dialog_filters.push_back(std::move(dialog_filter));
  }

  bool is_changed = false;
  dialog_filters_updated_date_ = G()->unix_time();
  if (server_dialog_filters_ != new_server_dialog_filters) {
    LOG(INFO) << "Change server chat filters from " << get_dialog_filter_ids(server_dialog_filters_) << " to "
              << get_dialog_filter_ids(new_server_dialog_filters);
    std::unordered_map<DialogFilterId, const DialogFilter *, DialogFilterIdHash> old_server_dialog_filters;
    for (const auto &filter : server_dialog_filters_) {
      old_server_dialog_filters.emplace(filter->dialog_filter_id, filter.get());
    }
    for (const auto &new_server_filter : new_server_dialog_filters) {
      auto dialog_filter_id = new_server_filter->dialog_filter_id;
      auto old_filter = get_dialog_filter(dialog_filter_id);
      auto it = old_server_dialog_filters.find(dialog_filter_id);
      if (it != old_server_dialog_filters.end()) {
        auto old_server_filter = it->second;
        if (*new_server_filter != *old_server_filter) {
          if (old_filter == nullptr) {
            // the filter was deleted, don't need to edit it
          } else {
            if (DialogFilter::are_equivalent(*old_filter, *new_server_filter)) {  // fast path
              // the filter was edited from this client, nothing to do
            } else {
              auto new_filter =
                  DialogFilter::merge_dialog_filter_changes(old_filter, old_server_filter, new_server_filter.get());
              LOG(INFO) << "Old  local filter: " << *old_filter;
              LOG(INFO) << "Old server filter: " << *old_server_filter;
              LOG(INFO) << "New server filter: " << *new_server_filter;
              LOG(INFO) << "New  local filter: " << *new_filter;
              sort_dialog_filter_input_dialog_ids(new_filter.get(), "on_get_dialog_filters 2");
              if (*new_filter != *old_filter) {
                is_changed = true;
                edit_dialog_filter(std::move(new_filter), "on_get_dialog_filters");
              }
            }
          }
        }
        old_server_dialog_filters.erase(it);
      } else {
        if (old_filter == nullptr) {
          // the filter was added from another client
          is_changed = true;
          add_dialog_filter(make_unique<DialogFilter>(*new_server_filter), false, "on_get_dialog_filters");
        } else {
          // the filter was added from this client
          // after that it could be added from another client, or edited from this client, or edited from another client
          // prefer local value, so do nothing
          // effectively, ignore edits from other clients, if didn't receive UpdateDialogFilterQuery response
        }
      }
    }
    vector<DialogFilterId> left_old_server_dialog_filter_ids;
    for (const auto &filter : server_dialog_filters_) {
      if (old_server_dialog_filters.count(filter->dialog_filter_id) == 0) {
        left_old_server_dialog_filter_ids.push_back(filter->dialog_filter_id);
      }
    }
    LOG(INFO) << "Still existing server chat filters: " << left_old_server_dialog_filter_ids;
    for (auto &old_server_filter : old_server_dialog_filters) {
      auto dialog_filter_id = old_server_filter.first;
      // deleted filter
      auto old_filter = get_dialog_filter(dialog_filter_id);
      if (old_filter == nullptr) {
        // the filter was deleted from this client, nothing to do
      } else {
        // the filter was deleted from another client
        // ignore edits done from the current client and just delete the filter
        is_changed = true;
        delete_dialog_filter(dialog_filter_id, "on_get_dialog_filters");
      }
    }
    bool is_order_changed = [&] {
      vector<DialogFilterId> new_server_dialog_filter_ids = get_dialog_filter_ids(new_server_dialog_filters);
      CHECK(new_server_dialog_filter_ids.size() >= left_old_server_dialog_filter_ids.size());
      new_server_dialog_filter_ids.resize(left_old_server_dialog_filter_ids.size());
      return new_server_dialog_filter_ids != left_old_server_dialog_filter_ids;
    }();
    if (is_order_changed) {  // if order is changed from this and other clients, prefer order from another client
      vector<DialogFilterId> new_dialog_filter_order;
      for (const auto &new_server_filter : new_server_dialog_filters) {
        auto dialog_filter_id = new_server_filter->dialog_filter_id;
        if (get_dialog_filter(dialog_filter_id) != nullptr) {
          new_dialog_filter_order.push_back(dialog_filter_id);
        }
      }
      is_changed = true;
      set_dialog_filters_order(dialog_filters_, new_dialog_filter_order);
    }

    server_dialog_filters_ = std::move(new_server_dialog_filters);
  }
  if (is_changed || !is_update_chat_filters_sent_) {
    send_update_chat_filters();
  }
  schedule_dialog_filters_reload(get_dialog_filters_cache_time());
  save_dialog_filters();

  if (need_synchronize_dialog_filters()) {
    synchronize_dialog_filters();
  }
  for (auto &promise : promises) {
    promise.set_value(Unit());
  }
}

bool MessagesManager::need_synchronize_dialog_filters() const {
  CHECK(!td_->auth_manager_->is_bot());
  size_t server_dialog_filter_count = 0;
  vector<DialogFilterId> dialog_filter_ids;
  for (const auto &dialog_filter : dialog_filters_) {
    if (dialog_filter->is_empty(true)) {
      continue;
    }

    server_dialog_filter_count++;
    auto server_dialog_filter = get_server_dialog_filter(dialog_filter->dialog_filter_id);
    if (server_dialog_filter == nullptr || !DialogFilter::are_equivalent(*server_dialog_filter, *dialog_filter)) {
      // need update dialog filter on server
      return true;
    }
    dialog_filter_ids.push_back(dialog_filter->dialog_filter_id);
  }
  if (server_dialog_filter_count != server_dialog_filters_.size()) {
    // need delete dialog filter on server
    return true;
  }
  if (dialog_filter_ids != get_dialog_filter_ids(server_dialog_filters_)) {
    // need reorder dialog filters on server
    return true;
  }
  return false;
}

void MessagesManager::synchronize_dialog_filters() {
  if (G()->close_flag()) {
    return;
  }
  CHECK(!td_->auth_manager_->is_bot());
  if (are_dialog_filters_being_synchronized_ || are_dialog_filters_being_reloaded_) {
    return;
  }
  if (need_dialog_filters_reload_) {
    return reload_dialog_filters();
  }
  if (!need_synchronize_dialog_filters()) {
    // reload filters to repair their order if the server added new filter to the beginning of the list
    return reload_dialog_filters();
  }

  LOG(INFO) << "Synchronize chat filter changes with server having local " << get_dialog_filter_ids(dialog_filters_)
            << " and server " << get_dialog_filter_ids(server_dialog_filters_);
  for (const auto &server_dialog_filter : server_dialog_filters_) {
    if (get_dialog_filter(server_dialog_filter->dialog_filter_id) == nullptr) {
      return delete_dialog_filter_on_server(server_dialog_filter->dialog_filter_id);
    }
  }

  vector<DialogFilterId> dialog_filter_ids;
  for (const auto &dialog_filter : dialog_filters_) {
    if (dialog_filter->is_empty(true)) {
      continue;
    }

    auto server_dialog_filter = get_server_dialog_filter(dialog_filter->dialog_filter_id);
    if (server_dialog_filter == nullptr || !DialogFilter::are_equivalent(*server_dialog_filter, *dialog_filter)) {
      return update_dialog_filter_on_server(make_unique<DialogFilter>(*dialog_filter));
    }
    dialog_filter_ids.push_back(dialog_filter->dialog_filter_id);
  }

  if (dialog_filter_ids != get_dialog_filter_ids(server_dialog_filters_)) {
    return reorder_dialog_filters_on_server(std::move(dialog_filter_ids));
  }

  UNREACHABLE();
}

vector<DialogId> MessagesManager::search_public_dialogs(const string &query, Promise<Unit> &&promise) {
  LOG(INFO) << "Search public chats with query = \"" << query << '"';

  if (utf8_length(query) < MIN_SEARCH_PUBLIC_DIALOG_PREFIX_LEN) {
    string username = clean_username(query);
    if (username[0] == '@') {
      username = username.substr(1);
    }

    for (auto &short_username : get_valid_short_usernames()) {
      if (2 * username.size() > short_username.size() && begins_with(short_username, username)) {
        username = short_username.str();
        auto it = resolved_usernames_.find(username);
        if (it == resolved_usernames_.end()) {
          td_->create_handler<ResolveUsernameQuery>(std::move(promise))->send(username);
          return {};
        }

        if (it->second.expires_at < Time::now()) {
          td_->create_handler<ResolveUsernameQuery>(Promise<>())->send(username);
        }

        auto dialog_id = it->second.dialog_id;
        force_create_dialog(dialog_id, "public dialogs search");

        auto d = get_dialog(dialog_id);
        if (d == nullptr || d->order != DEFAULT_ORDER ||
            (dialog_id.get_type() == DialogType::User &&
             td_->contacts_manager_->is_user_contact(dialog_id.get_user_id()))) {
          continue;
        }

        promise.set_value(Unit());
        return {dialog_id};
      }
    }
    promise.set_value(Unit());
    return {};
  }

  auto it = found_public_dialogs_.find(query);
  if (it != found_public_dialogs_.end()) {
    promise.set_value(Unit());
    return it->second;
  }

  send_search_public_dialogs_query(query, std::move(promise));
  return {};
}

void MessagesManager::send_search_public_dialogs_query(const string &query, Promise<Unit> &&promise) {
  auto &promises = search_public_dialogs_queries_[query];
  promises.push_back(std::move(promise));
  if (promises.size() != 1) {
    // query has already been sent, just wait for the result
    return;
  }

  td_->create_handler<SearchPublicDialogsQuery>()->send(query);
}

std::pair<int32, vector<DialogId>> MessagesManager::search_dialogs(const string &query, int32 limit,
                                                                   Promise<Unit> &&promise) {
  LOG(INFO) << "Search chats with query \"" << query << "\" and limit " << limit;
  CHECK(!td_->auth_manager_->is_bot());

  if (limit < 0) {
    promise.set_error(Status::Error(400, "Limit must be non-negative"));
    return {};
  }
  if (query.empty()) {
    return recently_found_dialogs_.get_dialogs(limit, std::move(promise));
  }

  auto result = dialogs_hints_.search(query, limit);
  vector<DialogId> dialog_ids;
  dialog_ids.reserve(result.second.size());
  for (auto key : result.second) {
    dialog_ids.push_back(DialogId(-key));
  }

  promise.set_value(Unit());
  return {narrow_cast<int32>(result.first), std::move(dialog_ids)};
}

std::pair<int32, vector<DialogId>> MessagesManager::get_recently_opened_dialogs(int32 limit, Promise<Unit> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());
  return recently_opened_dialogs_.get_dialogs(limit, std::move(promise));
}

vector<DialogId> MessagesManager::sort_dialogs_by_order(const vector<DialogId> &dialog_ids, int32 limit) const {
  CHECK(!td_->auth_manager_->is_bot());
  auto fake_order = static_cast<int64>(dialog_ids.size()) + 1;
  auto dialog_dates = transform(dialog_ids, [this, &fake_order](DialogId dialog_id) {
    const Dialog *d = get_dialog(dialog_id);
    CHECK(d != nullptr);
    auto order = get_dialog_base_order(d);
    if (is_dialog_inited(d) || order != DEFAULT_ORDER) {
      return DialogDate(order, dialog_id);
    }
    // if the dialog is not inited yet, we need to assume that server knows better and the dialog needs to be returned
    return DialogDate(fake_order--, dialog_id);
  });
  if (static_cast<size_t>(limit) >= dialog_dates.size()) {
    std::sort(dialog_dates.begin(), dialog_dates.end());
  } else {
    std::partial_sort(dialog_dates.begin(), dialog_dates.begin() + limit, dialog_dates.end());
    dialog_dates.resize(limit, MAX_DIALOG_DATE);
  }
  while (!dialog_dates.empty() && dialog_dates.back().get_order() == DEFAULT_ORDER) {
    dialog_dates.pop_back();
  }
  return transform(dialog_dates, [](auto dialog_date) { return dialog_date.get_dialog_id(); });
}

vector<DialogId> MessagesManager::search_dialogs_on_server(const string &query, int32 limit, Promise<Unit> &&promise) {
  LOG(INFO) << "Search chats on server with query \"" << query << "\" and limit " << limit;

  if (limit < 0) {
    promise.set_error(Status::Error(400, "Limit must be non-negative"));
    return {};
  }
  if (limit > MAX_GET_DIALOGS) {
    limit = MAX_GET_DIALOGS;
  }

  if (query.empty()) {
    promise.set_value(Unit());
    return {};
  }

  auto it = found_on_server_dialogs_.find(query);
  if (it != found_on_server_dialogs_.end()) {
    promise.set_value(Unit());
    return sort_dialogs_by_order(it->second, limit);
  }

  send_search_public_dialogs_query(query, std::move(promise));
  return {};
}

void MessagesManager::drop_common_dialogs_cache(UserId user_id) {
  auto it = found_common_dialogs_.find(user_id);
  if (it != found_common_dialogs_.end()) {
    it->second.is_outdated = true;
  }
}

std::pair<int32, vector<DialogId>> MessagesManager::get_common_dialogs(UserId user_id, DialogId offset_dialog_id,
                                                                       int32 limit, bool force,
                                                                       Promise<Unit> &&promise) {
  if (!td_->contacts_manager_->have_input_user(user_id)) {
    promise.set_error(Status::Error(400, "Have no access to the user"));
    return {};
  }

  if (user_id == td_->contacts_manager_->get_my_id()) {
    promise.set_error(Status::Error(400, "Can't get common chats with self"));
    return {};
  }
  if (limit <= 0) {
    promise.set_error(Status::Error(400, "Parameter limit must be positive"));
    return {};
  }
  if (limit > MAX_GET_DIALOGS) {
    limit = MAX_GET_DIALOGS;
  }

  int64 offset_chat_id = 0;
  switch (offset_dialog_id.get_type()) {
    case DialogType::Chat:
      offset_chat_id = offset_dialog_id.get_chat_id().get();
      break;
    case DialogType::Channel:
      offset_chat_id = offset_dialog_id.get_channel_id().get();
      break;
    case DialogType::None:
      if (offset_dialog_id == DialogId()) {
        break;
      }
    // fallthrough
    case DialogType::User:
    case DialogType::SecretChat:
      promise.set_error(Status::Error(400, "Wrong offset_chat_id"));
      return {};
    default:
      UNREACHABLE();
      break;
  }

  auto it = found_common_dialogs_.find(user_id);
  if (it != found_common_dialogs_.end() && !it->second.dialog_ids.empty()) {
    int32 total_count = it->second.total_count;
    vector<DialogId> &common_dialog_ids = it->second.dialog_ids;
    bool use_cache = (!it->second.is_outdated && it->second.received_date >= Time::now() - 3600) || force ||
                     offset_chat_id != 0 || common_dialog_ids.size() >= static_cast<size_t>(MAX_GET_DIALOGS);
    // use cache if it is up-to-date, or we required to use it or we can't update it
    if (use_cache) {
      auto offset_it = common_dialog_ids.begin();
      if (offset_dialog_id != DialogId()) {
        offset_it = std::find(common_dialog_ids.begin(), common_dialog_ids.end(), offset_dialog_id);
        if (offset_it == common_dialog_ids.end()) {
          promise.set_error(Status::Error(400, "Wrong offset_chat_id"));
          return {};
        }
        ++offset_it;
      }
      vector<DialogId> result;
      while (result.size() < static_cast<size_t>(limit)) {
        if (offset_it == common_dialog_ids.end()) {
          break;
        }
        auto dialog_id = *offset_it++;
        if (dialog_id == DialogId()) {  // end of the list
          promise.set_value(Unit());
          return {total_count, std::move(result)};
        }
        result.push_back(dialog_id);
      }
      if (result.size() == static_cast<size_t>(limit) || force) {
        promise.set_value(Unit());
        return {total_count, std::move(result)};
      }
    }
  }

  td_->create_handler<GetCommonDialogsQuery>(std::move(promise))->send(user_id, offset_chat_id, MAX_GET_DIALOGS);
  return {};
}

void MessagesManager::on_get_common_dialogs(UserId user_id, int64 offset_chat_id,
                                            vector<tl_object_ptr<telegram_api::Chat>> &&chats, int32 total_count) {
  td_->contacts_manager_->on_update_user_common_chat_count(user_id, total_count);

  auto &common_dialogs = found_common_dialogs_[user_id];
  if (common_dialogs.is_outdated && offset_chat_id == 0 &&
      common_dialogs.dialog_ids.size() < static_cast<size_t>(MAX_GET_DIALOGS)) {
    // drop outdated cache if possible
    common_dialogs = CommonDialogs();
  }
  if (common_dialogs.received_date == 0) {
    common_dialogs.received_date = Time::now();
  }
  common_dialogs.is_outdated = false;
  auto &result = common_dialogs.dialog_ids;
  if (!result.empty() && result.back() == DialogId()) {
    return;
  }
  bool is_last = chats.empty() && offset_chat_id == 0;
  for (auto &chat : chats) {
    DialogId dialog_id;
    switch (chat->get_id()) {
      case telegram_api::chatEmpty::ID: {
        auto c = static_cast<const telegram_api::chatEmpty *>(chat.get());
        ChatId chat_id(c->id_);
        if (!chat_id.is_valid()) {
          LOG(ERROR) << "Receive invalid " << chat_id;
          continue;
        }
        dialog_id = DialogId(chat_id);
        break;
      }
      case telegram_api::chat::ID: {
        auto c = static_cast<const telegram_api::chat *>(chat.get());
        ChatId chat_id(c->id_);
        if (!chat_id.is_valid()) {
          LOG(ERROR) << "Receive invalid " << chat_id;
          continue;
        }
        dialog_id = DialogId(chat_id);
        break;
      }
      case telegram_api::chatForbidden::ID: {
        auto c = static_cast<const telegram_api::chatForbidden *>(chat.get());
        ChatId chat_id(c->id_);
        if (!chat_id.is_valid()) {
          LOG(ERROR) << "Receive invalid " << chat_id;
          continue;
        }
        dialog_id = DialogId(chat_id);
        break;
      }
      case telegram_api::channel::ID: {
        auto c = static_cast<const telegram_api::channel *>(chat.get());
        ChannelId channel_id(c->id_);
        if (!channel_id.is_valid()) {
          LOG(ERROR) << "Receive invalid " << channel_id;
          continue;
        }
        dialog_id = DialogId(channel_id);
        break;
      }
      case telegram_api::channelForbidden::ID: {
        auto c = static_cast<const telegram_api::channelForbidden *>(chat.get());
        ChannelId channel_id(c->id_);
        if (!channel_id.is_valid()) {
          LOG(ERROR) << "Receive invalid " << channel_id;
          continue;
        }
        dialog_id = DialogId(channel_id);
        break;
      }
      default:
        UNREACHABLE();
    }
    CHECK(dialog_id.is_valid());
    td_->contacts_manager_->on_get_chat(std::move(chat), "on_get_common_dialogs");

    if (!td::contains(result, dialog_id)) {
      force_create_dialog(dialog_id, "get common dialogs");
      result.push_back(dialog_id);
    }
  }
  if (result.size() >= static_cast<size_t>(total_count) || is_last) {
    if (result.size() != static_cast<size_t>(total_count)) {
      LOG(ERROR) << "Fix total count of common groups with " << user_id << " from " << total_count << " to "
                 << result.size();
      total_count = narrow_cast<int32>(result.size());
      td_->contacts_manager_->on_update_user_common_chat_count(user_id, total_count);
    }

    result.emplace_back();
  }
  common_dialogs.total_count = total_count;
}

void MessagesManager::block_message_sender_from_replies(MessageId message_id, bool need_delete_message,
                                                        bool need_delete_all_messages, bool report_spam,
                                                        Promise<Unit> &&promise) {
  auto dialog_id = DialogId(ContactsManager::get_replies_bot_user_id());
  Dialog *d = get_dialog_force(dialog_id, "block_message_sender_from_replies");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return promise.set_error(Status::Error(400, "Not enough rights"));
  }

  auto *m = get_message_force(d, message_id, "block_message_sender_from_replies");
  if (m == nullptr) {
    return promise.set_error(Status::Error(400, "Message not found"));
  }
  if (m->is_outgoing || m->message_id.is_scheduled() || !m->message_id.is_server()) {
    return promise.set_error(Status::Error(400, "Wrong message specified"));
  }

  UserId sender_user_id;
  if (m->forward_info != nullptr) {
    sender_user_id = m->forward_info->sender_user_id;
  }
  bool need_update_dialog_pos = false;
  vector<int64> deleted_message_ids;
  if (need_delete_message) {
    auto p = delete_message(d, message_id, true, &need_update_dialog_pos, "block_message_sender_from_replies");
    CHECK(p.get() == m);
    deleted_message_ids.push_back(p->message_id.get());
  }
  if (need_delete_all_messages && sender_user_id.is_valid()) {
    vector<MessageId> message_ids;
    find_messages(d->messages.get(), message_ids, [sender_user_id](const Message *m) {
      return !m->is_outgoing && m->forward_info != nullptr && m->forward_info->sender_user_id == sender_user_id;
    });

    for (auto user_message_id : message_ids) {
      auto p = delete_message(d, user_message_id, true, &need_update_dialog_pos, "block_message_sender_from_replies 2");
      deleted_message_ids.push_back(p->message_id.get());
    }
  }

  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "block_message_sender_from_replies");
  }

  send_update_delete_messages(dialog_id, std::move(deleted_message_ids), true, false);

  block_message_sender_from_replies_on_server(message_id, need_delete_message, need_delete_all_messages, report_spam, 0,
                                              std::move(promise));
}

class MessagesManager::BlockMessageSenderFromRepliesOnServerLogEvent {
 public:
  MessageId message_id_;
  bool delete_message_;
  bool delete_all_messages_;
  bool report_spam_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(delete_message_);
    STORE_FLAG(delete_all_messages_);
    STORE_FLAG(report_spam_);
    END_STORE_FLAGS();

    td::store(message_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(delete_message_);
    PARSE_FLAG(delete_all_messages_);
    PARSE_FLAG(report_spam_);
    END_PARSE_FLAGS();

    td::parse(message_id_, parser);
  }
};

uint64 MessagesManager::save_block_message_sender_from_replies_on_server_log_event(MessageId message_id,
                                                                                   bool need_delete_message,
                                                                                   bool need_delete_all_messages,
                                                                                   bool report_spam) {
  BlockMessageSenderFromRepliesOnServerLogEvent log_event{message_id, need_delete_message, need_delete_all_messages,
                                                          report_spam};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::BlockMessageSenderFromRepliesOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::block_message_sender_from_replies_on_server(MessageId message_id, bool need_delete_message,
                                                                  bool need_delete_all_messages, bool report_spam,
                                                                  uint64 log_event_id, Promise<Unit> &&promise) {
  if (log_event_id == 0) {
    log_event_id = save_block_message_sender_from_replies_on_server_log_event(message_id, need_delete_message,
                                                                              need_delete_all_messages, report_spam);
  }

  td_->create_handler<BlockFromRepliesQuery>(get_erase_log_event_promise(log_event_id, std::move(promise)))
      ->send(message_id, need_delete_message, need_delete_all_messages, report_spam);
}

void MessagesManager::get_blocked_dialogs(int32 offset, int32 limit,
                                          Promise<td_api::object_ptr<td_api::messageSenders>> &&promise) {
  if (offset < 0) {
    return promise.set_error(Status::Error(400, "Parameter offset must be non-negative"));
  }

  if (limit <= 0) {
    return promise.set_error(Status::Error(400, "Parameter limit must be positive"));
  }

  td_->create_handler<GetBlockedDialogsQuery>(std::move(promise))->send(offset, limit);
}

void MessagesManager::on_get_blocked_dialogs(int32 offset, int32 limit, int32 total_count,
                                             vector<tl_object_ptr<telegram_api::peerBlocked>> &&blocked_peers,
                                             Promise<td_api::object_ptr<td_api::messageSenders>> &&promise) {
  LOG(INFO) << "Receive " << blocked_peers.size() << " blocked chats from offset " << offset << " out of "
            << total_count;
  vector<DialogId> dialog_ids;
  for (auto &blocked_peer : blocked_peers) {
    CHECK(blocked_peer != nullptr);
    DialogId dialog_id(blocked_peer->peer_id_);
    if (dialog_id.get_type() == DialogType::User) {
      if (td_->contacts_manager_->have_user(dialog_id.get_user_id())) {
        dialog_ids.push_back(dialog_id);
      } else {
        LOG(ERROR) << "Have no info about " << dialog_id.get_user_id();
      }
    } else {
      if (have_dialog_info(dialog_id)) {
        force_create_dialog(dialog_id, "on_get_blocked_dialogs");
        if (have_dialog(dialog_id)) {
          dialog_ids.push_back(dialog_id);
        }
      } else {
        LOG(ERROR) << "Have no info about " << dialog_id;
      }
    }
  }
  if (!dialog_ids.empty() && offset + dialog_ids.size() > static_cast<size_t>(total_count)) {
    LOG(ERROR) << "Fix total count of blocked chats from " << total_count << " to " << offset + dialog_ids.size();
    total_count = offset + narrow_cast<int32>(dialog_ids.size());
  }

  auto senders = transform(dialog_ids, [this](DialogId dialog_id) {
    return get_message_sender_object(dialog_id, "on_get_blocked_dialogs");
  });
  promise.set_value(td_api::make_object<td_api::messageSenders>(total_count, std::move(senders)));
}

bool MessagesManager::have_message_force(FullMessageId full_message_id, const char *source) {
  return get_message_force(full_message_id, source) != nullptr;
}

bool MessagesManager::have_message_force(Dialog *d, MessageId message_id, const char *source) {
  return get_message_force(d, message_id, source) != nullptr;
}

MessagesManager::Message *MessagesManager::get_message(FullMessageId full_message_id) {
  Dialog *d = get_dialog(full_message_id.get_dialog_id());
  if (d == nullptr) {
    return nullptr;
  }

  return get_message(d, full_message_id.get_message_id());
}

const MessagesManager::Message *MessagesManager::get_message(FullMessageId full_message_id) const {
  const Dialog *d = get_dialog(full_message_id.get_dialog_id());
  if (d == nullptr) {
    return nullptr;
  }

  return get_message(d, full_message_id.get_message_id());
}

MessagesManager::Message *MessagesManager::get_message_force(FullMessageId full_message_id, const char *source) {
  Dialog *d = get_dialog_force(full_message_id.get_dialog_id(), source);
  if (d == nullptr) {
    return nullptr;
  }

  return get_message_force(d, full_message_id.get_message_id(), source);
}

FullMessageId MessagesManager::get_replied_message_id(DialogId dialog_id, const Message *m) {
  auto full_message_id = get_message_content_replied_message_id(dialog_id, m->content.get());
  if (full_message_id.get_message_id().is_valid()) {
    CHECK(!m->reply_to_message_id.is_valid());
    return full_message_id;
  }
  if (!m->reply_to_message_id.is_valid()) {
    return {};
  }
  return {m->reply_in_dialog_id.is_valid() ? m->reply_in_dialog_id : dialog_id, m->reply_to_message_id};
}

void MessagesManager::get_message_force_from_server(Dialog *d, MessageId message_id, Promise<Unit> &&promise,
                                                    tl_object_ptr<telegram_api::InputMessage> input_message) {
  LOG(INFO) << "Get " << message_id << " in " << d->dialog_id << " using " << to_string(input_message);
  auto dialog_type = d->dialog_id.get_type();
  auto m = get_message_force(d, message_id, "get_message_force_from_server");
  if (m == nullptr) {
    if (message_id.is_valid() && message_id.is_server()) {
      if (d->last_new_message_id != MessageId() && message_id > d->last_new_message_id &&
          dialog_type != DialogType::Channel) {
        // message will not be added to the dialog anyway
        return promise.set_value(Unit());
      }

      if (d->deleted_message_ids.count(message_id) == 0 && dialog_type != DialogType::SecretChat) {
        return get_message_from_server({d->dialog_id, message_id}, std::move(promise), "get_message_force_from_server",
                                       std::move(input_message));
      }
    } else if (message_id.is_valid_scheduled() && message_id.is_scheduled_server()) {
      if (d->deleted_scheduled_server_message_ids.count(message_id.get_scheduled_server_message_id()) == 0 &&
          dialog_type != DialogType::SecretChat && input_message == nullptr) {
        return get_message_from_server({d->dialog_id, message_id}, std::move(promise), "get_message_force_from_server");
      }
    }
  }

  promise.set_value(Unit());
}

void MessagesManager::get_message(FullMessageId full_message_id, Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(full_message_id.get_dialog_id(), "get_message");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }

  get_message_force_from_server(d, full_message_id.get_message_id(), std::move(promise));
}

FullMessageId MessagesManager::get_replied_message(DialogId dialog_id, MessageId message_id, bool force,
                                                   Promise<Unit> &&promise) {
  LOG(INFO) << "Get replied message to " << message_id << " in " << dialog_id;
  Dialog *d = get_dialog_force(dialog_id, "get_replied_message");
  if (d == nullptr) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return FullMessageId();
  }

  message_id = get_persistent_message_id(d, message_id);
  auto m = get_message_force(d, message_id, "get_replied_message");
  if (m == nullptr) {
    if (force) {
      promise.set_value(Unit());
    } else {
      get_message_force_from_server(d, message_id, std::move(promise));
    }
    return FullMessageId();
  }

  tl_object_ptr<telegram_api::InputMessage> input_message;
  auto replied_message_id = get_replied_message_id(dialog_id, m);
  if (replied_message_id.get_dialog_id() != dialog_id) {
    dialog_id = replied_message_id.get_dialog_id();
    if (!have_dialog_info_force(dialog_id)) {
      promise.set_value(Unit());
      return {};
    }
    if (!have_input_peer(dialog_id, AccessRights::Read)) {
      promise.set_value(Unit());
      return {};
    }

    force_create_dialog(dialog_id, "get_replied_message");
    d = get_dialog_force(dialog_id, "get_replied_message");
    if (d == nullptr) {
      promise.set_error(Status::Error(500, "Chat with replied message not found"));
      return {};
    }
  } else if (m->message_id.is_valid() && m->message_id.is_server()) {
    input_message = make_tl_object<telegram_api::inputMessageReplyTo>(m->message_id.get_server_message_id().get());
  }
  get_message_force_from_server(d, replied_message_id.get_message_id(), std::move(promise), std::move(input_message));

  return replied_message_id;
}

Result<FullMessageId> MessagesManager::get_top_thread_full_message_id(DialogId dialog_id, const Message *m) const {
  CHECK(m != nullptr);
  if (m->message_id.is_scheduled()) {
    return Status::Error(400, "Message is scheduled");
  }
  if (dialog_id.get_type() != DialogType::Channel) {
    return Status::Error(400, "Chat can't have message threads");
  }
  if (!m->reply_info.is_empty() && m->reply_info.is_comment) {
    if (!is_visible_message_reply_info(dialog_id, m)) {
      return Status::Error(400, "Message has no comments");
    }
    if (m->message_id.is_yet_unsent()) {
      return Status::Error(400, "Message is not sent yet");
    }
    return FullMessageId{DialogId(m->reply_info.channel_id), m->linked_top_thread_message_id};
  } else {
    if (!m->top_thread_message_id.is_valid()) {
      return Status::Error(400, "Message has no thread");
    }
    if (!m->message_id.is_server()) {
      return Status::Error(400, "Message thread is unavailable for the message");
    }
    if (m->top_thread_message_id != m->message_id &&
        !td_->contacts_manager_->get_channel_has_linked_channel(dialog_id.get_channel_id())) {
      return Status::Error(400, "Root message must be used to get the message thread");
    }
    return FullMessageId{dialog_id, m->top_thread_message_id};
  }
}

void MessagesManager::get_message_thread(DialogId dialog_id, MessageId message_id,
                                         Promise<MessageThreadInfo> &&promise) {
  LOG(INFO) << "Get message thread from " << message_id << " in " << dialog_id;
  Dialog *d = get_dialog_force(dialog_id, "get_message_thread");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return promise.set_error(Status::Error(400, "Can't access the chat"));
  }
  if (dialog_id.get_type() != DialogType::Channel) {
    return promise.set_error(Status::Error(400, "Chat is not a supergroup or a channel"));
  }
  if (message_id.is_scheduled()) {
    return promise.set_error(Status::Error(400, "Scheduled messages can't have message threads"));
  }

  auto m = get_message_force(d, message_id, "get_message_thread");
  if (m == nullptr) {
    return promise.set_error(Status::Error(400, "Message not found"));
  }

  TRY_RESULT_PROMISE(promise, top_thread_full_message_id, get_top_thread_full_message_id(dialog_id, m));

  auto query_promise = PromiseCreator::lambda([actor_id = actor_id(this), dialog_id, message_id,
                                               promise = std::move(promise)](Result<MessageThreadInfo> result) mutable {
    if (result.is_error()) {
      return promise.set_error(result.move_as_error());
    }
    send_closure(actor_id, &MessagesManager::on_get_discussion_message, dialog_id, message_id, result.move_as_ok(),
                 std::move(promise));
  });

  td_->create_handler<GetDiscussionMessageQuery>(std::move(query_promise))
      ->send(dialog_id, message_id, top_thread_full_message_id.get_dialog_id(),
             top_thread_full_message_id.get_message_id());
}

void MessagesManager::process_discussion_message(
    telegram_api::object_ptr<telegram_api::messages_discussionMessage> &&result, DialogId dialog_id,
    MessageId message_id, DialogId expected_dialog_id, MessageId expected_message_id,
    Promise<MessageThreadInfo> promise) {
  LOG(INFO) << "Receive discussion message for " << message_id << " in " << dialog_id << ": " << to_string(result);
  td_->contacts_manager_->on_get_users(std::move(result->users_), "process_discussion_message");
  td_->contacts_manager_->on_get_chats(std::move(result->chats_), "process_discussion_message");

  for (auto &message : result->messages_) {
    auto message_dialog_id = get_message_dialog_id(message);
    if (message_dialog_id != expected_dialog_id) {
      return promise.set_error(Status::Error(500, "Expected messages in a different chat"));
    }
  }

  for (auto &message : result->messages_) {
    if (need_channel_difference_to_add_message(expected_dialog_id, message)) {
      return run_after_channel_difference(
          expected_dialog_id, PromiseCreator::lambda([actor_id = actor_id(this), result = std::move(result), dialog_id,
                                                      message_id, expected_dialog_id, expected_message_id,
                                                      promise = std::move(promise)](Unit ignored) mutable {
            send_closure(actor_id, &MessagesManager::process_discussion_message_impl, std::move(result), dialog_id,
                         message_id, expected_dialog_id, expected_message_id, std::move(promise));
          }));
    }
  }

  process_discussion_message_impl(std::move(result), dialog_id, message_id, expected_dialog_id, expected_message_id,
                                  std::move(promise));
}

void MessagesManager::process_discussion_message_impl(
    telegram_api::object_ptr<telegram_api::messages_discussionMessage> &&result, DialogId dialog_id,
    MessageId message_id, DialogId expected_dialog_id, MessageId expected_message_id,
    Promise<MessageThreadInfo> promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  MessageId max_message_id;
  MessageId last_read_inbox_message_id;
  MessageId last_read_outbox_message_id;
  if ((result->flags_ & telegram_api::messages_discussionMessage::MAX_ID_MASK) != 0) {
    max_message_id = MessageId(ServerMessageId(result->max_id_));
  }
  if ((result->flags_ & telegram_api::messages_discussionMessage::READ_INBOX_MAX_ID_MASK) != 0) {
    last_read_inbox_message_id = MessageId(ServerMessageId(result->read_inbox_max_id_));
  }
  if ((result->flags_ & telegram_api::messages_discussionMessage::READ_OUTBOX_MAX_ID_MASK) != 0) {
    last_read_outbox_message_id = MessageId(ServerMessageId(result->read_outbox_max_id_));
  }

  MessageThreadInfo message_thread_info;
  message_thread_info.dialog_id = expected_dialog_id;
  message_thread_info.unread_message_count = max(0, result->unread_count_);
  MessageId top_message_id;
  for (auto &message : result->messages_) {
    auto full_message_id =
        on_get_message(std::move(message), false, true, false, false, false, "process_discussion_message_impl");
    if (full_message_id.get_message_id().is_valid()) {
      CHECK(full_message_id.get_dialog_id() == expected_dialog_id);
      message_thread_info.message_ids.push_back(full_message_id.get_message_id());
      if (full_message_id.get_message_id() == expected_message_id) {
        top_message_id = expected_message_id;
      }
    }
  }
  if (!message_thread_info.message_ids.empty() && !top_message_id.is_valid()) {
    top_message_id = message_thread_info.message_ids.back();
  }
  if (top_message_id.is_valid()) {
    on_update_read_message_comments(expected_dialog_id, top_message_id, max_message_id, last_read_inbox_message_id,
                                    last_read_outbox_message_id);
  }
  if (expected_dialog_id != dialog_id) {
    on_update_read_message_comments(dialog_id, message_id, max_message_id, last_read_inbox_message_id,
                                    last_read_outbox_message_id);
  }
  promise.set_value(std::move(message_thread_info));
}

void MessagesManager::on_get_discussion_message(DialogId dialog_id, MessageId message_id,
                                                MessageThreadInfo &&message_thread_info,
                                                Promise<MessageThreadInfo> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  Dialog *d = get_dialog_force(dialog_id, "on_get_discussion_message");
  CHECK(d != nullptr);

  auto m = get_message_force(d, message_id, "on_get_discussion_message");
  if (m == nullptr) {
    return promise.set_error(Status::Error(400, "Message not found"));
  }
  if (message_thread_info.message_ids.empty()) {
    return promise.set_error(Status::Error(400, "Message has no thread"));
  }

  DialogId expected_dialog_id;
  if (m->reply_info.is_comment) {
    if (!is_active_message_reply_info(dialog_id, m->reply_info)) {
      return promise.set_error(Status::Error(400, "Message has no comments"));
    }
    expected_dialog_id = DialogId(m->reply_info.channel_id);
  } else {
    if (!m->top_thread_message_id.is_valid()) {
      return promise.set_error(Status::Error(400, "Message has no thread"));
    }
    expected_dialog_id = dialog_id;
  }

  if (expected_dialog_id != dialog_id && m->reply_info.is_comment &&
      m->linked_top_thread_message_id != message_thread_info.message_ids.back()) {
    auto linked_d = get_dialog_force(expected_dialog_id, "on_get_discussion_message 2");
    CHECK(linked_d != nullptr);
    auto linked_message_id = message_thread_info.message_ids.back();
    Message *linked_m = get_message_force(linked_d, linked_message_id, "on_get_discussion_message 3");
    CHECK(linked_m != nullptr && linked_m->message_id.is_server());
    if (linked_m->top_thread_message_id == linked_m->message_id &&
        is_active_message_reply_info(expected_dialog_id, linked_m->reply_info)) {
      if (m->linked_top_thread_message_id.is_valid()) {
        LOG(ERROR) << "Comment message identifier for " << message_id << " in " << dialog_id << " changed from "
                   << m->linked_top_thread_message_id << " to " << linked_message_id;
      }
      m->linked_top_thread_message_id = linked_message_id;
      on_dialog_updated(dialog_id, "on_get_discussion_message");
    }
  }
  promise.set_value(std::move(message_thread_info));
}

td_api::object_ptr<td_api::messageThreadInfo> MessagesManager::get_message_thread_info_object(
    const MessageThreadInfo &info) {
  if (info.message_ids.empty()) {
    return nullptr;
  }

  Dialog *d = get_dialog(info.dialog_id);
  CHECK(d != nullptr);
  td_api::object_ptr<td_api::messageReplyInfo> reply_info;
  vector<td_api::object_ptr<td_api::message>> messages;
  messages.reserve(info.message_ids.size());
  for (auto message_id : info.message_ids) {
    const Message *m = get_message_force(d, message_id, "get_message_thread_info_object");
    auto message = get_message_object(d->dialog_id, m, "get_message_thread_info_object");
    if (message != nullptr) {
      if (message->interaction_info_ != nullptr && message->interaction_info_->reply_info_ != nullptr) {
        reply_info = m->reply_info.get_message_reply_info_object(td_->contacts_manager_.get(), this);
        CHECK(reply_info != nullptr);
      }
      messages.push_back(std::move(message));
    }
  }
  if (reply_info == nullptr) {
    return nullptr;
  }

  MessageId top_thread_message_id;
  td_api::object_ptr<td_api::draftMessage> draft_message;
  if (!info.message_ids.empty()) {
    top_thread_message_id = info.message_ids.back();
    if (can_send_message(d->dialog_id).is_ok()) {
      const Message *m = get_message_force(d, top_thread_message_id, "get_message_thread_info_object 2");
      if (m != nullptr && !m->reply_info.is_comment && is_active_message_reply_info(d->dialog_id, m->reply_info)) {
        draft_message = get_draft_message_object(m->thread_draft_message);
      }
    }
  }
  return td_api::make_object<td_api::messageThreadInfo>(d->dialog_id.get(), top_thread_message_id.get(),
                                                        std::move(reply_info), info.unread_message_count,
                                                        std::move(messages), std::move(draft_message));
}

Status MessagesManager::can_get_message_viewers(FullMessageId full_message_id) {
  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id, "get_message_viewers");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }

  auto m = get_message_force(d, full_message_id.get_message_id(), "get_message_viewers");
  if (m == nullptr) {
    return Status::Error(400, "Message not found");
  }

  return can_get_message_viewers(dialog_id, m);
}

Status MessagesManager::can_get_message_viewers(DialogId dialog_id, const Message *m) const {
  if (td_->auth_manager_->is_bot()) {
    return Status::Error(400, "User is bot");
  }
  if (!m->is_outgoing) {
    return Status::Error(400, "Can't get viewers of incoming messages");
  }
  if (G()->unix_time() - m->date > G()->shared_config().get_option_integer("chat_read_mark_expire_period", 7 * 86400)) {
    return Status::Error(400, "Message is too old");
  }

  int32 participant_count = 0;
  switch (dialog_id.get_type()) {
    case DialogType::User:
      return Status::Error(400, "Can't get message viewers in private chats");
    case DialogType::Chat:
      if (!td_->contacts_manager_->get_chat_is_active(dialog_id.get_chat_id())) {
        return Status::Error(400, "Chat is deactivated");
      }
      participant_count = td_->contacts_manager_->get_chat_participant_count(dialog_id.get_chat_id());
      break;
    case DialogType::Channel:
      if (is_broadcast_channel(dialog_id)) {
        return Status::Error(400, "Can't get message viewers in channel chats");
      }
      participant_count = td_->contacts_manager_->get_channel_participant_count(dialog_id.get_channel_id());
      break;
    case DialogType::SecretChat:
      return Status::Error(400, "Can't get message viewers in secret chats");
    case DialogType::None:
    default:
      UNREACHABLE();
      return Status::OK();
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return Status::Error(400, "Can't access the chat");
  }
  if (participant_count == 0) {
    return Status::Error(400, "Chat is empty or have unknown number of members");
  }
  if (participant_count > G()->shared_config().get_option_integer("chat_read_mark_size_threshold", 50)) {
    return Status::Error(400, "Chat is too big");
  }

  if (m->message_id.is_scheduled()) {
    return Status::Error(400, "Scheduled messages can't have viewers");
  }
  if (m->message_id.is_yet_unsent()) {
    return Status::Error(400, "Yet unsent messages can't have viewers");
  }
  if (m->message_id.is_local()) {
    return Status::Error(400, "Local messages can't have viewers");
  }
  CHECK(m->message_id.is_server());

  return Status::OK();
}

void MessagesManager::get_message_viewers(FullMessageId full_message_id,
                                          Promise<td_api::object_ptr<td_api::users>> &&promise) {
  TRY_STATUS_PROMISE(promise, can_get_message_viewers(full_message_id));

  auto query_promise = PromiseCreator::lambda([actor_id = actor_id(this), dialog_id = full_message_id.get_dialog_id(),
                                               promise = std::move(promise)](Result<vector<UserId>> result) mutable {
    if (result.is_error()) {
      return promise.set_error(result.move_as_error());
    }
    send_closure(actor_id, &MessagesManager::on_get_message_viewers, dialog_id, result.move_as_ok(), false,
                 std::move(promise));
  });

  td_->create_handler<GetMessageReadParticipantsQuery>(std::move(query_promise))
      ->send(full_message_id.get_dialog_id(), full_message_id.get_message_id());
}

void MessagesManager::on_get_message_viewers(DialogId dialog_id, vector<UserId> user_ids, bool is_recursive,
                                             Promise<td_api::object_ptr<td_api::users>> &&promise) {
  if (!is_recursive) {
    bool need_participant_list = false;
    for (auto user_id : user_ids) {
      if (!user_id.is_valid()) {
        LOG(ERROR) << "Receive invalid " << user_id << " as viewer of a message in " << dialog_id;
        continue;
      }
      if (!td_->contacts_manager_->have_user_force(user_id)) {
        need_participant_list = true;
      }
    }
    if (need_participant_list) {
      auto query_promise = PromiseCreator::lambda([actor_id = actor_id(this), dialog_id, user_ids = std::move(user_ids),
                                                   promise = std::move(promise)](Unit result) mutable {
        send_closure(actor_id, &MessagesManager::on_get_message_viewers, dialog_id, std::move(user_ids), true,
                     std::move(promise));
      });

      switch (dialog_id.get_type()) {
        case DialogType::Chat:
          return td_->contacts_manager_->reload_chat_full(dialog_id.get_chat_id(), std::move(query_promise));
        case DialogType::Channel:
          return td_->contacts_manager_->get_channel_participants(
              dialog_id.get_channel_id(), td_api::make_object<td_api::supergroupMembersFilterRecent>(), string(), 0,
              200, 200, PromiseCreator::lambda([query_promise = std::move(query_promise)](DialogParticipants) mutable {
                query_promise.set_value(Unit());
              }));
        default:
          UNREACHABLE();
          return;
      }
    }
  }
  promise.set_value(td_->contacts_manager_->get_users_object(-1, user_ids));
}

void MessagesManager::get_dialog_info_full(DialogId dialog_id, Promise<Unit> &&promise, const char *source) {
  switch (dialog_id.get_type()) {
    case DialogType::User:
      send_closure_later(td_->contacts_manager_actor_, &ContactsManager::load_user_full, dialog_id.get_user_id(), false,
                         std::move(promise), source);
      return;
    case DialogType::Chat:
      send_closure_later(td_->contacts_manager_actor_, &ContactsManager::load_chat_full, dialog_id.get_chat_id(), false,
                         std::move(promise), source);
      return;
    case DialogType::Channel:
      send_closure_later(td_->contacts_manager_actor_, &ContactsManager::load_channel_full, dialog_id.get_channel_id(),
                         false, std::move(promise), source);
      return;
    case DialogType::SecretChat:
      return promise.set_value(Unit());
    case DialogType::None:
    default:
      UNREACHABLE();
      return promise.set_error(Status::Error(500, "Wrong chat type"));
  }
}

void MessagesManager::reload_dialog_info_full(DialogId dialog_id) {
  if (G()->close_flag()) {
    return;
  }

  switch (dialog_id.get_type()) {
    case DialogType::User:
      send_closure_later(td_->contacts_manager_actor_, &ContactsManager::reload_user_full, dialog_id.get_user_id());
      return;
    case DialogType::Chat:
      send_closure_later(td_->contacts_manager_actor_, &ContactsManager::reload_chat_full, dialog_id.get_chat_id(),
                         Promise<Unit>());
      return;
    case DialogType::Channel:
      send_closure_later(td_->contacts_manager_actor_, &ContactsManager::reload_channel_full,
                         dialog_id.get_channel_id(), Promise<Unit>(), "reload_dialog_info_full");
      return;
    case DialogType::SecretChat:
      return;
    case DialogType::None:
    default:
      UNREACHABLE();
      return;
  }
}

MessageId MessagesManager::get_dialog_pinned_message(DialogId dialog_id, Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(dialog_id, "get_dialog_pinned_message");
  if (d == nullptr) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return MessageId();
  }

  LOG(INFO) << "Get pinned message in " << dialog_id << " with "
            << (d->is_last_pinned_message_id_inited ? "inited" : "unknown") << " pinned " << d->last_pinned_message_id;

  if (!d->is_last_pinned_message_id_inited) {
    // must call get_dialog_info_full as expected in fix_new_dialog
    get_dialog_info_full(dialog_id, std::move(promise), "get_dialog_pinned_message 1");
    return MessageId();
  }

  get_dialog_info_full(dialog_id, Auto(), "get_dialog_pinned_message 2");

  if (d->last_pinned_message_id.is_valid()) {
    tl_object_ptr<telegram_api::InputMessage> input_message;
    if (dialog_id.get_type() == DialogType::Channel) {
      input_message = make_tl_object<telegram_api::inputMessagePinned>();
    }
    get_message_force_from_server(d, d->last_pinned_message_id, std::move(promise), std::move(input_message));
  } else {
    promise.set_value(Unit());
  }

  return d->last_pinned_message_id;
}

void MessagesManager::get_callback_query_message(DialogId dialog_id, MessageId message_id, int64 callback_query_id,
                                                 Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(dialog_id, "get_callback_query_message");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }
  if (!message_id.is_valid() || !message_id.is_server()) {
    return promise.set_error(Status::Error(400, "Invalid message identifier specified"));
  }

  LOG(INFO) << "Get callback query " << message_id << " in " << dialog_id << " for query " << callback_query_id;

  auto input_message = make_tl_object<telegram_api::inputMessageCallbackQuery>(message_id.get_server_message_id().get(),
                                                                               callback_query_id);
  get_message_force_from_server(d, message_id, std::move(promise), std::move(input_message));
}

bool MessagesManager::get_messages(DialogId dialog_id, const vector<MessageId> &message_ids, Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(dialog_id, "get_messages");
  if (d == nullptr) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return false;
  }

  bool is_secret = dialog_id.get_type() == DialogType::SecretChat;
  vector<FullMessageId> missed_message_ids;
  for (auto message_id : message_ids) {
    if (!message_id.is_valid() && !message_id.is_valid_scheduled()) {
      promise.set_error(Status::Error(400, "Invalid message identifier"));
      return false;
    }

    auto *m = get_message_force(d, message_id, "get_messages");
    if (m == nullptr && message_id.is_any_server() && !is_secret) {
      missed_message_ids.emplace_back(dialog_id, message_id);
      continue;
    }
  }

  if (!missed_message_ids.empty()) {
    get_messages_from_server(std::move(missed_message_ids), std::move(promise), "get_messages");
    return false;
  }

  promise.set_value(Unit());
  return true;
}

void MessagesManager::get_message_from_server(FullMessageId full_message_id, Promise<Unit> &&promise,
                                              const char *source,
                                              tl_object_ptr<telegram_api::InputMessage> input_message) {
  get_messages_from_server({full_message_id}, std::move(promise), source, std::move(input_message));
}

void MessagesManager::get_messages_from_server(vector<FullMessageId> &&message_ids, Promise<Unit> &&promise,
                                               const char *source,
                                               tl_object_ptr<telegram_api::InputMessage> input_message) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  if (message_ids.empty()) {
    LOG(ERROR) << "Empty message_ids from " << source;
    return promise.set_error(Status::Error(500, "There are no messages specified to fetch"));
  }

  if (input_message != nullptr) {
    CHECK(message_ids.size() == 1);
  }

  vector<tl_object_ptr<telegram_api::InputMessage>> ordinary_message_ids;
  std::unordered_map<ChannelId, vector<tl_object_ptr<telegram_api::InputMessage>>, ChannelIdHash> channel_message_ids;
  std::unordered_map<DialogId, vector<int32>, DialogIdHash> scheduled_message_ids;
  for (auto &full_message_id : message_ids) {
    auto dialog_id = full_message_id.get_dialog_id();
    auto message_id = full_message_id.get_message_id();
    if (!message_id.is_valid() || !message_id.is_server()) {
      if (message_id.is_valid_scheduled() && message_id.is_scheduled_server()) {
        scheduled_message_ids[dialog_id].push_back(message_id.get_scheduled_server_message_id().get());
      }
      continue;
    }

    if (input_message == nullptr) {
      input_message = make_tl_object<telegram_api::inputMessageID>(message_id.get_server_message_id().get());
    }

    switch (dialog_id.get_type()) {
      case DialogType::User:
      case DialogType::Chat:
        ordinary_message_ids.push_back(std::move(input_message));
        break;
      case DialogType::Channel:
        channel_message_ids[dialog_id.get_channel_id()].push_back(std::move(input_message));
        break;
      case DialogType::SecretChat:
        LOG(ERROR) << "Can't get " << full_message_id << " from server from " << source;
        break;
      case DialogType::None:
      default:
        UNREACHABLE();
        break;
    }
  }

  MultiPromiseActorSafe mpas{"GetMessagesOnServerMultiPromiseActor"};
  mpas.add_promise(std::move(promise));
  auto lock = mpas.get_promise();

  if (!ordinary_message_ids.empty()) {
    td_->create_handler<GetMessagesQuery>(mpas.get_promise())->send(std::move(ordinary_message_ids));
  }

  for (auto &it : scheduled_message_ids) {
    auto dialog_id = it.first;
    have_dialog_force(dialog_id, "get_messages_from_server");
    auto input_peer = get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      LOG(ERROR) << "Can't find info about " << dialog_id << " to get a message from it from " << source;
      mpas.get_promise().set_error(Status::Error(400, "Can't access the chat"));
      continue;
    }
    td_->create_handler<GetScheduledMessagesQuery>(mpas.get_promise())
        ->send(dialog_id, std::move(input_peer), std::move(it.second));
  }

  for (auto &it : channel_message_ids) {
    td_->contacts_manager_->have_channel_force(it.first);
    auto input_channel = td_->contacts_manager_->get_input_channel(it.first);
    if (input_channel == nullptr) {
      LOG(ERROR) << "Can't find info about " << it.first << " to get a message from it from " << source;
      mpas.get_promise().set_error(Status::Error(400, "Can't access the chat"));
      continue;
    }
    const auto *d = get_dialog_force(DialogId(it.first));
    td_->create_handler<GetChannelMessagesQuery>(mpas.get_promise())
        ->send(it.first, std::move(input_channel), std::move(it.second),
               d == nullptr ? MessageId() : d->last_new_message_id);
  }
  lock.set_value(Unit());
}

bool MessagesManager::is_message_edited_recently(FullMessageId full_message_id, int32 seconds) {
  if (seconds < 0) {
    return false;
  }
  if (!full_message_id.get_message_id().is_valid()) {
    return false;
  }

  auto m = get_message_force(full_message_id, "is_message_edited_recently");
  if (m == nullptr) {
    return true;
  }

  return m->edit_date >= G()->unix_time() - seconds;
}

Status MessagesManager::can_get_media_timestamp_link(DialogId dialog_id, const Message *m) {
  if (m == nullptr) {
    return Status::Error(400, "Message not found");
  }

  if (dialog_id.get_type() != DialogType::Channel) {
    auto forward_info = m->forward_info.get();
    if (!can_message_content_have_media_timestamp(m->content.get()) || forward_info == nullptr ||
        forward_info->is_imported || is_forward_info_sender_hidden(forward_info) ||
        !forward_info->message_id.is_valid() || !m->forward_info->message_id.is_server() ||
        !forward_info->sender_dialog_id.is_valid() ||
        forward_info->sender_dialog_id.get_type() != DialogType::Channel) {
      return Status::Error(400, "Message links are available only for messages in supergroups and channel chats");
    }
    return Status::OK();
  }

  if (m->message_id.is_yet_unsent()) {
    return Status::Error(400, "Message is not sent yet");
  }
  if (m->message_id.is_scheduled()) {
    return Status::Error(400, "Message is scheduled");
  }
  if (!m->message_id.is_server()) {
    return Status::Error(400, "Message is local");
  }
  return Status::OK();
}

Result<std::pair<string, bool>> MessagesManager::get_message_link(FullMessageId full_message_id, int32 media_timestamp,
                                                                  bool for_group, bool for_comment) {
  auto dialog_id = full_message_id.get_dialog_id();
  auto d = get_dialog_force(dialog_id, "get_message_link");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return Status::Error(400, "Can't access the chat");
  }

  auto *m = get_message_force(d, full_message_id.get_message_id(), "get_message_link");
  TRY_STATUS(can_get_media_timestamp_link(dialog_id, m));

  if (media_timestamp <= 0 || !can_message_content_have_media_timestamp(m->content.get())) {
    media_timestamp = 0;
  }
  if (media_timestamp != 0) {
    for_group = false;
    auto duration = get_message_content_media_duration(m->content.get(), td_);
    if (duration != 0 && media_timestamp > duration) {
      media_timestamp = 0;
    }
  }

  auto message_id = m->message_id;
  if (dialog_id.get_type() != DialogType::Channel) {
    CHECK(m->forward_info != nullptr);
    CHECK(m->forward_info->sender_dialog_id.get_type() == DialogType::Channel);

    dialog_id = m->forward_info->sender_dialog_id;
    message_id = m->forward_info->message_id;
    for_group = false;
    for_comment = false;
    auto channel_message = get_message({dialog_id, message_id});
    if (channel_message != nullptr && channel_message->media_album_id == 0) {
      for_group = true;  // default is true
    }
  } else {
    if (m->media_album_id == 0) {
      for_group = true;  // default is true
    }
  }

  if (!m->top_thread_message_id.is_valid() || !m->top_thread_message_id.is_server()) {
    for_comment = false;
  }
  if (d->deleted_message_ids.count(m->top_thread_message_id) != 0) {
    for_comment = false;
  }
  if (for_comment && is_broadcast_channel(dialog_id)) {
    for_comment = false;
  }

  if (!td_->auth_manager_->is_bot()) {
    td_->create_handler<ExportChannelMessageLinkQuery>(Promise<Unit>())
        ->send(dialog_id.get_channel_id(), message_id, for_group, true);
  }

  SliceBuilder sb;
  sb << G()->shared_config().get_option_string("t_me_url", "https://t.me/");

  if (for_comment) {
    auto *top_m = get_message_force(d, m->top_thread_message_id, "get_public_message_link");
    if (is_discussion_message(dialog_id, top_m) && is_active_message_reply_info(dialog_id, top_m->reply_info)) {
      auto linked_dialog_id = top_m->forward_info->from_dialog_id;
      auto linked_message_id = top_m->forward_info->from_message_id;
      auto linked_d = get_dialog(linked_dialog_id);
      CHECK(linked_d != nullptr);
      CHECK(linked_dialog_id.get_type() == DialogType::Channel);
      auto *linked_m = get_message_force(linked_d, linked_message_id, "get_public_message_link");
      auto channel_username = td_->contacts_manager_->get_channel_username(linked_dialog_id.get_channel_id());
      if (linked_m != nullptr && is_active_message_reply_info(linked_dialog_id, linked_m->reply_info) &&
          linked_message_id.is_server() && have_input_peer(linked_dialog_id, AccessRights::Read) &&
          !channel_username.empty()) {
        sb << channel_username << '/' << linked_message_id.get_server_message_id().get()
           << "?comment=" << message_id.get_server_message_id().get();
        if (!for_group) {
          sb << "&single";
        }
        if (media_timestamp > 0) {
          sb << "&t=" << media_timestamp;
        }
        return std::make_pair(sb.as_cslice().str(), true);
      }
    }
  }

  auto dialog_username = td_->contacts_manager_->get_channel_username(dialog_id.get_channel_id());
  bool is_public = !dialog_username.empty();
  if (m->content->get_type() == MessageContentType::VideoNote && is_broadcast_channel(dialog_id) && is_public) {
    return std::make_pair(
        PSTRING() << "https://telesco.pe/" << dialog_username << '/' << message_id.get_server_message_id().get(), true);
  }

  if (is_public) {
    sb << dialog_username;
  } else {
    sb << "c/" << dialog_id.get_channel_id().get();
  }
  sb << '/' << message_id.get_server_message_id().get();

  char separator = '?';
  if (for_comment) {
    sb << separator << "thread=" << m->top_thread_message_id.get_server_message_id().get();
    separator = '&';
  }
  if (!for_group) {
    sb << separator << "single";
    separator = '&';
  }
  if (media_timestamp > 0) {
    sb << separator << "t=" << media_timestamp;
    separator = '&';
  }

  return std::make_pair(sb.as_cslice().str(), is_public);
}

string MessagesManager::get_message_embedding_code(FullMessageId full_message_id, bool for_group,
                                                   Promise<Unit> &&promise) {
  auto dialog_id = full_message_id.get_dialog_id();
  auto d = get_dialog_force(dialog_id, "get_message_embedding_code");
  if (d == nullptr) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return {};
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    promise.set_error(Status::Error(400, "Can't access the chat"));
    return {};
  }
  if (dialog_id.get_type() != DialogType::Channel ||
      td_->contacts_manager_->get_channel_username(dialog_id.get_channel_id()).empty()) {
    promise.set_error(Status::Error(
        400, "Message embedding code is available only for messages in public supergroups and channel chats"));
    return {};
  }

  auto *m = get_message_force(d, full_message_id.get_message_id(), "get_message_embedding_code");
  if (m == nullptr) {
    promise.set_error(Status::Error(400, "Message not found"));
    return {};
  }
  if (m->message_id.is_yet_unsent()) {
    promise.set_error(Status::Error(400, "Message is not sent yet"));
    return {};
  }
  if (m->message_id.is_scheduled()) {
    promise.set_error(Status::Error(400, "Message is scheduled"));
    return {};
  }
  if (!m->message_id.is_server()) {
    promise.set_error(Status::Error(400, "Message is local"));
    return {};
  }

  if (m->media_album_id == 0) {
    for_group = true;  // default is true
  }

  auto &links = message_embedding_codes_[for_group][dialog_id].embedding_codes_;
  auto it = links.find(m->message_id);
  if (it == links.end()) {
    td_->create_handler<ExportChannelMessageLinkQuery>(std::move(promise))
        ->send(dialog_id.get_channel_id(), m->message_id, for_group, false);
    return {};
  }

  promise.set_value(Unit());
  return it->second;
}

void MessagesManager::on_get_public_message_link(FullMessageId full_message_id, bool for_group, string url,
                                                 string html) {
  LOG_IF(ERROR, url.empty() && html.empty()) << "Receive empty public link for " << full_message_id;
  auto dialog_id = full_message_id.get_dialog_id();
  auto message_id = full_message_id.get_message_id();
  message_embedding_codes_[for_group][dialog_id].embedding_codes_[message_id] = std::move(html);
}

void MessagesManager::get_message_link_info(Slice url, Promise<MessageLinkInfo> &&promise) {
  auto r_message_link_info = LinkManager::get_message_link_info(url);
  if (r_message_link_info.is_error()) {
    return promise.set_error(Status::Error(400, r_message_link_info.error().message()));
  }

  auto info = r_message_link_info.move_as_ok();
  CHECK(info.username.empty() == info.channel_id.is_valid());

  bool have_dialog = info.username.empty() ? td_->contacts_manager_->have_channel_force(info.channel_id)
                                           : resolve_dialog_username(info.username).is_valid();
  if (!have_dialog) {
    auto query_promise = PromiseCreator::lambda(
        [actor_id = actor_id(this), info, promise = std::move(promise)](Result<Unit> &&result) mutable {
          if (result.is_error()) {
            return promise.set_value(std::move(info));
          }
          send_closure(actor_id, &MessagesManager::on_get_message_link_dialog, std::move(info), std::move(promise));
        });
    if (info.username.empty()) {
      td_->contacts_manager_->reload_channel(info.channel_id, std::move(query_promise));
    } else {
      td_->create_handler<ResolveUsernameQuery>(std::move(query_promise))->send(info.username);
    }
    return;
  }

  return on_get_message_link_dialog(std::move(info), std::move(promise));
}

void MessagesManager::on_get_message_link_dialog(MessageLinkInfo &&info, Promise<MessageLinkInfo> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  DialogId dialog_id;
  if (info.username.empty()) {
    if (!td_->contacts_manager_->have_channel(info.channel_id)) {
      return promise.set_error(Status::Error(500, "Chat info not found"));
    }

    dialog_id = DialogId(info.channel_id);
    force_create_dialog(dialog_id, "on_get_message_link_dialog");
  } else {
    dialog_id = resolve_dialog_username(info.username);
    if (dialog_id.is_valid()) {
      force_create_dialog(dialog_id, "on_get_message_link_dialog", true);
    }
  }
  Dialog *d = get_dialog_force(dialog_id, "on_get_message_link_dialog");
  if (d == nullptr) {
    return promise.set_error(Status::Error(500, "Chat not found"));
  }

  get_message_force_from_server(d, info.message_id,
                                PromiseCreator::lambda([actor_id = actor_id(this), info = std::move(info), dialog_id,
                                                        promise = std::move(promise)](Result<Unit> &&result) mutable {
                                  if (result.is_error()) {
                                    return promise.set_value(std::move(info));
                                  }
                                  send_closure(actor_id, &MessagesManager::on_get_message_link_message, std::move(info),
                                               dialog_id, std::move(promise));
                                }));
}

void MessagesManager::on_get_message_link_message(MessageLinkInfo &&info, DialogId dialog_id,
                                                  Promise<MessageLinkInfo> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  Message *m = get_message_force({dialog_id, info.message_id}, "on_get_message_link_message");
  if (info.comment_message_id == MessageId() || m == nullptr || !is_broadcast_channel(dialog_id) ||
      !m->reply_info.is_comment || !is_active_message_reply_info(dialog_id, m->reply_info)) {
    return promise.set_value(std::move(info));
  }

  if (td_->contacts_manager_->have_channel_force(m->reply_info.channel_id)) {
    force_create_dialog(DialogId(m->reply_info.channel_id), "on_get_message_link_message");
    on_get_message_link_discussion_message(std::move(info), DialogId(m->reply_info.channel_id), std::move(promise));
    return;
  }

  auto query_promise = PromiseCreator::lambda([actor_id = actor_id(this), info = std::move(info),
                                               promise = std::move(promise)](Result<MessageThreadInfo> result) mutable {
    if (result.is_error() || result.ok().message_ids.empty()) {
      return promise.set_value(std::move(info));
    }
    send_closure(actor_id, &MessagesManager::on_get_message_link_discussion_message, std::move(info),
                 result.ok().dialog_id, std::move(promise));
  });

  td_->create_handler<GetDiscussionMessageQuery>(std::move(query_promise))
      ->send(dialog_id, info.message_id, DialogId(m->reply_info.channel_id), MessageId());
}

void MessagesManager::on_get_message_link_discussion_message(MessageLinkInfo &&info, DialogId comment_dialog_id,
                                                             Promise<MessageLinkInfo> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  CHECK(comment_dialog_id.is_valid());
  info.comment_dialog_id = comment_dialog_id;

  Dialog *d = get_dialog_force(comment_dialog_id, "on_get_message_link_discussion_message");
  if (d == nullptr) {
    return promise.set_error(Status::Error(500, "Chat not found"));
  }

  get_message_force_from_server(
      d, info.comment_message_id,
      PromiseCreator::lambda([info = std::move(info), promise = std::move(promise)](Result<Unit> &&result) mutable {
        return promise.set_value(std::move(info));
      }));
}

td_api::object_ptr<td_api::messageLinkInfo> MessagesManager::get_message_link_info_object(
    const MessageLinkInfo &info) const {
  CHECK(info.username.empty() == info.channel_id.is_valid());

  bool is_public = !info.username.empty();
  DialogId dialog_id = info.comment_dialog_id.is_valid()
                           ? info.comment_dialog_id
                           : (is_public ? resolve_dialog_username(info.username) : DialogId(info.channel_id));
  MessageId message_id = info.comment_dialog_id.is_valid() ? info.comment_message_id : info.message_id;
  td_api::object_ptr<td_api::message> message;
  int32 media_timestamp = 0;
  bool for_album = false;
  bool for_comment = false;

  const Dialog *d = get_dialog(dialog_id);
  if (d == nullptr) {
    dialog_id = DialogId();
  } else {
    const Message *m = get_message(d, message_id);
    if (m != nullptr) {
      message = get_message_object(dialog_id, m, "get_message_link_info_object");
      for_album = !info.is_single && m->media_album_id != 0;
      for_comment = (info.comment_dialog_id.is_valid() || info.for_comment) && m->top_thread_message_id.is_valid();
      if (can_message_content_have_media_timestamp(m->content.get())) {
        auto duration = get_message_content_media_duration(m->content.get(), td_);
        if (duration == 0 || info.media_timestamp <= duration) {
          media_timestamp = info.media_timestamp;
        }
      }
    }
  }

  return td_api::make_object<td_api::messageLinkInfo>(is_public, dialog_id.get(), std::move(message), media_timestamp,
                                                      for_album, for_comment);
}

InputDialogId MessagesManager::get_input_dialog_id(DialogId dialog_id) const {
  auto input_peer = get_input_peer(dialog_id, AccessRights::Read);
  if (input_peer == nullptr || input_peer->get_id() == telegram_api::inputPeerSelf::ID) {
    return InputDialogId(dialog_id);
  } else {
    return InputDialogId(input_peer);
  }
}

void MessagesManager::sort_dialog_filter_input_dialog_ids(DialogFilter *dialog_filter, const char *source) const {
  auto sort_input_dialog_ids = [contacts_manager =
                                    td_->contacts_manager_.get()](vector<InputDialogId> &input_dialog_ids) {
    std::sort(input_dialog_ids.begin(), input_dialog_ids.end(),
              [contacts_manager](InputDialogId lhs, InputDialogId rhs) {
                auto get_order = [contacts_manager](InputDialogId input_dialog_id) {
                  auto dialog_id = input_dialog_id.get_dialog_id();
                  if (dialog_id.get_type() != DialogType::SecretChat) {
                    return dialog_id.get() * 10;
                  }
                  auto user_id = contacts_manager->get_secret_chat_user_id(dialog_id.get_secret_chat_id());
                  return DialogId(user_id).get() * 10 + 1;
                };
                return get_order(lhs) < get_order(rhs);
              });
  };

  if (!dialog_filter->include_contacts && !dialog_filter->include_non_contacts && !dialog_filter->include_bots &&
      !dialog_filter->include_groups && !dialog_filter->include_channels) {
    dialog_filter->excluded_dialog_ids.clear();
  }

  sort_input_dialog_ids(dialog_filter->excluded_dialog_ids);
  sort_input_dialog_ids(dialog_filter->included_dialog_ids);

  std::unordered_set<DialogId, DialogIdHash> all_dialog_ids;
  for (auto input_dialog_ids :
       {&dialog_filter->pinned_dialog_ids, &dialog_filter->excluded_dialog_ids, &dialog_filter->included_dialog_ids}) {
    for (auto input_dialog_id : *input_dialog_ids) {
      LOG_CHECK(all_dialog_ids.insert(input_dialog_id.get_dialog_id()).second)
          << source << ' ' << input_dialog_id.get_dialog_id() << ' ' << dialog_filter;
    }
  }
}

Result<unique_ptr<DialogFilter>> MessagesManager::create_dialog_filter(DialogFilterId dialog_filter_id,
                                                                       td_api::object_ptr<td_api::chatFilter> filter) {
  CHECK(filter != nullptr);
  for (auto chat_ids : {&filter->pinned_chat_ids_, &filter->excluded_chat_ids_, &filter->included_chat_ids_}) {
    for (auto chat_id : *chat_ids) {
      DialogId dialog_id(chat_id);
      if (!dialog_id.is_valid()) {
        return Status::Error(400, "Invalid chat identifier specified");
      }
      const Dialog *d = get_dialog_force(dialog_id, "create_dialog_filter");
      if (d == nullptr) {
        return Status::Error(400, "Chat not found");
      }
      if (!have_input_peer(dialog_id, AccessRights::Read)) {
        return Status::Error(400, "Can't access the chat");
      }
      if (d->order == DEFAULT_ORDER) {
        return Status::Error(400, "Chat is not in the chat list");
      }
    }
  }

  auto dialog_filter = make_unique<DialogFilter>();
  dialog_filter->dialog_filter_id = dialog_filter_id;

  std::unordered_set<int64> added_dialog_ids;
  auto add_chats = [this, &added_dialog_ids](vector<InputDialogId> &input_dialog_ids, const vector<int64> &chat_ids) {
    for (auto &chat_id : chat_ids) {
      if (!added_dialog_ids.insert(chat_id).second) {
        // do not allow duplicate chat_ids
        continue;
      }

      input_dialog_ids.push_back(get_input_dialog_id(DialogId(chat_id)));
    }
  };
  add_chats(dialog_filter->pinned_dialog_ids, filter->pinned_chat_ids_);
  add_chats(dialog_filter->included_dialog_ids, filter->included_chat_ids_);
  add_chats(dialog_filter->excluded_dialog_ids, filter->excluded_chat_ids_);

  dialog_filter->title = clean_name(std::move(filter->title_), MAX_DIALOG_FILTER_TITLE_LENGTH);
  if (dialog_filter->title.empty()) {
    return Status::Error(400, "Title must be non-empty");
  }
  dialog_filter->emoji = DialogFilter::get_emoji_by_icon_name(filter->icon_name_);
  if (dialog_filter->emoji.empty() && !filter->icon_name_.empty()) {
    return Status::Error(400, "Invalid icon name specified");
  }
  dialog_filter->exclude_muted = filter->exclude_muted_;
  dialog_filter->exclude_read = filter->exclude_read_;
  dialog_filter->exclude_archived = filter->exclude_archived_;
  dialog_filter->include_contacts = filter->include_contacts_;
  dialog_filter->include_non_contacts = filter->include_non_contacts_;
  dialog_filter->include_bots = filter->include_bots_;
  dialog_filter->include_groups = filter->include_groups_;
  dialog_filter->include_channels = filter->include_channels_;

  TRY_STATUS(dialog_filter->check_limits());
  sort_dialog_filter_input_dialog_ids(dialog_filter.get(), "create_dialog_filter");

  return std::move(dialog_filter);
}

void MessagesManager::create_dialog_filter(td_api::object_ptr<td_api::chatFilter> filter,
                                           Promise<td_api::object_ptr<td_api::chatFilterInfo>> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());
  if (dialog_filters_.size() >= MAX_DIALOG_FILTERS) {
    return promise.set_error(Status::Error(400, "The maximum number of chat folders exceeded"));
  }
  if (!is_update_chat_filters_sent_) {
    return promise.set_error(Status::Error(400, "Chat folders are not synchronized yet"));
  }

  DialogFilterId dialog_filter_id;
  do {
    auto min_id = static_cast<int>(DialogFilterId::min().get());
    auto max_id = static_cast<int>(DialogFilterId::max().get());
    dialog_filter_id = DialogFilterId(static_cast<int32>(Random::fast(min_id, max_id)));
  } while (get_dialog_filter(dialog_filter_id) != nullptr || get_server_dialog_filter(dialog_filter_id) != nullptr);

  auto r_dialog_filter = create_dialog_filter(dialog_filter_id, std::move(filter));
  if (r_dialog_filter.is_error()) {
    return promise.set_error(r_dialog_filter.move_as_error());
  }
  auto dialog_filter = r_dialog_filter.move_as_ok();
  CHECK(dialog_filter != nullptr);
  auto chat_filter_info = dialog_filter->get_chat_filter_info_object();

  bool at_beginning = false;
  for (auto &recommended_dialog_filter : recommended_dialog_filters_) {
    if (DialogFilter::are_similar(*recommended_dialog_filter.dialog_filter, *dialog_filter)) {
      at_beginning = true;
    }
  }

  add_dialog_filter(std::move(dialog_filter), at_beginning, "create_dialog_filter");
  save_dialog_filters();
  send_update_chat_filters();

  synchronize_dialog_filters();
  promise.set_value(std::move(chat_filter_info));
}

void MessagesManager::edit_dialog_filter(DialogFilterId dialog_filter_id, td_api::object_ptr<td_api::chatFilter> filter,
                                         Promise<td_api::object_ptr<td_api::chatFilterInfo>> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());
  auto old_dialog_filter = get_dialog_filter(dialog_filter_id);
  if (old_dialog_filter == nullptr) {
    return promise.set_error(Status::Error(400, "Chat filter not found"));
  }
  CHECK(is_update_chat_filters_sent_);

  auto r_dialog_filter = create_dialog_filter(dialog_filter_id, std::move(filter));
  if (r_dialog_filter.is_error()) {
    return promise.set_error(r_dialog_filter.move_as_error());
  }
  auto new_dialog_filter = r_dialog_filter.move_as_ok();
  CHECK(new_dialog_filter != nullptr);
  auto chat_filter_info = new_dialog_filter->get_chat_filter_info_object();

  if (*new_dialog_filter == *old_dialog_filter) {
    return promise.set_value(std::move(chat_filter_info));
  }

  edit_dialog_filter(std::move(new_dialog_filter), "edit_dialog_filter");
  save_dialog_filters();
  send_update_chat_filters();

  synchronize_dialog_filters();
  promise.set_value(std::move(chat_filter_info));
}

void MessagesManager::update_dialog_filter_on_server(unique_ptr<DialogFilter> &&dialog_filter) {
  CHECK(!td_->auth_manager_->is_bot());
  CHECK(dialog_filter != nullptr);
  are_dialog_filters_being_synchronized_ = true;
  dialog_filter->remove_secret_chat_dialog_ids();
  auto dialog_filter_id = dialog_filter->dialog_filter_id;
  auto input_dialog_filter = dialog_filter->get_input_dialog_filter();

  auto promise = PromiseCreator::lambda(
      [actor_id = actor_id(this), dialog_filter = std::move(dialog_filter)](Result<Unit> result) mutable {
        send_closure(actor_id, &MessagesManager::on_update_dialog_filter, std::move(dialog_filter),
                     result.is_error() ? result.move_as_error() : Status::OK());
      });
  td_->create_handler<UpdateDialogFilterQuery>(std::move(promise))
      ->send(dialog_filter_id, std::move(input_dialog_filter));
}

void MessagesManager::on_update_dialog_filter(unique_ptr<DialogFilter> dialog_filter, Status result) {
  CHECK(!td_->auth_manager_->is_bot());
  if (result.is_error()) {
    // TODO rollback dialog_filters_ changes if error isn't 429
  } else {
    bool is_edited = false;
    for (auto &filter : server_dialog_filters_) {
      if (filter->dialog_filter_id == dialog_filter->dialog_filter_id) {
        if (*filter != *dialog_filter) {
          filter = std::move(dialog_filter);
        }
        is_edited = true;
        break;
      }
    }

    if (!is_edited) {
      server_dialog_filters_.push_back(std::move(dialog_filter));
    }
    save_dialog_filters();
  }

  are_dialog_filters_being_synchronized_ = false;
  synchronize_dialog_filters();
}

void MessagesManager::delete_dialog_filter(DialogFilterId dialog_filter_id, Promise<Unit> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());
  auto dialog_filter = get_dialog_filter(dialog_filter_id);
  if (dialog_filter == nullptr) {
    return promise.set_value(Unit());
  }

  delete_dialog_filter(dialog_filter_id, "delete_dialog_filter");
  save_dialog_filters();
  send_update_chat_filters();

  synchronize_dialog_filters();
  promise.set_value(Unit());
}

void MessagesManager::delete_dialog_filter_on_server(DialogFilterId dialog_filter_id) {
  CHECK(!td_->auth_manager_->is_bot());
  are_dialog_filters_being_synchronized_ = true;
  auto promise = PromiseCreator::lambda([actor_id = actor_id(this), dialog_filter_id](Result<Unit> result) {
    send_closure(actor_id, &MessagesManager::on_delete_dialog_filter, dialog_filter_id,
                 result.is_error() ? result.move_as_error() : Status::OK());
  });
  td_->create_handler<UpdateDialogFilterQuery>(std::move(promise))->send(dialog_filter_id, nullptr);
}

void MessagesManager::on_delete_dialog_filter(DialogFilterId dialog_filter_id, Status result) {
  CHECK(!td_->auth_manager_->is_bot());
  if (result.is_error()) {
    // TODO rollback dialog_filters_ changes if error isn't 429
  } else {
    for (auto it = server_dialog_filters_.begin(); it != server_dialog_filters_.end(); ++it) {
      if ((*it)->dialog_filter_id == dialog_filter_id) {
        server_dialog_filters_.erase(it);
        save_dialog_filters();
        break;
      }
    }
  }

  are_dialog_filters_being_synchronized_ = false;
  synchronize_dialog_filters();
}

void MessagesManager::reorder_dialog_filters(vector<DialogFilterId> dialog_filter_ids, Promise<Unit> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());

  for (auto dialog_filter_id : dialog_filter_ids) {
    auto dialog_filter = get_dialog_filter(dialog_filter_id);
    if (dialog_filter == nullptr) {
      return promise.set_error(Status::Error(400, "Chat filter not found"));
    }
  }
  std::unordered_set<DialogFilterId, DialogFilterIdHash> new_dialog_filter_ids_set(dialog_filter_ids.begin(),
                                                                                   dialog_filter_ids.end());
  if (new_dialog_filter_ids_set.size() != dialog_filter_ids.size()) {
    return promise.set_error(Status::Error(400, "Duplicate chat filters in the new list"));
  }

  if (set_dialog_filters_order(dialog_filters_, dialog_filter_ids)) {
    save_dialog_filters();
    send_update_chat_filters();

    synchronize_dialog_filters();
  }
  promise.set_value(Unit());
}

void MessagesManager::reorder_dialog_filters_on_server(vector<DialogFilterId> dialog_filter_ids) {
  CHECK(!td_->auth_manager_->is_bot());
  are_dialog_filters_being_synchronized_ = true;
  auto promise = PromiseCreator::lambda([actor_id = actor_id(this), dialog_filter_ids](Result<Unit> result) mutable {
    send_closure(actor_id, &MessagesManager::on_reorder_dialog_filters, std::move(dialog_filter_ids),
                 result.is_error() ? result.move_as_error() : Status::OK());
  });
  td_->create_handler<UpdateDialogFiltersOrderQuery>(std::move(promise))->send(dialog_filter_ids);
}

void MessagesManager::on_reorder_dialog_filters(vector<DialogFilterId> dialog_filter_ids, Status result) {
  CHECK(!td_->auth_manager_->is_bot());
  if (result.is_error()) {
    // TODO rollback dialog_filters_ changes if error isn't 429
  } else {
    if (set_dialog_filters_order(server_dialog_filters_, std::move(dialog_filter_ids))) {
      save_dialog_filters();
    }
  }

  are_dialog_filters_being_synchronized_ = false;
  synchronize_dialog_filters();
}

bool MessagesManager::set_dialog_filters_order(vector<unique_ptr<DialogFilter>> &dialog_filters,
                                               vector<DialogFilterId> dialog_filter_ids) {
  auto old_dialog_filter_ids = get_dialog_filter_ids(dialog_filters);
  if (old_dialog_filter_ids == dialog_filter_ids) {
    return false;
  }
  LOG(INFO) << "Reorder chat filters from " << old_dialog_filter_ids << " to " << dialog_filter_ids;

  if (dialog_filter_ids.size() != old_dialog_filter_ids.size()) {
    for (auto dialog_filter_id : old_dialog_filter_ids) {
      if (!td::contains(dialog_filter_ids, dialog_filter_id)) {
        dialog_filter_ids.push_back(dialog_filter_id);
      }
    }
    CHECK(dialog_filter_ids.size() == old_dialog_filter_ids.size());
  }
  if (old_dialog_filter_ids == dialog_filter_ids) {
    return false;
  }

  CHECK(dialog_filter_ids.size() == dialog_filters.size());
  for (size_t i = 0; i < dialog_filters.size(); i++) {
    for (size_t j = i; j < dialog_filters.size(); j++) {
      if (dialog_filters[j]->dialog_filter_id == dialog_filter_ids[i]) {
        if (i != j) {
          std::swap(dialog_filters[i], dialog_filters[j]);
        }
        break;
      }
    }
    CHECK(dialog_filters[i]->dialog_filter_id == dialog_filter_ids[i]);
  }
  return true;
}

void MessagesManager::add_dialog_filter(unique_ptr<DialogFilter> dialog_filter, bool at_beginning, const char *source) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  CHECK(dialog_filter != nullptr);
  auto dialog_filter_id = dialog_filter->dialog_filter_id;
  LOG(INFO) << "Add " << dialog_filter_id << " from " << source;
  CHECK(get_dialog_filter(dialog_filter_id) == nullptr);
  if (at_beginning) {
    dialog_filters_.insert(dialog_filters_.begin(), std::move(dialog_filter));
  } else {
    dialog_filters_.push_back(std::move(dialog_filter));
  }

  auto dialog_list_id = DialogListId(dialog_filter_id);
  CHECK(dialog_lists_.find(dialog_list_id) == dialog_lists_.end());

  auto &list = add_dialog_list(dialog_list_id);
  auto folder_ids = get_dialog_list_folder_ids(list);
  CHECK(!folder_ids.empty());

  for (auto folder_id : folder_ids) {
    auto *folder = get_dialog_folder(folder_id);
    CHECK(folder != nullptr);
    for (const auto &dialog_date : folder->ordered_dialogs_) {
      if (dialog_date.get_order() == DEFAULT_ORDER) {
        break;
      }

      auto dialog_id = dialog_date.get_dialog_id();
      Dialog *d = get_dialog(dialog_id);
      CHECK(d != nullptr);

      if (need_dialog_in_list(d, list)) {
        list.in_memory_dialog_total_count_++;

        add_dialog_to_list(d, dialog_list_id);
      }
    }
  }

  for (const auto &input_dialog_id : reversed(dialog_filters_.back()->pinned_dialog_ids)) {
    auto dialog_id = input_dialog_id.get_dialog_id();
    auto order = get_next_pinned_dialog_order();
    list.pinned_dialogs_.emplace_back(order, dialog_id);
    list.pinned_dialog_id_orders_.emplace(dialog_id, order);
  }
  std::reverse(list.pinned_dialogs_.begin(), list.pinned_dialogs_.end());
  list.are_pinned_dialogs_inited_ = true;

  update_list_last_pinned_dialog_date(list);
  update_list_last_dialog_date(list);
}

void MessagesManager::edit_dialog_filter(unique_ptr<DialogFilter> new_dialog_filter, const char *source) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  CHECK(new_dialog_filter != nullptr);
  LOG(INFO) << "Edit " << new_dialog_filter->dialog_filter_id << " from " << source;
  for (auto &old_dialog_filter : dialog_filters_) {
    if (old_dialog_filter->dialog_filter_id == new_dialog_filter->dialog_filter_id) {
      CHECK(*old_dialog_filter != *new_dialog_filter);

      auto dialog_list_id = DialogListId(old_dialog_filter->dialog_filter_id);
      auto *old_list_ptr = get_dialog_list(dialog_list_id);
      CHECK(old_list_ptr != nullptr);
      auto &old_list = *old_list_ptr;

      disable_get_dialog_filter_ = true;  // to ensure crash if get_dialog_filter is called

      auto folder_ids = get_dialog_filter_folder_ids(old_dialog_filter.get());
      CHECK(!folder_ids.empty());
      for (auto folder_id : get_dialog_filter_folder_ids(new_dialog_filter.get())) {
        if (!td::contains(folder_ids, folder_id)) {
          folder_ids.push_back(folder_id);
        }
      }

      DialogList new_list;
      new_list.dialog_list_id = dialog_list_id;

      auto old_it = old_list.pinned_dialogs_.rbegin();
      for (const auto &input_dialog_id : reversed(new_dialog_filter->pinned_dialog_ids)) {
        auto dialog_id = input_dialog_id.get_dialog_id();
        while (old_it < old_list.pinned_dialogs_.rend()) {
          if (old_it->get_dialog_id() == dialog_id) {
            break;
          }
          ++old_it;
        }

        int64 order;
        if (old_it < old_list.pinned_dialogs_.rend()) {
          order = old_it->get_order();
          ++old_it;
        } else {
          order = get_next_pinned_dialog_order();
        }
        new_list.pinned_dialogs_.emplace_back(order, dialog_id);
        new_list.pinned_dialog_id_orders_.emplace(dialog_id, order);
      }
      std::reverse(new_list.pinned_dialogs_.begin(), new_list.pinned_dialogs_.end());
      new_list.are_pinned_dialogs_inited_ = true;

      do_update_list_last_pinned_dialog_date(new_list);
      do_update_list_last_dialog_date(new_list, get_dialog_filter_folder_ids(new_dialog_filter.get()));

      new_list.server_dialog_total_count_ = 0;
      new_list.secret_chat_total_count_ = 0;

      std::map<DialogDate, const Dialog *> updated_position_dialogs;
      for (auto folder_id : folder_ids) {
        auto *folder = get_dialog_folder(folder_id);
        CHECK(folder != nullptr);
        for (const auto &dialog_date : folder->ordered_dialogs_) {
          if (dialog_date.get_order() == DEFAULT_ORDER) {
            break;
          }

          auto dialog_id = dialog_date.get_dialog_id();
          Dialog *d = get_dialog(dialog_id);
          CHECK(d != nullptr);

          const DialogPositionInList old_position = get_dialog_position_in_list(old_list_ptr, d);
          // can't use get_dialog_position_in_list, because need_dialog_in_list calls get_dialog_filter
          DialogPositionInList new_position;
          if (need_dialog_in_filter(d, new_dialog_filter.get())) {
            new_position.private_order = get_dialog_private_order(&new_list, d);
            if (new_position.private_order != 0) {
              new_position.public_order =
                  DialogDate(new_position.private_order, dialog_id) <= new_list.list_last_dialog_date_
                      ? new_position.private_order
                      : 0;
              new_position.is_pinned = get_dialog_pinned_order(&new_list, dialog_id) != DEFAULT_ORDER;
              new_position.is_sponsored = is_dialog_sponsored(d);
            }
          }

          if (need_send_update_chat_position(old_position, new_position)) {
            updated_position_dialogs.emplace(DialogDate(new_position.public_order, dialog_id), d);
          }

          bool was_in_list = old_position.private_order != 0;
          bool is_in_list = new_position.private_order != 0;
          if (is_in_list) {
            if (!was_in_list) {
              add_dialog_to_list(d, dialog_list_id);
            }

            new_list.in_memory_dialog_total_count_++;
            if (dialog_id.get_type() == DialogType::SecretChat) {
              new_list.secret_chat_total_count_++;
            } else {
              new_list.server_dialog_total_count_++;
            }

            auto unread_count = d->server_unread_count + d->local_unread_count;
            if (unread_count != 0) {
              new_list.unread_message_total_count_ += unread_count;
              if (is_dialog_muted(d)) {
                new_list.unread_message_muted_count_ += unread_count;
              }
            }
            if (unread_count != 0 || d->is_marked_as_unread) {
              new_list.unread_dialog_total_count_++;
              if (unread_count == 0 && d->is_marked_as_unread) {
                new_list.unread_dialog_marked_count_++;
              }
              if (is_dialog_muted(d)) {
                new_list.unread_dialog_muted_count_++;
                if (unread_count == 0 && d->is_marked_as_unread) {
                  new_list.unread_dialog_muted_marked_count_++;
                }
              }
            }
          } else {
            if (was_in_list) {
              remove_dialog_from_list(d, dialog_list_id);
            }
          }
        }
      }

      if (new_list.list_last_dialog_date_ == MAX_DIALOG_DATE) {
        new_list.is_message_unread_count_inited_ = true;
        new_list.is_dialog_unread_count_inited_ = true;
        new_list.need_unread_count_recalc_ = false;
      } else {
        if (old_list.is_message_unread_count_inited_) {  // can't stop sending updates
          new_list.is_message_unread_count_inited_ = true;
        }
        if (old_list.is_dialog_unread_count_inited_) {  // can't stop sending updates
          new_list.is_dialog_unread_count_inited_ = true;
        }
        new_list.server_dialog_total_count_ = -1;
        new_list.secret_chat_total_count_ = -1;
      }

      bool need_update_unread_message_count =
          new_list.is_message_unread_count_inited_ &&
          (old_list.unread_message_total_count_ != new_list.unread_message_total_count_ ||
           old_list.unread_message_muted_count_ != new_list.unread_message_muted_count_ ||
           !old_list.is_message_unread_count_inited_);
      bool need_update_unread_chat_count =
          new_list.is_dialog_unread_count_inited_ &&
          (old_list.unread_dialog_total_count_ != new_list.unread_dialog_total_count_ ||
           old_list.unread_dialog_muted_count_ != new_list.unread_dialog_muted_count_ ||
           old_list.unread_dialog_marked_count_ != new_list.unread_dialog_marked_count_ ||
           old_list.unread_dialog_muted_marked_count_ != new_list.unread_dialog_muted_marked_count_ ||
           get_dialog_total_count(old_list) != get_dialog_total_count(new_list) ||
           !old_list.is_dialog_unread_count_inited_);
      bool need_save_unread_chat_count = new_list.is_dialog_unread_count_inited_ &&
                                         (old_list.server_dialog_total_count_ != new_list.server_dialog_total_count_ ||
                                          old_list.secret_chat_total_count_ != new_list.secret_chat_total_count_);

      auto load_list_promises = std::move(old_list.load_list_queries_);

      disable_get_dialog_filter_ = false;

      old_list = std::move(new_list);
      old_dialog_filter = std::move(new_dialog_filter);

      if (need_update_unread_message_count) {
        send_update_unread_message_count(old_list, DialogId(), true, source);
      }
      if (need_update_unread_chat_count) {
        send_update_unread_chat_count(old_list, DialogId(), true, source);
      } else if (need_save_unread_chat_count) {
        save_unread_chat_count(old_list);
      }

      for (auto it : updated_position_dialogs) {
        send_update_chat_position(dialog_list_id, it.second, source);
      }

      if (old_list.need_unread_count_recalc_) {
        // repair unread count
        get_dialogs_from_list(dialog_list_id, static_cast<int32>(old_list.pinned_dialogs_.size() + 2), Auto());
      }

      if (!load_list_promises.empty()) {
        LOG(INFO) << "Retry loading of chats in " << dialog_list_id;
        for (auto &promise : load_list_promises) {
          promise.set_value(Unit());  // try again
        }
      }
      return;
    }
  }
  UNREACHABLE();
}

void MessagesManager::delete_dialog_filter(DialogFilterId dialog_filter_id, const char *source) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  LOG(INFO) << "Delete " << dialog_filter_id << " from " << source;
  for (auto it = dialog_filters_.begin(); it != dialog_filters_.end(); ++it) {
    if ((*it)->dialog_filter_id == dialog_filter_id) {
      auto dialog_list_id = DialogListId(dialog_filter_id);
      auto *list = get_dialog_list(dialog_list_id);
      CHECK(list != nullptr);
      auto folder_ids = get_dialog_list_folder_ids(*list);
      CHECK(!folder_ids.empty());

      for (auto folder_id : folder_ids) {
        auto *folder = get_dialog_folder(folder_id);
        CHECK(folder != nullptr);
        for (const auto &dialog_date : folder->ordered_dialogs_) {
          if (dialog_date.get_order() == DEFAULT_ORDER) {
            break;
          }

          auto dialog_id = dialog_date.get_dialog_id();
          Dialog *d = get_dialog(dialog_id);
          CHECK(d != nullptr);

          const DialogPositionInList old_position = get_dialog_position_in_list(list, d);

          if (is_dialog_in_list(d, dialog_list_id)) {
            remove_dialog_from_list(d, dialog_list_id);

            if (old_position.public_order != 0) {
              send_update_chat_position(dialog_list_id, d, source);
            }
          }
        }
      }

      if (G()->parameters().use_message_db) {
        postponed_unread_message_count_updates_.erase(dialog_list_id);
        postponed_unread_chat_count_updates_.erase(dialog_list_id);

        if (list->is_message_unread_count_inited_) {
          list->unread_message_total_count_ = 0;
          list->unread_message_muted_count_ = 0;
          send_update_unread_message_count(*list, DialogId(), true, source, true);
          G()->td_db()->get_binlog_pmc()->erase(PSTRING() << "unread_message_count" << dialog_list_id.get());
        }
        if (list->is_dialog_unread_count_inited_) {
          list->unread_dialog_total_count_ = 0;
          list->unread_dialog_muted_count_ = 0;
          list->unread_dialog_marked_count_ = 0;
          list->unread_dialog_muted_marked_count_ = 0;
          list->in_memory_dialog_total_count_ = 0;
          list->server_dialog_total_count_ = 0;
          list->secret_chat_total_count_ = 0;
          send_update_unread_chat_count(*list, DialogId(), true, source, true);
          G()->td_db()->get_binlog_pmc()->erase(PSTRING() << "unread_dialog_count" << dialog_list_id.get());
        }
      }

      auto promises = std::move(list->load_list_queries_);
      for (auto &promise : promises) {
        promise.set_error(Status::Error(400, "Chat list not found"));
      }

      dialog_lists_.erase(dialog_list_id);
      dialog_filters_.erase(it);
      return;
    }
  }
  UNREACHABLE();
}

Status MessagesManager::delete_dialog_reply_markup(DialogId dialog_id, MessageId message_id) {
  if (td_->auth_manager_->is_bot()) {
    return Status::Error(400, "Bots can't delete chat reply markup");
  }
  if (message_id.is_scheduled()) {
    return Status::Error(400, "Wrong message identifier specified");
  }
  if (!message_id.is_valid()) {
    return Status::Error(400, "Invalid message identifier specified");
  }

  Dialog *d = get_dialog_force(dialog_id, "delete_dialog_reply_markup");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }
  if (d->reply_markup_message_id != message_id) {
    return Status::OK();
  }

  Message *m = get_message_force(d, message_id, "delete_dialog_reply_markup");
  CHECK(m != nullptr);
  CHECK(m->reply_markup != nullptr);

  if (m->reply_markup->type == ReplyMarkup::Type::ForceReply) {
    set_dialog_reply_markup(d, MessageId());
  } else if (m->reply_markup->type == ReplyMarkup::Type::ShowKeyboard) {
    if (!m->reply_markup->is_one_time_keyboard) {
      return Status::Error(400, "Do not need to delete non one-time keyboard");
    }
    if (m->reply_markup->is_personal) {
      m->reply_markup->is_personal = false;
      set_dialog_reply_markup(d, message_id);

      on_message_changed(d, m, true, "delete_dialog_reply_markup");
    }
  } else {
    // non-bots can't have messages with RemoveKeyboard
    UNREACHABLE();
  }
  return Status::OK();
}

class MessagesManager::SaveDialogDraftMessageOnServerLogEvent {
 public:
  DialogId dialog_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
  }
};

Status MessagesManager::set_dialog_draft_message(DialogId dialog_id, MessageId top_thread_message_id,
                                                 tl_object_ptr<td_api::draftMessage> &&draft_message) {
  if (td_->auth_manager_->is_bot()) {
    return Status::Error(400, "Bots can't change chat draft message");
  }

  Dialog *d = get_dialog_force(dialog_id, "set_dialog_draft_message");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }
  TRY_STATUS(can_send_message(dialog_id));

  TRY_RESULT(new_draft_message, get_draft_message(td_->contacts_manager_.get(), dialog_id, std::move(draft_message)));
  if (new_draft_message != nullptr) {
    new_draft_message->reply_to_message_id =
        get_reply_to_message_id(d, top_thread_message_id, new_draft_message->reply_to_message_id, true);

    if (!new_draft_message->reply_to_message_id.is_valid() && new_draft_message->input_message_text.text.text.empty()) {
      new_draft_message = nullptr;
    }
  }

  if (top_thread_message_id != MessageId()) {
    if (!top_thread_message_id.is_valid()) {
      return Status::Error(400, "Invalid message thread specified");
    }

    auto m = get_message_force(d, top_thread_message_id, "set_dialog_draft_message");
    if (m == nullptr || m->message_id.is_scheduled() || m->reply_info.is_comment ||
        !is_active_message_reply_info(dialog_id, m->reply_info)) {
      return Status::OK();
    }

    auto &old_draft_message = m->thread_draft_message;
    if (((new_draft_message == nullptr) != (old_draft_message == nullptr)) ||
        (new_draft_message != nullptr &&
         (old_draft_message->reply_to_message_id != new_draft_message->reply_to_message_id ||
          old_draft_message->input_message_text != new_draft_message->input_message_text))) {
      old_draft_message = std::move(new_draft_message);
      on_message_changed(d, m, false, "set_dialog_draft_message");
    }
    return Status::OK();
  }

  if (update_dialog_draft_message(d, std::move(new_draft_message), false, true)) {
    if (dialog_id.get_type() != DialogType::SecretChat) {
      if (G()->parameters().use_message_db) {
        SaveDialogDraftMessageOnServerLogEvent log_event;
        log_event.dialog_id_ = dialog_id;
        add_log_event(d->save_draft_message_log_event_id, get_log_event_storer(log_event),
                      LogEvent::HandlerType::SaveDialogDraftMessageOnServer, "draft");
      }

      pending_draft_message_timeout_.set_timeout_in(dialog_id.get(), d->is_opened ? MIN_SAVE_DRAFT_DELAY : 0);
    }
  }
  return Status::OK();
}

void MessagesManager::save_dialog_draft_message_on_server(DialogId dialog_id) {
  if (G()->close_flag()) {
    return;
  }

  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  Promise<> promise;
  if (d->save_draft_message_log_event_id.log_event_id != 0) {
    d->save_draft_message_log_event_id.generation++;
    promise = PromiseCreator::lambda([actor_id = actor_id(this), dialog_id,
                                      generation = d->save_draft_message_log_event_id.generation](Result<Unit> result) {
      if (!G()->close_flag()) {
        send_closure(actor_id, &MessagesManager::on_saved_dialog_draft_message, dialog_id, generation);
      }
    });
  }

  // TODO do not send two queries simultaneously or use SequenceDispatcher
  td_->create_handler<SaveDraftMessageQuery>(std::move(promise))->send(dialog_id, d->draft_message);
}

void MessagesManager::on_saved_dialog_draft_message(DialogId dialog_id, uint64 generation) {
  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  delete_log_event(d->save_draft_message_log_event_id, generation, "draft");
}

void MessagesManager::clear_all_draft_messages(bool exclude_secret_chats, Promise<Unit> &&promise) {
  if (!exclude_secret_chats) {
    for (auto &dialog : dialogs_) {
      Dialog *d = dialog.second.get();
      if (d->dialog_id.get_type() == DialogType::SecretChat) {
        update_dialog_draft_message(d, nullptr, false, true);
      }
    }
  }
  td_->create_handler<ClearAllDraftsQuery>(std::move(promise))->send();
}

int32 MessagesManager::get_pinned_dialogs_limit(DialogListId dialog_list_id) {
  if (dialog_list_id.is_filter()) {
    return DialogFilter::MAX_INCLUDED_FILTER_DIALOGS;
  }

  Slice key{"pinned_chat_count_max"};
  int32 default_limit = 5;
  if (!dialog_list_id.is_folder() || dialog_list_id.get_folder_id() != FolderId::main()) {
    key = Slice("pinned_archived_chat_count_max");
    default_limit = 100;
  }
  int32 limit = clamp(narrow_cast<int32>(G()->shared_config().get_option_integer(key)), 0, 1000);
  if (limit <= 0) {
    return default_limit;
  }
  return limit;
}

vector<DialogId> MessagesManager::remove_secret_chat_dialog_ids(vector<DialogId> dialog_ids) {
  td::remove_if(dialog_ids, [](DialogId dialog_id) { return dialog_id.get_type() == DialogType::SecretChat; });
  return dialog_ids;
}

Status MessagesManager::toggle_dialog_is_pinned(DialogListId dialog_list_id, DialogId dialog_id, bool is_pinned) {
  if (td_->auth_manager_->is_bot()) {
    return Status::Error(400, "Bots can't change chat pin state");
  }

  Dialog *d = get_dialog_force(dialog_id, "toggle_dialog_is_pinned");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return Status::Error(400, "Can't access the chat");
  }
  if (d->order == DEFAULT_ORDER && is_pinned) {
    return Status::Error(400, "The chat can't be pinned");
  }

  auto *list = get_dialog_list(dialog_list_id);
  if (list == nullptr) {
    return Status::Error(400, "Chat list not found");
  }
  if (!list->are_pinned_dialogs_inited_) {
    return Status::Error(400, "Pinned chats must be loaded first");
  }

  bool was_pinned = is_dialog_pinned(dialog_list_id, dialog_id);
  if (is_pinned == was_pinned) {
    return Status::OK();
  }

  if (dialog_list_id.is_filter()) {
    CHECK(is_update_chat_filters_sent_);
    auto dialog_filter_id = dialog_list_id.get_filter_id();
    auto old_dialog_filter = get_dialog_filter(dialog_filter_id);
    CHECK(old_dialog_filter != nullptr);
    auto new_dialog_filter = make_unique<DialogFilter>(*old_dialog_filter);
    auto is_changed_dialog = [dialog_id](InputDialogId input_dialog_id) {
      return dialog_id == input_dialog_id.get_dialog_id();
    };
    if (is_pinned) {
      new_dialog_filter->pinned_dialog_ids.insert(new_dialog_filter->pinned_dialog_ids.begin(),
                                                  get_input_dialog_id(dialog_id));
      td::remove_if(new_dialog_filter->included_dialog_ids, is_changed_dialog);
      td::remove_if(new_dialog_filter->excluded_dialog_ids, is_changed_dialog);
    } else {
      bool is_removed = td::remove_if(new_dialog_filter->pinned_dialog_ids, is_changed_dialog);
      CHECK(is_removed);
      new_dialog_filter->included_dialog_ids.push_back(get_input_dialog_id(dialog_id));
    }

    TRY_STATUS(new_dialog_filter->check_limits());
    sort_dialog_filter_input_dialog_ids(new_dialog_filter.get(), "toggle_dialog_is_pinned");

    edit_dialog_filter(std::move(new_dialog_filter), "toggle_dialog_is_pinned");
    save_dialog_filters();
    send_update_chat_filters();

    if (dialog_id.get_type() != DialogType::SecretChat) {
      synchronize_dialog_filters();
    }

    return Status::OK();
  }

  CHECK(dialog_list_id.is_folder());
  auto folder_id = dialog_list_id.get_folder_id();
  if (is_pinned) {
    if (d->folder_id != folder_id) {
      return Status::Error(400, "Chat not in the list");
    }

    auto pinned_dialog_ids = get_pinned_dialog_ids(dialog_list_id);
    auto pinned_dialog_count = pinned_dialog_ids.size();
    auto secret_pinned_dialog_count =
        std::count_if(pinned_dialog_ids.begin(), pinned_dialog_ids.end(),
                      [](DialogId dialog_id) { return dialog_id.get_type() == DialogType::SecretChat; });
    size_t dialog_count = dialog_id.get_type() == DialogType::SecretChat
                              ? secret_pinned_dialog_count
                              : pinned_dialog_count - secret_pinned_dialog_count;

    if (dialog_count >= static_cast<size_t>(get_pinned_dialogs_limit(dialog_list_id))) {
      return Status::Error(400, "The maximum number of pinned chats exceeded");
    }
  }

  if (set_dialog_is_pinned(dialog_list_id, d, is_pinned)) {
    toggle_dialog_is_pinned_on_server(dialog_id, is_pinned, 0);
  }
  return Status::OK();
}

class MessagesManager::ToggleDialogIsPinnedOnServerLogEvent {
 public:
  DialogId dialog_id_;
  bool is_pinned_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(is_pinned_);
    END_STORE_FLAGS();

    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(is_pinned_);
    END_PARSE_FLAGS();

    td::parse(dialog_id_, parser);
  }
};

uint64 MessagesManager::save_toggle_dialog_is_pinned_on_server_log_event(DialogId dialog_id, bool is_pinned) {
  ToggleDialogIsPinnedOnServerLogEvent log_event{dialog_id, is_pinned};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ToggleDialogIsPinnedOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::toggle_dialog_is_pinned_on_server(DialogId dialog_id, bool is_pinned, uint64 log_event_id) {
  CHECK(!td_->auth_manager_->is_bot());
  if (log_event_id == 0 && dialog_id.get_type() == DialogType::SecretChat) {
    // don't even create new binlog events
    return;
  }

  if (log_event_id == 0 && G()->parameters().use_message_db) {
    log_event_id = save_toggle_dialog_is_pinned_on_server_log_event(dialog_id, is_pinned);
  }

  td_->create_handler<ToggleDialogPinQuery>(get_erase_log_event_promise(log_event_id))->send(dialog_id, is_pinned);
}

Status MessagesManager::set_pinned_dialogs(DialogListId dialog_list_id, vector<DialogId> dialog_ids) {
  if (td_->auth_manager_->is_bot()) {
    return Status::Error(400, "Bots can't reorder pinned chats");
  }

  int32 dialog_count = 0;
  int32 secret_dialog_count = 0;
  auto dialog_count_limit = get_pinned_dialogs_limit(dialog_list_id);
  for (auto dialog_id : dialog_ids) {
    Dialog *d = get_dialog_force(dialog_id, "set_pinned_dialogs");
    if (d == nullptr) {
      return Status::Error(400, "Chat not found");
    }
    if (!have_input_peer(dialog_id, AccessRights::Read)) {
      return Status::Error(400, "Can't access the chat");
    }
    if (d->order == DEFAULT_ORDER) {
      return Status::Error(400, "The chat can't be pinned");
    }
    if (dialog_list_id.is_folder() && d->folder_id != dialog_list_id.get_folder_id()) {
      return Status::Error(400, "Chat not in the list");
    }
    if (dialog_id.get_type() == DialogType::SecretChat) {
      secret_dialog_count++;
    } else {
      dialog_count++;
    }

    if (dialog_count > dialog_count_limit || secret_dialog_count > dialog_count_limit) {
      return Status::Error(400, "The maximum number of pinned chats exceeded");
    }
  }
  std::unordered_set<DialogId, DialogIdHash> new_pinned_dialog_ids(dialog_ids.begin(), dialog_ids.end());
  if (new_pinned_dialog_ids.size() != dialog_ids.size()) {
    return Status::Error(400, "Duplicate chats in the list of pinned chats");
  }

  auto *list = get_dialog_list(dialog_list_id);
  if (list == nullptr) {
    return Status::Error(400, "Chat list not found");
  }
  if (!list->are_pinned_dialogs_inited_) {
    return Status::Error(400, "Pinned chats must be loaded first");
  }

  auto pinned_dialog_ids = get_pinned_dialog_ids(dialog_list_id);
  if (pinned_dialog_ids == dialog_ids) {
    return Status::OK();
  }
  LOG(INFO) << "Reorder pinned chats in " << dialog_list_id << " from " << pinned_dialog_ids << " to " << dialog_ids;

  auto server_old_dialog_ids = remove_secret_chat_dialog_ids(pinned_dialog_ids);
  auto server_new_dialog_ids = remove_secret_chat_dialog_ids(dialog_ids);

  if (dialog_list_id.is_filter()) {
    CHECK(is_update_chat_filters_sent_);
    auto dialog_filter_id = dialog_list_id.get_filter_id();
    auto old_dialog_filter = get_dialog_filter(dialog_filter_id);
    CHECK(old_dialog_filter != nullptr);
    auto new_dialog_filter = make_unique<DialogFilter>(*old_dialog_filter);
    auto old_pinned_dialog_ids = std::move(new_dialog_filter->pinned_dialog_ids);
    new_dialog_filter->pinned_dialog_ids =
        transform(dialog_ids, [this](DialogId dialog_id) { return get_input_dialog_id(dialog_id); });
    auto is_new_pinned = [&new_pinned_dialog_ids](InputDialogId input_dialog_id) {
      return new_pinned_dialog_ids.count(input_dialog_id.get_dialog_id()) > 0;
    };
    td::remove_if(old_pinned_dialog_ids, is_new_pinned);
    td::remove_if(new_dialog_filter->included_dialog_ids, is_new_pinned);
    td::remove_if(new_dialog_filter->excluded_dialog_ids, is_new_pinned);
    append(new_dialog_filter->included_dialog_ids, old_pinned_dialog_ids);

    TRY_STATUS(new_dialog_filter->check_limits());
    sort_dialog_filter_input_dialog_ids(new_dialog_filter.get(), "set_pinned_dialogs");

    edit_dialog_filter(std::move(new_dialog_filter), "set_pinned_dialogs");
    save_dialog_filters();
    send_update_chat_filters();

    if (server_old_dialog_ids != server_new_dialog_ids) {
      synchronize_dialog_filters();
    }

    return Status::OK();
  }

  CHECK(dialog_list_id.is_folder());

  std::reverse(pinned_dialog_ids.begin(), pinned_dialog_ids.end());
  std::reverse(dialog_ids.begin(), dialog_ids.end());

  std::unordered_set<DialogId, DialogIdHash> old_pinned_dialog_ids(pinned_dialog_ids.begin(), pinned_dialog_ids.end());
  auto old_it = pinned_dialog_ids.begin();
  for (auto dialog_id : dialog_ids) {
    old_pinned_dialog_ids.erase(dialog_id);
    while (old_it < pinned_dialog_ids.end()) {
      if (*old_it == dialog_id) {
        break;
      }
      ++old_it;
    }
    if (old_it < pinned_dialog_ids.end()) {
      // leave dialog where it is
      ++old_it;
      continue;
    }
    set_dialog_is_pinned(dialog_id, true);
  }
  for (auto dialog_id : old_pinned_dialog_ids) {
    Dialog *d = get_dialog_force(dialog_id, "set_pinned_dialogs 2");
    if (d == nullptr) {
      LOG(ERROR) << "Failed to find " << dialog_id << " to unpin in " << dialog_list_id;
      force_create_dialog(dialog_id, "set_pinned_dialogs", true);
      d = get_dialog_force(dialog_id, "set_pinned_dialogs 3");
    }
    if (d != nullptr) {
      set_dialog_is_pinned(dialog_list_id, d, false);
    }
  }

  if (server_old_dialog_ids != server_new_dialog_ids) {
    reorder_pinned_dialogs_on_server(dialog_list_id.get_folder_id(), server_new_dialog_ids, 0);
  }
  return Status::OK();
}

class MessagesManager::ReorderPinnedDialogsOnServerLogEvent {
 public:
  FolderId folder_id_;
  vector<DialogId> dialog_ids_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(folder_id_, storer);
    td::store(dialog_ids_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    if (parser.version() >= static_cast<int32>(Version::AddFolders)) {
      td::parse(folder_id_, parser);
    } else {
      folder_id_ = FolderId();
    }
    td::parse(dialog_ids_, parser);
  }
};

uint64 MessagesManager::save_reorder_pinned_dialogs_on_server_log_event(FolderId folder_id,
                                                                        const vector<DialogId> &dialog_ids) {
  ReorderPinnedDialogsOnServerLogEvent log_event{folder_id, dialog_ids};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ReorderPinnedDialogsOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::reorder_pinned_dialogs_on_server(FolderId folder_id, const vector<DialogId> &dialog_ids,
                                                       uint64 log_event_id) {
  if (log_event_id == 0 && G()->parameters().use_message_db) {
    log_event_id = save_reorder_pinned_dialogs_on_server_log_event(folder_id, dialog_ids);
  }

  td_->create_handler<ReorderPinnedDialogsQuery>(get_erase_log_event_promise(log_event_id))
      ->send(folder_id, dialog_ids);
}

Status MessagesManager::toggle_dialog_is_marked_as_unread(DialogId dialog_id, bool is_marked_as_unread) {
  Dialog *d = get_dialog_force(dialog_id, "toggle_dialog_is_marked_as_unread");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return Status::Error(400, "Can't access the chat");
  }

  if (is_marked_as_unread == d->is_marked_as_unread) {
    return Status::OK();
  }

  set_dialog_is_marked_as_unread(d, is_marked_as_unread);

  toggle_dialog_is_marked_as_unread_on_server(dialog_id, is_marked_as_unread, 0);
  return Status::OK();
}

class MessagesManager::ToggleDialogIsMarkedAsUnreadOnServerLogEvent {
 public:
  DialogId dialog_id_;
  bool is_marked_as_unread_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(is_marked_as_unread_);
    END_STORE_FLAGS();

    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(is_marked_as_unread_);
    END_PARSE_FLAGS();

    td::parse(dialog_id_, parser);
  }
};

uint64 MessagesManager::save_toggle_dialog_is_marked_as_unread_on_server_log_event(DialogId dialog_id,
                                                                                   bool is_marked_as_unread) {
  ToggleDialogIsMarkedAsUnreadOnServerLogEvent log_event{dialog_id, is_marked_as_unread};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ToggleDialogIsMarkedAsUnreadOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::toggle_dialog_is_marked_as_unread_on_server(DialogId dialog_id, bool is_marked_as_unread,
                                                                  uint64 log_event_id) {
  if (log_event_id == 0 && dialog_id.get_type() == DialogType::SecretChat) {
    // don't even create new binlog events
    return;
  }

  if (log_event_id == 0 && G()->parameters().use_message_db) {
    log_event_id = save_toggle_dialog_is_marked_as_unread_on_server_log_event(dialog_id, is_marked_as_unread);
  }

  td_->create_handler<ToggleDialogUnreadMarkQuery>(get_erase_log_event_promise(log_event_id))
      ->send(dialog_id, is_marked_as_unread);
}

Status MessagesManager::toggle_message_sender_is_blocked(const td_api::object_ptr<td_api::MessageSender> &sender,
                                                         bool is_blocked) {
  if (sender == nullptr) {
    return Status::Error(400, "Message sender must be non-empty");
  }

  UserId sender_user_id;
  DialogId dialog_id;
  switch (sender->get_id()) {
    case td_api::messageSenderUser::ID:
      sender_user_id = UserId(static_cast<const td_api::messageSenderUser *>(sender.get())->user_id_);
      break;
    case td_api::messageSenderChat::ID: {
      auto sender_dialog_id = DialogId(static_cast<const td_api::messageSenderChat *>(sender.get())->chat_id_);
      if (!have_dialog_force(sender_dialog_id, "toggle_message_sender_is_blocked")) {
        return Status::Error(400, "Sender chat not found");
      }

      switch (sender_dialog_id.get_type()) {
        case DialogType::User:
          sender_user_id = sender_dialog_id.get_user_id();
          break;
        case DialogType::Chat:
          return Status::Error(400, "Basic group chats can't be blocked");
        case DialogType::Channel:
          dialog_id = sender_dialog_id;
          break;
        case DialogType::SecretChat:
          sender_user_id = td_->contacts_manager_->get_secret_chat_user_id(sender_dialog_id.get_secret_chat_id());
          break;
        case DialogType::None:
        default:
          UNREACHABLE();
      }
      break;
    }
    default:
      UNREACHABLE();
  }
  if (!dialog_id.is_valid()) {
    if (!td_->contacts_manager_->have_user_force(sender_user_id)) {
      return Status::Error(400, "Sender user not found");
    }
    dialog_id = DialogId(sender_user_id);
  }
  if (dialog_id == get_my_dialog_id()) {
    return Status::Error(400, is_blocked ? Slice("Can't block self") : Slice("Can't unblock self"));
  }

  Dialog *d = get_dialog_force(dialog_id, "toggle_message_sender_is_blocked");
  if (!have_input_peer(dialog_id, AccessRights::Know)) {
    return Status::Error(400, "Message sender isn't accessible");
  }
  if (d != nullptr) {
    if (is_blocked == d->is_blocked) {
      return Status::OK();
    }
    set_dialog_is_blocked(d, is_blocked);
  } else {
    CHECK(dialog_id.get_type() == DialogType::User);
    td_->contacts_manager_->on_update_user_is_blocked(dialog_id.get_user_id(), is_blocked);
  }

  toggle_dialog_is_blocked_on_server(dialog_id, is_blocked, 0);
  return Status::OK();
}

class MessagesManager::ToggleDialogIsBlockedOnServerLogEvent {
 public:
  DialogId dialog_id_;
  bool is_blocked_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(is_blocked_);
    END_STORE_FLAGS();

    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(is_blocked_);
    END_PARSE_FLAGS();

    td::parse(dialog_id_, parser);
  }
};

uint64 MessagesManager::save_toggle_dialog_is_blocked_on_server_log_event(DialogId dialog_id, bool is_blocked) {
  ToggleDialogIsBlockedOnServerLogEvent log_event{dialog_id, is_blocked};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ToggleDialogIsBlockedOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::toggle_dialog_is_blocked_on_server(DialogId dialog_id, bool is_blocked, uint64 log_event_id) {
  if (log_event_id == 0 && G()->parameters().use_message_db) {
    log_event_id = save_toggle_dialog_is_blocked_on_server_log_event(dialog_id, is_blocked);
  }

  send_closure(td_->create_net_actor<ToggleDialogIsBlockedActor>(get_erase_log_event_promise(log_event_id)),
               &ToggleDialogIsBlockedActor::send, dialog_id, is_blocked,
               get_sequence_dispatcher_id(dialog_id, MessageContentType::Text));
}

Status MessagesManager::toggle_dialog_silent_send_message(DialogId dialog_id, bool silent_send_message) {
  CHECK(!td_->auth_manager_->is_bot());

  Dialog *d = get_dialog_force(dialog_id, "toggle_dialog_silent_send_message");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return Status::Error(400, "Can't access the chat");
  }

  if (update_dialog_silent_send_message(d, silent_send_message)) {
    update_dialog_notification_settings_on_server(dialog_id, false);
  }

  return Status::OK();
}

class MessagesManager::UpdateDialogNotificationSettingsOnServerLogEvent {
 public:
  DialogId dialog_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
  }
};

void MessagesManager::update_dialog_notification_settings_on_server(DialogId dialog_id, bool from_binlog) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  if (!from_binlog && get_input_notify_peer(dialog_id) == nullptr) {
    // don't even create new binlog events
    return;
  }

  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  if (!from_binlog && G()->parameters().use_message_db) {
    UpdateDialogNotificationSettingsOnServerLogEvent log_event;
    log_event.dialog_id_ = dialog_id;
    add_log_event(d->save_notification_settings_log_event_id, get_log_event_storer(log_event),
                  LogEvent::HandlerType::UpdateDialogNotificationSettingsOnServer, "notification settings");
  }

  Promise<> promise;
  if (d->save_notification_settings_log_event_id.log_event_id != 0) {
    d->save_notification_settings_log_event_id.generation++;
    promise = PromiseCreator::lambda(
        [actor_id = actor_id(this), dialog_id,
         generation = d->save_notification_settings_log_event_id.generation](Result<Unit> result) {
          if (!G()->close_flag()) {
            send_closure(actor_id, &MessagesManager::on_updated_dialog_notification_settings, dialog_id, generation);
          }
        });
  }

  send_update_dialog_notification_settings_query(d, std::move(promise));
}

void MessagesManager::send_update_dialog_notification_settings_query(const Dialog *d, Promise<Unit> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());
  CHECK(d != nullptr);
  // TODO do not send two queries simultaneously or use SequenceDispatcher
  td_->create_handler<UpdateDialogNotifySettingsQuery>(std::move(promise))
      ->send(d->dialog_id, d->notification_settings);
}

void MessagesManager::on_updated_dialog_notification_settings(DialogId dialog_id, uint64 generation) {
  CHECK(!td_->auth_manager_->is_bot());
  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  delete_log_event(d->save_notification_settings_log_event_id, generation, "notification settings");
}

Status MessagesManager::set_dialog_client_data(DialogId dialog_id, string &&client_data) {
  Dialog *d = get_dialog_force(dialog_id, "set_dialog_client_data");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }

  d->client_data = std::move(client_data);
  on_dialog_updated(d->dialog_id, "set_dialog_client_data");
  return Status::OK();
}

bool MessagesManager::is_dialog_inited(const Dialog *d) {
  return d != nullptr && d->notification_settings.is_synchronized && d->is_last_read_inbox_message_id_inited &&
         d->is_last_read_outbox_message_id_inited;
}

int32 MessagesManager::get_dialog_mute_until(const Dialog *d) const {
  CHECK(!td_->auth_manager_->is_bot());
  CHECK(d != nullptr);
  return d->notification_settings.use_default_mute_until ? get_scope_mute_until(d->dialog_id)
                                                         : d->notification_settings.mute_until;
}

bool MessagesManager::is_dialog_muted(const Dialog *d) const {
  return get_dialog_mute_until(d) != 0;
}

bool MessagesManager::is_dialog_pinned_message_notifications_disabled(const Dialog *d) const {
  CHECK(!td_->auth_manager_->is_bot());
  CHECK(d != nullptr);
  if (d->notification_settings.use_default_disable_pinned_message_notifications) {
    auto scope = get_dialog_notification_setting_scope(d->dialog_id);
    return get_scope_notification_settings(scope)->disable_pinned_message_notifications;
  }

  return d->notification_settings.disable_pinned_message_notifications;
}

bool MessagesManager::is_dialog_mention_notifications_disabled(const Dialog *d) const {
  CHECK(!td_->auth_manager_->is_bot());
  CHECK(d != nullptr);
  if (d->notification_settings.use_default_disable_mention_notifications) {
    auto scope = get_dialog_notification_setting_scope(d->dialog_id);
    return get_scope_notification_settings(scope)->disable_mention_notifications;
  }

  return d->notification_settings.disable_mention_notifications;
}

void MessagesManager::create_dialog(DialogId dialog_id, bool force, Promise<Unit> &&promise) {
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    if (!have_dialog_info_force(dialog_id)) {
      return promise.set_error(Status::Error(400, "Chat info not found"));
    }
    if (!have_input_peer(dialog_id, AccessRights::Read)) {
      return promise.set_error(Status::Error(400, "Can't access the chat"));
    }
  }

  if (force || td_->auth_manager_->is_bot() || dialog_id.get_type() == DialogType::SecretChat) {
    force_create_dialog(dialog_id, "create dialog");
  } else {
    const Dialog *d = get_dialog_force(dialog_id, "create_dialog");
    if (!is_dialog_inited(d)) {
      return send_get_dialog_query(dialog_id, std::move(promise), 0, "create_dialog");
    }
  }

  promise.set_value(Unit());
}

DialogId MessagesManager::create_new_group_chat(const vector<UserId> &user_ids, const string &title, int64 &random_id,
                                                Promise<Unit> &&promise) {
  LOG(INFO) << "Trying to create group chat \"" << title << "\" with members " << format::as_array(user_ids);

  if (random_id != 0) {
    // request has already been sent before
    auto it = created_dialogs_.find(random_id);
    CHECK(it != created_dialogs_.end());
    auto dialog_id = it->second;
    CHECK(dialog_id.get_type() == DialogType::Chat);
    CHECK(have_dialog(dialog_id));

    created_dialogs_.erase(it);

    // set default notification settings to newly created chat
    on_update_dialog_notify_settings(dialog_id, make_tl_object<telegram_api::peerNotifySettings>(),
                                     "create_new_group_chat");

    promise.set_value(Unit());
    return dialog_id;
  }

  if (user_ids.empty()) {
    promise.set_error(Status::Error(400, "Too few users to create basic group chat"));
    return DialogId();
  }

  auto new_title = clean_name(title, MAX_TITLE_LENGTH);
  if (new_title.empty()) {
    promise.set_error(Status::Error(400, "Title can't be empty"));
    return DialogId();
  }

  vector<tl_object_ptr<telegram_api::InputUser>> input_users;
  for (auto user_id : user_ids) {
    auto input_user = td_->contacts_manager_->get_input_user(user_id);
    if (input_user == nullptr) {
      promise.set_error(Status::Error(400, "User not found"));
      return DialogId();
    }
    input_users.push_back(std::move(input_user));
  }

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || created_dialogs_.find(random_id) != created_dialogs_.end());
  created_dialogs_[random_id];  // reserve place for result

  td_->create_handler<CreateChatQuery>(std::move(promise))->send(std::move(input_users), new_title, random_id);
  return DialogId();
}

DialogId MessagesManager::create_new_channel_chat(const string &title, bool is_megagroup, const string &description,
                                                  const DialogLocation &location, bool for_import, int64 &random_id,
                                                  Promise<Unit> &&promise) {
  LOG(INFO) << "Trying to create " << (is_megagroup ? "supergroup" : "broadcast") << " with title \"" << title
            << "\", description \"" << description << "\" and " << location;

  if (random_id != 0) {
    // request has already been sent before
    auto it = created_dialogs_.find(random_id);
    CHECK(it != created_dialogs_.end());
    auto dialog_id = it->second;
    CHECK(dialog_id.get_type() == DialogType::Channel);
    CHECK(have_dialog(dialog_id));

    created_dialogs_.erase(it);

    // set default notification settings to newly created chat
    on_update_dialog_notify_settings(dialog_id, make_tl_object<telegram_api::peerNotifySettings>(),
                                     "create_new_channel_chat");

    promise.set_value(Unit());
    return dialog_id;
  }

  auto new_title = clean_name(title, MAX_TITLE_LENGTH);
  if (new_title.empty()) {
    promise.set_error(Status::Error(400, "Title can't be empty"));
    return DialogId();
  }

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || created_dialogs_.find(random_id) != created_dialogs_.end());
  created_dialogs_[random_id];  // reserve place for result

  td_->create_handler<CreateChannelQuery>(std::move(promise))
      ->send(new_title, is_megagroup, strip_empty_characters(description, MAX_DESCRIPTION_LENGTH), location, for_import,
             random_id);
  return DialogId();
}

void MessagesManager::create_new_secret_chat(UserId user_id, Promise<SecretChatId> &&promise) {
  auto user_base = td_->contacts_manager_->get_input_user(user_id);
  if (user_base == nullptr || user_base->get_id() != telegram_api::inputUser::ID) {
    return promise.set_error(Status::Error(400, "User not found"));
  }
  auto user = move_tl_object_as<telegram_api::inputUser>(user_base);

  send_closure(G()->secret_chats_manager(), &SecretChatsManager::create_chat, UserId(user->user_id_),
               user->access_hash_, std::move(promise));
}

DialogId MessagesManager::migrate_dialog_to_megagroup(DialogId dialog_id, Promise<Unit> &&promise) {
  LOG(INFO) << "Trying to convert " << dialog_id << " to supergroup";

  if (dialog_id.get_type() != DialogType::Chat) {
    promise.set_error(Status::Error(400, "Only basic group chats can be converted to supergroup"));
    return DialogId();
  }

  auto channel_id = td_->contacts_manager_->migrate_chat_to_megagroup(dialog_id.get_chat_id(), promise);
  if (!channel_id.is_valid()) {
    return DialogId();
  }

  if (!td_->contacts_manager_->have_channel(channel_id)) {
    LOG(ERROR) << "Can't find info about supergroup to which the group has migrated";
    promise.set_error(Status::Error(400, "Supergroup is not found"));
    return DialogId();
  }

  auto new_dialog_id = DialogId(channel_id);
  Dialog *d = get_dialog_force(new_dialog_id, "migrate_dialog_to_megagroup");
  if (d == nullptr) {
    d = add_dialog(new_dialog_id, "migrate_dialog_to_megagroup");
    if (d->pts == 0) {
      d->pts = 1;
      if (is_debug_message_op_enabled()) {
        d->debug_message_op.emplace_back(Dialog::MessageOp::SetPts, d->pts, "migrate");
      }
    }
    update_dialog_pos(d, "migrate_dialog_to_megagroup");
  }

  promise.set_value(Unit());
  return new_dialog_id;
}

Status MessagesManager::open_dialog(DialogId dialog_id) {
  Dialog *d = get_dialog_force(dialog_id, "open_dialog");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }

  open_dialog(d);
  return Status::OK();
}

Status MessagesManager::close_dialog(DialogId dialog_id) {
  Dialog *d = get_dialog_force(dialog_id, "close_dialog");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }

  close_dialog(d);
  return Status::OK();
}

DialogId MessagesManager::get_my_dialog_id() const {
  return DialogId(td_->contacts_manager_->get_my_id());
}

Status MessagesManager::view_messages(DialogId dialog_id, MessageId top_thread_message_id,
                                      const vector<MessageId> &message_ids, bool force_read) {
  CHECK(!td_->auth_manager_->is_bot());

  Dialog *d = get_dialog_force(dialog_id, "view_messages");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }
  for (auto message_id : message_ids) {
    if (!message_id.is_valid() && !message_id.is_valid_scheduled()) {
      return Status::Error(400, "Invalid message identifier");
    }
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return Status::Error(400, "Can't access the chat");
  }

  if (top_thread_message_id != MessageId()) {
    if (!top_thread_message_id.is_valid() || !top_thread_message_id.is_server()) {
      return Status::Error(400, "Invalid message thread ID specified");
    }
    if (dialog_id.get_type() != DialogType::Channel || is_broadcast_channel(dialog_id)) {
      return Status::Error(400, "There are no message threads in the chat");
    }
  }

  bool need_read = force_read || d->is_opened;
  MessageId max_message_id;  // max server or local viewed message_id
  vector<MessageId> read_content_message_ids;
  for (auto message_id : message_ids) {
    if (!message_id.is_valid()) {
      continue;
    }

    auto *m = get_message_force(d, message_id, "view_messages");
    if (m != nullptr) {
      if (m->message_id.is_server() && m->view_count > 0) {
        d->pending_viewed_message_ids.insert(m->message_id);
      }

      if (!m->message_id.is_yet_unsent() && m->message_id > max_message_id) {
        max_message_id = m->message_id;
      }

      auto message_content_type = m->content->get_type();
      if (message_content_type == MessageContentType::LiveLocation) {
        on_message_live_location_viewed(d, m);
      }

      if (need_read && message_content_type != MessageContentType::VoiceNote &&
          message_content_type != MessageContentType::VideoNote &&
          update_message_contains_unread_mention(d, m, false, "view_messages")) {
        CHECK(m->message_id.is_server());
        read_content_message_ids.push_back(m->message_id);
        on_message_changed(d, m, true, "view_messages");
      }
    } else if (!message_id.is_yet_unsent() && message_id > max_message_id &&
               message_id <= d->max_notification_message_id) {
      max_message_id = message_id;
    }
  }
  if (!d->pending_viewed_message_ids.empty()) {
    pending_message_views_timeout_.add_timeout_in(dialog_id.get(), MAX_MESSAGE_VIEW_DELAY);
    d->increment_view_counter |= d->is_opened;
  }
  if (!read_content_message_ids.empty()) {
    read_message_contents_on_server(dialog_id, std::move(read_content_message_ids), 0, Auto());
  }

  if (!need_read) {
    return Status::OK();
  }

  if (top_thread_message_id.is_valid()) {
    MessageId prev_last_read_inbox_message_id;
    MessageId max_thread_message_id;
    Message *top_m = get_message_force(d, top_thread_message_id, "view_messages 2");
    if (top_m != nullptr && is_active_message_reply_info(dialog_id, top_m->reply_info)) {
      prev_last_read_inbox_message_id = top_m->reply_info.last_read_inbox_message_id;
      if (top_m->reply_info.update_max_message_ids(MessageId(), max_message_id, MessageId())) {
        on_message_reply_info_changed(dialog_id, top_m);
        on_message_changed(d, top_m, true, "view_messages 3");
      }
      max_thread_message_id = top_m->reply_info.max_message_id;

      if (is_discussion_message(dialog_id, top_m)) {
        auto linked_dialog_id = top_m->forward_info->from_dialog_id;
        auto linked_d = get_dialog(linked_dialog_id);
        CHECK(linked_d != nullptr);
        CHECK(linked_dialog_id.get_type() == DialogType::Channel);
        auto *linked_m = get_message_force(linked_d, top_m->forward_info->from_message_id, "view_messages 4");
        if (linked_m != nullptr && is_active_message_reply_info(linked_dialog_id, linked_m->reply_info)) {
          if (linked_m->reply_info.last_read_inbox_message_id < prev_last_read_inbox_message_id) {
            prev_last_read_inbox_message_id = linked_m->reply_info.last_read_inbox_message_id;
          }
          if (linked_m->reply_info.update_max_message_ids(MessageId(), max_message_id, MessageId())) {
            on_message_reply_info_changed(linked_dialog_id, linked_m);
            on_message_changed(linked_d, linked_m, true, "view_messages 5");
          }
          if (linked_m->reply_info.max_message_id > max_thread_message_id) {
            max_thread_message_id = linked_m->reply_info.max_message_id;
          }
        }
      }
    }

    if (max_message_id.get_prev_server_message_id().get() >
        prev_last_read_inbox_message_id.get_prev_server_message_id().get()) {
      read_message_thread_history_on_server(d, top_thread_message_id, max_message_id.get_prev_server_message_id(),
                                            max_thread_message_id.get_prev_server_message_id());
    }

    return Status::OK();
  }

  if (max_message_id > d->last_read_inbox_message_id) {
    const MessageId last_read_message_id = max_message_id;
    const MessageId prev_last_read_inbox_message_id = d->last_read_inbox_message_id;
    MessageId read_history_on_server_message_id;
    if (dialog_id.get_type() != DialogType::SecretChat) {
      if (last_read_message_id.get_prev_server_message_id().get() >
          prev_last_read_inbox_message_id.get_prev_server_message_id().get()) {
        read_history_on_server_message_id = last_read_message_id.get_prev_server_message_id();
      }
    } else {
      if (last_read_message_id > prev_last_read_inbox_message_id) {
        read_history_on_server_message_id = last_read_message_id;
      }
    }

    if (read_history_on_server_message_id.is_valid()) {
      // add dummy timeout to not try to repair unread_count in read_history_inbox before server request succeeds
      // the timeout will be overwritten in the read_history_on_server call
      pending_read_history_timeout_.add_timeout_in(dialog_id.get(), 0);
    }
    read_history_inbox(d->dialog_id, last_read_message_id, -1, "view_messages");
    if (read_history_on_server_message_id.is_valid()) {
      // call read_history_on_server after read_history_inbox to not have delay before request if all messages are read
      read_history_on_server(d, read_history_on_server_message_id);
    }
  }
  if (d->is_marked_as_unread) {
    set_dialog_is_marked_as_unread(d, false);
  }

  return Status::OK();
}

Status MessagesManager::open_message_content(FullMessageId full_message_id) {
  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id, "open_message_content");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }

  auto *m = get_message_force(d, full_message_id.get_message_id(), "open_message_content");
  if (m == nullptr) {
    return Status::Error(400, "Message not found");
  }

  if (m->message_id.is_scheduled() || m->message_id.is_yet_unsent() || m->is_outgoing) {
    return Status::OK();
  }

  if (read_message_content(d, m, true, "open_message_content") &&
      (m->message_id.is_server() || dialog_id.get_type() == DialogType::SecretChat)) {
    read_message_contents_on_server(dialog_id, {m->message_id}, 0, Auto());
  }

  if (m->content->get_type() == MessageContentType::LiveLocation) {
    on_message_live_location_viewed(d, m);
  }

  return Status::OK();
}

class MessagesManager::ReadMessageContentsOnServerLogEvent {
 public:
  DialogId dialog_id_;
  vector<MessageId> message_ids_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
    td::store(message_ids_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
    td::parse(message_ids_, parser);
  }
};

uint64 MessagesManager::save_read_message_contents_on_server_log_event(DialogId dialog_id,
                                                                       const vector<MessageId> &message_ids) {
  ReadMessageContentsOnServerLogEvent log_event{dialog_id, message_ids};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ReadMessageContentsOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::read_message_contents_on_server(DialogId dialog_id, vector<MessageId> message_ids,
                                                      uint64 log_event_id, Promise<Unit> &&promise,
                                                      bool skip_log_event) {
  CHECK(!message_ids.empty());

  LOG(INFO) << "Read contents of " << format::as_array(message_ids) << " in " << dialog_id << " on server";

  if (log_event_id == 0 && G()->parameters().use_message_db && !skip_log_event) {
    log_event_id = save_read_message_contents_on_server_log_event(dialog_id, message_ids);
  }

  auto new_promise = get_erase_log_event_promise(log_event_id, std::move(promise));
  promise = std::move(new_promise);  // to prevent self-move

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
      td_->create_handler<ReadMessagesContentsQuery>(std::move(promise))->send(std::move(message_ids));
      break;
    case DialogType::Channel:
      td_->create_handler<ReadChannelMessagesContentsQuery>(std::move(promise))
          ->send(dialog_id.get_channel_id(), std::move(message_ids));
      break;
    case DialogType::SecretChat: {
      CHECK(message_ids.size() == 1);
      auto m = get_message_force({dialog_id, message_ids[0]}, "read_message_contents_on_server");
      if (m != nullptr) {
        send_closure(G()->secret_chats_manager(), &SecretChatsManager::send_open_message,
                     dialog_id.get_secret_chat_id(), m->random_id, std::move(promise));
      } else {
        promise.set_error(Status::Error(400, "Message not found"));
      }
      break;
    }
    case DialogType::None:
    default:
      UNREACHABLE();
  }
}

void MessagesManager::click_animated_emoji_message(FullMessageId full_message_id,
                                                   Promise<td_api::object_ptr<td_api::sticker>> &&promise) {
  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id, "click_animated_emoji_message");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }

  auto message_id = get_persistent_message_id(d, full_message_id.get_message_id());
  auto *m = get_message_force(d, message_id, "click_animated_emoji_message");
  if (m == nullptr) {
    return promise.set_error(Status::Error(400, "Message not found"));
  }

  if (m->message_id.is_scheduled() || dialog_id.get_type() != DialogType::User || !m->message_id.is_server()) {
    return promise.set_value(nullptr);
  }

  get_message_content_animated_emoji_click_sticker(m->content.get(), full_message_id, td_, std::move(promise));
}

void MessagesManager::on_animated_emoji_message_clicked(FullMessageId full_message_id, UserId user_id, Slice emoji,
                                                        string data) {
  CHECK(full_message_id.get_message_id().is_server());
  auto *m = get_message_force(full_message_id, "on_animated_emoji_message_clicked");
  if (m == nullptr) {
    return;
  }
  if (full_message_id.get_dialog_id().get_type() != DialogType::User ||
      full_message_id.get_dialog_id().get_user_id() != user_id) {
    return;
  }
  on_message_content_animated_emoji_clicked(m->content.get(), full_message_id, td_, emoji, std::move(data));
}

void MessagesManager::open_dialog(Dialog *d) {
  CHECK(!td_->auth_manager_->is_bot());
  DialogId dialog_id = d->dialog_id;
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return;
  }
  recently_opened_dialogs_.add_dialog(dialog_id);
  if (d->is_opened) {
    return;
  }
  d->is_opened = true;
  d->was_opened = true;

  auto min_message_id = MessageId(ServerMessageId(1));
  if (d->last_message_id == MessageId() && d->last_read_outbox_message_id < min_message_id && d->messages != nullptr &&
      d->messages->message_id < min_message_id) {
    Message *m = d->messages.get();
    while (m->right != nullptr) {
      m = m->right.get();
    }
    if (m->message_id < min_message_id) {
      read_history_inbox(dialog_id, m->message_id, -1, "open_dialog");
    }
  }

  if (d->has_unload_timeout) {
    LOG(INFO) << "Cancel unload timeout for " << dialog_id;
    pending_unload_dialog_timeout_.cancel_timeout(dialog_id.get());
    d->has_unload_timeout = false;
  }

  if (d->new_secret_chat_notification_id.is_valid()) {
    remove_new_secret_chat_notification(d, true);
  }

  get_dialog_pinned_message(dialog_id, Auto());

  if (d->active_group_call_id.is_valid()) {
    td_->group_call_manager_->reload_group_call(d->active_group_call_id, Auto());
  }

  switch (dialog_id.get_type()) {
    case DialogType::User:
      break;
    case DialogType::Chat:
      td_->contacts_manager_->repair_chat_participants(dialog_id.get_chat_id());
      reget_dialog_action_bar(dialog_id, "open_dialog");
      break;
    case DialogType::Channel:
      if (!is_broadcast_channel(dialog_id)) {
        auto participant_count = td_->contacts_manager_->get_channel_participant_count(dialog_id.get_channel_id());
        if (participant_count < 195) {  // include unknown participant_count
          td_->contacts_manager_->get_channel_participants(dialog_id.get_channel_id(),
                                                           td_api::make_object<td_api::supergroupMembersFilterRecent>(),
                                                           string(), 0, 200, 200, Auto());
        }
      }
      get_channel_difference(dialog_id, d->pts, true, "open_dialog");
      reget_dialog_action_bar(dialog_id, "open_dialog");
      break;
    case DialogType::SecretChat: {
      // to repair dialog action bar
      auto user_id = td_->contacts_manager_->get_secret_chat_user_id(dialog_id.get_secret_chat_id());
      if (user_id.is_valid()) {
        td_->contacts_manager_->reload_user_full(user_id);
      }
      break;
    }
    case DialogType::None:
    default:
      UNREACHABLE();
  }

  if (!td_->auth_manager_->is_bot()) {
    auto online_count_it = dialog_online_member_counts_.find(dialog_id);
    if (online_count_it != dialog_online_member_counts_.end()) {
      auto &info = online_count_it->second;
      CHECK(!info.is_update_sent);
      if (Time::now() - info.updated_time < ONLINE_MEMBER_COUNT_CACHE_EXPIRE_TIME) {
        info.is_update_sent = true;
        send_update_chat_online_member_count(dialog_id, info.online_member_count);
      }
    }

    if (d->has_scheduled_database_messages && !d->is_has_scheduled_database_messages_checked) {
      CHECK(G()->parameters().use_message_db);

      LOG(INFO) << "Send check has_scheduled_database_messages request";
      d->is_has_scheduled_database_messages_checked = true;
      G()->td_db()->get_messages_db_async()->get_scheduled_messages(
          dialog_id, 1,
          PromiseCreator::lambda([actor_id = actor_id(this), dialog_id](vector<MessagesDbDialogMessage> messages) {
            if (messages.empty()) {
              send_closure(actor_id, &MessagesManager::set_dialog_has_scheduled_database_messages, dialog_id, false);
            }
          }));
    }
  }
}

void MessagesManager::close_dialog(Dialog *d) {
  if (!d->is_opened) {
    return;
  }
  d->is_opened = false;

  auto dialog_id = d->dialog_id;
  if (have_input_peer(dialog_id, AccessRights::Write)) {
    if (pending_draft_message_timeout_.has_timeout(dialog_id.get())) {
      pending_draft_message_timeout_.set_timeout_in(dialog_id.get(), 0.0);
    }
  } else {
    pending_draft_message_timeout_.cancel_timeout(dialog_id.get());
  }

  if (have_input_peer(dialog_id, AccessRights::Read)) {
    if (pending_message_views_timeout_.has_timeout(dialog_id.get())) {
      pending_message_views_timeout_.set_timeout_in(dialog_id.get(), 0.0);
    }
    if (pending_read_history_timeout_.has_timeout(dialog_id.get())) {
      pending_read_history_timeout_.set_timeout_in(dialog_id.get(), 0.0);
    }
  } else {
    pending_message_views_timeout_.cancel_timeout(dialog_id.get());
    d->pending_viewed_message_ids.clear();
    d->increment_view_counter = false;

    pending_read_history_timeout_.cancel_timeout(dialog_id.get());
  }

  if (is_message_unload_enabled()) {
    CHECK(!d->has_unload_timeout);
    d->has_unload_timeout = true;
    pending_unload_dialog_timeout_.set_timeout_in(dialog_id.get(), get_unload_dialog_delay());
  }

  for (auto &it : d->pending_viewed_live_locations) {
    auto live_location_task_id = it.second;
    auto erased_count = viewed_live_location_tasks_.erase(live_location_task_id);
    CHECK(erased_count > 0);
  }
  d->pending_viewed_live_locations.clear();

  switch (dialog_id.get_type()) {
    case DialogType::User:
      break;
    case DialogType::Chat:
      break;
    case DialogType::Channel:
      channel_get_difference_timeout_.cancel_timeout(dialog_id.get());
      break;
    case DialogType::SecretChat:
      break;
    case DialogType::None:
    default:
      UNREACHABLE();
  }

  if (!td_->auth_manager_->is_bot()) {
    auto online_count_it = dialog_online_member_counts_.find(dialog_id);
    if (online_count_it != dialog_online_member_counts_.end()) {
      auto &info = online_count_it->second;
      info.is_update_sent = false;
    }
    update_dialog_online_member_count_timeout_.set_timeout_in(dialog_id.get(), ONLINE_MEMBER_COUNT_CACHE_EXPIRE_TIME);
  }
}

td_api::object_ptr<td_api::ChatType> MessagesManager::get_chat_type_object(DialogId dialog_id) const {
  switch (dialog_id.get_type()) {
    case DialogType::User:
      return td_api::make_object<td_api::chatTypePrivate>(
          td_->contacts_manager_->get_user_id_object(dialog_id.get_user_id(), "chatTypePrivate"));
    case DialogType::Chat:
      return td_api::make_object<td_api::chatTypeBasicGroup>(
          td_->contacts_manager_->get_basic_group_id_object(dialog_id.get_chat_id(), "chatTypeBasicGroup"));
    case DialogType::Channel: {
      auto channel_id = dialog_id.get_channel_id();
      auto channel_type = td_->contacts_manager_->get_channel_type(channel_id);
      return td_api::make_object<td_api::chatTypeSupergroup>(
          td_->contacts_manager_->get_supergroup_id_object(channel_id, "chatTypeSupergroup"),
          channel_type != ContactsManager::ChannelType::Megagroup);
    }
    case DialogType::SecretChat: {
      auto secret_chat_id = dialog_id.get_secret_chat_id();
      auto user_id = td_->contacts_manager_->get_secret_chat_user_id(secret_chat_id);
      return td_api::make_object<td_api::chatTypeSecret>(
          td_->contacts_manager_->get_secret_chat_id_object(secret_chat_id, "chatTypeSecret"),
          td_->contacts_manager_->get_user_id_object(user_id, "chatTypeSecret"));
    }
    case DialogType::None:
    default:
      UNREACHABLE();
      return nullptr;
  }
}

td_api::object_ptr<td_api::ChatActionBar> MessagesManager::get_chat_action_bar_object(const Dialog *d,
                                                                                      bool hide_unarchive) const {
  CHECK(d != nullptr);
  if (d->dialog_id.get_type() == DialogType::SecretChat) {
    auto user_id = td_->contacts_manager_->get_secret_chat_user_id(d->dialog_id.get_secret_chat_id());
    if (!user_id.is_valid()) {
      return nullptr;
    }
    const Dialog *user_d = get_dialog(DialogId(user_id));
    if (user_d == nullptr) {
      return nullptr;
    }
    return get_chat_action_bar_object(user_d, d->folder_id != FolderId::archive());
  }

  if (!d->know_action_bar) {
    if (d->know_can_report_spam && d->dialog_id.get_type() != DialogType::SecretChat && d->can_report_spam) {
      return td_api::make_object<td_api::chatActionBarReportSpam>(false);
    }
    return nullptr;
  }

  if (d->can_report_location) {
    CHECK(d->dialog_id.get_type() == DialogType::Channel);
    CHECK(!d->can_share_phone_number && !d->can_block_user && !d->can_add_contact && !d->can_report_spam &&
          !d->can_invite_members);
    return td_api::make_object<td_api::chatActionBarReportUnrelatedLocation>();
  }
  if (d->can_invite_members) {
    CHECK(!d->can_share_phone_number && !d->can_block_user && !d->can_add_contact && !d->can_report_spam);
    return td_api::make_object<td_api::chatActionBarInviteMembers>();
  }
  if (d->can_share_phone_number) {
    CHECK(d->dialog_id.get_type() == DialogType::User);
    CHECK(!d->can_block_user && !d->can_add_contact && !d->can_report_spam);
    return td_api::make_object<td_api::chatActionBarSharePhoneNumber>();
  }
  if (hide_unarchive) {
    if (d->can_add_contact) {
      return td_api::make_object<td_api::chatActionBarAddContact>();
    } else {
      return nullptr;
    }
  }
  if (d->can_block_user) {
    CHECK(d->dialog_id.get_type() == DialogType::User);
    CHECK(d->can_report_spam && d->can_add_contact);
    auto distance = d->hide_distance ? -1 : d->distance;
    return td_api::make_object<td_api::chatActionBarReportAddBlock>(d->can_unarchive, distance);
  }
  if (d->can_add_contact) {
    CHECK(d->dialog_id.get_type() == DialogType::User);
    CHECK(!d->can_report_spam);
    return td_api::make_object<td_api::chatActionBarAddContact>();
  }
  if (d->can_report_spam) {
    return td_api::make_object<td_api::chatActionBarReportSpam>(d->can_unarchive);
  }
  return nullptr;
}

string MessagesManager::get_dialog_theme_name(const Dialog *d) const {
  CHECK(d != nullptr);
  if (d->dialog_id.get_type() == DialogType::SecretChat) {
    auto user_id = td_->contacts_manager_->get_secret_chat_user_id(d->dialog_id.get_secret_chat_id());
    if (!user_id.is_valid()) {
      return string();
    }
    d = get_dialog(DialogId(user_id));
    if (d == nullptr) {
      return string();
    }
  }
  return d->theme_name;
}

td_api::object_ptr<td_api::chatJoinRequestsInfo> MessagesManager::get_chat_join_requests_info_object(
    const Dialog *d) const {
  if (d->pending_join_request_count == 0) {
    return nullptr;
  }
  return td_api::make_object<td_api::chatJoinRequestsInfo>(
      d->pending_join_request_count, td_->contacts_manager_->get_user_ids_object(d->pending_join_request_user_ids,
                                                                                 "get_chat_join_requests_info_object"));
}

td_api::object_ptr<td_api::videoChat> MessagesManager::get_video_chat_object(const Dialog *d) const {
  auto active_group_call_id = td_->group_call_manager_->get_group_call_id(d->active_group_call_id, d->dialog_id);
  auto default_participant_alias =
      d->default_join_group_call_as_dialog_id.is_valid()
          ? get_message_sender_object_const(d->default_join_group_call_as_dialog_id, "get_video_chat_object")
          : nullptr;
  return make_tl_object<td_api::videoChat>(active_group_call_id.get(),
                                           active_group_call_id.is_valid() ? !d->is_group_call_empty : false,
                                           std::move(default_participant_alias));
}

td_api::object_ptr<td_api::chat> MessagesManager::get_chat_object(const Dialog *d) const {
  CHECK(d != nullptr);

  auto chat_source = is_dialog_sponsored(d) ? sponsored_dialog_source_.get_chat_source_object() : nullptr;

  bool can_delete_for_self = false;
  bool can_delete_for_all_users = false;
  if (chat_source != nullptr) {
    switch (chat_source->get_id()) {
      case td_api::chatSourcePublicServiceAnnouncement::ID:
        // can delete for self (but only while removing from dialog list)
        can_delete_for_self = true;
        break;
      default:
        // can't delete
        break;
    }
  } else if (!td_->auth_manager_->is_bot() && have_input_peer(d->dialog_id, AccessRights::Read)) {
    switch (d->dialog_id.get_type()) {
      case DialogType::User:
        can_delete_for_self = true;
        can_delete_for_all_users = G()->shared_config().get_option_boolean("revoke_pm_inbox", true);
        if (d->dialog_id == get_my_dialog_id() || td_->contacts_manager_->is_user_deleted(d->dialog_id.get_user_id()) ||
            td_->contacts_manager_->is_user_bot(d->dialog_id.get_user_id())) {
          can_delete_for_all_users = false;
        }
        break;
      case DialogType::Chat:
        // chats can be deleted only for self with deleteChatHistory
        can_delete_for_self = true;
        break;
      case DialogType::Channel:
        if (is_broadcast_channel(d->dialog_id) ||
            td_->contacts_manager_->is_channel_public(d->dialog_id.get_channel_id())) {
          // deleteChatHistory can't be used in channels and public supergroups
        } else {
          // private supergroups can be deleted for self
          can_delete_for_self = true;
        }
        break;
      case DialogType::SecretChat:
        if (td_->contacts_manager_->get_secret_chat_state(d->dialog_id.get_secret_chat_id()) ==
            SecretChatState::Closed) {
          // in a closed secret chats there is no way to delete messages for both users
          can_delete_for_self = true;
        } else {
          // active secret chats can be deleted only for both users
          can_delete_for_all_users = true;
        }
        break;
      case DialogType::None:
      default:
        UNREACHABLE();
    }
  }

  // TODO hide/show draft message when can_send_message(dialog_id) changes
  auto draft_message = can_send_message(d->dialog_id).is_ok() ? get_draft_message_object(d->draft_message) : nullptr;

  return make_tl_object<td_api::chat>(
      d->dialog_id.get(), get_chat_type_object(d->dialog_id), get_dialog_title(d->dialog_id),
      get_chat_photo_info_object(td_->file_manager_.get(), get_dialog_photo(d->dialog_id)),
      get_dialog_default_permissions(d->dialog_id).get_chat_permissions_object(),
      get_message_object(d->dialog_id, get_message(d, d->last_message_id), "get_chat_object"),
      get_chat_positions_object(d), d->is_marked_as_unread, d->is_blocked, get_dialog_has_scheduled_messages(d),
      can_delete_for_self, can_delete_for_all_users, can_report_dialog(d->dialog_id),
      d->notification_settings.silent_send_message, d->server_unread_count + d->local_unread_count,
      d->last_read_inbox_message_id.get(), d->last_read_outbox_message_id.get(), d->unread_mention_count,
      get_chat_notification_settings_object(&d->notification_settings),
      d->message_ttl_setting.get_message_ttl_setting_object(), get_dialog_theme_name(d), get_chat_action_bar_object(d),
      get_video_chat_object(d), get_chat_join_requests_info_object(d), d->reply_markup_message_id.get(),
      std::move(draft_message), d->client_data);
}

tl_object_ptr<td_api::chat> MessagesManager::get_chat_object(DialogId dialog_id) const {
  return get_chat_object(get_dialog(dialog_id));
}

tl_object_ptr<td_api::chats> MessagesManager::get_chats_object(int32 total_count, const vector<DialogId> &dialog_ids) {
  if (total_count == -1) {
    total_count = narrow_cast<int32>(dialog_ids.size());
  }
  return td_api::make_object<td_api::chats>(total_count,
                                            transform(dialog_ids, [](DialogId dialog_id) { return dialog_id.get(); }));
}

tl_object_ptr<td_api::chats> MessagesManager::get_chats_object(const std::pair<int32, vector<DialogId>> &dialog_ids) {
  return get_chats_object(dialog_ids.first, dialog_ids.second);
}

td_api::object_ptr<td_api::chatFilter> MessagesManager::get_chat_filter_object(DialogFilterId dialog_filter_id) const {
  CHECK(!td_->auth_manager_->is_bot());
  auto filter = get_dialog_filter(dialog_filter_id);
  if (filter == nullptr) {
    return nullptr;
  }

  return get_chat_filter_object(filter);
}

td_api::object_ptr<td_api::chatFilter> MessagesManager::get_chat_filter_object(const DialogFilter *filter) const {
  auto get_chat_ids = [this,
                       dialog_filter_id = filter->dialog_filter_id](const vector<InputDialogId> &input_dialog_ids) {
    vector<int64> chat_ids;
    chat_ids.reserve(input_dialog_ids.size());
    for (auto &input_dialog_id : input_dialog_ids) {
      auto dialog_id = input_dialog_id.get_dialog_id();
      const Dialog *d = get_dialog(dialog_id);
      if (d != nullptr) {
        if (d->order != DEFAULT_ORDER) {
          chat_ids.push_back(dialog_id.get());
        } else {
          LOG(INFO) << "Skip nonjoined " << dialog_id << " from " << dialog_filter_id;
        }
      } else {
        LOG(ERROR) << "Can't find " << dialog_id << " from " << dialog_filter_id;
      }
    }
    return chat_ids;
  };
  return td_api::make_object<td_api::chatFilter>(
      filter->title, filter->get_icon_name(), get_chat_ids(filter->pinned_dialog_ids),
      get_chat_ids(filter->included_dialog_ids), get_chat_ids(filter->excluded_dialog_ids), filter->exclude_muted,
      filter->exclude_read, filter->exclude_archived, filter->include_contacts, filter->include_non_contacts,
      filter->include_bots, filter->include_groups, filter->include_channels);
}

td_api::object_ptr<td_api::updateScopeNotificationSettings>
MessagesManager::get_update_scope_notification_settings_object(NotificationSettingsScope scope) const {
  auto notification_settings = get_scope_notification_settings(scope);
  CHECK(notification_settings != nullptr);
  return td_api::make_object<td_api::updateScopeNotificationSettings>(
      get_notification_settings_scope_object(scope), get_scope_notification_settings_object(notification_settings));
}

std::pair<bool, int32> MessagesManager::get_dialog_mute_until(DialogId dialog_id, const Dialog *d) const {
  CHECK(!td_->auth_manager_->is_bot());
  if (d == nullptr || !d->notification_settings.is_synchronized) {
    return {false, get_scope_mute_until(dialog_id)};
  }

  return {d->notification_settings.is_use_default_fixed, get_dialog_mute_until(d)};
}

NotificationSettingsScope MessagesManager::get_dialog_notification_setting_scope(DialogId dialog_id) const {
  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::SecretChat:
      return NotificationSettingsScope::Private;
    case DialogType::Chat:
      return NotificationSettingsScope::Group;
    case DialogType::Channel:
      return is_broadcast_channel(dialog_id) ? NotificationSettingsScope::Channel : NotificationSettingsScope::Group;
    case DialogType::None:
    default:
      UNREACHABLE();
      return NotificationSettingsScope::Private;
  }
}

int32 MessagesManager::get_scope_mute_until(DialogId dialog_id) const {
  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::SecretChat:
      return users_notification_settings_.mute_until;
    case DialogType::Chat:
      return chats_notification_settings_.mute_until;
    case DialogType::Channel:
      return is_broadcast_channel(dialog_id) ? channels_notification_settings_.mute_until
                                             : chats_notification_settings_.mute_until;
    case DialogType::None:
    default:
      UNREACHABLE();
      return 0;
  }
}

vector<DialogId> MessagesManager::get_dialog_notification_settings_exceptions(NotificationSettingsScope scope,
                                                                              bool filter_scope, bool compare_sound,
                                                                              bool force, Promise<Unit> &&promise) {
  CHECK(!td_->auth_manager_->is_bot());
  bool have_all_dialogs = true;
  for (const auto &list : dialog_folders_) {
    if (list.second.folder_last_dialog_date_ != MAX_DIALOG_DATE) {
      have_all_dialogs = false;
    }
  }

  if (have_all_dialogs || force) {
    vector<DialogDate> ordered_dialogs;
    auto my_dialog_id = get_my_dialog_id();
    for (const auto &list : dialog_folders_) {
      for (const auto &dialog_date : list.second.ordered_dialogs_) {
        auto dialog_id = dialog_date.get_dialog_id();
        if (filter_scope && get_dialog_notification_setting_scope(dialog_id) != scope) {
          continue;
        }
        if (dialog_id == my_dialog_id) {
          continue;
        }

        const Dialog *d = get_dialog(dialog_id);
        CHECK(d != nullptr);
        LOG_CHECK(d->folder_id == list.first)
            << list.first << ' ' << dialog_id << ' ' << d->folder_id << ' ' << d->order;
        if (d->order == DEFAULT_ORDER) {
          break;
        }
        if (are_default_dialog_notification_settings(d->notification_settings, compare_sound)) {
          continue;
        }
        if (is_dialog_message_notification_disabled(dialog_id, std::numeric_limits<int32>::max())) {
          continue;
        }
        ordered_dialogs.push_back(DialogDate(get_dialog_base_order(d), dialog_id));
      }
    }
    std::sort(ordered_dialogs.begin(), ordered_dialogs.end());

    vector<DialogId> result;
    for (auto &dialog_date : ordered_dialogs) {
      CHECK(result.empty() || result.back() != dialog_date.get_dialog_id());
      result.push_back(dialog_date.get_dialog_id());
    }
    promise.set_value(Unit());
    return result;
  }

  for (const auto &folder : dialog_folders_) {
    load_folder_dialog_list(folder.first, MAX_GET_DIALOGS, true);
  }

  td_->create_handler<GetNotifySettingsExceptionsQuery>(std::move(promise))->send(scope, filter_scope, compare_sound);
  return {};
}

const ScopeNotificationSettings *MessagesManager::get_scope_notification_settings(NotificationSettingsScope scope,
                                                                                  Promise<Unit> &&promise) {
  const ScopeNotificationSettings *notification_settings = get_scope_notification_settings(scope);
  CHECK(notification_settings != nullptr);
  if (!notification_settings->is_synchronized && !td_->auth_manager_->is_bot()) {
    send_get_scope_notification_settings_query(scope, std::move(promise));
    return nullptr;
  }

  promise.set_value(Unit());
  return notification_settings;
}

DialogNotificationSettings *MessagesManager::get_dialog_notification_settings(DialogId dialog_id, bool force) {
  Dialog *d = get_dialog_force(dialog_id, "get_dialog_notification_settings");
  if (d == nullptr) {
    return nullptr;
  }
  if (!force && !have_input_peer(dialog_id, AccessRights::Read)) {
    return nullptr;
  }
  return &d->notification_settings;
}

ScopeNotificationSettings *MessagesManager::get_scope_notification_settings(NotificationSettingsScope scope) {
  switch (scope) {
    case NotificationSettingsScope::Private:
      return &users_notification_settings_;
    case NotificationSettingsScope::Group:
      return &chats_notification_settings_;
    case NotificationSettingsScope::Channel:
      return &channels_notification_settings_;
    default:
      UNREACHABLE();
      return nullptr;
  }
}

const ScopeNotificationSettings *MessagesManager::get_scope_notification_settings(
    NotificationSettingsScope scope) const {
  switch (scope) {
    case NotificationSettingsScope::Private:
      return &users_notification_settings_;
    case NotificationSettingsScope::Group:
      return &chats_notification_settings_;
    case NotificationSettingsScope::Channel:
      return &channels_notification_settings_;
    default:
      UNREACHABLE();
      return nullptr;
  }
}

tl_object_ptr<telegram_api::InputNotifyPeer> MessagesManager::get_input_notify_peer(DialogId dialog_id) const {
  if (get_dialog(dialog_id) == nullptr) {
    return nullptr;
  }
  auto input_peer = get_input_peer(dialog_id, AccessRights::Read);
  if (input_peer == nullptr) {
    return nullptr;
  }
  return make_tl_object<telegram_api::inputNotifyPeer>(std::move(input_peer));
}

Status MessagesManager::set_dialog_notification_settings(
    DialogId dialog_id, tl_object_ptr<td_api::chatNotificationSettings> &&notification_settings) {
  CHECK(!td_->auth_manager_->is_bot());
  auto current_settings = get_dialog_notification_settings(dialog_id, false);
  if (current_settings == nullptr) {
    return Status::Error(400, "Wrong chat identifier specified");
  }
  if (dialog_id == get_my_dialog_id()) {
    return Status::Error(400, "Notification settings of the Saved Messages chat can't be changed");
  }

  TRY_RESULT(new_settings, ::td::get_dialog_notification_settings(std::move(notification_settings),
                                                                  current_settings->silent_send_message));
  if (update_dialog_notification_settings(dialog_id, current_settings, new_settings)) {
    update_dialog_notification_settings_on_server(dialog_id, false);
  }
  return Status::OK();
}

Status MessagesManager::set_scope_notification_settings(
    NotificationSettingsScope scope, tl_object_ptr<td_api::scopeNotificationSettings> &&notification_settings) {
  CHECK(!td_->auth_manager_->is_bot());
  TRY_RESULT(new_settings, ::td::get_scope_notification_settings(std::move(notification_settings)));
  if (update_scope_notification_settings(scope, get_scope_notification_settings(scope), new_settings)) {
    update_scope_notification_settings_on_server(scope, 0);
  }
  return Status::OK();
}

class MessagesManager::UpdateScopeNotificationSettingsOnServerLogEvent {
 public:
  NotificationSettingsScope scope_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(scope_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(scope_, parser);
  }
};

uint64 MessagesManager::save_update_scope_notification_settings_on_server_log_event(NotificationSettingsScope scope) {
  UpdateScopeNotificationSettingsOnServerLogEvent log_event{scope};
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::UpdateScopeNotificationSettingsOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::update_scope_notification_settings_on_server(NotificationSettingsScope scope,
                                                                   uint64 log_event_id) {
  CHECK(!td_->auth_manager_->is_bot());
  if (log_event_id == 0) {
    log_event_id = save_update_scope_notification_settings_on_server_log_event(scope);
  }

  LOG(INFO) << "Update " << scope << " notification settings on server with log_event " << log_event_id;
  td_->create_handler<UpdateScopeNotifySettingsQuery>(get_erase_log_event_promise(log_event_id))
      ->send(scope, *get_scope_notification_settings(scope));
}

void MessagesManager::reset_all_notification_settings() {
  CHECK(!td_->auth_manager_->is_bot());
  DialogNotificationSettings new_dialog_settings;
  ScopeNotificationSettings new_scope_settings;
  new_dialog_settings.is_synchronized = true;
  new_scope_settings.is_synchronized = true;

  update_scope_notification_settings(NotificationSettingsScope::Private, &users_notification_settings_,
                                     new_scope_settings);
  update_scope_notification_settings(NotificationSettingsScope::Group, &chats_notification_settings_,
                                     new_scope_settings);
  update_scope_notification_settings(NotificationSettingsScope::Channel, &channels_notification_settings_,
                                     new_scope_settings);

  for (auto &dialog : dialogs_) {
    Dialog *d = dialog.second.get();
    update_dialog_notification_settings(d->dialog_id, &d->notification_settings, new_dialog_settings);
  }
  reset_all_notification_settings_on_server(0);
}

class MessagesManager::ResetAllNotificationSettingsOnServerLogEvent {
 public:
  template <class StorerT>
  void store(StorerT &storer) const {
  }

  template <class ParserT>
  void parse(ParserT &parser) {
  }
};

uint64 MessagesManager::save_reset_all_notification_settings_on_server_log_event() {
  ResetAllNotificationSettingsOnServerLogEvent log_event;
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ResetAllNotificationSettingsOnServer,
                    get_log_event_storer(log_event));
}

void MessagesManager::reset_all_notification_settings_on_server(uint64 log_event_id) {
  CHECK(!td_->auth_manager_->is_bot());
  if (log_event_id == 0) {
    log_event_id = save_reset_all_notification_settings_on_server_log_event();
  }

  LOG(INFO) << "Reset all notification settings";
  td_->create_handler<ResetNotifySettingsQuery>(get_erase_log_event_promise(log_event_id))->send();
}

tl_object_ptr<td_api::messages> MessagesManager::get_dialog_history(DialogId dialog_id, MessageId from_message_id,
                                                                    int32 offset, int32 limit, int left_tries,
                                                                    bool only_local, Promise<Unit> &&promise) {
  if (limit <= 0) {
    promise.set_error(Status::Error(400, "Parameter limit must be positive"));
    return nullptr;
  }
  if (limit > MAX_GET_HISTORY) {
    limit = MAX_GET_HISTORY;
  }
  if (offset > 0) {
    promise.set_error(Status::Error(400, "Parameter offset must be non-positive"));
    return nullptr;
  }
  if (offset <= -MAX_GET_HISTORY) {
    promise.set_error(Status::Error(400, "Parameter offset must be greater than -100"));
    return nullptr;
  }
  if (offset < -limit) {
    promise.set_error(Status::Error(400, "Parameter offset must be greater than or equal to -limit"));
    return nullptr;
  }
  bool is_limit_increased = false;
  if (limit == -offset) {
    limit++;
    is_limit_increased = true;
  }
  CHECK(0 < limit && limit <= MAX_GET_HISTORY);
  CHECK(-limit < offset && offset <= 0);

  if (from_message_id == MessageId() || from_message_id.get() > MessageId::max().get()) {
    from_message_id = MessageId::max();
  }
  if (!from_message_id.is_valid()) {
    promise.set_error(Status::Error(400, "Invalid value of parameter from_message_id specified"));
    return nullptr;
  }

  const Dialog *d = get_dialog_force(dialog_id, "get_dialog_history");
  if (d == nullptr) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return nullptr;
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    promise.set_error(Status::Error(400, "Can't access the chat"));
    return nullptr;
  }

  LOG(INFO) << "Get " << (only_local ? "local " : "") << "history in " << dialog_id << " from " << from_message_id
            << " with offset " << offset << " and limit " << limit << ", " << left_tries
            << " tries left. Last read inbox message is " << d->last_read_inbox_message_id
            << ", last read outbox message is " << d->last_read_outbox_message_id
            << ", have_full_history = " << d->have_full_history;

  MessagesConstIterator p(d, from_message_id);
  LOG(DEBUG) << "Iterator points to " << (*p ? (*p)->message_id : MessageId());
  bool from_the_end = (d->last_message_id != MessageId() && from_message_id > d->last_message_id) ||
                      from_message_id >= MessageId::max();

  if (from_the_end) {
    limit += offset;
    offset = 0;
    if (d->last_message_id == MessageId()) {
      p = MessagesConstIterator();
    }
  } else {
    bool have_a_gap = false;
    if (*p == nullptr) {
      // there is no gap if from_message_id is less than first message in the dialog
      if (left_tries == 0 && d->messages != nullptr && offset < 0) {
        const Message *cur = d->messages.get();
        while (cur->left != nullptr) {
          cur = cur->left.get();
        }
        CHECK(cur->message_id > from_message_id);
        from_message_id = cur->message_id;
        p = MessagesConstIterator(d, from_message_id);
      } else {
        have_a_gap = true;
      }
    } else if ((*p)->message_id != from_message_id) {
      CHECK((*p)->message_id < from_message_id);
      if (!(*p)->have_next && (d->last_message_id == MessageId() || (*p)->message_id < d->last_message_id)) {
        have_a_gap = true;
      }
    }

    if (have_a_gap) {
      LOG(INFO) << "Have a gap near message to get chat history from";
      p = MessagesConstIterator();
    }
    if (*p != nullptr && (*p)->message_id == from_message_id) {
      if (offset < 0) {
        offset++;
      } else {
        --p;
      }
    }

    while (*p != nullptr && offset < 0) {
      ++p;
      if (*p) {
        ++offset;
        from_message_id = (*p)->message_id;
      }
    }

    if (offset < 0 && ((d->last_message_id != MessageId() && from_message_id >= d->last_message_id) ||
                       (!have_a_gap && left_tries == 0))) {
      CHECK(!have_a_gap);
      limit += offset;
      offset = 0;
      p = MessagesConstIterator(d, from_message_id);
    }

    if (!have_a_gap && offset < 0) {
      offset--;
    }
  }

  LOG(INFO) << "Iterator after applying offset points to " << (*p ? (*p)->message_id : MessageId())
            << ", offset = " << offset << ", limit = " << limit << ", from_the_end = " << from_the_end;
  vector<tl_object_ptr<td_api::message>> messages;
  if (*p != nullptr && offset == 0) {
    while (*p != nullptr && messages.size() < static_cast<size_t>(limit)) {
      messages.push_back(get_message_object(dialog_id, *p, "get_dialog_history"));
      from_message_id = (*p)->message_id;
      from_the_end = false;
      --p;
    }
  }

  if (!messages.empty()) {
    // maybe need some messages
    CHECK(offset == 0);
    preload_newer_messages(d, MessageId(messages[0]->id_));
    preload_older_messages(d, MessageId(messages.back()->id_));
  } else if (messages.size() < static_cast<size_t>(limit) && left_tries != 0 &&
             !(d->is_empty && d->have_full_history && left_tries < 3)) {
    // there can be more messages in the database or on the server, need to load them
    if (from_the_end) {
      from_message_id = MessageId();
    }
    send_closure_later(actor_id(this), &MessagesManager::load_messages, dialog_id, from_message_id, offset,
                       limit - static_cast<int32>(messages.size()), left_tries, only_local, std::move(promise));
    return nullptr;
  }

  LOG(INFO) << "Have " << messages.size() << " messages out of requested "
            << (is_limit_increased ? "increased " : "exact ") << limit;
  if (is_limit_increased && static_cast<size_t>(limit) == messages.size()) {
    messages.pop_back();
  }

  LOG(INFO) << "Return " << messages.size() << " messages in result to getChatHistory";
  promise.set_value(Unit());                                   // can return some messages
  return get_messages_object(-1, std::move(messages), false);  // TODO return real total_count of messages in the dialog
}

class MessagesManager::ReadHistoryOnServerLogEvent {
 public:
  DialogId dialog_id_;
  MessageId max_message_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
    td::store(max_message_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
    td::parse(max_message_id_, parser);
  }
};

class MessagesManager::ReadHistoryInSecretChatLogEvent {
 public:
  DialogId dialog_id_;
  int32 max_date_ = 0;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
    td::store(max_date_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
    td::parse(max_date_, parser);
  }
};

class MessagesManager::ReadMessageThreadHistoryOnServerLogEvent {
 public:
  DialogId dialog_id_;
  MessageId top_thread_message_id_;
  MessageId max_message_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
    td::store(top_thread_message_id_, storer);
    td::store(max_message_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
    td::parse(top_thread_message_id_, parser);
    td::parse(max_message_id_, parser);
  }
};

void MessagesManager::read_history_on_server(Dialog *d, MessageId max_message_id) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(d != nullptr);
  CHECK(!max_message_id.is_scheduled());

  auto dialog_id = d->dialog_id;
  LOG(INFO) << "Read history in " << dialog_id << " on server up to " << max_message_id;

  bool is_secret = dialog_id.get_type() == DialogType::SecretChat;
  if (is_secret) {
    auto *m = get_message_force(d, max_message_id, "read_history_on_server");
    if (m == nullptr) {
      LOG(ERROR) << "Failed to read history in " << dialog_id << " up to " << max_message_id;
      return;
    }

    ReadHistoryInSecretChatLogEvent log_event;
    log_event.dialog_id_ = dialog_id;
    log_event.max_date_ = m->date;
    add_log_event(d->read_history_log_event_ids[0], get_log_event_storer(log_event),
                  LogEvent::HandlerType::ReadHistoryInSecretChat, "read history");

    d->last_read_inbox_message_date = m->date;
  } else if (G()->parameters().use_message_db) {
    ReadHistoryOnServerLogEvent log_event;
    log_event.dialog_id_ = dialog_id;
    log_event.max_message_id_ = max_message_id;
    add_log_event(d->read_history_log_event_ids[0], get_log_event_storer(log_event),
                  LogEvent::HandlerType::ReadHistoryOnServer, "read history");
  }

  d->updated_read_history_message_ids.insert(MessageId());

  bool need_delay = d->is_opened && !is_secret &&
                    (d->server_unread_count > 0 || (!need_unread_counter(d->order) && d->last_message_id.is_valid() &&
                                                    max_message_id < d->last_message_id));
  pending_read_history_timeout_.set_timeout_in(dialog_id.get(), need_delay ? MIN_READ_HISTORY_DELAY : 0);
}

void MessagesManager::read_message_thread_history_on_server(Dialog *d, MessageId top_thread_message_id,
                                                            MessageId max_message_id, MessageId last_message_id) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(d != nullptr);
  CHECK(top_thread_message_id.is_valid());
  CHECK(top_thread_message_id.is_server());
  CHECK(max_message_id.is_server());

  auto dialog_id = d->dialog_id;
  LOG(INFO) << "Read history in thread of " << top_thread_message_id << " in " << dialog_id << " on server up to "
            << max_message_id;

  if (G()->parameters().use_message_db) {
    ReadMessageThreadHistoryOnServerLogEvent log_event;
    log_event.dialog_id_ = dialog_id;
    log_event.top_thread_message_id_ = top_thread_message_id;
    log_event.max_message_id_ = max_message_id;
    add_log_event(d->read_history_log_event_ids[top_thread_message_id.get()], get_log_event_storer(log_event),
                  LogEvent::HandlerType::ReadMessageThreadHistoryOnServer, "read history");
  }

  d->updated_read_history_message_ids.insert(top_thread_message_id);

  bool need_delay = d->is_opened && last_message_id.is_valid() && max_message_id < last_message_id;
  pending_read_history_timeout_.set_timeout_in(dialog_id.get(), need_delay ? MIN_READ_HISTORY_DELAY : 0);
}

void MessagesManager::do_read_history_on_server(DialogId dialog_id) {
  if (G()->close_flag()) {
    return;
  }

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  for (auto top_thread_message_id : d->updated_read_history_message_ids) {
    if (!top_thread_message_id.is_valid()) {
      read_history_on_server_impl(d, MessageId());
    } else {
      read_message_thread_history_on_server_impl(d, top_thread_message_id, MessageId());
    }
  }
  reset_to_empty(d->updated_read_history_message_ids);
}

void MessagesManager::read_history_on_server_impl(Dialog *d, MessageId max_message_id) {
  CHECK(d != nullptr);
  auto dialog_id = d->dialog_id;

  {
    auto message_id = d->last_read_inbox_message_id;
    if (dialog_id.get_type() != DialogType::SecretChat) {
      message_id = message_id.get_prev_server_message_id();
    }
    if (message_id > max_message_id) {
      max_message_id = message_id;
    }
  }

  Promise<> promise;
  if (d->read_history_log_event_ids[0].log_event_id != 0) {
    d->read_history_log_event_ids[0].generation++;
    promise = PromiseCreator::lambda([actor_id = actor_id(this), dialog_id,
                                      generation = d->read_history_log_event_ids[0].generation](Result<Unit> result) {
      if (!G()->close_flag()) {
        send_closure(actor_id, &MessagesManager::on_read_history_finished, dialog_id, MessageId(), generation);
      }
    });
  }
  if (d->need_repair_server_unread_count && need_unread_counter(d->order)) {
    repair_server_unread_count(dialog_id, d->server_unread_count);
  }

  if (!max_message_id.is_valid() || !have_input_peer(dialog_id, AccessRights::Read)) {
    return promise.set_value(Unit());
  }

  LOG(INFO) << "Send read history request in " << dialog_id << " up to " << max_message_id;
  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
      td_->create_handler<ReadHistoryQuery>(std::move(promise))->send(dialog_id, max_message_id);
      break;
    case DialogType::Channel: {
      auto channel_id = dialog_id.get_channel_id();
      td_->create_handler<ReadChannelHistoryQuery>(std::move(promise))->send(channel_id, max_message_id);
      break;
    }
    case DialogType::SecretChat: {
      auto secret_chat_id = dialog_id.get_secret_chat_id();
      auto date = d->last_read_inbox_message_date;
      auto *m = get_message_force(d, max_message_id, "read_history_on_server_impl");
      if (m != nullptr && m->date > date) {
        date = m->date;
      }
      if (date == 0) {
        LOG(ERROR) << "Don't know last read inbox message date in " << dialog_id;
        return promise.set_value(Unit());
      }
      send_closure(G()->secret_chats_manager(), &SecretChatsManager::send_read_history, secret_chat_id, date,
                   std::move(promise));
      break;
    }
    case DialogType::None:
    default:
      UNREACHABLE();
  }
}

void MessagesManager::read_message_thread_history_on_server_impl(Dialog *d, MessageId top_thread_message_id,
                                                                 MessageId max_message_id) {
  CHECK(d != nullptr);
  auto dialog_id = d->dialog_id;
  CHECK(dialog_id.get_type() == DialogType::Channel);

  const Message *m = get_message_force(d, top_thread_message_id, "read_message_thread_history_on_server_impl");
  if (m != nullptr) {
    auto message_id = m->reply_info.last_read_inbox_message_id.get_prev_server_message_id();
    if (message_id > max_message_id) {
      max_message_id = message_id;
    }
  }

  Promise<> promise;
  if (d->read_history_log_event_ids[top_thread_message_id.get()].log_event_id != 0) {
    d->read_history_log_event_ids[top_thread_message_id.get()].generation++;
    promise = PromiseCreator::lambda(
        [actor_id = actor_id(this), dialog_id, top_thread_message_id,
         generation = d->read_history_log_event_ids[top_thread_message_id.get()].generation](Result<Unit> result) {
          if (!G()->close_flag()) {
            send_closure(actor_id, &MessagesManager::on_read_history_finished, dialog_id, top_thread_message_id,
                         generation);
          }
        });
  }

  if (!max_message_id.is_valid() || !have_input_peer(dialog_id, AccessRights::Read)) {
    return promise.set_value(Unit());
  }

  LOG(INFO) << "Send read history request in thread of " << top_thread_message_id << " in " << dialog_id << " up to "
            << max_message_id;
  td_->create_handler<ReadDiscussionQuery>(std::move(promise))->send(dialog_id, top_thread_message_id, max_message_id);
}

void MessagesManager::on_read_history_finished(DialogId dialog_id, MessageId top_thread_message_id, uint64 generation) {
  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  auto it = d->read_history_log_event_ids.find(top_thread_message_id.get());
  if (it == d->read_history_log_event_ids.end()) {
    return;
  }
  delete_log_event(it->second, generation, "read history");
  if (it->second.log_event_id == 0) {
    d->read_history_log_event_ids.erase(it);
  }
}

template <class T, class It>
vector<MessageId> MessagesManager::get_message_history_slice(const T &begin, It it, const T &end,
                                                             MessageId from_message_id, int32 offset, int32 limit) {
  int32 left_offset = -offset;
  int32 left_limit = limit + offset;
  while (left_offset > 0 && it != end) {
    ++it;
    left_offset--;
    left_limit++;
  }

  vector<MessageId> message_ids;
  while (left_limit > 0 && it != begin) {
    --it;
    left_limit--;
    message_ids.push_back(*it);
  }
  return message_ids;
}

std::pair<DialogId, vector<MessageId>> MessagesManager::get_message_thread_history(
    DialogId dialog_id, MessageId message_id, MessageId from_message_id, int32 offset, int32 limit, int64 &random_id,
    Promise<Unit> &&promise) {
  if (limit <= 0) {
    promise.set_error(Status::Error(400, "Parameter limit must be positive"));
    return {};
  }
  if (limit > MAX_GET_HISTORY) {
    limit = MAX_GET_HISTORY;
  }
  if (offset > 0) {
    promise.set_error(Status::Error(400, "Parameter offset must be non-positive"));
    return {};
  }
  if (offset <= -MAX_GET_HISTORY) {
    promise.set_error(Status::Error(400, "Parameter offset must be greater than -100"));
    return {};
  }
  if (offset < -limit) {
    promise.set_error(Status::Error(400, "Parameter offset must be greater than or equal to -limit"));
    return {};
  }
  bool is_limit_increased = false;
  if (limit == -offset) {
    limit++;
    is_limit_increased = true;
  }
  CHECK(0 < limit && limit <= MAX_GET_HISTORY);
  CHECK(-limit < offset && offset <= 0);

  Dialog *d = get_dialog_force(dialog_id, "get_message_thread_history");
  if (d == nullptr) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return {};
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    promise.set_error(Status::Error(400, "Can't access the chat"));
    return {};
  }
  if (dialog_id.get_type() != DialogType::Channel) {
    promise.set_error(Status::Error(400, "Can't get message thread history in the chat"));
    return {};
  }

  if (from_message_id == MessageId() || from_message_id.get() > MessageId::max().get()) {
    from_message_id = MessageId::max();
  }
  if (!from_message_id.is_valid()) {
    promise.set_error(Status::Error(400, "Parameter from_message_id must be identifier of a chat message or 0"));
    return {};
  }

  FullMessageId top_thread_full_message_id;
  {
    Message *m = get_message_force(d, message_id, "get_message_thread_history 1");
    if (m == nullptr) {
      promise.set_error(Status::Error(400, "Message not found"));
      return {};
    }

    auto r_top_thread_full_message_id = get_top_thread_full_message_id(dialog_id, m);
    if (r_top_thread_full_message_id.is_error()) {
      promise.set_error(r_top_thread_full_message_id.move_as_error());
      return {};
    }
    top_thread_full_message_id = r_top_thread_full_message_id.move_as_ok();

    if (!top_thread_full_message_id.get_message_id().is_valid()) {
      CHECK(m->reply_info.is_comment);
      get_message_thread(
          dialog_id, message_id,
          PromiseCreator::lambda([promise = std::move(promise)](Result<MessageThreadInfo> &&result) mutable {
            if (result.is_error()) {
              promise.set_error(result.move_as_error());
            } else {
              promise.set_value(Unit());
            }
          }));
      return {};
    }
  }

  if (random_id != 0) {
    // request has already been sent before
    auto it = found_dialog_messages_.find(random_id);
    CHECK(it != found_dialog_messages_.end());
    auto result = std::move(it->second.second);
    found_dialog_messages_.erase(it);

    auto dialog_id_it = found_dialog_messages_dialog_id_.find(random_id);
    if (dialog_id_it != found_dialog_messages_dialog_id_.end()) {
      dialog_id = dialog_id_it->second;
      found_dialog_messages_dialog_id_.erase(dialog_id_it);

      d = get_dialog(dialog_id);
      CHECK(d != nullptr);
    }
    if (dialog_id != top_thread_full_message_id.get_dialog_id()) {
      promise.set_error(Status::Error(500, "Receive messages in an unexpected chat"));
      return {};
    }

    auto yet_unsent_it = d->yet_unsent_thread_message_ids.find(top_thread_full_message_id.get_message_id());
    if (yet_unsent_it != d->yet_unsent_thread_message_ids.end()) {
      const std::set<MessageId> &message_ids = yet_unsent_it->second;
      auto merge_message_ids = get_message_history_slice(message_ids.begin(), message_ids.lower_bound(from_message_id),
                                                         message_ids.end(), from_message_id, offset, limit);
      vector<MessageId> new_result(result.size() + merge_message_ids.size());
      std::merge(result.begin(), result.end(), merge_message_ids.begin(), merge_message_ids.end(), new_result.begin(),
                 std::greater<>());
      result = std::move(new_result);
    }

    Message *top_m = get_message_force(d, top_thread_full_message_id.get_message_id(), "get_message_thread_history 2");
    if (top_m != nullptr && !top_m->local_thread_message_ids.empty()) {
      vector<MessageId> &message_ids = top_m->local_thread_message_ids;
      vector<MessageId> merge_message_ids;
      while (true) {
        merge_message_ids = get_message_history_slice(
            message_ids.begin(), std::lower_bound(message_ids.begin(), message_ids.end(), from_message_id),
            message_ids.end(), from_message_id, offset, limit);
        bool found_deleted = false;
        for (auto local_message_id : merge_message_ids) {
          Message *local_m = get_message_force(d, local_message_id, "get_message_thread_history 3");
          if (local_m == nullptr) {
            auto local_it = std::lower_bound(message_ids.begin(), message_ids.end(), local_message_id);
            CHECK(local_it != message_ids.end() && *local_it == local_message_id);
            message_ids.erase(local_it);
            found_deleted = true;
          }
        }
        if (!found_deleted) {
          break;
        }
        on_message_changed(d, top_m, false, "get_message_thread_history");
      }
      vector<MessageId> new_result(result.size() + merge_message_ids.size());
      std::merge(result.begin(), result.end(), merge_message_ids.begin(), merge_message_ids.end(), new_result.begin(),
                 std::greater<>());
      result = std::move(new_result);
    }

    if (is_limit_increased) {
      limit--;
    }

    std::reverse(result.begin(), result.end());
    result = get_message_history_slice(result.begin(), std::lower_bound(result.begin(), result.end(), from_message_id),
                                       result.end(), from_message_id, offset, limit);

    LOG(INFO) << "Return " << result.size() << " messages in result to getMessageThreadHistory";

    promise.set_value(Unit());
    return {dialog_id, std::move(result)};
  }

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || found_dialog_messages_.find(random_id) != found_dialog_messages_.end());
  found_dialog_messages_[random_id];  // reserve place for result

  td_->create_handler<SearchMessagesQuery>(std::move(promise))
      ->send(dialog_id, string(), DialogId(), from_message_id.get_next_server_message_id(), offset, limit,
             MessageSearchFilter::Empty, message_id, random_id);
  return {};
}

td_api::object_ptr<td_api::messageCalendar> MessagesManager::get_dialog_message_calendar(DialogId dialog_id,
                                                                                         MessageId from_message_id,
                                                                                         MessageSearchFilter filter,
                                                                                         int64 &random_id, bool use_db,
                                                                                         Promise<Unit> &&promise) {
  if (random_id != 0) {
    // request has already been sent before
    auto it = found_dialog_message_calendars_.find(random_id);
    if (it != found_dialog_message_calendars_.end()) {
      auto result = std::move(it->second);
      found_dialog_message_calendars_.erase(it);
      promise.set_value(Unit());
      return result;
    }
    random_id = 0;
  }
  LOG(INFO) << "Get message calendar in " << dialog_id << " filtered by " << filter << " from " << from_message_id;

  if (from_message_id.get() > MessageId::max().get()) {
    from_message_id = MessageId::max();
  }

  if (!from_message_id.is_valid() && from_message_id != MessageId()) {
    promise.set_error(Status::Error(400, "Parameter from_message_id must be identifier of a chat message or 0"));
    return {};
  }
  from_message_id = from_message_id.get_next_server_message_id();

  const Dialog *d = get_dialog_force(dialog_id, "get_dialog_message_calendar");
  if (d == nullptr) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return {};
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    promise.set_error(Status::Error(400, "Can't access the chat"));
    return {};
  }

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || found_dialog_message_calendars_.find(random_id) != found_dialog_message_calendars_.end());
  found_dialog_message_calendars_[random_id];  // reserve place for result

  if (filter == MessageSearchFilter::Empty || filter == MessageSearchFilter::Call ||
      filter == MessageSearchFilter::MissedCall || filter == MessageSearchFilter::Mention ||
      filter == MessageSearchFilter::UnreadMention) {
    promise.set_error(Status::Error(400, "The filter is not supported"));
    return {};
  }

  // Trying to use database
  if (use_db && G()->parameters().use_message_db) {
    MessageId first_db_message_id = get_first_database_message_id_by_index(d, filter);
    int32 message_count = d->message_count_by_index[message_search_filter_index(filter)];
    auto fixed_from_message_id = from_message_id;
    if (fixed_from_message_id == MessageId()) {
      fixed_from_message_id = MessageId::max();
    }
    LOG(INFO) << "Get message calendar in " << dialog_id << " from " << fixed_from_message_id << ", have up to "
              << first_db_message_id << ", message_count = " << message_count;
    if (first_db_message_id < fixed_from_message_id && message_count != -1) {
      LOG(INFO) << "Get message calendar from database in " << dialog_id << " from " << fixed_from_message_id;
      auto new_promise =
          PromiseCreator::lambda([random_id, dialog_id, fixed_from_message_id, first_db_message_id, filter,
                                  promise = std::move(promise)](Result<MessagesDbCalendar> r_calendar) mutable {
            send_closure(G()->messages_manager(), &MessagesManager::on_get_message_calendar_from_database, random_id,
                         dialog_id, fixed_from_message_id, first_db_message_id, filter, std::move(r_calendar),
                         std::move(promise));
          });
      MessagesDbDialogCalendarQuery db_query;
      db_query.dialog_id = dialog_id;
      db_query.filter = filter;
      db_query.from_message_id = fixed_from_message_id;
      db_query.tz_offset = static_cast<int32>(G()->shared_config().get_option_integer("utc_time_offset"));
      G()->td_db()->get_messages_db_async()->get_dialog_message_calendar(db_query, std::move(new_promise));
      return {};
    }
  }
  if (filter == MessageSearchFilter::FailedToSend) {
    promise.set_value(Unit());
    return {};
  }

  LOG(DEBUG) << "Get message calendar from server in " << dialog_id << " from " << from_message_id;

  switch (dialog_id.get_type()) {
    case DialogType::None:
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      td_->create_handler<GetSearchResultCalendarQuery>(std::move(promise))
          ->send(dialog_id, from_message_id, filter, random_id);
      break;
    case DialogType::SecretChat:
      promise.set_value(Unit());
      break;
    default:
      UNREACHABLE();
      promise.set_error(Status::Error(500, "Search messages is not supported"));
  }
  return {};
}

void MessagesManager::on_get_message_calendar_from_database(int64 random_id, DialogId dialog_id,
                                                            MessageId from_message_id, MessageId first_db_message_id,
                                                            MessageSearchFilter filter,
                                                            Result<MessagesDbCalendar> r_calendar,
                                                            Promise<Unit> promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  if (r_calendar.is_error()) {
    LOG(ERROR) << "Failed to get message calendar from the database: " << r_calendar.error();
    if (first_db_message_id != MessageId::min() && dialog_id.get_type() != DialogType::SecretChat &&
        filter != MessageSearchFilter::FailedToSend) {
      found_dialog_message_calendars_.erase(random_id);
    }
    return promise.set_value(Unit());
  }
  CHECK(!from_message_id.is_scheduled());
  CHECK(!first_db_message_id.is_scheduled());

  auto calendar = r_calendar.move_as_ok();

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  auto it = found_dialog_message_calendars_.find(random_id);
  CHECK(it != found_dialog_message_calendars_.end());
  CHECK(it->second == nullptr);

  vector<std::pair<MessageId, int32>> periods;
  periods.reserve(calendar.messages.size());
  for (size_t i = 0; i < calendar.messages.size(); i++) {
    auto m = on_get_message_from_database(d, calendar.messages[i], false, "on_get_message_calendar_from_database");
    if (m != nullptr && first_db_message_id <= m->message_id) {
      CHECK(!m->message_id.is_scheduled());
      periods.emplace_back(m->message_id, calendar.total_counts[i]);
    }
  }

  if (periods.empty() && first_db_message_id != MessageId::min() && dialog_id.get_type() != DialogType::SecretChat) {
    LOG(INFO) << "No messages found in database";
    found_dialog_message_calendars_.erase(it);
  } else {
    auto total_count = d->message_count_by_index[message_search_filter_index(filter)];
    vector<td_api::object_ptr<td_api::messageCalendarDay>> days;
    for (auto &period : periods) {
      const auto *m = get_message(d, period.first);
      CHECK(m != nullptr);
      days.push_back(td_api::make_object<td_api::messageCalendarDay>(
          period.second, get_message_object(dialog_id, m, "on_get_message_calendar_from_database")));
    }
    it->second = td_api::make_object<td_api::messageCalendar>(total_count, std::move(days));
  }
  promise.set_value(Unit());
}

std::pair<int32, vector<MessageId>> MessagesManager::search_dialog_messages(
    DialogId dialog_id, const string &query, const td_api::object_ptr<td_api::MessageSender> &sender,
    MessageId from_message_id, int32 offset, int32 limit, MessageSearchFilter filter, MessageId top_thread_message_id,
    int64 &random_id, bool use_db, Promise<Unit> &&promise) {
  if (random_id != 0) {
    // request has already been sent before
    auto it = found_dialog_messages_.find(random_id);
    if (it != found_dialog_messages_.end()) {
      CHECK(found_dialog_messages_dialog_id_.count(random_id) == 0);
      auto result = std::move(it->second);
      found_dialog_messages_.erase(it);
      promise.set_value(Unit());
      return result;
    }
    random_id = 0;
  }
  LOG(INFO) << "Search messages with query \"" << query << "\" in " << dialog_id << " sent by "
            << oneline(to_string(sender)) << " in thread of " << top_thread_message_id << " filtered by " << filter
            << " from " << from_message_id << " with offset " << offset << " and limit " << limit;

  std::pair<int32, vector<MessageId>> result;
  if (limit <= 0) {
    promise.set_error(Status::Error(400, "Parameter limit must be positive"));
    return result;
  }
  if (limit > MAX_SEARCH_MESSAGES) {
    limit = MAX_SEARCH_MESSAGES;
  }
  if (limit <= -offset) {
    promise.set_error(Status::Error(400, "Parameter limit must be greater than -offset"));
    return result;
  }
  if (offset > 0) {
    promise.set_error(Status::Error(400, "Parameter offset must be non-positive"));
    return result;
  }

  if (from_message_id.get() > MessageId::max().get()) {
    from_message_id = MessageId::max();
  }

  if (!from_message_id.is_valid() && from_message_id != MessageId()) {
    promise.set_error(Status::Error(400, "Parameter from_message_id must be identifier of a chat message or 0"));
    return result;
  }
  from_message_id = from_message_id.get_next_server_message_id();

  const Dialog *d = get_dialog_force(dialog_id, "search_dialog_messages");
  if (d == nullptr) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return result;
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    promise.set_error(Status::Error(400, "Can't access the chat"));
    return {};
  }

  DialogId sender_dialog_id;
  if (sender != nullptr) {
    switch (sender->get_id()) {
      case td_api::messageSenderUser::ID:
        sender_dialog_id = DialogId(UserId(static_cast<const td_api::messageSenderUser *>(sender.get())->user_id_));
        break;
      case td_api::messageSenderChat::ID:
        sender_dialog_id = DialogId(static_cast<const td_api::messageSenderChat *>(sender.get())->chat_id_);
        switch (sender_dialog_id.get_type()) {
          case DialogType::User:
          case DialogType::Chat:
          case DialogType::Channel:
            // ok
            break;
          case DialogType::SecretChat:
            promise.set_value(Unit());
            return result;
          case DialogType::None:
            if (sender_dialog_id == DialogId()) {
              break;
            }
            promise.set_error(Status::Error(400, "Invalid sender chat identifier specified"));
            return result;
          default:
            UNREACHABLE();
            return result;
        }
        break;
      default:
        UNREACHABLE();
    }
    if (sender_dialog_id != DialogId() && !have_input_peer(sender_dialog_id, AccessRights::Read)) {
      promise.set_error(Status::Error(400, "Invalid message sender specified"));
      return result;
    }
  }
  if (sender_dialog_id == dialog_id && is_broadcast_channel(dialog_id)) {
    sender_dialog_id = DialogId();
  }

  if (filter == MessageSearchFilter::FailedToSend && sender_dialog_id.is_valid()) {
    if (sender_dialog_id != get_my_dialog_id()) {
      promise.set_value(Unit());
      return result;
    }
    sender_dialog_id = DialogId();
  }

  if (top_thread_message_id != MessageId()) {
    if (!top_thread_message_id.is_valid() || !top_thread_message_id.is_server()) {
      promise.set_error(Status::Error(400, "Invalid message thread ID specified"));
      return result;
    }
    if (dialog_id.get_type() != DialogType::Channel || is_broadcast_channel(dialog_id)) {
      promise.set_error(Status::Error(400, "Can't filter by message thread ID in the chat"));
      return result;
    }
  }

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || found_dialog_messages_.find(random_id) != found_dialog_messages_.end());
  found_dialog_messages_[random_id];  // reserve place for result

  if (filter == MessageSearchFilter::UnreadMention) {
    if (!query.empty()) {
      promise.set_error(Status::Error(400, "Non-empty query is unsupported with the specified filter"));
      return result;
    }
    if (sender_dialog_id.is_valid()) {
      promise.set_error(Status::Error(400, "Filtering by sender is unsupported with the specified filter"));
      return result;
    }
    if (top_thread_message_id != MessageId()) {
      promise.set_error(Status::Error(400, "Filtering by message thread is unsupported with the specified filter"));
      return result;
    }
  }

  // Trying to use database
  if (use_db && query.empty() && G()->parameters().use_message_db && filter != MessageSearchFilter::Empty &&
      !sender_dialog_id.is_valid() && top_thread_message_id == MessageId()) {
    MessageId first_db_message_id = get_first_database_message_id_by_index(d, filter);
    int32 message_count = d->message_count_by_index[message_search_filter_index(filter)];
    auto fixed_from_message_id = from_message_id;
    if (fixed_from_message_id == MessageId()) {
      fixed_from_message_id = MessageId::max();
    }
    LOG(INFO) << "Search messages in " << dialog_id << " from " << fixed_from_message_id << ", have up to "
              << first_db_message_id << ", message_count = " << message_count;
    if ((first_db_message_id < fixed_from_message_id || (first_db_message_id == fixed_from_message_id && offset < 0)) &&
        message_count != -1) {
      LOG(INFO) << "Search messages in database in " << dialog_id << " from " << fixed_from_message_id
                << " and with limit " << limit;
      auto new_promise = PromiseCreator::lambda(
          [random_id, dialog_id, fixed_from_message_id, first_db_message_id, filter, offset, limit,
           promise = std::move(promise)](Result<vector<MessagesDbDialogMessage>> r_messages) mutable {
            send_closure(G()->messages_manager(), &MessagesManager::on_search_dialog_messages_db_result, random_id,
                         dialog_id, fixed_from_message_id, first_db_message_id, filter, offset, limit,
                         std::move(r_messages), std::move(promise));
          });
      MessagesDbMessagesQuery db_query;
      db_query.dialog_id = dialog_id;
      db_query.filter = filter;
      db_query.from_message_id = fixed_from_message_id;
      db_query.offset = offset;
      db_query.limit = limit;
      G()->td_db()->get_messages_db_async()->get_messages(db_query, std::move(new_promise));
      return result;
    }
  }
  if (filter == MessageSearchFilter::FailedToSend) {
    promise.set_value(Unit());
    return result;
  }

  LOG(DEBUG) << "Search messages on server in " << dialog_id << " with query \"" << query << "\" from "
             << sender_dialog_id << " in thread of " << top_thread_message_id << " from " << from_message_id
             << " and with limit " << limit;

  switch (dialog_id.get_type()) {
    case DialogType::None:
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      td_->create_handler<SearchMessagesQuery>(std::move(promise))
          ->send(dialog_id, query, sender_dialog_id, from_message_id, offset, limit, filter, top_thread_message_id,
                 random_id);
      break;
    case DialogType::SecretChat:
      if (filter == MessageSearchFilter::UnreadMention || filter == MessageSearchFilter::Pinned) {
        promise.set_value(Unit());
      } else {
        promise.set_error(Status::Error(500, "Search messages in secret chats is not supported"));
      }
      break;
    default:
      UNREACHABLE();
      promise.set_error(Status::Error(500, "Search messages is not supported"));
  }
  return result;
}

std::pair<int32, vector<FullMessageId>> MessagesManager::search_call_messages(MessageId from_message_id, int32 limit,
                                                                              bool only_missed, int64 &random_id,
                                                                              bool use_db, Promise<Unit> &&promise) {
  if (random_id != 0) {
    // request has already been sent before
    auto it = found_call_messages_.find(random_id);
    if (it != found_call_messages_.end()) {
      auto result = std::move(it->second);
      found_call_messages_.erase(it);
      promise.set_value(Unit());
      return result;
    }
    random_id = 0;
  }
  LOG(INFO) << "Search call messages from " << from_message_id << " with limit " << limit;

  std::pair<int32, vector<FullMessageId>> result;
  if (limit <= 0) {
    promise.set_error(Status::Error(400, "Parameter limit must be positive"));
    return result;
  }
  if (limit > MAX_SEARCH_MESSAGES) {
    limit = MAX_SEARCH_MESSAGES;
  }

  if (from_message_id.get() > MessageId::max().get()) {
    from_message_id = MessageId::max();
  }

  if (!from_message_id.is_valid() && from_message_id != MessageId()) {
    promise.set_error(Status::Error(400, "Parameter from_message_id must be identifier of a chat message or 0"));
    return result;
  }
  from_message_id = from_message_id.get_next_server_message_id();

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || found_call_messages_.find(random_id) != found_call_messages_.end());
  found_call_messages_[random_id];  // reserve place for result

  auto filter = only_missed ? MessageSearchFilter::MissedCall : MessageSearchFilter::Call;

  if (use_db && G()->parameters().use_message_db) {
    // try to use database
    MessageId first_db_message_id =
        calls_db_state_.first_calls_database_message_id_by_index[call_message_search_filter_index(filter)];
    int32 message_count = calls_db_state_.message_count_by_index[call_message_search_filter_index(filter)];
    auto fixed_from_message_id = from_message_id;
    if (fixed_from_message_id == MessageId()) {
      fixed_from_message_id = MessageId::max();
    }
    CHECK(fixed_from_message_id.is_valid() && fixed_from_message_id.is_server());
    LOG(INFO) << "Search call messages from " << fixed_from_message_id << ", have up to " << first_db_message_id
              << ", message_count = " << message_count;
    if (first_db_message_id < fixed_from_message_id && message_count != -1) {
      LOG(INFO) << "Search messages in database from " << fixed_from_message_id << " and with limit " << limit;

      MessagesDbCallsQuery db_query;
      db_query.filter = filter;
      db_query.from_unique_message_id = fixed_from_message_id.get_server_message_id().get();
      db_query.limit = limit;
      G()->td_db()->get_messages_db_async()->get_calls(
          db_query, PromiseCreator::lambda([random_id, first_db_message_id, filter, promise = std::move(promise)](
                                               Result<MessagesDbCallsResult> calls_result) mutable {
            send_closure(G()->messages_manager(), &MessagesManager::on_messages_db_calls_result,
                         std::move(calls_result), random_id, first_db_message_id, filter, std::move(promise));
          }));
      return result;
    }
  }

  LOG(DEBUG) << "Search call messages on server from " << from_message_id << " and with limit " << limit;
  td_->create_handler<SearchMessagesQuery>(std::move(promise))
      ->send(DialogId(), "", DialogId(), from_message_id, 0, limit, filter, MessageId(), random_id);
  return result;
}

void MessagesManager::search_dialog_recent_location_messages(DialogId dialog_id, int32 limit,
                                                             Promise<td_api::object_ptr<td_api::messages>> &&promise) {
  LOG(INFO) << "Search recent location messages in " << dialog_id << " with limit " << limit;

  if (limit <= 0) {
    return promise.set_error(Status::Error(400, "Parameter limit must be positive"));
  }
  if (limit > MAX_SEARCH_MESSAGES) {
    limit = MAX_SEARCH_MESSAGES;
  }

  const Dialog *d = get_dialog_force(dialog_id, "search_dialog_recent_location_messages");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      return td_->create_handler<GetRecentLocationsQuery>(std::move(promise))->send(dialog_id, limit);
    case DialogType::SecretChat:
      return promise.set_value(get_messages_object(0, vector<td_api::object_ptr<td_api::message>>(), true));
    default:
      UNREACHABLE();
      promise.set_error(Status::Error(500, "Search messages is not supported"));
  }
}

vector<FullMessageId> MessagesManager::get_active_live_location_messages(Promise<Unit> &&promise) {
  if (!G()->parameters().use_message_db) {
    are_active_live_location_messages_loaded_ = true;
  }

  if (!are_active_live_location_messages_loaded_) {
    load_active_live_location_messages_queries_.push_back(std::move(promise));
    if (load_active_live_location_messages_queries_.size() == 1u) {
      LOG(INFO) << "Trying to load active live location messages from database";
      G()->td_db()->get_sqlite_pmc()->get(
          "di_active_live_location_messages", PromiseCreator::lambda([](string value) {
            send_closure(G()->messages_manager(),
                         &MessagesManager::on_load_active_live_location_full_message_ids_from_database,
                         std::move(value));
          }));
    }
    return {};
  }

  promise.set_value(Unit());
  vector<FullMessageId> result;
  for (auto &full_message_id : active_live_location_full_message_ids_) {
    auto m = get_message(full_message_id);
    CHECK(m != nullptr);
    CHECK(m->content->get_type() == MessageContentType::LiveLocation);
    CHECK(!m->message_id.is_scheduled());

    if (m->is_failed_to_send) {
      continue;
    }

    auto live_period = get_message_content_live_location_period(m->content.get());
    if (live_period <= G()->unix_time() - m->date) {  // bool is_expired flag?
      // live location is expired
      continue;
    }
    result.push_back(full_message_id);
  }

  return result;
}

void MessagesManager::on_load_active_live_location_full_message_ids_from_database(string value) {
  if (G()->close_flag()) {
    return;
  }
  if (value.empty()) {
    LOG(INFO) << "Active live location messages aren't found in the database";
    on_load_active_live_location_messages_finished();

    if (!active_live_location_full_message_ids_.empty()) {
      save_active_live_locations();
    }
    return;
  }

  LOG(INFO) << "Successfully loaded active live location messages list of size " << value.size() << " from database";

  auto new_full_message_ids = std::move(active_live_location_full_message_ids_);
  vector<FullMessageId> old_full_message_ids;
  log_event_parse(old_full_message_ids, value).ensure();

  // TODO asynchronously load messages from database
  active_live_location_full_message_ids_.clear();
  for (auto full_message_id : old_full_message_ids) {
    Message *m = get_message_force(full_message_id, "on_load_active_live_location_full_message_ids_from_database");
    if (m != nullptr) {
      try_add_active_live_location(full_message_id.get_dialog_id(), m);
    }
  }

  for (auto full_message_id : new_full_message_ids) {
    add_active_live_location(full_message_id);
  }

  on_load_active_live_location_messages_finished();

  if (!new_full_message_ids.empty() || old_full_message_ids.size() != active_live_location_full_message_ids_.size()) {
    save_active_live_locations();
  }
}

void MessagesManager::on_load_active_live_location_messages_finished() {
  are_active_live_location_messages_loaded_ = true;
  auto promises = std::move(load_active_live_location_messages_queries_);
  load_active_live_location_messages_queries_.clear();
  for (auto &promise : promises) {
    promise.set_value(Unit());
  }
}

void MessagesManager::try_add_active_live_location(DialogId dialog_id, const Message *m) {
  CHECK(m != nullptr);

  if (td_->auth_manager_->is_bot()) {
    return;
  }
  if (m->content->get_type() != MessageContentType::LiveLocation || m->message_id.is_scheduled() ||
      m->message_id.is_local() || m->via_bot_user_id.is_valid() || m->forward_info != nullptr) {
    return;
  }

  auto live_period = get_message_content_live_location_period(m->content.get());
  if (live_period <= G()->unix_time() - m->date + 1) {  // bool is_expired flag?
    // live location is expired
    return;
  }
  add_active_live_location({dialog_id, m->message_id});
}

void MessagesManager::add_active_live_location(FullMessageId full_message_id) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }
  if (!active_live_location_full_message_ids_.insert(full_message_id).second) {
    return;
  }

  // TODO add timer for live location expiration

  if (!G()->parameters().use_message_db) {
    return;
  }

  if (are_active_live_location_messages_loaded_) {
    save_active_live_locations();
  } else if (load_active_live_location_messages_queries_.empty()) {
    // load active live locations and save after that
    get_active_live_location_messages(Auto());
  }
}

bool MessagesManager::delete_active_live_location(DialogId dialog_id, const Message *m) {
  CHECK(m != nullptr);
  return active_live_location_full_message_ids_.erase(FullMessageId{dialog_id, m->message_id}) != 0;
}

void MessagesManager::save_active_live_locations() {
  CHECK(are_active_live_location_messages_loaded_);
  LOG(INFO) << "Save active live locations of size " << active_live_location_full_message_ids_.size() << " to database";
  if (G()->parameters().use_message_db) {
    G()->td_db()->get_sqlite_pmc()->set("di_active_live_location_messages",
                                        log_event_store(active_live_location_full_message_ids_).as_slice().str(),
                                        Auto());
  }
}

void MessagesManager::on_message_live_location_viewed(Dialog *d, const Message *m) {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  CHECK(m->content->get_type() == MessageContentType::LiveLocation);
  CHECK(!m->message_id.is_scheduled());

  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  switch (d->dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      // ok
      break;
    case DialogType::SecretChat:
      return;
    default:
      UNREACHABLE();
      return;
  }
  if (!d->is_opened) {
    return;
  }

  if (m->is_outgoing || !m->message_id.is_server() || m->via_bot_user_id.is_valid() || !m->sender_user_id.is_valid() ||
      td_->contacts_manager_->is_user_bot(m->sender_user_id) || m->forward_info != nullptr) {
    return;
  }

  auto live_period = get_message_content_live_location_period(m->content.get());
  if (live_period <= G()->unix_time() - m->date + 1) {
    // live location is expired
    return;
  }

  auto &live_location_task_id = d->pending_viewed_live_locations[m->message_id];
  if (live_location_task_id != 0) {
    return;
  }

  live_location_task_id = ++viewed_live_location_task_id_;
  auto &full_message_id = viewed_live_location_tasks_[live_location_task_id];
  full_message_id = FullMessageId(d->dialog_id, m->message_id);
  view_message_live_location_on_server_impl(live_location_task_id, full_message_id);
}

void MessagesManager::view_message_live_location_on_server(int64 task_id) {
  if (G()->close_flag()) {
    return;
  }

  auto it = viewed_live_location_tasks_.find(task_id);
  if (it == viewed_live_location_tasks_.end()) {
    return;
  }

  auto full_message_id = it->second;
  Dialog *d = get_dialog(full_message_id.get_dialog_id());
  const Message *m = get_message_force(d, full_message_id.get_message_id(), "view_message_live_location_on_server");
  if (m == nullptr || get_message_content_live_location_period(m->content.get()) <= G()->unix_time() - m->date + 1) {
    // the message was deleted or live location is expired
    viewed_live_location_tasks_.erase(it);
    auto erased_count = d->pending_viewed_live_locations.erase(full_message_id.get_message_id());
    CHECK(erased_count > 0);
    return;
  }

  view_message_live_location_on_server_impl(task_id, full_message_id);
}

void MessagesManager::view_message_live_location_on_server_impl(int64 task_id, FullMessageId full_message_id) {
  auto promise = PromiseCreator::lambda([actor_id = actor_id(this), task_id](Unit result) {
    send_closure(actor_id, &MessagesManager::on_message_live_location_viewed_on_server, task_id);
  });
  read_message_contents_on_server(full_message_id.get_dialog_id(), {full_message_id.get_message_id()}, 0,
                                  std::move(promise), true);
}

void MessagesManager::on_message_live_location_viewed_on_server(int64 task_id) {
  if (G()->close_flag()) {
    return;
  }

  auto it = viewed_live_location_tasks_.find(task_id);
  if (it == viewed_live_location_tasks_.end()) {
    return;
  }

  pending_message_live_location_view_timeout_.add_timeout_in(task_id, LIVE_LOCATION_VIEW_PERIOD);
}

void MessagesManager::try_add_bot_command_message_id(DialogId dialog_id, const Message *m) {
  CHECK(m != nullptr);
  if (td_->auth_manager_->is_bot() || !is_group_dialog(dialog_id) || m->message_id.is_scheduled() ||
      !has_bot_commands(get_message_content_text(m->content.get()))) {
    return;
  }

  dialog_bot_command_message_ids_[dialog_id].message_ids.insert(m->message_id);
}

void MessagesManager::delete_bot_command_message_id(DialogId dialog_id, MessageId message_id) {
  if (message_id.is_scheduled()) {
    return;
  }
  auto it = dialog_bot_command_message_ids_.find(dialog_id);
  if (it == dialog_bot_command_message_ids_.end()) {
    return;
  }
  if (it->second.message_ids.erase(message_id) && it->second.message_ids.empty()) {
    dialog_bot_command_message_ids_.erase(it);
  }
}

FileSourceId MessagesManager::get_message_file_source_id(FullMessageId full_message_id) {
  if (td_->auth_manager_->is_bot()) {
    return FileSourceId();
  }

  auto dialog_id = full_message_id.get_dialog_id();
  auto message_id = full_message_id.get_message_id();
  if (!dialog_id.is_valid() || !(message_id.is_valid() || message_id.is_valid_scheduled()) ||
      dialog_id.get_type() == DialogType::SecretChat || !message_id.is_any_server()) {
    return FileSourceId();
  }

  auto &file_source_id = full_message_id_to_file_source_id_[full_message_id];
  if (!file_source_id.is_valid()) {
    file_source_id = td_->file_reference_manager_->create_message_file_source(full_message_id);
  }
  return file_source_id;
}

void MessagesManager::add_message_file_sources(DialogId dialog_id, const Message *m) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  if (dialog_id.get_type() != DialogType::SecretChat && m->is_content_secret) {
    // return;
  }

  auto file_ids = get_message_content_file_ids(m->content.get(), td_);
  if (file_ids.empty()) {
    return;
  }

  // do not create file_source_id for messages without file_ids
  auto file_source_id = get_message_file_source_id(FullMessageId(dialog_id, m->message_id));
  if (file_source_id.is_valid()) {
    for (auto file_id : file_ids) {
      td_->file_manager_->add_file_source(file_id, file_source_id);
    }
  }
}

void MessagesManager::remove_message_file_sources(DialogId dialog_id, const Message *m) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  auto file_ids = get_message_content_file_ids(m->content.get(), td_);
  if (file_ids.empty()) {
    return;
  }

  // do not create file_source_id for messages without file_ids
  auto file_source_id = get_message_file_source_id(FullMessageId(dialog_id, m->message_id));
  if (file_source_id.is_valid()) {
    for (auto file_id : file_ids) {
      td_->file_manager_->remove_file_source(file_id, file_source_id);
    }
  }
}

void MessagesManager::change_message_files(DialogId dialog_id, const Message *m, const vector<FileId> &old_file_ids) {
  if (dialog_id.get_type() != DialogType::SecretChat && m->is_content_secret) {
    // return;
  }

  auto new_file_ids = get_message_content_file_ids(m->content.get(), td_);
  if (new_file_ids == old_file_ids) {
    return;
  }

  FullMessageId full_message_id{dialog_id, m->message_id};
  if (need_delete_message_files(dialog_id, m)) {
    for (auto file_id : old_file_ids) {
      if (!td::contains(new_file_ids, file_id) && need_delete_file(full_message_id, file_id)) {
        send_closure(G()->file_manager(), &FileManager::delete_file, file_id, Promise<>(), "change_message_files");
      }
    }
  }

  auto file_source_id = get_message_file_source_id(full_message_id);
  if (file_source_id.is_valid()) {
    td_->file_manager_->change_files_source(file_source_id, old_file_ids, new_file_ids);
  }
}

MessageId MessagesManager::get_first_database_message_id_by_index(const Dialog *d, MessageSearchFilter filter) {
  CHECK(d != nullptr);
  auto message_id = filter == MessageSearchFilter::Empty
                        ? d->first_database_message_id
                        : d->first_database_message_id_by_index[message_search_filter_index(filter)];
  CHECK(!message_id.is_scheduled());
  if (!message_id.is_valid()) {
    if (d->dialog_id.get_type() == DialogType::SecretChat) {
      LOG(ERROR) << "Invalid first_database_message_id_by_index in " << d->dialog_id;
      return MessageId::min();
    }
    return MessageId::max();
  }
  return message_id;
}

void MessagesManager::on_search_dialog_messages_db_result(int64 random_id, DialogId dialog_id,
                                                          MessageId from_message_id, MessageId first_db_message_id,
                                                          MessageSearchFilter filter, int32 offset, int32 limit,
                                                          Result<vector<MessagesDbDialogMessage>> r_messages,
                                                          Promise<Unit> promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  if (r_messages.is_error()) {
    LOG(ERROR) << "Failed to get messages from the database: " << r_messages.error();
    if (first_db_message_id != MessageId::min() && dialog_id.get_type() != DialogType::SecretChat &&
        filter != MessageSearchFilter::FailedToSend) {
      found_dialog_messages_.erase(random_id);
    }
    return promise.set_value(Unit());
  }
  CHECK(!from_message_id.is_scheduled());
  CHECK(!first_db_message_id.is_scheduled());

  auto messages = r_messages.move_as_ok();

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  auto it = found_dialog_messages_.find(random_id);
  CHECK(it != found_dialog_messages_.end());
  auto &res = it->second.second;

  res.reserve(messages.size());
  for (auto &message : messages) {
    auto m = on_get_message_from_database(d, message, false, "on_search_dialog_messages_db_result");
    if (m != nullptr && first_db_message_id <= m->message_id) {
      if (filter == MessageSearchFilter::UnreadMention && !m->contains_unread_mention) {
        // skip already read by d->last_read_all_mentions_message_id mentions
      } else {
        CHECK(!m->message_id.is_scheduled());
        res.push_back(m->message_id);
      }
    }
  }

  auto &message_count = d->message_count_by_index[message_search_filter_index(filter)];
  auto result_size = narrow_cast<int32>(res.size());
  bool from_the_end =
      from_message_id == MessageId::max() || (offset < 0 && (result_size == 0 || res[0] < from_message_id));
  if ((message_count != -1 && message_count < result_size) ||
      (message_count > result_size && from_the_end && first_db_message_id == MessageId::min() &&
       result_size < limit + offset)) {
    LOG(INFO) << "Fix found message count in " << dialog_id << " from " << message_count << " to " << result_size;
    message_count = result_size;
    if (filter == MessageSearchFilter::UnreadMention) {
      d->unread_mention_count = message_count;
      update_dialog_mention_notification_count(d);
      send_update_chat_unread_mention_count(d);
    }
    on_dialog_updated(dialog_id, "on_search_dialog_messages_db_result");
  }
  it->second.first = message_count;
  if (res.empty() && first_db_message_id != MessageId::min() && dialog_id.get_type() != DialogType::SecretChat) {
    LOG(INFO) << "No messages found in database";
    found_dialog_messages_.erase(it);
  } else {
    LOG(INFO) << "Found " << res.size() << " messages out of " << message_count << " in database";
    if (from_the_end && filter == MessageSearchFilter::Pinned) {
      set_dialog_last_pinned_message_id(d, res.empty() ? MessageId() : res[0]);
    }
  }
  promise.set_value(Unit());
}

td_api::object_ptr<td_api::foundMessages> MessagesManager::get_found_messages_object(
    const FoundMessages &found_messages, const char *source) {
  vector<tl_object_ptr<td_api::message>> result;
  result.reserve(found_messages.full_message_ids.size());
  for (auto full_message_id : found_messages.full_message_ids) {
    auto message = get_message_object(full_message_id, source);
    if (message != nullptr) {
      result.push_back(std::move(message));
    }
  }

  return td_api::make_object<td_api::foundMessages>(found_messages.total_count, std::move(result),
                                                    found_messages.next_offset);
}

MessagesManager::FoundMessages MessagesManager::offline_search_messages(DialogId dialog_id, const string &query,
                                                                        string offset, int32 limit,
                                                                        MessageSearchFilter filter, int64 &random_id,
                                                                        Promise<> &&promise) {
  if (!G()->parameters().use_message_db) {
    promise.set_error(Status::Error(400, "Message database is required to search messages in secret chats"));
    return {};
  }

  if (random_id != 0) {
    // request has already been sent before
    auto it = found_fts_messages_.find(random_id);
    CHECK(it != found_fts_messages_.end());
    auto result = std::move(it->second);
    found_fts_messages_.erase(it);
    promise.set_value(Unit());
    return result;
  }

  if (query.empty()) {
    promise.set_value(Unit());
    return {};
  }
  if (dialog_id != DialogId() && !have_dialog_force(dialog_id, "offline_search_messages")) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return {};
  }
  if (limit <= 0) {
    promise.set_error(Status::Error(400, "Limit must be positive"));
    return {};
  }
  if (limit > MAX_SEARCH_MESSAGES) {
    limit = MAX_SEARCH_MESSAGES;
  }

  MessagesDbFtsQuery fts_query;
  fts_query.query = query;
  fts_query.dialog_id = dialog_id;
  fts_query.filter = filter;
  if (!offset.empty()) {
    auto r_from_search_id = to_integer_safe<int64>(offset);
    if (r_from_search_id.is_error()) {
      promise.set_error(Status::Error(400, "Invalid offset specified"));
      return {};
    }
    fts_query.from_search_id = r_from_search_id.ok();
  }
  fts_query.limit = limit;

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || found_fts_messages_.find(random_id) != found_fts_messages_.end());
  found_fts_messages_[random_id];  // reserve place for result

  G()->td_db()->get_messages_db_async()->get_messages_fts(
      std::move(fts_query),
      PromiseCreator::lambda([random_id, offset = std::move(offset), limit,
                              promise = std::move(promise)](Result<MessagesDbFtsResult> fts_result) mutable {
        send_closure(G()->messages_manager(), &MessagesManager::on_messages_db_fts_result, std::move(fts_result),
                     std::move(offset), limit, random_id, std::move(promise));
      }));

  return {};
}

void MessagesManager::on_messages_db_fts_result(Result<MessagesDbFtsResult> result, string offset, int32 limit,
                                                int64 random_id, Promise<Unit> &&promise) {
  if (G()->close_flag()) {
    result = Global::request_aborted_error();
  }
  if (result.is_error()) {
    found_fts_messages_.erase(random_id);
    return promise.set_error(result.move_as_error());
  }
  auto fts_result = result.move_as_ok();

  auto it = found_fts_messages_.find(random_id);
  CHECK(it != found_fts_messages_.end());
  auto &res = it->second.full_message_ids;

  res.reserve(fts_result.messages.size());
  for (auto &message : fts_result.messages) {
    auto m = on_get_message_from_database(message, false, "on_messages_db_fts_result");
    if (m != nullptr) {
      res.emplace_back(message.dialog_id, m->message_id);
    }
  }

  it->second.next_offset = fts_result.next_search_id <= 1 ? string() : to_string(fts_result.next_search_id);
  it->second.total_count = offset.empty() && fts_result.messages.size() < static_cast<size_t>(limit)
                               ? static_cast<int32>(fts_result.messages.size())
                               : -1;

  promise.set_value(Unit());
}

void MessagesManager::on_messages_db_calls_result(Result<MessagesDbCallsResult> result, int64 random_id,
                                                  MessageId first_db_message_id, MessageSearchFilter filter,
                                                  Promise<> &&promise) {
  CHECK(!first_db_message_id.is_scheduled());
  if (G()->close_flag()) {
    result = Global::request_aborted_error();
  }
  if (result.is_error()) {
    found_call_messages_.erase(random_id);
    return promise.set_error(result.move_as_error());
  }
  auto calls_result = result.move_as_ok();

  auto it = found_call_messages_.find(random_id);
  CHECK(it != found_call_messages_.end());
  auto &res = it->second.second;

  res.reserve(calls_result.messages.size());
  for (auto &message : calls_result.messages) {
    auto m = on_get_message_from_database(message, false, "on_messages_db_calls_result");

    if (m != nullptr && first_db_message_id <= m->message_id) {
      res.emplace_back(message.dialog_id, m->message_id);
    }
  }
  it->second.first = calls_db_state_.message_count_by_index[call_message_search_filter_index(filter)];

  if (res.empty() && first_db_message_id != MessageId::min()) {
    LOG(INFO) << "No messages found in database";
    found_call_messages_.erase(it);
  }

  promise.set_value(Unit());
}

std::pair<int32, vector<FullMessageId>> MessagesManager::search_messages(
    FolderId folder_id, bool ignore_folder_id, const string &query, int32 offset_date, DialogId offset_dialog_id,
    MessageId offset_message_id, int32 limit, MessageSearchFilter filter, int32 min_date, int32 max_date,
    int64 &random_id, Promise<Unit> &&promise) {
  if (random_id != 0) {
    // request has already been sent before
    auto it = found_messages_.find(random_id);
    CHECK(it != found_messages_.end());
    auto result = std::move(it->second);
    found_messages_.erase(it);
    promise.set_value(Unit());
    return result;
  }

  if (limit <= 0) {
    promise.set_error(Status::Error(400, "Parameter limit must be positive"));
    return {};
  }
  if (limit > MAX_SEARCH_MESSAGES) {
    limit = MAX_SEARCH_MESSAGES;
  }

  if (offset_date <= 0) {
    offset_date = std::numeric_limits<int32>::max();
  }
  if (!offset_message_id.is_valid()) {
    if (offset_message_id.is_valid_scheduled()) {
      promise.set_error(Status::Error(400, "Parameter offset_message_id can't be a scheduled message identifier"));
      return {};
    }
    offset_message_id = MessageId();
  }
  if (offset_message_id != MessageId() && !offset_message_id.is_server()) {
    promise.set_error(
        Status::Error(400, "Parameter offset_message_id must be identifier of the last found message or 0"));
    return {};
  }

  if (filter == MessageSearchFilter::Call || filter == MessageSearchFilter::MissedCall ||
      filter == MessageSearchFilter::Mention || filter == MessageSearchFilter::UnreadMention ||
      filter == MessageSearchFilter::FailedToSend || filter == MessageSearchFilter::Pinned) {
    promise.set_error(Status::Error(400, "The filter is not supported"));
    return {};
  }

  if (query.empty() && filter == MessageSearchFilter::Empty) {
    promise.set_value(Unit());
    return {};
  }

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || found_messages_.find(random_id) != found_messages_.end());
  found_messages_[random_id];  // reserve place for result

  LOG(DEBUG) << "Search all messages filtered by " << filter << " with query = \"" << query << "\" from date "
             << offset_date << ", " << offset_dialog_id << ", " << offset_message_id << " and limit " << limit;

  td_->create_handler<SearchMessagesGlobalQuery>(std::move(promise))
      ->send(folder_id, ignore_folder_id, query, offset_date, offset_dialog_id, offset_message_id, limit, filter,
             min_date, max_date, random_id);
  return {};
}

int64 MessagesManager::get_dialog_message_by_date(DialogId dialog_id, int32 date, Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(dialog_id, "get_dialog_message_by_date");
  if (d == nullptr) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return 0;
  }

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    promise.set_error(Status::Error(400, "Can't access the chat"));
    return 0;
  }

  if (date <= 0) {
    date = 1;
  }

  int64 random_id = 0;
  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 ||
           get_dialog_message_by_date_results_.find(random_id) != get_dialog_message_by_date_results_.end());
  get_dialog_message_by_date_results_[random_id];  // reserve place for result

  auto message_id = find_message_by_date(d->messages.get(), date);
  if (message_id.is_valid() && (message_id == d->last_message_id || get_message(d, message_id)->have_next)) {
    get_dialog_message_by_date_results_[random_id] = {dialog_id, message_id};
    promise.set_value(Unit());
    return random_id;
  }

  if (G()->parameters().use_message_db && d->last_database_message_id != MessageId()) {
    CHECK(d->first_database_message_id != MessageId());
    G()->td_db()->get_messages_db_async()->get_dialog_message_by_date(
        dialog_id, d->first_database_message_id, d->last_database_message_id, date,
        PromiseCreator::lambda([actor_id = actor_id(this), dialog_id, date, random_id,
                                promise = std::move(promise)](Result<MessagesDbDialogMessage> result) mutable {
          send_closure(actor_id, &MessagesManager::on_get_dialog_message_by_date_from_database, dialog_id, date,
                       random_id, std::move(result), std::move(promise));
        }));
  } else {
    get_dialog_message_by_date_from_server(d, date, random_id, false, std::move(promise));
  }
  return random_id;
}

MessageId MessagesManager::find_message_by_date(const Message *m, int32 date) {
  if (m == nullptr) {
    return MessageId();
  }

  if (m->date > date) {
    return find_message_by_date(m->left.get(), date);
  }

  auto message_id = find_message_by_date(m->right.get(), date);
  if (message_id.is_valid()) {
    return message_id;
  }

  return m->message_id;
}

void MessagesManager::find_messages_by_date(const Message *m, int32 min_date, int32 max_date,
                                            vector<MessageId> &message_ids) {
  if (m == nullptr) {
    return;
  }

  if (m->date >= min_date) {
    find_messages_by_date(m->left.get(), min_date, max_date, message_ids);
    if (m->date <= max_date) {
      message_ids.push_back(m->message_id);
    }
  }
  if (m->date <= max_date) {
    find_messages_by_date(m->right.get(), min_date, max_date, message_ids);
  }
}

void MessagesManager::on_get_dialog_message_by_date_from_database(DialogId dialog_id, int32 date, int64 random_id,
                                                                  Result<MessagesDbDialogMessage> result,
                                                                  Promise<Unit> promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  if (result.is_ok()) {
    Message *m = on_get_message_from_database(d, result.ok(), false, "on_get_dialog_message_by_date_from_database");
    if (m != nullptr) {
      auto message_id = find_message_by_date(d->messages.get(), date);
      if (!message_id.is_valid()) {
        LOG(ERROR) << "Failed to find " << m->message_id << " in " << dialog_id << " by date " << date;
        message_id = m->message_id;
      }
      get_dialog_message_by_date_results_[random_id] = {dialog_id, message_id};
      promise.set_value(Unit());
      return;
    }
    // TODO if m == nullptr, we need to just adjust it to the next non-nullptr message, not get from server
  }

  return get_dialog_message_by_date_from_server(d, date, random_id, true, std::move(promise));
}

void MessagesManager::get_dialog_message_by_date_from_server(const Dialog *d, int32 date, int64 random_id,
                                                             bool after_database_search, Promise<Unit> &&promise) {
  CHECK(d != nullptr);
  if (d->have_full_history) {
    // request can be always done locally/in memory. There is no need to send request to the server
    if (after_database_search) {
      return promise.set_value(Unit());
    }

    auto message_id = find_message_by_date(d->messages.get(), date);
    if (message_id.is_valid()) {
      get_dialog_message_by_date_results_[random_id] = {d->dialog_id, message_id};
    }
    promise.set_value(Unit());
    return;
  }
  if (d->dialog_id.get_type() == DialogType::SecretChat) {
    // there is no way to send request to the server
    return promise.set_value(Unit());
  }

  td_->create_handler<GetDialogMessageByDateQuery>(std::move(promise))->send(d->dialog_id, date, random_id);
}

void MessagesManager::on_get_dialog_message_by_date_success(DialogId dialog_id, int32 date, int64 random_id,
                                                            vector<tl_object_ptr<telegram_api::Message>> &&messages,
                                                            Promise<Unit> &&promise) {
  TRY_STATUS_PROMISE(promise, G()->close_status());

  auto it = get_dialog_message_by_date_results_.find(random_id);
  CHECK(it != get_dialog_message_by_date_results_.end());
  auto &result = it->second;
  CHECK(result == FullMessageId());

  for (auto &message : messages) {
    auto message_date = get_message_date(message);
    auto message_dialog_id = get_message_dialog_id(message);
    if (message_dialog_id != dialog_id) {
      LOG(ERROR) << "Receive message in wrong " << message_dialog_id << " instead of " << dialog_id;
      continue;
    }
    if (message_date != 0 && message_date <= date) {
      result = on_get_message(std::move(message), false, dialog_id.get_type() == DialogType::Channel, false, false,
                              false, "on_get_dialog_message_by_date_success");
      if (result != FullMessageId()) {
        const Dialog *d = get_dialog(dialog_id);
        CHECK(d != nullptr);
        auto message_id = find_message_by_date(d->messages.get(), date);
        if (!message_id.is_valid()) {
          LOG(ERROR) << "Failed to find " << result.get_message_id() << " in " << dialog_id << " by date " << date;
          message_id = result.get_message_id();
        }
        get_dialog_message_by_date_results_[random_id] = {dialog_id, message_id};
        // TODO result must be adjusted by local messages
        promise.set_value(Unit());
        return;
      }
    }
  }
  promise.set_value(Unit());
}

void MessagesManager::on_get_dialog_message_by_date_fail(int64 random_id) {
  auto erased_count = get_dialog_message_by_date_results_.erase(random_id);
  CHECK(erased_count > 0);
}

tl_object_ptr<td_api::message> MessagesManager::get_dialog_message_by_date_object(int64 random_id) {
  auto it = get_dialog_message_by_date_results_.find(random_id);
  CHECK(it != get_dialog_message_by_date_results_.end());
  auto full_message_id = std::move(it->second);
  get_dialog_message_by_date_results_.erase(it);
  return get_message_object(full_message_id, "get_dialog_message_by_date_object");
}

void MessagesManager::get_dialog_sparse_message_positions(
    DialogId dialog_id, MessageSearchFilter filter, MessageId from_message_id, int32 limit,
    Promise<td_api::object_ptr<td_api::messagePositions>> &&promise) {
  const Dialog *d = get_dialog_force(dialog_id, "get_dialog_sparse_message_positions");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }
  if (limit < 50 || limit > 2000) {  // server-side limits
    return promise.set_error(Status::Error(400, "Invalid limit specified"));
  }

  if (filter == MessageSearchFilter::Empty || filter == MessageSearchFilter::Call ||
      filter == MessageSearchFilter::MissedCall || filter == MessageSearchFilter::Mention ||
      filter == MessageSearchFilter::UnreadMention || filter == MessageSearchFilter::Pinned) {
    return promise.set_error(Status::Error(400, "The filter is not supported"));
  }

  if (from_message_id.is_scheduled()) {
    return promise.set_error(Status::Error(400, "Invalid from_message_id specified"));
  }
  if (!from_message_id.is_valid() || from_message_id > d->last_new_message_id) {
    if (d->last_new_message_id.is_valid()) {
      from_message_id = d->last_new_message_id.get_next_message_id(MessageType::Server);
    } else {
      from_message_id = MessageId::max();
    }
  } else {
    from_message_id = from_message_id.get_next_server_message_id();
  }

  if (filter == MessageSearchFilter::FailedToSend || dialog_id.get_type() == DialogType::SecretChat) {
    if (!G()->parameters().use_message_db) {
      return promise.set_error(Status::Error(400, "Unsupported without message database"));
    }

    LOG(INFO) << "Get sparse message positions from database";
    auto new_promise =
        PromiseCreator::lambda([promise = std::move(promise)](Result<MessagesDbMessagePositions> result) mutable {
          if (result.is_error()) {
            return promise.set_error(result.move_as_error());
          }

          auto positions = result.move_as_ok();
          promise.set_value(td_api::make_object<td_api::messagePositions>(
              positions.total_count, transform(positions.positions, [](const MessagesDbMessagePosition &position) {
                return td_api::make_object<td_api::messagePosition>(position.position, position.message_id.get(),
                                                                    position.date);
              })));
        });
    MessagesDbGetDialogSparseMessagePositionsQuery db_query;
    db_query.dialog_id = dialog_id;
    db_query.filter = filter;
    db_query.from_message_id = from_message_id;
    db_query.limit = limit;
    G()->td_db()->get_messages_db_async()->get_dialog_sparse_message_positions(db_query, std::move(new_promise));
    return;
  }

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      td_->create_handler<GetSearchResultPositionsQuery>(std::move(promise))
          ->send(dialog_id, filter, from_message_id, limit);
      break;
    case DialogType::SecretChat:
    case DialogType::None:
    default:
      UNREACHABLE();
  }
}

void MessagesManager::on_get_dialog_sparse_message_positions(
    DialogId dialog_id, MessageSearchFilter filter,
    telegram_api::object_ptr<telegram_api::messages_searchResultsPositions> positions,
    Promise<td_api::object_ptr<td_api::messagePositions>> &&promise) {
  auto message_positions = transform(
      positions->positions_, [](const telegram_api::object_ptr<telegram_api::searchResultPosition> &position) {
        return td_api::make_object<td_api::messagePosition>(
            position->offset_, MessageId(ServerMessageId(position->msg_id_)).get(), position->date_);
      });
  promise.set_value(td_api::make_object<td_api::messagePositions>(positions->count_, std::move(message_positions)));
}

void MessagesManager::get_dialog_message_count(DialogId dialog_id, MessageSearchFilter filter, bool return_local,
                                               Promise<int32> &&promise) {
  LOG(INFO) << "Get " << (return_local ? "local " : "") << "number of messages in " << dialog_id << " filtered by "
            << filter;

  const Dialog *d = get_dialog_force(dialog_id, "get_dialog_message_count");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }

  if (filter == MessageSearchFilter::Empty) {
    return promise.set_error(Status::Error(400, "Can't use searchMessagesFilterEmpty"));
  }

  auto dialog_type = dialog_id.get_type();
  int32 message_count = d->message_count_by_index[message_search_filter_index(filter)];
  if (message_count == -1 && filter == MessageSearchFilter::UnreadMention) {
    message_count = d->unread_mention_count;
  }
  if (message_count != -1 || return_local || dialog_type == DialogType::SecretChat ||
      filter == MessageSearchFilter::FailedToSend) {
    return promise.set_value(std::move(message_count));
  }

  LOG(INFO) << "Get number of messages in " << dialog_id << " filtered by " << filter << " from the server";

  switch (dialog_type) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      td_->create_handler<GetSearchCountersQuery>(std::move(promise))->send(dialog_id, filter);
      break;
    case DialogType::None:
    case DialogType::SecretChat:
    default:
      UNREACHABLE();
  }
}

void MessagesManager::preload_newer_messages(const Dialog *d, MessageId max_message_id) {
  CHECK(d != nullptr);
  CHECK(max_message_id.is_valid());
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  MessagesConstIterator p(d, max_message_id);
  int32 limit = MAX_GET_HISTORY * 3 / 10;
  while (*p != nullptr && limit-- > 0) {
    ++p;
    if (*p) {
      max_message_id = (*p)->message_id;
    }
  }
  if (limit > 0 && (d->last_message_id == MessageId() || max_message_id < d->last_message_id)) {
    // need to preload some new messages
    LOG(INFO) << "Preloading newer after " << max_message_id;
    load_messages_impl(d, max_message_id, -MAX_GET_HISTORY + 1, MAX_GET_HISTORY, 3, false, Promise<Unit>());
  }
}

void MessagesManager::preload_older_messages(const Dialog *d, MessageId min_message_id) {
  CHECK(d != nullptr);
  CHECK(min_message_id.is_valid());
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  /*
    if (d->first_remote_message_id == -1) {
      // nothing left to preload from server
      return;
    }
  */
  MessagesConstIterator p(d, min_message_id);
  int32 limit = MAX_GET_HISTORY * 3 / 10 + 1;
  while (*p != nullptr && limit-- > 0) {
    min_message_id = (*p)->message_id;
    --p;
  }
  if (limit > 0) {
    // need to preload some old messages
    LOG(INFO) << "Preloading older before " << min_message_id;
    load_messages_impl(d, min_message_id, 0, MAX_GET_HISTORY / 2, 3, false, Promise<Unit>());
  }
}

unique_ptr<MessagesManager::Message> MessagesManager::parse_message(DialogId dialog_id, MessageId expected_message_id,
                                                                    const BufferSlice &value, bool is_scheduled) {
  auto m = make_unique<Message>();

  auto status = log_event_parse(*m, value.as_slice());
  bool is_message_id_valid = [&] {
    if (is_scheduled) {
      if (!expected_message_id.is_valid_scheduled()) {
        return false;
      }
      if (m->message_id == expected_message_id) {
        return true;
      }
      return m->message_id.is_valid_scheduled() && expected_message_id.is_scheduled_server() &&
             m->message_id.is_scheduled_server() &&
             m->message_id.get_scheduled_server_message_id() == expected_message_id.get_scheduled_server_message_id();
    } else {
      if (!expected_message_id.is_valid()) {
        return false;
      }
      return m->message_id == expected_message_id;
    }
  }();
  if (status.is_error() || !is_message_id_valid) {
    // can't happen unless the database is broken, but has been seen in the wild
    LOG(ERROR) << "Receive invalid message from database: " << expected_message_id << ' ' << m->message_id << ' '
               << status << ' ' << format::as_hex_dump<4>(value.as_slice());
    if (!is_scheduled && dialog_id.get_type() != DialogType::SecretChat) {
      // trying to repair the message
      if (expected_message_id.is_valid() && expected_message_id.is_server()) {
        get_message_from_server({dialog_id, expected_message_id}, Auto(), "parse_message");
      }
      if (m->message_id.is_valid() && m->message_id.is_server()) {
        get_message_from_server({dialog_id, m->message_id}, Auto(), "parse_message");
      }
    }
    return nullptr;
  }

  LOG(INFO) << "Loaded " << m->message_id << " in " << dialog_id << " of size " << value.size() << " from database";
  return m;
}

void MessagesManager::on_get_history_from_database(DialogId dialog_id, MessageId from_message_id,
                                                   MessageId old_last_database_message_id, int32 offset, int32 limit,
                                                   bool from_the_end, bool only_local,
                                                   vector<MessagesDbDialogMessage> &&messages,
                                                   Promise<Unit> &&promise) {
  CHECK(-limit < offset && offset <= 0);
  CHECK(offset < 0 || from_the_end);
  CHECK(!from_message_id.is_scheduled());
  TRY_STATUS_PROMISE(promise, G()->close_status());

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    LOG(WARNING) << "Ignore result of get_history_from_database in " << dialog_id;
    promise.set_value(Unit());
    return;
  }

  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  LOG(INFO) << "Receive " << messages.size() << " history messages from database "
            << (from_the_end ? "from the end " : "") << "in " << dialog_id << " from " << from_message_id
            << " with offset " << offset << " and limit " << limit << ". First database message is "
            << d->first_database_message_id << ", last database message is " << d->last_database_message_id
            << ", have_full_history = " << d->have_full_history;

  if (old_last_database_message_id < d->last_database_message_id && old_last_database_message_id < from_message_id) {
    // new messages where added to the database since the request was sent
    // they should have been received from the database, so we must repeat the request to get them
    if (from_the_end) {
      get_history_from_the_end_impl(d, true, only_local, std::move(promise));
    } else {
      get_history_impl(d, from_message_id, offset, limit, true, only_local, std::move(promise));
    }
    return;
  }

  if (messages.empty() && from_the_end && d->messages == nullptr) {
    if (d->have_full_history) {
      set_dialog_is_empty(d, "on_get_history_from_database empty");
    } else if (d->last_database_message_id.is_valid()) {
      set_dialog_first_database_message_id(d, MessageId(), "on_get_history_from_database empty");
      set_dialog_last_database_message_id(d, MessageId(), "on_get_history_from_database empty");
    }
  }

  bool have_next = false;
  bool need_update = false;
  bool need_update_dialog_pos = false;
  bool added_new_message = false;
  MessageId first_added_message_id;
  MessageId last_added_message_id;
  Message *next_message = nullptr;
  Dependencies dependencies;
  bool is_first = true;
  bool had_full_history = d->have_full_history;
  auto debug_first_database_message_id = d->first_database_message_id;
  auto debug_last_message_id = d->last_message_id;
  auto debug_last_new_message_id = d->last_new_message_id;
  auto last_received_message_id = MessageId::max();
  size_t pos = 0;
  for (auto &message_slice : messages) {
    if (!d->first_database_message_id.is_valid() && !d->have_full_history) {
      break;
    }
    auto message = parse_message(dialog_id, message_slice.message_id, message_slice.data, false);
    if (message == nullptr) {
      if (d->have_full_history) {
        d->have_full_history = false;
        d->is_empty = false;  // just in case
        on_dialog_updated(dialog_id, "drop have_full_history in on_get_history_from_database");
      }
      break;
    }
    if (message->message_id >= last_received_message_id) {
      // TODO move to ERROR
      LOG(FATAL) << "Receive " << message->message_id << " after " << last_received_message_id
                 << " from database in the history of " << dialog_id << " from " << from_message_id << " with offset "
                 << offset << ", limit " << limit << ", from_the_end = " << from_the_end;
      break;
    }
    last_received_message_id = message->message_id;

    if (message->message_id < d->first_database_message_id) {
      if (d->have_full_history) {
        LOG(ERROR) << "Have full history in the " << dialog_id << " and receive " << message->message_id
                   << " from database, but first database message is " << d->first_database_message_id;
      } else {
        break;
      }
    }
    if (!have_next && (from_the_end || (is_first && offset < -1 && message->message_id <= from_message_id)) &&
        message->message_id < d->last_message_id) {
      // last message in the dialog must be attached to the next local message
      have_next = true;
    }

    message->have_previous = false;
    message->have_next = have_next;
    message->from_database = true;

    auto old_message = get_message(d, message->message_id);
    Message *m = old_message ? old_message
                             : add_message_to_dialog(d, std::move(message), false, &need_update,
                                                     &need_update_dialog_pos, "on_get_history_from_database");
    if (m != nullptr) {
      first_added_message_id = m->message_id;
      if (!have_next) {
        last_added_message_id = m->message_id;
      }
      if (old_message == nullptr) {
        add_message_dependencies(dependencies, m);
        added_new_message = true;
      } else if (m->message_id != from_message_id) {
        added_new_message = true;
      }
      if (next_message != nullptr && !next_message->have_previous) {
        LOG_CHECK(m->message_id < next_message->message_id)
            << m->message_id << ' ' << next_message->message_id << ' ' << last_received_message_id << ' ' << dialog_id
            << ' ' << from_message_id << ' ' << offset << ' ' << limit << ' ' << from_the_end << ' ' << only_local
            << ' ' << messages.size() << ' ' << debug_first_database_message_id << ' ' << last_added_message_id << ' '
            << added_new_message << ' ' << pos << ' ' << m << ' ' << next_message << ' ' << old_message << ' '
            << to_string(get_message_object(dialog_id, m, "on_get_history_from_database"))
            << to_string(get_message_object(dialog_id, next_message, "on_get_history_from_database"));
        LOG(INFO) << "Fix have_previous for " << next_message->message_id;
        next_message->have_previous = true;
        attach_message_to_previous(
            d, next_message->message_id,
            (PSLICE() << "on_get_history_from_database 1 " << m->message_id << ' ' << from_message_id << ' ' << offset
                      << ' ' << limit << ' ' << d->first_database_message_id << ' ' << d->have_full_history << ' '
                      << pos)
                .c_str());
      }

      have_next = true;
      next_message = m;
    }
    is_first = false;
    pos++;
  }
  resolve_dependencies_force(td_, dependencies, "on_get_history_from_database");

  if (from_the_end && !last_added_message_id.is_valid() && d->first_database_message_id.is_valid() &&
      !d->have_full_history) {
    if (last_received_message_id <= d->first_database_message_id) {
      // database definitely has no messages from first_database_message_id to last_database_message_id; drop them
      set_dialog_first_database_message_id(d, MessageId(), "on_get_history_from_database 8");
      set_dialog_last_database_message_id(d, MessageId(), "on_get_history_from_database 9");
    } else {
      CHECK(last_received_message_id.is_valid());
      // if a message was received, but wasn't added, then it is likely to be already deleted
      // if it is less than d->last_database_message_id, then we can adjust d->last_database_message_id and
      // try again database search without chance to loop
      if (last_received_message_id < d->last_database_message_id) {
        set_dialog_last_database_message_id(d, last_received_message_id, "on_get_history_from_database 12");

        get_history_from_the_end_impl(d, true, only_local, std::move(promise));
        return;
      }

      if (limit > 1) {
        // we expected to have messages [first_database_message_id, last_database_message_id] in the database, but
        // received no messages or newer messages [last_received_message_id, ...], none of which can be added
        // first_database_message_id and last_database_message_id are very wrong, so it is better to drop them,
        // pretending that the database has no usable messages
        if (last_received_message_id == MessageId::max()) {
          LOG(ERROR) << "Receive no usable messages in " << dialog_id
                     << " from database from the end, but expected messages from " << d->last_database_message_id
                     << " up to " << d->first_database_message_id;
        } else {
          LOG(ERROR) << "Receive " << messages.size() << " unusable messages up to " << last_received_message_id
                     << " in " << dialog_id << " from database from the end, but expected messages from "
                     << d->last_database_message_id << " up to " << d->first_database_message_id;
        }
        set_dialog_first_database_message_id(d, MessageId(), "on_get_history_from_database 13");
        set_dialog_last_database_message_id(d, MessageId(), "on_get_history_from_database 14");
      }
    }
  }

  if (!added_new_message && !only_local && dialog_id.get_type() != DialogType::SecretChat) {
    if (from_the_end) {
      from_message_id = MessageId();
    }
    load_messages_impl(d, from_message_id, offset, limit, 1, false, std::move(promise));
    return;
  }

  if (from_the_end && last_added_message_id.is_valid()) {
    // CHECK(d->first_database_message_id.is_valid());
    // CHECK(last_added_message_id >= d->first_database_message_id);
    if ((had_full_history || d->have_full_history) && !d->last_new_message_id.is_valid() &&
        (last_added_message_id.is_server() || d->dialog_id.get_type() == DialogType::SecretChat)) {
      LOG(ERROR) << "Trying to hard fix " << d->dialog_id << " last new message to " << last_added_message_id
                 << " from on_get_history_from_database 2";
      d->last_new_message_id = last_added_message_id;
      on_dialog_updated(d->dialog_id, "on_get_history_from_database 3");
    }
    if (last_added_message_id > d->last_message_id && d->last_new_message_id.is_valid()) {
      set_dialog_last_message_id(d, last_added_message_id, "on_get_history_from_database 4");
      need_update_dialog_pos = true;
    }
    if (last_added_message_id != d->last_database_message_id && d->last_new_message_id.is_valid()) {
      auto debug_last_database_message_id = d->last_database_message_id;
      set_dialog_last_database_message_id(d, last_added_message_id, "on_get_history_from_database 5");
      if (last_added_message_id < d->first_database_message_id || !d->first_database_message_id.is_valid()) {
        CHECK(next_message != nullptr);
        LOG_CHECK(had_full_history || d->have_full_history)
            << had_full_history << ' ' << d->have_full_history << ' ' << next_message->message_id << ' '
            << last_added_message_id << ' ' << d->first_database_message_id << ' ' << debug_first_database_message_id
            << ' ' << d->last_database_message_id << ' ' << debug_last_database_message_id << ' ' << dialog_id << ' '
            << d->last_new_message_id << ' ' << debug_last_new_message_id << ' ' << d->last_message_id << ' '
            << debug_last_message_id;
        CHECK(next_message->message_id <= d->last_database_message_id);
        LOG(ERROR) << "Fix first database message in " << dialog_id << " from " << d->first_database_message_id
                   << " to " << next_message->message_id;
        set_dialog_first_database_message_id(d, next_message->message_id, "on_get_history_from_database 6");
      }
    }
  }
  if (first_added_message_id.is_valid() && first_added_message_id != d->first_database_message_id &&
      last_received_message_id < d->first_database_message_id && d->last_new_message_id.is_valid() &&
      !d->have_full_history) {
    CHECK(first_added_message_id > d->first_database_message_id);
    set_dialog_first_database_message_id(d, first_added_message_id, "on_get_history_from_database 10");
    if (d->last_database_message_id < d->first_database_message_id) {
      set_dialog_last_database_message_id(d, d->first_database_message_id, "on_get_history_from_database 11");
    }
  }

  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "on_get_history_from_database 7");
  }

  promise.set_value(Unit());
}

void MessagesManager::get_history_from_the_end(DialogId dialog_id, bool from_database, bool only_local,
                                               Promise<Unit> &&promise) {
  get_history_from_the_end_impl(get_dialog(dialog_id), from_database, only_local, std::move(promise));
}

void MessagesManager::get_history_from_the_end_impl(const Dialog *d, bool from_database, bool only_local,
                                                    Promise<Unit> &&promise) {
  CHECK(d != nullptr);
  TRY_STATUS_PROMISE(promise, G()->close_status());

  auto dialog_id = d->dialog_id;
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    // can't get history in dialogs without read access
    return promise.set_value(Unit());
  }
  if (!d->first_database_message_id.is_valid() && !d->have_full_history) {
    from_database = false;
  }
  int32 limit = MAX_GET_HISTORY;
  if (from_database && G()->parameters().use_message_db) {
    if (!promise) {
      // repair last database message ID
      limit = 10;
    }
    LOG(INFO) << "Get history from the end of " << dialog_id << " from database";
    MessagesDbMessagesQuery db_query;
    db_query.dialog_id = dialog_id;
    db_query.from_message_id = MessageId::max();
    db_query.limit = limit;
    G()->td_db()->get_messages_db_async()->get_messages(
        db_query, PromiseCreator::lambda([dialog_id, old_last_database_message_id = d->last_database_message_id,
                                          only_local, limit, actor_id = actor_id(this), promise = std::move(promise)](
                                             vector<MessagesDbDialogMessage> messages) mutable {
          send_closure(actor_id, &MessagesManager::on_get_history_from_database, dialog_id, MessageId::max(),
                       old_last_database_message_id, 0, limit, true, only_local, std::move(messages),
                       std::move(promise));
        }));
  } else {
    if (only_local || dialog_id.get_type() == DialogType::SecretChat || d->last_message_id.is_valid()) {
      // if last message is known, there are no reasons to get message history from server from the end
      promise.set_value(Unit());
      return;
    }
    if (!promise && !G()->parameters().use_message_db) {
      // repair last message ID
      limit = 10;
    }

    LOG(INFO) << "Get history from the end of " << dialog_id << " from server";
    td_->create_handler<GetHistoryQuery>(std::move(promise))
        ->send_get_from_the_end(dialog_id, d->last_new_message_id, limit);
  }
}

void MessagesManager::get_history(DialogId dialog_id, MessageId from_message_id, int32 offset, int32 limit,
                                  bool from_database, bool only_local, Promise<Unit> &&promise) {
  get_history_impl(get_dialog(dialog_id), from_message_id, offset, limit, from_database, only_local,
                   std::move(promise));
}

void MessagesManager::get_history_impl(const Dialog *d, MessageId from_message_id, int32 offset, int32 limit,
                                       bool from_database, bool only_local, Promise<Unit> &&promise) {
  CHECK(d != nullptr);
  CHECK(from_message_id.is_valid());
  TRY_STATUS_PROMISE(promise, G()->close_status());

  auto dialog_id = d->dialog_id;
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    // can't get history in dialogs without read access
    return promise.set_value(Unit());
  }
  if ((!d->first_database_message_id.is_valid() || from_message_id <= d->first_database_message_id) &&
      !d->have_full_history) {
    from_database = false;
  }
  if (from_database && G()->parameters().use_message_db) {
    LOG(INFO) << "Get history in " << dialog_id << " from " << from_message_id << " with offset " << offset
              << " and limit " << limit << " from database";
    MessagesDbMessagesQuery db_query;
    db_query.dialog_id = dialog_id;
    db_query.from_message_id = from_message_id;
    db_query.offset = offset;
    db_query.limit = limit;
    G()->td_db()->get_messages_db_async()->get_messages(
        db_query,
        PromiseCreator::lambda([dialog_id, from_message_id, old_last_database_message_id = d->last_database_message_id,
                                offset, limit, only_local, actor_id = actor_id(this),
                                promise = std::move(promise)](vector<MessagesDbDialogMessage> messages) mutable {
          send_closure(actor_id, &MessagesManager::on_get_history_from_database, dialog_id, from_message_id,
                       old_last_database_message_id, offset, limit, false, only_local, std::move(messages),
                       std::move(promise));
        }));
  } else {
    if (only_local || dialog_id.get_type() == DialogType::SecretChat) {
      return promise.set_value(Unit());
    }

    LOG(INFO) << "Get history in " << dialog_id << " from " << from_message_id << " with offset " << offset
              << " and limit " << limit << " from server";
    td_->create_handler<GetHistoryQuery>(std::move(promise))
        ->send(dialog_id, from_message_id.get_next_server_message_id(), d->last_new_message_id, offset, limit);
  }
}

void MessagesManager::load_messages(DialogId dialog_id, MessageId from_message_id, int32 offset, int32 limit,
                                    int left_tries, bool only_local, Promise<Unit> &&promise) {
  load_messages_impl(get_dialog(dialog_id), from_message_id, offset, limit, left_tries, only_local, std::move(promise));
}

void MessagesManager::load_messages_impl(const Dialog *d, MessageId from_message_id, int32 offset, int32 limit,
                                         int left_tries, bool only_local, Promise<Unit> &&promise) {
  CHECK(d != nullptr);
  CHECK(offset <= 0);
  CHECK(left_tries > 0);
  auto dialog_id = d->dialog_id;
  LOG(INFO) << "Load " << (only_local ? "local " : "") << "messages in " << dialog_id << " from " << from_message_id
            << " with offset = " << offset << " and limit = " << limit << ". " << left_tries << " tries left";
  only_local |= dialog_id.get_type() == DialogType::SecretChat;
  if (!only_local && d->have_full_history) {
    LOG(INFO) << "Have full history in " << dialog_id << ", so don't need to get chat history from server";
    only_local = true;
  }
  bool from_database = (left_tries > 2 || only_local) && G()->parameters().use_message_db;

  if (from_message_id == MessageId()) {
    get_history_from_the_end_impl(d, from_database, only_local, std::move(promise));
    return;
  }
  if ((!d->first_database_message_id.is_valid() || from_message_id <= d->first_database_message_id) &&
      !d->have_full_history) {
    from_database = false;
  }
  if (offset >= -1) {
    // get history before some server or local message
    limit = min(max(limit + offset + 1, MAX_GET_HISTORY / 2), MAX_GET_HISTORY);
    offset = -1;
  } else {
    // get history around some server or local message
    int32 messages_to_load = max(MAX_GET_HISTORY, limit);
    int32 max_add = max(messages_to_load - limit - 2, 0);
    offset -= max_add;
    limit = MAX_GET_HISTORY;
  }
  get_history_impl(d, from_message_id, offset, limit, from_database, only_local, std::move(promise));
}

vector<MessageId> MessagesManager::get_dialog_scheduled_messages(DialogId dialog_id, bool force, bool ignore_result,
                                                                 Promise<Unit> &&promise) {
  LOG(INFO) << "Get scheduled messages in " << dialog_id;
  if (G()->close_flag()) {
    promise.set_error(Global::request_aborted_error());
    return {};
  }

  Dialog *d = get_dialog_force(dialog_id, "get_dialog_scheduled_messages");
  if (d == nullptr) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return {};
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    promise.set_error(Status::Error(400, "Can't access the chat"));
    return {};
  }
  if (is_broadcast_channel(dialog_id) &&
      !td_->contacts_manager_->get_channel_status(dialog_id.get_channel_id()).can_post_messages()) {
    promise.set_error(Status::Error(400, "Not enough rights to get scheduled messages"));
    return {};
  }

  if (!d->has_loaded_scheduled_messages_from_database) {
    load_dialog_scheduled_messages(dialog_id, true, 0, std::move(promise));
    return {};
  }

  vector<MessageId> message_ids;
  find_old_messages(d->scheduled_messages.get(),
                    MessageId(ScheduledServerMessageId(), std::numeric_limits<int32>::max(), true), message_ids);
  std::reverse(message_ids.begin(), message_ids.end());

  if (G()->parameters().use_message_db) {
    bool has_scheduled_database_messages = false;
    for (auto &message_id : message_ids) {
      CHECK(message_id.is_valid_scheduled());
      if (!message_id.is_yet_unsent()) {
        has_scheduled_database_messages = true;
        break;
      }
    }
    set_dialog_has_scheduled_database_messages(d->dialog_id, has_scheduled_database_messages);
  }

  if (d->scheduled_messages_sync_generation != scheduled_messages_sync_generation_) {
    vector<uint64> numbers;
    for (auto &message_id : message_ids) {
      if (!message_id.is_scheduled_server()) {
        continue;
      }

      numbers.push_back(message_id.get_scheduled_server_message_id().get());
      const Message *m = get_message(d, message_id);
      CHECK(m != nullptr);
      CHECK(m->message_id.get_scheduled_server_message_id() == message_id.get_scheduled_server_message_id());
      numbers.push_back(m->edit_date);
      numbers.push_back(m->date);
    }
    auto hash = get_vector_hash(numbers);

    if (!force && (d->has_scheduled_server_messages ||
                   (d->scheduled_messages_sync_generation == 0 && !G()->parameters().use_message_db))) {
      load_dialog_scheduled_messages(dialog_id, false, hash, std::move(promise));
      return {};
    }
    load_dialog_scheduled_messages(dialog_id, false, hash, Promise<Unit>());
  }

  if (!ignore_result) {
    d->sent_scheduled_messages = true;
  }

  promise.set_value(Unit());
  return message_ids;
}

void MessagesManager::load_dialog_scheduled_messages(DialogId dialog_id, bool from_database, int64 hash,
                                                     Promise<Unit> &&promise) {
  if (G()->parameters().use_message_db && from_database) {
    LOG(INFO) << "Load scheduled messages from database in " << dialog_id;
    auto &queries = load_scheduled_messages_from_database_queries_[dialog_id];
    queries.push_back(std::move(promise));
    if (queries.size() == 1) {
      G()->td_db()->get_messages_db_async()->get_scheduled_messages(
          dialog_id, 1000,
          PromiseCreator::lambda([dialog_id, actor_id = actor_id(this)](vector<MessagesDbDialogMessage> messages) {
            send_closure(actor_id, &MessagesManager::on_get_scheduled_messages_from_database, dialog_id,
                         std::move(messages));
          }));
    }
  } else {
    td_->create_handler<GetAllScheduledMessagesQuery>(std::move(promise))
        ->send(dialog_id, hash, scheduled_messages_sync_generation_);
  }
}

void MessagesManager::on_get_scheduled_messages_from_database(DialogId dialog_id,
                                                              vector<MessagesDbDialogMessage> &&messages) {
  if (G()->close_flag()) {
    auto it = load_scheduled_messages_from_database_queries_.find(dialog_id);
    CHECK(it != load_scheduled_messages_from_database_queries_.end());
    CHECK(!it->second.empty());
    auto promises = std::move(it->second);
    load_scheduled_messages_from_database_queries_.erase(it);

    for (auto &promise : promises) {
      promise.set_error(Global::request_aborted_error());
    }
    return;
  }
  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  d->has_loaded_scheduled_messages_from_database = true;

  LOG(INFO) << "Receive " << messages.size() << " scheduled messages from database in " << dialog_id;

  Dependencies dependencies;
  vector<MessageId> added_message_ids;
  for (auto &message_slice : messages) {
    auto message = parse_message(dialog_id, message_slice.message_id, message_slice.data, true);
    if (message == nullptr) {
      continue;
    }
    message->from_database = true;

    if (get_message(d, message->message_id) != nullptr) {
      continue;
    }

    bool need_update = false;
    Message *m = add_scheduled_message_to_dialog(d, std::move(message), false, &need_update,
                                                 "on_get_scheduled_messages_from_database");
    if (m != nullptr) {
      add_message_dependencies(dependencies, m);
      added_message_ids.push_back(m->message_id);
    }
  }
  resolve_dependencies_force(td_, dependencies, "on_get_scheduled_messages_from_database");

  // for (auto message_id : added_message_ids) {
  //   send_update_new_message(d, get_message(d, message_id));
  // }
  send_update_chat_has_scheduled_messages(d, false);

  auto it = load_scheduled_messages_from_database_queries_.find(dialog_id);
  CHECK(it != load_scheduled_messages_from_database_queries_.end());
  CHECK(!it->second.empty());
  auto promises = std::move(it->second);
  load_scheduled_messages_from_database_queries_.erase(it);

  for (auto &promise : promises) {
    promise.set_value(Unit());
  }
}

void MessagesManager::get_message_public_forwards(FullMessageId full_message_id, string offset, int32 limit,
                                                  Promise<td_api::object_ptr<td_api::foundMessages>> &&promise) {
  auto dc_id_promise = PromiseCreator::lambda([actor_id = actor_id(this), full_message_id, offset = std::move(offset),
                                               limit, promise = std::move(promise)](Result<DcId> r_dc_id) mutable {
    if (r_dc_id.is_error()) {
      return promise.set_error(r_dc_id.move_as_error());
    }
    send_closure(actor_id, &MessagesManager::send_get_message_public_forwards_query, r_dc_id.move_as_ok(),
                 full_message_id, std::move(offset), limit, std::move(promise));
  });
  td_->contacts_manager_->get_channel_statistics_dc_id(full_message_id.get_dialog_id(), false,
                                                       std::move(dc_id_promise));
}

void MessagesManager::send_get_message_public_forwards_query(
    DcId dc_id, FullMessageId full_message_id, string offset, int32 limit,
    Promise<td_api::object_ptr<td_api::foundMessages>> &&promise) {
  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id, "send_get_message_public_forwards_query");
  if (d == nullptr) {
    return promise.set_error(Status::Error(400, "Chat not found"));
  }

  const Message *m = get_message_force(d, full_message_id.get_message_id(), "send_get_message_public_forwards_query");
  if (m == nullptr) {
    return promise.set_error(Status::Error(400, "Message not found"));
  }

  if (m->view_count == 0 || m->forward_info != nullptr || m->had_forward_info || m->message_id.is_scheduled() ||
      !m->message_id.is_server()) {
    return promise.set_error(Status::Error(400, "Message forwards are inaccessible"));
  }

  if (limit <= 0) {
    return promise.set_error(Status::Error(400, "Parameter limit must be positive"));
  }
  if (limit > MAX_SEARCH_MESSAGES) {
    limit = MAX_SEARCH_MESSAGES;
  }

  int32 offset_date = std::numeric_limits<int32>::max();
  DialogId offset_dialog_id;
  ServerMessageId offset_message_id;

  if (!offset.empty()) {
    auto parts = full_split(offset, ',');
    if (parts.size() != 3) {
      return promise.set_error(Status::Error(400, "Invalid offset specified"));
    }
    auto r_offset_date = to_integer_safe<int32>(parts[0]);
    auto r_offset_dialog_id = to_integer_safe<int64>(parts[1]);
    auto r_offset_message_id = to_integer_safe<int32>(parts[2]);
    if (r_offset_date.is_error() || r_offset_dialog_id.is_error() || r_offset_message_id.is_error()) {
      return promise.set_error(Status::Error(400, "Invalid offset specified"));
    }

    offset_date = r_offset_date.ok();
    offset_dialog_id = DialogId(r_offset_dialog_id.ok());
    offset_message_id = ServerMessageId(r_offset_message_id.ok());
  }

  td_->create_handler<GetMessagePublicForwardsQuery>(std::move(promise))
      ->send(dc_id, full_message_id, offset_date, offset_dialog_id, offset_message_id, limit);
}

Result<int32> MessagesManager::get_message_schedule_date(
    td_api::object_ptr<td_api::MessageSchedulingState> &&scheduling_state) {
  if (scheduling_state == nullptr) {
    return 0;
  }

  switch (scheduling_state->get_id()) {
    case td_api::messageSchedulingStateSendWhenOnline::ID: {
      auto send_date = SCHEDULE_WHEN_ONLINE_DATE;
      return send_date;
    }
    case td_api::messageSchedulingStateSendAtDate::ID: {
      auto send_at_date = td_api::move_object_as<td_api::messageSchedulingStateSendAtDate>(scheduling_state);
      auto send_date = send_at_date->send_date_;
      if (send_date <= 0) {
        return Status::Error(400, "Invalid send date specified");
      }
      if (send_date <= G()->unix_time() + 10) {
        return 0;
      }
      if (send_date - G()->unix_time() > 367 * 86400) {
        return Status::Error(400, "Send date is too far in the future");
      }
      return send_date;
    }
    default:
      UNREACHABLE();
      return 0;
  }
}

tl_object_ptr<td_api::MessageSendingState> MessagesManager::get_message_sending_state_object(const Message *m) const {
  CHECK(m != nullptr);
  if (m->message_id.is_yet_unsent()) {
    return td_api::make_object<td_api::messageSendingStatePending>();
  }
  if (m->is_failed_to_send) {
    return td_api::make_object<td_api::messageSendingStateFailed>(
        m->send_error_code, m->send_error_message, can_resend_message(m), max(m->try_resend_at - Time::now(), 0.0));
  }
  return nullptr;
}

tl_object_ptr<td_api::MessageSchedulingState> MessagesManager::get_message_scheduling_state_object(int32 send_date) {
  if (send_date == SCHEDULE_WHEN_ONLINE_DATE) {
    return td_api::make_object<td_api::messageSchedulingStateSendWhenOnline>();
  }
  return td_api::make_object<td_api::messageSchedulingStateSendAtDate>(send_date);
}

tl_object_ptr<td_api::message> MessagesManager::get_message_object(FullMessageId full_message_id, const char *source) {
  return get_message_object(full_message_id.get_dialog_id(), get_message_force(full_message_id, source), source);
}

tl_object_ptr<td_api::message> MessagesManager::get_message_object(DialogId dialog_id, const Message *m,
                                                                   const char *source, bool for_event_log) const {
  if (m == nullptr) {
    return nullptr;
  }
  LOG_CHECK(have_dialog(dialog_id)) << source;

  m->is_update_sent = true;

  auto sending_state = get_message_sending_state_object(m);

  if (for_event_log) {
    CHECK(m->message_id.is_server());
    CHECK(sending_state == nullptr);
  }

  bool can_delete = can_delete_message(dialog_id, m);
  bool is_scheduled = m->message_id.is_scheduled();
  DialogId my_dialog_id = get_my_dialog_id();
  bool can_delete_for_self = false;
  bool can_delete_for_all_users = can_delete && can_revoke_message(dialog_id, m);
  if (can_delete) {
    switch (dialog_id.get_type()) {
      case DialogType::User:
      case DialogType::Chat:
        // TODO allow to delete yet unsent message just for self
        can_delete_for_self = !m->message_id.is_yet_unsent() || dialog_id == my_dialog_id;
        break;
      case DialogType::Channel:
      case DialogType::SecretChat:
        can_delete_for_self = !can_delete_for_all_users;
        break;
      case DialogType::None:
      default:
        UNREACHABLE();
    }
  }
  if (for_event_log) {
    can_delete_for_self = false;
    can_delete_for_all_users = false;
  } else if (is_scheduled) {
    can_delete_for_self = (dialog_id == my_dialog_id);
    can_delete_for_all_users = !can_delete_for_self;
  }

  bool is_outgoing = m->is_outgoing;
  if (dialog_id == my_dialog_id) {
    // in Saved Messages all non-forwarded messages must be outgoing
    // a forwarded message is outgoing, only if it doesn't have from_dialog_id and its sender isn't hidden
    // i.e. a message is incoming only if it's a forwarded message with known from_dialog_id or with a hidden sender
    auto forward_info = m->forward_info.get();
    is_outgoing = is_scheduled || forward_info == nullptr ||
                  (!forward_info->from_dialog_id.is_valid() && !is_forward_info_sender_hidden(forward_info));
  }

  int32 ttl = m->ttl;
  double ttl_expires_in = 0;
  if (!for_event_log) {
    if (m->ttl_expires_at != 0) {
      ttl_expires_in = clamp(m->ttl_expires_at - Time::now(), 1e-3, ttl - 1e-3);
    } else {
      ttl_expires_in = ttl;
    }
    if (ttl == 0 && m->ttl_period != 0) {
      ttl = m->ttl_period;
      ttl_expires_in = clamp(m->date + m->ttl_period - G()->server_time(), 1e-3, ttl - 1e-3);
    }
  } else {
    ttl = 0;
  }
  auto sender = get_message_sender_object_const(m->sender_user_id, m->sender_dialog_id, source);
  auto scheduling_state = is_scheduled ? get_message_scheduling_state_object(m->date) : nullptr;
  auto forward_info = get_message_forward_info_object(m->forward_info);
  auto interaction_info = get_message_interaction_info_object(dialog_id, m);
  auto can_be_edited = for_event_log ? false : can_edit_message(dialog_id, m, false, td_->auth_manager_->is_bot());
  auto can_be_forwarded = for_event_log ? false : can_forward_message(dialog_id, m);
  auto can_get_statistics = for_event_log ? false : can_get_message_statistics(dialog_id, m);
  auto can_get_message_thread = for_event_log ? false : get_top_thread_full_message_id(dialog_id, m).is_ok();
  auto can_get_viewers = for_event_log ? false : can_get_message_viewers(dialog_id, m).is_ok();
  auto can_get_media_timestamp_links = for_event_log ? false : can_get_media_timestamp_link(dialog_id, m).is_ok();
  auto via_bot_user_id = td_->contacts_manager_->get_user_id_object(m->via_bot_user_id, "via_bot_user_id");
  auto media_album_id = for_event_log ? static_cast<int64>(0) : m->media_album_id;
  auto reply_to_message_id = for_event_log ? static_cast<int64>(0) : m->reply_to_message_id.get();
  auto reply_in_dialog_id =
      reply_to_message_id == 0 ? DialogId() : (m->reply_in_dialog_id.is_valid() ? m->reply_in_dialog_id : dialog_id);
  auto top_thread_message_id = for_event_log || is_scheduled ? static_cast<int64>(0) : m->top_thread_message_id.get();
  auto contains_unread_mention = for_event_log ? false : m->contains_unread_mention;
  auto date = is_scheduled ? 0 : m->date;
  auto edit_date = m->hide_edit_date ? 0 : m->edit_date;
  auto is_pinned = for_event_log || is_scheduled ? false : m->is_pinned;
  auto has_timestamped_media = for_event_log || reply_to_message_id == 0 || m->max_own_media_timestamp >= 0;
  auto reply_markup = get_reply_markup_object(m->reply_markup);

  auto live_location_date = m->is_failed_to_send ? 0 : m->date;
  auto skip_bot_commands = for_event_log ? true : need_skip_bot_commands(dialog_id, m);
  auto max_media_timestamp =
      for_event_log ? get_message_own_max_media_timestamp(m) : get_message_max_media_timestamp(m);
  auto content = get_message_content_object(m->content.get(), td_, dialog_id, live_location_date, m->is_content_secret,
                                            skip_bot_commands, max_media_timestamp);
  return make_tl_object<td_api::message>(
      m->message_id.get(), std::move(sender), dialog_id.get(), std::move(sending_state), std::move(scheduling_state),
      is_outgoing, is_pinned, can_be_edited, can_be_forwarded, can_delete_for_self, can_delete_for_all_users,
      can_get_statistics, can_get_message_thread, can_get_viewers, can_get_media_timestamp_links, has_timestamped_media,
      m->is_channel_post, contains_unread_mention, date, edit_date, std::move(forward_info),
      std::move(interaction_info), reply_in_dialog_id.get(), reply_to_message_id, top_thread_message_id, ttl,
      ttl_expires_in, via_bot_user_id, m->author_signature, media_album_id,
      get_restriction_reason_description(m->restriction_reasons), std::move(content), std::move(reply_markup));
}

tl_object_ptr<td_api::messages> MessagesManager::get_messages_object(int32 total_count, DialogId dialog_id,
                                                                     const vector<MessageId> &message_ids,
                                                                     bool skip_not_found, const char *source) {
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  auto message_objects = transform(message_ids, [this, dialog_id, d, source](MessageId message_id) {
    return get_message_object(dialog_id, get_message_force(d, message_id, source), source);
  });
  return get_messages_object(total_count, std::move(message_objects), skip_not_found);
}

tl_object_ptr<td_api::messages> MessagesManager::get_messages_object(int32 total_count,
                                                                     const vector<FullMessageId> &full_message_ids,
                                                                     bool skip_not_found, const char *source) {
  auto message_objects = transform(full_message_ids, [this, source](FullMessageId full_message_id) {
    return get_message_object(full_message_id, source);
  });
  return get_messages_object(total_count, std::move(message_objects), skip_not_found);
}

tl_object_ptr<td_api::messages> MessagesManager::get_messages_object(int32 total_count,
                                                                     vector<tl_object_ptr<td_api::message>> &&messages,
                                                                     bool skip_not_found) {
  auto message_count = narrow_cast<int32>(messages.size());
  if (total_count < message_count) {
    if (total_count != -1) {
      LOG(ERROR) << "Have wrong total_count = " << total_count << ", while having " << message_count << " messages";
    }
    total_count = message_count;
  }
  if (skip_not_found && td::remove(messages, nullptr)) {
    total_count -= message_count - static_cast<int32>(messages.size());
  }
  return td_api::make_object<td_api::messages>(total_count, std::move(messages));
}

bool MessagesManager::is_anonymous_administrator(DialogId dialog_id, string *author_signature) const {
  CHECK(dialog_id.is_valid());

  if (is_broadcast_channel(dialog_id)) {
    return true;
  }

  if (td_->auth_manager_->is_bot()) {
    return false;
  }

  if (dialog_id.get_type() != DialogType::Channel) {
    return false;
  }

  auto status = td_->contacts_manager_->get_channel_status(dialog_id.get_channel_id());
  if (!status.is_anonymous()) {
    return false;
  }

  if (author_signature != nullptr) {
    *author_signature = status.get_rank();
  }
  return true;
}

int64 MessagesManager::generate_new_random_id() {
  int64 random_id;
  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || being_sent_messages_.find(random_id) != being_sent_messages_.end());
  return random_id;
}

unique_ptr<MessagesManager::Message> MessagesManager::create_message_to_send(
    Dialog *d, MessageId top_thread_message_id, MessageId reply_to_message_id, const MessageSendOptions &options,
    unique_ptr<MessageContent> &&content, bool suppress_reply_info, unique_ptr<MessageForwardInfo> forward_info,
    bool is_copy) const {
  CHECK(d != nullptr);
  CHECK(!reply_to_message_id.is_scheduled());
  CHECK(content != nullptr);

  bool is_scheduled = options.schedule_date != 0;
  DialogId dialog_id = d->dialog_id;

  auto dialog_type = dialog_id.get_type();
  auto my_id = td_->contacts_manager_->get_my_id();

  auto m = make_unique<Message>();
  bool is_channel_post = is_broadcast_channel(dialog_id);
  if (is_channel_post) {
    // sender of the post can be hidden
    if (!is_scheduled && td_->contacts_manager_->get_channel_sign_messages(dialog_id.get_channel_id())) {
      m->author_signature = td_->contacts_manager_->get_user_title(my_id);
    }
    m->sender_dialog_id = dialog_id;
  } else {
    if (is_anonymous_administrator(dialog_id, &m->author_signature)) {
      m->sender_dialog_id = dialog_id;
    } else {
      m->sender_user_id = my_id;
    }
  }
  m->send_date = G()->unix_time();
  m->date = is_scheduled ? options.schedule_date : m->send_date;
  m->reply_to_message_id = reply_to_message_id;
  if (!is_scheduled) {
    m->top_thread_message_id = top_thread_message_id;
    if (reply_to_message_id.is_valid()) {
      const Message *reply_m = get_message(d, reply_to_message_id);
      if (reply_m != nullptr && reply_m->top_thread_message_id.is_valid()) {
        m->top_thread_message_id = reply_m->top_thread_message_id;
      }
    }
  }
  m->is_channel_post = is_channel_post;
  m->is_outgoing = is_scheduled || dialog_id != DialogId(my_id);
  m->from_background = options.from_background;
  m->view_count = is_channel_post && !is_scheduled ? 1 : 0;
  m->forward_count = 0;
  if ([&] {
        if (suppress_reply_info) {
          return false;
        }
        if (is_scheduled) {
          return false;
        }
        if (dialog_type != DialogType::Channel) {
          return false;
        }
        if (td_->auth_manager_->is_bot()) {
          return false;
        }
        if (is_channel_post) {
          return td_->contacts_manager_->get_channel_has_linked_channel(dialog_id.get_channel_id());
        }
        return !reply_to_message_id.is_valid();
      }()) {
    m->reply_info.reply_count = 0;
    if (is_channel_post) {
      auto linked_channel_id = td_->contacts_manager_->get_channel_linked_channel_id(dialog_id.get_channel_id());
      if (linked_channel_id.is_valid()) {
        m->reply_info.is_comment = true;
        m->reply_info.channel_id = linked_channel_id;
      }
    }
  }
  m->content = std::move(content);
  m->forward_info = std::move(forward_info);
  m->is_copy = is_copy || m->forward_info != nullptr;

  if (td_->auth_manager_->is_bot() || options.disable_notification ||
      G()->shared_config().get_option_boolean("ignore_default_disable_notification")) {
    m->disable_notification = options.disable_notification;
  } else {
    m->disable_notification = d->notification_settings.silent_send_message;
  }

  if (dialog_type == DialogType::SecretChat) {
    CHECK(!is_scheduled);
    m->ttl = td_->contacts_manager_->get_secret_chat_ttl(dialog_id.get_secret_chat_id());
    if (is_service_message_content(m->content->get_type())) {
      m->ttl = 0;
    }
    m->is_content_secret = is_secret_message_content(m->ttl, m->content->get_type());
    if (reply_to_message_id.is_valid()) {
      // the message was forcely preloaded in get_reply_to_message_id
      auto *reply_to_message = get_message(d, reply_to_message_id);
      if (reply_to_message != nullptr) {
        m->reply_to_random_id = reply_to_message->random_id;
      } else {
        m->reply_to_message_id = MessageId();
      }
    }
  }
  return m;
}

MessagesManager::Message *MessagesManager::get_message_to_send(
    Dialog *d, MessageId top_thread_message_id, MessageId reply_to_message_id, const MessageSendOptions &options,
    unique_ptr<MessageContent> &&content, bool *need_update_dialog_pos, bool suppress_reply_info,
    unique_ptr<MessageForwardInfo> forward_info, bool is_copy) {
  d->was_opened = true;

  auto message = create_message_to_send(d, top_thread_message_id, reply_to_message_id, options, std::move(content),
                                        suppress_reply_info, std::move(forward_info), is_copy);

  MessageId message_id = options.schedule_date != 0 ? get_next_yet_unsent_scheduled_message_id(d, options.schedule_date)
                                                    : get_next_yet_unsent_message_id(d);
  set_message_id(message, message_id);

  message->have_previous = true;
  message->have_next = true;

  message->random_id = generate_new_random_id();

  bool need_update = false;
  CHECK(have_input_peer(d->dialog_id, AccessRights::Read));
  auto result =
      add_message_to_dialog(d, std::move(message), true, &need_update, need_update_dialog_pos, "send message");
  LOG_CHECK(result != nullptr) << message_id << " " << debug_add_message_to_dialog_fail_reason_;
  if (result->message_id.is_scheduled()) {
    send_update_chat_has_scheduled_messages(d, false);
  }
  return result;
}

int64 MessagesManager::begin_send_message(DialogId dialog_id, const Message *m) {
  LOG(INFO) << "Begin to send " << FullMessageId(dialog_id, m->message_id) << " with random_id = " << m->random_id;
  CHECK(m->random_id != 0 && being_sent_messages_.find(m->random_id) == being_sent_messages_.end());
  CHECK(m->message_id.is_yet_unsent());
  being_sent_messages_[m->random_id] = FullMessageId(dialog_id, m->message_id);
  return m->random_id;
}

Status MessagesManager::can_send_message(DialogId dialog_id) const {
  if (!have_input_peer(dialog_id, AccessRights::Write)) {
    return Status::Error(400, "Have no write access to the chat");
  }

  if (dialog_id.get_type() == DialogType::Channel) {
    auto channel_id = dialog_id.get_channel_id();
    auto channel_type = td_->contacts_manager_->get_channel_type(channel_id);
    auto channel_status = td_->contacts_manager_->get_channel_permissions(channel_id);

    switch (channel_type) {
      case ContactsManager::ChannelType::Unknown:
      case ContactsManager::ChannelType::Megagroup:
        if (!channel_status.can_send_messages()) {
          return Status::Error(400, "Have no rights to send a message");
        }
        break;
      case ContactsManager::ChannelType::Broadcast: {
        if (!channel_status.can_post_messages()) {
          return Status::Error(400, "Need administrator rights in the channel chat");
        }
        break;
      }
      default:
        UNREACHABLE();
    }
  }
  return Status::OK();
}

MessageId MessagesManager::get_persistent_message_id(const Dialog *d, MessageId message_id) {
  if (!message_id.is_valid() && !message_id.is_valid_scheduled()) {
    return MessageId();
  }
  if (message_id.is_yet_unsent()) {
    // it is possible that user tries to do something with an already sent message by its temporary id
    // we need to use real message in this case and transparently replace message_id
    auto it = d->yet_unsent_message_id_to_persistent_message_id.find(message_id);
    if (it != d->yet_unsent_message_id_to_persistent_message_id.end()) {
      return it->second;
    }
  }

  return message_id;
}

MessageId MessagesManager::get_reply_to_message_id(Dialog *d, MessageId top_thread_message_id, MessageId message_id,
                                                   bool for_draft) {
  CHECK(d != nullptr);
  if (!message_id.is_valid()) {
    if (!for_draft && message_id == MessageId() && top_thread_message_id.is_valid() &&
        top_thread_message_id.is_server() &&
        get_message_force(d, top_thread_message_id, "get_reply_to_message_id 1") != nullptr) {
      return top_thread_message_id;
    }
    return MessageId();
  }
  message_id = get_persistent_message_id(d, message_id);
  const Message *m = get_message_force(d, message_id, "get_reply_to_message_id 2");
  if (m == nullptr || m->message_id.is_yet_unsent() ||
      (m->message_id.is_local() && d->dialog_id.get_type() != DialogType::SecretChat)) {
    if (message_id.is_server() && d->dialog_id.get_type() != DialogType::SecretChat &&
        message_id > d->last_new_message_id && message_id <= d->max_notification_message_id) {
      // allow to reply yet unreceived server message
      return message_id;
    }
    if (!for_draft && top_thread_message_id.is_valid() && top_thread_message_id.is_server() &&
        get_message_force(d, top_thread_message_id, "get_reply_to_message_id 3") != nullptr) {
      return top_thread_message_id;
    }

    // TODO local replies to local messages can be allowed
    // TODO replies to yet unsent messages can be allowed with special handling of them on application restart
    return MessageId();
  }
  return m->message_id;
}

void MessagesManager::fix_server_reply_to_message_id(DialogId dialog_id, MessageId message_id,
                                                     DialogId reply_in_dialog_id, MessageId &reply_to_message_id) {
  CHECK(!reply_to_message_id.is_scheduled());
  if (!reply_to_message_id.is_valid()) {
    if (reply_to_message_id != MessageId()) {
      LOG(ERROR) << "Receive reply to " << reply_to_message_id << " for " << message_id << " in " << dialog_id;
      reply_to_message_id = MessageId();
    }
    return;
  }

  if (!message_id.is_scheduled() && !reply_in_dialog_id.is_valid() && reply_to_message_id >= message_id) {
    if (!can_overflow_message_id(dialog_id)) {
      LOG(ERROR) << "Receive reply to wrong " << reply_to_message_id << " in " << message_id << " in " << dialog_id;
    }
    reply_to_message_id = MessageId();
  }
}

vector<FileId> MessagesManager::get_message_file_ids(const Message *m) const {
  CHECK(m != nullptr);
  return get_message_content_file_ids(m->content.get(), td_);
}

void MessagesManager::cancel_upload_message_content_files(const MessageContent *content) {
  auto file_id = get_message_content_upload_file_id(content);
  // always cancel file upload, it should be a no-op in the worst case
  if (being_uploaded_files_.erase(file_id) || file_id.is_valid()) {
    cancel_upload_file(file_id);
  }
  file_id = get_message_content_thumbnail_file_id(content, td_);
  if (being_uploaded_thumbnails_.erase(file_id) || file_id.is_valid()) {
    cancel_upload_file(file_id);
  }
}

void MessagesManager::cancel_upload_file(FileId file_id) {
  // send the request later so they doesn't interfere with other actions
  // for example merge, supposed to happen soon, can auto-cancel the upload
  LOG(INFO) << "Cancel upload of file " << file_id;
  send_closure_later(G()->file_manager(), &FileManager::cancel_upload, file_id);
}

void MessagesManager::cancel_send_message_query(DialogId dialog_id, Message *m) {
  CHECK(m != nullptr);
  CHECK(m->content != nullptr);
  CHECK(m->message_id.is_valid() || m->message_id.is_valid_scheduled());
  CHECK(m->message_id.is_yet_unsent());
  LOG(INFO) << "Cancel send message query for " << m->message_id;

  cancel_upload_message_content_files(m->content.get());

  CHECK(m->edited_content == nullptr);

  if (!m->send_query_ref.empty()) {
    LOG(INFO) << "Cancel send query for " << m->message_id;
    cancel_query(m->send_query_ref);
    m->send_query_ref = NetQueryRef();
  }

  if (m->send_message_log_event_id != 0) {
    LOG(INFO) << "Delete send message log event for " << m->message_id;
    binlog_erase(G()->td_db()->get_binlog(), m->send_message_log_event_id);
    m->send_message_log_event_id = 0;
  }

  if (m->reply_to_message_id.is_valid() && !m->reply_to_message_id.is_yet_unsent()) {
    auto it = replied_by_yet_unsent_messages_.find({dialog_id, m->reply_to_message_id});
    CHECK(it != replied_by_yet_unsent_messages_.end());
    it->second--;
    CHECK(it->second >= 0);
    if (it->second == 0) {
      replied_by_yet_unsent_messages_.erase(it);
    }
  }

  if (m->media_album_id != 0) {
    send_closure_later(actor_id(this), &MessagesManager::on_upload_message_media_finished, m->media_album_id, dialog_id,
                       m->message_id, Status::OK());
  }

  if (!m->message_id.is_scheduled() && G()->parameters().use_file_db &&
      !m->is_copy) {  // ResourceManager::Mode::Baseline
    auto queue_id = get_sequence_dispatcher_id(dialog_id, m->content->get_type());
    if (queue_id & 1) {
      auto queue_it = yet_unsent_media_queues_.find(queue_id);
      if (queue_it != yet_unsent_media_queues_.end()) {
        auto &queue = queue_it->second;
        LOG(INFO) << "Delete " << m->message_id << " from queue " << queue_id;
        if (queue.erase(m->message_id) != 0) {
          if (queue.empty()) {
            yet_unsent_media_queues_.erase(queue_it);
          } else {
            // send later, because do_delete_all_dialog_messages can be called right now
            send_closure_later(actor_id(this), &MessagesManager::on_yet_unsent_media_queue_updated, dialog_id);
          }
        }
      }
    }
  }
}

void MessagesManager::cancel_send_deleted_message(DialogId dialog_id, Message *m, bool is_permanently_deleted) {
  CHECK(m != nullptr);
  if (m->message_id.is_yet_unsent()) {
    cancel_send_message_query(dialog_id, m);
  } else if (is_permanently_deleted || !m->message_id.is_scheduled()) {
    cancel_edit_message_media(dialog_id, m, "Message was deleted");
  }
}

bool MessagesManager::is_message_auto_read(DialogId dialog_id, bool is_outgoing) const {
  switch (dialog_id.get_type()) {
    case DialogType::User: {
      auto user_id = dialog_id.get_user_id();
      if (user_id == td_->contacts_manager_->get_my_id()) {
        return true;
      }
      if (is_outgoing && td_->contacts_manager_->is_user_bot(user_id) &&
          !td_->contacts_manager_->is_user_support(user_id)) {
        return true;
      }
      return false;
    }
    case DialogType::Chat:
      // TODO auto_read message content and messages sent to group with bots only
      return false;
    case DialogType::Channel:
      return is_outgoing && is_broadcast_channel(dialog_id);
    case DialogType::SecretChat:
      return false;
    case DialogType::None:
      return false;
    default:
      UNREACHABLE();
      return false;
  }
}

void MessagesManager::add_message_dependencies(Dependencies &dependencies, const Message *m) {
  dependencies.user_ids.insert(m->sender_user_id);
  add_dialog_and_dependencies(dependencies, m->sender_dialog_id);
  add_dialog_and_dependencies(dependencies, m->reply_in_dialog_id);
  add_dialog_and_dependencies(dependencies, m->real_forward_from_dialog_id);
  dependencies.user_ids.insert(m->via_bot_user_id);
  if (m->forward_info != nullptr) {
    dependencies.user_ids.insert(m->forward_info->sender_user_id);
    add_dialog_and_dependencies(dependencies, m->forward_info->sender_dialog_id);
    add_dialog_and_dependencies(dependencies, m->forward_info->from_dialog_id);
  }
  for (auto recent_replier_dialog_id : m->reply_info.recent_replier_dialog_ids) {
    add_message_sender_dependencies(dependencies, recent_replier_dialog_id);
  }
  add_message_content_dependencies(dependencies, m->content.get());
}

class MessagesManager::SendMessageLogEvent {
 public:
  DialogId dialog_id;
  const Message *m_in;
  unique_ptr<Message> m_out;

  SendMessageLogEvent() : dialog_id(), m_in(nullptr) {
  }

  SendMessageLogEvent(DialogId dialog_id, const Message *m) : dialog_id(dialog_id), m_in(m) {
  }

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id, storer);
    td::store(*m_in, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id, parser);
    td::parse(m_out, parser);
  }
};

Result<td_api::object_ptr<td_api::message>> MessagesManager::send_message(
    DialogId dialog_id, MessageId top_thread_message_id, MessageId reply_to_message_id,
    tl_object_ptr<td_api::messageSendOptions> &&options, tl_object_ptr<td_api::ReplyMarkup> &&reply_markup,
    tl_object_ptr<td_api::InputMessageContent> &&input_message_content) {
  if (input_message_content == nullptr) {
    return Status::Error(400, "Can't send message without content");
  }

  Dialog *d = get_dialog_force(dialog_id, "send_message");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }

  LOG(INFO) << "Begin to send message to " << dialog_id << " in reply to " << reply_to_message_id;

  reply_to_message_id = get_reply_to_message_id(d, top_thread_message_id, reply_to_message_id, false);

  if (input_message_content->get_id() == td_api::inputMessageForwarded::ID) {
    auto input_message = td_api::move_object_as<td_api::inputMessageForwarded>(input_message_content);
    TRY_RESULT(copy_options, process_message_copy_options(dialog_id, std::move(input_message->copy_options_)));
    copy_options.top_thread_message_id = top_thread_message_id;
    copy_options.reply_to_message_id = reply_to_message_id;
    TRY_RESULT_ASSIGN(copy_options.reply_markup, get_dialog_reply_markup(dialog_id, std::move(reply_markup)));
    return forward_message(dialog_id, DialogId(input_message->from_chat_id_), MessageId(input_message->message_id_),
                           std::move(options), input_message->in_game_share_, std::move(copy_options));
  }

  TRY_STATUS(can_send_message(dialog_id));
  TRY_RESULT(message_reply_markup, get_dialog_reply_markup(dialog_id, std::move(reply_markup)));
  TRY_RESULT(message_content, process_input_message_content(dialog_id, std::move(input_message_content)));
  TRY_RESULT(message_send_options, process_message_send_options(dialog_id, std::move(options)));
  TRY_STATUS(can_use_message_send_options(message_send_options, message_content));
  TRY_STATUS(can_use_top_thread_message_id(d, top_thread_message_id, reply_to_message_id));

  // there must be no errors after get_message_to_send call

  bool need_update_dialog_pos = false;
  Message *m = get_message_to_send(d, top_thread_message_id, reply_to_message_id, message_send_options,
                                   dup_message_content(td_, dialog_id, message_content.content.get(),
                                                       MessageContentDupType::Send, MessageCopyOptions()),
                                   &need_update_dialog_pos, false, nullptr, message_content.via_bot_user_id.is_valid());
  m->reply_markup = std::move(message_reply_markup);
  m->via_bot_user_id = message_content.via_bot_user_id;
  m->disable_web_page_preview = message_content.disable_web_page_preview;
  m->clear_draft = message_content.clear_draft;
  if (message_content.ttl > 0) {
    m->ttl = message_content.ttl;
    m->is_content_secret = is_secret_message_content(m->ttl, m->content->get_type());
  }
  m->send_emoji = std::move(message_content.emoji);

  if (message_content.clear_draft) {
    if (top_thread_message_id.is_valid()) {
      set_dialog_draft_message(dialog_id, top_thread_message_id, nullptr).ignore();
    } else {
      update_dialog_draft_message(d, nullptr, false, !need_update_dialog_pos);
    }
  }

  save_send_message_log_event(dialog_id, m);
  do_send_message(dialog_id, m);

  send_update_new_message(d, m);
  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "send_message");
  }

  return get_message_object(dialog_id, m, "send_message");
}

Result<InputMessageContent> MessagesManager::process_input_message_content(
    DialogId dialog_id, tl_object_ptr<td_api::InputMessageContent> &&input_message_content) {
  if (input_message_content == nullptr) {
    return Status::Error(400, "Can't send message without content");
  }

  if (input_message_content->get_id() == td_api::inputMessageForwarded::ID) {
    // for sendMessageAlbum/editMessageMedia/addLocalMessage
    auto input_message = td_api::move_object_as<td_api::inputMessageForwarded>(input_message_content);
    TRY_RESULT(copy_options, process_message_copy_options(dialog_id, std::move(input_message->copy_options_)));
    if (!copy_options.send_copy) {
      return Status::Error(400, "Can't use forwarded message");
    }

    DialogId from_dialog_id(input_message->from_chat_id_);
    Dialog *from_dialog = get_dialog_force(from_dialog_id, "send_message copy");
    if (from_dialog == nullptr) {
      return Status::Error(400, "Chat to copy message from not found");
    }
    if (!have_input_peer(from_dialog_id, AccessRights::Read)) {
      return Status::Error(400, "Can't access the chat to copy message from");
    }
    if (from_dialog_id.get_type() == DialogType::SecretChat) {
      return Status::Error(400, "Can't copy message from secret chats");
    }
    MessageId message_id = get_persistent_message_id(from_dialog, MessageId(input_message->message_id_));

    const Message *copied_message = get_message_force(from_dialog, message_id, "process_input_message_content");
    if (copied_message == nullptr) {
      return Status::Error(400, "Can't find message to copy");
    }
    if (!can_forward_message(from_dialog_id, copied_message)) {
      return Status::Error(400, "Can't copy message");
    }

    unique_ptr<MessageContent> content = dup_message_content(td_, dialog_id, copied_message->content.get(),
                                                             MessageContentDupType::Copy, std::move(copy_options));
    if (content == nullptr) {
      return Status::Error(400, "Can't copy message content");
    }

    return InputMessageContent(std::move(content), get_message_disable_web_page_preview(copied_message), false, 0,
                               UserId(), copied_message->send_emoji);
  }

  TRY_RESULT(content, get_input_message_content(dialog_id, std::move(input_message_content), td_));

  if (content.ttl < 0 || content.ttl > MAX_PRIVATE_MESSAGE_TTL) {
    return Status::Error(400, "Invalid message content TTL specified");
  }
  if (content.ttl > 0 && dialog_id.get_type() != DialogType::User) {
    return Status::Error(400, "Message content TTL can be specified only in private chats");
  }

  if (dialog_id != DialogId()) {
    TRY_STATUS(can_send_message_content(dialog_id, content.content.get(), false, td_));
  }

  return std::move(content);
}

Result<MessageCopyOptions> MessagesManager::process_message_copy_options(
    DialogId dialog_id, tl_object_ptr<td_api::messageCopyOptions> &&options) const {
  if (options == nullptr || !options->send_copy_) {
    return MessageCopyOptions();
  }
  MessageCopyOptions result;
  result.send_copy = true;
  result.replace_caption = options->replace_caption_;
  if (result.replace_caption) {
    TRY_RESULT_ASSIGN(result.new_caption,
                      process_input_caption(td_->contacts_manager_.get(), dialog_id, std::move(options->new_caption_),
                                            td_->auth_manager_->is_bot()));
  }
  return std::move(result);
}

Result<MessagesManager::MessageSendOptions> MessagesManager::process_message_send_options(
    DialogId dialog_id, tl_object_ptr<td_api::messageSendOptions> &&options) const {
  MessageSendOptions result;
  if (options != nullptr) {
    result.disable_notification = options->disable_notification_;
    result.from_background = options->from_background_;
    TRY_RESULT_ASSIGN(result.schedule_date, get_message_schedule_date(std::move(options->scheduling_state_)));
  }

  auto dialog_type = dialog_id.get_type();
  if (result.schedule_date != 0) {
    if (dialog_type == DialogType::SecretChat) {
      return Status::Error(400, "Can't schedule messages in secret chats");
    }
    if (td_->auth_manager_->is_bot()) {
      return Status::Error(400, "Bots can't send scheduled messages");
    }
  }
  if (result.schedule_date == SCHEDULE_WHEN_ONLINE_DATE) {
    if (dialog_type != DialogType::User) {
      return Status::Error(400, "Messages can be scheduled till online only in private chats");
    }
    if (dialog_id == get_my_dialog_id()) {
      return Status::Error(400, "Can't scheduled till online messages in chat with self");
    }
  }

  return result;
}

Status MessagesManager::can_use_message_send_options(const MessageSendOptions &options,
                                                     const unique_ptr<MessageContent> &content, int32 ttl) {
  if (options.schedule_date != 0) {
    if (ttl > 0) {
      return Status::Error(400, "Can't send scheduled self-destructing messages");
    }
    if (content->get_type() == MessageContentType::LiveLocation) {
      return Status::Error(400, "Can't send scheduled live location messages");
    }
  }

  return Status::OK();
}

Status MessagesManager::can_use_message_send_options(const MessageSendOptions &options,
                                                     const InputMessageContent &content) {
  return can_use_message_send_options(options, content.content, content.ttl);
}

Status MessagesManager::can_use_top_thread_message_id(Dialog *d, MessageId top_thread_message_id,
                                                      MessageId reply_to_message_id) {
  if (top_thread_message_id == MessageId()) {
    return Status::OK();
  }

  if (!top_thread_message_id.is_valid() || !top_thread_message_id.is_server()) {
    return Status::Error(400, "Invalid message thread ID specified");
  }
  if (d->dialog_id.get_type() != DialogType::Channel || is_broadcast_channel(d->dialog_id)) {
    return Status::Error(400, "Chat doesn't have threads");
  }
  if (reply_to_message_id.is_valid()) {
    const Message *reply_m = get_message_force(d, reply_to_message_id, "can_use_top_thread_message_id 1");
    if (reply_m != nullptr && top_thread_message_id != reply_m->top_thread_message_id) {
      if (reply_m->top_thread_message_id.is_valid() || reply_m->media_album_id == 0) {
        return Status::Error(400, "The message to reply is not in the specified message thread");
      }

      // if the message is in an album and not in the thread, it can be in the album of top_thread_message_id
      const Message *top_m = get_message_force(d, top_thread_message_id, "can_use_top_thread_message_id 2");
      if (top_m != nullptr &&
          (top_m->media_album_id != reply_m->media_album_id || top_m->top_thread_message_id != top_m->message_id)) {
        return Status::Error(400, "The message to reply is not in the specified message thread root album");
      }
    }
  }

  return Status::OK();
}

int64 MessagesManager::generate_new_media_album_id() {
  int64 media_album_id = 0;
  do {
    media_album_id = Random::secure_int64();
  } while (media_album_id >= 0 || pending_message_group_sends_.count(media_album_id) != 0);
  return media_album_id;
}

Result<vector<MessageId>> MessagesManager::send_message_group(
    DialogId dialog_id, MessageId top_thread_message_id, MessageId reply_to_message_id,
    tl_object_ptr<td_api::messageSendOptions> &&options,
    vector<tl_object_ptr<td_api::InputMessageContent>> &&input_message_contents) {
  if (input_message_contents.size() > MAX_GROUPED_MESSAGES) {
    return Status::Error(400, "Too much messages to send as an album");
  }
  if (input_message_contents.empty()) {
    return Status::Error(400, "There are no messages to send");
  }

  Dialog *d = get_dialog_force(dialog_id, "send_message_group");
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }

  TRY_STATUS(can_send_message(dialog_id));
  TRY_RESULT(message_send_options, process_message_send_options(dialog_id, std::move(options)));

  vector<std::pair<unique_ptr<MessageContent>, int32>> message_contents;
  std::unordered_set<MessageContentType, MessageContentTypeHash> message_content_types;
  for (auto &input_message_content : input_message_contents) {
    TRY_RESULT(message_content, process_input_message_content(dialog_id, std::move(input_message_content)));
    TRY_STATUS(can_use_message_send_options(message_send_options, message_content));
    auto message_content_type = message_content.content->get_type();
    if (!is_allowed_media_group_content(message_content_type)) {
      return Status::Error(400, "Invalid message content type");
    }
    message_content_types.insert(message_content_type);

    message_contents.emplace_back(std::move(message_content.content), message_content.ttl);
  }
  if (message_content_types.size() > 1) {
    for (auto message_content_type : message_content_types) {
      if (is_homogenous_media_group_content(message_content_type)) {
        return Status::Error(400, PSLICE() << message_content_type << " can't be mixed with other media types");
      }
    }
  }

  reply_to_message_id = get_reply_to_message_id(d, top_thread_message_id, reply_to_message_id, false);
  TRY_STATUS(can_use_top_thread_message_id(d, top_thread_message_id, reply_to_message_id));

  int64 media_album_id = 0;
  if (message_contents.size() > 1) {
    media_album_id = generate_new_media_album_id();
  }

  // there must be no errors after get_message_to_send calls

  vector<MessageId> result;
  bool need_update_dialog_pos = false;
  for (size_t i = 0; i < message_contents.size(); i++) {
    auto &message_content = message_contents[i];
    Message *m = get_message_to_send(d, top_thread_message_id, reply_to_message_id, message_send_options,
                                     dup_message_content(td_, dialog_id, message_content.first.get(),
                                                         MessageContentDupType::Send, MessageCopyOptions()),
                                     &need_update_dialog_pos, i != 0);
    result.push_back(m->message_id);
    auto ttl = message_content.second;
    if (ttl > 0) {
      m->ttl = ttl;
      m->is_content_secret = is_secret_message_content(m->ttl, m->content->get_type());
    }
    m->media_album_id = media_album_id;

    save_send_message_log_event(dialog_id, m);
    do_send_message(dialog_id, m);

    send_update_new_message(d, m);
  }

  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "send_message_group");
  }

  return result;
}

void MessagesManager::save_send_message_log_event(DialogId dialog_id, const Message *m) {
  if (!G()->parameters().use_message_db) {
    return;
  }

  CHECK(m != nullptr);
  LOG(INFO) << "Save " << FullMessageId(dialog_id, m->message_id) << " to binlog";
  auto log_event = SendMessageLogEvent(dialog_id, m);
  CHECK(m->send_message_log_event_id == 0);
  m->send_message_log_event_id =
      binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::SendMessage, get_log_event_storer(log_event));
}

void MessagesManager::do_send_message(DialogId dialog_id, const Message *m, vector<int> bad_parts) {
  bool is_edit = m->message_id.is_any_server();
  LOG(INFO) << "Do " << (is_edit ? "edit" : "send") << ' ' << FullMessageId(dialog_id, m->message_id);
  bool is_secret = dialog_id.get_type() == DialogType::SecretChat;

  if (m->media_album_id != 0 && bad_parts.empty() && !is_secret && !is_edit) {
    auto &request = pending_message_group_sends_[m->media_album_id];
    request.dialog_id = dialog_id;
    request.message_ids.push_back(m->message_id);
    request.is_finished.push_back(false);

    request.results.push_back(Status::OK());
  }

  auto content = is_edit ? m->edited_content.get() : m->content.get();
  CHECK(content != nullptr);
  auto content_type = content->get_type();
  if (content_type == MessageCon