// licence isn't clear on github but many files and commits have
// the very unsurprising LGPL2.1+ so  I  think that's OP's intent
// https://github.com/Mistyazur/QQREncode

#include "qqrencode.h"
#include "qqrencode_p.h"

QQREncodePrivate::~QQREncodePrivate()
{
    QRcode_free(m_code);
}

void QQREncodePrivate::paint(QPainter &painter)
{

    int symwidth = m_code->width + m_margin * 2;
    painter.setClipRect(QRect(0,0,symwidth, symwidth));
    painter.setPen(m_pen);
    painter.setBrush(m_fg);

    /* Make solid background */
    painter.fillRect(QRect(0, 0, symwidth, symwidth), m_bg);

    /* Write data */
    unsigned char *p = m_code->data;
    for(int y=0; y<m_code->width; y++) {
        unsigned char *row = (p+(y*m_code->width));
        /* no RLE */
        for(int x=0; x<m_code->width; x++) {
            if(*(row+x)&0x1) {
                painter.drawRect(m_margin + x, m_margin + y, 1, 1);
            }
        }

    }
}

QQREncode::QQREncode()
    : d_ptr(new QQREncodePrivate(this))
{
}

QQREncode::~QQREncode()
{
}


bool QQREncode::encode(QString input, bool caseSensitive)
{
    Q_D(QQREncode);
    if (input.isEmpty()) return false;
    QRcode *c = NULL;
    if (d->m_micro) {
        c = QRcode_encodeStringMQR(input.toStdString().c_str(),
                                   d->m_version,
                                   d->m_level,
                                   QR_MODE_8,
                                   (caseSensitive) ? 1 : 0);
    } else {
        c = QRcode_encodeString(input.toStdString().c_str(),
                                d->m_version,
                                d->m_level,
                                QR_MODE_8,
                                (caseSensitive) ? 1 : 0);
    }
    if (!c) {
        return false;
    }
    if (d->m_code) QRcode_free(d->m_code);
    d->m_code = c;
    return true;
}

QImage QQREncode::toQImage(int size)
{
    Q_D(QQREncode);
    if (size < 0) throw std::invalid_argument("Invalid size");

    if (!d->m_code) {
        throw std::logic_error("No qr code to convert");
    }

    int symwidth = d->m_code->width + d->m_margin * 2;
    QImage result(QSize(symwidth, symwidth), QImage::Format_Mono);
    result.fill(Qt::white);

    QPainter painter;
    painter.begin(&result);
    d->paint(painter);
    painter.end();

    if (size > 0)
        return result.scaled(size, size);
    return result;
}
